/*
 * Copyright (C) 2000-2024 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>

#include "lang.h"

#include "common.h"
#include "menus.h"
#include "acontrol.h"
#include "control.h"
#include "event_sender.h"
#include "mrl_browser.h"
#include "panel.h"
#include "playlist.h"
#include "videowin.h"
#include "actions.h"
#include "event.h"
#include "xine-toolkit/menu.h"

#define PLAYL_NO_LOOP    (PLAYLIST_LAST + 0)
#define PLAYL_LOOP       (PLAYLIST_LAST + 1)
#define PLAYL_REPEAT     (PLAYLIST_LAST + 2)
#define PLAYL_SHUFFLE    (PLAYLIST_LAST + 3)
#define PLAYL_SHUF_PLUS  (PLAYLIST_LAST + 4)
#define PLAYL_GET_FROM   (PLAYLIST_LAST + 5)
#define PLAYL_OPEN_MRLB  (PLAYLIST_LAST + 6)


#define AUDIO_DECRE_VOL          0
#define AUDIO_INCRE_VOL          1

#define CTRL_RESET               0

#define _MENU_GUI_ACTION_BASE   (0 << 24)
#define _MENU_ASPECT_BASE       (1 << 24)
#define _MENU_AUDIO_CMD_BASE    (2 << 24)
#define _MENU_AUDIO_CHAN_BASE   ((3 << 24) + 2)
#define _MENU_AUDIO_VIZ_BASE    (4 << 24)
#define _MENU_SUBT_CHAN_BASE    ((5 << 24) + 2)
#define _MENU_PLAYL_CMD_BASE    (6 << 24)
#define _MENU_CTRL_CMD_BASE     (7 << 24)
#define _MENU_NAV_BASE          (8 << 24)
#define _MENU_VOL_BASE          (9 << 24)

typedef struct {
  char *write, *end, buf[1024];
} menu_text_buf_t;

static void _menu_set_shortcuts (gGui_t *gui, menu_text_buf_t *tbuf, xitk_menu_entry_t *m, int n) {
  xitk_menu_entry_t *e;

  for (e = m; n > 0; e++, n--) {
    size_t sz;

    if (tbuf->write + 1 >= tbuf->end) {
      if (gui->verbosity >= 1)
        printf ("gui.menu: text buffer overflow.\n");
      break;
    }
    if (!e->shortcut)
      continue;
    sz = kbindings_get_shortcut (gui->kbindings, e->shortcut, tbuf->write, tbuf->end - tbuf->write, gui->shortcut_style);
    if (!sz) {
      /* no shortcut set (VOID) */
      e->shortcut = NULL;
    } else if (tbuf->write + sz == tbuf->end) {
      if (gui->verbosity >= 2)
        printf ("gui.menu: shortcut for action \"%s\" was truncated.\n", e->shortcut);
      e->shortcut = tbuf->write;
    } else {
      e->shortcut = tbuf->write;
    }
    tbuf->write += sz + 1;
  }
  for (; n > 0; e++, n--)
    e->shortcut = NULL;
}

static void _menu_action (xitk_widget_t *w, xitk_menu_entry_t *me, void *data) {
  gGui_t *gui = data;

  (void)w;
  if (!me)
    return;

  switch (me->user_id >> 24) {

    case (_MENU_GUI_ACTION_BASE >> 24):
      gui_execute_action_id (gui, me->user_id - _MENU_GUI_ACTION_BASE);
      break;

    case (_MENU_ASPECT_BASE >> 24):
      gui_toggle_aspect (gui, me->user_id - _MENU_ASPECT_BASE);
      break;

    case (_MENU_AUDIO_CMD_BASE >> 24):
      switch (me->user_id - _MENU_AUDIO_CMD_BASE) {
        static const uint8_t _step[LAST_MIXER] = {
          [SOUND_CARD_MIXER] = 100 / 10, [SOFTWARE_MIXER] = 200 / 10
        };
        case AUDIO_DECRE_VOL:
        case AUDIO_INCRE_VOL:
          gui_set_audio_vol (gui, GUI_AUDIO_VOL_RELATIVE +
            (int)_step[gui->mixer.type_volume] * (2 * ((int)me->user_id - _MENU_AUDIO_CMD_BASE - AUDIO_DECRE_VOL) - 1));
          break;
        default:
          printf("%s(): unknown control %d\n", __XINE_FUNCTION__, me->user_id - _MENU_CTRL_CMD_BASE);
          break;
      }
      break;

    case (_MENU_AUDIO_CHAN_BASE >> 24):
      gui_direct_change_audio_channel (w, gui, me->user_id - _MENU_AUDIO_CHAN_BASE);
      break;

    case (_MENU_AUDIO_VIZ_BASE >> 24):
      config_update_num (gui->xine, "gui.post_audio_plugin", me->user_id - _MENU_AUDIO_VIZ_BASE);
      break;

    case (_MENU_SUBT_CHAN_BASE >> 24):
      gui_direct_change_spu_channel (w, gui, me->user_id - _MENU_SUBT_CHAN_BASE);
      break;

    case (_MENU_PLAYL_CMD_BASE >> 24):
      switch (me->user_id - _MENU_PLAYL_CMD_BASE) {
        case PLAYLIST_LOAD:
        case PLAYLIST_SAVE:
        case PLAYLIST_CURR_REMOVE:
        case PLAYLIST_CLEAR:
        case PLAYLIST_CURR_UP:
        case PLAYLIST_CURR_DOWN:
	case PLAYLIST_CURR_PLAY:
        case PLAYLIST_CURR_SCAN:
	case PLAYLIST_SCAN:
          playlist_action (gui, me->user_id - _MENU_PLAYL_CMD_BASE);
          break;
        case PLAYL_NO_LOOP:
        case PLAYL_LOOP:
        case PLAYL_REPEAT:
        case PLAYL_SHUFFLE:
        case PLAYL_SHUF_PLUS:
          gui_execute_action_id (gui, ACTID_EVENT_NUMBER_0 + (me->user_id - _MENU_PLAYL_CMD_BASE - PLAYL_NO_LOOP));
          gui_execute_action_id (gui, ACTID_LOOPMODE);
          break;
        case PLAYL_GET_FROM:
          playlist_scan_input_s (gui, me->menu);
          break;
        case PLAYL_OPEN_MRLB:
          open_mrlbrowser_from_playlist (w, gui, 0);
          break;
        default:
          printf ("%s(): unknown control %d\n", __XINE_FUNCTION__, me->user_id - _MENU_PLAYL_CMD_BASE);
      }
      break;

    case (_MENU_CTRL_CMD_BASE >> 24):
      switch (me->user_id - _MENU_CTRL_CMD_BASE) {
        case CTRL_RESET:
          control_reset (gui->vctrl);
          break;
        default: ;
      }
      break;

    case (_MENU_NAV_BASE >> 24):
      event_sender_send (gui, me->user_id - _MENU_NAV_BASE);
      break;

    case (_MENU_VOL_BASE >> 24):
      switch (me->user_id - _MENU_VOL_BASE) {
        case SOUND_CARD_MIXER:
        case SOFTWARE_MIXER:
          if ((int)gui->mixer.type_volume == me->user_id - _MENU_VOL_BASE)
            break;
          gui->mixer.type_volume = me->user_id - _MENU_VOL_BASE;
          panel_update_mixer_display (gui->panel);
          break;
        case LAST_MIXER:
          acontrol_main (XUI_W_ON, gui);
          break;
        default: ;
      }
      break;

    default: ;
  }
}

static const uint8_t _menu_check_lut[16] = {
  XITK_MENU_ENTRY_CHECK, XITK_MENU_ENTRY_CHECK, XITK_MENU_ENTRY_CHECK, XITK_MENU_ENTRY_CHECK,
  XITK_MENU_ENTRY_CHECK, XITK_MENU_ENTRY_CHECK, XITK_MENU_ENTRY_CHECK, XITK_MENU_ENTRY_CHECK,
  XITK_MENU_ENTRY_CHECKED,
  XITK_MENU_ENTRY_CHECK, XITK_MENU_ENTRY_CHECK, XITK_MENU_ENTRY_CHECK, XITK_MENU_ENTRY_CHECK,
  XITK_MENU_ENTRY_CHECK, XITK_MENU_ENTRY_CHECK, XITK_MENU_ENTRY_CHECK
};

static const char num15[] = " 0 1 2 3 4 5 6 7 8 91011121314";

void video_window_menu (gGui_t *gui, xitk_widget_list_t *wl, int x, int y) {
  unsigned int         aspect = xine_get_param (gui->stream, XINE_PARAM_VO_ASPECT_RATIO) + 8;
  int                  achannel = xine_get_param (gui->stream, XINE_PARAM_AUDIO_CHANNEL_LOGICAL);
  int                  schannel = xine_get_param (gui->stream, XINE_PARAM_SPU_CHANNEL);
/* cd po && make update-po
 * will copy the next comment to the *.po files. */
/* TRANSLATORS: the next snippets define a menu tree.
 * Wherever you got something like "foo", "foo/bar" and "foo/bar/buff",
 * make sure to translate all "foo"s the same way,
 * and all "bar"s another same way.
 * Also, just leave "SEP" as is.
 * Different translations for the same word sometimes are good style.
 * However, here they will create unintended extra branches. */
  const char          *audio_channel_auto = _("Audio/Channel/Auto");
  const char          *subtitle_channel_auto = _("Subtitle/Channel/Auto");
  const char          *playlist_get_from = _("Playlist/Get from");
  const char          *audio_visualization = _("Audio/Visualization");
  const char          *audio_channel = _("Audio/Channel");
  const char          *subtitle_channel = _("Subtitle/Channel");
  const char          *net_remote = _("Settings/Network remote control");
  unsigned int         loop_mode = gui->playlist.loop + 8;
  char                 title[256];
  xitk_widget_t       *w;
  float                xmag, ymag;
  int                  fullscr_mode = video_window_get_fullscreen_mode (gui->vwin);
  menu_text_buf_t      tbuf = {tbuf.buf, tbuf.buf + sizeof (tbuf.buf) - 1, {0}};
  xitk_menu_entry_t    menu_entries[] = {
    { XITK_MENU_ENTRY_TITLE, 0,
      NULL, NULL},
    { panel_is_visible (gui->panel) > 1 ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK,
      _MENU_GUI_ACTION_BASE + ACTID_TOGGLE_VISIBLITY,
      _("Show controls"), "ToggleVisibility"},
    { video_window_is_visible (gui->vwin) > 1 ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK,
      _MENU_GUI_ACTION_BASE + ACTID_TOGGLE_WINOUT_VISIBLITY,
      _("Show video window"), "ToggleWindowVisibility"},
    { fullscr_mode & FULLSCR_MODE ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK,
      _MENU_GUI_ACTION_BASE + ACTID_TOGGLE_FULLSCREEN,
      _("Fullscreen"), "ToggleFullscreen"},
#ifdef HAVE_XINERAMA
    { fullscr_mode & FULLSCR_XI_MODE ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK,
      _MENU_GUI_ACTION_BASE + ACTID_TOGGLE_XINERAMA_FULLSCREEN,
      _("Xinerama fullscreen"), "ToggleXineramaFullscr"},
#endif
    { video_window_get_border_mode (gui->vwin) ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK,
      _MENU_GUI_ACTION_BASE + ACTID_TOGGLE_WINOUT_BORDER,
      _("Window frame"), "ToggleWindowBorder"},
    { XITK_MENU_ENTRY_SEPARATOR, 0,
      "SEP", NULL},
    /* no longer needed
    { XITK_MENU_ENTRY_BRANCH, 0,
      _("Open"), NULL},
    */
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_FILESELECTOR,
      _("Open/File..."), "FileSelector"},
    { XITK_MENU_ENTRY_PLAIN, _MENU_PLAYL_CMD_BASE + PLAYLIST_LOAD,
      _("Open/Playlist..."), NULL},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_MRLBROWSER,
      _("Open/Location..."), "MrlBrowser"},
    /* no longer needed
    { XITK_MENU_ENTRY_BRANCH, 0,
      _("Playback"), NULL},
    */
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_PLAY,
      _("Playback/Play"), "Play"},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_STOP,
      _("Playback/Stop"), "Stop"},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_PAUSE,
      _("Playback/Pause"), "Pause"},
    { XITK_MENU_ENTRY_SEPARATOR, 0,
      _("Playback/SEP"), NULL},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_MRL_NEXT,
      _("Playback/Next MRL"), "NextMrl"},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_MRL_PRIOR,
      _("Playback/Previous MRL"), "PriorMrl"},
    { XITK_MENU_ENTRY_SEPARATOR, 0,
      _("Playback/SEP"), NULL},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_SPEED_FAST,
      _("Playback/Increase Speed"), "SpeedFaster"},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_SPEED_SLOW,
      _("Playback/Decrease Speed"), "SpeedSlower"},
    { XITK_MENU_ENTRY_SEPARATOR, 0,
      _("Playback/SEP"), NULL},
    { gui->is_display_mrl ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK, _MENU_GUI_ACTION_BASE + ACTID_SHOW_MRL,
      _("Playback/Show MRL"), "ShowMrl"},
    { gui->runtime_mode ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK, _MENU_GUI_ACTION_BASE + ACTID_SHOW_TIME,
      _("Playback/Show remaining time"), "ShowTime"},
    /* no longer needed
    { XITK_MENU_ENTRY_BRANCH, 0,
      _("Playlist"), NULL},
    */
    { XITK_MENU_ENTRY_BRANCH, 0,
      playlist_get_from, NULL},
    { XITK_MENU_ENTRY_PLAIN, _MENU_PLAYL_CMD_BASE + PLAYLIST_LOAD,
      _("Playlist/Load..."), NULL},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_PLAYLIST,
      _("Playlist/Editor..."), "PlaylistEditor"},
    { XITK_MENU_ENTRY_SEPARATOR, 0,
      _("Playlist/SEP"), NULL},
    { XITK_MENU_ENTRY_BRANCH, 0,
      _("Playlist/Loop modes"), "ToggleLoopMode"},
    { _menu_check_lut[(loop_mode - PLAYLIST_LOOP_NO_LOOP) & 15], _MENU_PLAYL_CMD_BASE + PLAYL_NO_LOOP,
      _("Playlist/Loop modes/Disabled"), NULL},
    { _menu_check_lut[(loop_mode - PLAYLIST_LOOP_LOOP) & 15], _MENU_PLAYL_CMD_BASE + PLAYL_LOOP,
      _("Playlist/Loop modes/Loop"), NULL},
    { _menu_check_lut[(loop_mode - PLAYLIST_LOOP_REPEAT) & 15], _MENU_PLAYL_CMD_BASE + PLAYL_REPEAT,
      _("Playlist/Loop modes/Repeat Selection"), NULL},
    { _menu_check_lut[(loop_mode - PLAYLIST_LOOP_SHUFFLE) & 15], _MENU_PLAYL_CMD_BASE + PLAYL_SHUFFLE,
      _("Playlist/Loop modes/Shuffle"), NULL},
    { _menu_check_lut[(loop_mode - PLAYLIST_LOOP_SHUF_PLUS) & 15], _MENU_PLAYL_CMD_BASE + PLAYL_SHUF_PLUS,
      _("Playlist/Loop modes/Non-stop Shuffle"), NULL},
    { (gui->playlist.control & PLAYLIST_CONTROL_STOP) ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK,
      _MENU_GUI_ACTION_BASE + ACTID_PLAYLIST_STOP,
      _("Playlist/Continue Playback"), "PlaylistStop"},
    { XITK_MENU_ENTRY_SEPARATOR, 0,
      "SEP", NULL},
    /* no longer needed
    { XITK_MENU_ENTRY_BRANCH, 0,
      _("Menus"), NULL},
    */
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_EVENT_SENDER,
      _("Menus/Navigation..."), "EventSenderShow"},
    { XITK_MENU_ENTRY_SEPARATOR, 0,
      _("Menus/SEP"), NULL},
    { XITK_MENU_ENTRY_SEPARATOR, 0,
      "SEP", NULL},
    /* no longer needed
    { XITK_MENU_ENTRY_BRANCH, 0,
      _("Stream"), NULL},
    */
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_STREAM_INFOS,
      _("Stream/Information..."), "StreamInfosShow"},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_OSD_SINFOS,
      _("Stream/Information (OSD)"), "OSDStreamInfos"},
    /* no longer needed
    { XITK_MENU_ENTRY_BRANCH, 0,
      _("Video"), NULL},
    */
    { gui->deinterlace_enable ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK,
      _MENU_GUI_ACTION_BASE + ACTID_TOGGLE_INTERLEAVE,
      _("Video/Deinterlace"), "ToggleInterleave"},
    { XITK_MENU_ENTRY_SEPARATOR, 0,
      _("Video/SEP"), NULL},
    { XITK_MENU_ENTRY_BRANCH, 0,
      _("Video/Aspect ratio"), "ToggleAspectRatio"},
    { _menu_check_lut[(aspect - XINE_VO_ASPECT_AUTO) & 15], _MENU_ASPECT_BASE + XINE_VO_ASPECT_AUTO,
      _("Video/Aspect ratio/Automatic"), NULL},
    { _menu_check_lut[(aspect - XINE_VO_ASPECT_SQUARE) & 15], _MENU_ASPECT_BASE + XINE_VO_ASPECT_SQUARE,
      _("Video/Aspect ratio/Square"), NULL},
    { _menu_check_lut[(aspect - XINE_VO_ASPECT_4_3) & 15], _MENU_ASPECT_BASE + XINE_VO_ASPECT_4_3,
      _("Video/Aspect ratio/4:3"), NULL},
    { _menu_check_lut[(aspect - XINE_VO_ASPECT_ANAMORPHIC) & 15], _MENU_ASPECT_BASE + XINE_VO_ASPECT_ANAMORPHIC,
      _("Video/Aspect ratio/Anamorphic"), NULL},
    { _menu_check_lut[(aspect - XINE_VO_ASPECT_DVB) & 15], _MENU_ASPECT_BASE + XINE_VO_ASPECT_DVB,
      _("Video/Aspect ratio/DVB"), NULL},
    { (video_window_get_mag (gui->vwin, &xmag, &ymag), (xmag == 2.0f && ymag == 2.0f))
        ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK,
      _MENU_GUI_ACTION_BASE + ACTID_WINDOW200,
      _("Video/200%"), "Window200"},
    { (video_window_get_mag (gui->vwin, &xmag, &ymag), (xmag == 1.0f && ymag == 1.0f))
        ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK,
      _MENU_GUI_ACTION_BASE + ACTID_WINDOW100,
      _("Video/100%"), "Window100"},
    { (video_window_get_mag (gui->vwin, &xmag, &ymag), (xmag == 0.5f && ymag == 0.5f))
        ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK,
      _MENU_GUI_ACTION_BASE + ACTID_WINDOW50,
      _("Video/50%"), "Window50"},
    { XITK_MENU_ENTRY_SEPARATOR, 0,
      _("Video/SEP"), NULL},
    { (gui->transform.flags & 1) ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK,
      _MENU_GUI_ACTION_BASE + ACTID_FLIP_H,
      _("Video/mirrored"), "FlipH"},
    { (gui->transform.flags & 2) ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK,
      _MENU_GUI_ACTION_BASE + ACTID_FLIP_V,
      _("Video/upside down"), "FlipV"},
    { XITK_MENU_ENTRY_SEPARATOR, 0,
      _("Video/SEP"), NULL},
    /* no longer needed
    { XITK_MENU_ENTRY_BRANCH, 0,
      _("Video/Postprocess"), NULL},
    */
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_VPP,
      _("Video/Postprocess/Chain Reaction..."), "VPProcessShow"},
    { gui->post_video_enable ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK,
      _MENU_GUI_ACTION_BASE + ACTID_VPP_ENABLE,
      _("Video/Postprocess/Enable Postprocessing"), "VPProcessEnable"},
    /* no longer needed
    { XITK_MENU_ENTRY_BRANCH, 0,
      _("Audio"), NULL},
    { XITK_MENU_ENTRY_BRANCH, 0,
      _("Audio/Volume"), NULL},
    */
    { gui->mixer.mute[gui->mixer.type_mute] ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK,
      _MENU_GUI_ACTION_BASE + ACTID_MUTE,
      _("Audio/Volume/Mute"), "Mute"},
    { XITK_MENU_ENTRY_PLAIN, _MENU_AUDIO_CMD_BASE + AUDIO_INCRE_VOL,
      _("Audio/Volume/Increase 10%"), "Volume+"},
    { XITK_MENU_ENTRY_PLAIN, _MENU_AUDIO_CMD_BASE + AUDIO_DECRE_VOL,
      _("Audio/Volume/Decrease 10%"), "Volume-"},
    { XITK_MENU_ENTRY_BRANCH, 0,
      audio_channel, NULL},
    { achannel == -2 ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK, _MENU_AUDIO_CHAN_BASE - 2,
      _("Audio/Channel/Off"), NULL},
    { achannel ==  -1 ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK, _MENU_AUDIO_CHAN_BASE - 1,
      audio_channel_auto, NULL},
    /* need no extra translation. */
    { XITK_MENU_ENTRY_SEPARATOR, 0,
      audio_channel_auto, NULL},
    { XITK_MENU_ENTRY_BRANCH, 0,
      audio_visualization, NULL},
    { XITK_MENU_ENTRY_SEPARATOR, 0,
      _("Audio/SEP"), NULL},
    { XITK_MENU_ENTRY_BRANCH, 0,
      _("Audio/Postprocess"), NULL},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_APP,
      _("Audio/Postprocess/Chain Reaction..."), "APProcessShow"},
    { gui->post_audio_enable ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK,
      _MENU_GUI_ACTION_BASE + ACTID_APP_ENABLE,
      _("Audio/Postprocess/Enable Postprocessing"), "APProcessEnable"},
    /* no longer needed
    { XITK_MENU_ENTRY_BRANCH, 0,
      _("Subtitle"), NULL},
    */
    { XITK_MENU_ENTRY_BRANCH, 0,
      subtitle_channel, NULL},
    { schannel == -2 ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK, _MENU_SUBT_CHAN_BASE - 2,
        _("Subtitle/Channel/Off"), NULL},
    { schannel == -1 ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK, _MENU_SUBT_CHAN_BASE - 1,
        subtitle_channel_auto, NULL},
    /* need no extra translation. */
    { XITK_MENU_ENTRY_SEPARATOR, 0,
        subtitle_channel_auto, NULL},
    { XITK_MENU_ENTRY_SEPARATOR, 0,
      "SEP", NULL},
    /* no longer needed
    { XITK_MENU_ENTRY_BRANCH, 0,
      _("Settings"), NULL},
    */
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_SETUP,
      _("Settings/Setup..."), "SetupShow"},
#ifdef HAVE_CURL
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_SKINDOWNLOAD,
      _("Settings/Skin Downloader..."), "SkinDownload"},
#endif
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_KBEDIT,
      _("Settings/Keymap Editor..."), "KeyBindingEditor"},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_CONTROLSHOW,
      _("Settings/Video..."), "ControlShow"},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_ACONTROLSHOW,
      _("Settings/Audio..."), "AControlShow"},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_TVANALOG,
      _("Settings/TV Analog..."), "TVAnalogShow"},
    { XITK_MENU_ENTRY_SEPARATOR, 0,
      net_remote, NULL},
    { gui->network ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK, _MENU_GUI_ACTION_BASE + ACTID_NET_REMOTE,
      net_remote, NULL},
    { XITK_MENU_ENTRY_SEPARATOR, 0,
      "SEP", NULL},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_HELP_SHOW,
      _("Help..."), "HelpShow"},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_VIEWLOG,
      _("Logs..."), "ViewlogShow"},
    { XITK_MENU_ENTRY_SEPARATOR, 0,
      "SEP", NULL},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_QUIT,
      _("Quit"), "Quit"},
    { XITK_MENU_ENTRY_END, 0,
      NULL, NULL}
  };
  xitk_menu_widget_t menu = {
    .nw = { .wl = wl, .userdata = gui },
    .cb = _menu_action
  };

  if(gui->no_gui)
    return;

  /* reduce server resource consumnption. */
  xitk_menu_hide_menu (gui->xitk);

  _menu_set_shortcuts (gui, &tbuf, menu_entries, sizeof (menu_entries) / sizeof (menu_entries[0]) - 1);

  gui->nongui_error_msg = NULL;

  snprintf (title, sizeof (title), _("xine %s"), VERSION);
  menu_entries[0].menu = title;

  menu.menu_tree = &menu_entries[0];

  w = xitk_noskin_menu_create (&menu, x, y);

  /* Subtitle loader */
  if (gui->playlist.num) {
    xitk_menu_entry_t menu_entry = {
      .type = XITK_MENU_ENTRY_PLAIN,
      .user_id = _MENU_GUI_ACTION_BASE + ACTID_SUBSELECT,
      .menu = _("Open/Subtitle...")
    };

    xitk_menu_add_entry (w, &menu_entry);
    menu_entry.user_id = _MENU_PLAYL_CMD_BASE + PLAYLIST_SAVE;
    menu_entry.menu = _("Playlist/Save...");
    xitk_menu_add_entry (w, &menu_entry);
  }

  { /* Autoplay plugins */
    xitk_menu_entry_t   menu_entry = { .type = XITK_MENU_ENTRY_BASE, .menu = playlist_get_from };
    const char * const *plug_ids = xine_get_autoplay_input_plugin_ids (gui->xine);
    const char         *plug_id;

    xitk_menu_add_entry (w, &menu_entry);
    menu_entry.type = XITK_MENU_ENTRY_PLAIN;
    menu_entry.user_id = _MENU_PLAYL_CMD_BASE + PLAYL_GET_FROM;
    plug_id = *plug_ids++;
    while (plug_id) {
      menu_entry.menu = plug_id;
      xitk_menu_add_entry (w, &menu_entry);
      plug_id = *plug_ids++;
    }
  }

  { /* Audio Viz */
    const char * const *viz_names = post_get_audio_plugins_names (gui);

    if (viz_names && *viz_names) {
      xitk_menu_entry_t  menu_entry = { .type = XITK_MENU_ENTRY_BASE, .menu = audio_visualization };
      int i = 0;

      xitk_menu_add_entry (w, &menu_entry);
      while (viz_names[i]) {
        menu_entry.menu = viz_names[i];
        menu_entry.type = i == gui->visual_anim.post_plugin_num ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK;
        menu_entry.user_id = _MENU_AUDIO_VIZ_BASE + i;
        xitk_menu_add_entry (w, &menu_entry);
        i++;
      }
    } else {
      xitk_menu_entry_t menu_entry = { .menu = NULL, .type = XITK_MENU_ENTRY_BASE };

      xitk_menu_add_entry (w, &menu_entry);
      menu_entry.menu = _("Audio/Visualization/None");
      menu_entry.type = XITK_MENU_ENTRY_PLAIN;
      xitk_menu_add_entry (w, &menu_entry);
    }
  }

  { /* Audio channels */
    xitk_menu_entry_t  menu_entry = { .type = XITK_MENU_ENTRY_BASE, .menu = audio_channel };
    int i;

    xitk_menu_add_entry (w, &menu_entry);
    for (i = 0; i < 32; i++) {
      char   langbuf[XINE_LANG_MAX];

      memset (&langbuf, 0, sizeof (langbuf));
      if (!xine_get_audio_lang (gui->stream, i, &langbuf[0]))
        break;
      menu_entry.menu = get_language_from_iso639_1 (langbuf);
      menu_entry.type = achannel == i ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK;
      menu_entry.user_id = _MENU_AUDIO_CHAN_BASE + i;
      xitk_menu_add_entry (w, &menu_entry);
    }
    if (i == 0) {
      char buf[4];

      menu_entry.menu = buf;
      buf[2] = 0;
      for (i = 0; i < 15; i++) {
        memcpy (buf, num15 + 2 * i, 2);
        menu_entry.type = achannel == i ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK;
        menu_entry.user_id = _MENU_AUDIO_CHAN_BASE + i;
        xitk_menu_add_entry (w, &menu_entry);
      }
    }
  }

  { /* SPU channels */
    xitk_menu_entry_t  menu_entry = { .type = XITK_MENU_ENTRY_BASE, .menu = subtitle_channel };
    int i;

    xitk_menu_add_entry (w, &menu_entry);
    for (i = 0; i < 32; i++) {
      char langbuf[XINE_LANG_MAX];

      memset (&langbuf, 0, sizeof (langbuf));
      if (!xine_get_spu_lang (gui->stream, i, langbuf))
        break;
      menu_entry.menu = get_language_from_iso639_1 (langbuf);
      menu_entry.type = schannel == i ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK;
      menu_entry.user_id = _MENU_SUBT_CHAN_BASE + i;
      xitk_menu_add_entry (w, &menu_entry);
    }
    if (i == 0) {
      char buf[4];

      menu_entry.menu = buf;
      buf[2] = 0;
      for (i = 0; i < 15; i++) {
        memcpy (buf, num15 + 2 * i, 2);
        menu_entry.type = schannel == i ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK;
        menu_entry.user_id = _MENU_SUBT_CHAN_BASE + i;
        xitk_menu_add_entry (w, &menu_entry);
      }
    }
  }

  { /* Menus access */
    static const char menu_shortcuts[7][16] = {
      "Menu", "TitleMenu", "RootMenu", "SubpictureMenu", "AudioMenu", "AngleMenu", "PartMenu"
    };
    const char *menu_entries[7];
    xitk_menu_entry_t  menu_entry = { .type = XITK_MENU_ENTRY_BASE, .menu = _("Menus") };
    int i;

    event_sender_menu_names (gui, menu_entries);
    xitk_menu_add_entry (w, &menu_entry);
    menu_entry.type = XITK_MENU_ENTRY_PLAIN;
    for (i = 0; i < 7; i++) {
      menu_entry.menu = menu_entries [i];
      menu_entry.user_id = _MENU_NAV_BASE + XINE_EVENT_INPUT_MENU1 + i;
      menu_entry.shortcut = menu_shortcuts[i];
      _menu_set_shortcuts (gui, &tbuf, &menu_entry, 1);
      xitk_menu_add_entry (w, &menu_entry);
    }
  }

  /* Mediamark */
  if (xine_get_status (gui->stream) != XINE_STATUS_STOP) {
    xitk_menu_entry_t menu_entry = { .type = XITK_MENU_ENTRY_BASE, .menu = _("Playback") };

    xitk_menu_add_entry (w, &menu_entry);
    menu_entry.type = XITK_MENU_ENTRY_SEPARATOR;
    menu_entry.menu = "SEP";
    xitk_menu_add_entry (w, &menu_entry);
    menu_entry.type      = XITK_MENU_ENTRY_PLAIN;
    menu_entry.user_id   = _MENU_GUI_ACTION_BASE + ACTID_ADDMEDIAMARK;
    menu_entry.menu      = _("Add Mediamark");
    menu_entry.shortcut  = "AddMediamark";
    _menu_set_shortcuts (gui, &tbuf, &menu_entry, 1);
    xitk_menu_add_entry (w, &menu_entry);
  }

  xitk_menu_show_menu(w);
}

void lang_menu (gGui_t *gui, xitk_widget_list_t *wl, int mode, int x, int y) {
  static const unsigned int xine_param[2] = { XINE_PARAM_AUDIO_CHANNEL_LOGICAL, XINE_PARAM_SPU_CHANNEL };
  static const char title[2][10] = { N_("Audio"), N_("Subtitle") };
  static const unsigned int base[2] = { _MENU_AUDIO_CHAN_BASE, _MENU_SUBT_CHAN_BASE };
  typedef int (*get_func) (xine_stream_t *stream, int i, char *buf);
  static const get_func get_fn[2] = { xine_get_audio_lang, xine_get_spu_lang };

  uint32_t _mode = mode & 1;
  int channel = xine_get_param (gui->stream, xine_param[_mode]);
  char buffer[4];
  xitk_menu_entry_t menu_entry = { .menu = buffer, .user_id = base[_mode] };
  xitk_menu_entry_t menu_entries[] = {
    { XITK_MENU_ENTRY_TITLE, 0, gettext (title[_mode]), NULL },
    { channel == -2 ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK, menu_entry.user_id - 2, _("Off"),  NULL },
    { channel == -1 ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK, menu_entry.user_id - 1, _("Auto"), NULL },
    { XITK_MENU_ENTRY_SEPARATOR, 0, "SEP", NULL },
    { XITK_MENU_ENTRY_END, 0, NULL, NULL }
  };
  xitk_menu_widget_t menu = {
    .nw = { .wl = wl, .userdata = gui },
    .cb = _menu_action,
    .menu_tree = menu_entries
  };
  xitk_widget_t *w;
  int i;

  /* reduce server resource consumnption. */
  xitk_menu_hide_menu (gui->xitk);
  w = xitk_noskin_menu_create (&menu, x, y);

  for (i = 0; i < 32; i++) {
    char langbuf[XINE_LANG_MAX];

    memset (&langbuf, 0, sizeof (langbuf));
    if (!get_fn[_mode] (gui->stream, i, &langbuf[0]))
      break;
    menu_entry.menu = get_language_from_iso639_1 (langbuf);
    menu_entry.type = channel == i ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK;
    xitk_menu_add_entry (w, &menu_entry);
    menu_entry.user_id++;
  }
  if (i == 0) {
    buffer[2] = 0;
    for (i = 0; i < 15; i++) {
      memcpy (buffer, num15 + 2 * i, 2);
      menu_entry.type = channel == i ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK;
      xitk_menu_add_entry (w, &menu_entry);
      menu_entry.user_id++;
    }
  }

  xitk_menu_show_menu (w);
}

void playlist_menu (gGui_t *gui, xitk_widget_list_t *wl, int x, int y, uint32_t flags) {
  xitk_widget_t *w = NULL;
  menu_text_buf_t tbuf = {tbuf.buf, tbuf.buf + sizeof (tbuf.buf) - 1, {0}};
  xitk_menu_entry_t menu_entries[15], *e = menu_entries;
  xitk_menu_widget_t menu = {
    .nw = { .wl = wl, .userdata = gui },
    .cb = _menu_action,
    .menu_tree = menu_entries
  };

  e->type = XITK_MENU_ENTRY_TITLE;
  e->user_id = 0;
  e->menu = _("Playlist");
  e->shortcut = NULL;
  e++;
  if (flags & PLAYLIST_MENU_FLAG_PLAY) {
    e->type = XITK_MENU_ENTRY_PLAIN;
    e->user_id = _MENU_PLAYL_CMD_BASE + PLAYLIST_CURR_PLAY;
    e->menu = _("Play");
    e->shortcut = NULL;
    e++;
  }
  e->type = gui->is_display_mrl ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK;
  e->user_id = _MENU_GUI_ACTION_BASE + ACTID_SHOW_MRL;
  e->menu = _("Show MRL");
  e->shortcut = "ShowMrl";
  e++;
  e->type = XITK_MENU_ENTRY_SEPARATOR;
  e->user_id = 0;
  e->menu = "SEP";
  e->shortcut = NULL;
  e++;
  e->type = XITK_MENU_ENTRY_PLAIN;
  e->user_id = (flags & PLAYLIST_MENU_FLAG_PLAY)
             ? (_MENU_PLAYL_CMD_BASE + PLAYLIST_CURR_SCAN)
             : (_MENU_GUI_ACTION_BASE + ACTID_SCANPLAYLIST);
  e->menu = _("Scan");
  e->shortcut = "ScanPlaylistInfo";
  e++;
  e->type = XITK_MENU_ENTRY_PLAIN;
  e->user_id = _MENU_PLAYL_CMD_BASE + PLAYL_OPEN_MRLB;
  e->menu = _("Add");
  e->shortcut = NULL;
  e++;
  if (flags & PLAYLIST_MENU_FLAG_PLAY) {
    e->type = XITK_MENU_ENTRY_PLAIN;
    e->user_id = _MENU_GUI_ACTION_BASE + ACTID_MMKEDITOR;
    e->menu = _("Edit");
    e->shortcut = "MediamarkEditor";
    e++;
    e->type = XITK_MENU_ENTRY_PLAIN;
    e->user_id = _MENU_PLAYL_CMD_BASE + PLAYLIST_CURR_REMOVE;
    e->menu = _("Delete");
    e->shortcut = NULL;
    e++;
  }
  if (flags & PLAYLIST_MENU_FLAG_CLEAR) {
    e->type = XITK_MENU_ENTRY_PLAIN;
    e->user_id = _MENU_PLAYL_CMD_BASE + PLAYLIST_CLEAR;
    e->menu = _("Delete All");
    e->shortcut = NULL;
    e++;
  }
  if (flags & (PLAYLIST_MENU_FLAG_UP | PLAYLIST_MENU_FLAG_DOWN)) {
    e->type = XITK_MENU_ENTRY_SEPARATOR;
    e->user_id = 0;
    e->menu = "SEP";
    e->shortcut = NULL;
    e++;
  }
  if (flags & PLAYLIST_MENU_FLAG_UP) {
    e->type = XITK_MENU_ENTRY_PLAIN;
    e->user_id = _MENU_PLAYL_CMD_BASE + PLAYLIST_CURR_UP;
    e->menu = _("Move Up");
    e->shortcut = NULL;
    e++;
  }
  if (flags & PLAYLIST_MENU_FLAG_DOWN) {
    e->type = XITK_MENU_ENTRY_PLAIN;
    e->user_id = _MENU_PLAYL_CMD_BASE + PLAYLIST_CURR_DOWN;
    e->menu = _("Move Down");
    e->shortcut = NULL;
    e++;
  }
  e->type = XITK_MENU_ENTRY_END;
  e->user_id = 0;
  e->menu = NULL;
  e->shortcut = NULL;

  _menu_set_shortcuts (gui, &tbuf, menu_entries, e - menu_entries);
  /* reduce server resource consumnption. */
  xitk_menu_hide_menu (gui->xitk);
  w = xitk_noskin_menu_create (&menu, x, y);
  xitk_menu_show_menu (w);
}

void control_menu (gGui_t *gui, xitk_widget_list_t *wl, int x, int y) {
  xitk_menu_entry_t    menu_entries[] = {
    { XITK_MENU_ENTRY_TITLE, 0, _("Video Control"), NULL },
    { XITK_MENU_ENTRY_PLAIN, _MENU_CTRL_CMD_BASE + CTRL_RESET, _("Reset video settings"), NULL },
    { XITK_MENU_ENTRY_END,   0, NULL, NULL }
  };
  xitk_menu_widget_t menu = {
    .nw = { .wl = wl, .userdata = gui },
    .cb = _menu_action,
    .menu_tree = menu_entries
  };
  /* reduce server resource consumnption. */
  xitk_menu_hide_menu (gui->xitk);
  xitk_widget_t *w = xitk_noskin_menu_create (&menu, x, y);

  xitk_menu_show_menu (w);
}

void volume_menu (gGui_t *gui, xitk_widget_list_t *wl, int x, int y) {
  xitk_menu_entry_t    menu_entries[] = {
    { XITK_MENU_ENTRY_TITLE, 0, _("Volume control"), NULL },
    { (gui->mixer.type_volume == SOUND_CARD_MIXER) ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK,
      _MENU_VOL_BASE + SOUND_CARD_MIXER, _("Sound card"), NULL },
    { (gui->mixer.type_volume == SOFTWARE_MIXER) ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK,
      _MENU_VOL_BASE + SOFTWARE_MIXER, _("Software"), NULL },
    { XITK_MENU_ENTRY_SEPARATOR, 0, "SEP", NULL },
/* TRANSLATORS: last of menu tree snippets. */
    { XITK_MENU_ENTRY_PLAIN, _MENU_VOL_BASE + LAST_MIXER, _("Settings..."), NULL },
    { XITK_MENU_ENTRY_END,   0, NULL, NULL }
  };
  xitk_menu_widget_t menu = {
    .nw = { .wl = wl, .userdata = gui },
    .cb = _menu_action,
    .menu_tree = menu_entries
  };
  /* reduce server resource consumnption. */
  xitk_menu_hide_menu (gui->xitk);
  xitk_widget_t *w = xitk_noskin_menu_create (&menu, x, y);

  xitk_menu_show_menu (w);
}

