/*
	SRG - Squid Report Generator
	IP address to username conversion
	
	Copyright 2005 University of Waikato

	This file is part of SRG.

	SRG is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	SRG is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with SRG; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

*/

#include "srg.h"
#include "prototypes.h"

/* Global Data Structures */
list<ip2user_line> ip2users;

/* Initialise the IP address to user mappings */
void init_ip2user(void) {

	/* Exit if no ip2user file specified */
	if (!srg.ip2user)
		return;

	/* Read the file */
	Line *iLine = new Line(srg.ip2user);
	if (iLine->getError() || iLine->eof()) {
		fprintf(stderr, "%s: Could not open the ip to user "
				"conversion file: %s\n", progname, srg.ip2user);
		exit(1);
	}

	/* Process each line of the file */
	while(!iLine->eof()) {
		ip2user_line thisline;
		char *ipLine = iLine->getline();

		/* Check for comment lines */
		if (ipLine[0] == '\0' || ipLine[0] == '#') 
			continue;

		/* Parse the IP Address and User */
		char netmask[1024];
		char address[1024];
		char username[1024];
		if (sscanf(ipLine, "%1023s %1023s %1023s", address,
					netmask, username) != 3) {
			fprintf(stderr, "%s: Error parsing ip to "
					"user file: %s\n", progname, srg.ip2user);
			exit(1);
		}
		thisline.username = strdup(username);
		if (inet_aton(address, &thisline.ipaddress)==0) {
			fprintf(stderr, "Invalid IP Address in ip to "
					"user file: %s\n", address);
			exit(1);
		}
		if (inet_aton(netmask, &thisline.netmask)==0) {
			fprintf(stderr, "Invalid netmask in ip to "
					"user file: %s\n", netmask);
			exit(1);
		}
		/* Add this entry to the list */
		ip2users.push_back(thisline);
	}

	/* Free memory */
	delete iLine;
	
}

/* Looks up the specified ip address in the ip address to username mapping
 * and returns the associated username if it exists via the username 
 * paramter. Returns true if a username was found, false otherwise */
bool ip2username(const in_addr ipaddress, char **username) {
	
	
	list<ip2user_line>::const_iterator iter;
	for (iter=ip2users.begin(); iter != ip2users.end(); iter++) {
		if ((ipaddress.s_addr&(*iter).netmask.s_addr)==
				(*iter).ipaddress.s_addr) {
			*username = strdup((*iter).username);
			return true;
		}
	}
	
	/* otherwise return the original IP address. */
	*username = strdup(inet_ntoa(ipaddress));
	return false;

}
