#  Copyright (C) 1999-2012
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

proc SkyViewDef {} {
    global skyview
    global iskyview

    set iskyview(top) .skyview
    set iskyview(mb) .skyviewmb

    set skyview(sky) fk5
    set skyview(rformat) arcsec
    set skyview(width) 300
    set skyview(height) 300
    set skyview(mode) new
    set skyview(save) 0
    set skyview(valid) 0
    set skyview(survey) sdssi
}

proc SkyViewDialog {} {
    global skyview
    global iskyview
    global wcs

    if [winfo exists $iskyview(top)] {
	raise $iskyview(top)
	return
    }

    global debug
    if {$debug(tcl,image)} {
	puts stderr "SkyViewDialog"
    }

    set varname dskyview
    upvar #0 $varname var
    global $varname

    set var(top) $iskyview(top)
    set var(mb) $iskyview(mb)
    set var(sky) $skyview(sky)
    set var(skyformat) $wcs(skyformat)
    set var(rformat) $skyview(rformat)
    set var(width) $skyview(width)
    set var(height) $skyview(height)
    set var(mode) $skyview(mode)
    set var(save) $skyview(save)
    set var(valid) $skyview(valid)
    set var(survey) $skyview(survey)

    set w $var(top)
    IMGSVRInit $varname "HEASARC-SkyView [msgcat::mc {Server}]" \
	SkyViewExec SkyViewAck

    menu $var(mb).survey
    $var(mb) add cascade -label Survey -menu $var(mb).survey
    $var(mb).survey add cascade -label {Optical} \
	-menu $var(mb).survey.opt
    $var(mb).survey add cascade -label {Infrared} \
	-menu $var(mb).survey.ir
    $var(mb).survey add cascade -label {CMB} \
	-menu $var(mb).survey.cmb

    set f $var(mb).survey.opt
    menu $f
    $f add radiobutton -label {SDSSi} \
	-variable ${varname}(survey) -value sdssi
    $f add radiobutton -label {SDSSr} \
	-variable ${varname}(survey) -value sdssr
    $f add radiobutton -label {SDSSg} \
	-variable ${varname}(survey) -value sdssg
    $f add radiobutton -label {SDSSu} \
	-variable ${varname}(survey) -value sdssu
    $f add radiobutton -label {SDSSz} \
	-variable ${varname}(survey) -value sdssz

    set f $var(mb).survey.ir
    menu $f
    $f add radiobutton -label {WISE 3.4} \
	-variable ${varname}(survey) -value wise3.4
    $f add radiobutton -label {WISE 4.6} \
	-variable ${varname}(survey) -value wise4.6
    $f add radiobutton -label {WISE 12} \
	-variable ${varname}(survey) -value wise12
    $f add radiobutton -label {WISE 22} \
	-variable ${varname}(survey) -value wise22

    set f $var(mb).survey.cmb
    menu $f
    $f add radiobutton -label {WMAP Ka} \
	-variable ${varname}(survey) -value wmapka
    $f add radiobutton -label {WMAP K} \
	-variable ${varname}(survey) -value wmapk
    $f add radiobutton -label {WMAP Q} \
	-variable ${varname}(survey) -value wmapq
    $f add radiobutton -label {WMAP V} \
	-variable ${varname}(survey) -value wmapv
    $f add radiobutton -label {WMAP W} \
	-variable ${varname}(survey) -value wmapw
    $f add separator
    $f add radiobutton -label {COBE DIRBE/AAM} \
	-variable ${varname}(survey) -value cobeaam
    $f add radiobutton -label {COBE DIRBE/ZSMA} \
	-variable ${varname}(survey) -value cobezsma
    $f add radiobutton -label {COBE DIRBE (Old)} \
	-variable ${varname}(survey) -value cobe

    IMGSVRUpdate $varname 1
}

proc SkyViewExec {varname} {
    upvar #0 $varname var
    global $varname

    global debug
    if {$debug(tcl,image)} {
	puts stderr "SkyViewExec $varname"
    }

    if {$var(save)} {
	set var(fn) [SaveFileDialog savefitsfbox]
	if {$var(fn) == {}} {
	    return
	}

    } else {
	set var(fn) [tmpnam ds9skyview ".fits"]
    }

    # skyformat
    switch -- $var(skyformat) {
	degrees {
	    set xx $var(x)
	    set yy $var(y)
	}
	sexagesimal {
	    set xx [h2d [Sex2H $var(x)]]
	    set yy [Sex2D $var(y)]
	}
    }

    # size - convert to arcsec
    switch -- $var(rformat) {
	degrees {
	    set ww $var(width)
	    set hh $var(height)
	}
	arcmin {
	    set ww [expr $var(width)/60.]
	    set hh [expr $var(height)/60.]
	}
	arcsec {
	    set ww [expr $var(width)/60./60.]
	    set hh [expr $var(height)/60./60.]
	}
    }
    if {$ww>5} {
	set ww 5
    }
    if {$hh>5} {
	set hh 5
    }

    # now average
    set rr [expr ($ww+$hh)/2.]

    # query
    switch $var(survey) {
	wmapka -
	wmapk -
	wmapq -
	wmapv -
	wmapw -
	cobeaam -
	cobezsma -
	cobe {
	    set var(query) [http::formatQuery Position $xx,$yy Survey $var(survey) Return FITS]
	}
	default {
	    set var(query) [http::formatQuery Position $xx,$yy Survey $var(survey) Size $ww,$hh Return FITS]
	}
    }
    set var(url) "http://skyview.gsfc.nasa.gov/cgi-bin/images"

    IMGSVRLoad $varname
}

proc SkyViewAck {varname} {
    upvar #0 $varname var
    global $varname

    global debug
    if {$debug(tcl,image)} {
	puts stderr "SkyViewAck $varname"
    }

set msg {Acknowledgments for the DSS-SAO

SkyView was developed and is maintained under NASA ADP Grant
NAS5-32068 with P.I. Thomas A. McGlynn under the auspices of the High
Energy Astrophysics Science Archive Research Center (HEASARC) at the
GSFC Laboratory for High Energy Astrophysics.

We gratefully acknowledge the support of NASA and the contributors of
SkyView data surveys.

Maintained by: Laura McDonald lmm@skyview.gsfc.nasa.gov
}

SimpleTextDialog ${varname}ack [msgcat::mc {Acknowledgment}] 80 40 insert top $msg
}

# Process Cmds


proc ProcessSkyViewCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    SkyViewDialog
    IMGSVRProcessCmd $varname $iname dskyview
}

proc ProcessSendSkyViewCmd {proc id param} {
    SkyViewDialog
    IMGSVRProcessSendCmd $proc $id $param dskyview
}
