//roarlight.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2009-2019
 *
 *  This file is part of roarclients a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  RoarAudio is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 */

#include <roaraudio.h>
#include <libroarlight/libroarlight.h>
#include <stdio.h>

#define CONVAR struct roar_connection * con

static struct roar_connection * g_connection;
static struct roar_vio_calls  * g_stream;

static int __run_argv(int argc, char * argv[]);
static int __open_and_run_file(const char * file);

void usage (void) {
 printf("roarlight [OPTIONS]... command [command...]\n");

 printf("\nOptions:\n\n");

 printf("  --server    SERVER    - Set server hostname\n"
        "  --mixer     MIXERID   - ID of the light mixer to use\n"
        "  --help                - Show this help\n"
       );

 printf("\nCommands:\n\n");
 printf(
        "  help                    - Show this help\n"
        "  sleep TIME              - Sleeps for TIME seconds\n"
        "  sset  chan=val          - Set a DMX Channel\n"
        "  set   chan=val          - Same as sset\n"
        "  event EVENT             - Send event EVENT\n"
        "  @FILE                   - Read commands from FILE or stdin in case of @-\n"
       );
}

static int cmd_sset (char * arg) {
 char * next = arg;
 char * k, * v;
 int32_t chan, val;
 struct roar_roardmx_message mes;

 roar_roardmx_message_new_sset(&mes);

 while (next != NULL) {
  arg  = next;
  next = strstr(next, ",");
  if ( next != NULL ) {
   *next = 0;
    next++;
  }

  k = arg;
  v = strstr(arg, "=");
  if ( v == NULL )
   return -1;

  *v = 0;
   v++;

  chan = atoi(k);
  val  = atoi(v);
//  printf("k='%s'(%i), v='%s'(%i)\n", k, chan, v, val);
  if ( roar_roardmx_message_add_chanval(&mes, chan, val) == -1 )
   return -1;
 }

 if ( roar_roardmx_message_send(&mes, g_stream) == -1 ) {
  return -1;
 }

 return 0;
}

static int cmd_rangeset (char * arg) {
 char * next = arg;
 char * k, * v;
 int32_t start, end, val;
 struct roar_roardmx_message mes;

 roar_roardmx_message_new_rangeset(&mes);

 while (next != NULL) {
  arg  = next;
  next = strstr(next, ",");
  if ( next != NULL ) {
   *next = 0;
    next++;
  }

  k = arg;
  v = strstr(arg, "=");
  if ( v == NULL )
   return -1;

  *v = 0;
   v++;

  val  = atoi(v);

  v = strstr(arg, "-");
  if ( v == NULL ) {
   start = end = atoi(k);
  } else {
  *v = 0;
   v++;
   start = atoi(k);
   end = atoi(v);
  }

//  printf("k='%s'(%i), v='%s'(%i)\n", k, chan, v, val);
  if ( roar_roardmx_message_add_rangeval(&mes, start, end, val) == -1 ) {
   return -1;
  }
 }

 if ( roar_roardmx_message_send(&mes, g_stream) == -1 ) {
  return -1;
 }

 return 0;
}

static int cmd_event (char * arg) {
 struct roar_roardmx_message mes;
 int event;
 char * p;

 if ( roar_roardmx_message_new_event(&mes) == -1 )
  return -1;

 while (arg != NULL) {
  p = strstr(arg, ",");
  if ( p != NULL ) {
   *p = 0;
    p++;
  }

  event = roar_roardmx_str2event(arg);
  if ( event == -1 )
   return -1;

  roar_roardmx_message_add_event(&mes, event);

  arg = p;
 }

 if ( roar_roardmx_message_send(&mes, g_stream) == -1 )
  return -1;

 return 0;
}

static int __run_argv(int argc, char * argv[]) {
 char * k;
 int i;

 for (i = 0; i < argc; i++) {
  k = argv[i];
  // cmd is in k

  printf("--- [ %s ] ---\n", k);

  if ( !strcmp(k, "help") ) {
   usage();

  } else if ( !strcmp(k, "sleep") ) {
   ROAR_CKHAVEARGS(1);
   roar_sleep(atoi(argv[++i]));

  } else if ( !strcmp(k, "sset") || !strcmp(k, "set") ) {
   ROAR_CKHAVEARGS(1);
   i++;
   if ( cmd_sset(argv[i]) == -1 ) {
    fprintf(stderr, "Error: can not set channels\n");
   } else {
    printf("channels changed\n");
   }
  } else if ( !strcmp(k, "rangeset") ) {
   ROAR_CKHAVEARGS(1);
   i++;
   if ( cmd_rangeset(argv[i]) == -1 ) {
    fprintf(stderr, "Error: can not set channels\n");
   } else {
    printf("channels changed\n");
   }
  } else if ( !strcmp(k, "event") ) {
   ROAR_CKHAVEARGS(1);
   i++;
   if ( cmd_event(argv[i]) == -1 ) {
    fprintf(stderr, "Error: can not send event\n");
   } else {
    printf("event sent\n");
   }

  } else if ( *k == '@' ) {
   __open_and_run_file(k);
  } else {
   fprintf(stderr, "Error: invalid command: %s\n", k);
   return 1;
  }
 }

 return 0;
}

static inline void __strip_space(char ** str) {
 for (; **str == ' '; (*str)++);
}

static int __parse_line_and_run(char * line) {
#define MAX_ARGS 16
 int argc = 0;
 char * argv[MAX_ARGS];
 char * p;

 while (line != NULL) {
  if ( argc == MAX_ARGS ) {
   fprintf(stderr, "Error: too many arguments.\n");
   return 1;
  }

  p = strstr(line, " ");
  if ( p != NULL ) {
   *p = 0;
    p++;
   __strip_space(&p);
  }

  argv[argc++] = line;

  line = p;
 }

 return __run_argv(argc, argv);
}

static int __run_file(FILE * file) {
 char buffer[1024];
 ssize_t len;
 int ret;

 while (fgets(buffer, sizeof(buffer), file) != NULL) {
  len = roar_mm_strlen(buffer);
  if ( len > 0 && buffer[len-1] == '\n' )
   buffer[len-1] = 0;

  if ( !len || buffer[0] == 0 )
   continue;

  ret = __parse_line_and_run(buffer);
  if ( ret != 0 )
   return ret;
 }

 return 0;
}

static int __open_and_run_file(const char * file) {
 FILE * input;
 int ret;

 if ( !strcmp(file, "@-") ) {
  return __run_file(stdin);
 } else {
  input = fopen(file+1, "r");
  if ( input == NULL ) {
   fprintf(stderr, "Error: can not open input file: %s\n", file+1);
   return 1;
  }
  ret = __run_file(input);
  fclose(input);
  return ret;
 }
}

int main (int argc, char * argv[]) {
 struct roar_connection con;
 struct roar_vio_calls vio;
 int    mixer = -1; // -1 = Default
 const char * server   = NULL;
 const char * k;
 int    i;
 int    ret;

 for (i = 1; i < argc; i++) {
  k = argv[i];

  if ( !strcmp(k, "--server") || !strcmp(k, "-s") ) {
   ROAR_CKHAVEARGS(1);
   server = argv[++i];
  } else if ( !strcmp(k, "--mixer") ) {
   ROAR_CKHAVEARGS(1);
   mixer = atoi(argv[++i]);
  } else if ( !strcmp(k, "--help") || !strcmp(k, "-h") ) {
   usage();
   return 0;
  } else if ( *k == '-' ) {
   fprintf(stderr, "Error: unknown argument: %s\n", k);
   usage();
   return 1;
  } else {
   break;
  }
 }

 if ( i == argc ) {
  fprintf(stderr, "Error: No Commands given\n");
  return 0; // this is not a fatal error...
 }

 if ( roar_simple_connect(&con, server, "roarlight") == -1 ) {
  fprintf(stderr, "Error: Can not connect to server\n");
  return 1;
 }

 if ( roar_vio_simple_new_stream_obj(&vio, &con, NULL,
                                     ROAR_RATE_DEFAULT, ROAR_CHANNELS_DEFAULT, ROAR_BITS_DEFAULT,
                                     ROAR_CODEC_ROARDMX, ROAR_DIR_LIGHT_IN, mixer) == -1 ) {
 }

 g_connection = &con;
 g_stream = &vio;

 ret = __run_argv(argc - i, argv + i);

 // try to flush all data:

 roar_vio_sync(&vio);
 roar_vio_close(&vio);
 roar_usleep(100);

 roar_disconnect(&con);

 return ret;
}

//ll
