/*
 *  Copyright (c) 2009 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Template.h"

#include <fstream>

#ifdef LLVM_27_OR_28
#include <llvm/System/Path.h>
#else
#include <llvm/Support/Path.h>
#endif

#include <GTLCore/CompilationMessage.h>
#include <GTLCore/CompilationMessages.h>
#include <GTLCore/Macros_p.h>
#include <GTLCore/SysUtils_p.h>

#include "Debug.h"
#include "templatecompiler/GenerationContext_p.h"
#include "templatecompiler/TemplateAST_p.h"
#include "templatecompiler/TemplateLexer_p.h"
#include "templatecompiler/TemplateParser_p.h"
#include "Module.h"

using namespace OpenCTL;

namespace {
  std::list<GTLCore::String> TemplateIncludeDirectories;
}

struct Template::Private {
  Private() : isCompiled(false), node(0) {}
  GTLCore::String source;
  GTLCore::String filename;
  GTLCore::String name;
  bool isCompiled;
  TemplateAST::Node* node;
  GTLCore::CompilationMessages errorMessages;
};

STATIC_INITIALISATION(TemplateDirectories)
{
  TemplateIncludeDirectories.push_back("./");
}

Template::Template() : d(new Private)
{
}

Template::~Template()
{
  delete d;
}

GTLCore::String Template::name() const
{
  return d->name;
}

void Template::setSource(const GTLCore::String& name, const GTLCore::String& source)
{
  d->name = name;
  d->source = source;
}

void Template::loadFromFile(const GTLCore::String& fileName)
{
  d->filename = fileName;
  d->name = llvm::sys::Path((const std::string&)d->filename).getBasename().str();
  std::ifstream in;
  in.open(fileName.c_str() );
  if(not in)
  {
    OCTL_DEBUG( "Impossible to open file " << fileName );
    return;
  }
  std::string str;
  std::getline(in,str);
  while ( in ) {
    d->source += str;
    d->source += "\n";
    std::getline(in,str);
  }
}

GTLCore::String Template::fileName() const
{
  return d->filename;
}

void Template::compile()
{
  delete d->node;
  d->node = 0;
  
  std::istringstream iss(d->source);
  TemplateLexer lexer(&iss);
  std::list<GTLCore::String> directories = TemplateIncludeDirectories;
  GTLCore::String dir = GTLCore::extractDir( d->filename );
  if( dir != "" )
  {
    directories.push_back( dir );
  }
  TemplateParser parser(&lexer, d->filename, directories);
  d->node = parser.parse();
  if(d->node)
  {
    d->isCompiled = true;
  } else {
    d->isCompiled = false;
    d->errorMessages = parser.errorMessages();
  }
}

bool Template::isCompiled() const
{
  return d->isCompiled;
}

GTLCore::String Template::generateSource( const std::map< GTLCore::String, const GTLCore::Type* >& _namedTypes,
                  const GTLCore::PixelDescription& _pixelDescription )
{
  TemplateGenerationContext context(_namedTypes, _pixelDescription);
  d->node->generate(&context);
  return context.code();
}

Module* Template::generateModule( const std::map< GTLCore::String, const GTLCore::Type* >& _namedTypes,
                  const GTLCore::PixelDescription& _pixelDescription )
{
  Module* module = new Module();
  module->setSource( name(), generateSource(_namedTypes, _pixelDescription));
  return module;
}

Module* Template::generateModule( const GTLCore::PixelDescription& _pixelDescription )
{
  std::map< GTLCore::String, const GTLCore::Type* > namedTypes;
  return generateModule(namedTypes, _pixelDescription);
}

void Template::addIncludeDirectory(const GTLCore::String& directory)
{
  TemplateIncludeDirectories.push_back(directory);
}

GTLCore::CompilationMessages Template::compilationMessages() const
{
  return d->errorMessages;
}
