/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#ifndef MEDIASCANNER_MEDIAUTILS_H
#define MEDIASCANNER_MEDIAUTILS_H

// C++ Standard Library
#include <string>
#include <vector>

// Media Scanner Library
#include "mediascanner/property.h"

typedef struct _GList GList;
typedef struct _GrlMedia GrlMedia;

namespace mediascanner {

class MimeType {
public:
    static const MimeType kApplicationOgg;
    static const MimeType kAudioPrefix;
    static const MimeType kImagePrefix;
    static const MimeType kVideoPrefix;

    explicit MimeType(const std::string &str)
        : str_(str.begin(), str.end()) {
    }

    explicit MimeType(const std::wstring &str)
        : str_(str) {
    }

    const std::wstring& str() const {
        return str_;
    }

    bool is_audio() const;
    bool is_image() const;
    bool is_video() const;

    GrlMedia* make_media() const;

private:
    std::wstring str_;
};

class MediaInfo {
private:
    typedef std::vector<Property::ValueMap> PropertyVector;

public:
    typedef PropertyVector::const_iterator const_iterator;
    typedef PropertyVector::value_type value_type;

    MediaInfo();

    void add_related(const Property::ValueMap &properties);
    void add_single(const Property::BoundValue &value);

    Property::Value first(Property key) const;
    size_t count(Property key) const;

    template<typename PropertyType, typename ValueType>
    ValueType first(const GenericProperty<PropertyType, ValueType> &key) const;

    std::wstring first(StringProperty key) const {
        return first<StringProperty, std::wstring>(key);
    }

    std::wstring first(TextProperty key) const {
        return first<TextProperty, std::wstring>(key);
    }


    GrlMedia* make_media(GList *const requested_keys) const;
    GrlMedia* make_media(GList *const requested_keys,
                         const std::string &url) const;

    void copy_to_media(GList *const requested_keys,
                       GrlMedia *const media) const;
    bool fill_from_media(GrlMedia *media, const GList *const requested_keys,
                         GList **failed_keys, std::string *error_message);

    const_iterator begin() const {
        return related_properties_.begin();
    }

    const_iterator end() const {
        return related_properties_.end();
    }

    bool empty() const;

    bool operator==(const MediaInfo &other) const {
        return related_properties_ == other.related_properties_;
    }

    bool operator!=(const MediaInfo &other) const {
        return not operator==(other);
    }

private:
    std::vector<Property::ValueMap> related_properties_;
};

template<typename P, typename V>
inline V MediaInfo::first(const GenericProperty<P, V> &key) const {
    const Property::Value value = first(static_cast<Property>(key));
    return value.which() ? boost::get<V>(value) : V();
}

} // namespace mediascanner

#endif // MEDIASCANNER_MEDIAUTILS_H
