/*
 *  Copyright (C) 2004-2021 Edward F. Valeev
 *
 *  This file is part of Libint.
 *
 *  Libint is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Libint is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Libint.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

// this table ensures relative precision of std::numeric_limits<double>::epsilon()
#ifndef LIBINT2_STATICS_INITIALIZATION
static constexpr std::size_t interpolation_order=7;
static constexpr double cheb_table_tmax=117;
static constexpr double cheb_table_delta=1./7;
static constexpr std::size_t cheb_table_mmax=40;
static constexpr std::size_t cheb_table_nintervals=819;
#if LIBINT2_CONSTEXPR_STATICS
static constexpr double cheb_table[cheb_table_nintervals][(cheb_table_mmax+1)*(interpolation_order+1)]=
#else
static double cheb_table[cheb_table_nintervals][(cheb_table_mmax+1)*(interpolation_order+1)];
#endif
#else   // !defined(LIBINT2_STATICS_INITIALIZATION)
template<> double libint2::FmEval_Chebyshev7<double>::cheb_table[cheb_table_nintervals][(cheb_table_mmax+1)*(interpolation_order+1)]=
#endif  // !defined(LIBINT2_STATICS_INITIALIZATION)
#if defined(LIBINT2_STATICS_INITIALIZATION) || LIBINT2_CONSTEXPR_STATICS
{
{
// [0,0.14286]: m=0 maxabs=7.297617753308889496258e-18 maxrel=7.478980514002315388053e-18
0.9766921224504732990438633071118362588895,
-0.04562934274645054017356506017838234492795,
0.001939529855103683169128257025295091276273,
-0.00006566955542969862727950927413983990490283,
1.818857587867905565500200711878183916361e-6,
-4.243327998671117373380960815115153532947e-8,
8.537417592293087334825086125469820771628e-10,
-1.508276918584763772580805119911679784338e-11,
// [0,0.14286]: m=1 maxabs=7.297617753308889496258e-18 maxrel=2.0722766605258094264564e-17
0.319405399225153775456283591023291092481,
-0.02715341797143857969800930144391348933798,
0.001379060664024224008010229743617960221257,
-0.00005092801297969231295847420056437053318377,
1.485164792163734714719768621024369418151e-6,
-3.585216770778211289186290118526474004134e-8,
7.390792778520443447168787470469963866672e-10,
-1.329653037894695700592364221950776502797e-11,
// [0,0.14286]: m=2 maxabs=7.297617753308889496258e-18 maxrel=3.1670458294190631015376e-17
0.190073925800070052678926111261289220845,
-0.01930684929632752658653425099341660766782,
0.001069488272574038459904062218943916029337,
-0.00004158461464496939791753244408918401188164,
1.254825863107184145291243565147078919311e-6,
-3.103687155112712862726645726318253592317e-8,
6.515513172172901361891330628375095624461e-10,
-1.18883627663911729941724020163795972694e-11,
// [0,0.14286]: m=3 maxabs=7.297617753308889496258e-18 maxrel=4.0778745159626759141959e-17
0.1351479450742926813537616365727581761942,
-0.01497283581602604080670875606375251064214,
0.0008732769075448135484189655079021303182164,
-0.00003513512458690992993301635327527180863936,
1.086290498206869481535421145719479585072e-6,
-2.736112417652293234082669772881308388939e-8,
5.825492398476542238318110362868802255651e-10,
-1.074976689292430595116593287874376234897e-11,
// [0,0.14286]: m=4 maxabs=7.297617753308889496258e-18 maxrel=4.8453586635483669774831e-17
0.1048098507121822812769943917693614629993,
-0.01222587670561780965320452881125083734356,
0.0007378376163255280475111118254171014309852,
-0.00003041613433299656772147475567209224164849,
9.576393405847002991223702796042010637093e-7,
-2.446338929235875026090714999870839443226e-8,
5.267564773165162457207262772668043220916e-10,
-9.81012276024828654009305532375040607286e-12,
// [0,0.14286]: m=5 maxabs=7.297617753308889496258e-18 maxrel=5.5001464048942564670597e-17
0.08558113693932466352764413216585287483266,
-0.01032972662854858277766027758916339218978,
0.0006387388209933162237172175359710377820026,
-0.00002681390188877011986879447363345754420946,
8.562186200551863657746845607029912776818e-7,
-2.212038900230846530217797340307656102915e-8,
4.807125828692685901133330044410986617216e-10,
-9.021490008104035744038284010170312823173e-12,
// [0,0.14286]: m=6 maxabs=7.297617753308889496258e-18 maxrel=6.0650673509140293217002e-17
0.07230808639984007567898311600168201548165,
-0.008942343493898272808294304517600119456532,
0.0005630919396645339244124441903037323522861,
-0.00002397412168772098470351264473326636505008,
7.7421361026201932580183098496680152658e-7,
-2.018679717512494718702432404896340883814e-8,
4.420684305147781618346604551246398781613e-10,
-8.350182027491879451778568764091250496859e-12,
// [0,0.14286]: m=7 maxabs=7.297617753308889496258e-18 maxrel=6.5572962479887997230058e-17
0.06259640445728790613728548294359938147268,
-0.007883287155295885399839375137464523369999,
0.0005034565554424786748748488586614077492194,
-0.00002167798139092086375079818239962547434521,
7.065378966227445918411995928791072203254e-7,
-1.856395965519384501727107629234980135508e-8,
4.091733405881009352927940052250331763391e-10,
-7.771836051887704188947557714067554682182e-12,
// [0,0.14286]: m=8 maxabs=7.297617753308889496258e-18 maxrel=6.9899443903376558111729e-17
0.05518301008707119449229981457718273007999,
-0.007048391776187603538220716736795303890095,
0.0004552376092096555716853661840331791870261,
-0.00001978306138935574318599131437660393664853,
6.497385836993325350811031263475987755256e-7,
-1.718255466735326429132766547039184976779e-8,
3.808335104170289986627027144098923015768e-10,
-7.268396656191740656408778528575251572674e-12,
// [0,0.14286]: m=9 maxabs=7.297617753308889496258e-18 maxrel=7.3731725584369483181547e-17
0.04933874243331322165060445510332508643419,
-0.006373326528928511919960900033946225669923,
0.0004154442891767698347560688442715045448391,
-0.0000181926806102270146265538937084483072027,
6.013894093676471297250678536501343018779e-7,
-1.599244762366717743378630035536388255619e-8,
3.561642032747418803540293933937646572152e-10,
-6.826199010225702312213221419681658368542e-12,
// [0,0.14286]: m=10 maxabs=7.297617753308889496258e-18 maxrel=7.7149675160983773699922e-17
0.04461328570249958049185290554551061792291,
-0.00581622004846849390899221375452413982585,
0.0003820462928150503044836594327679608051708,
-0.00001683890371364746028043652310911722465072,
5.597356630550416886178955781567574769602e-7,
-1.495648353115244324502750565537649032724e-8,
3.344958261167897422356186828578279672606e-10,
-6.434710967522695487105604265013904327126e-12,
// [0,0.14286]: m=11 maxabs=7.297617753308889496258e-18 maxrel=8.0216895943502318940455e-17
0.04071354033927945456674483329769601647104,
-0.005348648099404761325831786659095477222327,
0.0003536169779868651025638986124751371019118,
-0.00001567259880326077233341591634007698382925,
5.234769200111685617124718062996173204791e-7,
-1.404654257508074775964534601804387549642e-8,
3.153122907669408623697355206536624781445e-10,
-6.085684133642764835504657996066299166524e-12,
// [0,0.14286]: m=12 maxabs=7.297617753308889496258e-18 maxrel=8.298464458731385708569e-17
0.03744053669583332662145288304583440743487,
-0.0049506376918104742723101163546955375446,
0.0003291245748687315198337848252886471118998,
-0.00001465735398580129132898197567925937665898,
4.91628986723804291985301178773087351094e-7,
-1.324095150894344058916146852970381354566e-8,
2.982094154415195685342919417852083559778e-10,
-5.7725667734490790763436106687882845557e-12,
// [0,0.14286]: m=13 maxabs=7.297617753308889496258e-18 maxrel=8.549467663933769096701e-17
0.03465446384267331737086712599892225029297,
-0.004607744048156879966491312218218460771136,
0.0003078044337020705083533741453258849577301,
-0.00001376561184272090526004382887173158713013,
4.634332995678041652332335303650443642882e-7,
-1.252273667404490088538245550672125550101e-8,
2.82866156613049624274765070907245499016e-10,
-5.490088953612890401171798584138202069261e-12,
// [0,0.14286]: m=14 maxabs=7.297617753308889496258e-18 maxrel=8.7781343904707952989812e-17
0.03225420833709815734314282260222243304638,
-0.004309262071823875923630352596388841717765,
0.0002890778486973715521495979566748131466721,
-0.00001297613259234810442892936445646701728842,
4.382957804910057791774570704616596361393e-7,
-1.187841584983229300785663377210171840242e-8,
2.690242805585634374355618857242122844432e-10,
-5.233963723887964184103493582242166766594e-12,
// [0,0.14286]: m=15 maxabs=7.297617753308889496258e-18 maxrel=8.987316189389649041898e-17
0.03016483450276712914648014672916228962433,
-0.004047089881758318360408568673914337835218,
0.0002724987844395328116995438224401647333663,
-0.00001227228204908472845319842389270473263102,
4.157445517758715148187776167088451192521e-7,
-1.129714454103297754811029123280423062832e-8,
2.564737209187751018882432989804537421595e-10,
-5.00066816073173664100558488776474472743e-12,
// [0,0.14286]: m=16 maxabs=7.297617753308889496258e-18 maxrel=9.1793996439627416567762e-17
0.02832962917230822629883212743466784626609,
-0.003814982982148784376810355257401892654795,
0.0002577179230309928053352640132890308771797,
-0.00001164084763672559099603580564455949163963,
3.954000560893741526893117409580757290639e-7,
-1.077010105623890063421508151313466090641e-8,
2.450418495915794976237269381264351271201e-10,
-4.787280435870649837175586629517266882484e-12,
// [0,0.14286]: m=17 maxabs=7.297617753308889496258e-18 maxrel=9.3563972810403843310743e-17
0.02670488087504148850108713740081702738451,
-0.003608050922429415615853037410126147906894,
0.0002444578003714425242127692990145341855685,
-0.00001107120174985047449125949627188169624016,
3.769535342335087554253617989522636956647e-7,
-1.029003593153802467006582951709921714299e-8,
2.345854929053313233007555109757312938091e-10,
-4.591356870707562480581323101648644261891e-12,
// [0,0.14286]: m=18 maxabs=7.297617753308889496258e-18 maxrel=9.5200179963746799227437e-17
0.02525635645700590725521314683454501600643,
-0.003422409205195887964907157298651159082061,
0.0002324952367470573181591941107771382131447,
-0.00001055469913083479265841460251486276147748,
3.601512549724377838412845101266769151287e-7,
-9.850936645026079164355654021127186723178e-9,
2.249849071406793817620653725867103206438e-10,
-4.410837978636374758064694809035892240664e-12,
// [0,0.14286]: m=19 maxabs=7.297617753308889496258e-18 maxrel=9.67172217164244212257e-17
0.02395686443637121377352722655918507944522,
-0.003254933314454658028494891617675251024422,
0.0002216486817477208058502236420069906570981,
-0.00001008423530500681621127912971300729321273,
3.447827800404375699601770109840753140585e-7,
-9.447774615964110804126815518557145853411e-9,
2.161391744913621799335792711866077504896e-10,
-4.243975823470225697424287645208998101215e-12,
// [0,0.14286]: m=20 maxabs=7.297617753308889496258e-18 maxrel=9.8127652219365967712285e-17
0.02278453320118260428830616564549892976047,
-0.003103081544464097926204441064996279068398,
0.0002117689414053266126059243921513472201951,
-9.653918000867956872500990303256243815014e-6,
3.306721091124403401939566143593794467278e-7,
-9.076311856417397230192267363324867870152e-9,
2.079626435383235292523914760634604887827e-10,
-4.089277257943477393123269000674865986898e-12,
// [0,0.14286]: m=21 maxabs=7.297617753308889496258e-18 maxrel=9.9442323066363910304297e-17
0.02172157081124868363720447249500168796207,
-0.002964765179670719665570228792590358434535,
0.0002027322780184043330397382757822056806681,
-9.258819209266194468710562948524092594488e-6,
3.17670912611418309307347937446017843168e-7,
-8.732951487964090995535010244737921502963e-9,
2.003821478655518285681454409164787695852e-10,
-3.945459134806472454178874758341484200067e-12,
// [0,0.14286]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.0067066224333129230397e-16
0.02075335625769503587342963426046031083362,
-0.002838251892253938653984833239273400342559,
0.0001944352033947614987242960298019416008901,
-8.894785702001438511921301459205829070267e-6,
3.056532997932039730070528810305788476693e-7,
-8.414620936933802985802247805737224163685e-9,
1.933348113469101475767831342038238776573e-10,
-3.811412644493307352917360400131906324588e-12,
// [0,0.14286]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.01820900019833259726547e-16
0.0198677632457775688491368702880194393875,
-0.002722092847523061273982909095814876561306,
0.0001867904997421961703233589271831424367064,
-8.558292538199377883377608365691786896278e-6,
2.945117305798548939734524363034797605573e-7,
-8.118679767192195790271333492650289108795e-9,
1.867663006458848435025749782525184224976e-10,
-3.686174681086306665628272278284744030175e-12,
// [0,0.14286]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.0290025318322965141101e-16
0.01905464993266142724242572683701994417316,
-0.002615066996387261195715244345789166896391,
0.0001797241433023477800744161256080426878948,
-8.24632859564478976151519787825402101442e-6,
2.841537897071261793119687993967523413431e-7,
-7.842846293819240938864719953914955912996e-9,
1.806294221103744438325884313350149807522e-10,
-3.568904672068584148388231957408154221924e-12,
// [0,0.14286]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.03915076302954144156339e-16
0.01830546897471082674466598310768209366348,
-0.002516138006229491190147474969703322669413,
0.0001731729005086966185552022179834920998218,
-7.956306246910031291567102557684163939002e-6,
2.744996182032190925391399603463630676139e-7,
-7.585138659772294424537266964021169818988e-9,
1.748829863998172803513683264556936602756e-10,
-3.458865693083727091410004034081625461123e-12,
// [0,0.14286]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.04870986703882078978514e-16
0.01761296604360643675289485322439242825712,
-0.00242442060711847595892496878152593937486,
0.0001670824311852621591503117565770031245814,
-7.685989440759395371528920596823531735917e-6,
2.654798510719966317068756843921523817516e-7,
-7.343827156036840289214077018327545315321e-9,
1.69490883083024491400879742294300955738e-10,
-3.355408970864357926640714487439607886866e-12,
// [0,0.14286]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.05772968326302645716397e-16
0.01697094424982933017887632271684459504273,
-0.002339154036590488689402761528467588038843,
0.0001614057782560945290621033932254590068146,
-7.433435957278646942886122222019044003436e-6,
2.570339484982661464737359270406766897119e-7,
-7.117395359534507561552829959337847841533e-9,
1.644213212605865032851483790977238065068e-10,
-3.257961085774600845511622767239545830882e-12,
// [0,0.14286]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.06625458517520824100018e-16
0.01637407825613341933431500298472400263968,
-0.00225968089558223165917989257812429011434,
0.0001561021551029947710030020597188809982185,
-7.196950681622520998610969264664694109054e-6,
2.491088356745654372235740843152384699317e-7,
-6.904508243250752960646105054013515286826e-9,
1.596462024718337153045869377612473707379e-10,
-3.166013340813559622658430641417293155251e-12,
// [0,0.14286]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.07432420938634321384878e-16
0.01581776626907562016260009352024162708384,
-0.002185430171438919908464023860729899669468,
0.0001511359643142123010166412373250799388075,
-6.975047519164386460509005600943583476697e-6,
2.416577866556362736138472244093457570381e-7,
-6.703985841657445841148767448549269953009e-9,
1.551405997612103486324608740374277949847e-10,
-3.079112880955790197145363732052965704356e-12,
// [0,0.14286]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.08197407109441878664183e-16
0.01529801120007243794536076308493744400641,
-0.002115903500396045656850078799978341383381,
0.0001464759979025878496094742512932400145685,
-6.766418143421213390617610125750761362199e-6,
2.346395026482187698649393776991576218637e-7,
-6.514781374297139439978113727628890753564e-9,
1.508823225134705103172941097458329226909e-10,
-2.996855235607175547133230561238088070739e-12,
// [0,0.14286]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.0892360860405178317043e-16
0.01481132450277231821991920703926080354864,
-0.002050663970633379485308759817928267681195,
0.0001420947810119777729420007083239971130169,
-6.56990618816756965917073469314008404181e-6,
2.280173463365041869787156002981185494751e-7,
-6.335962971221382143744164372238607155434e-9,
1.468515510234285753020947082026326883138e-10,
-2.918878025045912721588655785626389622859e-12,
// [0,0.14286]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.09613901512591746212981e-16
0.0143546477944336550532124711947545177489,
-0.001989326934164910698074720244529972784951,
0.0001379680299516479826446985799003293361797,
-6.38448580854809096514095382500649114673e-6,
2.217587022724786448046252443329892639099e-7,
-6.16669832692128231608537890166943979947e-9,
1.430305281025508762857764416380356253849e-10,
-2.844855624274066283659874469006481240738e-12,
// [0,0.14286]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.10270884473258061737601e-16
0.01392528853915437357501153777892621440405,
-0.001931552419320362344507484967644935840985,
0.0001340742019796358160686770451951873193886,
-6.209243772006927257937875117898866888636e-6,
2.15834439763495300738668313942578966763e-7,
-6.0062417495085226118424302232181987184e-9,
1.394032975999706341225134886586126429036e-10,
-2.774494618566230801779650446871657132757e-12,
// [0,0.14286]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.10896911333589195781909e-16
0.01352086693524253513094938643079050703605,
-0.001877038827712257406395920835710813374331,
0.0001303941192122684109861636799798982130206,
-6.043364419139611926640584603935753160839e-6,
2.102184595936119197839051545558627960496e-7,
-5.853923180055411837568485879525947462065e-9,
1.359554817177514022704437642751369532081e-10,
-2.707529916989854067053377864778508469359e-12,
// [0,0.14286]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.11494119305205086118067e-16
0.01313927179398580059365368019892513297246,
-0.001825517668969176047023822194828015627205,
0.0001269106528020519584306400045340594025736,
-5.886116971890383410132167118658338181583e-6,
2.048873097004944729503691599614657151944e-7,
-5.709138841087330761975401812429210999488e-9,
1.326740905676763919389744853615786051802e-10,
-2.643721415377101756240269142874207850906e-12,
// [0,0.14286]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.12064453320940858897182e-16
0.0127786236827842311062070693813542828192,
-0.001776749139226205153740467740887222239614,
0.0001236084564098154563939781978101580592954,
-5.736844772505373605093247943080951186955e-6,
1.998198578726542831342532391885866654792e-7,
-5.571343239046422285939581232956354137549e-9,
1.295473586519337885000023091734122428122e-10,
-2.582851120211953342928172087003194314589e-12,
// [0,0.14286]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.12609687178689666732069e-16
0.01243724397458343488014213544747213782451,
-0.001730518389734950892149984636969729971779,
0.0001204737402227276662938931541244399860545,
-5.594956119055150771885801512816784992065e-6,
1.949970118356785199025613827547483418012e-7,
-5.440042297413295235907177628658804323139e-9,
1.265646039293001733490665824092715763333e-10,
-2.524720660833308425938891008872854211552e-12,
// [0,0.14286]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.13131441955759617906486e-16
0.01211362872814465507476785067821393732643,
-0.001686632363115776098764333914190999717062,
0.0001174940785002705139100635146880367616075,
-5.4599164278490890416226179544564003981e-6,
1.904014789114909981180793702509461251077e-7,
-5.314787438296462403085397626229676296545e-9,
1.237161059093619888522289668139899652595e-10,
-2.469149130135704813885503017925798373548e-12,
// [0,0.14286]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.13631202096074671639006e-16
0.01180642654181043154574127159180351828985,
-0.001644917099001427895945221081210773956304,
0.0001146582449849408489562106190842612452258,
-5.331241503894597328462109970093061377061e-6,
1.860175588739835277117413034578035931988e-7,
-5.195170463094758002170329531867866668538e-9,
1.20992999843591836524539460660838659077e-10,
-2.415971204253371224874837715380048286338e-12,
// [0,0.14286]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.01151441969300999414966161027872617499011,
-0.00160521542978686232751717661566523769638,
0.000111956071581894252171370582912389144011,
-5.208491740854732963737773635811985886078e-6,
1.818309647722014169920653908747747779402e-7,
-5.080819109138476499981414950944980015164e-9,
1.183871845870118415119608598602633921596e-10,
-2.365035500062773871216766057390893453808e-12
}
,{
// [0.14286,0.28571]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.9329384172667980434921615752003961875958,
-0.04194022342046958698769446769711362281236,
0.001753022495821579192744478706632895851805,
-0.00005880190016643391682732231117157831460152,
1.618983899776876428434589754743296265734e-6,
-3.761536330758135162547438758622507124978e-8,
7.545333304062723717396367819161434412989e-10,
-1.329951713278505620049195087164518593186e-11,
// [0.14286,0.28571]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.2935815639432871038526319086088606457284,
-0.02454231494149067991968779262799469698178,
0.001234839903495598133712195337469600461104,
-0.00004533154965090734613847796925179001683083,
1.316537709286924239504636763411303408247e-6,
-3.16860111675211641276960967873318238275e-8,
6.516970704998676080560465710905518757683e-10,
-1.170327002395462499298537637751226252599e-11,
// [0.14286,0.28571]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.1717962045904347548668398413248447203275,
-0.01728775864892817039568720487468183902483,
0.0009519625426694930846109865902538033915201,
-0.00003686305626817626377659919647010602085375,
1.109010385012347656449705845193508065605e-6,
-2.736735877276177052550792445226586260899e-8,
5.734789540680289822444432562185847052824e-10,
-1.044854259391278493499958240494788604304e-11,
// [0.14286,0.28571]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.1210143105424971886026097318569674219874,
-0.01332747559736368806473664037142221654869,
0.0007741241816321015925470298960258621126409,
-0.00003105229114895359015827290770685196041228,
9.578575517126030207439454269886338071354e-7,
-2.408257741583785395872075341412865394192e-8,
5.119956561239982358565681384569836163272e-10,
-9.436450725859069967180330645686659559688e-12,
// [0.14286,0.28571]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.0932923291815458126242565999102236609934,
-0.01083773854284102118684895493553445755825,
0.0006520981141283929695457974025380175175848,
-0.00002682001178400007001558477910896734048163,
8.428902046532939275954339562264484649034e-7,
-2.150059148609608372122629339367137640048e-8,
4.624017688974586738492917305922697025667e-10,
-8.602890834881680517946331224367486025477e-12,
// [0.14286,0.28571]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.0758641697998871447665884218642988343758,
-0.009129373597789782481533097151245311574383,
0.0005632202474643414420500956242428624964976,
-0.00002360092603905854942727312898859612313721,
7.525206974803929171488112161412623160351e-7,
-1.941791012016155357679098883987816718015e-8,
4.215561564424899973103740265207885831713e-10,
-7.904494875053044238262083675548506821991e-12,
// [0.14286,0.28571]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.06390561518452847407678483735486204948563,
-0.007885083464493640790518814047360176666935,
0.0004956194468205457584781724937327937412018,
-0.00002107057981502939201872099140086996334737,
6.796268499893681009620313928279605380158e-7,
-1.770261700227379311082099225411770171451e-8,
3.873337410211293914640661844754285173549e-10,
-7.310878854658818245910714272669598173812e-12,
// [0.14286,0.28571]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.05519558425145548245480466824101425763801,
-0.006938672255480999995586567192757343923034,
0.0004424821761159128013036374187898014083584,
-0.0000190295520653295431702240365579634691337,
6.195915911386518272953832465775664584115e-7,
-1.626546708662145989032361290521897636112e-8,
3.582456748831298517556499984601669192886e-10,
-6.800124894281809784198108861061335637481e-12,
// [0.14286,0.28571]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.04857070578836699707904081807797044977794,
-0.006194750465616572278203159393976483758811,
0.0003996205933721978542621470364483552708758,
-0.0000173485648001620315325349886993319089546,
5.692913443324371342434307721599645284829e-7,
-1.504393484525770298877831660124491037464e-8,
3.332179576489406504743399643051248088968e-10,
-6.356026113368197917633563290535531415812e-12,
// [0.14286,0.28571]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.04336325325931600322432305207856815898254,
-0.005594688307204943564920119608935108355191,
0.0003643198608036640810519593628118241877021,
-0.00001594015787436608922512314515308882235824,
5.265377160984238192986437532736661888161e-7,
-1.399291685271994875246707751042995615163e-8,
3.114564334847419395171650783407733601423e-10,
-5.966341200977565565381173035063897761532e-12,
// [0.14286,0.28571]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.03916281815043460238007926919417200433495,
-0.005100478051245807344824244387726953237454,
0.0003347433153619350137298250230694625988362,
-0.00001474305627034941586693846870495316579752,
4.897520865499878427626014147322450963235e-7,
-1.307906209579147386627272959582917235137e-8,
2.923612635431095107801241491299278291222e-10,
-5.621651986289502406454547591090265015506e-12,
// [0.14286,0.28571]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.03570334635872064897274639842552894924352,
-0.004686406415061900258198336211129154731617,
0.0003096041816775720726747630027597928714681,
-0.00001371305863099887699642265325427401374867,
4.577671702281654825652871863418522781783e-7,
-1.227718010444105848938655790213534941604e-8,
2.754709458647250477689778389004185275684e-10,
-5.314594285546738250064501513674248978182e-12,
// [0.14286,0.28571]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.03280484490543329948657689062784547939658,
-0.004334458543481087894341399344650770128527,
0.0002879742312511992160011901330143750066342,
-0.00001281748096323532720873816486066238429284,
4.297013006847732891447672366312669137839e-7,
-1.156788998673418235875478198071274659661e-8,
2.604246261358843390693288591689391538978e-10,
-5.03932712914226728748726429644970926708e-12,
// [0.14286,0.28571]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.03034120980436761304851046246906556711154,
-0.004031639237512110248901365679494448839101,
0.0002691671002281542129248854371860822552073,
-0.00001203163660632634685957892653440089591153,
4.048761467044668273235538969983059834658e-7,
-1.093603762102037314156289701905192585975e-8,
2.469360892817552751950333753122430961616e-10,
-4.791158435346863721252444451034016193516e-12,
// [0.14286,0.28571]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.02822147466258476962959855092154954083581,
-0.003768339403189699813831167351339117354575,
0.0002526643687330561053871187158388909034208,
-0.00001133653228609337347814720043155735339791,
3.827613140314199474939324347130479540101e-7,
-1.036960340377865691291548548757435058408e-8,
2.347754170883732923819254180051085264649e-10,
-4.566275866645697847994056035187616007563e-12,
// [0.14286,0.28571]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.02637837582232789667476768536139846947207,
-0.003537301162258526232400209943701733908285,
0.0002380671780081549609242206754827106171038,
-0.00001071731696325103037070336439556163335259,
3.629361165440062584435271092797132918208e-7,
-9.858931943575140734797274324277636150099e-9,
2.237557998728756318284211547460553091154e-10,
-4.361549939153730754351671828371134094834e-12,
// [0.14286,0.28571]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.02476110813580968168795319888069281770908,
-0.003332940492110092982230080777043111106035,
0.0002250636562284577682158174030578307570624,
-0.00001016221142629249335801731152255229257007,
3.450626155433828685701807545531208641053e-7,
-9.396178206072276195621362636767309674158e-9,
2.137238886261323368577985063349702820361e-10,
-4.174387730419170957776376758797799056561e-12,
// [0.14286,0.28571]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.02333058344477064901338994280412319161895,
-0.003150891187194500020622277297224162844635,
0.0002134064399523211346586727667428937567827,
-9.661753391565529406377183270162674441613e-6,
3.288662348288667783687252330874068236845e-7,
-8.974902345358303307159761901291779446625e-9,
2.045526265281758667056012180671093299206e-10,
-4.002622641937830040321599422142325475083e-12,
// [0.14286,0.28571]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.02205623831036149835293785820731511821821,
-0.00298769015932874163275731367367252439135,
0.0002028968212230480945964723170436013263501,
-9.20825472537974898504719663072627232308e-6,
3.141215797945057022363610394504899721181e-7,
-8.589768659147088907452598384426354966545e-9,
1.961358471824910176416009488015355184107e-10,
-3.844430260794553541804362292619170911701e-12,
// [0.14286,0.28571]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.02091383111530118970350028761328663387122,
-0.002840555497119061810361034498074841634717,
0.0001933733492331404064681401922224001543859,
-8.795404378708051087763788806008588841099e-6,
3.006419008611038566092685401335482317822e-7,
-8.236318738983097414751370756445185379961e-9,
1.883841517358239790913458269364238496537e-10,
-3.698263386690196991146939503709899985421e-12,
// [0.14286,0.28571]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.0198838884798334310077075349525051992184,
-0.002707226889260486461305258221969868172187,
0.0001847034919530288964003006347618282403037,
-8.417973363281364319967556059707386767068e-6,
2.88271153733420722505931417309600752041e-7,
-7.910798328290863744827033854850089683859e-9,
1.812217251231883448053541531209807517322e-10,
-3.562801318477911219080707201560311334305e-12,
// [0.14286,0.28571]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.01895058822482340362113677312732288259675,
-0.00258584888733904825894031424882179013769,
0.0001767774406290630195604070235976218428545,
-8.071592438788650717096563648794871310393e-6,
2.768779394319223142331857533397943547519e-7,
-7.610023619656025075741424485792535003851e-9,
1.745838510450498774306521308671197448714e-10,
-3.436909878631180066962186477321859068961e-12,
// [0.14286,0.28571]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.01810094221137333625765216035013569778053,
-0.002474884168803640532092398794032555602554,
0.0001695034412147109391546542641183452995854,
-7.752582433764699118387080379689263224233e-6,
2.663508246976331078176958952788944757648e-7,
-7.331276895791771809684870405718381309712e-9,
1.684149531156469526319491334708126647895e-10,
-3.319609613970790294311909155140547951505e-12,
// [0.14286,0.28571]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.01732418918162548221939632159946801519972,
-0.002373048177002818395945516701886161668629,
0.0001628042311092031863187086447202590727605,
-7.45782321692498270971155455258566240933e-6,
2.565946894259746125075657368375991390461e-7,
-7.072224267334299020453233970767182297251e-9,
1.626670366576909418427879336703894586922e-10,
-3.210050286271363952415527353322752208967e-12,
// [0.14286,0.28571]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.01661133723901972731297207773723605898163,
-0.002279259235525810010652479060495582248097,
0.0001566142875555646677362079966196171823792,
-7.184651425312333389101498551679407833914e-6,
2.475278474896165860896832533555407862006e-7,
-6.830850235954367301324844509765901479083e-9,
1.572984387087593482007707554021973824311e-10,
-3.107490247810234069889780299165658710274e-12,
// [0.14286,0.28571]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.01595481464868066865972610049404194897705,
-0.002192600025774964704396935174845349617746,
0.0001508776799316948266742583825125084468102,
-6.930779847336119670039274288724902242207e-6,
2.390797564473902043709338006430872439262e-7,
-6.605405200924648363927772857230617139353e-9,
1.522728173958632940588383274515745559871e-10,
-3.011279644423379851096876122295594705506e-12,
// [0.14286,0.28571]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.01534820018042475155718864186600717352265,
-0.002112287519040875241340475979423866794203,
0.0001455463767941903784240232558173607177314,
-6.694233294621277417284829108210820668089e-6,
2.311891802741522043333451851742205823028e-7,
-6.39436301801647552813311027186154653161e-9,
1.475583288626372861337177578239353654674e-10,
-2.920846642351481524146540719104925106315e-12,
// [0.14286,0.28571]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.01478601263328612535470782851441321046911,
-0.002037649275115896128890177393849480463171,
0.0001405788991871749553143140558054069711771,
-6.473297158444061880362255124049778763005e-6,
2.238027039221770015727292350803871405539e-7,
-6.196386434703992026112143819073733525165e-9,
1.431269523660749370341461680818259610286e-10,
-2.835686062394261663359424600498623672852e-12,
// [0.14286,0.28571]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.01426354492581127160432628085164597803123,
-0.001968104588617758657103048373936458061712,
0.0001359392403274498990485091804679880324602,
-6.266475817450658412091212770087271583641e-6,
2.16873523553306764936351781674040640518e-7,
-6.010298747765337307124328042427281488297e-9,
1.389539333344628782654428653667193754974e-10,
-2.755349944449920795894510925108533187146e-12,
// [0.14286,0.28571]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.01377673212032430933661545163750743572487,
-0.001903149364581681999095163754043025958279,
0.0001315959921665850854862976776136599024801,
-6.072458764157076865592101702420697635602e-6,
2.103604545549970360834156749689235618559e-7,
-5.835060414669736568310763100715197927415e-9,
1.350173210166966259202182149535680533096e-10,
-2.679439670514971621921873291785860936291e-12,
// [0.14286,0.28571]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.01332204555207177276354225677260191390844,
-0.001842343890329644764441810250480329462777,
0.0001275216340474167067215006422808092825565,
-5.890092829398170710775447425495871563488e-6,
2.04227112938868320234061031584387637308e-7,
-5.669749637327798391987091399988391626234e-9,
1.312975824982094910295113706015885673151e-10,
-2.607599353911842458725247186884811945517e-12,
// [0.14286,0.28571]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.01289640723230751215227238611037718718053,
-0.001785302876661353953712011987440831589629,
0.0001236919494174766723086237438823131453971,
-5.718359261486863772716864355494791492467e-6,
1.984412357719694331432584298169809245668e-7,
-5.513546152851443886832979288272743258553e-9,
1.277772787637501928510787451962315113244e-10,
-2.539510263496793680419123347398152634305e-12,
// [0.14286,0.28571]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.01249712013662947650691393035166490923024,
-0.001731687291842256580254477983734663096261,
0.0001200855444913363705817022030664185814573,
-5.556354698289339377109547173720379055369e-6,
1.929741138533775147721041954202186643515e-7,
-5.365717629962445211015435801510087321743e-9,
1.244407914755908159163653281685414744162e-10,
-2.474886098633636786994461129861665026357e-12,
// [0.14286,0.28571]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.01212181104289579492102067877751226589027,
-0.001681197622876352332589042313587893460446,
0.0001166834486641856405489311598309063066348,
-5.403275282169683627497779439902003178727e-6,
1.878001155884990092286230508782416352249e-7,
-5.225608195185380488254480626462052694294e-9,
1.212740914404178119611379600315539896311e-10,
-2.413468967257875822725021666694100257632e-12,
// [0.14286,0.28571]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.01176838336013446521433362959846428243708,
-0.001633568281296299184288797018706012541824,
0.0001134687809256702805773632872205206899899,
-5.258403328470178550324062123400598958899e-6,
1.828962854058249162490357684309927599523e-7,
-5.092628709746277828305667036167129922926e-9,
1.182645415286207786132966200544525094232e-10,
-2.355025947939527849349244212784924731409e-12,
// [0.14286,0.28571]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.14110329506187512891951e-16
0.01143497796907409320195715160800903287658,
-0.001588562932957138518447183345829854418732,
0.0001104264698979782043206961523309926878293,
-5.121096081180333882567121626531866293494e-6,
1.782420034483849079499354186313210124078e-7,
-4.966248493289186322334022055726311276385e-9,
1.154007282103273408837927064767234363363e-10,
-2.299346139359966155563232859494084153825e-12,
// [0.14286,0.28571]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.14449814586570137777866e-16
0.01111994053069996856562968742257925593731,
-0.001545970578569501313149261503021199262171,
0.0001075430177048891081574705347466571089085,
-4.990776184297503050195857498257563918965e-6,
1.738186959038286763346224861075305789895e-7,
-4.845988249342835621582661448950339527553e-9,
1.126723169754079224036785684239474519726e-10,
-2.246238118450763195234399622123820971486e-12,
// [0.14286,0.28571]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.15011929764542917359271e-16
0.01082179404998650815202356540589766092471,
-0.001505602247866303487752681880698528206887,
0.0001048062998703474066499344245674673327623,
-4.866923571069075049913028453223584729698e-6,
1.696095873957602008016851074388648620861e-7,
-4.73141399378498540920142190905545966406e-9,
1.10069927778495826156069178921695138763e-10,
-2.1955277426549701801119651995265094912e-12,
// [0.14286,0.28571]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.15549776768383406006304e-16
0.0105392157350641233967126021990894437837,
-0.001467288198182767001173826183791211995118,
0.0001022053949925482619834777318211706093223,
-4.749068530949759457027485399948042603231e-6,
1.655994884799910087643654255724439616097e-7,
-4.622131824248303118323801278272467696997e-9,
1.075850273464967915129250569083512590316e-10,
-2.147056243162064349350391472678911820606e-12,
// [0.14286,0.28571]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16064893489770560082031e-16
0.01027101738727936801217579267996677705625,
-0.001430875529893624265660728403288086427636,
0.00009973043915004056906455359578109231247948,
-4.636785759496612888902958916433504724667e-6,
1.617746125737467943934474598937002670536e-7,
-4.517783397658173852972619737357926342997e-9,
1.052098357442258526879036637182669971377e-10,
-2.100678565145885046362554020128066700169e-12,
// [0.14286,0.28571]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01001612870925536888420837161827555951771,
-0.00139622614809855993957927901377678444014,
0.00009737250094952251119407663417430766313125,
-4.529689232386767717092201250488121692302e-6,
1.581224176694915528682435340887589567696e-7,
-4.418042006532932808613277790754779152315e-9,
1.029372450435477402614666319381193781643e-10,
-2.056261918467725541614989285658792013471e-12
}
,{
// [0.28571,0.42857]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.8926939965518722076563701458987119062499,
-0.03860429183534837992582427138194827622138,
0.001585965588103371037213285167629505733256,
-0.00005268748353413898163528028499510635853604,
1.441772347568960792835318636659862436723e-6,
-3.335671080991053733799276205798909115956e-8,
6.67044611785091649690849854653871210825e-10,
-1.172979940260941865889345238364144888078e-11,
// [0.28571,0.42857]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.270230042847438655031815101873106822251,
-0.02220351823343713330156349111626912434909,
0.001106437154217345716035647670183935611291,
-0.00004036962613438288952229508402657144040259,
1.167484872652163269832503680086782000049e-6,
-2.801201013534872453872482195958471604804e-8,
5.747783938202968565506049692233554195245e-10,
-1.030286480394406856804664019586421636256e-11,
// [0.28571,0.42857]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.1554246276340599290978788148080023375669,
-0.01549012015903387089840165904383006603168,
0.0008477621488224259361354020877276155484548,
-0.00003268957679302853926644380496249456300841,
9.804203496004415234937328578622428318883e-7,
-2.413724834924991679267087841884012856204e-8,
5.048568130694031281771730877254689070278e-10,
-9.184545516498479082018943082241291653556e-12,
// [0.28571,0.42857]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.1084308411132370926340088696762045762119,
-0.0118686700835058727350986316095911014453,
0.000686481112653950187384563371333080235883,
-0.00002745177011242987863757560023054232934767,
8.448036875455627575489059083275603927146e-7,
-2.120087940358318490079199530659224542503e-8,
4.500577005277370369747653061946517319851e-10,
-8.284686938756308189392543222973754488614e-12,
// [0.28571,0.42857]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.08308069058454110579053378311934937065134,
-0.009610735577147934512863573454031351163254,
0.0005764871723613495480305796724723096532222,
-0.00002365450354600265867911636820808698283166,
7.420307748307767510255931508786600328419e-7,
-1.889959402803457677053229739975520670703e-8,
4.059634028575438916975827934733255227921e-10,
-7.545081255001516760433860143897023224948e-12,
// [0.28571,0.42857]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.06727514904003553848919628330261254629724,
-0.008070820413052129650443009274138851914645,
0.0004967445744663535152382064325603288165148,
-0.00002077686196582493427079242940587580422041,
6.614857870120918412472542673843373488734e-7,
-1.704786549868106814014213723433773246292e-8,
3.697216826993570211169991779678003660658e-10,
-6.926485004934059453620128169646659180348e-12,
// [0.28571,0.42857]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.05649574289136490467077588001943786594265,
-0.006954424042522697877816165261812274601338,
0.0004363141012826003243103954854530200525912,
-0.00001852160228639414631300522399154369996644,
5.966752887644311533882232110988210034219e-7,
-1.552591012606957504601168089201666970269e-8,
3.394095713657058613788668662413766246421e-10,
-6.401485734438999962650539478533221353894e-12,
// [0.28571,0.42857]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.04868096829765888245217552035775847839754,
-0.006108397417950593749807886340385597842114,
0.0003889536480145355575103865582610952018687,
-0.0000167069083178377147194933169103817140723,
5.434068509659587027640558460203710217995e-7,
-1.42529706214584112755684136865389857138e-8,
3.13683829734666840143193726716598962649e-10,
-5.950360372828675202871128365368233820168e-12,
// [0.28571,0.42857]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.04275878192565415372248211997194110608402,
-0.005445351072198069627641931569172805856751,
0.000350845074674834522976802294975101390709,
-0.00001521539204417584711872419006790484573752,
4.988539685175161512807762116848548449384e-7,
-1.317263639824176163178065513494782637e-8,
2.91578005580340301430800955251105012584e-10,
-5.558558173710439657967896942598935376819e-12,
// [0.28571,0.42857]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.03811745750538648501434233625173001636869,
-0.004911831045442590536298376279303594676907,
0.0003195232329279211891930341295669561303578,
-0.00001396791132220366592149299240276607463161,
4.610422708931228372845259147940611326001e-7,
-1.224432057601172608272344117306990905088e-8,
2.723790956162778757166458894780381719025e-10,
-5.215110892991753971648122194493950848951e-12,
// [0.28571,0.42857]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.03438281731809813150580202000760015738905,
-0.004473325260986100260541087464814490100087,
0.0002933261377664928209935984474410414132252,
-0.00001290918377686467072779659384984244442653,
4.28551217282579497759530066904190819252e-7,
-1.143808017555237723678904500902309685917e-8,
2.555496428347639844049070635527426094489e-10,
-4.911594999658389233156694964060156481315e-12,
// [0.28571,0.42857]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.03131327682690269969273103128315808926994,
-0.004106565928726366929683854411108053389282,
0.0002710928593143626677895711733813248046545,
-0.00001199943426521641533169194549774608193737,
4.003328034165578660906826140549408261965e-7,
-1.073134446847170153040445193094256654309e-8,
2.406768838826711175021278607446677402684e-10,
-4.641434655197230281011795410659716777714e-12,
// [0.28571,0.42857]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02874596150108456648234979215203191293983,
-0.003795300030396781121443499330790893049548,
0.0002519881195697391649612794169855206173635,
-0.00001120931866751426848363602878148037926425,
3.755970538039280718011597527587932909571e-7,
-1.010677934696868233202073632355514747528e-8,
2.274385943829102447081254701236631181591e-10,
-4.399421762687052757183819442132147625708e-12,
// [0.28571,0.42857]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02656710021277746592031850719262265166128,
-0.003527833673972265011333349887029126137433,
0.0002353956920179848985925866940700558660292,
-0.00001051671766984338518293930193968391459603,
3.537372746737568808582206964465647752605e-7,
-9.550852954015298296547532946743260143795e-9,
2.155795708588169548354773845534777090996e-10,
-4.181378149745768260978453465440270659287e-12,
// [0.28571,0.42857]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02469483571780585323657724697804011987445,
-0.003295539688247898115714994953215824683479,
0.0002208510710668879943779093109315215883697,
-9.904643846485186484912954652508901013474e-6,
3.342798510317874317048521227754277872057e-7,
-9.052848015101912623971077540753536722675e-9,
2.048950773472264088776316144460454403496e-10,
-3.983913111452743641392005941484427180547e-12,
// [0.28571,0.42857]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02306877781773528504677187475550812252259,
-0.003091914994932716909421015435064539897046,
0.000207997520776358187463197669195499067534,
-9.359835977491875091388643529909302096253e-6,
3.168496782716093142878304762049713154401e-7,
-8.60416666239004258089142168026930552159e-9,
1.952189647410107140233294907943281537383e-10,
-3.804246333557817275201056711423025145223e-12,
// [0.28571,0.42857]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02164340496452901667566104827656426071641,
-0.002911965290865459931819959624725544557109,
0.0001965565555274916452497717889663691571372,
-8.871791133794065884819461744459667767863e-6,
3.011458310200667978225114129348562470262e-7,
-8.197831496363231472225749141317546852447e-9,
1.864149938301780186918397325269279754201e-10,
-3.640076520652617045815573455954501590427e-12,
// [0.28571,0.42857]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02038375703605821789960754917519524689465,
-0.00275179177738147540260862476469706603505,
0.0001863076138098312051062471190402774980449,
-8.43208340486835473980159716999594402883e-6,
2.869241002950860724942117193595508739293e-7,
-7.828121190624387347047781116378202559682e-9,
1.78370397932754271337382405105580814003e-10,
-3.489482539424755876986274876754319198605e-12,
// [0.28571,0.42857]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01926254244167032625715240767330241835094,
-0.002608306593334364623367046971776011796558,
0.0001770737515023853167077189610576024258582,
-8.033874939153537752009462712690521691552e-6,
2.7398423967361814117817573937944120324e-7,
-7.490300150650900809145053497995595125335e-9,
1.709910387989201875317093128615777503615e-10,
-3.350848063553252810365013275016760995926e-12,
// [0.28571,0.42857]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01825814615334055085992178260582276978639,
-0.002479032521030247227065216612625327714986,
0.0001687113737223686437361390443686641266203,
-7.671558836750741867189403224682968206137e-6,
2.621605033480960396857377728719185308475e-7,
-7.180415083563742120542363388626428236286e-9,
1.641977141209767469780620911719931228931e-10,
-3.222803452621334193486343284089488865902e-12,
// [0.28571,0.42857]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01735322764721172913919125399095341953081,
-0.002361959232110129646120799078023069789287,
0.0001611027355719048054298228833686984119774,
-7.340494215002456939114186375017722781635e-6,
2.513145260683753602431682183214853745766e-7,
-6.895139930539128211413811391284319936815e-9,
1.57923309529272809462536334603881488053e-10,
-3.104180437006110574722696322262707419158e-12,
// [0.28571,0.42857]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.016533714624770906122303604812009960196,
-0.002255438298003743528938779435626998722249,
0.0001541503785151860484817983727977860235029,
-7.036806846865476516517264004979991002632e-6,
2.413298957761601794877015373828152648397e-7,
-6.631656264018695462766871856533258713985e-9,
1.521105780955347210859768775265630944717e-10,
-2.993976434536775248827339077928663768059e-12,
// [0.28571,0.42857]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01578806808602620334845734295125219681535,
-0.002158105299209781174180929637545010544298,
0.0001477729437843050025085222586074427858458,
-6.757237194673716116391813502620866632418e-6,
2.32107967507373197933938574786060969418e-7,
-6.387560037483410364674598721413750754027e-9,
1.467103918040233423766048943821967026956e-10,
-2.8913261928953467554015894136082600351e-12,
// [0.28571,0.42857]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01510673709446846690860186521683438902143,
-0.002068821212977540129522783684368228638088,
0.0001419019810882738629353736064241371599387,
-6.499023199403689325477310194617755753467e-6,
2.23564599634253996857050008056315423192e-7,
-6.160788155765116049853205715279655185423e-9,
1.416803519927420701417374820331300938484e-10,
-2.795479061675813472348945039996439870363e-12,
// [0.28571,0.42857]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01448174849084277963674352081570475324921,
-0.001986627735233191769980045190010154424598,
0.0001364794871875993884330859319565910461934,
-6.259808895452541808285000548333040991106e-6,
2.156275838262724308016297410470239288823e-7,
-5.949560120462272733526315527898071158674e-9,
1.369836756546906084542526851459815796659e-10,
-2.705780632316909687615476859582395455908e-12,
// [0.28571,0.42857]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01390639414663234115823639431402254062153,
-0.001910712820623831276609128192604418514766,
0.0001314559868046216145169992521706702743515,
-6.037572449543042673794355443763015666083e-6,
2.082346026396871334334881558803566002763e-7,
-5.752331260195427225833990609637510000506e-9,
1.325882957702814966895245783169261922186e-10,
-2.621657797224743071585363063546825282364e-12,
// [0.28571,0.42857]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01337498974436681774069098139322625565939,
-0.001840383815262219638851615941544792956775,
0.0001267890214405186717782360609409246473035,
-5.830568973230744438855020020897471415358e-6,
2.013315925764932098162794729102322357254e-7,
-5.567754949159452183592201735898296731233e-9,
1.284661291843189281262199950044116824304e-10,
-2.54260650774636638703673990843201825262e-12,
// [0.28571,0.42857]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01288268670683553631038945718759518750832,
-0.001775046300164851119629156169960521134782,
0.0001224419484379571447256851807971928551421,
-5.637284688554124071960174829317815012007e-6,
1.94871421733755802996990771846232099138e-7,
-5.394651861742337553871160585833645517639e-9,
1.245924767300868602636531614898885436531e-10,
-2.468181678987269848941184971035191925326e-12,
// [0.28571,0.42857]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0124253241011539567079533278064767503482,
-0.001714187278129058287309826641849636256205,
0.000118382978459745033380317984762367803129,
-5.456399902215595711704272722267171111745e-6,
1.888128137152721268666696254889464376048e-7,
-5.231984780877506727280698348456468305089e-9,
1.209455285515426441654509210072917444299e-10,
-2.39798881480897586000543034205049614177e-12,
// [0.28571,0.42857]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01199931094690340691211082799718138156021,
-0.001657361698434153484851967650772334608022,
0.0001145843979466330200902717787424274741461,
-5.286758875107774648753291693739538311314e-6,
1.831194659239061652556687383462165260897e-7,
-5.078837824196167271800706588999969586897e-9,
1.175059537165712104281528531627785049908e-10,
-2.331677020556897625048301730165014229255e-12,
// [0.28571,0.42857]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01160153188903907332370614165777964794306,
-0.001604181571250646570938455049205506538562,
0.0001110219363773660129224940448173610949215,
-5.127345134498619974663128789678596418782e-6,
1.77759322476924010706272967977923488199e-7,
-4.934399209981685265781034486254071562266e-9,
1.142565578308482415093317771719787370822e-10,
-2.268933142511180258076252949845690289033e-12,
// [0.28571,0.42857]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01122927099875452495364133983411067396144,
-0.001554307109280966532284067568093336716524,
0.0001076742478245711410933559068559262722042,
-4.977261115660631092217178340770936981276e-6,
1.727039710143995604383423602327551209064e-7,
-4.797946878804410205570772262886948185059e-9,
1.111819958626295779110140224933576922613e-10,
-2.209476827678765029471430083589047707618e-12,
// [0.28571,0.42857]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01088014976496676470902967626966671675179,
-0.001507439469541893423750100445997793836854,
0.0001045224834289708815202065191114515356208,
-4.835711272506043760281354260425549932516e-6,
1.679281394564355952520213616546726797943e-7,
-4.668836433731882564684141292185090244858e-9,
1.082685300654995200977360349254963291871e-10,
-2.153056339639023463828366512176559606446e-12,
// [0.28571,0.42857]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01055207628679325297399935602283055789957,
-0.001463314768003542143542353427230271052274,
0.0001015499367227221756218994631341378602794,
-4.701987986788881253006639912228481906366e-6,
1.634092739105229298243417543639494487872e-7,
-4.546490974419149648527042378110180201855e-9,
1.055038249487725664037176354941442255573e-10,
-2.099444998832301595222225975148634529159e-12,
// [0.28571,0.42857]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01024320337602479403608071520976287773649,
-0.001421699114116110071388507990403384536671,
0.00009874174772265950354072519880012617266161,
-4.57545974951089176695652024693036581738e-6,
1.591271828647030852434554029806294017055e-7,
-4.430392487006135461752219302673852021112e-9,
1.02876772846455692435781211285672625427e-10,
-2.048438141224306431511100152634989125575e-12,
// [0.28571,0.42857]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.009951893798812769553448949836843019250835,
-0.001382384468115280110192922240002962681519,
0.00009608465473981956958761635771217139776732,
-4.455561198330001341690015898190191001441e-6,
1.550637358339776273134497889352560280888e-7,
-4.320074518992400863177783256083704879914e-9,
1.003773448873458484449136872251378522e-10,
-1.999850509378480487291568804019110530189e-12,
// [0.28571,0.42857]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.009676691276806959846508620544780883751024,
-0.001345185166355566284463389306505652782185,
0.00009356678516501881348410276583613613435705,
-4.341784679659687576272303574407733221675e-6,
1.512026069809259758289014468571746538628e-7,
-4.215115920823700463965717441277315013471e-9,
9.799646315372397700327358606770619273035e-11,
-1.953514005882691063445192716006083912581e-12,
// [0.28571,0.42857]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.009416296164488963086881777059083891842078,
-0.001309934992308398899594568380810079814882,
0.00009117747827294025833073176114249245727749,
-4.233673070046202445965746721315612101786e-6,
1.475290560712359400500231280687941749253e-7,
-4.11513547728522776983905676974645275179e-9,
9.57258905959218573390769215239995856903e-11,
-1.909275751751909445083283354845630261619e-12,
// [0.28571,0.42857]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.009169544946158791412392685142997507520374,
-0.00127648469581934041507022447455619230955,
0.00008890713447105518968944719110420584156852,
-4.130813642923361520825899139669901250598e-6,
1.440297405724682071179655803645770083725e-7,
-4.019787284544863012820676012895530691893e-9,
9.355813589114638024243906554205247351826e-11,
-1.866996402581183656074197645036186996965e-12,
// [0.28571,0.42857]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.008935392870735382039484124404283716662585,
-0.001244699882592988953047318551169826373253,
0.0000867470865014737291146347074879825810253,
-4.032832807377892409529427211456681160193e-6,
1.406925538505708091115445923291615997738e-7,
-3.928756754773220759220194165490036900544e-9,
9.148637093243778173143093209240528591799e-11,
-1.826548683398850642724698141287092759746e-12,
// [0.28571,0.42857]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.008712899178150921823306561572963274461984,
-0.001214459211018886329039526820866915405285,
0.00008468948895501715142144129843318336723633,
-3.939391577651789881174785920230577147473e-6,
1.375064853315814835531118465139297118378e-7,
-3.841757151159473723531696352457154340258e-9,
8.950435903425551181625845430099795259026e-11,
-1.787816109787225637208815316175660383655e-12
}
,{
// [0.42857,0.57143]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.8556243918921487982296917400819559430323,
-0.03558481888278791129722748745906949235645,
0.001436229967611987017312189145907131372228,
-0.00004724102255439676902800562710636647573242,
1.284595142243078464271678882802072765669e-6,
-2.959131154660376977208853299002300462252e-8,
5.898731915274688785079008446027458456238e-10,
-1.034778340091797590212098711896075536817e-11,
// [0.42857,0.57143]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.2490937321795153751692166836800454126339,
-0.02010721954655895923775115594118308007449,
0.0009920614736427076434423158284793490678638,
-0.00003596866433716920073669217154975178758492,
1.035695899124532945210212005585684885152e-6,
-2.477127214127638560737284897121453477563e-8,
5.070574077922223161211107427370441310722e-10,
-9.071775370902517939769942928577626611263e-12,
// [0.42857,0.57143]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.1407505368259127111405288694788742307844,
-0.01388886063099002164558061796247106169878,
0.0007553419510808914951123694282244225335914,
-0.00002899948549090399153877837330551880767227,
8.669945204342651796144298607878383353185e-7,
-2.129338310699669677289367866687897738471e-8,
4.445314265087667583685166029025875827842e-10,
-8.074752001704793641931887499790737280334e-12,
// [0.42857,0.57143]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.09722202441693014831328340543750579541964,
-0.01057478731512537706673153593945757700786,
0.0006089891953092915787940048574193268194863,
-0.00002427584685631637093163241923954442210599,
7.452684046414525676872116924312007524304e-7,
-1.866759199163313469318119127144809344192e-8,
3.956759790911501863523657997908565394102e-10,
-7.274486635345044599095552466802674312758e-12,
// [0.42857,0.57143]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.07402351120587763652679666410133669892027,
-0.008525848734323619224095843035828422854851,
0.0005097927839829260620805474439535180019024,
-0.00002086751558847800498389394068424453346258,
6.533657159435149329977294899358638895385e-7,
-1.661590934152798376970050755280789978555e-8,
3.564618888191668799211614501003837581952e-10,
-6.618107282241158174189453752475484896135e-12,
// [0.42857,0.57143]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.05968094114026533185329532915859152085986,
-0.00713709897575503715283729736986828147624,
0.0004382178273582987820273585105356752083117,
-0.00001829424028352908758802991976053643885801,
5.815568234777707387261143005286721230968e-7,
-1.496912305506383933856096822588755335017e-8,
3.24298379120003983538936622539813944539e-10,
-6.070092175235442010447148096973324870255e-12,
// [0.42857,0.57143]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.04995969283028525754752492091921412075389,
-0.006135049583010708729822007333284314848093,
0.0003841790459543529848491003176853066829025,
-0.00001628359127634820700752551586813117215885,
5.239193036986178714286021133864763224966e-7,
-1.361842979297054176470928298046645624625e-8,
2.974448481803300073296077169197311733514e-10,
-5.60570315131945878356672035709117118139e-12,
// [0.42857,0.57143]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0429453470810749587538776928822056982179,
-0.005378506643355856738112909301141190910082,
0.0003419554168035384164454329778889078321516,
-0.00001466974070696505795026600770325019685698,
4.76645039739702540507348247232663260806e-7,
-1.249073093619525743282963519782071154126e-8,
2.746891000869476373223385322579644927313e-10,
-5.207187878487597995795688886350771360905e-12,
// [0.42857,0.57143]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.03764954650349099495855560636064052546452,
-0.004787375835244790380599857336544340749434,
0.0003080645548464782085853219779859089901768,
-0.00001334606130261131957496413127972681393944,
4.371755799402702037647486455654657531728e-7,
-1.153511915638355120045359822843959413502e-8,
2.551612510687250532786824416403269493418e-10,
-4.861479654870722591033345934387818305125e-12,
// [0.42857,0.57143]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.03351163084671353058543772474270012517697,
-0.004312903767846243101070112078411584042129,
0.0002802672873550372731923277340174545287784,
-0.00001224091641640190155450874920962842336997,
4.037291678125877666591241817039524800822e-7,
-1.071506302120864951411325046988394328563e-8,
2.38221017783219745602793183983617462919e-10,
-4.558748957357520222839391336087877243593e-12,
// [0.42857,0.57143]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.03019032637492369974387967402754568237371,
-0.003923742022966331025045223785505190158413,
0.0002570592447446284402951665750193664634767,
-0.00001130441686638593177373870700918746891419,
3.750272032288595221753765225149624618287e-7,
-1.000367345601597859348307974592290305905e-8,
2.233867419455847474653519318220958309819e-10,
-4.291460489309549512809159505301310274147e-12,
// [0.42857,0.57143]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0274661941607643153147820931249700452735,
-0.003598829426420839508063484961986761798876,
0.0002373927541942831794950796517176427357474,
-0.00001050076184875572354370334113297749273163,
3.501285685790551328354469372574511840012e-7,
-9.380723015157362762781828266077151207235e-9,
2.102891848053390867116172560333262802211e-10,
-4.053741392089873403928880648453572471305e-12,
// [0.42857,0.57143]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02519180598494587478871550778385964085222,
-0.003323498558716213660126343780327460011695,
0.0002205159988240398972855279022077600031862,
-9.803600070249002883352554424270769057782e-6,
3.283253032677956363169717308366323350268e-7,
-8.830705412795585409769144744093850740869e-9,
1.986405689062652245758203142030631686742e-10,
-3.840947163492461315073088285768763112247e-12,
// [0.42857,0.57143]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02326448991101349393715104358907575544202,
-0.003087223983532994805949222964308757384157,
0.000205875601475390699809024320408265109166,
-9.193108634049805611559974172701978669117e-6,
3.090746872926485280635350203231385337125e-7,
-8.341535391166703323319510222810269692115e-9,
1.88213307656012273154492542220766094437e-10,
-3.649356732664634066870962806327938582073e-12,
// [0.42857,0.57143]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02161056788473096203429331443481787684094,
-0.002882258420652075376888868734099312577976,
0.0001930552813152002243825923047175516182895,
-8.654091379972206346599126031539464611717e-6,
2.919537366334075236639313155424910480055e-7,
-7.903655444423485123116609127584858369771e-9,
1.788250636812615834454277474665501295192e-10,
-3.475954005940881078684095788901498229654e-12,
// [0.42857,0.57143]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02017580894456452610062901979468232128345,
-0.002702773938409562707847243518032769236337,
0.0001817359189795639429999002785549441166513,
-8.174704755353929919371042236771228103011e-6,
2.766279385866860622374742791553765597258e-7,
-7.509408329283146413455201701064329897554e-9,
1.70328044339476770203592668644376007055e-10,
-3.318268584342794485911141089855141612433e-12,
// [0.42857,0.57143]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0189194175688669374812971421331638998507,
-0.002544302865710795401525491384877573249282,
0.0001716687998625739978859782544691213389173,
-7.745582404420360478806689780371174775388e-6,
2.628292896386430111053282240177102137298e-7,
-7.152587520767316336714488266192993713793e-9,
1.6260119670048252197815208575059463868e-10,
-3.174257772685515631477570456140691973619e-12,
// [0.42857,0.57143]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01781012005997556639589940068482682195507,
-0.002403363198073065112774231898725336178934,
0.0001626572304929633895718655927114488843977,
-7.359220228717397832035811261948312735712e-6,
2.503405614159233459591943672047061997808e-7,
-6.828109434677026589428817799566412602231e-9,
1.555444258587577420812132264740451617275e-10,
-3.042217916188108150723776215688572274023e-12,
// [0.42857,0.57143]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01682354238651145442897749629175038216873,
-0.002277201226898635247553445683701773847824,
0.000154543624803195957620114160900962718018,
-7.009535833735159880739462160608729075063e-6,
2.389838284723150070591362670476478954247e-7,
-6.531770622020564136414156526576831309107e-9,
1.490742503241736500484368290250448186312e-10,
-2.920716902840137006502862298145668707927e-12,
// [0.42857,0.57143]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01594040858829044542275192660488298469279,
-0.002163610747242000743868371043318526390043,
0.0001472002525085641299050249539008423582652,
-6.69154730693234342824925164919208488738e-6,
2.286119700937488531713930887122673727939e-7,
-6.260065314923748300088416929272142202041e-9,
1.43120494557607419149114476249070016613e-10,
-2.808542165179640714827356130156866448058e-12,
// [0.42857,0.57143]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01514527523069400394368869316501435339651,
-0.002060803535117256601239284738670100004747,
0.0001405224934457004980803751220329645743039,
-6.401135268274641021281442634338195728945e-6,
2.191022844051781111653229746596312298241e-7,
-6.010046528311704097335710243932603081875e-9,
1.376237409948580159165086546134550002657e-10,
-2.704660184172158428886676850423482486926e-12,
// [0.42857,0.57143]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0144256247458207949888000177805227055731,
-0.001967314908237259968358152510581200181007,
0.0001344238406338845700797403496242200852918,
-6.134864065226121191918412778357341718675e-6,
2.103516269294561240885379864481099359197e-7,
-5.779219056859106362713001943121128157891e-9,
1.325333456865383404172068424355052084336e-10,
-2.608184633704758849458735162254246738899e-12,
// [0.42857,0.57143]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01377120435766081859925070608801425094489,
-0.001881933768871924702791089982239149304467,
0.0001288321453698617542563510147434492620494,
-5.8898456523968174794740614167796326657e-6,
2.022726654806074522347602270979176663536e-7,
-5.565456141349762867528605681752264142887e-9,
1.278058773381009637109086565159577875927e-10,
-2.518351089540214732888257437667668593032e-12,
// [0.42857,0.57143]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01317353638210347177828329174407862881741,
-0.001803650035175687168713177129741939614308,
0.0001236867587004427280821469610887429861731,
-5.663634728553528182113917515438472754527e-6,
1.947909634864234172520184425563641072766e-7,
-5.366933915952079743159125080237480047171e-9,
1.234038780161608192321589542584526273939e-10,
-2.434496777562348763634315225064385450055e-12,
// [0.42857,0.57143]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01262555024622980907536817588488303856698,
-0.001731614621803897414657705638509785571372,
0.0001189363292997302323137334347191918421016,
-5.454147069651853088445698023234323954617e-6,
1.878426856431116328624869754546482475702e-7,
-5.182079364014274521556043286342624489653e-9,
1.192948707861462172968659232197796198915e-10,
-2.356044227964619978677537239853127058347e-12,
// [0.42857,0.57143]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01212130235262728083045362984446491352921,
-0.001665108610193994304818943673212938113908,
0.0001145370884627918418497433576107896163881,
-5.259595287165861158933613047993056635659e-6,
1.813727763681352084889424124573800028848e-7,
-5.009528643829856899739124689607306641207e-9,
1.154505587459484777676500764836829467175e-10,
-2.282487984172681463288090204316901572211e-12,
// [0.42857,0.57143]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01165576027135795909308949227512976623564,
-0.001603519238476924321682026843140389699277,
0.000110451501030582986623537898507840393596,
-5.078437824767163151792829899171579051761e-6,
1.753335012020101360232397355254023841962e-7,
-4.848093452131657954273312276797740010166e-9,
1.118461737453351927627253665341168814691e-10,
-2.213383720823547134374305886085351539244e-12,
// [0.42857,0.57143]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01122463466933846924084000705612733316199,
-0.001546321014426063867277754990291933359178,
0.0001066471943202074764012783118636092014458,
-4.909338117574888170552952923553325476419e-6,
1.696832695306009218362897460076716557619e-7,
-4.696733673669909304934935343467970866188e-9,
1.084599431521789646172355361204492404539e-10,
-2.148339276477296872665680167399105503514e-12,
// [0.42857,0.57143]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0108242471009824460880704033405935205179,
-0.001493060720480866617615541422612239011098,
0.0001030961004691670079925149256717122866677,
-4.751131628379670960584411579493418428094e-6,
1.643856773203572007703338799839980773837e-7,
-4.554534988181700623883438271447982716766e-9,
1.052726504430324925545643303319453671362e-10,
-2.087007219324491872390961472253741383204e-12,
// [0.42857,0.57143]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01045142504336606536698005702951252800182,
-0.001443345406566356629684779099430387170017,
0.00009977376419606489823562352496184346177554,
-4.602799044230034974371632040937131942917e-6,
1.594087233622479705381609377421966198008e-7,
-4.420690417519558633197135212729897765569e-9,
1.022672709124828575109019625327986706299e-10,
-2.029078648697876303874108241152855673649e-12,
// [0.42857,0.57143]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0101034178459644954766141744506767053524,
-0.001396832698742980608324918425829657240042,
0.00009665877992892012816498454123672621060182,
-4.463444331262347974518181360783851552628e-6,
1.547241634212645707425109182807727962097e-7,
-4.294485027392753917429268587994901481014e-9,
9.942866793853975101697173689608558504045e-11,
-1.974277999249939650753054654504342836472e-12,
// [0.42857,0.57143]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.009777828891200863350956131817514702018393,
-0.001353222919003004528840677639856762041846,
0.0000937323309565964105017801234906393178277,
-4.332276650886734456999691740140327981043e-6,
1.503069747973687188007591937489252230478e-7,
-4.175283172155111538626039324092045608007e-9,
9.67433383799840646305310100730238759068e-11,
-1.922358663591360307705366116090282792048e-12,
// [0.42857,0.57143]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.009472560433021030817235474176172809386187,
-0.001312252633390520585491728506268852569285,
0.00009097780966870635039578621068229932507411,
-4.208595367493475914944376130211762432572e-6,
1.461349098930678350666858459908812390393e-7,
-4.062517802883182727717319229046504568219e-9,
9.419919807940010796723605905408296104454e-11,
-1.873099286862467664837111499601495705646e-12,
// [0.42857,0.57143]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00918576843373364323535701859958684815171,
-0.001273689335360105444954688509457986298038,
0.00008838050271744585083964239110985712531112,
-4.091777548344996865021439373009314954617e-6,
1.421881219961527203130102974911807355727e-7,
-3.955681459685143166281006413641712921815e-9,
9.178540029183873334791818354951506290976e-11,
-1.826300615935345174006186372991501206523e-12,
// [0.42857,0.57143]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.008915825347520737272135917044777118248094,
-0.001237327038042501924750436720007728433457,
0.00008592732851532581911572741881612408201413,
-3.981267485492414714669151102659417854778e-6,
1.38448850010510426011744255362856352544e-7,
-3.854318646713631062263817060092429455267e-9,
8.949218129114303863378867786287858618656e-11,
-1.781782808772875128257138550285867827086e-12,
// [0.42857,0.57143]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.008661289266297512650289829282323382146282,
-0.001202982599212862885753272334726269487236,
0.00008360661719541971391560944003610920324521,
-3.876567868238210013227573828816611083985e-6,
1.349011515815748181783531856103216027435e-7,
-3.758019348496551408099458848893939021209e-9,
8.731072852460173398830792925459650920451e-11,
-1.739383127421330992304997641499573635566e-12,
// [0.42857,0.57143]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.008420878194490039396003863608453283653126,
-0.001170492640734216893802341370080970174407,
0.00008140792523307962053406347600697225983765,
-3.777232310648764674057899868685610807967e-6,
1.315306761679057900061079601611476687076e-7,
-3.666413493177150037744738417079138817673e-9,
8.523306756618263942969230978605963074908e-11,
-1.698953952316277722327093227233884990262e-12,
// [0.42857,0.57143]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.008193448485139517470211284027104092174086,
-0.001139710953261493274093045740041176032616,
0.00007932187852369955346442570436569989599217,
-3.68285899755851292123034225353513744221e-6,
1.283244712545927758887050574606084670768e-7,
-3.579166205193091641648235249375401536006e-9,
8.325196481457131096732257981654790429747e-11,
-1.660361066886814295015857706120125025175e-12,
// [0.42857,0.57143]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007977976672830452149334000318945305581839,
-0.001110506299330208348964189453458902435085,
0.00007734003894880262621889258610209632289021,
-3.593085258545181594269858493591264311482e-6,
1.252708161970232839007282303968870750406e-7,
-3.495973719151728584763738201832985321697e-9,
8.136084343621265899481855241401540454506e-11,
-1.623482170492854984897657962046611980597e-12,
// [0.42857,0.57143]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007773544095311457689793138089334610199148,
-0.001082760545281685816645243372531703095325,
0.00007545479042952109767570086694420210850937,
-3.507582915555258534147088011299246878226e-6,
1.223590792065180041659576331380177809619e-7,
-3.416559849922876563733765394426513012331e-9,
7.955371049699856580538898803967538206635e-11,
-1.588205585013304355990974687957779747416e-12,
// [0.42857,0.57143]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007579323816971799979240370512175541123896,
-0.00105636706601177740121894829272674038872,
0.00007365924122673120813900733445212328455206,
-3.426054278501731533932641899675591429033e-6,
1.195795938035785692594921599438514284259e-7,
-3.340672932592899812903254814647927634112e-9,
7.782509358312877443488681117798635393975e-11,
-1.554429126294574250586261427122106393974e-12
}
,{
// [0.57143,0.71429]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.8214298175378663038412668196447443338068,
-0.03284908812356517157753282890915977980553,
0.00130192719181872268435452869285205766684,
-0.00004238711245848151373539867327845841414491,
1.145134391875108930413797896908750325738e-6,
-2.626104814486002468421595506169243826779e-8,
5.217862202175997655156970276612506819874e-10,
-9.130788088484260041977291950390940747104e-12,
// [0.57143,0.71429]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.2299436168649561976034147424598204041849,
-0.01822698068545431560845555770474813218496,
0.0008901293616284419642543555986253827476881,
-0.00003206376328458442593220462259787906116566,
9.191366806677441955440576720148288617441e-7,
-2.191202525208701725417287402292325523883e-8,
4.47422703903781244923036943471025485585e-10,
-7.989355651077446557094321725981574612972e-12,
// [0.57143,0.71429]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.1275888647981802061647018718955324984053,
-0.01246181106279125381483457640552774392789,
0.0006733390289765700167656916529274576065117,
-0.00002573582733604647631566889706570140595585,
7.669208798620716039940415557115670378596e-7,
-1.878909099302724019284914473881410461347e-8,
3.91491102044073897031177224129167118947e-10,
-7.100216577057653403918796604777102209265e-12,
// [0.57143,0.71429]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.08723267743953877389158532828574726818893,
-0.009426746405665741725435729202659944056687,
0.000540452374057245980596429912541234809528,
-0.00002147378488568043788606787950699510929013,
6.576181811562365864601515099182320441801e-7,
-1.64402306653001301199946580488624728465e-8,
3.479221313922388648086995024893701492968e-10,
-6.38834660158225640316565012055394344833e-12,
// [0.57143,0.71429]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0659872248396601895010056301787733174261,
-0.007566333236795774197211413705206424031784,
0.0004509494825995365931055812816986740726969,
-0.00001841330929915777237353648360233840630118,
5.754080699868614163472706695510687757478e-7,
-1.461055238807819999824538002680035852897e-8,
3.130395391565405397320743072329964823068e-10,
-5.805704247039594677022159479131423188756e-12,
// [0.57143,0.71429]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.05296433265757041700244658834286366769987,
-0.006313292756388316961750330241411774673151,
0.0003866794952825415121881742254752062850224,
-0.00001611142616744755473805406789762992479107,
5.113693305388299100581612410791461606101e-7,
-1.314566560509566793507891784350215714802e-8,
2.844892486271928613947402755267595355079e-10,
-5.320126805624988479534507019760594113838e-12,
// [0.57143,0.71429]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.04419304929471821652477096546941723003746,
-0.005413512933950787036762010176279448062643,
0.0003383399495166105687828020400177704097607,
-0.00001431834144685424012545633348155357605936,
4.600982933527493333346015836307780780347e-7,
-1.194670746857107656446763088400297358376e-8,
2.606952554109036448850880535177257958642e-10,
-4.909283476408456840918568055118217067891e-12,
// [0.57143,0.71429]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.03789459053765550719763067653201274691059,
-0.004736759293228097663016112351603243353099,
0.0003006851703841367751714365987094373566734,
-0.00001288275239189052338808213749124303231657,
4.181347587635457672783777550656077617554e-7,
-1.094749178737045481480430636892118060216e-8,
2.405633269753882800748563607581686698319e-10,
-4.557191859632315727470708830126856439888e-12,
// [0.57143,0.71429]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.03315731505259668171068126536967978660114,
-0.00420959238537376247384389552407489421488,
0.0002705378002298863135303127159105923848392,
-0.00001170777341147587683806550684251559100202,
3.831622100869932074470354226729164505515e-7,
-1.010206519635097353982798342526237445738e-8,
2.233103082509503071525673333371688944571e-10,
-4.252115355115386062610339864078258913262e-12,
// [0.57143,0.71429]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02946714669761633550050733395102174547309,
-0.003787529203214516623886551925103313908125,
0.0002458632416411677888972236199306835546555,
-0.00001072854203810780336195358303351773273914,
3.535722795472728696074352348947600283871e-7,
-9.377538486620686873814370460060791443252e-9,
2.0836109245193805514600436790263714259e-10,
-3.985243177261306735174793841070412330581e-12,
// [0.57143,0.71429]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02651270442250161465213833427929680113472,
-0.00344208538297268716752542296181203052292,
0.0002252993828004285179425528854483674785131,
-9.900023973800021306474667503947840825765e-6,
3.282138448364192836932867301974273793449e-7,
-8.74975970140716811264595203641605849211e-9,
1.95283940675681829863266391785936289748e-10,
-3.749833373318018264301592152587114953369e-12,
// [0.57143,0.71429]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02409459768080880854825028976414787586371,
-0.003154191359202541661030942997070906805871,
0.0002079005034499563933100496950732550738846,
-9.189987793724580822801689600864433917471e-6,
3.062415874699660214497520356715966359923e-7,
-8.200597774006705207598650826795375123507e-9,
1.837484890180979247690495898331855565791e-10,
-3.54064003852102702937413233730262515783e-12,
// [0.57143,0.71429]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02207933951441779008434175049275466827659,
-0.002910607048296114646800672841433250550507,
0.0001929897436683643140448371828954243863607,
-8.57476458015460547712067607899553217052e-6,
2.870209201153382260181726517373799458213e-7,
-7.716178973892286874406324814846643590428e-9,
1.73497681569508541866093290849519851558e-10,
-3.353520641516349370595094771534329849286e-12,
// [0.57143,0.71429]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02037424933807280105849538727930800312988,
-0.002701856411353989948183250429823197250523,
0.0001800700561833877502607995509384176530762,
-8.036585887648529750117456542443399121554e-6,
2.700662622057541866585041185045479252099e-7,
-7.285708195770351972004879594673664947941e-9,
1.643285289697706683679030343344193987623e-10,
-3.18516073751635394043129095848812625186e-12,
// [0.57143,0.71429]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01891299487947792823521506053322059162324,
-0.002520980786564466767067242706888820694607,
0.0001687683036407537239660248267310058622292,
-7.561855460231653270843181462750146059167e-6,
2.549997850573004175509893825454415631002e-7,
-6.900660892719105468981093914334478572006e-9,
1.56078619068590239952274660598774138789e-10,
-3.032877106724009279999697476761194921843e-12,
// [0.57143,0.71429]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0176468655059512660285991243453031816777,
-0.002362756250967725555376583360890098557433,
0.0001587989646649934430452391236791266088736,
-7.139994094668645422775975275274447095409e-6,
2.415231295288384333071942646416438831812e-7,
-6.554216577658874247497714515332380251816e-9,
1.486164704980166007294941406709697238126e-10,
-2.894474454861310463651203784834292430697e-12,
// [0.57143,0.71429]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01653929375677407760284701598116858729354,
-0.002223185505307204994328150370495831578479,
0.0001499398759881648942886546182971290751852,
-6.762647734936794894781729389972258463848e-6,
2.293975785735165148309066551379188673853e-7,
-6.240853713142592530971799649752083911257e-9,
1.41834510840534479478888531424446987062e-10,
-2.768139424302004271558027677442449633575e-12,
// [0.57143,0.71429]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01556229853715043372709158907304176061807,
-0.002099158263831718374502960643288282057888,
0.0001420156024337910811502614791382834204956,
-6.42313230366523241427134894901063358228e-6,
2.184298783814741911612673934469355987346e-7,
-5.95605482424020537632770474793412075071e-9,
1.356438830545933585223199876330323416688e-10,
-2.652361061743823627608703743813018201161e-12,
// [0.57143,0.71429]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01469410784682202743592001329146523041936,
-0.001988218434070588983511790694170387519715,
0.0001348857783770836989805509721375621675847,
-6.116036694128294243435792218185190992045e-6,
2.084619173308252094839897114987298281028e-7,
-5.696088391086147508688435887347082879582e-9,
1.299705482982339267455314361468213187756e-10,
-2.54587035071659456990325049631985688363e-12,
// [0.57143,0.71429]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01391752903849412174305065132736180706336,
-0.00188840089727678160468842494486976715435,
0.0001284367705768037667643508175817298052512,
-5.836933780871227923545723650484581504724e-6,
1.993630922268417357338152400041576401226e-7,
-5.457845184918918625318245730668250204832e-9,
1.247523229501727749264882914442624075652e-10,
-2.447593685409805814650269597157151456406e-12,
// [0.57143,0.71429]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01321880628093747013219957396404077828743,
-0.001798114788072951396764432215166879152082,
0.0001225756093984014464285233929450818740523,
-5.582166674405936079818781739265739443859e-6,
1.910245800633571248649165260422058635661e-7,
-5.238713836539291761293663700183170982e-9,
1.199365987709223976087675267994253787748e-10,
-2.356616676829610165665846506866635585054e-12,
// [0.57143,0.71429]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01258680351651065871481492278670633516773,
-0.001716058531575401391951402598364545515454,
0.0001172255001626266616951564265167721781935,
-5.348688330529143003515878375972081430549e-6,
1.833549829188172006398575542656402548284e-7,
-5.036485093742858686769491914295009697518e-9,
1.154785693597258256818848919364245804485e-10,
-2.272155711366053087089077364735576547169e-12,
// [0.57143,0.71429]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01201240972102780871666131029945699873305,
-0.001641157002274631181915059418401791141559,
0.0001123224549412105954663511795467907279897,
-5.133939607410949302702382604679458075004e-6,
1.762769769664305343474655100768772222368e-7,
-4.849277340970680535830769233484096579248e-9,
1.113398364885433686382185200070155052393e-10,
-2.193535392334825040464810193165129588158e-12,
// [0.57143,0.71429]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01148809901592241727964565349881761979491,
-0.001572514369174877898786195094797041315227,
0.0001078127317557253368134168183860424217328,
-4.93575543787833474571376115856125479913e-6,
1.697247056619502407066707884988644212527e-7,
-4.675478072104382401106969143335974528361e-9,
1.07487304708721973709453118304612711342e-10,
-2.12017049285094933769504582169865089675e-12,
// [0.57143,0.71429]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01100760058422414432889945268702005467853,
-0.001509378244578151287221980402360069413676,
0.0001036508641955374401397818849533573208473,
-4.752291838672480039192320472700514078755e-6,
1.636417312915096202549294720743479801159e-7,
-4.513697469403960209331181005670676205478e-9,
1.038922970183448478453043520588627296108e-10,
-2.051551401932970740690304932847579929943e-12,
// [0.57143,0.71429]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01056564771204705807721264231054320109185,
-0.00145111209873558354423392163217379263675,
0.00009979812861221168206990612626804460241835,
-4.581968553787702924013096428144743327783e-6,
1.579794102344513981370658269152004596085e-7,
-4.362731265970637720517286168916373720295e-9,
1.005298417021426070613657313095137050853e-10,
-1.987232300004320783715557724977104584247e-12,
// [0.57143,0.71429]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01015778469114908390383314981876613690498,
-0.001397173800569081925079672709172240865093,
0.00009622133962962871909328444554601001755874,
-4.423423561830298976953778791181173248496e-6,
1.526955931495326582847813518810592911323e-7,
-4.221530796650841578240300312229834021844e-9,
9.737809291525468534563780709749699595843e-11,
-1.926821484977399927962631883429660401138e-12,
// [0.57143,0.71429]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.009780216603983572595712777387359199786208,
-0.00134709875481297595805831799295485368647,
0.00009289189479852074409261164026010229436243,
-4.275476681231968596490545025484113335266e-6,
1.477535767565680622354051442726551057951e-7,
-4.089178665541437934210060125264793275662e-9,
9.441785664825475043247201161192458212411e-11,
-1.869973406207731445890490768366643131926e-12,
// [0.57143,0.71429]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.009429691283690830851076845295756994921577,
-0.001300486527177516642349649993122953532267,
0.00008978501030595345278977177622699020981202,
-4.137100220135569329082564494033467755775e-6,
1.431212521991165406421520344619088153411e-7,
-3.964868838995697485011783766174171955435e-9,
9.163220037999334571418859882982049842443e-11,
-1.816382064745334107794456491093913084153e-12,
// [0.57143,0.71429]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.009103405690242615664301367460034190771673,
-0.00125699014428162398362820158167786261934,
0.00008687910462292684238358544112577684509219,
-4.00739513055012842271708983909383120434e-6,
1.387704082996929259315832460649055046103e-7,
-3.847890253104859976977450175345569469538e-9,
8.900612968077474785165546872573444463624e-11,
-1.765775514186945954137004704109647218526e-12,
// [0.57143,0.71429]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.008798931009971367075213155846006924153625,
-0.001216307464719298179490477442660559194542,
0.00008415529774163047499387898534462121938451,
-3.885571499496588589250640823992431362423e-6,
1.34676157821625115015828612180833668581e-7,
-3.737613232759110214808109280465781540862e-9,
8.652631874641978526877670354452116786769e-11,
-1.717911253865884274452257202819250233723e-12,
// [0.57143,0.71429]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.008514152253035086467082030253795575695603,
-0.001178174168381193311459851175847094457446,
0.00008159700148950413851804358068234395828099,
-3.770932484339657042161165154488140490784e-6,
1.308164621361902643435281963132801444842e-7,
-3.633478175532797980005321347805945855915e-9,
8.418088465805776885090129294919725169187e-11,
-1.672572349954759085205597382362329811153e-12,
// [0.57143,0.71429]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.008247219178668352410656449039012004564828,
-0.001142358020851466600183110965062954168816,
0.00007918958217120667629413449691305638879893,
-3.662861003467490768298057792924432237101e-6,
1.271717351585992163306753923646809183027e-7,
-3.534986071826325917069592872954273872283e-9,
8.195919731062572859596207860055061550297e-11,
-1.629564153782769182391840862437254760166e-12,
// [0.57143,0.71429]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007996506145960265450540481908344661933587,
-0.001108654150395342023437464997700821481642,
0.00007692008107288937769692176525421275494126,
-3.560808646499151522533279941108814873752e-6,
1.23724511552964051377909602821867438822e-7,
-3.441690522908998492490311064748221702153e-9,
7.985171860569271100384849319509622216677e-11,
-1.588711512810012096657615970909997258443e-12,
// [0.57143,0.71429]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007760579052767393431243836093413923474092,
-0.001076881135018937786799943129720782721618,
0.00007477698157655253293069249421366769538727,
-3.46428638402360429587033044314519140718e-6,
1.204591673637990663875368251176853942907e-7,
-3.353190987910335240328708483010141408626e-9,
7.784986578515311231036831261166300439598e-11,
-1.549856390105178536343608946573482013214e-12,
// [0.57143,0.71429]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007538167945132563791868518219908220846755,
-0.001046877742070258092738922239325379876776,
0.00007275001406456440077791835803744558154784,
-3.372856745281665021616223699439529873789e-6,
1.173616836607178266405795739107612768882e-7,
-3.269127044600240641094909095580595547748e-9,
7.594589478215119888555460885855292776051e-11,
-1.512855824205371077667367094571587002287e-12,
// [0.57143,0.71429]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007328144194491805949750245946735342641255,
-0.001018500196902458690809508544128711855164,
0.0000708299916509821103583871361645387752539,
-3.286127200217449465930793792384561542353e-6,
1.144194456657400512932261653934041092028e-7,
-3.189173490786032513394554226129856824801e-9,
7.413280025120506339004880096110932874919e-11,
-1.477580173913806806079223651482668083543e-12,
// [0.57143,0.71429]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007129501378317210151826760363639898172997,
-0.0009916198831123395037614576484472752212078,
0.00006900867120463208776966479771699503428391,
-3.203744535042906818144990724826463645652e-6,
1.116210713021884213973060348240124474324e-7,
-3.113036146146005394350901771687359791665e-9,
7.240422956068824852003269786965584898843e-11,
-1.443911602675164868942516287333522642709e-12,
// [0.57143,0.71429]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00694133918178637585739381943536957980561,
-0.000966121396863470601788181818281397044615,
0.00006727863523596526143068062708186480206711,
-3.125390051606879013424564208728908244652e-6,
1.08956264258864006204881510854406425978e-7,
-3.040448240406159539323920954386805374111e-9,
7.075440852495620566082693669895407726099e-11,
-1.411742765235408487823209909672684117899e-12,
// [0.57143,0.71429]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006762849778044293557848731356002137836355,
-0.0009419008933021650784520216471834672640459,
0.00006563319108380730781135114949217009654502,
-3.050775453191968065764677852239877006435e-6,
1.064156875762323956725181372435775145378e-7,
-2.971167294517708272284278405160847945343e-9,
6.91780770487015248449446037483032665907e-11,
-1.380975665781437055614812638803320871776e-12,
// [0.57143,0.71429]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006593306253115154908167678271940427713614,
-0.0009188646751719824918715621661457972334225,
0.00006406628451709286538494890352931635278729,
-2.979639304927708874567150280913538470392e-6,
1.039908544876369979674973714763309389019e-7,
-2.904972418093473546458837527935033040592e-9,
6.767043317406810642188344408449466463943e-11,
-1.351520662001734830299138040696931832019e-12
}
,{
// [0.71429,0.85714]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.7898413888797413156093208161322204400987,
-0.03036794277742946910849758358681709712129,
0.001181381688299203182030990510232914572136,
-0.00003805906275150718159705406820491489775583,
1.021345070027059237000455816903810187519e-6,
-2.331474296142132262582841252891022700616e-8,
4.616997342936863504707072783433239891676e-10,
-8.058896774115931065933663376254103038886e-12,
// [0.71429,0.85714]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.2125755994420062807347531149636940645292,
-0.01653934363618197211109788669719192695703,
0.0007992403177819411889885272222571905843302,
-0.00002859766223565732638667321551367092671399,
8.160159997780626338655035254551844101567e-7,
-1.938874978960874732219470645613649207683e-8,
3.948983313428235509135833657743933743326e-10,
-7.03749510901354064300936727232996206516e-12,
// [0.71429,0.85714]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.1157754054532738020597709582142432047283,
-0.01118936444894107876778954324577456937946,
0.0006005509069490647751452474367915188512655,
-0.0000228484482377028495916004407337929004025,
6.786062391573490968294576092696971245847e-7,
-1.658338829964578016884908059300196238831e-8,
3.448483930257108344197278123422456402122e-10,
-6.244335403217850011550272928771326961089e-12,
// [0.71429,0.85714]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.07832555114258754890719148732967652231044,
-0.008407712697281427516373129717769669138349,
0.0004798174129919967067731151126181488174587,
-0.00001900097491558100821356528178369489908251,
5.804185873293883719319783709272229730454e-7,
-1.44815284132727949278191268194796030155e-8,
3.059825410620477763087994936882612960873e-10,
-5.610938848310057346023564063325740781093e-12,
// [0.71429,0.85714]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.05885398888096999035572936583832074799453,
-0.006717443781882979727893593316487151944365,
0.0003990204732274180262842695833947018158924,
-0.00001625172064419121783001912935382258433913,
5.068534915731552276327609683406507720862e-7,
-1.284935661784537283281772574914209958368e-8,
2.749452560418491389001254675149768966252e-10,
-5.093638340507401412709658085998109482113e-12,
// [0.71429,0.85714]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.04702210647318085601246626542253099237962,
-0.005586286625179298443210577128886904079467,
0.0003412861335282155231663863815507226521724,
-0.00001419189794620688279281366919035116896481,
4.497274789585968805698240326163398209291e-7,
-1.154595202191581309450005634849026537442e-8,
2.495968098737160385925695758837224240666e-10,
-4.663303754008063088204388727126350753247e-12,
// [0.71429,0.85714]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.03910400637625508717039138388244630710996,
-0.004778005869390818405553149574835237814431,
0.0002980298568705300195233868546072701674376,
-0.00001259236957879890501730897141574318964255,
4.04108318293967619674314054277573483994e-7,
-1.048145360682495836886075270851440048244e-8,
2.285097978371100052719463084781402754137e-10,
-4.299771893411926920002837614370333782425e-12,
// [0.71429,0.85714]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.03344604108573572703721581470286138113267,
-0.004172417996183525179350516304521162818039,
0.0002644397611549499652452094886007147415779,
-0.00001131503306803620256022798406687768013912,
3.6685087393273471922537402955795767484e-7,
-9.595915771469304040855671635953345188246e-9,
2.106962024364973990965560623242014300237e-10,
-3.988650297106406881615546579015213366119e-12,
// [0.71429,0.85714]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02920692597328467456777036970312539555358,
-0.003702156656165667360039308183179661329195,
0.0002376156944289222721900349105802930356737,
-0.00001027182461540397766372550880766702062345,
3.35857049841173301155349552091900593336e-7,
-8.847845735113201912080041669423948629678e-9,
1.954507773799328715258796934340872176622e-10,
-3.719394731884435764162949316166904686028e-12,
// [0.71429,0.85714]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02591509659315966993303395817425168680806,
-0.003326619722001509427621184434626041764564,
0.0002157083169236359068827671616818518523819,
-9.403997531649372440531120016152172714319e-6,
3.096745986972764911143932903706503807055e-7,
-8.207626115695064018058851982501059257397e-9,
1.822568432696528239271725689491394387258e-10,
-3.484105975294893779466979212291331624382e-12,
// [0.71429,0.85714]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02328633805401056449528840834930322128206,
-0.003019916436927702805207435012533944079183,
0.0001974839481647806357494192782798371901206,
-8.670888891520522471495451582682553831654e-6,
2.872669121317946008213688759380873946827e-7,
-7.653558640968582770627354359416061280708e-9,
1.707273289066892789802626464794903706347e-10,
-3.276750783844523492184926636981087281051e-12,
// [0.71429,0.85714]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02113941505849391821810278583743230762962,
-0.002764775274303908799525996104668889319909,
0.0001820886667220671341437613083399741401194,
-8.043473660495364790393313457807130399516e-6,
2.678745506183981244166217902057007849906e-7,
-7.169388078072691867981863968211234703095e-9,
1.605665980278305136702838091571392654023e-10,
-3.092642488681482588741136251232952172575e-12,
// [0.71429,0.85714]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01935342692012736025000291217838529108928,
-0.002549241334106080474252603258405922754886,
0.0001689129468705319424687886080542571651133,
-7.50048753169232338980664831973246049558e-6,
2.509285810087804958872106555269526161428e-7,
-6.742699089716916086530658394606820270228e-9,
1.51544998000912716615641311354390338154e-10,
-2.928085706748587073000565815486371715627e-12,
// [0.71429,0.85714]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01784468933874256203787538568670136283849,
-0.002364781256184732278458525475312963218351,
0.0001575102381656618535491375641514384433094,
-7.026000376843476491946457541604746749814e-6,
2.359944664992553470053056576112709287789e-7,
-6.363847372599336689619306062136037121497e-9,
1.434814503192797509347583122519713667034e-10,
-2.780127764932051584456027911556536038257e-12,
// [0.71429,0.85714]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01655346879329312472616417909853622181167,
-0.002205143334316681643097919242293261773769,
0.0001475460079138304193222642672837487791125,
-6.607845165352348792671041693853584985681e-6,
2.227346564754652445444297954946643055632e-7,
-6.02522852471152889629493355798187550709e-9,
1.362312701292377867902532868807914101817e-10,
-2.646381265938513711554228532621113124844e-12,
// [0.71429,0.85714]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01543600334021677033232824328384548739733,
-0.002065644110791160200601957398838772953609,
0.0001387647484725115835347655143239287564792,
-6.236570479940719370793176472884545995854e-6,
2.108829968681135426303699153608670903034e-7,
-5.720766513844467785540652308801446548614e-9,
1.296774717690982221403557244255578902319e-10,
-2.524895147127732319147318707842350689671e-12,
// [0.71429,0.85714]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01445950877553812028403497202175367506314,
-0.00194270647861280473561655802050222442539,
0.0001309679800788626445197986366294379700128,
-5.904724006605393963517183325128832911682e-6,
2.002268265507153298907326434673848574929e-7,
-5.445548545936533061465697304494965411232e-9,
1.237244505104101048845067687258653201551e-10,
-2.414059458384404618943851078780441158919e-12,
// [0.71429,0.85714]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01359894535028963207435054734014517355237,
-0.001833551721101818733564857822755234271791,
0.0001239992041388164704619083532467194669021,
-5.606351233752444326659863936344916118692e-6,
1.905941977318111876614441617558202478078e-7,
-5.195559724962109723072748477703500827183e-9,
1.182933165661864609481939315729356183862e-10,
-2.312534010350974241528920650656114296545e-12,
// [0.71429,0.85714]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01283486204771273010169742136010484283223,
-0.001735988857941263447261131629931406352098,
0.0001177333759089005240255298746251223515592,
-5.336637623177077198154442032394899372782e-6,
1.818445890510940286369896073489997449989e-7,
-4.967487101593330350830765830919836013303e-9,
1.133183987717031061307643852503268219694e-10,
-2.219194197644111747319544974951432670141e-12,
// [0.71429,0.85714]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01215192200558884313616742404689144657259,
-0.001648267262722524282368670807274593268955,
0.0001120693900868141090841052914174586242866,
-5.091648576753559417016683447890084275139e-6,
1.738620472825901364957187097212973054308e-7,
-4.758572843406775298994691095963012890531e-9,
1.087445898579153937286855651593284068249e-10,
-2.133089363667823436139449422194726181419e-12,
// [0.71429,0.85714]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01153787083905766901773921404654404557337,
-0.001568971461213392283201181555732312835564,
0.0001069246201119167970373937817879149530675,
-4.868137404123023274301326686492514013693e-6,
1.665500482919094190432481575782019534463e-7,
-4.566502748499108104034801523551584499645e-9,
1.045253062769119531159776285709032344795e-10,
-2.053410448166366552947783110366439024148e-12,
// [0.71429,0.85714]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01098280022849374505689781439858094729402,
-0.001496944681564902125965652175121733274063,
0.0001022308854866723382457788412729836877011,
-4.663401429495481573910080933719421657692e-6,
1.598275950128051245729330789347050780923e-7,
-4.389320567679633585108184321225082824682e-9,
1.006209028920808657726413834794013559902e-10,
-1.979464591145387275063380404275893153668e-12,
// [0.71429,0.85714]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01047861277095431398734247447700404136518,
-0.001431232396811546898493972027241198713019,
0.0000979314300194909775592879157491780473791,
-4.475172734992713632952967386317886784185e-6,
1.536262187239233480493975503371578353914e-7,
-4.22536142900346360442676488798423677279e-9,
9.699742853833284375735637103965044111301e-11,
-1.91065500964705319029008259812807584177e-12,
// [0.71429,0.85714]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01001862677768082742411428035030374993436,
-0.001371040020271070533356107992326225644498,
0.00009397862743493005972252690104134927381677,
-4.301534196396623174746452497675238686761e-6,
1.478876489074718681222100415079909436224e-7,
-4.073199577332406006471532285070679469298e-9,
9.362563995836939901917474250785151797844e-11,
-1.846464913659170978556924385373730808536e-12,
// [0.71429,0.85714]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.009597280141897492895393486170375932072382,
-0.001315700784087276296184903677280317164047,
0.00009033221812440954463840443544661748385358,
-4.140854239191459533214926484543930595843e-6,
1.42561984133857783167054090129762397608e-7,
-3.931606964518076081416194041139940373069e-9,
9.048021366135543748085523176354243004739e-11,
-1.786444546478950426485386466191491716587e-12,
// [0.71429,0.85714]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.009209905488610933260777274768191565825374,
-0.001264651053740044009642177225230482412602,
0.0000869579390230986522606121154901049778711,
-3.991735623333260500992405807306626456287e-6,
1.376062427181017451959657320373672311886e-7,
-3.799520151411389159092895690045001257133e-9,
8.753911088367905948169634116655090632452e-11,
-1.730200664029858457480082277258871292505e-12,
// [0.71429,0.85714]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.008852557376180307279045870481277850931285,
-0.001217411146321743090312404454894664616483,
0.0000838264480900741620746555157161836910526,
-3.85297486162911491823385030385706980929e-6,
1.329832042901745887197543501743222588597e-7,
-3.676013639439709242546208554497360909491e-9,
8.478306206143554921584362814326912743477e-11,
-1.677387934201899594225208552819947188477e-12,
// [0.71429,0.85714]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.008521878024252200866421159375422720363411,
-0.001173570273259448748348346377342672823794,
0.00008091247209428492719041911848432885924153,
-3.723529783706311831142649854656850784765e-6,
1.286604764002011659009361993352240706153e-7,
-3.560278221105105019895600435793640194575e-9,
8.219514538647556725836546915393154384018e-11,
-1.627701859686019250789452383375237034965e-12,
// [0.71429,0.85714]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.008214991912816140494088445267598040215208,
-0.001132774609318445190488880404193500473722,
0.00007819412545790400644435566991053335405291,
-3.602493400957818997782180233231727376651e-6,
1.246097367859023321478552087791468431352e-7,
-3.451603281535617398895674340327833151858e-9,
7.976044001561032347196150367892557756824e-11,
-1.580872918638944207478439979125593307019e-12,
// [0.71429,0.85714]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007929422265229115609323246874673586523168,
-0.00109471775640915547425429215059611360738,
0.0000756523614201837128315737011580410201853,
-3.489072690030467897746342188830545868838e-6,
1.208061139268918425659146493976868288628e-7,
-3.34936223510152015668538118099455352777e-9,
7.74657389550839737061193747599656239963e-11,
-1.53666168561791036186699690946428093388e-12,
// [0.71429,0.85714]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007663024294864087614860005511678502594366,
-0.001059133059881112189984555259414119332402,
0.00007327052649070749854907075352625974051986,
-3.382571248345107269486661794648151433875e-6,
1.172276773262476060120666661951592551512e-7,
-3.253000468094975560684690231439101887265e-9,
7.529930997955506578270131736027797755332e-11,
-1.494854746727039890988456649306190325311e-12,
// [0.71429,0.85714]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007413931419167784643161451799556612184899,
-0.001025787370868483854801403454813233356845,
0.00007103399621531317914443982549187149737119,
-3.282375021980463950605153272736217991802e-6,
1.138550155043014339995422652493641755598e-7,
-3.162025298616284258525735611635078224598e-9,
7.325069546850205604280020590362508190029e-11,
-1.455261262194588483824733261277958680704e-12,
// [0.71429,0.85714]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00718051159607938631415430598790161069378,
-0.0009944759470130000535286340165856835796455,
0.00006892987546165401104720803380062028119588,
-3.187940489499142309841919052682494334456e-6,
1.106708845946593176687760008155820901159e-7,
-3.075997570792880707248839887908607342559e-9,
7.131054396760978076631820299813843345316e-11,
-1.417710059793037334885858093901887079638e-12,
// [0.71429,0.85714]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006961331629090999721672422545554759678925,
-0.0009650182564618065265418873883271759742196,
0.00006694675027954467954262984501682972265845,
-3.098784822636095560027755245980828036091e-6,
1.076599141418675737642090202962772519693e-7,
-2.994524581283414887179638363248349844709e-9,
6.947046776217490090314788105842163534496e-11,
-1.382047165897465178523805423206000427686e-12,
// [0.71429,0.85714]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006755127795232645048406121387616307633213,
-0.0009372545039123090034586242304255940722979,
0.00006507448127541919625979033125996317806869,
-3.014477648633195569960938326048131451513e-6,
1.048083595290568233089998360782930434578e-7,
-2.917254098150629376144172447711197052085e-9,
6.77229218953863207560189864672892946152e-11,
-1.348133699215830729463943748296082054035e-12,
// [0.71429,0.85714]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006560781527386162401732778811022719877287,
-0.0009110427378545837697334999592373422085736,
0.00006330403062135686514886717074489065320836,
-2.934634118213193789621818333874867127289e-6,
1.021038926384936512119843351598846494571e-7,
-2.843869280305725957630890451730770778179e-9,
6.606110095803059204761707802766594126645e-11,
-1.315844066545250382792695572309420673748e-12,
// [0.71429,0.85714]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006377299164982085779885097966455940927428,
-0.0008862564286977411918551922098067588305263,
0.00006162731648253546184836185652276629238292,
-2.858909044075107112339988516630571630102e-6,
9.953542403213426789216785579402529876882e-8,
-2.774084343257821014474918045727697980096e-9,
6.447885067789243235080348486367309671236e-11,
-1.285064411224057521270666725290216459716e-12,
// [0.71429,0.85714]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006203795000884187748329462982585940624011,
-0.0008627824307542702297954167100390028424391,
0.0000600370899256343367366690420849757221668,
-2.786991921949668088731333169978935292571e-6,
9.709295125285616727579450171058625648997e-8,
-2.707640846370960670964359744798834936168e-9,
6.297059189160412911914702451247399827655e-11,
-1.255691273944725399273219132846610931591e-12,
// [0.71429,0.85714]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006039477015279891026909538412242399937063,
-0.0008405192589576818808636625662695163431389,
0.00005852683036099886937787495050958892964776,
-2.718602683033766039697478541792148072563e-6,
9.476742887845428183299952081479195385546e-8,
-2.644304500116782759511249095793932062533e-9,
6.153125492247855608123419435150063974788e-11,
-1.227630432784388596846281660048739214592e-12,
// [0.71429,0.85714]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005883634812703772596829841991665712665007,
-0.0008193756250528115429328405306003747183416,
0.00005709065634376373185387267624959793739689,
-2.653488055502300018332612293341675164346e-6,
9.255065677548469638830928264627263457153e-8,
-2.583862410321544962319787302748695848842e-9,
6.015622274026079493361249749046329533179e-11,
-1.200795895090670122888315869114649100738e-12,
// [0.71429,0.85714]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005735629375369680243235654859661193313377,
-0.000799269188811544701947232888666635771774,
0.00005572324916560180092865984515903588921842,
-2.591418435615768462929703385237535020249e-6,
9.043518364791110968762414870321507067341e-8,
-2.526120691205297997311500748292722658631e-9,
5.884128156200851316556268729940155389895e-11,
-1.175109018531630188433155912024379389596e-12
}
,{
// [0.85714,1.]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.7606177672118986028493670056542602353391,
-0.02811538506807779229252569281071314672746,
0.001073106181883654100352363875734382885853,
-0.00003419787226476047258428537893253488753986,
9.114224563944403241238909498492957080924e-7,
-2.070732049383865400881704157748858304366e-8,
4.086605188075233932379881983944269366829e-10,
-7.114617766472916738180319367614212381963e-12,
// [0.85714,1.]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.1968076954765445433871088558542121751025,
-0.01502348654636510252144370965265466673886,
0.000718155317560225340290375577913208665109,
-0.00002551982902124157863992460153922386662243,
7.247562138787963119932580697097692080727e-7,
-1.716141668409616370830557711933954526587e-8,
3.486271683587886861592063051395143633401e-10,
-6.200305470073048528815684663930395439674e-12,
// [0.85714,1.]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.1051644058245557152625977604333003791011,
-0.01005417444583779103326031443641869615587,
0.0005359164094463023533284371068323716279791,
-0.00002029317420315726090335091212647639831722,
6.006495808873317123653931698872159189024e-7,
-1.464028137072844528961081010538599072256e-8,
3.038247473612451367740585208806604151077e-10,
-5.492555355545087234013498037125081677723e-12,
// [0.85714,1.]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.07037922112086453506783981221573332706858,
-0.007502829732243419712050868754401177684652,
0.0004261566582665103182950955171298229849493,
-0.00001681818845737625398890338233400527954583,
5.124098452042964655387303215498116914037e-7,
-1.275879108175964941335200529827604993094e-8,
2.691440802762216467266915973244415328003e-10,
-4.928839078814091387137283132848296283287e-12,
// [0.85714,1.]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.05251980812570393600414660177442890922531,
-0.005966193215726779835050495124277678100505,
0.0003531819576050914348182712528422547210401,
-0.0000143474758403066037493137885928089894554,
4.465576853268995152637573770610848367811e-7,
-1.130237533375452138011539121747320855258e-8,
2.415212258802404209382676981406396186781e-10,
-4.469452050033320095014717274585590969156e-12,
// [0.85714,1.]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.04176335251008745702098538025368138064121,
-0.004944547406467287969770970572150059378631,
0.0003012969926466138189489990931983829022469,
-0.00001250361534883924675023368303072929717529,
3.955831343461983852645215456068893629769e-7,
-1.014235849213219566431152384646511198244e-8,
2.190106231381653063314074942489833683368e-10,
-4.088002727422432310590094736586607314793e-12,
// [0.85714,1.]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.03461183184527101409721828688312677809673,
-0.004218157897048915524897843994069416819692,
0.0002625759223257865357128473995733361239423,
-0.00001107632790881243114168918493604437334438,
3.549825450599008389464147473525232013309e-7,
-9.197033822520590015734915613255047408843e-9,
2.003190607809452353627569134645827792741e-10,
-3.766280123190738463836936648084711472999e-12,
// [0.85714,1.]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02952710527934240709823390248254800111027,
-0.003676062912557602070695559425115940240042,
0.0002326028860852123556440082447348535824019,
-9.939511398055575294615640792730297095142e-6,
3.21896181770858893951775711905962215157e-7,
-8.412091313067569653474293592891487348006e-9,
1.845541816071430933862304361961699324183e-10,
-3.491320342515786797734704780524272018557e-12,
// [0.85714,1.]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.025732440387903213019314528441962826286,
-0.003256440405189795647579496970937974468026,
0.0002087297393593087351364529944105971483463,
-9.013093216678416142755329516432631905926e-6,
2.944231940686397842599081809267724993767e-7,
-7.750055514473946791873404707431289330718e-9,
1.710807407161952793143321638858988021078e-10,
-3.25364826343153952151826034658562257892e-12,
// [0.85714,1.]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02279508283632856814598359279965517287503,
-0.002922216351027347562369076095540276750243,
0.0001892749575503798986842199343703187742732,
-8.243849552912142461891930219374574495851e-6,
2.712519412311201384783061639423377211765e-7,
-7.184248797186682411483427704496497208756e-9,
1.594344464175072333635922778182193677593e-10,
-3.046180477802270090522045512222673059379e-12,
// [0.85714,1.]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02045551445719143162800557471771070620368,
-0.002649849405702522231382225711028654653666,
0.0001731208406112806158379373093590625736183,
-7.595054466326362519674796119284021406095e-6,
2.514487062265401409873237302745448710469e-7,
-6.695172935207382220129262872044573989504e-9,
1.492682034034560545022130217963496834747e-10,
-2.863516952513528648675934730544754708355e-12,
// [0.85714,1.]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01854894583991765438120304297772304789799,
-0.002423691768555290517881840736750201553294,
0.0001594961437929725068992946716991564057856,
-7.040563879868218295311952660677898566278e-6,
2.34331051147000248767869319683885344463e-7,
-6.268251496020970004190565665213742799116e-9,
1.40317403509800861945253975567522202873e-10,
-2.701469928242482818061493177222328864312e-12,
// [0.85714,1.]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01696584237988703244969059900681345590342,
-0.002232946013099118325786786210115665553468,
0.0001478518414773454310948418009297058848924,
-6.561269531987733952552230144946320157703e-6,
2.193888008561039556670706583484944684626e-7,
-5.892372173105462166812266221989009122171e-9,
1.323768413099986532518661832459615196243e-10,
-2.556742391930683200043217138729608751124e-12,
// [0.85714,1.]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01563062209169382716192938622645074078651,
-0.002069925780680466253339771024941326441379,
0.0001377866601718497970704554068669966292651,
-6.142886518763043806601700058103693017129e-6,
2.062330246268990768612842418875875937566e-7,
-5.558917325080073774957857285592874992197e-9,
1.252849589490074181300938854805955659542e-10,
-2.426703577995864435129217309049199285922e-12,
// [0.85714,1.]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01448948046476326270645841612052127218197,
-0.001929013242403642096231093850465194776349,
0.000129000616894126344800416662848242654319,
-5.774524779756500207448126794910154766914e-6,
1.94562105012133409542774940872882211439e-7,
-5.261102318706745285563229293251799031424e-9,
1.189128454724271683792953286693709585149e-10,
-2.309229016471888952841675655985193033143e-12,
// [0.85714,1.]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01350309269682549365380223588685986029098,
-0.001806008636515617907645168270711426960673,
0.0001212650203749844071583746352965515838014,
-5.447739026377299236691540124277946798539e-6,
1.841385798493612197738978521082851605585e-7,
-4.993513544229127590528332525857008884723e-9,
1.13156399015592354101449926676178178143e-10,
-2.20258449254153900143762048431672841352e-12,
// [0.85714,1.]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01264206045560932437682761648514802427551,
-0.001697710285247725743809800908905383794227,
0.0001144025195540170465736193138184395981626,
-5.155880318021120089044142020072490361755e-6,
1.747729727978474147287332319527966697888e-7,
-4.751779259400187904757822623539989120594e-9,
1.079306406936523675330769735128123280418e-10,
-2.105340484375426135926665146240126859942e-12,
// [0.85714,1.]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01188397199673407926961201561851320987933,
-0.001601635273754269639704515791943412810238,
0.0001082734866785334797850252304659321212815,
-4.893643317100941060578329171104479940225e-6,
1.663122728795638495625458708633022569873e-7,
-4.532330796873694714145064604138276295171e-9,
1.031655219593554531605937148584230247241e-10,
-2.016308141345860393013548713621713958211e-12,
// [0.85714,1.]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01121144691627988657741989844543703469756,
-0.001515828813497579603024371539854192170114,
0.0001027665096592062118614383984775414516536,
-4.656743716193039785602958117802247058207e-6,
1.586315767379144049272608680177501658428e-7,
-4.33222649143622622921535662606412423137e-9,
9.880278746159431762393313235081709414444e-11,
-1.934490736998944071162673409055464262492e-12,
// [0.85714,1.]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0106108016944830563544634677591116815482,
-0.001438731135227071526854602187811801958766,
0.00009779161804013703983112713979640328827477,
-4.441684221279839829360880289179791363048e-6,
1.516279260908732155642360061317993405637e-7,
-4.14901993403968851192203465412788061982e-9,
9.479359618365395350302484797291011008426e-11,
-1.859046406007404832777781820765568578133e-12,
// [0.85714,1.]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01007111794658949985263613556956597856056,
-0.001369082652560171268726871731757552474787,
0.00009327536864695683007378696446577722590298,
-4.245582000436651323951173810260478231749e-6,
1.452156966221368666844644976376895736474e-7,
-3.980660070453509483337040610903549808521e-9,
9.10966955089021798752385349845739339692e-11,
-1.789259221991363114441581678224232086972e-12,
// [0.85714,1.]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.009583578567921198074915001209718931795196,
-0.001305855161055711556254834887303824166327,
0.00008915722200924707084006080377552396098677,
-4.066039572783046561721500211657992121537e-6,
1.39323101433962315551101456627686349419e-7,
-3.825414520528367613820818513395929606841e-9,
8.767700399838698957197155019996672494805e-11,
-1.724516518181310336738930525311922016832e-12,
// [0.85714,1.]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.009140986127389980114816720979671893469778,
-0.001248201108127833739939185164529934716302,
0.00008538683102851875904612010859817376361279,
-3.901046905161620524514060966242452223291e-6,
1.338895072214062844341691012836384449978e-7,
-3.681810061587549752933972860685176358455e-9,
8.450450007487828235679429918807089586249e-11,
-1.664290934645601343148903174907480473316e-12,
// [0.85714,1.]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00873740775689483542603843515477453839972,
-0.001195415634397692222291604710582523287886,
0.00008192198500846637085114136344713785327834,
-3.748906265016133890190995608953606152925e-6,
1.288633511910299405216013288359459366192e-7,
-3.548585958532690721318552375167539781303e-9,
8.155334231411619400757593410542982191801e-11,
-1.60812608228143480738349408349868617634e-12,
// [0.85714,1.]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.008367909440783844826329951557128029351053,
-0.001146907790117010057189885922846193132666,
0.00007872703156540886435900911487932265268834,
-3.608173894114793249163599750779265598734e-6,
1.242005076146055860530790532544980396682e-7,
-3.424657020404103506130685386813250645408e-9,
7.880116696179496977393724159008086177468e-11,
-1.555625001711064043536391648461261763619e-12,
// [0.85714,1.]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.008028354530819069692983301095430734011442,
-0.001102178441914252996843702909861538679626,
0.00007577165177647856380982037268462972829314,
-3.477614272053672748540497810945867695886e-6,
1.198629948087329566270879317087219160421e-7,
-3.309084099604641693595232552603414958807e-9,
7.622852240430457600726615879812057130315e-11,
-1.506440801812766179726439061324681145338e-12,
// [0.85714,1.]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007715249093399770291596183871395410767452,
-0.001060803124869273877943459281881109266885,
0.00007302989971319301381459059094946976776344,
-3.356163911685671731262300927874078927143e-6,
1.158179426076783050916195174666540645615e-7,
-3.2010503425385093043441637155525277248e-9,
7.381841044427216562665328082239984713805e-11,
-1.460269012597207600351873947421368501811e-12,
// [0.85714,1.]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007425621874084916479116423735906945033593,
-0.001022418595983318587086685011599978337648,
0.00007047944214546308953515837006718731042802,
-3.242902448359763462838568925492817378517e-6,
1.120367611357359714303487723105725452892e-7,
-3.09984192554198622554764270619911288341e-9,
7.155591158118189025231310519825926030001e-11,
-1.41684129722997243262311535912408939489e-12,
// [0.85714,1.]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007156930171883229461828843376257410752036,
-0.0009867121900351396083770831015077516774126,
0.00006810095141561721974434287308558365996077,
-3.137029365546915871523522626280825672689e-6,
1.084944665648064571442585850424180610829e-7,
-3.004832318620349021853591081611551791078e-9,
6.942787689188915951136897376320532830551e-11,
-1.375920249644580536274339659073230846566e-12,
// [0.85714,1.]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006906985330245976628550345613622198992709,
-0.0009534133198173351504944240638408928908496,
0.00006587761667654572220118899302174429829682,
-3.037845116052108522194963506560520237761e-6,
1.051691303451908995706274613587662833612e-7,
-2.915469346052921924659602797692101720594e-9,
6.74226731064715151581572499384160048164e-11,
-1.337295065322500436823467586601388059566e-12,
// [0.85714,1.]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006673893238721345440120650428646788892358,
-0.0009222866334703698454853156403597409241212,
0.00006379474743715315996076014797545510534358,
-2.944735700476436402793171985857983998238e-6,
1.020414263267697453724961157625770242258e-7,
-2.831264480937167060703481466265203850934e-9,
6.552997047040488690816254324264535504532e-11,
-1.300777919004945816411250348494685081017e-12,
// [0.85714,1.]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006456006434292588320938930087667758306417,
-0.0008931264641189077400752540546183985785043,
0.00006183944971006252076963421567770858849756,
-2.857159986609993783094144690194235949038e-6,
9.909425606804750482241641093324440537377e-8,
-2.751783936542482363458190773168188161661e-9,
6.374056524733350605718024615035365047788e-11,
-1.266200918295439777761034298977289070127e-12,
// [0.85714,1.]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006251885248832353598149226667172710020934,
-0.0008657522959396708097197042709348833438651,
0.00006000035971886577799850026302160504882792,
-2.774639218085026768445334313432067252005e-6,
9.631243703353703392142297146231355160608e-8,
-2.676641212391978175972703537019513346915e-9,
6.204623044125884259829651996609808820264e-11,
-1.233413529143728352164478504611352591764e-12,
// [0.85714,1.]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006060266071577695099998916647426527619643,
-0.0008400050360629468137949185131784051543006,
0.00005826742357984009418394175318361333202662,
-2.696748283902607819906279279581998985399e-6,
9.368244170662287106582078384726109013441e-8,
-2.605490825414056796842980702142795701558e-9,
6.043958964155354525685336012446476903501e-11,
-1.202280390120410164421038048143222668021e-12,
// [0.85714,1.]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005880035252440627142175213501982233810779,
-0.0008157439301166161470029133681331365612715,
0.00005663171396200798587718548150462559748923,
-2.623108413592736379898996676073649401929e-6,
9.119217817986751054921546193752419564399e-8,
-2.53802301213189115211738040534424356339e-9,
5.891400991925223188035062052931599955866e-11,
-1.172679448695282481776703108966950598523e-12,
// [0.85714,1.]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005710207510816312487640671818235289945871,
-0.0007928439954669941488122434266053714841438,
0.00005508527668549943671897743810428922389349,
-2.553381033742852198264714031376910154865e-6,
8.88308047316440065132655057172659442493e-8,
-2.473959230906028127612524561942855432385e-9,
5.746351050196905212074007234565874640938e-11,
-1.144500365524721871427446238547812251332e-12,
// [0.85714,1.]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005549907968268958512719175995883992497251,
-0.000771193873595900687845692239671684840149,
0.00005362100170865067723174570705592313082177,
-2.487262576143494665332715335532256285059e-6,
8.658857240462782329356612212494857920808e-8,
-2.413048326795273243868771048073223024364e-9,
5.608268458163144810543326360339995756753e-11,
-1.117643142854731361479720195679194723769e-12,
// [0.85714,1.]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005398357115171304297810273233972348317208,
-0.0007506940239200430801261496185592261737322,
0.00005223251409906303076640186632300294115624,
-2.424480069986028397391713904518274390169e-6,
8.445669077592843430298382695215930273667e-8,
-2.355063247925263260203594735322397071417e-9,
5.4766632104196630097131915100388512109e-11,
-1.092016941170986715333069613200958109665e-12,
// [0.85714,1.]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005254858167440301055110745779754854266004,
-0.0007312551973858399333323619541690569268787,
0.0000509140814697551507558492636118677972688,
-2.364787383426287983893795288582951728544e-6,
8.24272130299899150490448113380780223212e-8,
-2.299798223040766758824148523836161395516e-9,
5.351090178373783915401978019656805673865e-11,
-1.067539054638888095910012469415849877941e-12,
// [0.85714,1.]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005118786381700879038405238930344477901762,
-0.0007127971405755524692172488660696099768573,
0.00004966053505199956037019221861080461086839,
-2.307962005625759823647008417197609732062e-6,
8.049293717292194456047489757597159373784e-8,
-2.247066326431084483607917045314597743373e-9,
5.23114408974677316199772413326444851501e-11,
-1.04413402102753756341386603532299860557e-12,
// [0.85714,1.]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004989579984028866799994806079581602542778,
-0.0006952474907269960795383768612886009688945,
0.00004846720211818747104524392026150188634472,
-2.253802280752819862734962068386975201098e-6,
7.864732080488923773254039704956179522018e-8,
-2.196697369611932389230460942119364117752e-9,
5.116455167066771245146407137139437170629e-11,
-1.021732845971195296901931927321069204216e-12
}
,{
// [1.,1.1429]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.7335421815701841994914580532034651913534,
-0.02606822176514257311507346801919614813014,
0.0009757800094048974575233053883686921035498,
-0.00003075132642067823000786201862796140703563,
8.1377350487499674884232543734594482899e-7,
-1.839907177607305420194587591592019753082e-8,
3.618301957616000500348605593759921428567e-10,
-6.282586938894203667979394405577956390631e-12,
// [1.,1.1429]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.1824775523559980094648555876243945499575,
-0.01366092013166322870776902236375677103315,
0.0006457778548344675344566459041741234626956,
-0.00002278565834992948930103895066792268354755,
6.439675091664912168508454204917869494721e-7,
-1.519481928727646227637816372262983770231e-8,
3.078563474340399697066002541396773139013e-10,
-5.463846351640445438500431728185241538308e-12,
// [1.,1.1429]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0956264409216425988568098207155438187868,
-0.00904088996767782669720526634287795052939,
0.0004784988253487206433283287900715093242256,
-0.00001803109044541468150171559102961723210149,
5.318186723697616959670809964035359617767e-7,
-1.292815455439881471789759429511538661179e-8,
2.677370629732209041372473025681010382998e-10,
-4.832119411010565264763591882164696629437e-12,
// [1.,1.1429]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.06328622977374478498052066640369652153871,
-0.006698983554877860282969035925420716018482,
0.0003786528993538907043805309794381999909016,
-0.00001489092299550366145789106441252443892852,
4.524854069720468057224687589170581632578e-7,
-1.124333279278034984011425344050580945354e-8,
2.367816332721686992539905528172097862406e-10,
-4.330289184264966617543801887637483669929e-12,
// [1.,1.1429]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.04689288488414502024468174347202973262074,
-0.005301140590950639615743742090857598587884,
0.0003127093829057435571844529673950257768862,
-0.00001266959154842841242352176081857416932285,
3.935166455250847360339058877132353915978e-7,
-9.943357762670649017584711830379182019602e-9,
2.121912811710727718267927714056214323345e-10,
-3.922241613693081626778802807323857668871e-12,
// [1.,1.1429]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0371079841366544757120464335175591182891,
-0.004377931360676909806483900188822425085737,
0.0002660614225171500849984277804686762411389,
-0.00001101846621470332088173232091882590854599,
3.480175196478118500615916699738946957428e-7,
-8.910689792501847284092044580407342839685e-9,
1.921963851988279476029520245126844440165e-10,
-3.584058926305823612094588846076681447289e-12,
// [1.,1.1429]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.03064551952473836716495414267631167607124,
-0.003724859915236879287365232264342634161028,
0.0002313877905089118608176859212178046938695,
-9.744490679015939338401258297337854956828e-6,
3.11874140842594892850135837829869661076e-7,
-8.071010950099009840381306119702466978831e-9,
1.756249513044793873761021782211082327861e-10,
-3.299288784584967291462526959633282503403e-12,
// [1.,1.1429]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02607401940665815363276022627359801105115,
-0.003239429067121781486471098178371010628466,
0.0002046343042594670912354758804683892604375,
-8.73247606297628818751497248462578670126e-6,
2.824853814885915595228499466248247696649e-7,
-7.375101865323075703383948905800421218544e-9,
1.616707980522898976112635017851421177267e-10,
-3.056251155738609015604477580086082920047e-12,
// [1.,1.1429]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02267600346985246911512566044117748588571,
-0.002864880259629759612735100912300034428765,
0.0001833819973226259090886209346345666008448,
-7.909590792868115894238153428646952259631e-6,
2.581285636348738822927645840771420282181e-7,
-6.789106111488255631413619113850704550547e-9,
1.497615912301057932765820151209134523699e-10,
-2.84642970980218997867229219701626079215e-12,
// [1.,1.1429]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02005416181740831607692733633739160915486,
-0.00256734796251416173004277102077844960026,
0.0001661014066503468073444386757258459637328,
-7.227599885817271110031728715702848328947e-6,
2.376187123504365925917212565802205618031e-7,
-6.288988034108581756801578076873816599742e-9,
1.394800210781145048614303702575095728371e-10,
-2.663471297130462244887581250955498698507e-12,
// [1.,1.1429]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01797143573759913096717687605373984183046,
-0.002325419693102411460756812129613527277777,
0.0001517795976022724336437884902842957664573,
-6.653324043566772364603681402128922615816e-6,
2.20114579730591351042176812764312813513e-7,
-5.857222436085146451288266213151701244433e-9,
1.305147758379729259626114523031139269318e-10,
-2.50254173523764313224809451917916542018e-12,
// [1.,1.1429]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01627793785171687914383832767533500109785,
-0.00212491436642950952668252816513652938001,
0.0001397198049150060402982486384645494242445,
-6.163208324639151916033117621127596772433e-6,
2.05002783878700558591570726928219863098e-7,
-5.480735627038051103525138042278139533499e-9,
1.226289747305316083750833811209166880822e-10,
-2.359898425017604911939124224369932897899e-12,
// [1.,1.1429]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01487440056500656566068438830326331024667,
-0.001956077268807904333237713017779818623071,
0.0001294273748175206957123815010493984304066,
-5.740078035813634030223550691680227983161e-6,
1.918257456329213018191381933719020821369e-7,
-5.149579717508130155745606333001127536128e-9,
1.156392257482616751699887215072012387212e-10,
-2.232599317241670510041776662822664969457e-12,
// [1.,1.1429]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01369254088165532935655549053789012173512,
-0.001811983247443221127401376217362528745256,
0.0001205416387521800215457833411151671864625,
-5.371120960467043928529758303282516570056e-6,
1.802352888633551139746377034374000825124e-7,
-4.856053122285741749007799906295317848137e-9,
1.094013647275032394825221191039975008652e-10,
-2.118300120437742314053690733755978128191e-12,
// [1.,1.1429]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01268388273210254696105147654664888603958,
-0.001687582942528552458818196149867008906912,
0.0001127935401698972757561760994470911439957,
-5.046588166888367636782322632629426553642e-6,
1.69961858088620408648505290621212270591e-7,
-4.594101655526608592656349798444793223657e-9,
1.038005183273069743672088774026514832494e-10,
-2.015110083440970838601034019304075550969e-12,
// [1.,1.1429]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01181308059769986632230001793120744324585,
-0.001579109562376685444400249223907141551063,
0.0001059783515047411058531654327453024427143,
-4.758932101538700168026712079282796959367e-6,
1.607935568049545870609145532705563331822e-7,
-4.358901215030922630761906604655864976102e-9,
9.874403722185073195127789343319790291599e-11,
-1.921487547765724041801059972633429547748e-12,
// [1.,1.1429]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01105376693663679725919655008492797990427,
-0.001483696921064582388970471055393007004032,
0.00009993757413239445806405398813205036540925,
-4.502219662263100875810854582888743304914e-6,
1.525615414360183242249926801781959215235e-7,
-4.146561005159998690378638467409788449053e-9,
9.415637805267280539789353210918821036273e-11,
-1.836163053651669623671152338947262208293e-12,
// [1.,1.1429]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01038587844745207590592957576895141031889,
-0.001399126037851805569689856627125124482683,
0.00009454661290760353772964626171985410122173,
-4.271723228882869221257216890791425630081e-6,
1.451296341350054231473542855730699912525e-7,
-3.953908600374910986835636037482226540749e-9,
8.997533553860500567591451377594373065176e-11,
-1.758081887000867393887694976872222130453e-12,
// [1.,1.1429]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.009793882264962638202987222974864263638456,
-0.001323652580704802724086613577904896947035,
0.00008970618780661559886170923355049294560488,
-4.063629821652921165246328441316576610148e-6,
1.383868000085157732640511627025531114212e-7,
-3.778331710169190622212909202178933946713e-9,
8.614922720951169197653062762434803679123e-11,
-1.686360571233547462933115425784586405406e-12,
// [1.,1.1429]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.009265568064933618313373843855713475036431,
-0.001255886629291036136432699230798979201734,
0.00008533622625478384717633495537424229365331,
-3.874830463528929113248486554956093860498e-6,
1.322416088892157961406311832816306268482e-7,
-3.61765995039367229765108889146916198466e-9,
8.263476145591641849840610968107569322263e-11,
-1.62025351303926309232928809631583773219e-12,
// [1.,1.1429]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.008791206405037252227255585646178376932885,
-0.001194707167565451305463196358244500307351,
0.00008137143973417737799347757147246930904527,
-3.702765109800804296736686248187714705274e-6,
1.266180973322206394810499462544218184079e-7,
-3.470075311037795816747176221623809500199e-9,
7.939540313061931547751090556927426766554e-11,
-1.559127143681651443291177882476702450441e-12,
// [1.,1.1429]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.008362950172958158436003780162709650543833,
-0.001139200156277016094750319947652021197899,
0.00007775806730588430550174421089804182245292,
-3.545306783990606980945046420054749592803e-6,
1.214526349874496615104765544450980132367e-7,
-3.334043519093322897183544529377442460354e-9,
7.640010644084867392065508030607398632271e-11,
-1.502439663506575236955227979155991763968e-12,
// [1.,1.1429]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007974401093939111984819181377956437922992,
-0.001088612942280964558052970071490712338981,
0.00007445144246386787652705903157261898977541,
-3.40067383627787432438902379043873903713e-6,
1.166915222998644429056877855221721631811e-7,
-3.208260826031479926265530522092452720238e-9,
7.362232240380457381138052630205503413017e-11,
-1.449725023823707362315658742296125983479e-12,
// [1.,1.1429]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007620290595966751250183814546524736964553,
-0.001042320194492782544557539045720885958614,
0.0000714141505618983551045985041427178505973,
-3.267362679105785529266521569900131764216e-6,
1.122891280711751564430014238944036405332e-7,
-3.091612325727974046000363046157478121089e-9,
7.103921393846224543409883930355313371484e-11,
-1.400580147710641765738692615650950184206e-12,
// [1.,1.1429]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007296241361449477176550661700516472685903,
-0.0009997981078652540929227702344262569617974,
0.00006861461626128249024981608260960746892494,
-3.144095638908538678371738866081451123299e-6,
1.082064305872212953029087802886155872335e-7,
-2.983138992173278184686499080498250678671e-9,
6.86310296684444312676749091814622739432e-11,
-1.354654651197154663455326381097697268507e-12,
// [1.,1.1429]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006998586755056778034660967129158207298285,
-0.0009606046276566739882094672493141320776174,
0.00006602600841713842921742246496162307417641,
-3.029780107677685710907312150936490814462e-6,
1.044098639378358653266902790585742731481e-7,
-2.88201138231728460208068174483205419701e-9,
6.638060024655848097786442839119383294641e-11,
-1.311642512533888533957067486885075301882e-12,
// [1.,1.1429]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006724232393596717320054908223120198634675,
-0.0009243641178386965541466339016383318433291,
0.00006362538226128875173362053776515706451224,
-2.923476239918064187227322312633655224002e-6,
1.008703976164117233725645655663422817171e-7,
-2.787508484249811381219533165477346376115e-9,
6.427293013787616840162965287406290506792e-11,
-1.271275272307857327625507024698399563023e-12,
// [1.,1.1429]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006470548824870875298936812788375388359951,
-0.0008907553516568381378389791438507546177457,
0.00006139300103833503684089321993758822946167,
-2.824371181435436512692089275808401706715e-6,
9.75627962062221687404303236851434175315e-8,
-2.699000574185165396244529013787641631775e-9,
6.229486441622795688271157965420229206269e-11,
-1.233316446195886586862431485059274836289e-12,
// [1.,1.1429]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006235287461597866401129554508802251144797,
-0.00085950201453552105010506423155563058619,
0.00005931179481019828642030980180938938816491,
-2.73175834055328605162215307199179413252e-6,
9.446501937488302897620345733003107806183e-8,
-2.615935223647190435794071227515764548793e-9,
6.043481498149937525253038542230351198957e-11,
-1.19755690551216019570313995978707576586e-12,
// [1.,1.1429]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006016514101748646802443139122676218996738,
-0.0008303651273416394985411193806158980369645,
0.00005736692515167164343517882284333908227016,
-2.64502058795760546765700284575116679922e-6,
9.155773212583133593845519914079309792819e-8,
-2.537825802036318452607020100320568979974e-9,
5.868253420008720473803248029144987050785e-11,
-1.163811035586242936671200756154751475421e-12,
// [1.,1.1429]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005812555891391475956122728911018308954932,
-0.000803136952122297646069176405972795723159,
0.00005554543234716094695302355306496089875461,
-2.563616543738174173043173513334477610641e-6,
8.882390238817379364307879570721102336673e-8,
-2.464241970733348616214491752749954622499e-9,
5.702892666006542800574845022088425141858e-11,
-1.131913523434433689676867969203944181565e-12,
// [1.,1.1429]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00562195866485608300268667585360354264688,
-0.0007776360528591773838814374087186858502566,
0.00005383594741855155847458188822066434427009,
-2.487069309904209959057145860667607123778e-6,
8.624846831032045911285513746221493305912e-8,
-2.394801777828926558643915786206184920005e-9,
5.546589176255671119066838732167691182765e-11,
-1.101716657725594957453973619854419314738e-12,
// [1.,1.1429]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005443452370014241180538039802547304168721,
-0.0007537032638586739022733240615482997981177,
0.00005222845550803704608269991997352223279322,
-2.414957154606022977123965718421512383001e-6,
8.381806157551771581123036346207789457829e-8,
-2.329165047816344334319204805701041516905e-9,
5.398619141621857491468815770948208699887e-11,
-1.073088048246122837001925905276092822328e-12,
// [1.,1.1429]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0052759228470107168217967323481643050277,
-0.0007311983771114972706177197849329631246766,
0.00005071410024677391796913131243930708503026,
-2.346905764969863405128356535684385526577e-6,
8.152077604109729479645235231632790270254e-8,
-2.267027825485625515900224978070798631433e-9,
5.258333828775473289004498876759871829745e-11,
-1.045908690784178206961098082679864752791e-12,
// [1.,1.1429]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005118388639780480412119895711565087657545,
-0.000709997403453838708350683641991953963972,
0.00004928502106441342336143908412346547571202,
-2.282581768996829618355713650518082797426e-6,
7.934597327477012816596907571867935477161e-8,
-2.208117683063449621300276639799703437338e-9,
5.125150097844212723511336625004319348726e-11,
-1.020071317928743334895462672088083771415e-12,
// [1.,1.1429]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004969981824176870487602487379598592436769,
-0.000689990294900815799326337194300025209546,
0.00004793421714897862405636864267717784087004,
-2.221687290579040955053686655296876764638e-6,
7.728411830452445584192471108534599910504e-8,
-2.152189738156546653482901509225402821291e-9,
4.998542321087881043734808398151683924396e-11,
-9.954789877071978745103230257304293746068e-13,
// [1.,1.1429]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004829932064305710135261994629631665353047,
-0.0006710790400847514944682305933043525681084,
0.00004665543310220402244801352651113351020167,
-2.163955350496737340331176131978550066725e-6,
7.532664024664528283225039430498552528172e-8,
-2.099023260049824898390888142628569107989e-9,
4.878035467015038735282049956855525757145e-11,
-9.720438710030289883814499819674809906698e-13,
// [1.,1.1429]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004697553280593260011598435549547969848124,
-0.000653176063429928905184839223387302044721,
0.00004544306236047465358760966393807083188348,
-2.109145964002783242113910172802948429052e-6,
7.346581352614941328862013001151962590297e-8,
-2.048418765427498045689638956711356657525e-9,
4.763199158550219065924640956967184319743e-11,
-9.496862058546340579909311152736515443103e-13,
// [1.,1.1429]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004572232444009501896503216224587611534195,
-0.0006362028730457385930904605867132150145394,
0.00004429206524410066822205652036767973334784,
-2.057042814994813273411613515451391368034e-6,
7.169465622702538701322236765191661996421e-8,
-2.000195523142691269591044734781851266916e-9,
4.653642548942131248265823649809264342546e-11,
-9.283333924528547932670343546153765434242e-13,
// [1.,1.1429]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004453420111320169721305740586959481054867,
-0.0006200889134165227333119747657020855745106,
0.00004319789911493239040968856300778801657559,
-2.007450409821919715699675474582630606267e-6,
7.000684275917011680988858050030840439759e-8,
-1.954189402393596597314050432831128369168e-9,
4.549009887115730201423774478961995678339e-11,
-9.07919207244710272523814234753744733998e-13,
// [1.,1.1429]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004340622393915658711921023488618764763233,
-0.0006047705876081859217782650989120187267986,
0.0000421564586063007554834659588167589233426,
-1.960191631958851619740718423505070034892e-6,
6.839662854455469737225544744192613781821e-8,
-1.910251010428095605969417845593236510724e-9,
4.448976666661790781067725636075402203463e-11,
-8.883831182551941876982660896471455005127e-13
}
,{
// [1.1429,1.2857]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.7084197842180618016762146087925441110842,
-0.02420575049942150751983483815853138885755,
0.0008882299779833421519465754116186783990267,
-0.00002767320197755110148316750103069034040444,
7.26991661096155358050923955717789215563e-7,
-1.635500824947489321661622897940909434448e-8,
3.204712632411284656042802851612604771762e-10,
-5.549307003319095035343627611904529212998e-12,
// [1.1429,1.2857]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.1694402534959505505792681828401431164501,
-0.01243521969176208719359729139489618509823,
0.0005811372415287708513403043141571464794846,
-0.00002035576669881117069896853069286394375404,
5.724252860953445872590595978404186134931e-7,
-1.345798710601218475755730978434329409668e-8,
2.719244792638363906467796364740285502412e-10,
-4.815884948708917100984903895019674092257e-12,
// [1.1429,1.2857]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.08704653784233460851206051922605268839652,
-0.008135921381398639799025757568714967977807,
0.0004274711006752115250955581676601949578958,
-0.00001602790817675577198171408036000555509676,
4.710295463512141255629398507983910899932e-7,
-1.141923369375752028697792205088979225073e-8,
2.359859119806878388982258766765763354036e-10,
-4.251843791847534181936084419559498902318e-12,
// [1.1429,1.2857]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.05695144966979047692566690693226965719881,
-0.005984595409449245606529944099576022669865,
0.0003365860717120312936812951097632385779091,
-0.00001318882744646500562013334813977842860136,
3.996731771470794384398524581599596558499e-7,
-9.909981437697988280469038582246891336558e-9,
2.083471760018981556211844426708803958658e-10,
-3.804989487976773634465325882958171053388e-12,
// [1.1429,1.2857]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.04189216786614471772345834460588292258618,
-0.004712205003965076391803392412601185162504,
0.0002769653763759112548655062764175117177594,
-0.0000111908490945881337610135587019840081767,
3.468493483709325828655207632936934451019e-7,
-8.749290473780017959886228524389392820064e-9,
1.864507201135953918577967777534792409888e-10,
-3.44246245383516682236313080916791283462e-12,
// [1.1429,1.2857]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.03298543502775553334249150930791694471743,
-0.003877515269259688697786497737231632143497,
0.0002350078309864852223474990236958098526248,
-9.711781877141962379913890109774936611748e-6,
3.062251647901170411099518892182776936265e-7,
-7.829751781974642796281037839424838068408e-9,
1.686863952368619391971601966142829480044e-10,
-3.142580096616737198763313775887650540283e-12,
// [1.1429,1.2857]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02714260688481781958844793317056227816405,
-0.003290109633807970435307641943011142537896,
0.0002039474194201056321309890057609252321634,
-8.57430472703134449029596683671072267351e-6,
2.740413107101454373137857339107286111605e-7,
-7.083744676325416693250167603194403872604e-9,
1.539917334395302300711276470743913792395e-10,
-2.890474758570256977812303228004559098568e-12,
// [1.1429,1.2857]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02303076743665579184083499142744785903237,
-0.002855263871878865987555242387772875340004,
0.0001800603992677740414620972696535871995025,
-7.673156804399465903577141629259521430163e-6,
2.479310621272893217462925693318581962844e-7,
-6.466649450905016260140697550090721013039e-9,
1.416382043008465276526814285498859203784e-10,
-2.675620482866563791885950048948963136557e-12,
// [1.1429,1.2857]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01998684710315206078473236420655729435018,
-0.002520845589746404632737582356029158303977,
0.0001611362928924970873468584798637703995761,
-6.942069836842857846411806885256014784566e-6,
2.263327293376261660132687425194314629748e-7,
-5.947870699145456069726599740913161444979e-9,
1.311100246280375913428893016746841960903e-10,
-2.49036097260056029468129755723265741847e-12,
// [1.1429,1.2857]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01764591912822483136969472371725092032431,
-0.002255908100492684854643262866188286505216,
0.0001457834665738017242621420635480236418209,
-6.337316512429060062763119570063220295501e-6,
2.081754731139602574626006783452765600077e-7,
-5.505747664201382415688717273903202600813e-9,
1.220320272846542069047699745045525069509e-10,
-2.328996875642366650140768135513935370267e-12,
// [1.1429,1.2857]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01579135670344879298386275141335877542268,
-0.00204096853203108824288548146782287340035,
0.0001330836467611061312650758294089063281262,
-5.828913332627514099842967522693097504262e-6,
1.927011669687784355259677617610357846218e-7,
-5.124524949502026246033905220490154552797e-9,
1.141249373788213596788392881058630402756e-10,
-2.187199752660154588363496365825008199503e-12,
// [1.1429,1.2857]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01428677972421761675580072184069110468243,
-0.001863171054653472571149467080857281822536,
0.0001224071799852684587450526000673290141473,
-5.395632755657175234512077189417944908202e-6,
1.79358372023731814437821411569345228519e-7,
-4.792474264085600052156962255757546385004e-9,
1.071766561735723471141672310487298156373e-10,
-2.061624268548974287855713403393028893916e-12,
// [1.1429,1.2857]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01304219738257430710231729773678828742695,
-0.001713700519791854509027795083225184708766,
0.0001133082878688866703579894817475547255908,
-5.022034492821708370123957484762369612969e-6,
1.677365980964532764619287603149683135203e-7,
-4.500688445640110303833524212002159251302e-9,
1.010232581033341133403511084295960409037e-10,
-1.949644707070484188308869105259728724865e-12,
// [1.1429,1.2857]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01199590363854298071137573362975855390892,
-0.001586316030162607588022901632185507271684,
0.0001054627243493376508122254690979049705355,
-4.696624818933220895048669663458019138768e-6,
1.575240945070665449347445888902283739355e-7,
-4.242283403902712169670434275174689946749e-9,
9.553607973323267238959522195229616219018e-11,
-1.84917175182030717111889631229355916627e-12,
// [1.1429,1.2857]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01110421221113825230415702339349318806942,
-0.001476478140889009837553892717251555426704,
0.00009862912119767559152496533371837122729506,
-4.410674714889436531400771769404104343268e-6,
1.484799180972219003294448112239835160497e-7,
-4.011855905720594974743715779595208487108e-9,
9.061274183993060930633351817034497040042e-11,
-1.758522432414377537053949373418722224746e-12,
// [1.1429,1.2857]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01033534698622306808714104264196986994783,
-0.001380807696765821275970354577013804899025,
0.0000926241690127526382315696588744113842128,
-4.157437772203022421010580775730450125142e-6,
1.404149557072700246480172426115080468627e-7,
-3.805106537706737824297073898704619344365e-9,
8.617077663762710337560940473004326590826e-11,
-1.67632609603047005582054307341369400205e-12,
// [1.1429,1.2857]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.009665653877360748189228200029217547256364,
-0.001296738366176973044576297893744702903995,
0.00008730619321633475739301860249457367538113,
-3.93161882235975699794218350424804293743e-6,
1.331787278692454207516706052132518837549e-7,
-3.618572075649803718814571426031839851935e-9,
8.214302028128071108345659949144181408919e-11,
-1.601455292842268732691161523388287130474e-12,
// [1.1429,1.2857]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.009077168563238810599925936967899845569519,
-0.001222286705027189587691200489062073748749,
0.00008256399526962325970530968295236579734512,
-3.72900444022005145617339068444654207382e-6,
1.266500217362368861774970492394893489846e-7,
-3.449431988986596779129503864086400941887e-9,
7.847422617550330410769797758408779241586e-11,
-1.532974210277401743696056885577386268599e-12,
// [1.1429,1.2857]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.008556006935190326429790490283528397439744,
-0.001155895933773290019864636547676120887638,
0.00007830909324468674953220861773484285397138,
-3.546200666039799685093049376979424591006e-6,
1.207301187389420297655938037214013075559e-7,
-3.295366214399646133056324820065056360002e-9,
7.511853819395970329687497606203797014531e-11,
-1.470099675472233998533093474969484158369e-12,
// [1.1429,1.2857]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.008091271536413029480940497264822938325973,
-0.001096327305424235447093857723587669312761,
0.00007447021398689897247544255415670576650028,
-3.380443379852738365274634305920247299384e-6,
1.153378166615970443930785613786377389586e-7,
-3.154449042229185358340395232427069847264e-9,
7.203757975377500241064283061530146305791e-11,
-1.41217129600022085983150419009676339578e-12,
// [1.1429,1.2857]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007674291137969647495588827494752977486555,
-0.001042582995815258855386737320715815830806,
0.00007098931097696837654459000105508194986228,
-3.229458919595577389931709051526653792636e-6,
1.104057156664072055750532901549833597385e-7,
-3.025068866269160460063290974099523530949e-9,
6.919899067532593523066109519789258764768e-11,
-1.358628337403371139476560839773366878717e-12,
// [1.1429,1.2857]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007298080970706811375990115844372803838948,
-0.0009938503536762789848086802151419966403583,
0.00006781863731156585034296501909991415301693,
-3.091360089791347083294966013972920384635e-6,
1.058774095364160131001334845193999570149e-7,
-2.905866738663223063674879200146957626945e-9,
6.657529415292275595432617738477462330026e-11,
-1.308991630456363124364942104026540905131e-12,
// [1.1429,1.2857]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006956952475733952302774650475782940768294,
-0.0009494609223606886780432803901491801206619,
0.00006491856188567501412375252485446768564326,
-2.96456751634917583170433565672624171058e-6,
1.017053350968720165696977729495691971242e-7,
-2.795688788033405244092685467088691306162e-9,
6.414301018814678433586295054048817697967e-11,
-1.262849277694353722432480845700636529341e-12,
// [1.1429,1.2857]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006646226456524820174877410472305962280788,
-0.0009088598663982589663036391038136973139346,
0.00006225591784338754961719200403698855233679,
-2.847749430362116178878448599547349736752e-6,
9.784910684973811110381963831595577498221e-8,
-2.693548987952505172992745091881436685377e-9,
6.188195519154762616510846135045157379627e-11,
-1.21984526082128347483688759297348876058e-12,
// [1.1429,1.2857]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006362019064787812210920884409598010427507,
-0.0008715828498062736958536914645371546415487,
0.00005980273803765754768646067188839155466358,
-2.739775037873047473153812898671933808047e-6,
9.427421387022963201440380446957095451229e-8,
-2.598599743648657634556600703839393574095e-9,
5.977468373093963855790190924949190838949e-11,
-1.179670285248628500759913915953369035563e-12,
// [1.1429,1.2857]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006101079948643915334867285551494247650591,
-0.0008372383325260904024308508448506899481895,
0.00005753527579538546363849504637840234643934,
-2.639678032977452905201870779461158297476e-6,
9.095099034147807221384125198871885849922e-8,
-2.510108448214083936743412540675577950531e-9,
5.780603990180281234384380962658480787512e-11,
-1.142054365942961448477458377007108796657e-12,
// [1.1429,1.2857]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005860668327682632296979467784659182888731,
-0.0008054938611343156915027298884849334734094,
0.00005543323869257643478516457048614207594676,
-2.546627772793767385122405617028198006511e-6,
8.785379502184040757780759710433270924868e-8,
-2.427438642429558495120878766293050534279e-9,
5.5962794023962144503570993000694498759e-11,
-1.106760780394510017021960386874852501153e-12,
// [1.1429,1.2857]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005638457027940209335619959740943208882606,
-0.0007760653416950216889721428842956428333497,
0.00005347918322871758567147339548533661906661,
-2.459906302547844708414303596335082386306e-6,
8.496035183875797084900086816481837525844e-8,
-2.35003475788681513128760155420550652826e-9,
5.423334632890788972347148667177303852134e-11,
-1.073581103607302349286410403415412327374e-12,
// [1.1429,1.2857]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005432457391865151332187632818978889285017,
-0.0007487085652010283183778821773534977162639,
0.00005165803235355183840241093536932232696347,
-2.378889892200845786118390681012533804293e-6,
8.225121589804278589295860721896318684965e-8,
-2.277409673394300657393251800245970548558e-9,
5.260748366754847833619224095238096049821e-11,
-1.042331105944323842554775717861254213547e-12,
// [1.1429,1.2857]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005240959956407197751524516093738488895634,
-0.0007232124529487366487998671691030446086702,
0.00004995668773626356534346783334668589999579,
-2.303034084709122833403378509819675877807e-6,
7.970933795808070960875234092418330969748e-8,
-2.209134497981240903614918702567292971497e-9,
5.107617849897054277700008868158718698694e-11,
-1.012847343933614288772611104338850574935e-12,
// [1.1429,1.2857]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005062487170641156077252998582553444822548,
-0.0006993936283067280354485211134055002819241,
0.00004836371577893615696965093268350337440857,
-2.231861501301796056151828010123211080565e-6,
7.73197068349752267595950194876475617407e-8,
-2.144830129493921209187819466630719585631e-9,
4.963142183518928898079106814585685427869e-11,
-9.849843113003778470852483745649794470747e-13,
// [1.1429,1.2857]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004895755398147095795902445421993782879078,
-0.000677092020904170071888066305374433994504,
0.00004686909152738113218984207945057178530674,
-2.164951828824686422129329733724494354892e-6,
7.506905395341851978117216149427404989133e-8,
-2.084160239172653513504982531588263240768e-9,
4.82660836377134871008976810927385636336e-11,
-9.586120457549093578220657710981768095867e-13,
// [1.1429,1.2857]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004739644146329190062473094129122218521181,
-0.0006561672813824241365504205795058247285573,
0.00004546398840536072646167639019226458995573,
-2.101933547164626700121828188623111319846e-6,
7.294560780688637929974844455023397962224e-8,
-2.026825409065098347013861960472185921714e-9,
4.697379554675440037359308880233621956865e-11,
-9.336141087393512777938754495831803976427e-13,
// [1.1429,1.2857]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004593170969676968526034037045352933379112,
-0.0006364958376741616280166563934756064716847,
0.00004414060449049842354555790681773395038515,
-2.042477054134852027425383865208544312847e-6,
7.093888876710537872460915636563642221905e-8,
-1.972558207295542304974527796661914768477e-9,
4.574885188592115462297269677713231260092e-11,
-9.098858720818991069374954763857271014226e-13,
// [1.1429,1.2857]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004455470863719130976694039634439078327237,
-0.0006179684628661114109984499254215755083183,
0.00004289201813687215736389205515698727458194,
-1.986288920140023788821942902726090229852e-6,
6.903953671847835986550994280559918976503e-8,
-1.921119030809398506359699414280677540689e-9,
4.458612570579988499256181373151476273487e-11,
-8.873330585385644302698650512871288606956e-13,
// [1.1429,1.2857]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004325779240062779467472203466173609057416,
-0.0006004882539153646435864377549527870823182,
0.00004171206732297981340972353058716806534904,
-1.933107061940094449419374672976436102063e-6,
6.723916555414261370626716161345603783914e-8,
-1.872292579671671754606506360094967063343e-9,
4.348099726837304263723720308490991760696e-11,
-8.658704934128146629763485838499252661649e-13,
// [1.1429,1.2857]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00420341777740755210503691062015309678342,
-0.0005839689425208917980180650808865038379905,
0.00004059524830078039019340377718923336209526,
-1.882696668539894739289450182195127584355e-6,
6.553023977641673401003425368328346157935e-8,
-1.825884853814590817855508007700039169759e-9,
4.242929287454820635234881548877876179115e-11,
-8.454210324947320223365633608774138506568e-13,
// [1.1429,1.2857]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0040877825976462421950811484540962816201,
-0.0005683334762101189218333690684956009541293,
0.00003953663003937533008281293190478174863455,
-1.834846746001609456601105480049140876046e-6,
6.390596938296824280366798884813587230295e-8,
-1.78172058414035925738482573106628662344e-9,
4.142723233159070621558425978252290417466e-11,
-8.259146379490952953488880563147023899923e-13,
// [1.1429,1.2857]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003978334333470832070413329178437248018376,
-0.0005535128205504662704187729553114228659673,
0.00003853178166607051113688840310309891915889,
-1.7893671742574397534730040871505000153e-6,
6.236021995541029964010205801017980372063e-8,
-1.739641026453113132428181919252658646643e-9,
4.047138367026277279869802688527316600444e-11,
-8.072875788791599900788813717476906676295e-13,
// [1.1429,1.2857]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003874589743853263518764141334624803855019,
-0.0005394449433242161933541545003138661313687,
0.00003757671065944215508042573962861059567671,
-1.746086189590283888381445204758964801087e-6,
6.088743544692059597386852737376691310276e-8,
-1.699502059838993186504033942461879352465e-9,
3.955862397124970040689949521161947559622e-11,
-7.894817373830458612996911634817573905677e-13,
// [1.1429,1.2857]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003776114603269512987216284669517449548893,
-0.000526073949231435846650219349166591853618,
0.00003666780998143112308009450744393726916586,
-1.704848222687042662047770551126104432418e-6,
5.948257162554420901943764429965063116465e-8,
-1.661172541602668471743972151638568028736e-9,
3.868610536087535614068432239181979455706e-11,
-7.724440042188363385809740101762988642917e-13
}
,{
// [1.2857,1.4286]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.6850753014462695856658762429662164697451,
-0.02250948207335271348746318882454459893688,
0.0008094134642402922117042569505830615930351,
-0.00002492256631311370969047286132064295584832,
6.498347100384602517863520075734337755573e-7,
-1.454429414404679926886118028570700414504e-8,
2.839348446646229208977877073094742278839e-10,
-4.902918938457128829965234029939952928195e-12,
// [1.2857,1.4286]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.1575663745134689925996701321632338665023,
-0.0113317884993599448560053023628974832584,
0.0005233738925755619112448067472929889304847,
-0.00001819537204692249703327642367277717607078,
5.090502927215281919445022297457549024558e-7,
-1.19236713497650828741743206747287323043e-8,
2.402504523496071351368269751640351011e-10,
-4.245685527581999741756451757510593052797e-12,
// [1.2857,1.4286]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.07932251949551961237225804906579638194847,
-0.007327234496054212598586610871011166772122,
0.0003821028129855279372322493823423260231193,
-0.00001425340834237031397224379935606634406117,
4.173284951684448442215875647035590485461e-7,
-1.008911578295310655995343341604255554409e-8,
2.08045225529978501689923384737609917049e-10,
-3.741922889201201103510223749368088605924e-12,
// [1.2857,1.4286]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.05129064147237948672637966106171264028698,
-0.005349439381794125634516255863452019681325,
0.0002993215751899171118549474139490650154441,
-0.00001168519799533699294988191217435472981403,
3.531190505297740962467826816880406214454e-7,
-8.736645508266290417379731803471461409943e-9,
1.833602170533385666772504009679140202477e-10,
-3.34391699917699973617476989572434405308e-12,
// [1.2857,1.4286]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.03744607567255887810672343746906617887349,
-0.004190502052655888681961135318108083077833,
0.0002453891579022050020530313834710550256573,
-9.887333532870016418926578391462084507328e-6,
3.057825910806470385411993876951492615086e-7,
-7.699995961078679124857105530988800629437e-9,
1.638574007244552903132141335304178032873e-10,
-3.021759061275125826787051937177887393749e-12,
// [1.2857,1.4286]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02933351436859121954696769416304813316743,
-0.003435448210628178879473951166509355609753,
0.0002076340041903881143692143990420307575198,
-8.56191265790499560211219803676108591782e-6,
2.694998570674881128854084318224533617227e-7,
-6.880977414583452972373362285557275944994e-9,
1.480712188626080565361388459413814069715e-10,
-2.755786649536466482622313620001111318679e-12,
// [1.2857,1.4286]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02404813747439725102157747530763650384703,
-0.002906876058662960442903544696487665578524,
0.0001798001658161138432509983696385092452208,
-7.54599609681684216705076389788999471321e-6,
2.408342080579417060655001800540322947444e-7,
-6.218041485945362465369927759176943005709e-9,
1.350381937699315660999205608454931474654e-10,
-2.532560384727098651143224488743329567098e-12,
// [1.2857,1.4286]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02034813241064072204482960694081891768716,
-0.002517202321423306160336838950462508387732,
0.0001584659180332550135632358363113256352702,
-6.74335791712896408617998490962782015156e-6,
2.17631450657042827456739645370219228917e-7,
-5.670725669961380794759057271259334274132e-9,
1.240997822039723617803399336267830616367e-10,
-2.342591956513395990772383714961079430798e-12,
// [1.2857,1.4286]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01762041624996314213580971654736284236102,
-0.002218522852463442303131877983687437988466,
0.0001416105162598029549183583301779997401986,
-6.093680703513410115523347464168862618331e-6,
1.984753971858563286631103214791711853723e-7,
-5.211373732200538769168475145734713125451e-9,
1.147910468118412408732381619700538769498e-10,
-2.178996674380678094080979414003815233638e-12,
// [1.2857,1.4286]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01552965996724409519590180594203942343805,
-0.001982547227635252479842907933666117044765,
0.0001279672947738705109336580854183199337826,
-5.557311200760235732838055001316963820207e-6,
1.823980794417017490548186502939527536658e-7,
-4.820460221548431972577116539932813111327e-9,
1.067746300672326606679630251137389687438e-10,
-2.036660639021570396922347123630702399802e-12,
// [1.2857,1.4286]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01387783059344676648643431587476038841483,
-0.001791542126832320286552291880204557901599,
0.0001167035352160487075646266871801308153329,
-5.107146299042969521370695551984505904767e-6,
1.687161066374293035504299384808423683464e-7,
-4.483817575523823962731355163217682123913e-9,
9.979994497009436453706381494696462366485e-11,
-1.911707509773758677619041694308634832056e-12,
// [1.2857,1.4286]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01254079488782624118111910484466437774345,
-0.00163384949302292300726017236190373807205,
0.0001072500722799815360686122673939621710052,
-4.724051056204591846739226677996552813077e-6,
1.569336140876489309720755324006889639372e-7,
-4.190922261655398963365244480145488418137e-9,
9.367704617768029860665654945763577954803e-11,
-1.801146528828786622803906464931172353374e-12,
// [1.2857,1.4286]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0114369464511604602688592871537817740937,
-0.001501501011918078808374972808991454428122,
0.00009920507218037149752007629411208386730903,
-4.39414126096262819339455838599476478741e-6,
1.466822781570192953210256782863060362973e-7,
-3.93379745445844313590184680023277830367e-9,
8.82593828622787497565046486753049183233e-11,
-1.702633927454524673389485950504209518192e-12,
// [1.2857,1.4286]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0105105070834265509152395639703514519112,
-0.001388871010523624523611448858472615004772,
0.0000922769664802865574214508461613363942309,
-4.10710385145477378927778609600233787873e-6,
1.376829099545043615674663450976512948463e-7,
-3.706288717546782513475539556022414352091e-9,
8.343210738337293644727798537438985298724e-11,
-1.614307359312184934088316154845671945471e-12,
// [1.2857,1.4286]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.009722097073665370956853593071205454769536,
-0.001291877530722513133094167325816179041472,
0.00008624918088061825888391653790949509138612,
-3.85512153867349564477174599556420185734e-6,
1.297201042073437170940185726117263749642e-7,
-3.503573011851680267010183823161902227828e-9,
7.910396235219451753589442687097270329987e-11,
-1.534668598422937401286452729622568399973e-12,
// [1.2857,1.4286]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.009043142715057591255058438247829043066002,
-0.001207488532327227430565842233321825678832,
0.00008095755231220836255610174831957810263006,
-3.632162974933894437378761036367209249335e-6,
1.226250545487524998629884774092227292819e-7,
-3.321817984081509987753875907826246877258e-9,
7.520153270881129213742177040857827604705e-11,
-1.462498876480142091892391567662596452342e-12,
// [1.2857,1.4286]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.008452419726290591366455343560745451528921,
-0.001133405732369553043064032134163122721645,
0.00007627542247367394404979072072282160935078,
-3.433501581926875490531637488927127328662e-6,
1.162636286140386636902935879873416648387e-7,
-3.157940577258820555866190489545577994597e-9,
7.166509715858521825550702980866323431518e-11,
-1.396796751000268076927632876587436531279e-12,
// [1.2857,1.4286]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007933840126586870680643548858806616246545,
-0.001067855914630129840084035763477201975791,
0.00007210353322052398286926082059174343033491,
-3.255381653408624071613601425154793001477e-6,
1.105279194094219246062147426967049112262e-7,
-3.009432808409683866094676831938179950497e-9,
6.844558364220427764643324059106571899504e-11,
-1.336731816900752845953245817309509053754e-12,
// [1.2857,1.4286]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00747499140241090828437350921100562896797,
-0.001009449465086084212729154698457789386932,
0.00006836301472163834242650892237817866801839,
-3.094781793526170300404143151135232359786e-6,
1.053301475311775071859914307312956342155e-7,
-2.874233911414745126497792453464865060937e-9,
6.550230114995452816880938758257310782177e-11,
-1.281609747170385610848736592685648901005e-12,
// [1.2857,1.4286]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007066146255602588915608750042743380485195,
-0.00095708220610173482003088030435928781751,
0.00006499041766410463215702520446704688078508,
-2.949244178952369759371158475284565062743e-6,
1.005981861654357262184788496639060247251e-7,
-2.750635083233844002008700847557851583577e-9,
6.280122667354094424103796103725161165609e-11,
-1.230845558939767615887404970270404419914e-12,
// [1.2857,1.4286]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006699575442712143187775004958963180047899,
-0.0009098658472963086787630785160412731906828,
0.00006193412775805279958432207112194615514251,
-2.816749258879484041384154313098229781638e-6,
9.627222720605374816671814453903903927019e-8,
-2.63720754363681603789952588367150740589e-9,
6.031369521147133061800647846922655499409e-11,
-1.183942935218054914907184667227626943704e-12,
// [1.2857,1.4286]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006369060931074160218465688139009633008814,
-0.0008670777886116254681291089041387856889752,
0.00005915173443652032122873256282777442287765,
-2.695622406318958718107681024580451922245e-6,
9.230226334520155005887396180966680865018e-8,
-2.532747521490626254378485450602280139253e-9,
5.801538650884765794214850403593454139153e-11,
-1.140478062153731551190579074802600575357e-12,
// [1.2857,1.4286]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006069544520281377762256648056468814833745,
-0.0008281242821102102149417492134465462313655,
0.00005660807053274753947132601802804819048681,
-2.584463416637331915915397389847379105616e-6,
8.864616259341789326559855075832833923967e-8,
-2.436233702614046575652662814543688434192e-9,
5.588553306299042550734825228276464413072e-11,
-1.100086873152519522790673733482287197352e-12,
// [1.2857,1.4286]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005796869974771471006969589830418868182084,
-0.0007925129874574280196372810891195818732223,
0.00005427373174943174226884606181567660263897,
-2.482092594117404808440569862398179841276e-6,
8.526817895452909029383892199352406757884e-8,
-2.346793969822918302976105362984056409635e-9,
5.390629506903453966164772330254757913087e-11,
-1.06245489140823957959577197405615898596e-12,
// [1.2857,1.4286]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005547590912201995655773871858012942512865,
-0.0007598322444910411802490200438495040875165,
0.00005212394447651174319040117949758615790249,
-2.387509050855648508793598404024277738002e-6,
8.213778832723728295184731409621123973323e-8,
-2.263679153703541232690140207503374636397e-9,
5.206226269084832591407641215221379216937e-11,
-1.027309074244805080269673611787120317119e-12,
// [1.2857,1.4286]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005318825711437287795003820672182800510522,
-0.0007297352226701933193652460621851452308496,
0.000050137690068013425903943379852443147053,
-2.299858112006418085969449380921123772531e-6,
7.922876978219239601350679946002344266516e-8,
-2.186242130465896351474266390636392919441e-9,
5.034005642312070583529742513323208503507e-11,
-9.944112140975461691081523139209386393541e-13,
// [1.2857,1.4286]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005108146558691352782866382542486529652355,
-0.0007019276609512470121917772674653770034182,
0.00004829702035217823831563589230325293225969,
-2.218405591539757191141775765664991828949e-6,
7.651847398685766740388702821238773112289e-8,
-2.113921039168048848545606800544990228489e-9,
4.872800373069134212314284180869013460174e-11,
-9.635525605184468225188449759056188820357e-13,
// [1.2857,1.4286]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004913493626658728645880903836872145581457,
-0.0006761582849295827088120701421142850409463,
0.00004658651742237708955261845944706941241734,
-2.142517308137460543587592310954662463795e-6,
7.398723580836601244809798341729668657233e-8,
-2.046225702222670712889234798166938303643e-9,
4.721587551957901225977850267322707716891e-11,
-9.34549407744506246526867698025797625203e-13,
// [1.2857,1.4286]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004733107994507078534701298188222704274438,
-0.0006522112439123932955035663098535476663237,
0.00004499286347092766202672499379174272802961,
-2.071642638072909823368502802806226213548e-6,
7.161789903125019078588577420307735299554e-8,
-1.982726558546379045916121948884896460463e-9,
4.579466992182807031834652342576375717609e-11,
-9.072394516281509768157423919216313347957e-13,
// [1.2857,1.4286]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004565478707386752653331697986960435312439,
-0.0006299000885921264665844480211715076755249,
0.00004350449539957096506328322425890642057303,
-2.00530120730740015578566277840973125205e-6,
6.939542901767121406686530637502019060041e-8,
-1.923045583657908594344349562289534146354e-9,
4.445643378008502457521151923654928514841e-11,
-8.814787639708479952561183002568877992667e-13,
// [1.2857,1.4286]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004409300620144884862054770379077581183826,
-0.0006090629355931564776844595946715836161256,
0.00004211132535349419097601892850227739808412,
-1.9430720459764368210971110757927156159e-6,
6.730659491085569993776489599353968410185e-8,
-1.866848792977907640948712239002318447521e-9,
4.319411438573649675225016747388177709963e-11,
-8.571392656356336081704835688899795053774e-13,
// [1.2857,1.4286]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004263440549152094950328334326189391595296,
-0.0005895585549481041328652144407324865064819,
0.00004080451296554294588789036826900417710657,
-1.884584690085898299661878518620300823137e-6,
6.53397072505898379484773119882011420667e-8,
-1.813840015626254443175480071514702162106e-9,
4.200143565787199570095772883260357317671e-11,
-8.341066051475455316144177071726205502814e-13,
// [1.2857,1.4286]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004126909884636728546628244642481175984188,
-0.0005712631815168080178596956023558504038861,
0.00003957627849184067360856662681032392493021,
-1.829511834745797404071063488657279279007e-6,
6.348440005612640554522448770101062204551e-8,
-1.763755694609664019427108329915367233419e-9,
4.087279419169503122640128514724106387905e-11,
-8.122783689000771002485670532978518809446e-13,
// [1.2857,1.4286]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003998842270617655751125521438300362275405,
-0.0005540678988849964358384579339623639297855,
0.00003841974852969763934911680512248489317937,
-1.777563232491618466310317028227350091794e-6,
6.173144883275180643777719120370310918807e-8,
-1.716360521423859871219370728804494180877e-9,
3.980317155607665474504031346949142191047e-11,
-7.915625640701164796389805812924655240272e-13,
// [1.2857,1.4286]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003878475292194974686050300282277518635653,
-0.0005378764794150131806766068626145250888242,
0.00003732882788235901060107057629442601101294,
-1.728480597468143958474124462093000448124e-6,
6.007261778286276556515357792533302140224e-8,
-1.671443753035179293182705209913731236369e-9,
3.878805995419646829311668504126221646046e-11,
-7.718763269958625550988030296706399471297e-13,
// [1.2857,1.4286]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003765135355905091908561142677605148266379,
-0.0005226035903522906703390294102525353043039,
0.00003629809254686521612733972809480077201921,
-1.68203332733955881615842617541097652209e-6,
5.85005309003231026588091313465352599012e-8,
-1.628816090041599987533907916304725622306e-9,
3.782339893217981777329865749415313357114e-11,
-7.531448188950038246990091144356187848268e-13,
// [1.2857,1.4286]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003658225132466034344442048825917433942446,
-0.0005081732956553949736789407023219563405711,
0.00003532269987416413671548464416681559169253,
-1.638014893931403072159769656061805700058e-6,
5.700856270610017013120285125239284124402e-8,
-1.588307018790954205528279380874810348906e-9,
3.690552126766813331055828260231882411997e-11,
-7.353002779893152898150573649574515581393e-13,
// [1.2857,1.4286]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003557213069587764475692625277942660705835,
-0.0004945177982375964817811635915356456731803,
0.00003439831277259211045255568825154544868282,
-1.596239783828360215162220879103231764883e-6,
5.559074522240278590522180086967939480186e-8,
-1.549762539007145614403880878672313565771e-9,
3.603110652252296291625211761220545509366e-11,
-7.182812028015716809452917669852658790672e-13,
// [1.2857,1.4286]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003461624587663175039931339127977018541111,
-0.0004815763788156039824995643478389616762529,
0.00003352103546042748822902271620665568193181,
-1.556540893650091722502101047094517571809e-6,
5.424168843959920369992887272268319834521e-8,
-1.513043213268550042114451477722005188174e-9,
3.519714102315890177365315946150417283757e-11,
-7.020316459361452133445521813328593416415e-13,
// [1.2857,1.4286]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003371034651709227552157826242108691769804,
-0.0004692944964453144381104734087468412308861,
0.00003268735876668315890202818465803858448509,
-1.518767303124916291685124192226838611178e-6,
5.295651204796146984860016440312920586894e-8,
-1.478022486411789907114160348779278316764e-9,
3.440088325473447275374370378396195822835e-11,
-6.865006012997388689559488505992893109088e-13,
// [1.2857,1.4286]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00328506147511720074832706215854818011144,
-0.0004576230227329083381191620079406219605164,
0.00003189411336565381313558473882829567555434,
-1.482782363578630993809035890016889290923e-6,
5.17307866167978171726551709827116999568e-8,
-1.444585232287792043037915061513230996791e-9,
3.363983383404379489648937170974819421881e-11,
-6.716414706572109123989711924186722461037e-13
}
,{
// [1.4286,1.5714]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.6633509458403348036226607487905011232217,
-0.02096289455675737981577440705336210038024,
0.0007384034882433738899403114141261615191478,
-0.00002246315987863912089519382429109268898987,
5.812052857623885956644634383295227236095e-7,
-1.293974775719394886686999754546086747329e-8,
2.516499373300836548192351359890815097672e-10,
-4.333001585148754295724900272536795505419e-12,
// [1.4286,1.5714]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.1467402618973016571149375796603213449781,
-0.01033764883540357844106630612928503988906,
0.0004717263574515747059081691112747049871435,
-0.00001627374814758874356615468058422038583473,
4.528911694595543673289567330919009011257e-7,
-1.056789343869037307523096748277086914425e-8,
2.123236128327929244922147721411921089311e-10,
-3.743825013031182541545124637851953698442e-12,
// [1.4286,1.5714]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.07236354184782504766374446062590605933721,
-0.006604169004318829376760846189886019114139,
0.0003417487110995002926303418123291968751642,
-0.00001268095287352878842649929602898053134147,
3.698762685317879635881384989110201895378e-7,
-8.916356584243105695713056337923286122376e-9,
1.83453257249949566989229019876407108118e-10,
-3.293758489789603412802353731069261198591e-12,
// [1.4286,1.5714]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.04622918303023180435231351002918343497844,
-0.004784481955390133867302526510723142095431,
0.0002663000103442279187464586619583462978372,
-0.00001035653563370018043937838021964669037804,
3.120724788036800466268436775856796908689e-7,
-7.703934621376202402737675958801655880274e-9,
1.613994294615903301261909195005017946194e-10,
-2.939166408468784953257523371103665355079e-12,
// [1.4286,1.5714]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.03349137368773093590039601811698786479711,
-0.003728200144816600267652964537153244634412,
0.0002174872483078161790600275199749312768966,
-8.738029510127693164578522399719985125251e-6,
2.696377102496315522626320623441943307919e-7,
-6.777781235225165719678910228849056056294e-9,
1.440239493381104021479688287566398015669e-10,
-2.652816247772331967785717804862306990468e-12,
// [1.4286,1.5714]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02609740101371620079860960511818952917585,
-0.003044821476307066322926304576087710255807,
0.000183498619712784753240278592446089472893,
-7.549855981397834888359875988386062570953e-6,
2.37222341856919566011715663750748589833e-7,
-6.048099548859810962716644764530861365588e-9,
1.299923992255668918991260869666083198767e-10,
-2.41687193203897152793791270687101054381e-12,
// [1.4286,1.5714]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02131375033414946326689738165665329080548,
-0.002568980675976819415296030271088986493351,
0.0001585469756094499174637920850745344370685,
-6.642225658679689435135615282684047495882e-6,
2.116834829382920989462803989061429752787e-7,
-5.458848577705766660742348229396316817745e-9,
1.184307944422162890970404076067975390786e-10,
-2.219181722069575722059231659400577680841e-12,
// [1.4286,1.5714]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01798286473183773498347326483231218749767,
-0.002219657658530295765761033540691932902028,
0.0001394867388323621440872754333153474690768,
-5.927137602396018265640245573167265028192e-6,
1.91059699037485569989407180996043801733e-7,
-4.973324173296541140962615190633943397414e-9,
1.08743687481140031729989654727847983365e-10,
-2.05119048488757897549110293647132271208e-12,
// [1.4286,1.5714]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01553760360971206949755266882093822344274,
-0.001952814343651208034386203782635135554794,
0.0001244698896503992103610406424438856514713,
-5.349671647529556924802874750090639819803e-6,
1.740663449610184445293341513318266407347e-7,
-4.566519862438341272075703134113224555372e-9,
1.005118677203175064649596754380269152345e-10,
-1.906705945993262365448996686204378881168e-12,
// [1.4286,1.5714]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01366970040555845543127171197856741096881,
-0.001742578455103849584599909243405923714785,
0.0001123431045981984012663479482636916351499,
-4.87385772848354444718053768903517686522e-6,
1.598281941492607974973044662351856950808e-7,
-4.220830520081698296450689217428832843608e-9,
9.343190682018237239689206847062107960622e-11,
-1.78113855836146050476945303830059463517e-12,
// [1.4286,1.5714]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01219804918572694632992436432173907343079,
-0.001572803464373145796844432287489259332393,
0.0001023510122982276121707875154813783823313,
-4.475189501452713934435230632666885383341e-6,
1.477290672271393340581145978992632910463e-7,
-3.923513458008380923715129992986748940827e-9,
8.727891167053650410381174396192756765179e-11,
-1.671016195960712364757357772503974430873e-12,
// [1.4286,1.5714]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01100962425061201985762017239096141057739,
-0.0014329141721736498177842052887453518128,
0.00009397897953057614090618267264151697031046,
-4.136413943830551760995121331744542727865e-6,
1.373229701083133363070817454351747870944e-7,
-3.665124168423339345434974665502954569304e-9,
8.188274394422494657913242039967350740526e-11,
-1.57366462889058643312096212200859511505e-12,
// [1.4286,1.5714]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01003039920521554804181439786959617703795,
-0.001315705713426613860427727011809640047093,
0.00008686469282050712412866636243872866019106,
-3.845043221117782789719074236661809756267e-6,
1.282793450209854880677874172697432812496e-7,
-3.438517626245717552592982086743934553426e-9,
7.711236308195478852521286404894955775014e-11,
-1.486991425110557004802114330763436680496e-12,
// [1.4286,1.5714]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00920993999398629637421269113981108120865,
-0.001216105699485723459303525207957981700518,
0.00008074590764353572192676803899796804869897,
-3.591821715639228040996541385396290955131e-6,
1.203481160881529176994013326697494063197e-7,
-3.238190750587517378877209459802273188278e-9,
7.28652372670524177055226744892318731578e-11,
-1.409336306689524550921863241276876508072e-12,
// [1.4286,1.5714]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.008512739896400063597039761262919680425784,
-0.001130442707008192158311533659557445109612,
0.00007542825602848312532704648120909311257555,
-3.369747302786700343285361139983548618288e-6,
1.133366754794077125666825439304650211323e-7,
-3.059837705375603218809740185700954872104e-9,
6.906001073041637211215677995696316881421e-11,
-1.339365330341932793642356367619297091484e-12,
// [1.4286,1.5714]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007913098949057344518026075419996114980496,
-0.001055995584397517690256662083379144393552,
0.00007076469335857736235760268770037120972732,
-3.17342696326644053374699935709390925144e-6,
1.070943189327417213204684538117036506642e-7,
-2.900041954750588281579096831279119896683e-9,
6.563131848594806526523879183186294734159e-11,
-1.275994644920771045610461785485354548648e-12,
// [1.4286,1.5714]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007391969090782623267162978519492622844088,
-0.000990705707018893316291335673955620197204,
0.00006664196622864945632890618232837023896654,
-2.998640977707469775593548801318480514698e-6,
1.015014676935333396835630700096779774917e-7,
-2.756058502903609253506524787765969048969e-9,
6.252605036525470508371001611098989814833e-11,
-1.21833462811163958965876242271671307716e-12,
// [1.4286,1.5714]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006934939949132252672815652779604442518647,
-0.0009329875271999540824298222030364569354161,
0.00006297146053190882304053024472103293928672,
-2.842041140951596240463026772771627751516e-6,
9.646204690885405770037575632355502212229e-8,
-2.625656999117776705457409845581393905774e-9,
5.970061365346940977080566558036945999687e-11,
-1.165648329178822336701391524430899146898e-12,
// [1.4286,1.5714]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006530912690399678057335003338720088039619,
-0.0008816004474456324110510888781990747765948,
0.00005968286396003340999782149823125917798524,
-2.700937357092774081697040736816021924048e-6,
9.189799430393406684962943516717546295456e-8,
-2.507006780226423066198626220854895728028e-9,
5.711889673631157101939248977607324558038e-11,
-1.117320125402967564217603535355524924662e-12,
// [1.4286,1.5714]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006171203132119426377586615050464836663083,
-0.0008355600954394200733096033131094516912627,
0.00005671968449899623399068448100367896889463,
-2.57314388241720533083122916782271054432e-6,
8.774523666821240642185215568835189601757e-8,
-2.398591352766659278237874795918198240915e-9,
5.475073322883776586214775407195417321945e-11,
-1.072831783289135008314218809797073485407e-12,
// [1.4286,1.5714]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005848920668075940031833354114480412645063,
-0.0007940755829849397332992777489833134312934,
0.00005403602153080752024837392218850661867524,
-2.456866667012021553344128898770596250617e-6,
8.395069673072039991970900083131802071538e-8,
-2.299143893322392412009747099260216126921e-9,
5.257072894598499545160875945289532053994e-11,
-1.03174396390150249956871548500671161253e-12,
// [1.4286,1.5714]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005558529080894577668888492459899000674633,
-0.0007565043014303349103221961380389238658467,
0.00005159420000729701668079500581851674645104,
-2.350619547275455998948014536464948673441e-6,
8.047003567209433058285831378568262428618e-8,
-2.207597986702225774588614914057507355402e-9,
5.055735564138238608389077985994629683736e-11,
-9.936817825963663743784131010270508446667e-13,
// [1.4286,1.5714]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005295530110012343924001184605471549990799,
-0.0007223188001012223893847536562603632692276,
0.00004936301049282760861343092281668442785549,
-2.253161036252753796036548609089891034763e-6,
7.726592896080756093991509598208583669976e-8,
-2.123049567238882756421435677697226897158e-9,
4.869224341629779188734813599707814704224e-11,
-9.583234241014079419305898384784685522049e-13,
// [1.4286,1.5714]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005056231600708556292332963755459278219974,
-0.0006910821468986828363314166311305443475693,
0.00004731638176134943253286960580747361733883,
-2.163446047050317505895491579039629330525e-6,
7.430673429865487243999343129786918163722e-8,
-2.044727203361393857456474724120248254481e-9,
4.695962284410584283273805292516441736741e-11,
-9.25391085342042461502651246466460473643e-13,
// [1.4286,1.5714]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004837575028290779434896906664854245168197,
-0.0006624293446580183973774430948793345590005,
0.00004543236698809693245207195749596978224676,
-2.080588595308984816681616389406996169024e-6,
7.156545158078265346476210420750973820327e-8,
-1.971968669547397327931012418416263314075e-9,
4.534588115712431441505264896317898983172e-11,
-8.946437097051572848772061532653674647204e-13,
// [1.4286,1.5714]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004637005412606128375288994576584461066346,
-0.0006360531378325114939589233675042438534374,
0.00004369236050152769126148999485519666634388,
-2.003832678084644354825129093264872570287e-6,
6.901890291402238761714360664928174933831e-8,
-1.904202308373024598498635283404866327104e-9,
4.383920621604652613021367994682633893936e-11,
-8.658711129995097757739448413534742640454e-13,
// [1.4286,1.5714]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004452371964827580063640296952231371905281,
-0.00061169304702056846631965029163267831143,
0.00004208048623981536258615059934572674412766,
-1.932529314361386507210742127075347674323e-6,
6.664708028863907534266811924336877393843e-8,
-1.840932079022828104914090609449295434181e-9,
4.242929867421981309439424571274431397268e-11,
-8.388892000615390694300271901632467369863e-13,
// [1.4286,1.5714]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004281851329143978881726630439713130561836,
-0.0005891268073566206233399970998075409388812,
0.0000405831156016258379009943812122990712222,
-1.866118279860306037320351964339115949117e-6,
6.443262227618071398060366983361722513939e-8,
-1.781725469660100606580086852813496848798e-9,
4.110713758491049303742999465834276676053e-11,
-8.135360430799850072068499081904435338153e-13,
// [1.4286,1.5714]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004123887651496343991772004786066526316622,
-0.0005681636184219905853113958580402266264194,
0.0000391884838771021013449905920548301259767,
-1.804113454579161136422380479132818936988e-6,
6.236039096244113900139236178750821754896e-8,
-1.726203654145355685236430002113004730959e-9,
3.986478823388190289113944309203845325157e-11,
-7.896686459802732321361843829593765851989e-13,
// [1.4286,1.5714]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003977145328953933735871057659627805333016,
-0.0005486387742786802539564515570530704599847,
0.00003788638254619706969188400419805511074672,
-1.746090976915228077278624726270203936425e-6,
6.041712743260823118804717904296247937291e-8,
-1.674033422011658982822704622151153517306e-9,
3.869524358978213388819438899463218915044e-11,
-7.671602589390948436509579997526131934464e-13,
// [1.4286,1.5714]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003840471419950761426130764239266505300938,
-0.0005304093556460305742082115029647407441832,
0.00003666791051525353999058714236883841232122,
-1.69167959724936346208569506518287408091e-6,
5.859116932040167598575938847092657182959e-8,
-1.624920520221427130054412096497139227218e-9,
3.759229271162715033919612951828415843191e-11,
-7.458981370061035916276992447212765249678e-13,
// [1.4286,1.5714]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003712865489522213677126091070299359520181,
-0.0005133507472128408030121387584453077075666,
0.00003552527154226949669821054894783066929191,
-1.640552769321788024455532876681803640274e-6,
5.687221776956190920366698590412010707259e-8,
-1.578604126985936228305466911841859125346e-9,
3.655041091811770174922629309023282290968e-11,
-7.257816595175876876868540419619030117983e-13,
// [1.4286,1.5714]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003593455230489885287514460886102007229295,
-0.0004973538015910828107370730603520551689423,
0.00003445160815578957145918297413488986339819,
-1.592422125153715167133773646096401784295e-6,
5.525114401753374682527418290218578321111e-8,
-1.534852239471277532719340098228181817367e-9,
3.556466763624381457392004206842619768098e-11,
-7.067207443700716064612613260381364205787e-13,
// [1.4286,1.5714]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003481476611137579349913032772169501129177,
-0.0004823225141803815193947965726348832942571,
0.00003344086462825924234417767619100676649883,
-1.547032059390440120989317192259316674533e-6,
5.371982796517554271014251168232259934852e-8,
-1.493457803943515269591982675934681098284e-9,
3.463064869842313966171562564400995284458e-11,
-6.886345046312649800594726590533913485516e-13,
// [1.4286,1.5714]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003376257599262670318436135630090528039236,
-0.0004681721047949736931013468498452269589222,
0.00003248767324722970614346661254807302282027,
-1.504155209253151706546489902254041490943e-6,
5.227102273184031789429931026022661962432e-8,
-1.454235452677178813385017317618050086123e-9,
3.37443905144992303415485840232625557713e-11,
-6.714501053852114594730293498721378704981e-13,
// [1.4286,1.5714]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003277204733564815541924864905228328429898,
-0.0004548274254605761511379370622147487176921,
0.00003158725939434592532012129915675877983926,
-1.46358866208116579565478670398994786271e-6,
5.089824044717350644767951763808654911754e-8,
-1.417018739545516076865203229611984157007e-9,
3.290232405549091542477291109099720417266e-11,
-6.551017868603176234744175892850590982387e-13,
// [1.4286,1.5714]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003183791978224032755373873807583469533132,
-0.0004422216315202184261142327407954998481701,
0.00003073536190373353067151129607428063810275,
-1.425150757502229456666899224434984251691e-6,
4.959565549677227167811857092692165974492e-8,
-1.381657787652261495813810243198086808914e-9,
3.210122698543355609404487520327160035069e-11,
-6.39530026308222636800907785046038752056e-13,
// [1.4286,1.5714]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003095551420641528687074612744433997217647,
-0.0004302950666516594019342179074369934239295,
0.00002992816590757520834902987256698017841864,
-1.388678378310953612447549928657457845919e-6,
4.835802218929777243621240796305527467178e-8,
-1.348017279139648908910891349356823150635e-9,
3.1338182592205768885517694211290965262e-11,
-6.246808161876671187734779898027130519873e-13,
// [1.4286,1.5714]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003012065466561615524376654353683200130869,
-0.0004189943227054567326816252444093520417647,
0.00002916224594455778565111555641613622068177,
-1.354024645147932125892767859952719799252e-6,
4.718060439975594943172177425336591796992e-8,
-1.315974730517096852167065478820207975633e-9,
3.061054441746533290453169962615522460172e-11,
-6.105050402611241658611007192135258459005e-13,
// [1.4286,1.5714]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002932960258938196845885932789210865308797,
-0.0004082714432232260442506805874224250562413,
0.00002843451754813373179725942046496384444894,
-1.321056946511582678917810339333950762287e-6,
4.605911520600180250618081104511940072486e-8,
-1.28541900732144124686937063383291798957e-9,
2.991590568444981228745106919907454559797e-11,
-5.96957932460631156760914155189941114449e-13,
// [1.4286,1.5714]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002857900102562582032880154285429019969924,
-0.0003980832456733019456420520683936924330292,
0.00002774219587676981636763646460386754061321,
-1.289655248580248142065492692427578956389e-6,
4.498966490184778957799957077737623562919e-8,
-1.256249040261145072790855723834118902641e-9,
2.925207278158673880400456478134355725318e-11,
-5.839986059966767435589727898797043839905e-13
}
,{
// [1.5714,1.7143]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.6431045601248349583866401253001611513462,
-0.01955121545758445368127879040862787969209,
0.0006743755279781216393398488505366127762292,
-0.00002026285183507777706808088185174934510426,
5.201337202830846697162885671349116534956e-7,
-1.151740320177955894750174348566295214507e-8,
2.231139755669173516107286031021234992839e-10,
-3.830395918580775410926831946271676019176e-12,
// [1.5714,1.7143]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.1368585082030911743642951086059798824123,
-0.009441257391690478364339364859545190670017,
0.0004255198885367681660743933598887051795291,
-0.00001456374429691082875341134642080179888181,
4.031091102643109434619044427485609843762e-7,
-9.369548716605611672410318490374267225779e-9,
1.876951535367584526955960587467733618348e-10,
-3.302031419066267619693939186440986242819e-12,
// [1.5714,1.7143]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.06608880174183334729878847245699544459242,
-0.005957278439511922527433358637461928521585,
0.0003058386302352475567529930424738032060134,
-0.00001128705520067351334916227977057520817559,
3.279342034791527515687325158846839310848e-7,
-7.882109023820592050747737193437955927655e-9,
1.618046660837035430908596953985204010128e-10,
-2.899810265336544703503164081869380039376e-12,
// [1.5714,1.7143]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.04170094907658345656376524114437137339257,
-0.004281740823290942585650507665091074251387,
0.0002370281592142520946105779474786752564545,
-9.182157798345454180214946168098626023845e-6,
2.758738143895261516180524673649356241371e-7,
-6.794827050157534518321390081828052963788e-9,
1.420953244330636199203971324137234755886e-10,
-2.583810969036875062986333385372586101574e-12,
// [1.5714,1.7143]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02997218576303659707269332871779667007498,
-0.003318394228997254727153268466184367009096,
0.0001928253137653531168503657098981218666363,
-7.724466893891381662363993781385304692815e-6,
2.378189454411222893570592628983271945749e-7,
-5.967130168731045174825342399481499860888e-9,
1.266109435436117517511201677828578586194e-10,
-2.329229074129226242341362244333333217256e-12,
// [1.5714,1.7143]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02322875960298078214803377036547495262496,
-0.002699554392712873477782047643380412589749,
0.0001622138047718094511139479143219662340165,
-6.658930555158445072171109209130098398867e-6,
2.088495546997669249041631290903803540884e-7,
-5.316864676975650640497350640432204862519e-9,
1.141360832629095020987086107740093832872e-10,
-2.11988010134356867174798709181360813228e-12,
// [1.5714,1.7143]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01889688074899011347440689842428908262957,
-0.002270993266803433157404135090698017278743,
0.0001398375416584108734093418810103047229438,
-5.847787607560448133868141133148374873655e-6,
1.860902625804524958948888459743935604487e-7,
-4.792986209951320000747388114952713379846e-9,
1.038776887978874983708122911839327614516e-10,
-1.944773458007461150065228200356851168251e-12,
// [1.5714,1.7143]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01589695286762403129358222785140360963711,
-0.001957725583215998164921930028413532799181,
0.0001228035397588470028899506730760111648024,
-5.210527422415785898478707139448262721782e-6,
1.677545163137319999046278624336647502206e-7,
-4.362189359734103860277577335211823062278e-9,
9.529721005452713326382409509148973097462e-11,
-1.796193431420654485602809416789006349608e-12,
// [1.5714,1.7143]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01370407908251198639987219299594517355209,
-0.001719249556622228608643942637934597989541,
0.0001094210758708039440452039745977703317833,
-4.697126521962334657990446685930167341191e-6,
1.526766266248216408068114001323571179751e-7,
-4.001857111416280689212376341034833474541e-9,
8.801656893632052345235953662538969636348e-11,
-1.668569230728329994708466181926925531162e-12,
// [1.5714,1.7143]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01203474689635559955294316438223154008006,
-0.00153189506218973397464257149460281320247,
0.00009863965696127695435361006863241780423254,
-4.274945606345218378747450775187440067845e-6,
1.400649979938798788474954479343133553754e-7,
-3.696111729875975157407822427548582897663e-9,
8.176279051948882315320468760122049442472e-11,
-1.557781027416988593999891783584319775593e-12,
// [1.5714,1.7143]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0107232654353281371564604300235801606702,
-0.001380955197456450905332456979413866956965,
0.00008977385773331352589218910615914676508344,
-3.921820000887364602647706102478637334084e-6,
1.293639096931239634080171131035905930895e-7,
-3.433489434808821376538732367521227130142e-9,
7.633399846162876932056450353812929629253e-11,
-1.460718161072123636017185846246406544339e-12,
// [1.5714,1.7143]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.009666686382195155708231497507347541096422,
-0.00125683400826504662533033968198773859165,
0.00008235822001869505015808620380382600805791,
-3.622189525117434200369273026556404509558e-6,
1.20172129413059527127038285349824556978e-7,
-3.20551231668982893540810713509195715269e-9,
7.157776669542013449449093069318953649916e-11,
-1.37498900884279415081690612849108968683e-12,
// [1.5714,1.7143]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.008797838057855325781290631545331373591236,
-0.001153015080260462440131721460342309703454,
0.00007606598002752333659554462437562752375131,
-3.364819674296603006680446998505294479785e-6,
1.121929303212297743052473219322406780129e-7,
-3.00577918132053374436157078244052043626e-9,
6.73769027639825592685282233683099494273e-11,
-1.298725221156382556689521460625198531094e-12,
// [1.5714,1.7143]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.008071105561823236514681730905708905214009,
-0.001064923720384125127594711263848107446683,
0.0000706612131602830224974523249634038724159,
-3.141402097058254820308302156696516447298e-6,
1.052022706214249601261066238704726562405e-7,
-2.829368500643199757591494349994865271812e-9,
6.363985518145951699232672655045207376743e-11,
-1.230446443488165364808711690478613837271e-12,
// [1.5714,1.7143]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007454466042688875353878513480719231147948,
-0.0009892569842428207698009172837354448895792,
0.00006596944403827512280919784012819231664803,
-2.945663623062117233033602730904862666482e-6,
9.902789683222202946407785791011949808918e-8,
-2.672435557689525305307147102222349781561e-9,
6.029408466347581834197831689907247973123e-11,
-1.168964838311161418177556672578139238589e-12,
// [1.5714,1.7143]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006924798889699744873835095158579650221662,
-0.000923572216534764517370436548375611989964,
0.00006185893608435388020360928077225755876434,
-2.772781154790689061571651776390775579049e-6,
9.353524386022048595973870426000005113439e-8,
-2.531934064011516426345413289260113928298e-9,
5.728138560300820838894970852283775145449e-11,
-1.113316413831478550140177467379652005372e-12,
// [1.5714,1.7143]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006465005515743351129208992397681518543432,
-0.0008660251051799165395681024995676218108837,
0.000058228404250651739415207283724559250584,
-2.618986869597882386720481646291408752837e-6,
8.861769161014609530571584195917680308155e-8,
-2.405419680506153219021852307375819175163e-9,
5.455452110437999163592846179593883915809e-11,
-1.062710788918810897285047573215495272295e-12,
// [1.5714,1.7143]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006062175736259415305118469591168235930706,
-0.0008151976595081317014231416643322423007315,
0.00005499872426160082875533681660291945944878,
-2.481295404790468632444042105997388708549e-6,
8.418968821373163396475514598604494514888e-8,
-2.290908702836301323159756972076070964587e-9,
5.207476140909072172070258767742425125256e-11,
-1.016493877972547073892688040796614703744e-12,
// [1.5714,1.7143]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005706383616556921456990520627985772132437,
-0.0007699821396614603323875822049806682906677,
0.00005210720350064332677596423427904335058487,
-2.357311308035641661035950411575327576433e-6,
8.018180401946208373382078280475466614137e-8,
-2.186774685855141763082533006521401349202e-9,
4.9810055411718824731612880287538468042e-11,
-9.741197852892828066180430756931202731404e-13,
// [1.5714,1.7143]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005389874977630221891177742467109996816916,
-0.0007295008490080933805653442487564149416534,
0.00004950353746879028649879581101077944028167,
-2.24509054907304577865264580688497053709e-6,
7.65371134474399338736741889325478139834e-8,
-2.091671655313558838906115736347134340659e-9,
4.773365345110527204044403108293517292506e-11,
-9.351293663946353469670382148296836684514e-13,
// [1.5714,1.7143]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005106505943056653244568217556888531211091,
-0.0006930495245621859680756111618027306189055,
0.00004714690153057422293146106984421291969924,
-2.143039211650357962530544343774278065989e-6,
7.3208507399151757493729831991120686017e-8,
-2.004476271293710735562847625515415628654e-9,
4.582305679001184256178698834037012815988e-11,
-8.991336843783024709067856304320924168658e-13,
// [1.5714,1.7143]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004851346671935301372134035402377990954816,
-0.0006600566214271936288796572147532231424689,
0.00004500382344469633936649693026297903385001,
-2.049838240996247986189920092520064005763e-6,
7.015666897764948389656774564595653011676e-8,
-1.924243711193226861251695443334037307926e-9,
4.405920695537569218137538402912794469807e-11,
-8.658011070550441520990827896939409051111e-13,
// [1.5714,1.7143]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004620396349990355011722998654508826230259,
-0.0006300535282249334868907156803009687428226,
0.00004304660306095868963503452086893373076908,
-1.964386763985057078971437588051840294195e-6,
6.734852939200231336420623135608549095878e-8,
-1.850173625826790472795175066850509740358e-9,
4.242585348317202203514675542265952485377e-11,
-8.348471445588763407564387266908473881262e-13,
// [1.5714,1.7143]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004410374697574534030830835565059013887876,
-0.0006026524428526345354130761792220715002666,
0.00004125212204372242965603110257698913985127,
-1.885758854461136038746657405612763994252e-6,
6.475607642085612285821376875600068909729e-8,
-1.781583587739196241566257345728585166632e-9,
4.090905594768535371257123077582215624704e-11,
-8.060263724503729021174487230415420667281e-13,
// [1.5714,1.7143]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004218567099968441382677414415348779913229,
-0.0005775297086113531651530666735627628763233,
0.00003960093594371891756002111905719557864574,
-1.813170170218255720469591282598961343214e-6,
6.235542510339372309555767468031654754537e-8,
-1.717888178862316459805663108138396840773e-9,
3.949678818339204930053951871757241465105e-11,
-7.791259581695045818695794559350607407206e-13,
// [1.5714,1.7143]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004042707960279471802285678533914986553024,
-0.0005544131032113285710581246868279174967912,
0.00003807657357461733375083580445054698373067,
-1.745951932346289970689108424840143812057e-6,
6.012608580733131394990603193138098584874e-8,
-1.658582369795106196620467188525113065546e-9,
3.817862107270723083601460923669296056095e-11,
-7.539604318476676190685057404263470856509e-13,
// [1.5714,1.7143]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003880891722479299654356833345796734523581,
-0.0005330720300439294373359922428845425279526,
0.00003666499057930502236826630840127007367616,
-1.683530431134950172854452148228569032625e-6,
5.805038250372082621548550615335860775289e-8,
-1.60322819848455797972147973878383010134e-9,
3.694546630885799886947279023746067157777e-11,
-7.303674314032678144209908686009339353839e-13,
// [1.5714,1.7143]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003731504210307505728406103083089866867518,
-0.0005133098681095787213360421727621510585531,
0.00003535413905386591660583373939705483328145,
-1.625410737768114654077273944595407702635e-6,
5.611298652078511741625987736906967708107e-8,
-1.551444009573544096448042313425471051502e-9,
3.578936789986480787755622885264138447157e-11,
-7.082042167514694300252491359531950761509e-13,
// [1.5714,1.7143]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003593169076767050725933169605060098963703,
-0.0004949579467534516108046181487275138352874,
0.00003413362549316145614194286965238193527165,
-1.571163649431101904641055778816584154407e-6,
5.430053992109393033220051801915137489842e-8,
-1.502895698645765508957111395171939871749e-9,
3.470333136008958539060328475267202634677e-11,
-6.873447958450749272264411746964515127203e-13,
// [1.5714,1.7143]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003464705627274160961210515749338340876986,
-0.0004778707569036083702490136755712680336927,
0.00003299443663808332465712080803119237014008,
-1.520415143871504303005693336912496407922e-6,
5.260134905013833782320561374682505969645e-8,
-1.457289539164668809974561284743462636776e-9,
3.368118288229389731852028990785927023505e-11,
-6.676775409441092428552725517520400794093e-13,
// [1.5714,1.7143]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003345095298325258285832085868960809500083,
-0.0004619221129325326681204548454799904198488,
0.00003192871802133095798158351627161505799794,
-1.472837798759194904679411114212031824412e-6,
5.100513347919387001272147192027474892471e-8,
-1.414366268445488150176216948758481707082e-9,
3.271745253154062857026588334115074615871e-11,
-6.49103200345225931523477857677750995408e-13,
// [1.5714,1.7143]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00323345479052772837899484570301840554306,
-0.000447002052298016692508154440138138395775,
0.00003092959377397168659620944474129513814746,
-1.428143762086429877984152616952759339679e-6,
4.950281901434285290267561342897634780618e-8,
-1.373896182424322384335308667811734650507e-9,
3.180727681714717826046801430223369223405e-11,
-6.315332311585935333365971305896211864443e-13,
// [1.5714,1.7143]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003129014366086116557357737356251808378885,
-0.0004330143128350031475250097613307440015582,
0.00002999101900384288672829106505133195634977,
-1.38607895642041910220441343989636439095e-6,
4.808636601339284611265658629504061272947e-8,
-1.335675044207049244621762194647900815547e-9,
3.094631699639249089037997881445405118143e-11,
-6.148883942898500855426066256358814506832e-13,
// [1.5714,1.7143]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003031100189845021749742054232695756645614,
-0.0004198742660532153698098778942636737713687,
0.00002910765808485596286560870003789448433369,
-1.346418271776995953797112615974798523619e-6,
4.674862618508926630251623804869365045791e-8,
-1.299520653271139470306543914957725409332e-9,
3.013069022662528120180277775189530301189e-11,
-5.990975647858336222074882756004736463275e-13,
// [1.5714,1.7143]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002939119862372507312647786500366602051848,
-0.0004075072131874130847126933525035072007344,
0.0000282747837073434132266499913105130548153,
-1.308961555998531238576255410956183334861e-6,
4.548322251117941755114175930137334905325e-8,
-1.265269954233631112398645656511540326446e-9,
2.93569112704901405417462149710618499623e-11,
-5.84096720020785734090656146497394253614e-13,
// [1.5714,1.7143]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002852550492311891323549803599966736186453,
-0.0003958469719022513237214842676657353508224,
0.00002748819267599502230202423330517219853386,
-1.273530252571692284325561088737675969526e-6,
4.428444805329205153440082473856245792651e-8,
-1.232776588794704315933205954523669823978e-9,
2.862184291558105867789875221943193309862e-11,
-5.69828075484497778559351450773892107554e-13,
// [1.5714,1.7143]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002770928803315759002887276611471087494846,
-0.0003848346974633871207659076499998742212914,
0.00002674413530403080176965970568059771982671,
-1.239964567220041676183971626200260695725e-6,
4.314718027096288047759890094666325298093e-8,
-1.201908813641046734048582464062431343647e-9,
2.792265362679475429172695847723031806417e-11,
-5.562393436663794135970542678164887549378e-13,
// [1.5714,1.7143]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002693842882243709588188692082397084761009,
-0.0003744178942559006856530363048658417105735,
0.00002603925591164556391226684353766124303394,
-1.208121068808726698158750565323534829435e-6,
4.206680814825270841591059892266444307328e-8,
-1.17254772208393768868535097230768761631e-9,
2.725678123055637448395040212364946583772e-11,
-5.432830960684544044033128379225614447633e-13,
// [1.5714,1.7143]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002620925259791304548122591930870576939029,
-0.0003645495827625194324501254614561962900175,
0.00002537054244500739986689039331642349218434,
-1.177870648889765476966648189847320405038e-6,
4.103916995108067312522696490550259029043e-8,
-1.144585719003430509538235898016812676908e-9,
2.662190165250810720545042821824954852184e-11,
-5.309162119955042916487285441973079706654e-13,
// [1.5714,1.7143]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002551847079337635781177141206718056739406,
-0.0003551875942295966754450238189352337625874,
0.00002473528362670868862624323240363610357808,
-1.149096778907359088722518003207108911544e-6,
4.00604998502708857281458310915109861206e-8,
-1.117925208010481956805088982431718153033e-9,
2.601590190740144298805008724494975568348e-11,
-5.190994006661530939150226079540319711534e-13,
// [1.5714,1.7143]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002486313159607176487383181121488180276094,
-0.0003462939707734257555707902526671780001859,
0.00002413103235707765126229936344549541059021,
-1.121694015643181230054750330578593424269e-6,
3.912738197222718539681010747545676322171e-8,
-1.092477457178311383302202505108659891115e-9,
2.543685668182062288490713830167051497871e-11,
-5.077967855202490734827147379397799037463e-13
}
,{
// [1.7143,1.8571]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.6242079661790276016600343479428947260621,
-0.01826122869708922044234978184301697232228,
0.0006165958682489933018359868413265719861091,
-0.00001829316009389535252216625734933280264958,
4.657629481267384013475686570354959180736e-7,
-1.025612523312424092947620629349302610821e-8,
1.97884546681985132170932883919773400234e-10,
-3.387050941603276202548628911920930832172e-12,
// [1.7143,1.8571]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.1278286008796245418595602959009011198749,
-0.008632342155483061505013842856484093988283,
0.0003841563619719211447819258669084734397832,
-0.00001304136266133838318151306200514801864859,
3.589643815761198018325686207789694689952e-7,
-8.310058573812876289123168083356106412019e-9,
1.659705624795901622508707246684865494942e-10,
-2.913042279918150920935548233843354395163e-12,
// [1.7143,1.8571]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.06042639508838142943465587211942354931201,
-0.005378189067604402451375178037367775043174,
0.0002738686158882116896518031146813043837013,
-0.00001005100278387516993736766185348564508497,
2.908520486748754588825616169518506920819e-7,
-6.969806078441357475732731205027686334754e-9,
1.42743579165221871108324718182292895412e-10,
-2.553464866251017066404112136038419127509e-12,
// [1.7143,1.8571]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.03764732347323081616884876384402653668628,
-0.003834160622432745157824230628204260143402,
0.0002110710584614736837825617409281754993918,
-8.143857451637125507672106463790901794354e-6,
2.439432114772425308474132231773296778429e-7,
-5.994378410445807616083222200085636678193e-9,
1.251238367100811222858304085888723438873e-10,
-2.271780767459934279984463842590144092108e-12,
// [1.7143,1.8571]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02683912435702921520398814815438670500621,
-0.00295499481845863415804562415199853842962,
0.000171021006484466111105967537073057795886,
-6.830410001260047776267172229531749911866e-6,
2.098032432125939424071106192877333114338e-7,
-5.254434123929803002356305714047014386064e-9,
1.113209472426120341259706856570212934234e-10,
-2.04538910153245452299947814906495569702e-12,
// [1.7143,1.8571]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02068496372921043828068289409976561614536,
-0.002394294090780709573120684659384818282327,
0.0001434386100265402647969040399662589862578,
-5.874490882592534472698037810377687402575e-6,
1.839051932807200182369467605902234818057e-7,
-4.674782437209209707157971905476856379438e-9,
1.002274478156873504077919013151853234347e-10,
-1.859599379018794034734311205885084033884e-12,
// [1.7143,1.8571]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01676005863546496624987392336475233847234,
-0.002008140540369899217592702888452679535411,
0.0001233643085345163734974470422841444097001,
-5.14934547844030894845816043335178869999e-6,
1.636173843269917904957474227511288740968e-7,
-4.208913635212213819970687947819529158516e-9,
9.112349063594787036438412776545644492048e-11,
-1.704468356155735860284682900389144220617e-12,
// [1.7143,1.8571]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01405698378258929381579249334921387737224,
-0.001727100319481693089691918250272115846732,
0.0001081362550473143944788732647839599898337,
-4.58128682260186902345000575477426877871e-6,
1.473119763270039128057649236071161909256e-7,
-3.826596720781276011418838677492882993151e-9,
8.35218468129335685008101009587656214332e-11,
-1.573035579343584615653327999144304480315e-12,
// [1.7143,1.8571]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01208970223637185096784205651432504090115,
-0.001513907570660975486524501440656141976423,
0.00009620702327470260994800569593503096547023,
-4.124735394198051587082920439511364832251e-6,
1.339308843825359949701353186702774098556e-7,
-3.507369960332743527834288597952802941783e-9,
7.708144678109326750206662370245275039955e-11,
-1.460288193632537390061581657628868651901e-12,
// [1.7143,1.8571]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01059735299462682778711600354356718841928,
-0.00134689832584450597121698740937586302937,
0.00008661944327821846496319274157628670645355,
-3.750064815934195644322561212833991841848e-6,
1.227579478198884508475863070043518255652e-7,
-3.236909819223439942562458945463989630208e-9,
7.155665511744618853770697145677729935124e-11,
-1.362527228988593663861960571882168533764e-12,
// [1.7143,1.8571]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.009428288280911541216536333278158139412274,
-0.001212672205893811496588359948499494617286,
0.00007875136113467397914795476915125375119745,
-3.437222588837832052384847779034095418479e-6,
1.132918429278765184648460691202910324227e-7,
-3.004900654947326653884307778352209008625e-9,
6.676621804579876099869610269626534087687e-11,
-1.276965319785239775793592623140933672388e-12,
// [1.7143,1.8571]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.008488705441256679926644167975215716760053,
-0.001102519055884262426462170203561537838501,
0.00007218167436564722291291246903926473358777,
-3.172171648912223193808343058518618471729e-6,
1.051715222198233945584202342282927356345e-7,
-2.803730611124656971777689133834909403582e-9,
6.257355133989399850551726232494433921123e-11,
-1.201463205430750366259979775090806933157e-12,
// [1.7143,1.8571]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007717633391189836464843832456932323889291,
-0.001010543441117953375994962975858141739147,
0.00006661560462720664490233134597836572323452,
-2.944802666465241853064677569786119540747e-6,
9.813057072325646741791537953558014978944e-8,
-2.627663775951812973945889043603730124322e-9,
5.887382861146160361761196146740786458226e-11,
-1.134352349792981252275500834003995618728e-12,
// [1.7143,1.8571]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00707380408782567313774357544878640226856,
-0.0009326184647798439153753071352771565035664,
0.00006184085599581752439615512418811469461523,
-2.7476560222160878158975935182089710785e-6,
9.196823152570503075124559307602629037027e-8,
-2.472297936167597959985211410168715681981e-9,
5.558528949107611734873848601382648323737e-11,
-1.074312625738978757979858884160657792702e-12,
// [1.7143,1.8571]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006528329253458906937078748520511986359472,
-0.0008657719839404489875947020927035142237751,
0.00005770077646658301701937572718458913908455,
-2.575110522574258142488305767083911204274e-6,
8.65304271635588763187238243157551437422e-8,
-2.334199552960255365250444864996403637252e-9,
5.264324604785605984490007256272020491041e-11,
-1.020286152317184802072013134298759108276e-12,
// [1.7143,1.8571]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006060403887583142464130037021945620346747,
-0.000807810870531213608790464222875437272849,
0.00005407732097410252837747867768230467059181,
-2.422851998525171279955777637962691350102e-6,
8.169698377884196966845819887584044948237e-8,
-2.21065205479313582157460887269853507518e-9,
4.999586072174786335348618422040521662197e-11,
-9.714154324573902652559155531649110915718e-13,
// [1.7143,1.8571]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005654676093718494832139897725954562655851,
-0.0007570824936365301432635015410024126888528,
0.0000508798919690698188606061506552753387941,
-2.287515582018065111617082544595146746462e-6,
7.737282136813157520614271876672918181125e-8,
-2.099478527503449630023363175222120579564e-9,
4.76011150043340761544048219007369551147e-11,
-9.269981719788941989260069185569544866336e-13,
// [1.7143,1.8571]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005299577455455710591449191163351677515872,
-0.0007123184875661118034617698002976795943255,
0.00004803782722241886150589911285884847714059,
-2.166439032934423444566212335423992832604e-6,
7.348174793603022111278090495139564668731e-8,
-1.99891441146279695270268879402454747226e-9,
4.542459552022230746320570784669013493415e-11,
-8.864537686047875992850347202280400538413e-13,
// [1.7143,1.8571]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004986229412962782229390675797501038449034,
-0.0006725295811130346846676783750845952483678,
0.00004549521969166079734370596904950481169991,
-2.057488975384206436997799005772473419565e-6,
6.9962003895796162033877071055639477653e-8,
-1.907514526447607781460856524752225022488e-9,
4.343785195064595635602822233926517750634e-11,
-8.492981062806155856278722104025765307665e-13,
// [1.7143,1.8571]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004707707067791242413108018439908125368569,
-0.0006369330756824551585968697094363089506311,
0.00004320726848310477368313182285946156558217,
-1.958936140922753474700184123216708597317e-6,
6.676300793981795465550695476623658505623e-8,
-1.824084111631157670639370636592479619576e-9,
4.161716192571326795623090303559916666296e-11,
-8.15124353026259507755502507645195994393e-13,
// [1.7143,1.8571]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004458531529777185744752835708905794330855,
-0.0006049017587627016238491243521242094354161,
0.00004113765895941290398220044464956555476078,
-1.869364252923493155054486792148660987613e-6,
6.384294343934215759933860576160230419425e-8,
-1.747626956608119053593356471378563363413e-9,
3.994259009627192759127439997399298126652e-11,
-7.835881606505286361443135464315485627069e-13,
// [1.7143,1.8571]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004234312311338911014645423807093005484634,
-0.0005759272254310439553506617353206169054032,
0.0000392566493114271770899579948430491021279,
-1.787602445769868479974588555967515032081e-6,
6.116694303033825253600627338945613709808e-8,
-1.677305886736771620396574821442899961933e-9,
3.839726290192391455633858922261396677139e-11,
-7.543961343240025115960808158698646442011e-13,
// [1.7143,1.8571]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004031490578017307347374353308998220617335,
-0.0005495930903592702426029958263473373384966,
0.00003753965136119988596323512423313931927044,
-1.712674433259200268705296747156985251941e-6,
5.870570560048047378259577307124061661795e-8,
-1.612412306797672894446763354732266453213e-9,
3.696680356580234683399588150028613170214e-11,
-7.272967604271985904887577081226306549675e-13,
// [1.7143,1.8571]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003847151632514891369541252871649795554256,
-0.0005255551190561127987521279100756862574538,
0.0000359661630984747590671691441146586991161,
-1.643759784237897003812711138007668827792e-6,
5.643443031720485501546501328321235176885e-8,
-1.552342473475571553530759991144047034099e-9,
3.563888754768677706751603091867138955508e-11,
-7.020732030996650097119037608654272196116e-13,
// [1.7143,1.8571]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003678885833392789273247413852787712513706,
-0.0005035262833779840058288766731863002972439,
0.00003451895546902636692450731503889366541412,
-1.58016407538682681688830437394937373092e-6,
5.433198616457908210635202561228043265778e-8,
-1.496578826553352895237304191298595163466e-9,
3.440288956567160753319862346812948795547e-11,
-6.785375359473407346168175354659510491404e-13,
// [1.7143,1.8571]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00352468398364588773277793441137123466085,
-0.0004832653765657280109855029773929909991099,
0.00003318344558315293363937482149089953296756,
-1.521295638253491511609962324514328949681e-6,
5.238025853509293132018826309084403098557e-8,
-1.444675165567940910535215210744749703696e-9,
3.324960094231661887434267641000108152489e-11,
-6.565260864848649930424248644195047188121e-13,
// [1.7143,1.8571]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003382857635960095778259593879293836438854,
-0.0004645682381635200915248302958826705304536,
0.00003194720840335199123746397056182497052798,
-1.466647263822010977539820233667573767732e-6,
5.056363041261676313021479482319368681389e-8,
-1.396244779768041935124397454151022295996e-9,
3.21710014761721550161273957555803190549e-11,
-6.35895650999665106088292889383522711644e-13,
// [1.7143,1.8571]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003251977667144640350865932448269208757452,
-0.000447260917646325818688836455127619520583,
0.00003079959254029005223707162213183118559961,
-1.415781675635839978409102632830248285168e-6,
4.88685669209241329141248855338691411844e-8,
-1.350950867880177632640389721171203749433e-9,
3.116007396505414241374446318755528380268e-11,
-6.165203959413821716020976984824695476268e-13,
// [1.7143,1.8571]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003130826423524280449338418367358720322171,
-0.0004311942955634764761462351467212903908073,
0.00002973141518837966210467996373067262268665,
-1.368319897156301528627018238696592130576e-6,
4.728328001550425738092224344712970480549e-8,
-1.308498749043580815400121840667415707812e-9,
3.021065236992749470492186202218964838593e-11,
-5.982893049935056232183276834317727069425e-13,
// [1.7143,1.8571]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003018360068944335059400372940397055475126,
-0.0004162398126367477964370230452708112233502,
0.00002873471784030860007961916853733857922628,
-1.323931863133253296009185167241655466142e-6,
4.579745586628446195483913853553826046115e-8,
-1.268629486486811452437416318755517115675e-9,
2.931729671791649536104889978671730966396e-11,
-5.811040630316899796006959755350469785337e-13,
// [1.7143,1.8571]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002913678688457234308869516371815243731137,
-0.0004022860497637687742402622859862730562922,
0.00002780256912582387356262606557683075238865,
-1.282328786321247227834264803635567215802e-6,
4.440203168631265756897533349883741712215e-8,
-1.231114634115642482048484924608637702823e-9,
2.847518941332947388383978519823770465633e-11,
-5.648772922518969778629544967245622604974e-13,
// [1.7143,1.8571]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00281600234834638116053304817664268923072,
-0.0003892359677609975892574039671442110879841,
0.00002692890451277107020513878933270704940458,
-1.243256908682581160572786587369010722317e-6,
4.308901186233411603514424764391391000308e-8,
-1.195751882130407558464002982689146346566e-9,
2.768004880546758780985601911406775408825e-11,
-5.495310740018280108400184982075135389692e-13,
// [1.7143,1.8571]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002724651774326982872331422048014646485019,
-0.000377004663178272536678200012632307458896,
0.00002610839508235844166460826884032237578186,
-1.206492353043399898425057503072097928274e-6,
4.18513155513992944301299154883390599613e-8,
-1.162361427340837567979746512564167932429e-9,
2.692805675627459986875336553905763578204e-11,
-5.349957037962347048704697165317194925948e-13,
// [1.7143,1.8571]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00263903264224790751062016272890875166395,
-0.0003655175311525092007839335165879480330788,
0.00002533633941393502621214609363580927870794,
-1.17183685579867314106017464143388331528e-6,
4.068264964188366847145093759102623991632e-8,
-1.130782931402441897211802194810457145617e-9,
2.62157976343100974657763542225415808152e-11,
-5.21208637738300030056931343707191033057e-13,
// [1.7143,1.8571]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002558622718067564165392848788822214078702,
-0.0003547087517945941723291009426629343749096,
0.00002460857397179518517948866159927803601569,
-1.139114209820715833400409795898852599953e-6,
3.957740229176155097572160616769255131628e-8,
-1.100872958899238752015478326920081739175e-9,
2.554020668788014231765863983598844749391e-11,
-5.081135969029405534329973598525493822766e-13,
// [1.7143,1.8571]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002482961262562158971953205479814031038919,
-0.0003445200356046485595455669190002667544218,
0.00002392139840625753027399285001929576467888,
-1.108167283530825215967785852966452129449e-6,
3.853055326150205937393707745424754192511e-8,
-1.07250280930196212111834813129628413698e-9,
2.489852615851874862240221913563643638322e-11,
-4.956598027483319862699606882287387788573e-13,
// [1.7143,1.8571]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002411640249232539687944461463909760186448,
-0.0003348995776871329711925075352053360408769,
0.00002327151295416930160969039954133337820589,
-1.078855510220342314967255314936697313972e-6,
3.75375980326067090891435013639958462358e-8,
-1.045556673980048377228581618150297924133e-9,
2.42882678150337512917428352201690363057e-11,
-4.838013217415605945385194646274196307538e-13,
// [1.7143,1.8571]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00234429704380993057469918779831803290055,
-0.0003258011813579088095418693118297634556225,
0.00002265596571464865897638754806146274716682,
-1.051052763369682528794143076880071594978e-6,
3.659448330302924653755561199455837358095e-8,
-1.019930062843033235820997893417261037849e-9,
2.370718083919744393206542579015266501808e-11,
-4.724965014353146710848460852232402566695e-13,
// [1.7143,1.8571]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002280608269505361448137715164317463974657,
-0.0003171835200046303486213530299376818340997,
0.00002207210803078432413669476273744784069077,
-1.024645550520071928365026228134289473846e-6,
3.56975519196248043743097183368811305606e-8,
-9.955284557221869844500738875734738217693e-10,
2.315322419266705713641238247429795949077e-11,
-4.617074834569932678470325526584519519365e-13,
// [1.7143,1.8571]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002220284640032412226469198438908619346528,
-0.0003090095124305397267849874689363620389662,
0.00002151755656094204311590973026465731693607,
-9.995314713821071175751844944048251350487e-7,
3.484349567650736205983048194985399303621e-8,
-9.72266141939445031794372230932916622345e-10,
2.262454275272295575770140954416011080995e-11,
-4.51399781452631919550408763875602751161e-13,
// [1.7143,1.8571]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002163066587013777878185787918797663827009,
-0.0003012457918527574191298661147433935561495,
0.0000209901608990443432567380696558621444788,
-9.756178961897497940545671833606622696322e-7,
3.402931469996251355353599449591740737513e-8,
-9.500652181460298698387339667700098583841e-10,
2.211944663088587135870469268901322851574e-11,
-4.415419141051083733036914370759609395553e-13
}
,{
// [1.8571,2.]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.6065454958914433277227336459896739581464,
-0.01708110350583691488748996447709234710037,
0.0005644113026158633893311630944321927866274,
-0.00001652882804835706010117633127803518909369,
4.173352177308250069059308913464154785441e-7,
-9.13727067389290224761079637395852984907e-9,
1.755721179186784654949105966634293024026e-10,
-2.995888523640128972249041904481045359743e-12,
// [1.8571,2.]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.119567724540858403123061200273179672087,
-0.007901758236619577257154771799883460431844,
0.0003471053890156028419899156142348621752574,
-0.00001168538619687161361063022781916462842083,
3.198044721918495013194594132629271115651e-7,
-7.373065025901058347025262833121891593385e-9,
1.468029997532794855095243175289053798929e-10,
-2.570480593257731923926329307509824378124e-12,
// [1.8571,2.]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.05531230765633703983237867128695150969001,
-0.00485947544621624361294941618823345726785,
0.0002453931101343967859172889097247904889461,
-8.954525309219478535435674346426387700801e-6,
2.580572746678656026120057861269125995149e-7,
-6.164882647334543981815918040744891238466e-9,
1.259575128257422008488011759928249563357e-10,
-2.248921294659839267454601124331237241386e-12,
// [1.8571,2.]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.03401632812351370442048492845470632467332,
-0.003435503541879604102919485563275233759429,
0.0001880450314936925851014845455278783965818,
-7.225603768736865990450677662302024291977e-6,
2.157708915428943858917116447376510781301e-7,
-5.289466382995446968313633935262981131639e-9,
1.102007076520975719534118220756323491292e-10,
-1.99775620234105208847976468209056231897e-12,
// [1.8571,2.]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02404852479315722793016034457646852385252,
-0.00263263044090994194029637660978782072964,
0.0001517376791435500526721215288118830158859,
-6.041585033371666728713958028816278343351e-6,
1.851313223932793702653755676710674474973e-7,
-4.627756079693144406503959019286967088616e-9,
9.789329091712643953594123233726659959348e-11,
-1.796384910921604426985551741131775323033e-12,
// [1.8571,2.]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01842841308636959285838346813734410824269,
-0.002124327508008107534817235163043280749478,
0.0001268732857008744759997141595112108637853,
-5.183677090740459952124380686641539125688e-6,
1.61971461862928007948296816556888258217e-7,
-4.110906420181606314956694321466021628316e-9,
8.802582490358762047030778420462106650707e-11,
-1.63146874226500770229349225978263552854e-12,
// [1.8571,2.]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01487029255605675207635671537634325678698,
-0.001776225999810783696972860185846918113592,
0.0001088572189056137262532853837008036566991,
-4.535200990521160430336988295351115528447e-6,
1.438817238521235347980969419842550624781e-7,
-3.696524394689259735012853410078553010529e-9,
7.994470192106522501548377744410575301264e-11,
-1.494009401572404860325385992780060097342e-12,
// [1.8571,2.]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01243358199867548525970122088129120116295,
-0.001524001064677246756613311310574059572072,
0.00009523922080100380378290015283016034530467,
-4.028688321676358130570994231064685069995e-6,
1.293783530216584051958127252466776232813e-7,
-3.357160835496989010195659487452989313646e-9,
7.320899657237837359094223304014415849449e-11,
-1.377726130880165836767168679142998768611e-12,
// [1.8571,2.]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01066800745274072671897648568801454060184,
-0.001333349091212805124702864655450110890906,
0.0000846024547552589434276194747476892581116,
-3.622593934531998272746072128097953398687e-6,
1.175006285034232091923400465396502685736e-7,
-3.07429856786701931306340868935524281584e-9,
6.751094512645869847774689463206929288256e-11,
-1.278107068194581405083906055101505087952e-12,
// [1.8571,2.]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.009333443638489635332142547395097256879062,
-0.001184434366572461332909358351837235835522,
0.00007607447262522387863529667646820031206569,
-3.290017644651259797271353328945606361739e-6,
1.076004491831447705059168747526262944065e-7,
-2.835012760772003565328410329231615735021e-9,
6.262946138899101825124629277044776684452e-11,
-1.191830397742818956618384037465252594839e-12,
// [1.8571,2.]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.008291040566007228821801348087902523569924,
-0.001065042616752044089080520136333521149909,
0.00006909037053772527815499015822765100439765,
-3.012812620736910079482256033575627707347e-6,
9.922544597605260438076946266282863618641e-8,
-2.630018730857812590655747943118122447345e-9,
5.840177258977956767755462329608250503179e-11,
-1.116397960195294248391603203282621706761e-12,
// [1.8571,2.]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007455298317264308143608898099782449202976,
-0.0009672651875271286246918867759792653787312,
0.00006326906503552118756453665265894505253993,
-2.778312528340615541182057257591913750064e-6,
9.205065496567624777151560819894251990312e-8,
-2.452480739475192717293105903221340019025e-9,
5.470546596468437887932836877468903515427e-11,
-1.049895886073922236794124864648542300289e-12,
// [1.8571,2.]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006770856312689899918464294516576300248902,
-0.0008857669104963290332203350620985920095084,
0.00005834456309519654696898651540504051908118,
-2.577418377742987365453142703105683152388e-6,
8.583682530002186139170105968972116066279e-8,
-2.297258009404778046854809820326118716163e-9,
5.144675957315025494283375963769549032523e-11,
-9.908338388933658566940957554590174778676e-13,
// [1.8571,2.]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00620036837347430280116211013369366420192,
-0.000816823883331835625944797883166243994482,
0.0000541257859326441473951169966913301560746,
-2.403431145042205427536465560282561370701e-6,
8.040402977699587951524262222484123093043e-8,
-2.160412140674339786206098457684458449522e-9,
4.855262505782251228288368301580445087651e-11,
-9.380343988112199114241883759453385661194e-13,
// [1.8571,2.]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005717767183322848971024405363710709775455,
-0.0007577610030561483974163813835810811959711,
0.000050472054045925730747950598494970095763,
-2.251312868542842596268288577215503102978e-6,
7.561442439804329599433278528978488492858e-8,
-2.038876295626521058485402756765168806093e-9,
4.596536733281576650033355528230461761839e-11,
-8.905552921063327548539880196987187088417e-13,
// [1.8571,2.]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005304327021393038390211235211850042823408,
-0.0007066087566421324792805399259817491189688,
0.00004727757023943729825098161245949915586812,
-2.117203916255559494199126288711842030836e-6,
7.136066984554358196125599891746324784052e-8,
-1.930227566727785029353201955888255713544e-9,
4.363881374751377945990731978058984492901e-11,
-8.476336532789628250515989064298803206758e-13,
// [1.8571,2.]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004946261296494926980490453991133516244615,
-0.0006618859833513324981290651344871944106912,
0.00004446128224140269900899129061749144625152,
-1.998098787262602631783095910626922458754e-6,
6.755796435614253271908026354027383022045e-8,
-1.832526936688654585381223957741714489915e-9,
4.153558286975713995039259569110911635725e-11,
-8.086453832210407799333747033736830817231e-13,
// [1.8571,2.]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004633201883459327128212589955158406419733,
-0.0006224579513788828447728115810261622034199,
0.00004196007453254908977469295509917868451835,
-1.891623032169920381119691453875953711134e-6,
6.413844232497467426396175357583131010744e-8,
-1.744204578644733879573108768175029938255e-9,
3.962509299132915561544324918238109469816e-11,
-7.730750500976896117301858428766726733231e-13,
// [1.8571,2.]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004357205659652179569228257334914179666951,
-0.000587441043454964133052668101188361368703,
0.00003972408367560136959741215520463467064152,
-1.795876414024507235855530381864899766771e-6,
6.104715981200968269525246880027097631443e-8,
-1.663976221865126773459417820524126979598e-9,
3.788208723702127711493217465401807812832e-11,
-7.404932812553850824066238417188200787474e-13,
// [1.8571,2.]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004112087304184748600571370075050074027202,
-0.0005561371714577253017189061136567926048769,
0.00003771340469454640866522673376932175813147,
-1.709320502491431637629127412708401816614e-6,
5.823916734185524032780808706473399654723e-8,
-1.590781212783613777003447592442371176225e-9,
3.628552574185887744699631735165639330891e-11,
-7.105395620289399762417471341509025155092e-13,
// [1.8571,2.]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003892960200204076793619571883729629596847,
-0.0005279876657229828311528592409856036336408,
0.00003589573055235063218109121725230403325724,
-1.630696712247798675329067476869483542294e-6,
5.56773420398546181473462836922309977627e-8,
-1.523735991407964978300794647109498799353e-9,
3.481774277219686306503838088419252585891e-11,
-6.829089934893788061219515668580970132297e-13,
// [1.8571,2.]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00369591366006087951114991708121108640225,
-0.0005025402277322669276961897420791879052817,
0.00003424463095723323667060076097619032427424,
-1.558965602793079850355351537607285865882e-6,
5.333075930641765046351014834403600198129e-8,
-1.462098694270389136919853917008344456464e-9,
3.346379783929474738201879626129204544734e-11,
-6.573419872577814904539501897942635035922e-13,
// [1.8571,2.]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003517781594125868197646847946072977298954,
-0.0004794248334006465598693887025492285878118,
0.00003273827765868311475369568207734619583503,
-1.493261285330063972714917802813323351569e-6,
5.117345392029043696878334761642518616605e-8,
-1.405241904235083746219645170484784987981e-9,
3.221097073244683747299914060665677535599e-11,
-6.336161657557330256983796761101498592073e-13,
// [1.8571,2.]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003355973833804525632834186033910919512362,
-0.0004583358872209664114500925956176482210615,
0.0000313584869919588237236278220939055199741,
-1.43285673365615783894093910279563662534e-6,
4.918346628182278212131545345404178706715e-8,
-1.35263144429911856664099373931548097393e-9,
3.104836462105520708979708689523948262482e-11,
-6.115399370772298294741753584727167335299e-13,
// [1.8571,2.]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00320835121054676460322535604736346662285,
-0.0004390188178868464465984393616215131762268,
0.00003008999140680589959050245352848297232597,
-1.377137078974673740426832428822395836318e-6,
4.734210019600168899389961787333579536432e-8,
-1.303809709807134941117550790081184151826e-9,
2.996659121512850668174471669868824185407e-11,
-5.90947354614241310433781083277313701661e-13,
// [1.8571,2.]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003073131725207924858002509615712562324369,
-0.000421259879694724310514681739942659328805,
0.00002891987865849389459969445436323839135283,
-1.325578827819605369064556870890105472386e-6,
4.563333949996792329192714383994336450072e-8,
-1.258382446746755125357848297350371630577e-9,
2.895751888025293846376417621354684199326e-11,
-5.716939718188716491679343298073875763568e-13,
// [1.8571,2.]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002948819157863069913621152972050514463501,
-0.0004048783012183738579855694005362247071706,
0.00002783715538423667112215410696473279472394,
-1.27773352762596006664094890884101986228e-6,
4.404338530335704789674834979434519338597e-8,
-1.216008173840182686805045818277785179099e-9,
2.801406951547262787230324767170867493383e-11,
-5.536534746741230599547412591941442789791e-13,
// [1.8571,2.]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002834148108528616753635920457336185499362,
-0.0003897201753787892705386953786836813015634,
0.00002683240408016937878626571506986030275119,
-1.233214809459200188713854778563454355555e-6,
4.256028576150684302496068634690771248587e-8,
-1.176389652448174809336335800948180272788e-9,
2.713005353985718996008681427521474379365e-11,
-5.367149270271453068370962453720072590334e-13,
// [1.8571,2.]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002728041227651524648781909531225539660242,
-0.0003756536571218627385032676506117051187254,
0.0000258975109986667230318891444261189406262,
-1.191688021664962278428258603727485234547e-6,
4.11736375220975016589021111049829279053e-8,
-1.13926695685635414480671114539786673266e-9,
2.630003491010369840785867687623500379786e-11,
-5.207805026510084470080878197881475269981e-13,
// [1.8571,2.]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002629575599853038931400839840433889777316,
-0.0003625651539808402226011910735476214707132,
0.00002502544845498706768772646922164442722765,
-1.152861870374909158047360186371472453007e-6,
3.987434318517351464698698556139111455543e-8,
-1.104411805718947780345904203231785051478e-9,
2.551921998845859670098035735144995766865e-11,
-5.057636066884611807187673226651096116629e-13,
// [1.8571,2.]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002537956077865881326579261670413624040954,
-0.0003503562783693388915584119787719471581361,
0.00002421009927789532883252671031025137239201,
-1.116481628387281582872732640496117076588e-6,
3.865441290367534719971658161011121728213e-8,
-1.071622895098347580732566463173214880462e-9,
2.478336549086140300487420155962218905827e-11,
-4.91587310739857843252528529389697356988e-13,
// [1.8571,2.]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002452493948585372015428502508708654724609,
-0.0003389413898900676374416891356994964354281,
0.00002344611419615455947610538105297869700129,
-1.082323579981733986585859381454580699906e-6,
3.750680103982473492467819588963035447359e-8,
-1.040722032778634956283332107590763105639e-9,
2.408870180406416303477245263417797220965e-11,
-4.781830422209819725775388974571813614273e-13,
// [1.8571,2.]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002372589729230473242442515797105766498815,
-0.0003282455987457092622702454499458192149854,
0.00002272879517963750016812423768508742335128,
-1.050190447298079674798632084930341798255e-6,
3.642527086609862927114025834693760182179e-8,
-1.011550917998315972619173018226995339935e-9,
2.343186876231338262873971244209909230382e-11,
-4.654894811110630846441867250519387371283e-13,
// [1.8571,2.]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002297719191219964621779826853513246947132,
-0.0003182031325144821874499266719711812776239,
0.00002205399939328022802578418049428976739712,
-1.01990760196352484755172456790219800711e-6,
3.540428185587541794053763594453023288304e-8,
-9.839684444208837192698774271507587549703e-10,
2.280986158642982734920219142976549314151e-11,
-4.534516268263794938047520619166775030284e-13,
// [1.8571,2.]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002227421927601375103302971521283275943669,
-0.0003087559915054915409351199595517862546722,
0.00002141805964125407117483284473310311570646,
-9.913199092307318018657450190300514792272e-7,
3.44388952874050272321780440867695109419e-8,
-9.578484298734731777561591428878525523179e-10,
2.221998515927283583093605543372208926098e-11,
-4.420200054087608287970292896278929017644e-13,
// [1.8571,2.]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002161291940538440582712206855353379660425,
-0.0002998528349771359600634829957432288189167,
0.00002081771809386493597618679252504960381776,
-9.642890848889553723140382571514209368185e-7,
3.352469478466220178710974248848294056561e-8,
-9.3307769617004464547896467935280758698e-10,
2.165981517682698590388761559191775472981e-11,
-4.311499930371977865910867032479615524885e-13,
// [1.8571,2.]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002098969844839951521388856412036565719097,
-0.0002914480533136981732946462377520560082592,
0.0000202500707826871722547797113819392333143,
-9.38691470407912124021742232207739459972e-7,
3.265771911115823350037009517851458557498e-8,
-9.095544376743671407879011609667289935818e-10,
2.112716499928192722347522173847938505093e-11,
-4.208012364433349075533147372141793553348e-13,
// [1.8571,2.]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00204013637319588701856643443234628222101,
-0.0002835009909572191139008808543274386090999,
0.00001971252087858482633253215586645430238333,
-9.144161511644892903379702256495467408966e-7,
3.183440506964564750501077700806296750918e-8,
-8.87186829232302257412101578550058951347e-10,
2.062005725053774454610316807487244049346e-11,
-4.109371544273524064136714733251004617907e-13,
// [1.8571,2.]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001984506936700533607165501969613551108594,
-0.0002759752922997954628488940434066677695498,
0.0000192027391744725287021329735949520989453,
-8.913633576344591715055896058292788908583e-7,
3.105153877974836820312345052429986925149e-8,
-8.658918335123955016699231532632529225143e-10,
2.013669939174356312141894780363237857036e-11,
-4.015245075468899159716437783899881661552e-13,
// [1.8571,2.]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00193182704609856805396639105464496773408,
-0.0002688383484422320765967910450493155940252,
0.00001871863051034149638403819509320923932945,
-8.694431011661092225385107936990405193842e-7,
3.030621393488261695489888532477660168094e-8,
-8.455941752342839373101498373768025757332e-10,
1.967546263541197349406175294824498997694e-11,
-3.925330253524629562243876060103993487754e-13,
// [1.8571,2.]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001881868439095624354188075630158472924038,
-0.0002620608271444060204084550327198565602683,
0.00001825830512450576475875702956767881145944,
-8.485740051740145326022641389778267900069e-7,
2.959579590025135480950326907260832812909e-8,
-8.262254556829994055187812431598378365306e-10,
1.923486367939392090420694159451650485001e-11,
-3.839350823932868415451552676433714991925e-13
}
,{
// [2.,2.1429]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.5900126832307232010933613458464683298194,
-0.01600024269753142599751114717526935674149,
0.0005172400286845131736516788072265181138142,
-0.00001494745121429049553253579282022360864418,
3.741803912199233237285086719780617917353e-7,
-8.144390752594270615563420752898622562882e-9,
1.558336501866554515723585186135007842507e-10,
-2.650684842412395976202035353414752171471e-12,
// [2.,2.1429]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.1120016988827199810229455853258760907961,
-0.007241360401580968895238048924126679031401,
0.0003138964755001925313233723987029103351613,
-0.00001047705104277996937220238738161131950935,
2.850536751124608666270441779584918606821e-7,
-6.544162531165255916464999027769604346876e-9,
1.29887487969003556491286215227436553215e-10,
-2.268746098504955328406784838467956173738e-12,
// [2.,2.1429]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.05068952281106678141554664863539435282659,
-0.004394550657000760812569880779258102704846,
0.0002200180718984610646927932614600904157435,
-7.981502980534554021421053270805547452512e-6,
2.290456875013425156633872267162638024598e-7,
-5.454531588547110330585934608059784239965e-9,
1.111720450459420338966281910151988408415e-10,
-1.981090523525446530952032621552383404812e-12,
// [2.,2.1429]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.03076185459900532492365396998127558306822,
-0.003080253006576739042092689143366233583931,
0.0001676115625912990109816986134325130177768,
-6.413279318672689418362344078019397632102e-6,
1.909086046207922787337042602833272907155e-7,
-4.668566991429110794909272498854878502924e-9,
9.707656639983811494469098616797588033763e-11,
-1.757074763614368213121144706106190939717e-12,
// [2.,2.1429]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0215617710460371726012384600761656925911,
-0.002346561876276645248222514003390098066303,
0.0001346788656921930455058959027017369540718,
-5.345440991018912170031743177440231967445e-6,
1.633998438124466711818601437942968575005e-7,
-4.076624072952778114783700367259558910318e-9,
8.60995036537179896740172183260011915074e-11,
-1.577915094699649706854503780620692981692e-12,
// [2.,2.1429]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01642593313393651610315808273565343017203,
-0.001885504119689304716474785020893760401692,
0.000112254260811458058225750091261384029157,
-4.575195682665798744237030676580862317325e-6,
1.426818417413094199585439776023486081783e-7,
-3.615642344460505350445301688279099119366e-9,
7.732043816643546208818329794891380575153e-11,
-1.431496355886161221241738498749159543372e-12,
// [2.,2.1429]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0131985288378251324307668676759254626871,
-0.001571559651359133860811255168334706532153,
0.00009607910933603789134629993822955849159294,
-3.995091619915271255800866395696059191701e-6,
1.2654748130787920760254960153580762413e-7,
-3.246967277583728317373796678867252662281e-9,
7.014571580635406551230765553968596126431e-11,
-1.309673035782495336828939926546605681945e-12,
// [2.,2.1429]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01100091755951393648376466824992852523976,
-0.001345107530703352008096130163425908897704,
0.00008389692401827272039029315278676764866502,
-3.54332952375985191302893222417996616256e-6,
1.136438540217748061943652861281794361377e-7,
-2.945667524633782182213749677631838320621e-9,
6.417619845602520838048368920662844940415e-11,
-1.206776213448528004145206693648012871499e-12,
// [2.,2.1429]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.009415752714923463551646254821652023163011,
-0.001174556936254725582304151542208982541823,
0.00007440991999900537297997470600237039204587,
-3.182027956310199215426247671137226417239e-6,
1.030983627157429792515560753204887042285e-7,
-2.6949808078877200568968461175987889268e-9,
5.913410306927034035664142145043394648628e-11,
-1.118743985254687615735849245541503513381e-12,
// [2.,2.1429]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.008221898553783078603319580704801966750674,
-0.001041738879985057083940255615638236865734,
0.00006682258708255957346738516221536975148908,
-2.886754196766670137581644283290061433412e-6,
9.432432767086908942794842718256929148393e-8,
-2.483241358313367573830145275084100414207e-9,
5.482039192502237065481231809262510825594e-11,
-1.042592590244909906551750992809775586632e-12,
// [2.,2.1429]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007292172159895399143148748632408662929517,
-0.0009355162191548808408201969677840980044676,
0.00006062183813214273868194540820374048219537,
-2.641081212912178830400806655324188619793e-6,
8.691344697208886205080747606668811268168e-8,
-2.302090431398841566740584287503656110618e-9,
5.108885733857909988597542297935453000293e-11,
-9.760826276372771396946141513518472564157e-13,
// [2.,2.1429]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006548613534084165466488962837367482745933,
-0.0008487057338491023609303826848287374233293,
0.00005546270547119600387992264651324091526444,
-2.433576551058030215453026147533789399237e-6,
8.057316456231189147339663260961591299143e-8,
-2.145387946584975423176269865528059344779e-9,
4.782976603000614350163168782006219130731e-11,
-9.175015636014429740062668370910443695129e-13,
// [2.,2.1429]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005940940136943716129753835654874334794599,
-0.0007764778765958988380252931547659106163161,
0.00005110510757225672357286673288580038458168,
-2.256048641553685939760157325651144737239e-6,
7.50885776226185664607337452215032470108e-8,
-2.008525605383757588527572738312706733999e-9,
4.495920175770876491386123439087803753244e-11,
-8.655180095517127017428107565589164125724e-13,
// [2.,2.1429]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005435345136171291489631342051548015095069,
-0.0007154715060107942609352829002816000404619,
0.00004737702147266355331331100601436432524651,
-2.102480205428369879449083825414528084457e-6,
7.029839570644894947260044485019074832778e-8,
-1.887979319509181756964785845625849029665e-9,
4.241192481549138362908859702213438394558e-11,
-8.190816663545624058958330630646033620501e-13,
// [2.,2.1429]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005008300542075559468262756039394616997774,
-0.0006632783006165306271285585570016432679581,
0.00004415208431403016293573073611903310687515,
-1.96835511014561425885214490209497781559e-6,
6.607927572421367664913985539926088114331e-8,
-1.781009336081844806175065732408678470628e-9,
4.013646919906732064850980053171486686297e-11,
-7.77353115892991576522142495497728502122e-13,
// [2.,2.1429]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004642948104315714048194208341921393224103,
-0.0006181291803956999771351731612487028610911,
0.00004133545731309070335731205007408553923686,
-1.850219749170402816585634558372203792343e-6,
6.233532632547755677082505578394350836625e-8,
-1.685454337457944938263487098789420449137e-9,
3.809170232016400198723532756346305091611e-11,
-7.396535927019004729750228999917015576249e-13,
// [2.,2.1429]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004326904262769899513357743902276671988725,
-0.0005786964023825809654649062660882026883508,
0.00003885461473260981180823684951157482284456,
-1.745389164668883283963113279629662191964e-6,
5.899090139299127107074926249482875051267e-8,
-1.599586962933358198590617105000213284241e-9,
3.62443637630087601034091607334340914178e-11,
-7.054284194510256051795990742021302844255e-13,
// [2.,2.1429]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004050874816678066445506306802082072563682,
-0.0005439646062558789602358682410887923884254,
0.00003665317245807657293524493749332093125686,
-1.651745265340198789149756189670212544721e-6,
5.598554330234662631842929375542111329756e-8,
-1.522010446655510018129096875308135931371e-9,
3.456727358275285959382487645584649354415e-11,
-6.742199682911190928257662565234289723702e-13,
// [2.,2.1429]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003807752243791152421621435314718770663174,
-0.0005131444144124415184298209523181961937419,
0.00003468665057217297757223716993758095941666,
-1.567595237686042720880637698748494042396e-6,
5.327036524890161399761776458806908325318e-8,
-1.451583373762550912114377042919859187835e-9,
3.303800738085761834886641940423964622683e-11,
-6.456473798203184188915742773625430376719e-13,
// [2.,2.1429]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003592010900887090340706246640024583690055,
-0.0004856131080098168237300116500516549892137,
0.0000329194999914345743273199096323475322621,
-1.491570251163958406279271384824046773938e-6,
5.080541771265890696832594393044885733614e-8,
-1.387364039328596351335455606730541071604e-9,
3.163790251093542260605723064674602761612e-11,
-6.193911523132115659600786730063813585305e-13,
// [2.,2.1429]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003399291756068717488654736014135694044782,
-0.0004608729998795028203116162211520418369947,
0.00003132297527446976238865081376765734369813,
-1.422551719203474141755744914193854883473e-6,
4.855774102135496622169457177120447200601e-8,
-1.328568713452673461117972234410038045338e-9,
3.035130293269786123325576107624422281019e-11,
-5.951812920224930924628649058629280657702e-13,
// [2.,2.1429]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003226110999156519474788382822305575833351,
-0.0004385216538420173211187875529752817935217,
0.00002987358610329862683697720095855478881373,
-1.359616770972238371628519972761664280122e-6,
4.649990462857765965383407063898152130616e-8,
-1.274539928578737815964787695575462357908e-9,
2.91649785623870622027165976735492218393e-11,
-5.727881019648090224167415764328674469924e-13,
// [2.,2.1429]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003069651576894120989798265472019065457784,
-0.000418230205445641709302958700366688410122,
0.00002855195219044177713016558012703545633451,
-1.301997351163030501375502033609455031903e-6,
4.460889716997040359646853166817701363298e-8,
-1.224722094925920131475545873193539439697e-9,
2.806767391191017688800867507903503170407e-11,
-5.520149495509548010781602431433628400336e-13,
// [2.,2.1429]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002927611438119491715815402714445098308144,
-0.0003997273306656646826033453091528641011267,
0.0000273419443744475739684346698899023070741,
-1.249049141567252975500909216762940832312e-6,
4.286527300329363756652595881472022090894e-8,
-1.178642545481479674562122010463786623359e-9,
2.704975369362936240655416134641531779089e-11,
-5.326925350882519607082372989580788827761e-13,
// [2.,2.1429]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002798091314659652537076058618184122756498,
-0.0003827872212417634339340324782309154452288,
0.00002623003197293546275791009112851387192218,
-1.200227664196473616379075913495779745329e-6,
4.125248878318048228967445292453620480784e-8,
-1.135896653119950742232404065851815705375e-9,
2.610292196965449578746447763537392070021e-11,
-5.146743105204726329568658280948328014447e-13,
// [2.,2.1429]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002679510548692343804032801985726836374104,
-0.000367220447620610323478377997437713227229,
0.00002520478094814836258701104378053438158235,
-1.155069705375439862462724579636674163859e-6,
3.975638256030872423809136269526965705737e-8,
-1.09613603625763265780517208950884647632e-9,
2.52199976641545687293329471379996019582e-11,
-4.978327882269334576096881275628004213823e-13,
// [2.,2.1429]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002570543133344272038016515359522393651833,
-0.0003528669332736063272844204732877387087019,
0.00002425646381290596511493483043403100117382,
-1.113178730518777294773662852334379785679e-6,
3.836476097930948402403557158470994062024e-8,
-1.059059131493262789619663784211121380747e-9,
2.439473368965399808973558677552144513408e-11,
-4.820565447611400087565488785433447624058e-13,
// [2.,2.1429]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002470068532915244071405227925791330585774,
-0.0003395904933802272240953334262886115595827,
0.00002337675334091540353426285122704345912574,
-1.074213325672336684445810606533936333713e-6,
3.706706932119202137293776697601574742599e-8,
-1.024403597830873507214602362496619788417e-9,
2.362167012622837796109391205690871322168e-11,
-4.672477717444661682302600151790583001402e-13,
// [2.,2.1429]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002377133453661590355438102098264024760207,
-0.0003272745467723729627935790082303280375064,
0.00002255847983913954049180521584671299425065,
-1.037877958701009824642117314770131445525e-6,
3.585412565114476271158889212244223056261e-8,
-9.919401509616075763880125663492250489839e-10,
2.289601421198557346970667998647488315515e-11,
-4.533202609427738419266740759890196300837e-13,
// [2.,2.1429]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002290921827406610532325182122393375229098,
-0.0003158187177475236949214691879215051914208,
0.00002179543713274110049428775843848723839583,
-1.003915535427093153108765944468939948119e-6,
3.471790501839970090733778465931838846478e-8,
-9.614675234893661657912156520697247665203e-10,
2.221354160906516364361411482602167084543e-11,
-4.401977364154106870269402882226390173237e-13,
// [2.,2.1429]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002210731024232665662891885333454207341289,
-0.0003051361198579576297167159723974507540448,
0.00002108222624398830592794613383556976099842,
-9.721013572264367072133272295225536382547e-7,
3.365136306413079608980514819054930680983e-8,
-9.328083186234694022701734932392892740518e-10,
2.157051467663343528626393284546356744378e-11,
-4.27812466020044966291871969436945050013e-13,
// [2.,2.1429]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002135952839005703211828327309302556578567,
-0.0002951511674154299395216756121845723538693,
0.00002041412850177400506886085497223415273671,
-9.422381820495539495581490989634842740546e-7,
3.264829090069770130537490793656594653443e-8,
-9.058055780802631515367782104949262015147e-10,
2.096361443268039146572472396209571447083e-11,
-4.161040992299932153754779745021145760458e-13,
// [2.,2.1429]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002066058171908009385554431660042406722314,
-0.0002857977990244405528828284901557667113421,
0.0000197870018230589783824044391840593242472,
-9.141521610404078944595351010468120100649e-7,
3.170319498820251229573640640107346283353e-8,
-8.803199248452368425394715258087526651721e-10,
2.03898836054289483082484224959151764433e-11,
-4.05018689414990493272162556346801321756e-13,
// [2.,2.1429]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002000584593171083683916629223723385686915,
-0.0002770180255224404435501530823916082124984,
0.00001919719538186644714666654076072340077299,
-8.876894750799678169584634281754690022387e-7,
3.081119713116432652015302605070553989342e-8,
-8.562271716421451865592505849116547998311e-10,
1.98466787237001331613948750426486051506e-11,
-3.945078673438193265321998486167644520107e-13,
// [2.,2.1429]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001939126178657082923183765739898789115748,
-0.0002687607353457547518904670704309601041786,
0.00001864147897669676431384396881367194295218,
-8.627135346930694896752128400338338928878e-7,
2.996795077493887445919491013963680951274e-8,
-8.334163089915391527286556972718678241804e-10,
1.933162961735426764092291426102613652472e-11,
-3.845281393348144528709531003485916707095e-13,
// [2.,2.1429]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001881325147420283085940737931228878694308,
-0.0002609807056733884575314237517809483046485,
0.00001811698422857147946035132764403295520119,
-8.391026363481414776967015153017246760905e-7,
2.916957058776742157877720689170729712337e-8,
-8.117878044537602737913029718541793131318e-10,
1.884260502564867152233194303739155876142e-11,
-3.750402886814670168670453768947328235746e-13,
// [2.,2.1429]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001826864939713719029596681832345223753436,
-0.0002536377791996432891636271110640609670724,
0.00001762115536332759095896333558827161729101,
-8.167479907545543707779728000290597771871e-7,
2.841257293428184225641438205987572721127e-8,
-7.912521583721839517778969568732075794485e-10,
1.837768326621794786897286947324780907047e-11,
-3.660088630646490630578404706886961677814e-13,
// [2.,2.1429]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001775464454397502854999962470957034346687,
-0.0002466961750862372553809516074719046894929,
0.00001715170780586187983736437325535543457625,
-7.95552056120745640056783939479405180501e-7,
2.769382532651837123586402136358422424267e-8,
-7.717286721356391670581399416352402875332e-10,
1.793512711751856273075163678020595541641e-11,
-3.574017338904350479955116023837934474914e-13,
// [2.,2.1429]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001726873225603660622320571865656471041221,
-0.0002401239092817253190715501922447902480749,
0.00001670659317855153175384415928998992833645,
-7.754271227825898260990956457244797025969e-7,
2.701050331310249758698282547826246802092e-8,
-7.53144393383667306687647963209328750059e-10,
1.751336222572884036230261367722989730472e-11,
-3.491897160581345044847896894050245500203e-13,
// [2.,2.1429]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001680867364972077071787323712603942347604,
-0.0002338923044993881053991178953857186736385,
0.00001628396957844991093295079701596628405001,
-7.562941061005627087133804826472228530744e-7,
2.636005356143320106146250895972328706004e-8,
-7.354332092197887738627565761144557640296e-10,
1.711095847281499926734962441618107375061e-11,
-3.413462387142867641314128899470599093211e-13,
// [2.,2.1429]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001637246131495716579556819266457526128203,
-0.0002279755740979727371813811242156376519472,
0.00001588217622812700771999690156378300653812,
-7.380815127608887509609343776145719229591e-7,
2.57401621201474380154796907049791863128e-8,
-7.185350637775469652247047799757536858243e-10,
1.672661384298219918269424644600473326175e-11,
-3.338470591970620219955139110182053029726e-13
}
,{
// [2.1429,2.2857]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.5745150992998683751819006385473489586199,
-0.0150091480775353971325260148044533478466,
0.0004745635961567578554536914876193873530476,
-0.00001352914781879410142128150728826093373768,
3.357056407261980979751879706351629517678e-7,
-7.262969369669517228232574466286712206325e-9,
1.383669896781332224644261494604442486419e-10,
-2.345966376863515740795088262502118732947e-12,
// [2.1429,2.2857]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.1050640365427477790821650560513890047504,
-0.006643890346192654031891135673296209487459,
0.000284112104194757299857205955052288494919,
-9.399758018571924124686695297501885764312e-6,
2.542039268561633092284738846888992062568e-7,
-5.810662474225371296924121416638152108806e-9,
1.1495581573603064226006772797648755686e-10,
-2.002919984932098825059124913371116062054e-12,
// [2.1429,2.2857]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.04650723242334857747453176086205136344525,
-0.003977569458724897629528721322630779864882,
0.0001973949183900822827512219209287411464171,
-7.117710020155848720089339877586349244412e-6,
2.033731856395368486477290515695210461257e-7,
-4.827489698025153731643116972981309804884e-9,
9.814614599112248715500205225010179390228e-11,
-1.745507583503490919592823425968118650994e-12,
// [2.1429,2.2857]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02784298621107428273522589449836897616667,
-0.002763528857459642561308710497646654751239,
0.000149471910423337285046291194303444970426,
-5.694449258283406887019818626534422487043e-6,
1.689621385713855785894890257831240974319e-7,
-4.121558515310477707630939319179721143546e-9,
8.553262198032069890680354175969161723976e-11,
-1.54564944535654028614476457910587226895e-12,
// [2.1429,2.2857]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01934470200221749732066903406284602483233,
-0.002092606745925368291679931375761586521277,
0.0001195834344240099601386067742961515609504,
-4.730939934147199869131927561053056659933e-6,
1.44254547256990957158519983974424558741e-7,
-3.591850295650599778742837672988153900995e-9,
7.573931522977498438858706190797238340602e-11,
-1.386212151768051989335464087617650356894e-12,
// [2.1429,2.2857]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01464824722147757748557576533052656157976,
-0.00167416808193491271753549024104077818336,
0.00009934973861714459115621305991237881200272,
-4.039127372265132653859440299072746008812e-6,
1.25714759635850154099795696384800563361e-7,
-3.180580170924560174531036756188133545156e-9,
6.792667358782280942680534752303073624658e-11,
-1.256188263220066550092797777096390259696e-12,
// [2.1429,2.2857]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01171917657354438851067700640957652302772,
-0.001390896340638903005425798436049161801883,
0.00008482167481761694483378107384179959192103,
-3.520013314655014265053036897172759974597e-6,
1.113203053269029357391672716849877343611e-7,
-2.852489716644202453783753124852873197241e-9,
6.155532236332529196441875594819723378084e-11,
-1.148202077980157327648862556671973246174e-12,
// [2.1429,2.2857]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.009736274384472320563597420346114592072745,
-0.001187503447445604887663080274546256703259,
0.00007392027960780084058040354732038379383534,
-3.116968590447236047823824044633667033902e-6,
9.983713947533172125218783846252756771773e-8,
-2.584927115040617320928038211633674638694e-9,
5.626384491412151040575923815530927336444e-11,
-1.057135559587951822538605402960734361415e-12,
// [2.1429,2.2857]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.008312524132119233771821101635409843417581,
-0.001034883914508255407899310394722828721341,
0.00006545634039943437198466780856585392285019,
-2.795439943564027973959426932851542482322e-6,
9.047244846088680521168857051987446084985e-8,
-2.362714238774570976480423436621357296054e-9,
5.180145213486242282277634002137532196127e-11,
-9.793310144919018026850982541174969336863e-13,
// [2.1429,2.2857]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007244187401557787441884468891983374482679,
-0.0009163887655911904942064472546520779512636,
0.00005870423881488427508456313661104842359548,
-2.533228592533684428135586171986623718685e-6,
8.269499782793993454619045462690325923469e-8,
-2.17531895067829311108462166082985013274e-9,
4.79889130577409787571755696034165511746e-11,
-9.121077504131432550003106442203190493871e-13,
// [2.1429,2.2857]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006414721359138333071033647598997689875448,
-0.0008218593434075464118754119420952515146177,
0.00005319780044324466068606489319969192927244,
-2.315459972520183500518317431411604703933e-6,
7.613616277656944665184414871730421340086e-8,
-2.015214303042746717032740374159864899072e-9,
4.469487072013421434611386990157621080842e-11,
-8.534579247743758218838985164022422018855e-13,
// [2.1429,2.2857]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005753015403852824516881321021404468240138,
-0.0007447692062046422089051996022271599617185,
0.00004862465942295901336383531116156426344106,
-2.131812589065849661476399285186753259748e-6,
7.053250013769663219325749467123580965557e-8,
-1.876883878182267836787090245740292802901e-9,
4.182093847547598074556915193579175759524e-11,
-8.018488811475605219097801389267023206702e-13,
// [2.1429,2.2857]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005213384443432495115874358668188623901649,
-0.0006807452319206878310354465623606790749733,
0.00004476806437041610342072058987586434449205,
-1.974910033390010985195050250546803112757e-6,
6.56909352929042526013931181781985044184e-8,
-1.756195883036351550825152877243918307064e-9,
3.929201429958976530705523832133369735743e-11,
-7.560910286040626939586089527058426263188e-13,
// [2.1429,2.2857]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004765216623444814488552493834484432875023,
-0.0006267529011851269775781008458167339856312,
0.00004147311070122178563569301433716826382833,
-1.839346216140381673626610181296193712292e-6,
6.146685548553835565375622858653508122571e-8,
-1.649996383981016253289876902088167212991e-9,
3.704980675306632648522750327065424121522e-11,
-7.152473204720255898175281222208256486889e-13,
// [2.1429,2.2857]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004387270308295888530391946523726460564807,
-0.0005806235498164423457954175175821514225197,
0.00003862627053897803016159350900936887727406,
-1.721071980101436607758755980124945124619e-6,
5.774987303913409479795706544061840395117e-8,
-1.555837421021081697347510924189604147508e-9,
3.504839935055192552879302922288906357607e-11,
-6.785698610679308517493765183718976718537e-13,
// [2.1429,2.2857]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004064364848715096122467328331224093577026,
-0.0005407677875450621076779807377372895357985,
0.00003614251158215878657176272093939363078308,
-1.616996470308565894208322546590517888897e-6,
5.445430935416586079777265725685609218727e-8,
-1.471790728276921848618129206392875604319e-9,
3.32511442253145188353149638961010211425e-11,
-6.454546345228441757059728885713523465948e-13,
// [2.1429,2.2857]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003785374512815434468909564327841707859523,
-0.0005059951621496220387275464155776224087586,
0.00003395692587650722820882660687505840845223,
-1.524720685955793343960138692271423596356e-6,
5.15126751250986940950354657575876030727e-8,
-1.3963172802342806609766755669808557967e-9,
3.162844379352975808653725406239502771275e-11,
-6.154086032679313281420136602834892809578e-13,
// [2.1429,2.2857]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00354196613504735399839460578576557314423,
-0.0004753969622705269625634007647804184643836,
0.00003201913440509783939205102442322007854915,
-1.442354926472267078735457414880449695114e-6,
4.887110445914311913171731942143380156162e-8,
-1.328174130754140067713505422239439018581e-9,
3.015613854396828532416559011711545164054e-11,
-5.88025415019931494202197880120373556627e-13,
// [2.1429,2.2857]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003327778735893688476394628897588267436915,
-0.0004482678816708199895946914614390357526455,
0.00003028945345594271750894858768684978110209,
-1.368390946846684113225462423211763554806e-6,
4.648609424160908579404359644481967157387e-8,
-1.266346707063089010579612729428361939201e-9,
2.881431665288627793763979967863608598333e-11,
-5.62967206724030349224189629379946658497e-13,
// [2.1429,2.2857]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003137875171695739675889489880608065390753,
-0.0004240523483826707597876470455534658193834,
0.00002873620988380448874902874932375361743313,
-1.301610659856661323398624495468029943645e-6,
4.432213442557546082373927921175220667653e-8,
-1.209998818756911178634523503852499129716e-9,
2.758642235618493652788619105293608559272e-11,
-5.399507960220291089022353109449357680417e-13,
// [2.1429,2.2857]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002968366438678695076741249486995302736636,
-0.0004023069383727562732942945983328330273916,
0.00002733382385701309805533573117815800649764,
-1.241019784179066954851646539330423235556e-6,
4.234995834702070424277241115294641628156e-8,
-1.158435213408110690414248778438787496471e-9,
2.645857931501293662258389221191147457083e-11,
-5.1873707634932222085838138180016344807e-13,
// [2.1429,2.2857]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00281614856860929368009832527331404924613,
-0.0003826735339976959578152319485929750657372,
0.00002606141546778277049947854284541811297543,
-1.185798853213357911073701147703955717659e-6,
4.054523217109027276907562500636458186811e-8,
-1.111073161009083381842342667698015891566e-9,
2.541907096270378927620841825898043893605e-11,
-4.991227825952330583840029554394335658494e-13,
// [2.1429,2.2857]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002678714737983871479937373909165651182031,
-0.0003648598165484891341571791768725111882305,
0.00002490177591750209409740015349026058290774,
-1.135266519576814762254498243211831274932e-6,
3.888756034761078529808122133643639199629e-8,
-1.067420630965449667121921255611383379647e-9,
2.445793701198698006420942310653986887389e-11,
-4.809340325138154701011462685278675529539e-13,
// [2.1429,2.2857]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002554018715839423721968129678689400392397,
-0.0003486248628445761806471190390468024116648,
0.00002384059691113395480446081550442999881665,
-1.088851707858738642612699055507799417061e-6,
3.735972180585920485431737068795062339031e-8,
-1.027059347316397612971351940885946089823e-9,
2.356665697601540754687599717623459322009e-11,
-4.640212134410276466691892729923264094767e-13,
// [2.1429,2.2857]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002440374039912033054533979759857534345131,
-0.0003337683567554376270510681775594874778124,
0.00002286588586505367120647202719374925178997,
-1.046072228073828839820230217732885305967e-6,
3.594707688727688436409853833292354749753e-8,
-9.896314981490920821239947900419572890955e-10,
2.273789961098473450591607789464511268057e-11,
-4.482548989308623753912008105735570049067e-13,
// [2.1429,2.2857]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002336378497288063186044654185903324108709,
-0.0003201224021103280434586672068690936384285,
0.00002196751678956992377362429663189500038082,
-1.006518169778047671209440356333051400004e-6,
3.463710217497553978630675360436605705575e-8,
-9.548292134213849964908288215193800543997e-10,
2.19653228260138156535345237884483796187e-11,
-4.335225615516966108833136815368214514039e-13,
// [2.1429,2.2857]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002240856814772296107169411279109725016428,
-0.0003075452350535690524418066306037718914849,
0.0000211368815653579171595728072032388518878,
-9.698388772946835852522875715332898796418e-7,
3.341902221753345952124883693804380910284e-8,
-9.223861621766010948321140784672133884351e-10,
2.124341260584518565222770997201909497145e-11,
-4.197259067237171051912968637644548763622e-13,
// [2.1429,2.2857]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002152816645374983175948286277465347191931,
-0.0002959163419146136033683813757148747035143,
0.00002036661642320670524924819153393629269307,
-9.35732637980559962225811780867034304039e-7,
3.228351543151411596066404982838348707563e-8,
-8.920707881135648146488514852284018927686e-10,
2.056735236532680852311682883831215940961e-11,
-4.067786950948978046237610534597332587976e-13,
// [2.1429,2.2857]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002071414393402295037989090414079568907074,
-0.0002851326299245085248199355037618370435918,
0.00001965038539760957590396650020213164316459,
-9.03938447496486305379859004458916970198e-7,
3.122247734641802455548811439761557199563e-8,
-8.636808231461862548663233510368771007195e-10,
1.993291624289672034727312951225883169888e-11,
-3.946049522618794950266471977590200357404e-13,
// [2.1429,2.2857]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001995928409471559493391520700722097388257,
-0.0002751053955661599124693644367951040275438,
0.00001898270739744352591865794175155671757515,
-8.742293806658524479285393349153625468142e-7,
3.022882857926901642457913084360918849022e-8,
-8.370388062851490222868790975777308897637e-10,
1.933638137446160053062237587694147334659e-11,
-3.831374878774931757422187801208578718303e-13,
// [2.1429,2.2857]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00192573776896311921189157473413934413582,
-0.00026575790356384577968527909686734520424,
0.00001835881699399973932076168963752115249195,
-8.464072147629960592875876675884506293579e-7,
2.929635799547395199294972757845740065345e-8,
-8.119883996028297206597579267901605224013e-10,
1.877445532762057412845483776516555729164e-11,
-3.723166635946773777283171046535506039456e-13,
// [2.1429,2.2857]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001860305324946920287092519416112982465267,
-0.000257023437915642754722974301058497316943,
0.00001777455151003930496091491595528809815266,
-8.202980380172343690172535184193402494539e-7,
2.841959376759544038443913215864460735923e-8,
-7.883913408571755690537700751827443612565e-10,
1.824421572919787684371616977022514897978e-11,
-3.620893624552310596308239274687438360467e-13,
// [2.1429,2.2857]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001799164065409499116802024345392234090913,
-0.0002488437211402061278402808256713817625119,
0.0000172262587983778826817091661442219538983,
-7.957486392584664613526524842428351773452e-7,
2.759369671718800999005236197731032817084e-8,
-7.661249081730231317061064640060721635787e-10,
1.774305976383918355230754358318083563339e-11,
-3.524081223620784926754915388416889044033e-13,
// [2.1429,2.2857]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001741906047981442732843427252727650723536,
-0.0002411676231769551783750845265859694540056,
0.00001671072142444335147393299363822744479693,
-7.726235214885574537170309905516073541247e-7,
2.681437157864464927692660420923105214814e-8,
-7.450797992576804719420523488774921335386e-10,
1.726866171291973995100717389830592996478e-11,
-3.432304039798181593707998868901601453612e-13,
// [2.1429,2.2857]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001688173362238686090598624411364589270509,
-0.0002339500999418802495395123203154807146394,
0.00001622509395127487720071988523207684052524,
-7.508024172690173676333031655810471879648e-7,
2.60777927717425105848192549185484157432e-8,
-7.251583482680785051784849914392890171336e-10,
1.681893708062034704759922749821641233019e-11,
-3.345179693730509584269484023204059970289e-13,
// [2.1429,2.2857]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001637650699593161592567592110941100858805,
-0.0002271513153175296970752459700803366636339,
0.00001576685076266416886560822035256059971429,
-7.301782103522601451615149284168362550807e-7,
2.538054199199348857948949557518198841852e-8,
-7.062730193045860100083617835328351460547e-10,
1.639201214630592722657483131937757713064e-11,
-3.262363523414398788547593381782549644946e-13,
// [2.1429,2.2857]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001590059207222707728955771865551412381917,
-0.000220735910676987477449089709653586976584,
0.00001533374241741191793961220610304623931064,
-7.106551882114021363355620880456461318818e-7,
2.471955548292798718542671213846707343931e-8,
-6.883451277809191127445549666687268349335e-10,
1.598619801017050367355060762181252015776e-11,
-3.183544050588966454531743497494326045962e-13,
// [2.1429,2.2857]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001545151374738912195043206537103330297141,
-0.0002146723938434632988123949950135695262219,
0.00001492375895245356658176827106603108922042,
-6.921475656641923559004471685013640169161e-7,
2.409207928404195987457983445144443393456e-8,
-6.713037504873179842483622627285676753316e-10,
1.559996837789068981760541289484620150881e-11,
-3.108439085053785268682166785699070012713e-13,
// [2.1429,2.2857]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001502706756904242947900654470258746682803,
-0.0002089326253340533764168281116410498166557,
0.00001453509887896184301709355560110616351454,
-6.745782318155164865750634763091409226434e-7,
2.349563108300827437453691014189029789485e-8,
-6.550847926719923106301929512985787985141e-10,
1.523194047120671053216814895465613736785e-11,
-3.036792364685908561703823445976807572977e-13,
// [2.1429,2.2857]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001462528377338373494300085088266866031297,
-0.0002034913843051759281897907254609881588462,
0.00001416614286813934559335922909811823432298,
-6.578776819193037307284762911941406013772e-7,
2.292796756352746074543943779660645300518e-8,
-6.396301863945821958266080521652966395039e-10,
1.488085856350469710934986386701580850168e-11,
-2.96837064721581760182826128909554782578e-13,
// [2.1429,2.2857]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001424439690136231359742729345542858702233,
-0.0001983260001536673517758941938465554493753,
0.00001381543132031858665605913484938911176589,
-6.419831031183378905442178009449325414609e-7,
2.23870563476989794140108072328198028063e-8,
-6.248871991153285248454142124847085378521e-10,
1.454557972908220795883720911649962229631e-11,
-2.902961184512354248357763239999255623025e-13
}
,{
// [2.2857,2.4286]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.5599673142493835785142982532149159879971,
-0.01409930100699517692186633743161331537987,
0.0004359197838277337061078105459058943688952,
-0.00001225626809492917412475089733084606911518,
3.013863727902209306075192290845548494745e-7,
-6.480192917113689839310215328982934382474e-9,
1.229059419366949686594261452929176925273e-10,
-2.076918654497493555437967358432506470936e-12,
// [2.2857,2.4286]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.09869510704896623770793982207908263553881,
-0.006102876973586544685847811564991831391063,
0.0002573816299935841889039730103820901964075,
-8.438818507214699496671622768471293183139e-6,
2.268067511452127280095316533459666331352e-7,
-5.161386308239488470315300411005150338286e-9,
1.017720661285137960397719435841892689228e-10,
-1.768681362040169375516928948673437419691e-12,
// [2.2857,2.4286]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.04272013881510581214220601095356143980567,
-0.003603342819908676468219692128400705939154,
0.0001772151886515719276796573356678551146118,
-6.350589092153485802981212083425975287628e-6,
1.806485199452030578677889306001949427874e-7,
-4.273849934094460855708159678783460511955e-9,
8.666808491796463452474068208553311666179e-11,
-1.538254562048132374538965718849265783177e-12,
// [2.2857,2.4286]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02522339973936073468755556744742642932109,
-0.002481012641120678985661640450125743141848,
0.0001333623709352798404326622892803216982263,
-5.058158611586186990261462144242494846724e-6,
1.495847469381230244698204386794440965807e-7,
-3.639549607007091318654441384383801978652e-9,
7.537689013092833631976472234938565143594e-11,
-1.359897334616874187055825499615544151238e-12,
// [2.2857,2.4286]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01736708848784475236559218721390620225403,
-0.001867073193092728357352682786340781079606,
0.000106221330843361194817589404175606966563,
-4.188372961844182995968233415107453910233e-6,
1.273842355616725706446166931018773548451e-7,
-3.165372646314377288771851927482338207878e-9,
6.663715684248986638672887664928507840693e-11,
-1.217975914849154859504309072986338543402e-12,
// [2.2857,2.4286]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01306951235164909801380513982564691399854,
-0.001487098631805980100147365777455605853958,
0.00008795583219877465885588870595866729930633,
-3.56675863879228282323446564692587858726e-6,
1.107880419967888151751727220851298727414e-7,
-2.798347156759552550580067126363733539241e-9,
5.968281731760131955073533264118585961291e-11,
-1.102486041803482959839125257858436338422e-12,
// [2.2857,2.4286]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01040969042264186025241510419392081326504,
-0.001231381650781862090309428749005916195554,
0.0000749019314146810066828693388760507918417,
-3.102065215235765983750114601358910100747e-6,
9.794214991235068548328779681768874225778e-8,
-2.506300798692992484466995816173973371301e-9,
5.40236535997346478200134194747851056609e-11,
-1.006747106883425029457226679802641245094e-12,
// [2.2857,2.4286]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.008619671555473034216864109257429515605234,
-0.001048627039804629679270303337565738247705,
0.00006514336951999095484158356944916414619206,
-2.742380233722700027678448997013300302251e-6,
8.772052742266402292354539997017027889234e-8,
-2.268646151153702926053258144864493599662e-9,
4.933230933100659543689302123937358666397e-11,
-9.261371968854669993588686960486326769459e-13,
// [2.2857,2.4286]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007340389278632407368339061843565777474647,
-0.0009120071732790361158816287756607887311126,
0.0000575899849082138098640945104830550873253,
-2.456174801324957858890527725210803176036e-6,
7.940261479558765290878070933735965077299e-8,
-2.071635482545331747858291619813554052311e-9,
4.538230598485092881473874602641209765165e-11,
-8.573618106512556056547788414166053966781e-13,
// [2.2857,2.4286]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006384050212953252449673638803736812392048,
-0.0008062597887142140440392387083751345743909,
0.00005157967082785881899999431046885145394037,
-2.223273245448488179300686955067826795493e-6,
7.250724142636567593791344414416457445236e-8,
-1.905757598100232257488899715590855616531e-9,
4.201220943194357213065880935818413310429e-11,
-7.98012013339457689657130226477626064274e-13,
// [2.2857,2.4286]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005643818520999497968803233146646635869153,
-0.0007221153915892946836017875749229735545665,
0.00004668873815445084119173926345208378912355,
-2.030202789089790816346398218413283044542e-6,
6.670151549898109617747697871072929401881e-8,
-1.76423293960530052203477936858783452279e-9,
3.910397914301247235297895228655893416606e-11,
-7.462872172420715107516733734381662602519e-13,
// [2.2857,2.4286]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005054807741125062465253376845683870967728,
-0.0006536423341616273995152515960702064128828,
0.00004263425857091632338005030077689696229778,
-1.867642461346799499150981483030191835318e-6,
6.174815247663441695931660969950853413985e-8,
-1.642104319300904402542905640902572646838e-9,
3.656938421111132980995394300189824585481e-11,
-7.008154989582640643894808644855843537701e-13,
// [2.2857,2.4286]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004575496339131391494050925633565589377049,
-0.0005968796199921834870936707181729202544433,
0.00003922049168831183499424222440002876848827,
-1.728948295147602304871324752950552148031e-6,
5.747365078825694491176479619722188914792e-8,
-1.535666437746707068457803034791337231895e-9,
3.434119873061608036069344627414592004208e-11,
-6.605337849190368943194638896754022045121e-13,
// [2.2857,2.4286]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00417815733994528412271698457787978552226,
-0.0005490868836357557348745491000391688709242,
0.0000363079141981271946632489771722029009815,
-1.609262246469614526907304102733400796684e-6,
5.374832495385011005346683801379697070245e-8,
-1.442096120456236112521797172255579127345e-9,
3.236733077434270101670279852645345400792e-11,
-6.246059399494666958715386174218074397942e-13,
// [2.2857,2.4286]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003843608185450289871274479940947530042038,
-0.0005083107987732022544809853107445434502167,
0.00003379450717588809855024420254700653852863,
-1.504953121846843314997882720911382607658e-6,
5.047336386672068855878629126555915822961e-8,
-1.35920575640460112152613174714481593161e-9,
3.060680865212491858935284336268773156653e-11,
-5.923655170761995491780449105125582109672e-13,
// [2.2857,2.4286]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003558175591412415521297413733007517190267,
-0.0004731231004618833171596551821948276346344,
0.00003160401555880867641771074878519111706474,
-1.413254210270879772392513308934627330491e-6,
4.757220114126927708365346240112370876464e-8,
-1.285274736192558074644902092151551692783e-9,
2.902697559263143339864595133485038939995e-11,
-5.632749472893005365116368525020438586061e-13,
// [2.2857,2.4286]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003311861703233182971687317850978609957599,
-0.00044245621782279716760768441525778102422,
0.00002967833841571232465291638398976891004095,
-1.332021652927819594665739554478840885474e-6,
4.49846154487460817170810710349908802844e-8,
-1.218931639786639401765939261673473700559e-9,
2.760148999838446213420924028685063911716e-11,
-5.368959293140207004815961877412771896844e-13,
// [2.2857,2.4286]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003097193524759579935469271454029304685736,
-0.0004154967378194717076468914635720600949908,
0.00002797245471150703892374113382267562991361,
-1.259569252598572452249374363921615236179e-6,
4.266260708816559376712784505222240495324e-8,
-1.159070255415961332374477549071920877317e-9,
2.630887451218029310654146815112992952797e-11,
-5.128676001428760742054743121003709569936e-13,
// [2.2857,2.4286]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002908477164736301725517595349151855764518,
-0.0003916143659606191693502234769312239990967,
0.00002645095430459191063640794404804534276886,
-1.194553017643835114632541107161703540114e-6,
4.056745864770251958877669705099936686864e-8,
-1.104788646484769704105166460393197057449e-9,
2.513144634860875061151413781950494515441e-11,
-4.90890207304172176600559961588428075152e-13,
// [2.2857,2.4286]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002741300561724333966445501532442771031152,
-0.0003703133602638270775343230016771065150532,
0.00002508561337054156210227998265050758046669,
-1.135888860522694109192404888949884128639e-6,
3.866760234663676927655003149049729696677e-8,
-1.05534423013668868310631138761178139481e-9,
2.405451721637761011460368240224336673114e-11,
-4.707127343439804967636785298396047464969e-13,
// [2.2857,2.4286]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002592193521846789332056626198971516519963,
-0.0003511985871871403513641577976080323884196,
0.00002385366607099680203095848348245727695044,
-1.082692883366714623648628815990385437625e-6,
3.693704778510672652060557577919394829239e-8,
-1.010120177551857519860735149841223381324e-9,
2.306578695232256628159737728194870709944e-11,
-4.521234087687075940644845405735412636452e-13,
// [2.2857,2.4286]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002458390110309982256580232583422779846547,
-0.0003339513249935304884456826832006912829671,
0.00002273655055072049221358960719050758243146,
-1.034237354972744488584755571203199511623e-6,
3.535420595451261155792645725443761562852e-8,
-9.685999493360656346076970327776513913008e-10,
2.215487839914831026294392058813854791203e-11,
-4.349423401130679529346636651424481878179e-13,
// [2.2857,2.4286]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002337659274954713223322053811426364779926,
-0.0003183117077096777040251021433797806939808,
0.00002171898445444643094365892638214438402529,
-9.899177830939830200088039083606551354967e-7,
3.39009979761390378954415570555255214947e-8,
-9.303477625489595889081824407721292723252e-10,
2.131297666170723937700285949553762995512e-11,
-4.190157517049537882360145099324095803385e-13,
// [2.2857,2.4286]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00222818195396774373906089433374757251785,
-0.0003040657823618553033352345241635763881226,
0.00002078827344499179854149805126047177521859,
-9.492279591212343968100176646666580769958e-7,
3.2562171447144513198604789214298942869e-8,
-8.949934411785797433168832375810533556429e-10,
2.053254645625603851459073865923890740273e-11,
-4.042114184360019014537892295758843733286e-13,
// [2.2857,2.4286]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002128460476532986940474597459664123411306,
-0.0002910358282295039224931857222550556267682,
0.00001993378714156347814473143446464337227695,
-9.117408157704715014824221176990015244765e-7,
3.132477020717203757974992211295246135984e-8,
-8.622205461751104647552887563159888164234e-10,
1.980710855540458370887375750896351792759e-11,
-3.904150268202828915300960053700104966079e-13,
// [2.2857,2.4286]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002037250797606527280424910486701448206056,
-0.0002790730199815200224256454294593616230906,
0.00001914655713119689625313757632302276278186,
-8.770935805478183098093087362500862691319e-7,
3.017771888953183197262313609584678750638e-8,
-8.317569872366178176509216720131482806594e-10,
1.913106142628413342843098872324614353207e-11,
-3.775272472951770023070879974916133063151e-13,
// [2.2857,2.4286]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001953511139870639985435496398830440807883,
-0.0002680517998363996588139450478765132796216,
0.0000184189651915206528219760716448099375924,
-8.449761431825736046920185275863779299124e-7,
2.911149433370335795353220032828682034753e-8,
-8.033675325018518616549556707158706395832e-10,
1.849953776944821896835598256339980090822e-11,
-3.654613615784900055997386927309970872323e-13,
// [2.2857,2.4286]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001876362598854797445308109856574137988182,
-0.0002578655126809433052916244830486866631626,
0.00001774449900685001917841572450749716546844,
-8.151218551771918911070173342118285259574e-7,
2.811786342458438451496129031021485088173e-8,
-7.768477839102316463660980007386019171096e-10,
1.790828825621498649588687750041903168442e-11,
-3.541413262686456816280547464501616496906e-13,
// [2.2857,2.4286]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00180505858876660297550620172777150295338,
-0.0002484229860955648258279274210472523755384,
0.00001711755895873653717448292472072180691905,
-7.873001893061951252766294664377998410272e-7,
2.718967223009127821894181045847421077834e-8,
-7.520192947667749974242863869595852684739e-10,
1.735358664246754267422194247933811957506e-11,
-3.435001820357337939301199688122977572343e-13,
// [2.2857,2.4286]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001738960902668953623839915629734656884441,
-0.0002396458254219858641532317491695445083977,
0.00001653330397544516544880500244019905368781,
-7.613108354689294724832245838948497687181e-7,
2.632067511593222360601058339666046431111e-8,
-7.287255850148335094085313563197456048209e-10,
1.68321518168460357271649256905291036257e-11,
-3.334787386292544744514156232115867907768e-13,
// [2.2857,2.4286]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001677520777953900896444543612642442577369,
-0.0002314662556559158924911386393205674027691,
0.00001598752754486217129820508028117454136142,
-7.369789159031727709376716057688435038067e-7,
2.550539528015351465566507718137884897389e-8,
-7.06828867670154646158294063368751119836e-10,
1.634108336431365842606567183665677188725e-11,
-3.240244815553017340913634148035224264562e-13,
// [2.2857,2.4286]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001620263789591411098905521776060166917679,
-0.0002238253856277626986533641820386233935202,
0.00001547655723398088738322430372565296981355,
-7.141510801523949898725949210318761871144e-7,
2.47390101783322045028209619398842719361e-8,
-6.862073428334023497395026820176189775147e-10,
1.587780799181478090539313823101686083567e-11,
-3.150906580775437785529530670289439235125e-13,
// [2.2857,2.4286]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001566777699394338745922985850451880511338,
-0.0002166718012754329806670712559154131737414,
0.00001499717268321418131792233521254542111867,
-6.926922969711224205642509054423648235455e-7,
2.401725681401472885778462797155242894981e-8,
-6.667529478558099568745466145827570060418e-10,
1.544003474103624426444805865364388305762e-11,
-3.066355091843890089453762574505875179399e-13,
// [2.2857,2.4286]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001516702608928030723703360768157240571609,
-0.0002099604175647069216653914679727100716351,
0.00001454653823640710365575408941217321910497,
-6.724832024571937993091996993961282225872e-7,
2.333635299451509918445564436981264610272e-8,
-6.483694765183504052424272103980570960678e-10,
1.502571735370696266326100550448526977396e-11,
-2.986216210644383885187031750850064217681e-13,
// [2.2857,2.4286]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001469722922952948314193241290551274586556,
-0.0002036515353094152622269350462295615132835,
0.00001412214725161426645028347349278265634059,
-6.534178952140882268346319782363408332358e-7,
2.269293150218615053980246659279192714835e-8,
-6.309709985806204777113149827463268657063e-10,
1.463302249296510700244656215160235502586e-11,
-2.910153749687078983216336377845476839004e-13,
// [2.2857,2.4286]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001425560747165906701456155498984650715555,
-0.0001977100615223226006835064785490748338012,
0.00001372177579950872954415284413814888041844,
-6.35402093146501798739217667996975203393e-7,
2.208398477863073875471114746817414834804e-8,
-6.144805252540303900047736718539493044856e-10,
1.426030278581137921472579184848328045786e-11,
-2.837864784946732043408977992642558771925e-13,
// [2.2857,2.4286]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001383970430656258073826741101176707018399,
-0.0001921048611928518015572720456519867758901,
0.00001334344395608910979232465262741714509925,
-6.183515846182454626364162281448716577886e-7,
2.150681821626358955189660705889982313978e-8,
-5.988288771352156837071463211384425704449e-10,
1.390607385534384107373833093270158793262e-11,
-2.769075645865396099803595372289605617475e-13,
// [2.2857,2.4286]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001344734028349962482971061992179417674178,
-0.0001868082153849835250224822659938192423215,
0.00001298538327699543604828323322428869547651,
-6.021909206159633397798911609108062577342e-7,
2.09590105359808989046061380399328649216e-8,
-5.839537196890559639313377499989708350406e-10,
1.35689946711820095913748849651368442875e-11,
-2.703538471182212820352269722148450707806e-13,
// [2.2857,2.4286]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001307657507694884550118789807896231805948,
-0.0001817953658776781970022589311460461514879,
0.0000126460093329472339062222751466526518711,
-5.868523053238699365408684560887390143894e-7,
2.04383800290661481385137444389493883049e-8,
-5.697987380778336959742396119913520976338e-10,
1.324785067252313092653630247317263832572e-11,
-2.641028239675144650307806663017057733034e-13,
// [2.2857,2.4286]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001272567561143747256740821201661938726979,
-0.0001770441306610091958100903562519609174042,
0.00001232389841181300713169852863504842519234,
-5.722746508971029323601940721629387140233e-7,
1.994295567621080009264298060386688150075e-8,
-5.563129284259332406536730799826878133905e-10,
1.29415392183341775614553784395736260057e-11,
-2.581340201204301272409252316540610766567e-13,
// [2.2857,2.4286]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001239308914627064251039838684182580995571,
-0.0001725345777651355923967702982112577722449,
0.00001201776766885064619959264408073768176725,
-5.584027687942939607070866468797051761083e-7,
1.947095234177888578805365053533685417311e-8,
-5.434499868115360085987458748320067686951e-10,
1.264905699907876521259193261388872305456e-11,
-2.52428764653664642844379241251229491152e-13
}
,{
// [2.4286,2.5714]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.5462919717851479908790205566331839466516,
-0.01326305637603569131858107166501501781343,
0.0004008962964753145269514818734058581344479,
-0.00001111313767239721817144713647383569155207,
2.707582328776486845614380515838951375231e-7,
-5.784747813973872125671767990902316508808e-9,
1.092159447245723992791658493263276638182e-10,
-1.839306179371292319579970597480709148504e-12,
// [2.4286,2.5714]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.09284139463224983857327807918702883675627,
-0.005612548150652877782990891912752867159874,
0.0002333758911204046336735068287047841219355,
-7.581230581598400090145110834089102424405e-6,
2.024661726483889076748243777126009800333e-7,
-4.586483842789708111903257270753669547785e-9,
9.012869302329165261468095177058579935138e-11,
-1.562234032672455743173700634540884748816e-12,
// [2.4286,2.5714]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.03928783705457014390127075149365076649502,
-0.00326726247568434077924022257349426559247,
0.0001592058422136220500411670957017563473363,
-5.669052887118991730951908153888124162969e-6,
1.60526933755662388454828696999667023028e-7,
-3.784896648941528884196819137143467574393e-9,
7.65518419330685156018549378972081545158e-11,
-1.355893154103205753000440388719122634944e-12,
// [2.4286,2.5714]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02287083732979038493622541595798882774568,
-0.002228881790989540090683319397298985817624,
0.0001190501106295485983776366723239489174078,
-4.49475419190331759847215418573379087538e-6,
1.324713820493245715707268798807223751571e-7,
-3.214728608989929079303008647895039256062e-9,
6.644088816772515601776943734576846129249e-11,
-1.196677102011888941006060365202874807837e-12,
// [2.4286,2.5714]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01560217253692678016602772103629849661688,
-0.001666701548812635165817018555926710019611,
0.00009438983803001467031609933045951044339981,
-3.709198739189888901864842356762203994627e-6,
1.125155007146357918252847518127134074979e-7,
-2.790115936366370634962280013918399916775e-9,
5.863909803983929355751233455741027622139e-11,
-1.070315311988992707685127185575880614088e-12,
// [2.4286,2.5714]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01166691084168844573308750168302333622574,
-0.001321457732419260369734420837331913030744,
0.00007789317352302871977784551090961664213414,
-3.150434057810704256603181621770075315673e-6,
9.76540572254501707534688497287775873653e-8,
-2.462479227000758719909263241407846515191e-9,
5.244720188387926874361520015851767561123e-11,
-9.67712293269105074744321071248863810751e-13,
// [2.4286,2.5714]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00925020412693482219507883121298107623561,
-0.001090504429321539968221874966220200235072,
0.00006615911521406252353992013191298167146464,
-2.734313636797242384177729004827418352995e-6,
8.618677244190303249730118782784787988816e-8,
-2.202451424740267146139386238827434285438e-9,
4.741951236867157565776730666488031065302e-11,
-8.828152331547954747287188657519296628617e-13,
// [2.4286,2.5714]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007633531005250779413943892296648949078045,
-0.000926227612996082913261027526757819198192,
0.00005742058637277699672999609987355041440022,
-2.41322966007020740080288016151703997262e-6,
7.708579940048577901060801354925288090682e-8,
-1.991315227296592343061801369904644009508e-9,
4.325943578301537638381797483343353831319e-11,
-8.114491321277354432173483124482022689453e-13,
// [2.4286,2.5714]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006483593290972580054603033928368887578892,
-0.0008038882092181449151334718595284669530567,
0.00005067782286150682510124567704949011921278,
-2.158402412535418168420829643570569590555e-6,
6.969603252245028570587295045752303157284e-8,
-1.816614811831075174871369081996969791094e-9,
3.976239285450569837936125675306632999024e-11,
-7.506458978222062240343403085310943338866e-13,
// [2.4286,2.5714]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005627217464527014089809738204075877156963,
-0.0007094895200604136888160052745493392604778,
0.00004532645066327412965064243984706035576806,
-1.951488937903349427179967821306256940883e-6,
6.358151800944486904609014770281609031561e-8,
-1.669758660852358502546745407946624123304e-9,
3.678294385278205880119869306053726486399e-11,
-6.982403253645787944897630663334045407649e-13,
// [2.4286,2.5714]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004966426640422895524996559804327112571009,
-0.000634570309285200505445125086882968547846,
0.00004098126769599882280364440278247250522464,
-1.78028252975706651514742263615364810587e-6,
5.844155275003359955074935114434675702399e-8,
-1.544638911327878389356534704584422314048e-9,
3.421499130200796925519099443096467243781e-11,
-6.526173633959576492050943191333300646759e-13,
// [2.4286,2.5714]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004441992164996403258579241728368883882373,
-0.0005737377477433853384471265184491913339188,
0.00003738593312492523247436891411329586480371,
-1.6363635009283843223812634916536714654e-6,
5.406236153866587794650263817072489902001e-8,
-1.436799929873564080447964315149146764267e-9,
3.197939580035157781573378532804827013332e-11,
-6.125487388932151316935502872597275211507e-13,
// [2.4286,2.5714]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004016164234203697104903311708980364948377,
-0.0005234030637483897065100442801562310584777,
0.00003436363351952143665363320994772419483317,
-1.513746145624770223388145957668285591841e-6,
5.028799720736190087516502630141726824591e-8,
-1.342918217926070925420433727334497830945e-9,
3.001597048912363029614636550838314649272e-11,
-5.770842794688141811317757643136236387709e-13,
// [2.4286,2.5714]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003663821446238727695072722545233618903563,
-0.0004810908692727674302131643152855778694219,
0.00003178866905814422258582901447652047934949,
-1.408063943113641224929109657476898211139e-6,
4.700213730677287638712186902567620931799e-8,
-1.260466207247229383145214669992851359875e-9,
2.827815574286641153773808551156023978595e-11,
-5.454777781179686948055644075256862806599e-13,
// [2.4286,2.5714]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003367636084909371773374257892320109357863,
-0.0004450413668135141110026893797371996693003,
0.00002956934280540932516177473756710094176963,
-1.316059864790030012659613338972461301805e-6,
4.411631694885955341310700837922574737237e-8,
-1.187488616299723540797784642913161723011e-9,
2.672938646895874301405945586218445066569e-11,
-5.171352812447762540293831209176574812113e-13,
// [2.4286,2.5714]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003115289567694598550120101766769166935885,
-0.0004139707992752505046241058439250691179652,
0.00002763725716061241265874968444338521769198,
-1.235256893770146471295318376139445136262e-6,
4.156210128005749538229714631238104388735e-8,
-1.122449890628678092394498521456976459717e-9,
2.534055816801497269580490381021429296068e-11,
-4.915782934306607001209258757744988021009e-13,
// [2.4286,2.5714]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002897795594926753315683909446091655181168,
-0.0003869216002481163475101812554603103242809,
0.00002594039476919387775165702704554239041747,
-1.163738854138964239495240233715567872074e-6,
3.928574589464478874144581847251386006127e-8,
-1.064127787393073920564591120458624869298e-9,
2.408822392861294511936219621362329009778e-11,
-4.684171249437960488463751987344921062209e-13,
// [2.4286,2.5714]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002708451201736814225223888458420582565466,
-0.000363165526768277450095920109867440069396,
0.00002443851593693815448415481150853848785497,
-1.100000902523750197132725242059247265079e-6,
3.724447229335067387087415092885436930309e-8,
-1.011537656501322487983763537100324055954e-9,
2.295328842618196139091121736599474545882e-11,
-4.473312730209831317140837178586163120272e-13,
// [2.4286,2.5714]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002542158687377941951892861396862087905141,
-0.0003421392231167161487241059036695584343241,
0.00002310001895301783698559243931256158956422,
-1.042845240934533739332509411323978834042e-6,
3.540381772310752551925934950766131843577e-8,
-9.638775940595406937658182491360654035592e-10,
2.192004658380915355914555931916745970341e-11,
-4.280547685460792552256986856882948377081e-13,
// [2.4286,2.5714]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002394974561817012850181176252344298461417,
-0.000323400265341848978513238242065479537195,
0.00002189975005964353383065523306453446061916,
-9.913069122767289930390191465337125535426e-7,
3.37357155477457412373787981251544082361e-8,
-9.204880702844967384962113111831432725231e-10,
2.097546554262344209352971803287264212043e-11,
-4.1036508514385745243352057555451573353e-13,
// [2.4286,2.5714]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002263801857392942665995454831808306833713,
-0.0003065965008346246426903573332765737060205,
0.00002081744515782893428118586665185848951344,
-9.446000507302597791416921935463144646495e-7,
3.221708222495092816819642857588700140856e-8,
-8.808217754461237038034690919894964217203e-10,
2.010864119433377596020257199020944353122e-11,
-3.940746421226924565546881597770310223553e-13,
// [2.4286,2.5714]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002146175505842372321990859957970587804051,
-0.0002914442322092349464049094709572966178578,
0.00001983660106535243225122425412621806040099,
-9.02078317104103706401412407626539207861e-7,
3.082876191425507289687437223487397732101e-8,
-8.444207967026432135030641995883910348574e-10,
1.931038181519812235925971160934927414299e-11,
-3.790242217417879798392214160028815762353e-13,
// [2.4286,2.5714]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002040109625464644454270125826722665154312,
-0.0002777124149145775372707302665365475620818,
0.00001894364465920255209015184628642644955126,
-8.632053478598436690018312495754771653549e-7,
2.955472766626839374088059836271810023388e-8,
-8.108991326593502409556112547543474429748e-10,
1.857288550404031115986881231974874335357e-11,
-3.650778169632115370286027240502641320074e-13,
// [2.4286,2.5714]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001943986904402042596179012502350983541097,
-0.0002652110252284918703055890251686286150576,
0.00001812731230507252988036940507735765223258,
-8.275323884100006886928834353626031905909e-7,
2.838146943223882101177382974678060103014e-8,
-7.799291472919380460522307958868830040467e-10,
1.78894877156613179397121206696029931482e-11,
-3.521185604596534168026263971552454670008e-13,
// [2.4286,2.5714]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00185647717659944293288437974162822205838,
-0.0002537823722706833492675744480263660245648,
0.00001737818015662530300104389423944662882363,
-7.94681157385572605428048719425434499154e-7,
2.729751995136998670591204244656810884521e-8,
-7.512309675666415500077397101365995590303e-10,
1.725446177704537775580459156847214444369e-11,
-3.400454796191522624094597796118414570551e-13,
// [2.4286,2.5714]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001776476605894783290729909776841906743041,
-0.0002432945221924331830941546326832832085954,
0.00001668830430511182253329536585454210480063,
-7.643305714808359238956202553916783694237e-7,
2.629308366752755190007526944058949038141e-8,
-7.245641060988005468590895020826539444137e-10,
1.666285987840184293497214539103651139619e-11,
-3.287708887864717548870519906951156008163e-13,
// [2.4286,2.5714]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001703061655347032132310361472298106866215,
-0.0002336362602712547616368995351347391882863,
0.0000160509420011118919572745135510271550901,
-7.362063551210404958063064212452568052337e-7,
2.535974352228998687752790682011163416207e-8,
-6.997207835684752191718182571025995721172e-10,
1.611038528957591960416436173540331796838e-11,
-3.182182776364530070761173895530659684138e-13,
// [2.4286,2.5714]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001635453821898783186615312340545988482799,
-0.0002247131880152653991022259752362219488248,
0.00001546033345755575541446296988763284473171,
-7.100728306677640966000253863987561235391e-7,
2.449022723949599070438321529062967171424e-8,
-6.765205624554614688221107737481155208063e-10,
1.55932888875380452887500678917753715556e-11,
-3.083205891301073070685003369466463102568e-13,
// [2.4286,2.5714]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00157299231610685765311493924395992093678,
-0.0002164446684054885711041267391729384160047,
0.00001491152944403654376421207636126653853002,
-6.857263743861849522105483700787052783236e-7,
2.367821950597075186093585816141823268968e-8,
-6.548060017239723687327782094491771266873e-10,
1.510828477391545693811701906490344323101e-11,
-2.990188058364521897789854954188605536336e-13,
// [2.4286,2.5714]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001515112678838419861129592114337937185225,
-0.000208761412216228160596094573596167278439,
0.00001440025386212299760067998640440994628216,
-6.629901575053714211801153981978228209761e-7,
2.29182098854904006592952421212408786442e-8,
-6.344391131968754322923389407875339823118e-10,
1.465248100283701330325741168660098978425e-11,
-2.902607821649157106567960151634783981309e-13,
// [2.4286,2.5714]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00146132988551359699135365458272525700761,
-0.0002016035540694465810557692850815757828455,
0.00001392279330762555053954203795535989342723,
-6.417098878092487609830571952320317724656e-7,
2.220536879188081629507016938669864156642e-8,
-6.152984524890299946896912695288307534827e-10,
1.422332235871261125123436936601381710203e-11,
-2.820002740825431585252376422839604599547e-13,
// [2.4286,2.5714]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001411224878486125938148403443053298435484,
-0.0001949191063064899432905099580893431010009,
0.00001347590764400663127912900850515853996425,
-6.217503368833340601702487383170041968581e-7,
2.153544567168485729919543273502750151284e-8,
-5.972767159780110290900545906778855230597e-10,
1.381854281102557609123945231342653800716e-11,
-2.741961284757337262118057819285313108938e-13,
// [2.4286,2.5714]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001364433744145429477181292477124409288268,
-0.0001886627070158322849581438111096193859139,
0.00001305675707456209714874379856925145517715,
-6.029924892293920983000730629429217510709e-7,
2.090468489813805132953328496708484520351e-8,
-5.802787441606808621441005010850704177318e-10,
1.343612579191783510754005729113732408187e-11,
-2.668116023710812291184082542328959924057e-13,
// [2.4286,2.5714]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001320638949110825872071430401174747915115,
-0.0001827945990436156408044529078403962517174,
0.00001266284227382900714456003568884756818496,
-5.853311872967322178168166138121438064547e-7,
2.030975588864890723627031723365840389703e-8,
-5.642198535272134410105495878824030401807e-10,
1.307427083704463888998925711557071803378e-11,
-2.598137884081381878146312683572433165269e-13,
// [2.4286,2.5714]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00127956219330530936605218877425808788715,
-0.0001772797918333588656414385212242734348225,
0.00001229195493324285621977900138412380937883,
-5.6867317477163803438800746036695897439e-7,
1.974769472039002160003733664330893322446e-8,
-5.490244356596574491795800890950043096556e-10,
1.273136543291698452716551597476705394773e-11,
-2.531731277315536766871691798421607660825e-13,
// [2.4286,2.5714]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001240958542833511942819224364992546647108,
-0.000172087369065158914814594611385620420072,
0.00001194213667021559918521908795465051910682,
-5.529354618139022167937441638069349417927e-7,
1.921585509874800771490432390056608261604e-8,
-5.34624774975869344880248684253700152572e-10,
1.240596114791259562067412947554925836363e-11,
-2.468629951862633321688227800073305774682e-13,
// [2.4286,2.5714]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001204611583456112289801541003352351662306,
-0.000167189913382783179157547450929874413984,
0.00001161164469810288107283104526646524018738,
-5.380439521734106428596630403336470446988e-7,
1.871186697836134074192190169864822252844e-8,
-5.209600463649888450454363596070194122695e-10,
1.209675330623604194259119746923243912798e-11,
-2.408593446113874147927183167857879635416e-13,
// [2.4286,2.5714]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001170329393679482142844094068587542193744,
-0.0001625630257732107103786426422722417806387,
0.00001129892299565230439808760102371121926737,
-5.239322845791901687256652035604736712732e-7,
1.8233601480362162350882609898238373698e-8,
-5.079754616079568059213163620149693031417e-10,
1.180256360679909400784046878306172063415e-11,
-2.351404043248787576976520551158712601245e-13,
// [2.4286,2.5714]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001137941180412474863914009538549330840244,
-0.0001581849219389079629906560910632328258529,
0.00001100257797617343230407359208860806596711,
-5.105408504221689084873337860904374134687e-7,
1.77791410170945472505096812845967485863e-8,
-4.956215394688937513420950211975039961788e-10,
1.152232520151820341839790260429856946765e-11,
-2.296864147128958335063131037092558276269e-13,
// [2.4286,2.5714]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001107294453572355634608656196444292177035,
-0.0001540360916662088385496714969473894250804,
0.00001072135785887575442479797932434499092465,
-4.978159572472788323486895948038927950768e-7,
1.734675374531287378168671587930328911333e-8,
-4.838534790686955206864434656834711631811e-10,
1.125506983680305088739287013440041290531e-11,
-2.244794012917301594098950758138320105092e-13,
// [2.4286,2.5714]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001078252641663461765827871115472489313827,
-0.0001500990100240462079337610560011109453189,
0.00001045413510220284143842894560238589883499,
-4.857091134430024506748143885150003847487e-7,
1.693487163425777789982571122653357792571e-8,
-4.726306199014012121862152025967705038536e-10,
1.099991673325081602745040048532779870495e-11,
-2.195029777766288500923029251742600767934e-13
}
,{
// [2.5714,2.7143]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.5334189636487927199034691620799520685703,
-0.01249354844434353469369191778999612260587,
0.0003691251855659887317916264392971468489709,
-0.00001008583100930499761194813859318570159581,
2.434100600342414868434702622039309051292e-7,
-5.1666423902811843992246486407361018861e-9,
9.709026635261071171105587390449949843555e-11,
-1.629402162202379213796768026827204382585e-12,
// [2.5714,2.7143]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.08745483911040474227679353121904419034679,
-0.005167752597922494374085048296383277519219,
0.0002118024511954605388942834586190110258528,
-6.815481734874009682778362467032857238921e-6,
1.80832482918652180072791517551245455026e-7,
-4.077273597816233565508772946879288420305e-9,
7.984307775792146777863965815957249062961e-11,
-1.380242291270020784014116218266703286028e-12,
// [2.5714,2.7143]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.03617426818545746010841293491433343616921,
-0.002965234316735280175792653622995560257254,
0.0001431251164324031810780826605921398727578,
-5.063309568417374667165039375185500067481e-6,
1.427045752705298126823160205104005317279e-7,
-3.352960990408878951999602604112328880504e-9,
6.763396199889865761999200398866046394157e-11,
-1.195405574059430558349893408914981257714e-12,
// [2.5714,2.7143]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02075664021714696077487876508497101742265,
-0.00200375163005261600362747856683531548988,
0.0001063295009368086125209854081821599823382,
-3.995728148716422583333586983397465459512e-6,
1.173536340810489531813394138704027681345e-7,
-2.840231442603693531522538981935867442787e-9,
5.857673957026696023276586833613504533814e-11,
-1.053234279955274836822800366303433630864e-12,
// [2.5714,2.7143]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01402626141036831161388719459499201258725,
-0.001488613013114401941546949701703635498785,
0.0000839102911230843789366170680451794653978,
-3.28590179101502058505879519369536468312e-6,
9.940809996439858520052014629300191555796e-8,
-2.459870301792057957549019625614902273217e-9,
5.161016525928300500784744310495944749568e-11,
-9.406974151920447560723705872356689878225e-13,
// [2.5714,2.7143]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01042029109180081321579745601482950552754,
-0.00117474407572235170758835251986778585824,
0.00006900393761135143545785795045735328695847,
-2.783426832187336890946365459659683889655e-6,
8.609546008267832258397253644078993181256e-8,
-2.167308371032007769654117101140777165792e-9,
4.60957094872934382336655308168902125749e-11,
-8.495229799301858122852977350962211088737e-13,
// [2.5714,2.7143]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.008223208530056461608697856168043240717885,
-0.0009660551265581640306328420638925951836937,
0.00005845196347596713925329872100293787766208,
-2.410672912557878818920624494312572156523e-6,
7.585579254525837513616601796778703650976e-8,
-1.935729465132684851874812869467674677892e-9,
4.162803677746019055748821030076491724794e-11,
-7.742253279804288544759446965640937709424e-13,
// [2.5714,2.7143]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006762385885907147896052091761240675964138,
-0.000818327488662845594902948954995338997859,
0.00005062413116374601962026437055063267757818,
-2.123962219041656998727056593881166928748e-6,
6.775053087211709973512781166349089491736e-8,
-1.748110908682067891203584267444060373498e-9,
3.793834515965140907094112418327605530979e-11,
-7.110321430068384555096979276437095519204e-13,
// [2.5714,2.7143]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005728292420639918868360792449446069510489,
-0.0007087378362918024225202033830141287316908,
0.00004460320659990320926324411430360306443933,
-1.897014890093586502254721482557362469214e-6,
6.118388142504068834392016036994927175079e-8,
-1.593164021935851249942856547794998135552e-9,
3.484178727119985837037113108183980795537e-11,
-6.572687482121333384844905982486953650838e-13,
// [2.5714,2.7143]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004961164854042616681176527849551690139437,
-0.0006244448923980482722695556290785061790936,
0.00003983731269199185731328355144056564472033,
-1.71314870376764339420201305933381562813e-6,
5.576074041387682713724203827182130642418e-8,
-1.463125945622917405441973761571282142584e-9,
3.220730107420766634441914323951882866848e-11,
-6.109885759835237546799533848531085250716e-13,
// [2.5714,2.7143]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004371114246786337646526724264286980572154,
-0.0005577223776873286469356931674090624929768,
0.00003597612277914540998253323887932887087471,
-1.561300753882964556338641582160783114986e-6,
5.120940776481835780177087390882122442206e-8,
-1.35249261749764112191873733928413573255e-9,
2.993950257338913634260187761996982778547e-11,
-5.707426994140030003325179187658297040856e-13,
// [2.5714,2.7143]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003904056643811300284315644885602521666836,
-0.0005036657189075128675647158210393853804771,
0.00003278731583156570228873060607872472257932,
-1.433863438329986258103704016891305892578e-6,
4.733724129979505724682879894006949274745e-8,
-1.257258322207333794577865140589107742601e-9,
2.796739266498598641208023323356513116806e-11,
-5.354312505286916980461137322875632022269e-13,
// [2.5714,2.7143]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003525660032352589842188624282971284936885,
-0.0004590224216414274538922377230109093353233,
0.00003011113220495186491694425477693017610958,
-1.32544277608956255619406432873660421319e-6,
4.400404098187580058913012986152841792963e-8,
-1.174441415920549542945015427156573803095e-9,
2.62370764967030564858593132483155351392e-11,
-5.042048230880177111299606746989555535851e-13,
// [2.5714,2.7143]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003213156951489991958551412405057813931683,
-0.0004215558508688608859448844955444538059819,
0.00002783429829790180843841723884638817973917,
-1.232113166101604362199763759772320317578e-6,
4.110544927728824130730771535756992016436e-8,
-1.101778564610828205774652808498116328833e-9,
2.470693211236997270724379674278793463382e-11,
-4.763973619155195626188194358601540754253e-13,
// [2.5714,2.7143]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002950890956082025993796727685513851196918,
-0.0003896801761701844287268358115418784054268,
0.00002587437648815364218791217263176642945344,
-1.150952597399805568166897831091760916023e-6,
3.856224949537038941515669990444404813089e-8,
-1.037521844651764883929508217889787316423e-9,
2.334432196316870912067710427004604864119e-11,
-4.514794447054539062426132293674007346881e-13,
// [2.5714,2.7143]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002727761233191290803121677693113875940896,
-0.0003622412708337320289056033818202458772923,
0.00002417000454541492178474701810191864892384,
-1.07974300262899111075594107582050050714e-6,
3.631326430941292261918230028064781761122e-8,
-9.803006387314083151276700605354732095371e-10,
2.212330366859939157711172718004210845754e-11,
-4.290251010307325696253931754215993433102e-13,
// [2.5714,2.7143]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00253568889583612401333674719624917742804,
-0.0003383800636354098035379926552868037704025,
0.00002267460305522695766007045380790967339762,
-1.016771416627764420129071485371938156982e-6,
3.431052211367406344886347551285517580797e-8,
-9.290254968105870882189291245621104427573e-10,
2.102300411288032193601367153096989004373e-11,
-4.086878182475889306865748566853904657153e-13,
// [2.5714,2.7143]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002368660445447868443953901119090797362205,
-0.0003174444427727963766729839353392782422642,
0.00002135219974920041087187236322314714456172,
-9.606946344242351347069125258677231986516e-7,
3.25158921569291517503999836903016577963e-8,
-8.828198547846462665686923946308344909745e-10,
2.002644370818061446545124541872042340604e-11,
-3.901829069536608344369429810999636226494e-13,
// [2.5714,2.7143]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002222111099409574463411568139412077669664,
-0.0002989307964884412336059594712180365937954,
0.0000201745873229255746525586600324710073893,
-9.104449949748933648353084538494041462697e-7,
3.089869469563759074127355519328940629318e-8,
-8.409706584713921020417775815681975992339e-10,
1.911967228234067608115237578985504218671e-11,
-3.732743484880928528113889058955076099519e-13,
// [2.5714,2.7143]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002092515575419088468857915301002401088696,
-0.0002824442225206086722533955039350353850482,
0.00001911934489448873359219492582152104717255,
-8.651634654528958342521693264846357619649e-7,
2.943397283352562879152696324573935128872e-8,
-8.028920746235081890687390152576197047444e-10,
1.829112459128346199249394291696178124352e-11,
-3.577648531544365356163032440484498137911e-13,
// [2.5714,2.7143]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001977109557644260545776543845319660724033,
-0.0002676708285225068391471989260276759687192,
0.00001816843277452617233556420142188974648271,
-8.241512527560866111504365638913192147454e-7,
2.810122240702457476997294658961493092843e-8,
-7.680984253025642199832612916608132310435e-10,
1.753113316025671884217781250210054813443e-11,
-3.434882529028592124938383296854190562037e-13,
// [2.5714,2.7143]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001873695799657547719949075300624859222423,
-0.0002543580588430438569421768207922365065851,
0.00001730717630789261072045928136969673991007,
-7.868342402990371900456589315161040382654e-7,
2.688344468836395822034118409923219966253e-8,
-7.361837294186983394472265500542660792183e-10,
1.683155551612923680779652756551296037555e-11,
-3.303036146246012582700071167479405277643e-13,
// [2.5714,2.7143]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001780506411901306850009198091293692994522,
-0.0002423004683101859208467545740316379391225,
0.00001652351904629404532795705419667466138231,
-7.52736463699475190547377585664605222872e-7,
2.57664303394626643679038757362935290285e-8,
-7.068060482130140511646477428597620594119e-10,
1.618548573161091932073467339319418673041e-11,
-3.180906375934593604164080134175968098266e-13,
// [2.5714,2.7143]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001696103278171301302459095776523373996333,
-0.0002313292666478170838746535775981818403414,
0.00001580746573770275202207726030919035003093,
-7.214600614870942231668644322331108170345e-7,
2.473821150381461972955822742114432622165e-8,
-6.796753714772069340445815483784797149882e-10,
1.558702889414505422025163673703539086306e-11,
-3.06746020431377998009119061339091526141e-13,
// [2.5714,2.7143]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001619304866534719448432316232332772350343,
-0.0002213045203275492952254583646645243094433,
0.00001515066129124229302371989331506552874842,
-6.926699336762400718901002358031965206737e-7,
2.378863782417716751755447877160922243056e-8,
-6.545441463319839816066300109899545780745e-10,
1.503112308261320749191657710459700360514e-11,
-2.961805679157014941486198096942629447667e-13,
// [2.5714,2.7143]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001549131642292844932358652021280266436958,
-0.0002121092580771125015882654897462810889015,
0.0000145460686072139266572968107714684050714,
-6.660818702610944638782034138737019377741e-7,
2.290904494981690986712551018818221648049e-8,
-6.311998011197923353782199349407811226228e-10,
1.451339759716145305668302687081097226015e-11,
-2.863168679767485827299528658293714654316e-13,
// [2.5714,2.7143]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001484764806539787381090424430792094889944,
-0.0002036449605007243854299440188953126430842,
0.00001398771927549546644322015492398429119702,
-6.414532694184852346366068240749710579225e-7,
2.209199287275616311541436892637292484952e-8,
-6.094587917623292147168183263106691559976e-10,
1.403005912904276969780279237326102911769e-11,
-2.770874122031341185098477139605481362202e-13,
// [2.5714,2.7143]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00142551472350507057192091548824577591792,
-0.0001958280698566743937856043779668262620663,
0.0000134705186578002945080307949467048481155,
-6.185758109227272501848174463545328022703e-7,
2.133105755028658433737973934322533059377e-8,
-5.891618214682513059018405728269990907884e-10,
1.357779966288502608232147236023886893781e-11,
-2.684330642936213266611365991560461097034e-13,
// [2.5714,2.7143]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00137079648899672063411815227049020953236,
-0.0001885872612089499272273051366575286280174,
0.00001299009202938902090173540798081573136899,
-5.97269621575954873384352883730279637184e-7,
2.062066359473964300386070421186664863366e-8,
-5.70169973099568371989163578622107986956e-10,
1.315372142877401016731761675049659683154e-11,
-2.603018036844164879415792363514474471514e-13,
// [2.5714,2.7143]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001320110828462649371706271532328480772444,
-0.0001818612884111995713195565564629976594648,
0.00001254266205310646535064752223936862132407,
-5.773785905216546587093883507300695364409e-7,
1.995594890631093126943179716378953080013e-8,
-5.523615575480441607083482606673026133033e-10,
1.275527533828224856569685692017627725125e-11,
-2.526476884426447236418766679327755557187e-13,
// [2.5714,2.7143]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001273029018878396883654392831995521035096,
-0.0001755972687432508420000525265224082352845,
0.00001212495040096584381413514803277993950649,
-5.587665789637124017795461839722257439677e-7,
1.933265436623464156264672802643779786047e-8,
-5.356295283713381751759543555503141435599e-10,
1.238021016482271337555929326218600341395e-11,
-2.454299941132128802670069716092140806874e-13,
// [2.5714,2.7143]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001229180881202755781542034405429880313794,
-0.00016974930561328879807629182794724611812,
0.00001173409815824875649674619876932224080323,
-5.413143315752704255122741076940170066059e-7,
1.874703334904890273738432786889856585699e-8,
-5.198793476374918177991942974235491705563e-10,
1.202653034594773712916373822466095979017e-11,
-2.386124947018527861352719160915930183664e-13,
// [2.5714,2.7143]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001188245139293021477035708577506858095912,
-0.0001642773742152558739846598183207914475443,
0.00001136760096309119083373848607042081971487,
-5.249169428421334191617669601026187650454e-7,
1.819577702715311070958105862721783210581e-8,
-5.050272138477865960074547531111053168503e-10,
1.169246075050821034227738783805863783971e-11,
-2.321628591963426574290621146335785520197e-13,
// [2.5714,2.7143]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001149941619506791011202702486501830068327,
-0.0001591464134830559220849168191103729201537,
0.00001102325579969504409083942660268620481632,
-5.094817655903538132669822687015595322179e-7,
1.767595234810823309383664826416858095644e-8,
-4.909985823483069825603846659670305236832e-10,
1.137641710733763375927555136433854871097e-11,
-2.26052142560604205498377633942917519908e-13,
// [2.5714,2.7143]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001114024894381391350572674625878981130448,
-0.0001543255811955155294780442543957538132262,
0.00001069911707740741622108262588508740912081,
-4.949266743506234609452826940837836166377e-7,
1.718495024904173693802156763610664500964e-8,
-4.777269234964172886630423529092068627902e-10,
1.107698106323447913404804825792425969822e-11,
-2.202543544087435027432652541624174561895e-13,
// [2.5714,2.7143]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001080279068368608604862729806809915538985,
-0.0001497876390834941183604630941861878957456,
0.00001039346016137283515738313077102375694427,
-4.811786153615972741063061263353717753588e-7,
1.67204421924744738326585613951461800625e-8,
-4.651526752336888938712325051980898148427e-10,
1.079287904736762946668620251455642286699e-11,
-2.147460918889067490009494691484618032227e-13,
// [2.5714,2.7143]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001048513473584458729457078685703816258713,
-0.0001455084422590151004265031129746769183618,
0.0000101047509226030531605079135878420613137,
-4.681723895730337497958242122772337822832e-7,
1.628034350637329990188032201188823864145e-8,
-4.532223555085938119306006095707341501688e-10,
1.052296428205108896715556763145154630027e-11,
-2.0950622590864894979590889185462752562e-13,
// [2.5714,2.7143]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001018559095813105606224434915177446430853,
-0.0001414665129162430260029259610414818221879,
9.831620181042997861512109664504832154112e-6,
-4.558496261672578124131328565559427572615e-7,
1.586278231894549440539347159872574100888e-8,
-4.418878068300112889457534631077481365016e-10,
1.026620140732689518319893843392871523665e-11,
-2.045156318839076396425115722839109651572e-13,
// [2.5714,2.7143]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0009902655904137010874597737569402064485322,
-0.0001376426825344068988695128934804972654621,
9.57284214952149193898125604587332170383e-6,
-4.441579127333954752825077209056850252348e-7,
1.546607311801161498088165194364641462094e-8,
-4.311055505866796659987269044797472757235e-10,
1.002165328726704999740208815188788309793e-11,
-1.997569578193726962645502519395035824075e-13,
// [2.5714,2.7143]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0009634987777408481996285304237242711781531,
-0.0001340197900931102519362381197717999382904,
9.327316167410181003944901787383408445023e-6,
-4.330500549298059561015402441304000123074e-7,
1.50886941521864206620060574316709333233e-8,
-4.208362329870587102971605358058296367049e-10,
9.788469645566595582704815044727087952918e-12,
-1.952144238244933353153465798318707153828e-13,
// [2.5714,2.7143]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0009381385306517716731068868364081890636957,
-0.0001305824263435561378018852406686965750906,
9.094051153534608017253862072350140071311e-6,
-4.224834437171407283582433253704119435772e-7,
1.472926803877414730620509719226919621255e-8,
-4.110441478194527673083511927581683734981e-10,
9.565877241527676248033475837986482671357e-12,
-1.908736482089552940572761592413535295977e-13
}
,{
// [2.7143,2.8571]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.5212846928978983376043360292777981373183,
-0.01178460718794672562287171384216570844854,
0.000340277909118959074811728470659896921807,
-9.161971297774777570030332482334148293865e-6,
2.189776774857387394945983181312074026366e-7,
-4.61705016421032159700696838311253794671e-9,
8.634666517583309535808669914598268627887e-11,
-1.443926844943538684586286461352718912236e-12,
// [2.7143,2.8571]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.08249225031562707884947922484583969283129,
-0.004763890727664235880450662459253309257291,
0.0001924013972533193489796891002404688070164,
-6.131375017247720531573965926135576986698e-6,
1.615967550937593517039630439339672402946e-7,
-3.62610252354163088346992557920351918643e-9,
7.07545069321316140981520254547143276316e-11,
-1.219774628843029009099194489167626666184e-12,
// [2.7143,2.8571]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.03334723509364965071404529733612365956611,
-0.002693619561545441466565894760015552045316,
0.0001287588753622452458607690393265304767172,
-4.524709183802348273497064265741510783246e-6,
1.269135877490928359999370015710384205611e-7,
-2.971293989069040267207555675501685797222e-9,
5.977079638232410105243969271663520172346e-11,
-1.054142787599024196472384580626418353053e-12,
// [2.7143,2.8571]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0188553369308180898654130303205546145959,
-0.001802624255070528034337122219633160692561,
0.00009501889285988776655688499133132422045441,
-3.553580493191196986357983980143901761894e-6,
1.039952891047107654229539015925344359687e-7,
-2.510025766871040770936525201713919920377e-9,
5.165463725353733839196487055165803392271e-11,
-9.271533543346322058185571888174477572972e-13,
// [2.7143,2.8571]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0126183697854936958790657892363665652353,
-0.001330264500037621223568357860481992836873,
0.00007462519035704982111187259214656221690142,
-2.911868127232493700950783143132986423772e-6,
8.785090137802643831971078987655748876526e-8,
-2.169184677278628827786244760931078350858e-9,
4.543199423722776320825939887745127970882e-11,
-8.269028610990419709766315114255705604491e-13,
// [2.7143,2.8571]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.009311851500263348236040642014518133645536,
-0.001044752664997969123984697911883116211438,
0.00006114923067191394590998674193144907904004,
-2.459825267719844993888896542109927004337e-6,
7.592146328370823398722589720944416864513e-8,
-1.907864059412890615213780651199007062791e-9,
4.051958753658130015433095473100327902035e-11,
-7.458657453439070986320570159091604020844e-13,
// [2.7143,2.8571]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007313268654985783566064772885042171383451,
-0.0008560892294061321014735044694647181353741,
0.00005165633062214572051461767304422160653906,
-2.125800998469704239499346745972267764193e-6,
6.677524169310811230384178939720557408545e-8,
-1.701568026721575688397356922460248565648e-9,
3.654865782210118631068684925285126077286e-11,
-6.790688918382910662180158376755801240647e-13,
// [2.7143,2.8571]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005992624605842924431517535662231281292952,
-0.0007231886287094315993753683940972212599215,
0.00004464182096789055367564441971134953402033,
-1.869706791746747510962141963286827819839e-6,
5.95548805783921236842456400826248638807e-8,
-1.534809965886673971307769003803778186921e-9,
3.327551766405440261128927475447407873491e-11,
-6.231029033693185798686799916939923356925e-13,
// [2.7143,2.8571]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005062320400966020936630958228454919177646,
-0.0006249854935499056945177176195168663302317,
0.00003926384262670656153196337105275485341628,
-1.667536678677450551198610038785576366848e-6,
5.371834847451522270918280328464090826908e-8,
-1.397355908929842622411117513022910979983e-9,
3.053310312602235338582746290338387280955e-11,
-5.755569192777766672904321505968382871569e-13,
// [2.7143,2.8571]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004374898454849339619826829845452935905666,
-0.0005496937967733697221994216855757097779719,
0.00003501827025224967422452000183513033364211,
-1.504113778172177533935095937121925777706e-6,
4.890745650304155026180529052213777242789e-8,
-1.282189826919667242535533234185050090467e-9,
2.820327945605103061599300042311606929961e-11,
-5.346805397991696668207168129716832590565e-13,
// [2.7143,2.8571]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003847856577413587828674404020472098194648,
-0.0004902557835310079740612123473475208485012,
0.00003158638934163749359340133214620537280923,
-1.369408801583937409793963261324044216535e-6,
4.487664365198237920323320900995624966434e-8,
-1.18435054783110644953960433860869649783e-9,
2.620027511119154808151477653591865497408e-11,
-4.991736092739178004268055932233573265381e-13,
// [2.7143,2.8571]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003431790484717055605026682512657989977188,
-0.0004422094507824678378350504976779810491293,
0.00002875758483328317228512028348794306045314,
-1.256546040538566758762751529552089003291e-6,
4.14522689009321495993447393072785200644e-8,
-1.100236036260758901265731985365395573444e-9,
2.446038095727931433021922829287699931956e-11,
-4.680510570214019920420360955955712300645e-13,
// [2.7143,2.8571]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00309546615547727466329541779921388338507,
-0.0004026061876655341922192142320159597926142,
0.00002638746685132925082879511555411043652108,
-1.160663546435043835902240272515847511882e-6,
3.85082610111404902269058170352476590331e-8,
-1.027170793369123421178120891116384806288e-9,
2.293532735119984223367194890401409172504e-11,
-4.4055342638091121929672405480910009802e-13,
// [2.7143,2.8571]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002818243313658739154597178237559448530533,
-0.0003694245359182031852839034974859378718089,
0.00002437393447515425063293694697512631741341,
-1.078231324565131598219489356475302558911e-6,
3.595097752351749744183483593844368819893e-8,
-9.631277171221600886716168938838389628901e-10,
2.158789998013676282123966709557671651768e-11,
-4.160861108446954201504249415739923723378e-13,
// [2.7143,2.8571]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002585971751427422115607077222777979335851,
-0.0003412350826517745776509565331907162224506,
0.00002264285781588517615810438964241732302966,
-1.006627386055876516961190023607120002179e-6,
3.370946986710692909703846774774613722543e-8,
-9.065439833701221749067887045834061664512e-10,
2.03889623727129222734591797555714073323e-11,
-3.941771395149592646617744206991570607373e-13,
// [2.7143,2.8571]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002388645578562421870827253414960427033468,
-0.0003170000094220268021362582664532298322153,
0.00002113917510718998897040679568585771806988,
-9.438651709056889914289566519479909480375e-7,
3.172903919685871775043736700406120476377e-8,
-8.561960025380740224437197968658682915809e-10,
1.931538734351757025753901850135109428625e-11,
-3.744472501174872719968916241864313906275e-13,
// [2.7143,2.8571]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002219000065954187450091408914530210813923,
-0.0002959484515003116215965705377807951653992,
0.00001982116858903529569440042477614299299758,
-8.884131114411298028379124688977146695065e-7,
2.996685987780692613750153134385786211756e-8,
-8.111125483976555131703202473544652708377e-10,
1.834859053931895652123105791230077842395e-11,
-3.565882846190457290320032618383825883683e-13,
// [2.7143,2.8571]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002071639160502181193498340176953107425202,
-0.0002774963602461617757517913821874260074261,
0.00001865667534027886299506027148443399913715,
-8.390720898732735955762170909295112133785e-7,
2.838893899208882434306787239497569701985e-8,
-7.705131729530441563290244350781506066083e-10,
1.7473471800890451716741183697993384634e-11,
-3.403473357638615181111705081869974234203e-13,
// [2.7143,2.8571]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001942474521723132279172070383888333312553,
-0.0002611934547635862024627799673778398634999,
0.00001762051388735325026975814226151104261595,
-7.948903044937818266021439759682867410529e-7,
2.696796085995908918698167262174193755735e-8,
-7.337637480652986293058093045345070714933e-10,
1.667763832561381067290410228984725534395e-11,
-3.255149399842006576796284108756059425174e-13,
// [2.7143,2.8571]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001828354183345103272209923543189445542808,
-0.0002466871944226409041229550759302341529249,
0.00001669269639438334126884503289163126925164,
-7.551029162629521899856281748989655083959e-7,
2.568173099664605157723511807618433026953e-8,
-7.003438416240281541801814632313075412901e-10,
1.595082610658254029248615782485827592716e-11,
-3.119161644893123920629898206445703409909e-13,
// [2.7143,2.8571]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001726810360958486189426264174098476724206,
-0.0002336977495210743969983244461843741904847,
0.00001585716124153538176619300525746893870801,
-7.19088479601427466160426808317084912809e-7,
2.451203427836338910455742023905840522389e-8,
-6.698224205440798504783746404994556737048e-10,
1.52844631895140159416296900309502296201e-11,
-2.9940379565693454571579820606430550418e-13,
// [2.7143,2.8571]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001635884246647520644734694528608859364893,
-0.0002220002573812142437565101137480212213046,
0.00001510085807164286520227368778134825164633,
-6.863369710383172557534362259413006066737e-7,
2.344378454719673295260189957380410939366e-8,
-6.418395095301803297653731284658204921824e-10,
1.467133589577462769708335565349190187757e-11,
-2.878530741307590354118288560533137381304e-13,
// [2.7143,2.8571]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001554001801668499576852802661033046818803,
-0.0002114120130027294564853984329297486533915,
0.00001441307639181708894378819164300449463117,
-6.564259781478623500296110399064441605785e-7,
2.246438266786813418971889477970267164618e-8,
-6.160921739859794232607008273546747049682e-10,
1.410533083572204155427524401015927275035e-11,
-2.771575828413509008710641700257880259042e-13,
// [2.7143,2.8571]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001479884091019106070433962223336784920051,
-0.0002017830694851782874944569267585669715491,
0.00001378494554111710594304401018330579666798,
-6.29002725138713126338708748696214213068e-7,
2.156322592955419379112080326763210355307e-8,
-5.923236857903732250696050818256856724518e-10,
1.358123341661164627015399136044904289786e-11,
-2.672260044681784277218903901964375971237e-13,
// [2.7143,2.8571]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001412481486396247891677552895059081508795,
-0.0001929892375753875550653031799417028945957,
0.00001320905722792457094609822961052916016232,
-6.037703361047364163955328411313755593702e-7,
2.073132884805865233393028837927924227787e-8,
-5.703150615969934765157148645749762268439e-10,
1.309456895412984085857872490908378395655e-11,
-2.579795416487312001056389008060330692078e-13,
// [2.7143,2.8571]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001350924663027712768583753047243001045128,
-0.0001849268011907004923867150291930720812454,
0.00001267917705821068464884266339792189853129,
-5.804772174857671770037752452799093414869e-7,
1.996102700629555578417560667843467669372e-8,
-5.498783902946800995280421268025949038307e-10,
1.264147625932134090251486620022809385957e-11,
-2.49349847440195277121138572193981609993e-13,
// [2.7143,2.8571]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001294487608334903333499270941823562755663,
-0.0001775084788147139673736204264003480704594,
0.00001219002156721197871885705486351945876412,
-5.589087656010717797398978347173693033466e-7,
1.924574351540663880050704017183740180725e-8,
-5.308515242858134102789064269593893932455e-10,
1.22186062285084254129627310695425935873e-11,
-2.41277352286581977554678496178881391652e-13,
// [2.7143,2.8571]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001242559351702997661850814682397937023056,
-0.0001706603019407394743033383816247365235521,
0.0000117370840776330448268431570597353120735,
-5.388808275605815312559629246570637456416e-7,
1.857980320950364457892941398288725120564e-8,
-5.130938207737619035816754719723838297405e-10,
1.182303986246466806720881303284655370316e-11,
-2.337099017759328762477917201717398985005e-13,
// [2.7143,2.8571]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001194622113585176213599225011526978175713,
-0.0001643191770866413413461311864773649782302,
0.00001131649737878243853014915614057003153128,
-5.202344987176340096317482445159091160518e-7,
1.795828359072956844165933246887577917341e-8,
-4.964826989869261443747680175983162782419e-10,
1.145222151469034729994508985443029503696e-11,
-2.26601639977563700557705664451800085357e-13,
// [2.7143,2.8571]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001150234239606489285953711125528108254236,
-0.0001584309633027393858340808969119117706237,
0.00001092492447308024730061983036646702725482,
-5.028319491306519032333680399177600565582e-7,
1.73768943321006679636247640493176031779e-8,
-4.809108369517312154136973887071385494573e-10,
1.11039041734208361778310880314790789376e-11,
-2.199120883051210575550580799452857503832e-13,
// [2.7143,2.8571]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001109016743119175600254295000737698634269,
-0.0001529489426229148673974202500034531991828,
0.00001055947093175334611096858711663900993978,
-4.86553049642689436900315845674968554604e-7,
1.683187916456159409607433597509464287207e-8,
-4.66283873623611026316869859626987403193e-10,
1.077610432465092591619511624536522031349e-11,
-2.136053811622302812587114986352104581429e-13,
// [2.7143,2.8571]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001070642598360403973926386959465626518148,
-0.0001478325930443440667965797434473201363934,
0.0000102176140425058723598270018927291433889,
-4.712926247189511296893080295046335893626e-7,
1.631993545157017369546048368098519339337e-8,
-4.52518513373202705040426370818179743709e-10,
1.046706449771008933939233455138649999991e-11,
-2.076496281462215677619232809909645630545e-13,
// [2.7143,2.8571]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00103482815131040837230533368279839236897,
-0.0001430465965948849353953262448693507979338,
9.897145119107119763393639266954037445099e-6,
-4.569582005351448077324682771144189279629e-7,
1.583814784611449285100599898553725981224e-8,
-4.395409531011289357228784241309432988372e-10,
1.017522201235025697621148506529350291271e-11,
-2.020163790562655820997863073395418067892e-13,
// [2.7143,2.8571]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00100132617616419445494857349392691601943,
-0.0001385600316673076100842558391653374700185,
9.596122211246951607694109253064740545143e-6,
-4.43468147373942320909113534748742146278e-7,
1.538393323973051553410253319548995261709e-8,
-4.272855697848037612940251387121566220414e-10,
9.899182763386621043154130802353352186803e-12,
-1.966801729072875463633733431496917639744e-13,
// [2.7143,2.8571]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0009699202216711531801002436842828203948272,
-0.0001343457109572701991854324425935546271874,
9.312831094861475783578610936551361291129e-6,
-4.307501381974579450083490495454779379e-7,
1.49549948266404881562850469007841595291e-8,
-4.156938195765792366107721768406353038489e-10,
9.637699121734731347008437796646312528436e-12,
-1.916181559742937001256282684394311459792e-13,
// [2.7143,2.8571]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0009404199767008913060237852721371421624164,
-0.000130379635327878233259662298487202192009,
9.045752902155091219228217760797179159235e-6,
-4.187398624431110169046966593351551978367e-7,
1.45492835721882445389017386192900673904e-8,
-4.047133097691515443374017869697945793416e-10,
9.389651218031739170483817793601472201879e-12,
-1.868097568630119966181184149253318011282e-13,
// [2.7143,2.8571]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0009126574472951475466529797994365682977329,
-0.0001266405406299933154326541351551882629726,
8.793537111313603207771646343112811608734e-6,
-4.073799471398036307428634778925283829167e-7,
1.4164965731628567588309118847071073444e-8,
-3.942970128117462200169014578806424327001e-10,
9.154031020625408535888293269281869709763e-12,
-1.822364089276387630113325050023947926289e-13,
// [2.7143,2.8571]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0008864837844099531238751697298451935983434,
-0.000123109519558216736214153637129007841222,
8.554978889943955017633254777164120133793e-6,
-3.966190474340134647862535902415665212678e-7,
1.380039534069379926895426098810979673372e-8,
-3.84402597674937611151215333139560990556e-10,
8.929928733636995977814576744051472806195e-12,
-1.778813121873726210482203200760624866175e-13,
// [2.7143,2.8571]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0008617666369075170712652855450399566464044,
-0.0001197697044590457162429902540217627222138,
8.328999996122177248115185755612344297888e-6,
-3.864110763256508536104525551296913500718e-7,
1.345409081336263149283814172112850584041e-8,
-3.74991858646157915600424727685680353262e-10,
8.716521130519982744914984777966248065247e-12,
-1.737292283441211258104145265257706137687e-13,
// [2.7143,2.8571]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0008383879312133199333012477992504382349685,
-0.0001166059999455446974381956882120692301328,
8.114632602846386338318827129959869081591e-6,
-3.76714549405578014217069865047031442136e-7,
1.312471494970301768659165166385705529084e-8,
-3.660302254055846602606272458848928985741e-10,
8.513061509622985627970765136406426447344e-12,
-1.697663036599330880568074104831529755984e-13,
// [2.7143,2.8571]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0008162419996188128034218612490826302369935,
-0.0001136048564396873222738801898328015633184,
7.911005537524688309322736562339865434024e-6,
-3.674920250752045708328013271204863784795e-7,
1.281105778852831738231491665227450356015e-8,
-3.574863412173744040284341363617892842635e-10,
8.318871014918861521558740913000639698663e-12,
-1.65979915379252643717756634918580185527e-13
}
,{
// [2.8571,3.]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.5098314160933151939561808383037389190294,
-0.01113068395005669678172560259490471379356,
0.0003140609561154062329816565736337063933434,
-8.330553704623691978888133443991637422004e-6,
1.97138418841737556646239213560261908426e-7,
-4.128171918707473263624359103191081643211e-9,
7.682445387610835088316717203324531065229e-11,
-1.279993361783812004962972590131417788379e-12,
// [2.8571,3.]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.07791478765039687702169769036063739344408,
-0.00439685338561463429591088205386102917976,
0.0001749416277971407683793177133399598370708,
-5.519875769687601211776412560629615680939e-6,
1.444860165782690273409840249828787033694e-7,
-3.226222718855599972707932560322180823503e-9,
6.272151950686988874539199034852700558676e-11,
-1.078254365489555701881445001728392850065e-12,
// [2.8571,3.]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.03077797369930243967595606759328329601187,
-0.002449182789159062784949078625374210188435,
0.0001159173911634775859541304310295946049579,
-4.045608500510708937257221464155225513128e-6,
1.129177946538044954276244378545133466749e-7,
-2.633955153394288332580539774656266491846e-9,
5.28360835648133771322723277093612812166e-11,
-9.297791522866523476992550414124142303668e-13,
// [2.8571,3.]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01714427952411343914249419491184492732518,
-0.001622843476287889033538480659377600589578,
0.00008495777851075869418112208659824969386062,
-3.161698282193572444834816552192326524475e-6,
9.218842991804549059219255365638560534115e-8,
-2.218809392450739126302211480752560327298e-9,
4.556062087951018429286513893876028031775e-11,
-8.163158177536225123614307456059812160686e-13,
// [2.8571,3.]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01135990433401522291751763544093625462789,
-0.001189408899149911782812009475952911001761,
0.00006639566392609547765476226468023531285211,
-2.581276066102933516546612063801164200641e-6,
7.76583283296905151537523629285326808319e-8,
-1.913273457928206801219506334123298859114e-9,
4.000077454557391751127472084556300980136e-11,
-7.269868439005039117042440114319334231843e-13,
// [2.8571,3.]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00832586229404938219114140032715018216576,
-0.0009295392949646971052758771454153212285266,
0.00005420679738818930407930766805454817143715,
-2.174433218814821677391035373817629130183e-6,
6.696457065814972651088535627065061304605e-8,
-1.679786928082090557372836167711792432648e-9,
3.562353723348085076628439531641580486651e-11,
-6.549434651594822005256248681935021110759e-13,
// [2.8571,3.]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006506775064752879472402007418199738125819,
-0.0007588951634340685528760517160017957018365,
0.00004566309759513665014809868660788381692697,
-1.87500800169655703725091832189458984636e-6,
5.879254214427318841514038300079476038622e-8,
-1.495965186255038563455507858549072527039e-9,
3.209331331490723660268666837217703014563e-11,
-5.956758508765835340844123862961925366974e-13,
// [2.8571,3.]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005312266144038479625974125639727755479182,
-0.0006392833663313798092814761648369105549414,
0.00003937516803565113709067173924463116357321,
-1.646191201371542283979453658039588048488e-6,
5.235878120640129599495225394359696891278e-8,
-1.34771437189269103946444864147009690996e-9,
2.918911677513347001564585615924901154387e-11,
-5.461017663141796069411633101058954899435e-13,
// [2.8571,3.]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004474983564319658438302972335235518502418,
-0.0005512523524986236943422562756351477609951,
0.00003457001522882414814052161940826345534739,
-1.466045893468402565143083588015530795336e-6,
4.717000272610761115019510269635131070048e-8,
-1.22575388747651250726448741541895688949e-9,
2.675991498831826380329432021713416881013e-11,
-5.040476067919685021225669326409154932878e-13,
// [2.8571,3.]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003858766467490365648904532836703164048332,
-0.000483980213203081110766076287723983526395,
0.0000307869637628567571328552969022906312784,
-1.320760094609699721700441340887378878649e-6,
4.290138579096690945518686989777560399289e-8,
-1.123740953103122893378888134524914959936e-9,
2.469919900986831010068693478410398812047e-11,
-4.679390162640535172091168486674900268946e-13,
// [2.8571,3.]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003387861492421567577160040406622824997013,
-0.0004310174926795682317824339610543871202847,
0.00002773596198682272169307656309615358870589,
-1.201238819201945742543716554348333538899e-6,
3.933093310490801918341637177666334705257e-8,
-1.037202630685868647314915530006215104852e-9,
2.292982364271352306766926983096316424275e-11,
-4.366090948144264920918423604290895908036e-13,
// [2.8571,3.]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003017122448756977436004958371521918482038,
-0.0003883034678151185258755920570910525993424,
0.00002522601520325876200582328673447139667442,
-1.101266142920678469208621003807339765605e-6,
3.630209183531916161679106685401925036643e-8,
-9.628991528548840425014972659528428455697e-10,
2.139460944346180926876827193175513470446e-11,
-4.091754210607228079671177338896734248178e-13,
// [2.8571,3.]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002718124274705829505086296222462975218973,
-0.0003531642128452467388540116819336544355898,
0.00002312658900135114805491892597241528938084,
-1.016458586426238993278204579408911898703e-6,
3.370147012458421607359407075544994534616e-8,
-8.984292376681462035028261613273664475929e-10,
2.005031671100055395855436383121001851063e-11,
-3.8495884516153176222997900150116169701e-13,
// [2.8571,3.]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002472149489916727005267123933242512043414,
-0.0003237722460185611689477526226236332498088,
0.0000213456303149670231946794488563740916303,
-9.436411776846375435372013001770944477161e-7,
3.144502310499330154579102853089559749133e-8,
-8.419770167683273184942269897364575325292e-10,
1.886366626815654847022238394262644106572e-11,
-3.634284604494262824952573022157646406991e-13,
// [2.8571,3.]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002266405722129928024321451563214063095841,
-0.0002988388244092022340643832713412005690892,
0.00001981646473139258652473488367646405387874,
-8.804606603835594597556431044963918341122e-7,
2.94691953842493430959762843419048503961e-8,
-7.921449225195556278304424641247465844511e-10,
1.780864301814970768266831538227872664177e-11,
-3.441634512216171953306028567372473545404e-13,
// [2.8571,3.]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002091871770864415487735023133283514025211,
-0.0002774305062391770514002784310128086951862,
0.00001848967386806921743473931535198519037401,
-8.251374835254944944354813442607568267212e-7,
2.772507209526676975870360374633532792199e-8,
-7.478404475022576419676511950026499514591e-10,
1.686462644773251580036144250382756988879e-11,
-3.268260959097230366938956382489396075554e-13,
// [2.8571,3.]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001942013543674239215991306113366077088567,
-0.0002588554341526652000662096848586842429182,
0.00001732788715404919027818235457465494443141,
-7.763020308213403639230418305328591971728e-7,
2.617441547849983041942335207763640093054e-8,
-7.081976329409463536466752379502994353656e-10,
1.601506775422630495995512893928776275557e-11,
-3.111423112123861252219528688167217441742e-13,
// [2.8571,3.]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001811988039068656262955759040765223222172,
-0.0002425904201563987404409151367458760850138,
0.00001630234264726134845278602076250301412973,
-7.32883644995038781555494111610966088516e-7,
2.478691697692175411635812333395438509969e-8,
-6.72521513385154811677680437062009153442e-10,
1.524653648698621327110783549902412455888e-11,
-2.968873976311937492707619496975888581096e-13,
// [2.8571,3.]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001698132941094791051354690825092811845021,
-0.0002282327970613816616489062311126750596472,
0.00001539055654490846072495157214907238253772,
-6.940336864425778537312135570923064288935e-7,
2.3538252799862380186787972647377550168e-8,
-6.402480796262663558120566746356531329647e-10,
1.454802206280509708600187864954287560364e-11,
-2.838754386365362193891436882411022586488e-13,
// [2.8571,3.]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001597629579429671505122863273305530901711,
-0.0002154677916284528779069269689631296385163,
0.00001457470741530627126357670470141011660047,
-6.59071089019133554037031094821480612369e-7,
2.240868262510099863422204107966639291971e-8,
-6.109149453413223116452056628068629951109e-10,
1.391041431293768936625465432519279207286e-11,
-2.719513090630141903880970767736887877134e-13,
// [2.8571,3.]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001508274541399170023830986800829420164187,
-0.0002040459038140329349542585719826254773275,
0.00001384049286941347037773189663109060912646,
-6.274431236974309775500935694243544451812e-7,
2.138202293140253940157163118719472792674e-8,
-5.841395323554951520968556001409143668542e-10,
1.332611188513170010504194058515950420714e-11,
-2.609845752027218643328985212786882590373e-13,
// [2.8571,3.]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001428321326698230427699632810909736485209,
-0.000193766900171543604979098563500668241232,
0.00001317630559765728068614472375990519373611,
-5.986966518785736829548013693305663429757e-7,
2.044488348270641109128539680669007000814e-8,
-5.596026252933941997086590765594389957493e-10,
1.278872334090886405072130058733684214322e-11,
-2.508647854596152540633886103548142875757e-13,
// [2.8571,3.]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001356368301200805122085289266402242893675,
-0.0001844682783669660965792880846146351054949,
0.00001257262968946087316706256758700058799073,
-5.724567469491876579390463790354817570711e-7,
1.958609174092399512283697771205207558714e-8,
-5.37035819042655505162609969600237180905e-10,
1.229283639188633050548995276613812245109e-11,
-2.414977962429798716018130119991589420232e-13,
// [2.8571,3.]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001291277948568762567206566294604031016292,
-0.0001760168156522248822868401113539642241607,
0.0000120215916859433903245915821424757955082,
-5.484105778396380968771543385141817394284e-7,
1.879625352716571791115056350163073859453e-8,
-5.162118277648085683465838710353810901564e-10,
1.183383786399199630514362785315099862219e-11,
-2.328028776431213272310441641201764668818e-13,
// [2.8571,3.]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001232117709565574070816786891936108137109,
-0.0001683022836029880251433057498053868036638,
0.00001151662213464249843429244164618681274635,
-5.262951075382975564182735332071347460346e-7,
1.806741383712252938706690991687860329164e-8,
-4.969369242550319296060142607198053845617e-10,
1.140777187191483344752444412582435424511e-11,
-2.247104128600464830652896217091935915806e-13,
// [2.8571,3.]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001178115985220916074232272445259413720339,
-0.0001612327098847829119387800584502661957186,
0.00001105219725831401874127651321416465619991,
-5.058875959221553269756439638129179783531e-7,
1.73927922186582721598232751561337410296e-8,
-4.790449839605299397121003176652479142963e-10,
1.10112270881551613023102089323109987459e-11,
-2.171600542963802084787613257015992820595e-13,
// [2.8571,3.]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0011286289691934802850059704610182654803,
-0.0001547307616161910914397070027189262380309,
0.00001062363951437472420513090175503544378002,
-4.869981903293455905499329948594076257286e-7,
1.676657431245361979580768290757442745629e-8,
-4.623927508391624249395573294330925814298e-10,
1.064124638917330828787618202269853605474e-11,
-2.100992341707237379093808186282484625656e-13,
// [2.8571,3.]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001083115331313337544522570045736580187894,
-0.0001487309532010474300011708151924196978624,
0.00001022696199692543076805164853831672953524,
-4.694640886971303585880242040835803667938e-7,
1.618374615705873052241093712343703426847e-8,
-4.468560429520963914898613354064841895041e-10,
1.029525387347480466335710375306486385872e-11,
-2.034819527593521547002182307341482558476e-13,
// [2.8571,3.]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001041116672407331917284937761006917187272,
-0.0001431774679567633902900537730192641702119,
9.858745862648639011830121131961567537621e-6,
-4.53144900103335003708051586196455685698e-7,
1.563996138463656419307250197405171313854e-8,
-4.32326687595596840571421691822299384352e-10,
9.970995483830163890093512899032616662591e-12,
-1.97267785825761697350865874965914219195e-13,
// [2.8571,3.]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001002242275697343641976550722677898384576,
-0.0001380224420768940152818566390911369173365,
9.516042902178680292527505859719508145196e-6,
-4.379189262471291740621499709701937001628e-7,
1.513143395057598149912051154870944577893e-8,
-4.187100277410813107298768151129458886969e-10,
9.66649036997650954680859708501085729668e-12,
-1.914210664213965509323691347393087612843e-13,
// [2.8571,3.]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0009661570945382580194395137564935790910348,
-0.0001332246006303199748087167019590516577842,
9.196297451198115915781987532904563383291e-6,
-4.236801578781670680783528603524934574776e-7,
1.465485085889400375991465046109039343697e-8,
-4.059228795230200049989048011937042543423e-10,
9.379990795730598806143060904172495979963e-12,
-1.859102063986169609504641633797483300206e-13,
// [2.8571,3.]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0009325722044122397385109154317031301142212,
-0.0001287481643165971545662299493320910007176,
8.897283315449682884332109975514272931125e-6,
-4.103358311078286075634498907194233909258e-7,
1.420730067431261146044148561836743887498e-8,
-3.938918485498535709537071373551470701559e-10,
9.109948892183179043261085990064794629022e-12,
-1.807071306200273820484103899066413984711e-13,
// [2.8571,3.]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0009012371502161799990707768433847990255319,
-0.0001245619664161238970419640649979546901135,
8.617052453272358514594193239287799265186e-6,
-3.978044257473512691865937191169370914983e-7,
1.378621459314111432166661827157258675659e-8,
-3.825519337160215801854818390702309018103e-10,
8.854988933163206662212935222843957869988e-12,
-1.757868026488734404729099667660230152731e-13,
// [2.8571,3.]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0008719337649128671985418064417634774736247,
-0.0001206387343456459065377762873241646799031,
8.353892940702128882088328127791167757299e-6,
-3.860140152925208369770086012619955846119e-7,
1.338931757669735781267207873203824865436e-8,
-3.718453629213113649615869640523412964554e-10,
8.613884093396159428115887037766677556356e-12,
-1.711268251432940615370796389503414200601e-13,
// [2.8571,3.]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0008444711404195212670457643579250320639382,
-0.0001169545011696670077361017754531149251105,
8.106294321150494610515843807351392090872e-6,
-3.749008986594519715909579464519956143294e-7,
1.301458760148511075971650270634353511964e-8,
-3.617206170403532448805844594719408243696e-10,
8.385536867251746689855968161803449211847e-12,
-1.667071015988714754199205666879803803019e-13,
// [2.8571,3.]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0008186815081876689773675999246294774026838,
-0.0001134881204959482270279005540440945502643,
7.872918871855862814945662621305306592189e-6,
-3.644084591895438287879162823628683348359e-7,
1.266022149812655054573696816802362571076e-8,
-3.521316076178768224995829645392464662028e-10,
8.168962493641605789619014235382906650217e-12,
-1.625095487410599136261980103402562962612e-13,
// [2.8571,3.]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0007944168434716375142511822721107148763364,
-0.0001102208642058272799570854933373374185641,
7.65257764298761508002353902477767773536e-6,
-3.544862081395800521859030385818054806573e-7,
1.232460617069636399594120054490196548762e-8,
-3.430369808065058567771172134371673219588e-10,
7.963274862830583836701279127141159834744e-12,
-1.585178509466650487131079423482895139461e-13,
// [2.8571,3.]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0007715460494407908865103839530832242985589,
-0.0001071360870016755231210492762009824978166,
7.444210370938207299408634870501593965868e-6,
-3.45088978823075374092543333030704765131e-7,
1.200629423454468028772313696309101590348e-8,
-3.343995255317825546887766165289992402482e-10,
7.767674482729743502715629763989305605071e-12,
-1.547172497085524182791188856303591218802e-13,
// [2.8571,3.]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0007499526090117285903328455158968984285474,
-0.0001042189451929873520997103990244883204372,
7.246868555291448285948596818583367557538e-6,
-3.361762444693106595722682360058498446736e-7,
1.17039833020730181867857170893572361278e-8,
-3.261856681444172911918494912574026562978e-10,
7.581438162356580611817206095901399806893e-12,
-1.510943624524025139460069119666221893023e-13,
// [2.8571,3.]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000729532616350911394783363211163362735923,
-0.0001014561597739361021507852882736669052343,
7.059701133862235690783498390297746341072e-6,
-3.277115382250533161707348825780894093659e-7,
1.141649829556311034297189079676367723233e-8,
-3.183650391843017559033494796678493105237e-10,
7.403910133586912408337674353022262017104e-12,
-1.476370260454659045167690365596111031368e-13,
// [2.8571,3.]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0007101931184175526466708509397454966830974,
-0.00009883581587393035120969589311032207043556,
6.881942302732684602073956046495097634514e-6,
-3.196619579137590760620696672347986700275e-7,
1.114277628391743655360680341898297445329e-8,
-3.10910100544950621352500984863440320695e-10,
7.234494382849863421263792322322220799302e-12,
-1.443341611631144949247638562195012876516e-13
}
,{
// [3.,3.1429]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.4990066556314715699526703412755283443677,
-0.01052678533403660333811289979976862022552,
0.0002902119696792542836383272574919179872162,
-7.581789184853335457337521064010464914166e-6,
1.77606301721802374450233099264225898946e-7,
-3.693114300777202324234019439711292311714e-9,
6.838191909135921299016351941395261793538e-11,
-1.135060209518681453195654619016250258313e-12,
// [3.,3.1429]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.07368749733825622296945327306382876415048,
-0.004062967575508628915202623860377921681879,
0.0001592175728819581891308815714105702125737,
-4.972976485452974412123120220443592739071e-6,
1.292590000186070120177332786045572640472e-7,
-2.871683071978130016938012239569497842427e-9,
5.561957777357859023051446348565494963116e-11,
-9.534163531397656067123789095891828754031e-13,
// [3.,3.1429]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02844077302856040205820378168130270720675,
-0.002229046020346613613573921993040803073818,
0.00010443250619454589140984902685808678575,
-3.61925203256261438912678971058953086637e-6,
1.005089070735165731708884264676227104424e-7,
-2.335714665371605281876859139083714151713e-9,
4.671882760402283547167567892525296977493e-11,
-8.202726696558771892917464827858137791894e-13,
// [3.,3.1429]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0156033221424262949853698710017677202341,
-0.00146205508672294047660099983791107698821,
0.00007600429268384462848051589790762534453192,
-2.814249426138812414852042206877792502797e-6,
8.175001289165427302490474035842231087752e-8,
-1.961921186598136775319287990678359302626e-9,
4.019462914166042587051188285697734097946e-11,
-7.18863439360430786545783172543473879125e-13,
// [3.,3.1429]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01023438560706058305759118129277794726331,
-0.001064060097573200546898775253800265799771,
0.00005909923794894180795942899139465872427835,
-2.289000385936595290808754189213723009936e-6,
6.866724117430378758699885307473059399289e-8,
-1.68793470800747506018308490016501457587e-9,
3.522544975006213016368930941981167675056e-11,
-6.392449813955195191478087930735996779083e-13,
// [3.,3.1429]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007448420683012403575153791833716707690582,
-0.0008273893312846236198359629285363470513471,
0.00004806900810469280243895753458419480711937,
-1.922682775348355382721743309272441211586e-6,
5.907771445624291529619950297325477572671e-8,
-1.47925319696190487923635320165814657398e-9,
3.132404095028953371756718952026480792038e-11,
-5.751823697609015720158166700483164679259e-13,
// [3.,3.1429]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005791725318992365106988839969405987750396,
-0.0006729661134651889067293675978214758347846,
0.00004037633828233772198602596157016168915391,
-1.654176025188069341855686910794468833335e-6,
5.17738615968798024940596669713898099428e-8,
-1.315413751447242584012752639182598582093e-9,
2.818488583814716762527692576285273590804e-11,
-5.225846585838541104885004401706291260046e-13,
// [3.,3.1429]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004710762794256322133263471912873988566437,
-0.0005652687359522606703933808487840707102207,
0.00003473769652896998512365602024515541153297,
-1.449668143410289618938323920522518491463e-6,
4.603948102693340422996008623597171711397e-8,
-1.183585706697000791356058038417552049916e-9,
2.560752417664200677707830421391596368224e-11,
-4.786646152473653038103240067737300093657e-13,
// [3.,3.1429]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003956881151665824494364502193179767412499,
-0.000486327751405148684373255227416272600916,
0.00003044303101163512745386883332830089642803,
-1.289105485998577631498373430728535876586e-6,
4.142549948045484061728169776716170670225e-8,
-1.075350467825579412794867967036307246363e-9,
2.345537875736555726835237966473160077599e-11,
-4.41462053184335857954013112758513142028e-13,
// [3.,3.1429]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003404294259836040605616115533965572698561,
-0.0004262024341624918302247810325739615662173,
0.0000270712152059878900325811694534280170998,
-1.15991400145103933654970131100959562202e-6,
3.763726613709760568745179004812556935818e-8,
-9.849723232056816725749465392550356739896e-10,
2.163239836013634930675724375196424362072e-11,
-4.095606330332706084947867844209945049906e-13,
// [3.,3.1429]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002983417039137442638291818524435454150607,
-0.0003789970128834575056859417593353618383564,
0.00002435819403048846119140465769123734742046,
-1.053843466757054125629952766432830716351e-6,
3.447403109039652237237457273030520603907e-8,
-9.084175144082292469871364696726741782954e-10,
2.006918078756667175747038039023091859186e-11,
-3.819128358088442999995624864271004399834e-13,
// [3.,3.1429]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002652979090184202376851738825893950778911,
-0.0003410147164264891194779743614953221915681,
0.00002213071280191377990518188252044943634013,
-9.652728845047137101426790910787394469149e-7,
3.179461279571048279114262295215567688052e-8,
-8.427714456240887282005613944323090911259e-10,
1.871439640414682389561346106249487157029e-11,
-3.577280428701332638696835218066214044177e-13,
// [3.,3.1429]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002387103014985423682574191591804262846111,
-0.0003098299792264644104446101547573990924627,
0.00002027073057461375006628454918043453153235,
-8.902491714203390865978793310327191221273e-7,
2.949700040001377969002185758303771371038e-8,
-7.858784999514184893852077368388842799994e-10,
1.752930395579169690894555922030424243469e-11,
-3.36398813602251622235905796729901012848e-13,
// [3.,3.1429]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002168809854585250727547071427617349827571,
-0.0002837902280442824960678349118137744948217,
0.00001869523259984109515091893552033661780437,
-8.259160236006907840612821670060439584771e-7,
2.750574731197463003851288238718734188633e-8,
-7.361117225082010730852779473389364121048e-10,
1.648413810781425358847866412964846465025e-11,
-3.17451041098709570313363406859088816761e-13,
// [3.,3.1429]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001986531596309977334289971989423503201476,
-0.0002617332563974818714748075251596081015224,
0.00001734423649562777228285977867918198275746,
-7.701609364738203723597736309852809744319e-7,
2.576391011090891133050463122417867774947e-8,
-6.922211099598366487757355234839807054579e-10,
1.555566702503235930479055582797336569467e-11,
-3.005094648591348338148971944238541936977e-13,
// [3.,3.1429]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001832132794782372968811105190818529722624,
-0.0002428193109385102301168399827732620115233,
0.00001617337966596285309085924279341025523886,
-7.213894942488236053237421368998455004856e-7,
2.42277386802567980194762427459403939102e-8,
-6.532310380178043293873634831052024180532e-10,
1.472550245918774330751165025933404725088e-11,
-2.852733129302750775507259254563187998657e-13,
// [3.,3.1429]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00169973517656957148536637040914996228329,
-0.0002264273153232148128861162407048963606595,
0.00001514917937923733912063408998245504922835,
-6.783766936525017175508977414427297743157e-7,
2.286308617004385563999336727978779652707e-8,
-6.183692916827070041924289164604500167356e-10,
1.397890618842134578973913126170208082649e-11,
-2.714987779984733456562691198915477626899e-13,
// [3.,3.1429]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00158499120726250357028099768031248056178,
-0.0002120885113090698364016471682248554157471,
0.00001424591056671404862424009107360902541408,
-6.401664228777713316489753402406845797339e-7,
2.164292505539351218107328149188563914932e-8,
-5.870169405092440175308715279193038398256e-10,
1.330393132691747774708170704891005542903e-11,
-2.589861986675411347905898744343920142151e-13,
// [3.,3.1429]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001484619579163488739954703761555837811342,
-0.00019944274793375491735648709321893057958,
0.00001344349488044422427890876704901759961774,
-6.060019112216418155810435271363658024135e-7,
2.054559277080554531769712221751808558976e-8,
-5.586722771809540779927349253409405787147e-10,
1.2690794193300458673407825873264632121e-11,
-2.475705402644294378079989486248027975936e-13,
// [3.,3.1429]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001396099235536284311294854347396382814359,
-0.0001882089283259875875799034211327569344456,
0.00001272604013566505743724376802283172466374,
-5.752766068447329853057253205544883810714e-7,
1.955352956027547164070401932239235792318e-8,
-5.329244386703069745292143563368176437537e-10,
1.213140785927277536466011675815017595486e-11,
-2.371142283315057110889708143729183501926e-13,
// [3.,3.1429]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0013174624982819130071536921340012547494,
-0.0001781645618990886388752790583129747684574,
0.00001208080874374955968675456164489570395938,
-5.474988365744045958172212793262219577845e-7,
1.865235521790036887220861285637391752477e-8,
-5.0943381733022540032843204667660197075e-10,
1.161903098237775653058859133705009068645e-11,
-2.275016853599661079930232280422814778444e-13,
// [3.,3.1429]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001247151933293620370195244176256434056151,
-0.0001691313224122803289677106827569625372277,
0.00001149747556807228200901857208969771223377,
-5.222659546415547971104193152530682461675e-7,
1.783018347608417541696449556631789035241e-8,
-4.879173134499579337539260824003576571631e-10,
1.114800009942247251689497507986415172244e-11,
-2.186351178692369414656139631575418641405e-13,
// [3.,3.1429]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001183919256885962204529761362733518843692,
-0.0001609646579528064529385213838494307868403,
0.00001096758504748208223479639015995472711107,
-4.992451455502402711162179865335768500476e-7,
1.707710584499484672670124381812017854147e-8,
-4.681370928100507427107923593754149962088e-10,
1.071352318824189864513293324145795954972e-11,
-2.104312331708392355034756558429589018034e-13,
// [3.,3.1429]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001126752605669645075756300772746049031263,
-0.0001535461906645510901200655518203618222891,
0.0000104841480565641478242608305309058810762,
-4.781589715823756815883420006918436779277e-7,
1.638479812698963712555831868707053357988e-8,
-4.49891917245358340191768920739340091011e-10,
1.031151878505731975250724163498476867985e-11,
-2.028186555849062591591418294979721860369e-13,
// [3.,3.1429]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00107482333465185753922715578463770279516,
-0.0001467780727917069250226013811252838968522,
0.0000100413384032386842382150867487396105423,
-4.587743552015614180735597891850371943323e-7,
1.574621698632149498946392631769982837272e-8,
-4.330103883435124261685967575124972008371e-10,
9.938489375825731547028802786791629888838e-12,
-1.957358746583161119220709601471470129222e-13,
// [3.,3.1429]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001027446509541948386536605920725694820619,
-0.0001405787376451568861358480659915471791341,
9.634261459241297499804219392013661099491e-6,
-4.408940830047984803955308298687415834522e-7,
1.515536347858629438864779581004622847204e-8,
-4.173456304977325901477700572389262739337e-10,
9.591420856297369022796361718117436916028e-12,
-1.891296021306775555836475947585467868879e-13,
// [3.,3.1429]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0009840511635160981171323393925574413228552,
-0.0001348796604291995030878702235886043729722,
9.258775743109006718546816346422055675715e-6,
-4.243501845469591673596864689731549971479e-7,
1.460709695757187806710671139852616904863e-8,
-4.027710687264986334929573545521825776638e-10,
9.267702021210160831382899808734327422659e-12,
-1.829534459138181270084133663763039613051e-13,
// [3.,3.1429]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0009441576230043964384280714670030087167081,
-0.0001296228604033530830342320268742093545892,
8.911353875494128512106553233980115564103e-6,
-4.089987217325181714610503643456172861295e-7,
1.409698729894713580434611171927625826145e-8,
-3.891770476239631408792845346931880414909e-10,
8.965059587533986574266142133391191922706e-12,
-1.771668321003829935787773721000550246841e-13,
// [3.,3.1429]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0009073600228234715005279754669627549692007,
-0.0001247589542567500934244479662955552068947,
8.588973156390629961357492105028940008648e-6,
-3.947156510788123068229710161158451955963e-7,
1.362119656352689506977084040975082176526e-8,
-3.764681026670872046813037137389272663363e-10,
8.681505371449927239293721828474539328381e-12,
-1.717341226225413363174538540321273426036e-13,
// [3.,3.1429]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0008733126797972505755921298612085360194369,
-0.0001202456241893061040826739428966595821181,
8.289028672662582869088589611999601908963e-6,
-3.813935102874290562490928805600522114967e-7,
1.317638349302204482883736538451436979905e-8,
-3.645607419213263895056917573315244607955e-10,
8.415293052431827804072611347321396570032e-12,
-1.666238884297585873033277735202980328074e-13,
// [3.,3.1429]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0008417193693251426524015990594421355385688,
-0.0001160464014171181474177562280329637426308,
8.009263716043323224753237486801649369533e-6,
-3.689387441251865101988421826689690772256e-7,
1.275962586973885552716748847074213756508e-8,
-3.533816303579223335678024589255514043408e-10,
8.164882557961213738382438161471040237007e-12,
-1.618083071775510346110007115054972231464e-13,
// [3.,3.1429]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0008123248099198269578289798382139226212759,
-0.0001121296920244529514199034957375843206304,
7.747713626636029903398423856882373368142e-6,
-3.572695304956947190667636958901244405836e-7,
1.236835696768444588658378578281417721527e-8,
-3.428660942001599681360245242147357449308e-10,
7.928910549441872128206023821410350265142e-12,
-1.572626612772094532684450061297494740842e-13,
// [3.,3.1429]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0007849078441711705878152770935704149613682,
-0.0001084679907727550418563715599448436688831,
7.502660140416513047064232937013326138885e-6,
-3.463140010702923362962705807309846234568e-7,
1.200031320468600514051368152927739212861e-8,
-3.329568814894815500521863081778623423171e-10,
7.706165825933732896280351383594411913877e-12,
-1.529649173567163985855519599609198714119e-13,
// [3.,3.1429]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0007592759354092852227398421159611035641132,
-0.000105037241965685779956854076210205556076,
7.272594022482883556241556218404379697056e-6,
-3.360087755473710089640606018434318907081e-7,
1.165349076220496991981120117414344623374e-8,
-3.236031291750975706185350135723044458069e-10,
7.495568717143422831161388553394745936974e-12,
-1.488953721584831970693651789981219985625e-13,
// [3.,3.1429]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000735260693759800391218230823839631876712,
-0.0001018163163146187355864419688483356277736,
7.056184286501365280305912914888776130033e-6,
-3.262977470071606284401076117257655171083e-7,
1.132610943347356391647098338466345091617e-8,
-3.147594977319311366573190964914539503189e-10,
7.296153731908248742212382269963197631503e-12,
-1.45036352962025287562577146464112213038e-13,
// [3.,3.1429]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000712714214202331082313025001969779657678,
-0.00009878658001088105816031400529996348770284,
6.852252687156785271273387344225498789324e-6,
-3.171310696595424962159797758571290831499e-7,
1.101658233512298593950944804537660314053e-8,
-3.063854424924153792496244816400069123955e-10,
7.107054878476188291959377410256328123063e-12,
-1.413719629961060681831930318741876212861e-13,
// [3.,3.1429]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006915060600761673419367565093563337154886,
-0.00009593153762006034040850127727923914559308,
6.659752462856650257537287545513130222956e-6,
-3.08464310769630102153137276787912162905e-7,
1.072349040379643466144376304921668650637e-8,
-2.984445971797317030972261973932939865408e-10,
6.927493189331527876877058514535878818602e-12,
-1.378878641616219092101125041500980927496e-13,
// [3.,3.1429]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006715207633404223192053362262116028895106,
-0.00009323653447986168919001875377505050275125,
6.47775052616834297963478700691279019147e-6,
-3.002577365629265445974674805188256711831e-7,
1.044556081981254664768246580593780718239e-8,
-2.909042500203056744483104998661319636063e-10,
6.756766074298462184981185569626388763602e-12,
-1.345710908466904751526428969270724533504e-13,
// [3.,3.1429]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006526557413590317621370419025251959226937,
-0.00009068850736622847434999792574801554025604,
6.305412467827428006222289272968445530199e-6,
-2.924757080878945391693810471670041084889e-7,
1.018164867110283816580214752938012229067e-8,
-2.83734896634216414888305593090101358665e-10,
6.594238197206079493600029935054781803582e-12,
-1.314098897708518348272410363891584694868e-13,
// [3.,3.1429]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006348195515635992596525046975995962180917,
-0.00008827577454945861027836913124633328082336,
6.141989869851621913096862752064033280941e-6,
-2.850861678061993882943834683024206807184e-7,
9.930721304376108242869257727969249160223e-9,
-2.769098569069770821805828638580921764837e-10,
6.439333628016536788602856277352131685956e-12,
-1.283935817149810001005212785202765348171e-13,
// [3.,3.1429]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006179304218462102124855358951460089751003,
-0.00008598785817780013853314378231923452759847,
5.986809523935895638496050239343314616574e-6,
-2.780602014253075874873148292973505559043e-7,
9.691844915630820057606090552484659535694e-9,
-2.704049454238399298675427402997471073212e-10,
6.291529067384173024759694575555673678079e-12,
-1.255124417296467339614288367612903181816e-13
}
,{
// [3.1429,3.2857]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.488762674458373291705196450788546348713,
-0.009968414400162214714249138495549022864224,
0.00026849631103550783535018578445739244419,
-6.906966436606726388991217967990952799208e-6,
1.601277712959872776643474096109172441139e-7,
-3.305782944579974632752205970804178247165e-9,
6.089405303006291515515999670586479575861e-11,
-1.006889514922171077686074717736941795906e-12,
// [3.1429,3.2857]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.06977890080113550264912331688488955923401,
-0.00375894835449628620316626767871330225005,
0.0001450462951687749139030299475793313429894,
-4.483577629227558588294378605703209066383e-6,
1.157024026115014434152539772954613123792e-7,
-2.557234002510116031954137433178851837232e-9,
4.933902238631308269330068376450319402404e-11,
-8.432689978230671675052418789284132620462e-13,
// [3.1429,3.2857]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02631263848147400311546072063107412792757,
-0.002030648132362141940931211218191198352019,
0.00009415513021380817399308589223291139977916,
-3.239667301396406717435607811280783163694e-6,
8.950318969527111942218531607542246879167e-8,
-2.071967504914381948746006807868828585604e-9,
4.1321437161068300980762769020874599426e-11,
-7.238301511417260463659469221746554117248e-13,
// [3.1429,3.2857]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01421453692653499331419916756179303305571,
-0.001318171822992696120121826001416762914817,
0.00006803301332935068384361759408849865102177,
-2.506089336200418372038446175419328115261e-6,
7.25188623234320129970312711995196278079e-8,
-1.735262924372442363963012639576549251087e-9,
3.546879283639100205351014320126780352763e-11,
-6.331660997266461186035649483836036141837e-13,
// [3.1429,3.2857]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.009227202760948872596134090373994193439643,
-0.0009524621866103605760606466026863576097774,
0.00005262787606023227892032913097405693445152,
-2.03052816701618404477312757479346664366e-6,
6.073420203979314411049586001987118290463e-8,
-1.489478481149521694074269499677549762211e-9,
3.102614126401995432315442817418067664741e-11,
-5.621834431022717347825673959504788124994e-13,
// [3.1429,3.2857]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006667235306272523810321373153004324813575,
-0.00073679026484275855015455387952723337284,
0.00004264109150736118694718044029909610150768,
-1.700557676848559836873700758493903421774e-6,
5.213174655593899935234819208022397973752e-8,
-1.302908482270897280211958905480667881807e-9,
2.754789845542235234096199698325701172834e-11,
-5.05204163360988524012394086889062329262e-13,
// [3.1429,3.2857]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005157531853899309647830432402020702988164,
-0.000596975281102608855601176796246993298215,
0.00003571171121383926880969489079277415661715,
-1.459688921476908200064741251369420647218e-6,
4.560179661931749363314620032650677434334e-8,
-1.156839792253035333795022766256031906062e-9,
2.475583653085372069233318617836905679033e-11,
-4.585161592324492742433935787957673134775e-13,
// [3.1429,3.2857]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004178826967718261801901170073227389513485,
-0.0004999639569933400068444637979646109310936,
0.00003065346735103305376991159609401729853481,
-1.276850321731288393110294719659931741786e-6,
4.048939248910126874081464928122239339228e-8,
-1.039587785580472556336425742183114453849e-9,
2.246805901982467579367008195585453788015e-11,
-4.195982678835310033515771906830559435422e-13,
// [3.1429,3.2857]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003499747698953379874258579397516559899629,
-0.0004291485429140851402594955529954131177501,
0.00002681385675637372699568945323037737384447,
-1.133703004799465760271299296602058510898e-6,
3.638557227303932657060523137193474523635e-8,
-9.435134735537951941043989728798437403071e-10,
2.056102641481343613005174156685223721632e-11,
-3.866823218300952715411351589786436594936e-13,
// [3.1429,3.2857]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0030040398003985958199838252854960038256,
-0.0003753939945888820927853125262427220310696,
0.00002380776310080431697873934935406682468921,
-1.018796037648628519422487362796081435314e-6,
3.302297136723535716478678525623537796293e-8,
-8.634286747836634356924408087037471379863e-10,
1.894809664294616459086300382692593090006e-11,
-3.584938468584516911656241237983127433887e-13,
// [3.1429,3.2857]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002627757962122174497993388324560900567769,
-0.0003333086834109341817983285542862256444994,
0.00002139471679063574334737252454709989996921,
-9.246432113329400827152107704027485166472e-7,
3.022000342350176214598919083430456478724e-8,
-7.956947749138505203830177185045324247472e-10,
1.7566819062006013804729319506745683224e-11,
-3.340922951365058253737475883840128331408e-13,
// [3.1429,3.2857]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002333160783876539130185386558224807689245,
-0.0002995260350685949739825891755555801745459,
0.0000194175074380054124874431386960430797765,
-8.461601080754720239615750436296720869781e-7,
2.784931693970752993402253619628066166648e-8,
-7.376891126587466342780325582049788879506e-10,
1.63711057500586726153697639422565354582e-11,
-3.127691646697858037336092750128127457944e-13,
// [3.1429,3.2857]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002096682245480164683553258238411524407333,
-0.0002718451041317886893954403565499506589163,
0.00001776936226959780775746282018652023863825,
-7.797808857953296737108408604325052376415e-7,
2.581911877111887152144653706186848148958e-8,
-6.874761990708547408848689800274449805817e-10,
1.532623926919945389700888375754692901745e-11,
-2.939810572264684844991301064615628062321e-13,
// [3.1429,3.2857]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001902915728922520698660546663778420532537,
-0.0002487710717740985085781021783838784246689,
0.00001637539860171412553524038139172345167224,
-7.229353364234259536472562151373416076807e-7,
2.406166680478090744100623937237344905264e-8,
-6.435980605523090381117186301343606643122e-10,
1.440559244201423370985798163891597933506e-11,
-2.773045205248247069865716868270903540751e-13,
// [3.1429,3.2857]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001741397502418689439425563445549685913576,
-0.0002292555804237415076578640202028603738721,
0.00001518164206490352471807702452387586444619,
-6.737266807839506359940908895723187812873e-7,
2.252593196493444257559660204659813184163e-8,
-6.049364811588401229437958510019308761167e-10,
1.358841557515200720620856908654781516326e-11,
-2.624048678767231920046929107117627426525e-13,
// [3.1429,3.2857]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00160478906296619043884440171483833071975,
-0.0002125429889084061728161475963185253348937,
0.00001414826029647398043614411297804408052929,
-6.307261047451814205186065080291983571955e-7,
2.117277669366453503307003110350690017013e-8,
-5.706200742325806044865201792644939639761e-10,
1.285830859053458476605771635413492981423e-11,
-2.490141974444586025866686129193625013237e-13,
// [3.1429,3.2857]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001487800922358843100272962204068231191155,
-0.0001980756441504043676920310025051729508647,
0.00001324524819965931613344294792183604381244,
-5.928377566770272103038249868456263011157e-7,
1.997170245805584121120512710510605735547e-8,
-5.399601178327596838699538048245281129054e-10,
1.220214394606559033458549802543764130666e-11,
-2.369156058341190452846684233997902571826e-13,
// [3.1429,3.2857]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001386529509052830469255762776202083448952,
-0.0001854334747950097937747139874350098062739,
0.00001244959289022761196145489175661421564617,
-5.59207677650927409940932706591403539199e-7,
1.889860399027222686510341856382914046476e-8,
-5.124053217289878418295323144137941155028e-10,
1.160929308474298009559596216588869512142e-11,
-2.259316587138693168729405363498345738813e-13,
// [3.1429,3.2857]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001298034323565068456276308786923248245953,
-0.0001742943004629757162633108177533382614154,
0.00001174336123067908974228847453507825914803,
-5.291609201617470858745350779354541736379e-7,
1.793418613232571642572738683449687808355e-8,
-4.875093414699103617300661957082172144946e-10,
1.107106148220209393650075319238252565099e-11,
-2.159158416501962247600873730908475127896e-13,
// [3.1429,3.2857]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001220060103240829917778013642410288812524,
-0.0001644070572293053598276660308091446681401,
0.00001111237932340591110836073291644773401873,
-5.021572197810564756105481631160700316353e-7,
1.706282682848240056549505589699442540263e-8,
-4.649070527903132666053343241980793232517e-10,
1.058026972798203143245919082985552741667e-11,
-2.067461326293676596691778953078705959666e-13,
// [3.1429,3.2857]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001150849400605137426491734902913867358953,
-0.0001555733105274890873428386337822344856628,
0.00001054530161541104702048220386189505501881,
-4.777591589443052405976149461449395797324e-7,
1.62717467295134810760459615870318064122e-8,
-4.442969588594443580780218174807939613129e-10,
1.013093857159847537673184442292999822436e-11,
-1.983201083549106988995832947449396737835e-13,
// [3.1429,3.2857]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001089013173692423522578585039012210518409,
-0.0001476342226155685768231844506000453590383,
0.0000100329423378389369419153858185996408364,
-4.556089158697044862132195000631554104418e-7,
1.55503934463883265847927388032728476212e-8,
-4.254279636295488025809006073340004961369e-10,
9.718049125298176487697834192740729762504e-12,
-1.905511749647196259312903994642836016337e-13,
// [3.1429,3.2857]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001033439558308979952169625275037654332632,
-0.0001404611927295660527246491663147840993437,
9.567787233272011150810900630227938721301e-6,
-4.35411023661517702134433046700237964433e-7,
1.488997861747464837229516800071077556821e-8,
-4.080893014519110777689823758989369035689e-10,
9.337358164620280807581121683607504805535e-12,
-1.833656337461739695069477245461775527883e-13,
// [3.1429,3.2857]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0009832283491069622864828033889910784082719,
-0.0001339490212656356005715896993470104632104,
9.143631496899800402889872124470701361012e-6,
-4.169194081915755760527726729235869762736e-7,
1.428312544510110320974765088137195727344e-8,
-3.921027805654824021086119617201222079569e-10,
8.985254344806026386004882717400470670737e-12,
-1.767003743247447283117842814546774809429e-13,
// [3.1429,3.2857]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0009376431488594491242114125667661924242712,
-0.0001280108409564307040214950782705588884321,
8.755307572030746951367842739818749594679e-6,
-3.999275191229573859632558043009481510503e-7,
1.372359721766016123821825515685666428714e-8,
-3.773167448723676655185029739139998928683e-10,
8.658645164135344289835811219028311632636e-12,
-1.705010445792969733684587785613140148476e-13,
// [3.1429,3.2857]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0008960758866950148509777094594272108948652,
-0.0001225743060082696005737705126473876279243,
8.39847790158951373012040680009660017675e-6,
-3.842607285288141064419149066102993017252e-7,
1.320608597175088060291913395553408826432e-8,
-3.636013269470307257077239126681466659874e-10,
8.354867287434084641834894933763444191249e-12,
-1.647205864560820168085143313926935921753e-13,
// [3.1429,3.2857]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0008580201420578871292948672526972477385738,
-0.0001175786906220976112870499960886266973513,
8.069475299112269540911775294847046521301e-6,
-3.697704134323199783519064127025776927195e-7,
1.272604634750168748301473890603408656473e-8,
-3.508446820663523604855489498020665127542e-10,
8.071614799056724387559735538074461490806e-12,
-1.593180552864752974020240307100922569076e-13,
// [3.1429,3.2857]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0008230508343546832065885438588879064328502,
-0.0001129726541874211343369795237807253403937,
7.765178682085671980745750961856910637985e-6,
-3.563293037556728860954977190758932819734e-7,
1.227956377962289078778501391020187580418e-8,
-3.389499751945559887526994734103752894922e-10,
7.806881347899152206942462241523520402279e-12,
-1.542576607119399857192851972245529842202e-13,
// [3.1429,3.2857]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0007908085793119478701016772156386490517228,
-0.0001087125015490534067651536308962861531042,
7.482915378875875334130833739827904356482e-6,
-3.438277916695341279839312365759949930361e-7,
1.186324904187948274723239697585000245742e-8,
-3.278329513679007409425973182486878123622e-10,
7.558913151440744194695775858541806171133e-12,
-1.495079822652231666366299254922250542151e-13,
// [3.1429,3.2857]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0007609875108433737791373000757323439458677,
-0.000104760815304120615601385359612426349222,
7.220383625066765725981255088305061981215e-6,
-3.32170978838241610262215330943937302172e-7,
1.14741532105557279264732141159199603757e-8,
-3.174199621065858592712343626569515126868e-10,
7.326170558162018455223665992900094238423e-12,
-1.450413236689942223509925834624800284738e-13,
// [3.1429,3.2857]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0007333257071288442429146455002424197527032,
-0.0001010853707507971905265085960865165437505,
6.975590555609438175427997349108940580533e-6,
-3.212762953967974518533231711135342082663e-7,
1.110969858887209336303180367374725656684e-8,
-3.076463512368146396276930495847099568955e-10,
7.107296407294303070643044815503647403185e-12,
-1.408331781075528795461782355472258386381e-13,
// [3.1429,3.2857]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0007075975952555802692089438441714734621296,
-0.00009765826777839848306063334490581852978619,
6.74680220333893627810672441471814290877e-6,
-3.110715658345246320406902048213432258034e-7,
1.076762221075771773684963746416263187547e-8,
-2.984551261672844171752665601665083451622e-10,
6.901089826393576900149296804542129635732e-12,
-1.368617828815309549744154263636290787238e-13,
// [3.1429,3.2857]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006836078744487893186693616818318494313129,
-0.00009445523084661512247932157268013318295572,
6.532502882531041792897761575700737386531e-6,
-3.014934271006815613330782177701975503616e-7,
1.044592933552775543303915948243696852938e-8,
-2.89795857527121133745192446773346165828e-10,
6.706484408798240589683413387487792818846e-12,
-1.331077465179620428981338622211027895005e-13,
// [3.1429,3.2857]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006611866159263057962323612404239283202528,
-0.00009145504035530807033438845230632839155619,
6.331361969120180617749459675878825880172e-6,
-2.924860264554154574732916613165019908611e-7,
1.014285493521125091816027681852965119315e-8,
-2.816237627367806637454645561206281783337e-10,
6.522529941496074385385102743994771472235e-12,
-1.295537349693675088167357066045539631074e-13,
// [3.1429,3.2857]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000640185282487156432767418279689443383214,
-0.00008863906756755930437454362210051611424336,
6.142206555569443675380892645189678494665e-6,
-2.839999431149324633329820171080854026014e-7,
9.85683161953282488532190413814844180553e-9,
-2.738989386779620721206549600748435430345e-10,
6.348377028432288530549827804876338856407e-12,
-1.261842062767688511884599112422689462851e-13,
// [3.1429,3.2857]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006204734729729150725216040303266025650257,
-0.00008599089177785211116531599641508468987252,
5.963998805419159380132759958845182868617e-6,
-2.759912901509760554298766778431259468356e-7,
9.586462779359757482937065782636511566375e-9,
-2.665857159570601772958503419537536578971e-10,
6.183264088615687822296973324803162164608e-12,
-1.229851851970531194656017284052409235016e-13,
// [3.1429,3.2857]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006019362424449647214591386290699445317441,
-0.00008349598327575110081270599510918450366229,
5.795817093175940209172127173488857385178e-6,
-2.684209625073378646942573234901413947006e-7,
9.330499985922930112236982672583667515247e-9,
-2.59652112897717344859430520059541404274e-10,
6.026506312532874208922724694869293049077e-12,
-1.199440709546179644351444979057098055803e-13,
// [3.1429,3.2857]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005844718829302576503270253040402770218416,
-0.00008114143930434885912085014249667018110638,
5.636840212659409932004853874594063558221e-6,
-2.612540041780376148227514749932318271632e-7,
9.08782388055622598796923588308129442553e-9,
-2.530693717719672750504501249574379249345e-10,
5.877486241697952006821867141493006189561e-12,
-1.170494725813854099407681491426075378515e-13,
// [3.1429,3.2857]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00056799007513044195975876174938435189111,
-0.00007891576297712012894345090251104159106093,
5.486334087743982310223719126827586068145e-6,
-2.544590731200208077031239766032651957973e-7,
8.857427942786638453950112380381655813597e-9,
-2.468115631944135219455406326647707201058e-10,
5.735645700070791111092379742967489816343e-12,
-1.142910673405684899648895952092789594976e-13,
// [3.1429,3.2857]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005524103408398408497344002231620477730392,
-0.00007680867722830671209956176007226301916864,
5.343640535525512486508854668653189901376e-6,
-2.48007986759676983761114423300252727809e-7,
8.638404644130584757284810098740226037435e-9,
-2.408552472876126671417513180064073421976e-10,
5.600478856611663272194798597228202943926e-12,
-1.116594785499200498399351780193640273263e-13,
// [3.1429,3.2857]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005376607405981469329910876495189652587053,
-0.00007481096749725058892291713161057117449639,
5.208167721958180447636181931340730960698e-6,
-2.418753342991323746537168059948060625256e-7,
8.429933588882374129404085507817859714731e-9,
-2.351791823490458970302097328816764411497e-10,
5.471526238438026193822855160129088073834e-12,
-1.091461697764111012547079503017799873726e-13
}
,{
// [3.2857,3.4286]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.4790560062846940233352855607118712924524,
-0.009451518336835778099459072725333450484911,
0.0002487040131005670663491651279314640133013,
-6.298329857131978886017977140218489283429e-6,
1.444779456033913848444906819622634327381e-7,
-2.960788363288015744901772151553676054013e-9,
5.425055903727194942678853285364129709535e-11,
-8.935103871841250212562269822330844322491e-13,
// [3.2857,3.4286]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0661606283578504463867439793585728429141,
-0.003481856183407210365327271533260935797721,
0.0001322649269998012658145472494825025969857,
-4.045382506037717369787558628166578954223e-6,
1.036275923189567089334568508585432594806e-7,
-2.278243707699534315979079861172019383936e-9,
4.37832765705922653716733082880041945133e-11,
-7.460609440589343323736620714345102254407e-13,
// [3.2857,3.4286]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02437299328385047228709786231748726687971,
-0.001851708977996593828654228108558262314507,
0.0000849530326268180034168767531453983035679,
-2.901572609886689449959301150204028873777e-6,
7.973852942363405934390547119032666944932e-8,
-1.838658038083126044535367365157200120573e-9,
3.65580929797833237271559574319297203891e-11,
-6.388766873212931098634197929737480011877e-13,
// [3.2857,3.4286]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01296196284597615656104933553775001494753,
-0.001189342456774907336716530832847305407938,
0.00006093302480764347346269745459212094697329,
-2.232678845650370430271603532312287157911e-6,
6.43530310263083630586019213607360576774e-8,
-1.535230733329249413738986110619080410224e-9,
3.130593880394988736663739357054999848059e-11,
-5.577936209242881129343913149141101605557e-13,
// [3.2857,3.4286]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.008325397197424351142040130788741812781189,
-0.0008530623473065257336993737586337390174816,
0.00004688625575867841679104331044276382939042,
-1.801884888052581217323193841406900455565e-6,
5.373307539135286487410859425338086460382e-8,
-1.31466399567633054874932243881317721307e-9,
2.733276797374947713730098566360576021418e-11,
-4.944928062145867986767884609315940178502e-13,
// [3.2857,3.4286]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005971436431145679940998731695435791031998,
-0.0006564075806210644426752830734804528390558,
0.0000378395826491229157560327761339336646269,
-1.504526128293718739408971933059503188243e-6,
4.601323959920160867643636547459090476575e-8,
-1.147809829935332482848312580670949652352e-9,
2.423094580994704027924490364738397399362e-11,
-4.438016368236075873308381327497951415993e-13,
// [3.2857,3.4286]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004594853064347450920537799021786742234569,
-0.000529754157087327906973461467454308432909,
0.00003159504869418519977371419030887390512354,
-1.288370724494320745334343320013279736912e-6,
4.01733438196526783653305008640428000972e-8,
-1.017548843090947716590887598656837938961e-9,
2.174701007446804968329971846559987278506e-11,
-4.023507383755402408616145757173611067371e-13,
// [3.2857,3.4286]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003708279099611295184734544042784486641152,
-0.0004423306817182335660581782104828762541982,
0.00002705578521439648738069866427139636721815,
-1.124853641319627556076967813644591114646e-6,
3.561420929815949064745894104721939479744e-8,
-9.132364765647618781804106440283766328937e-10,
1.971585825752051787178847616047129864972e-11,
-3.678589835428022423703359705745629479975e-13,
// [3.2857,3.4286]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003096314772027634810394497608388807364065,
-0.0003787809930012200373952766294949737167922,
0.00002362192646772677197533363968302887575184,
-9.971978735800163805860386462986044773309e-7,
3.196327648518877299181503100131479329601e-8,
-8.279390232048112332417835387815275053515e-10,
1.802571284411243751433617187042663811258e-11,
-3.387310102024599618506283522758380065587e-13,
// [3.2857,3.4286]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002651466951008540120187724060530352196924,
-0.0003307069705478683487466736446754621268599,
0.00002094115534519393566595996469205850031561,
-8.949717538437479397968818057647984112053e-7,
2.897786563094552364005036304755427924613e-8,
-7.569622575364291235247577427070849220159e-10,
1.659839941428520031769683978405745864133e-11,
-3.138197353608434856391567105259923533299e-13,
// [3.2857,3.4286]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002314948793835078308755252627422493543262,
-0.0002931761748324296744999658354354381538122,
0.00001879440683073142406148195683169653872175,
-8.113802490835671630796619235618267665288e-7,
2.649367884421015353632534683338070307859e-8,
-6.970231706747652185368961732750289466161e-10,
1.537770964136649463011710791310723552213e-11,
-2.922804809579080974396589411053926506489e-13,
// [3.2857,3.4286]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002052233223827007597045770795147994510012,
-0.0002631216956299728733248258227253093808662,
0.00001703898523076685806949528451476561800255,
-7.41823018320519609797782731285411969742e-7,
2.439581081431435746117307314735889971924e-8,
-6.45761251037612485469372988580206687164e-10,
1.432225333575244280045811187416023035838e-11,
-2.73478210459869864978042553119046402245e-13,
// [3.2857,3.4286]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00184185186940980999593022077241808413209,
-0.0002385457932304851127199509206892579117856,
0.00001557828338474217684611632536410511247347,
-6.830827128369009237972465328182158291162e-7,
2.260164363611542998237602913812459322403e-8,
-6.014382929837245722364203432933004848328e-10,
1.340091295675624252208372143911404260644e-11,
-2.569267301457030007458341667271607769719e-13,
// [3.2857,3.4286]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001669820552613395678044098127691703921759,
-0.0002180959673861539102714872690380358654531,
0.00001434473696958557502907962871848230289921,
-6.328460312739392398185599416478078252514e-7,
2.105034011235485627984379082469992078546e-8,
-5.627475016555339887415274132521294919645e-10,
1.25898644197027328716175669258369926072e-11,
-2.422477688127659277271920198874281218691e-13,
// [3.2857,3.4286]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001526671771703077266606030786770185867498,
-0.0002008263175739742824511372274577296505514,
0.00001328976665676283234981832536493930139371,
-5.894095320967343313646680059278483755955e-7,
1.96961624231657494404428995191825441626e-8,
-5.286883774505533515494707838327706811093e-10,
1.187057196214098216640543883913955148824e-11,
-2.291427818976857958015336455911706075128e-13,
// [3.2857,3.4286]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001405784223017819877011420670018028014566,
-0.0001860567331944673785592338889944535821566,
0.00001237760017404103507726054332563972511783,
-5.514925563396908929860342219558991994596e-7,
1.850409308258071310742521376598118404175e-8,
-4.984825078419647095570987946199462990289e-10,
1.122840651754651903093636317729598430304e-11,
-2.173731123323353689150206024549277122242e-13,
// [3.2857,3.4286]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001302397132361271554438989155963323984287,
-0.0001732864024363725948279154572873876290817,
0.00001158134368314267446352302261293776646383,
-5.18114614388183090506603999310643640499e-7,
1.744688765225889846554767375703892875783e-8,
-4.715155444079302909526900562495503002254e-10,
1.065167357668629608419331959545265442089e-11,
-2.067457669954441479227238687107792114424e-13,
// [3.2857,3.4286]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001213004817054608072576717472508281952049,
-0.0001621388115638049627889829754720691381331,
0.00001088040690216060194155911041388388311638,
-4.885128619625070284880566222515114698102e-7,
1.650304393751664169060681185330640691735e-8,
-4.472963768981652204484185829230126147323e-10,
1.013091621851143572410314019030206692151e-11,
-1.971030453673824967612214033739602602036e-13,
// [3.2857,3.4286]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001134971680946634652199643577480960679164,
-0.0001523256966300645295400087322213404993876,
0.00001025877010122103067380043300611146930794,
-4.620852376064386356589288306625072470727e-7,
1.565537307966108382810660373576824017825e-8,
-4.254278634110900305503490847829198015232e-10,
9.658406902802355089793491639547439543987e-12,
-1.883148604236155237342157236488753910898e-13,
// [3.2857,3.4286]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001066279876410451623034744645450433217162,
-0.0001436227814169183850555754298938765754468,
9.703789989743250942297909118154493273918e-6,
-4.383504532723498099798574884083607538878e-7,
1.488997511219322696778966727576548956369e-8,
-4.055854878469650344236254086687205835647e-10,
9.227771185249497916271584757239507030276e-12,
-1.802729731811047704069564647940370403441e-13,
// [3.2857,3.4286]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001005359469918428614941245010933953078931,
-0.0001358530598562366819305292904612841388323,
9.205359518727069038249738566719125444772e-6,
-4.169193098947227393573508094939431615739e-7,
1.419549197166972852646009179835537399319e-8,
-3.875015575853084302204375927743431277703e-10,
8.833705203535063680747712007045291206785e-12,
-1.728866086129412215322649108222275525718e-13,
// [3.2857,3.4286]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0009509714189936566961146583524146443760014,
-0.0001288750332620167831270535603404676567481,
8.755305507796607875027675185308557067273e-6,
-3.974737816941483934980840270781573063684e-7,
1.356255441641416107528764823525347732517e-8,
-3.709533391651154551836496232737466388523e-10,
8.471760851946254936173980708610744591539e-12,
-1.660790828745350676921255899783573896711e-13,
// [3.2857,3.4286]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0009021252328341174073172999670583966024952,
-0.000122574277108996473118255656710248960776,
8.346949415584275221333758584752507831396e-6,
-3.797515299011393496677814020707437611221e-7,
1.298336677532595984927982751909203221839e-8,
-3.557540366021331165665710285256268603614e-10,
8.138180511371798972076835121178265088469e-12,
-1.597851805803225755086252070558320215174e-13,
// [3.2857,3.4286]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0008580199397629752398841875796053393727777,
-0.0001168572918180295151638127934409200991551,
7.974782127930832966221224065716222114909e-6,
-3.635342757226998643955001353406664016244e-7,
1.245139118898605053990507263573969655812e-8,
-3.417458508211195227249077862590322092713e-10,
7.829768532621506237295585120683120189648e-12,
-1.539490950498113082814898376539431425667e-13,
// [3.2857,3.4286]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0008180010427262065366531928307881296375802,
-0.0001116469497908866226144912547324067056907,
7.63421979018336856420197121126922381788e-6,
-3.486389590932196821982639048279209715584e-7,
1.196110468978673498506579772518050718123e-8,
-3.287945825674548450301468899298993812425e-10,
7.543790305893356609536617161238601490128e-12,
-1.485227957972583270873468684341807963418e-13,
// [3.2857,3.4286]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0007815286485362062910971854745784101937291,
-0.000106879077062427060417982246620350291695,
7.321418140964064969572629323709861611862e-6,
-3.349109369180598613179722251210895396026e-7,
1.150781030383872672075565050906730110743e-8,
-3.167853938141074480136666161960159507807e-10,
7.277892265681697036022968146061501982042e-12,
-1.4346472359828740409420902278359612027e-13,
// [3.2857,3.4286]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0007481535394369893578652977002612644132872,
-0.0001024998539733614252261819513355686432333,
7.033129675285502937206792248542478480039e-6,
-3.22218693926908700292012456225713074282e-7,
1.108748870021549464216374246268609018712e-8,
-3.056194483620411275531300766909049895445e-10,
7.030037947348342953045992703478283395694e-12,
-1.387387391187616361095003924783485589601e-13,
// [3.2857,3.4286]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0007174989778135299135342223424692779324538,
-0.00009846381545386587843962218934507799520782,
6.766592572471135448106108847409964820552e-6,
-3.104496888525900266668884155713632360164e-7,
1.069668061196794876471190166111471529264e-8,
-2.952112265362072472643767759931455619565e-10,
6.798456468553575722155994518916082725641e-12,
-1.343132695630686419225222670868400291245e-13,
// [3.2857,3.4286]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006892467081770610879195054046074136949687,
-0.00009473229601446878884625208459263827266463,
6.519443465910261745709020094346206228155e-6,
-2.995070622194471554673403566162095790569e-7,
1.033239285048452069813990638122727056918e-8,
-2.854863616671790715501174088710264297292e-10,
6.581600713883874300838781998288741092759e-12,
-1.301606112520358185376712495913381689745e-13,
// [3.2857,3.4286]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006631260721012814625472476902075360494347,
-0.00009127220852262036968998662556727417366694,
6.289648306614090441664621090520786680472e-6,
-2.893070047454027332679353414511444105897e-7,
9.992022582348659672690137946703495623891e-9,
-2.763798840606458226312291219163658811784e-10,
6.378113160227778939396923718335258422376e-12,
-1.262563559428586639307784665829031235946e-13,
// [3.2857,3.4286]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006389054596583425301341361410567058974846,
-0.0000880550762924775626151711257485486976533,
6.075447099658996222960561640545885946046e-6,
-2.797766370939500815824114016533830927382e-7,
9.673295868271369542432722896560302852683e-9,
-2.678347858412350766245472306745186029299e-10,
6.186797765668354580984051214517900211912e-12,
-1.225789160645498956340749156075912981226e-13,
// [3.2857,3.4286]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006163855340473428821988552322774449410144,
-0.00008505625939510963195225692399934738404843,
5.87530937897833804772270178512749234215e-6,
-2.708522889642487974453189075222161101401e-7,
9.374217432625197669453093151577614675729e-9,
-2.598008404341778261144361808148368618761e-10,
6.006596705364127903329831605550821349795e-12,
-1.191091295656929568474800868367048413046e-13,
// [3.2857,3.4286]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005953938157657673690619515881845038692663,
-0.00008225433130558361978190427508224143349318,
5.687898068254466744497068464316939644744e-6,
-2.624780926380635257696742919835271437091e-7,
9.093029345302684547781850858707246615873e-9,
-2.522336255963058549323554075293661494631e-10,
5.836571008537219640514648961503894819775e-12,
-1.158299292519476804295729388037579829417e-13,
// [3.2857,3.4286]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005757803191390852852946680612980285884284,
-0.00007963057295545245259651224582553303588391,
5.512039945404439220831379588533361575594e-6,
-2.546048260717896854395633746141737325935e-7,
8.828176827801417643768226476221126565305e-9,
-2.450937102738422138627406399026917200424e-10,
5.675884355544807452100637708881440486002e-12,
-1.127260646813258984910398825373353990176e-13,
// [3.2857,3.4286]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005574140106881671163497496533732414867333,
-0.0000771685592355549404498731248448004270917,
5.346701347512558701941479313268512261196e-6,
-2.471889554668257962517680403804063191877e-7,
8.578279793246821755893760233735851427859e-9,
-2.383459741675367358128410059352506225797e-10,
5.523789450351757778043032543465780844159e-12,
-1.097838671391887165490570617600822206585e-13,
// [3.2857,3.4286]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005401799146488845326091067854437936407713,
-0.00007485381886507134049334525829170929440687,
5.190968064808193588574873265780409080077e-6,
-2.401918383902038152244388640976949408692e-7,
8.342109031172007520986906383732278412567e-9,
-2.319590354513370895609928793896439837619e-10,
5.379616503967970831385895122974699589916e-12,
-1.069910501161829677742183613159811127924e-13,
// [3.2857,3.4286]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000523976732055499334137095691564298398149,
-0.00007267355290721282115206551112536722099575,
5.044028606199014512720648469811108347571e-6,
-2.335790569484182498038793398610892383014e-7,
8.118566177671348591714661112346739009542e-9,
-2.259047671405492652134772335844510139231e-10,
5.24276345757831819305891898677402016682e-12,
-1.043365391956612574324739458164394454223e-13,
// [3.2857,3.4286]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005087148703504896999140662741895845814041,
-0.00007061640048668678104710431919629707153031,
4.905160195921405709131774909261515114737e-6,
-2.273198569517206301455491290390260190327e-7,
7.9066667882861762300041994128325108106e-9,
-2.201578865179066330614023805200457327998e-10,
5.112687646777896778297752063561376824558e-12,
-1.018103264220256134896085605341555490061e-13,
// [3.2857,3.4286]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000494314803406807420291499765217384783195,
-0.00006867224274280260831900130589946543383169,
4.773716995990648928205692183801643771083e-6,
-2.213866739549140687578636955401391367525e-7,
7.705525967917263306962876683578434857617e-9,
-2.146956050783849732117585071766710455487e-10,
4.988898665406659569629197952422428616343e-12,
-9.94033451419845521430808955544366171114e-14,
// [3.2857,3.4286]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000480705699199618212256803312605317604259,
-0.0000668320379437742555135235058256733471083,
4.649120153057609184158158385555828764606e-6,
-2.157547308948985952680801115859839176992e-7,
7.514346118893408286030540683518487365522e-9,
-2.094973288507425936806290091996878295202e-10,
4.870952232585584093197998522847167337942e-12,
-9.710736204249679831784067351739727324025e-14,
// [3.2857,3.4286]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004678242656064197436336568654176687268813,
-0.00006508768214271384014736397279184769521152,
4.530849348797186781332164597505190291003e-6,
-2.104016950365878573342031378585974247387e-7,
7.332406452225389685540898096338651724873e-9,
-2.045444008480854988566018080261134655336e-10,
4.758444902414860681915169459883248571905e-12,
-9.491488369419060772379028256742956997947e-14
}
,{
// [3.4286,3.5714]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.4698470352016235162971303272825239811054,
-0.008972441873047041136287501353403600990267,
0.0002306470785884853594736012930914729014101,
-5.748971615023306602191046124030243204334e-6,
1.304573027413251007650978888173572868124e-7,
-2.653363068501816522145926680747281186643e-9,
4.835409774746725378470029824134565857353e-11,
-7.931867314280001204317776091929225313224e-13,
// [3.4286,3.5714]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.06280709311132928768079896163257104748855,
-0.003229059100238150265640642733298669054832,
0.000120728403915515667263888818164289697592,
-3.652804502547972871863859827400931802985e-6,
9.286770704784773497081449887921539159317e-8,
-2.030624511822193421314772380053943032026e-9,
3.886726893267225023396167293386474170615e-11,
-6.602518456280767556302512157599328058578e-13,
// [3.4286,3.5714]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02260341370166705162140813044698771352789,
-0.001690197654816668541266343254814590999695,
0.00007670889455353028574565772349083658038629,
-2.600295819371922281197360728631577334026e-6,
7.10718576090367752017258840547819341666e-8,
-1.632213785110589883649701024279145491473e-9,
3.235331560555241878854382346262743761358e-11,
-5.640288861169328400052947529871475613398e-13,
// [3.4286,3.5714]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01183138358371667957814000568147291191421,
-0.001073924523748944036762920132124979399909,
0.00005460621220683059754246532263709474691535,
-1.990012032251727855205151900452025739869e-6,
5.712748220914138672049379171928873307738e-8,
-1.358654946963932886051030232394469702709e-9,
2.763827855498391941566691984678108520363e-11,
-4.914910933570560805399117650667163552985e-13,
// [3.4286,3.5714]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007517471666242608068466169399103851654709,
-0.0007644869708952035436550303360203913404637,
0.00004179025267730441697633053910081015042709,
-1.599569518848973241549544768767839241628e-6,
4.755292290197672839474558215984576494365e-8,
-1.160644565495213474566632812367387804009e-9,
2.408383721093218663856768704611206600014e-11,
-4.350251723835792028270932861357142809262e-13,
// [3.4286,3.5714]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005351408796266424634541114326032821346597,
-0.0005850635374818810657169428778370975289835,
0.0000335909598958448583859563904196901374617,
-1.331481856486351448321393339134871589346e-6,
4.062255957339431934860141596305000849056e-8,
-1.011374944429926115156613493963814402248e-9,
2.131693405024805915040259048763547145832e-11,
-3.899173349203442406907873151424022238869e-13,
// [3.4286,3.5714]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00409544476237316730378477833742408603376,
-0.0004702734385414831911950044787953502770824,
0.00002796111898622837893571199982575008034053,
-1.137431681848204272865009369606935383523e-6,
3.539812285506654196879262223686124611989e-8,
-8.951788150064629969881192981207685365156e-10,
1.910658935115159498093747025092084294627e-11,
-3.531083457470399226531037593461923160674e-13,
// [3.4286,3.5714]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003291914069790382194619032571901938094616,
-0.0003914556658068823365053986511601680241126,
0.00002388606531882608941499070508565938066208,
-9.911474525407134212022741867633559583017e-7,
3.133125834122985567545641106629917908789e-8,
-8.023558031046354311143885787871377974515e-10,
1.730289773109909290354432196544611638788e-11,
-3.2253368113498119353995115248066326313e-13,
// [3.4286,3.5714]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00274018966064817622245738355472454690677,
-0.0003344049144632754587502589117129432270029,
0.00002081409650336775762879234496010690380124,
-8.772752451462574380007684056371580399214e-7,
2.808245293831794371227529164065131016126e-8,
-7.266104225857530559047907378149703292112e-10,
1.580469547846282387084529348566361716843e-11,
-2.967535291129374036392240494276253738977e-13,
// [3.4286,3.5714]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002340834401242928087382299154986203126656,
-0.0002913973510468803156495197958465333070295,
0.00001842278014808329773182665405537930975931,
-7.863086930046412359991616474986732041071e-7,
2.543136463194708848671202987281745640128e-8,
-6.636941848097567892626024417264865113287e-10,
1.454143030122627755860698151271000587951e-11,
-2.747352248434156299024111295400856654758e-13,
// [3.4286,3.5714]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002039781457328162093709088841969218577366,
-0.0002579189220729164463180688106978540564784,
0.00001651248255310858641781330981386921921611,
-7.120782196834825959536590018554394961315e-7,
2.322929632006820675278688627715156837581e-8,
-6.106441780390481331688420286307612211175e-10,
1.346250049279993795422941783550801427508e-11,
-2.557200273512125387840007416327385390594e-13,
// [3.4286,3.5714]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00180543245451041501545272139840582962958,
-0.0002311747557432866804171963094298711701893,
0.00001495364261336357684996246467116647039042,
-6.504203063031690874148362236925940294726e-7,
2.137254609213512058969670973122914206083e-8,
-5.653353440834614041425828524522783816809e-10,
1.25307268821351846214756147150965851019e-11,
-2.391386374557933093701195691910714653264e-13,
// [3.4286,3.5714]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001618223290203006660405939474841467574445,
-0.0002093509965868707871175825706236668434113,
0.00001365882643237639227260175532434366054368,
-5.984312993514121958613533675316228686619e-7,
1.978673691170158132695238912664566204306e-8,
-5.262063209167294034415097876957061349625e-10,
1.171821313882402864207119979792626530522e-11,
-2.245559664993069348503708301177918955808e-13,
// [3.4286,3.5714]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001465456976108095412892522280544667967457,
-0.0001912235700530628218804705227122145981566,
0.00001256705728638896149507985746604224377894,
-5.540286417945152498798107468565928633814e-7,
1.841722110801338008813180368094982472557e-8,
-4.920855893986468944143098430128506152244e-10,
1.100363939017312586302328185108665516135e-11,
-2.116340483298005110162852941417073143841e-13,
// [3.4286,3.5714]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001338564990371439661244169961428495060556,
-0.0001759388020092500481362924471047557336834,
0.0000116346014776936445297114842512912025752,
-5.156821988409563523453341225374085114585e-7,
1.722299551129517394469511275102206062585e-8,
-4.62077814755385775132003018988864175623e-10,
1.037044487284700535656862125361574498077e-11,
-2.001065373563585104112588477585905864406e-13,
// [3.4286,3.5714]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001231571614064750249557099338241159519384,
-0.000162884420687525713705359727171739076618,
0.00001082932617566847355052144862763397291978,
-4.822438817287199307216305403309476407677e-7,
1.617272340456946428264903712581369920055e-8,
-4.354875246645210809156744735785363903877e-10,
9.805578312138423494192911608802235238363e-12,
-1.897607991439364146287324439337780791764e-13,
// [3.4286,3.5714]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001140190944812679912641392120001916697516,
-0.0001516105664591824367372838005905856081412,
0.00001012712151631111501562726946404798782725,
-4.528362623757274484910722148115545802669e-7,
1.524206325663829248305959373993566759653e-8,
-4.117666299909581953590593948039834486454e-10,
9.29862034260129359497566492310436096959e-12,
-1.804250928545316097820977677550067648149e-13,
// [3.4286,3.5714]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001061273965214276977600283250224893211625,
-0.0001417797012281876845366160332559702792173,
9.509561509897914286468612954796715912692e-6,
-4.267777779029603908691648954472877673271e-7,
1.441183194784496968058359600094877409038e-8,
-3.904775699526647864670055979714118753933e-10,
8.841155433980448958143768167226852451898e-12,
-1.719592401590399896099722678778986516634e-13,
// [3.4286,3.5714]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0009924579085973137156121454954242189293255,
-0.0001331338611384104049689036802291558784629,
8.96233333596947808728508576058823409156e-6,
-4.035313009555192227699947602370720798007e-7,
1.366671485087790154872560518109660869436e-8,
-3.712669355966932210130804890647057946223e-10,
8.426314657630137212456608322025345391322e-12,
-1.642477266257210529447539329447474856254e-13,
// [3.4286,3.5714]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0009319370279688727617747839134251312776244,
-0.0001254726667034191859444220856912064664269,
8.474157320072912439726787767197179283875e-6,
-3.826680219649284398999263424826650605736e-7,
1.299434265243380832517105894772130546134e-8,
-3.538462679283765906225168706567932409447e-10,
8.0484376467551262903031663988458826339e-12,
-1.571945293879033675149820789905740898009e-13,
// [3.4286,3.5714]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0008783086669239342314930233349164025830964,
-0.0001186382024808735903386599440040745362961,
8.036028461270228595461612173442196503376e-6,
-3.638416001555532910574956773572758219454e-7,
1.238461928774145720007492557476888913171e-8,
-3.379778617144697079157207946494651044435e-10,
7.702819146139542456810679567979393839834e-12,
-1.507191890941217134724079008273934806283e-13,
// [3.4286,3.5714]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0008304674173661150649232408697151850948883,
-0.0001125043984576418419963866706381070180518,
7.640673603273094095447440537016865397818e-6,
-3.467693457111463816859247514576827025703e-7,
1.182922507367305085837285241485790227112e-8,
-3.234641217036367333945088774100764611013e-10,
7.385516533051816318461097624522033160715e-12,
-1.447537915622821248661180250339339105558e-13,
// [3.4286,3.5714]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0007875307892034928290027284845258334131558,
-0.0001069694304456873428509980821181502252589,
7.282156259940311359000431070160611471442e-6,
-3.31218307501874971095477238264046472419e-7,
1.132124417646243488351007516394334231348e-8,
-3.101394793836493585008302964755338060176e-10,
7.093201914649283829545616851697449374859e-12,
-1.392406232649132032393730546686892282136e-13,
// [3.4286,3.5714]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0007487860131198113372858182478748755195203,
-0.0001019501876390333749676957868496847708851,
6.955584457545390857737646675902992874298e-6,
-3.169948421803589303829550153231670467871e-7,
1.085488169820793719108161512432682928919e-8,
-2.978641817621370277081252191311890724564e-10,
6.823047243865811278401169435613373978601e-12,
-1.341303319717253324926810516738902285706e-13,
// [3.4286,3.5714]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0007136513134732335642467014767966519982535,
-0.0000973781824055091264009464902001443958265,
6.656891685793348177798830425800875966186e-6,
-3.039366926036933557102863926281279807617e-7,
1.042524628419934697235273761762695747342e-8,
-2.865194667800217981316741720080934053351e-10,
6.572634188589448288382888193966757975475e-12,
-1.293804703328936873835584250775544389007e-13,
// [3.4286,3.5714]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006816472768385638262820974186435995560887,
-0.00009319648360098485120142657486947364609084,
6.382670544683178855156152026058933945775e-6,
-2.919069008385585448398920218249046127508e-7,
1.00281812623887153233764306365101936172e-8,
-2.760037782531469276263360075668955579745e-10,
6.339882765385555564166663835083724057502e-12,
-1.249543327630388901189875084066405512938e-13,
// [3.4286,3.5714]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006523753852068939017602331409529702292154,
-0.00008935738762544651802471050414235742633972,
6.130044917615167847545336294267676448794e-6,
-2.807890800663658404315570760383849057473e-7,
9.660132166347826650255205746262338325148e-9,
-2.662297688397632349160352712834254678291e-10,
6.122994345305224828237816163570503679888e-12,
-1.208200191316447580385855769075193985129e-13,
// [3.4286,3.5714]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006255017133781255712819627416683870837203,
-0.00008582062884649814347984864720038492076046,
5.896570681398952214862669510935881108701e-6,
-2.704837052260371675111670730197372037072e-7,
9.318041839130605772041389016098671574855e-9,
-2.57121906569238035626637260761829567158e-10,
5.920405773485351739311864203522034406473e-12,
-1.169496754131871107321530179569496329935e-13,
// [3.4286,3.5714]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006007444019254869511210590395617356658196,
-0.0000825519895394746702605083355905948490245,
5.680157809751891382304686443596288247036e-6,
-2.609051759221282088698544560335910337415e-7,
8.999266661778246732847522621931133583154e-9,
-2.486145480978046781576276068601654611954e-10,
5.730752159993582756410520202860361296401e-12,
-1.133188735612135277476831175367979112404e-13,
// [3.4286,3.5714]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005778639267763226401424051706835303019466,
-0.00007952220933641915132670811091187524780521,
5.479008694369653804146857114609488173922e-6,
-2.519794708229521005220370312750277457689e-7,
8.701509117270707936540029883382243785714e-9,
-2.406503761151994797091191225092237482619e-10,
5.55283649282180852566828488492422687298e-12,
-1.099061017755729456652786552487262589048e-13,
// [3.4286,3.5714]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005566554653549340090743041720495669505942,
-0.00007670612172107096360939019068104271474543,
5.291568887286814552056738875369316482895e-6,
-2.436422594512048285291621227179520959255e-7,
8.422763099759222316950907365652307936328e-9,
-2.33179123248471873500844376052968999617e-10,
5.385604660284703350155367043363673197998e-12,
-1.066923429452304688777007988695521975336e-13,
// [3.4286,3.5714]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005369428520474966964406282860586430160767,
-0.00007408196442191417458922304460387769400481,
5.116487448479988634756385433633323520442e-6,
-2.358373708424618169322279512997296476131e-7,
8.161269251199834307234490115100161246182e-9,
-2.261565231342132785123038959959090286789e-10,
5.228124794135836545705894155908172066146e-12,
-1.036607240065376125944193850871816088691e-13,
// [3.4286,3.5714]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005185737509533991746125563912263452503554,
-0.00007163082427862140905823544174347419050243,
4.952584787696259333143151883635302169665e-6,
-2.285155429709055084002653599246110136466e-7,
7.915478248881563109739410252791353108721e-9,
-2.195434429393974304322081783420417334955e-10,
5.079570087633860841642493799238188418307e-12,
-1.007962227058381337216419767771476061913e-13,
// [3.4286,3.5714]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00050141576995034981714011317850958849054,
-0.00006933618702765184605221252128504929464222,
4.7988264023934573842265403217929304035e-6,
-2.216333948001043075262855460232333213919e-7,
7.684020443655943363171871483871542028143e-9,
-2.133051618125920330769760451500720382274e-10,
4.939204426497128816272419408572429995566e-12,
-9.808542111399731001690732022252626864584e-14,
// [3.4286,3.5714]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004853533091935628772790272140726936767174,
-0.00006718356963341512762911812456124465955959,
4.654301290806518782128595911433099767993e-6,
-2.151525761534315871897813317064732245059e-7,
7.465680605729542469271834119957822077393e-9,
-2.074107674621232754983427655975691257117e-10,
4.806370310765140490797468267260666833654e-12,
-9.551629743735263458083653385801877587094e-14,
// [3.4286,3.5714]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004702849874339058494397601818068175697662,
-0.00006516021807120036825560481059923035682853,
4.518204099226283904696242861429473186678e-6,
-2.090390605962492734749192098559198886063e-7,
7.2593768048998120276938843105962425709e-9,
-2.018326489404991463032793164040781619794e-10,
4.680478653236451283600972871079813155866e-12,
-9.307804937039764866699004154570329414008e-14,
// [3.4286,3.5714]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004561215264984025348931393870407452019316,
-0.00006325485738907934272659042519523386786582,
4.389820272525352790793383089976573958411e-6,
-2.032625540825058056497099147795896589005e-7,
7.064142658009983703200905973360446575727e-9,
-1.965460682352517381001019244597795251359e-10,
4.56100012349460819014719064615359927112e-12,
-9.076094356147158128010106899343881806797e-14,
// [3.4286,3.5714]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004427840017235553572074473390338183310489,
-0.00006145748381526846017313598256121683508815,
4.268513635736642294717954714069041806242e-6,
-1.977959978834681888704458821706726539309e-7,
6.879112334628618488435893016740018714235e-9,
-1.91528796766329942730555745538955752027e-10,
4.447457771507398837913812705761763213991e-12,
-8.855618680330684540603657361892915951397e-14,
// [3.4286,3.5714]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004302023867068791803036131368867056188558,
-0.00005975919090022856433234037369550472319947,
4.153715955556759188558518518097671405953e-6,
-1.926151487467450512717407657480213644412e-7,
6.703507834458410191868673350773617944659e-9,
-1.86760805618761885001070922622050260815e-10,
4.339420715774383035726172095390333019856e-12,
-8.645581538204325764404215879337226816354e-14,
// [3.4286,3.5714]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000418314336301599910344515416605754773209,
-0.00005815202337771215707350109077053136997388,
4.044918123685484357246177857778584525239e-6,
-1.876982226637294148607102203999029635976e-7,
6.536628145479421803990458928570314918725e-9,
-1.822240004806284014557450459062689284315e-10,
4.236498721264795326606177497610360522028e-12,
-8.445259967114624976097140867357428448312e-14,
// [3.4286,3.5714]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004070641636439850604181421401004153935649,
-0.00005662885373151617720945813610147265633611,
3.941662675942070983999405212864192372241e-6,
-1.830255912976060890650149929128425376721e-7,
6.37783996677820169408445052981808269874e-9,
-1.779019939474122743054953784649865285007e-10,
4.138337524377330011060896754615378200866e-12,
-8.253996157824073173215312134436943012992e-14
}
,{
// [3.5714,3.7143]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.4610996192870572004388132762009650378363,
-0.008527885783685623267948765838579677119294,
0.0002141570828268022190109943940736573031599,
-5.25273617887361376277233214173987922828e-6,
1.1788875724439607142495771178572397605e-7,
-2.379288563641893591761533957326913567808e-9,
4.311874455202860440216344199900127566153e-11,
-7.043889763553571858054224727434520023969e-13,
// [3.5714,3.7143]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.05969520048579936263437735705363665299871,
-0.002998199159574660287335163224813375366306,
0.0001103074597563690504657824150816466548768,
-3.300885225674410604525537449980823551219e-6,
8.327509941864614928320333956821137820174e-8,
-1.810768089433123615232637446168103299331e-9,
3.45160480675783224655310626770543258234e-11,
-5.844867184490469076442273901057961894688e-13,
// [3.5714,3.7143]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02098739411702262180152644097769837051868,
-0.001544304436588680316603697238522131795665,
0.00006931858973918276547876213706936979832495,
-2.331702803177837200242976772193851529138e-6,
6.337688286158813786851176398158177482273e-8,
-1.449487242736776908334039357834428153316e-9,
2.864070863338353599770278454751599577653e-11,
-4.980714122638098845899972258693431672792e-13,
// [3.5714,3.7143]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01081013105612076203081165978196490331182,
-0.0009704602563481357186754896549089317121765,
0.00004896575886675238105386595226422377056276,
-1.774552737044521719138575418603566302076e-6,
5.073205325845520647495509514728981069963e-8,
-1.202747329235320694308405112463661248099e-9,
2.440625866470577452236655153100420295736e-11,
-4.331572633314145502654611803827099366436e-13,
// [3.5714,3.7143]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006793221794436949864762402572196373877261,
-0.0006855206241341595383200205435898333235349,
0.00003726560747795088891656117589196145391677,
-1.420497506188723247661643178216611635212e-6,
4.209615631079808561545981605222484130727e-8,
-1.024919324165306586669775990824141189394e-9,
2.12253857065844531539004988747386125447e-11,
-3.827741370130483593914531337316235683191e-13,
// [3.5714,3.7143]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004798644368939116618110938753600728919763,
-0.0005217185046909778560800082596374912759084,
0.00002983044762997760067467677103701203830766,
-1.178692390086006068860163080298434243023e-6,
3.587217615361885265365996193644855387778e-8,
-8.913377158422700183459392982682376387279e-10,
1.875654764884492649623816794647383233908e-11,
-3.426248941327283665155568147082890094432e-13,
// [3.5714,3.7143]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003652029532836844855561647138447229015249,
-0.0004176262668193837478231603873947526298932,
0.00002475254019181927935833831149668932613317,
-1.004420944407893521151773234704071092935e-6,
3.119681987912010150857532494341489495532e-8,
-7.876587775764459823120601719675726466746e-10,
1.678918096352769861219405408957009792928e-11,
-3.09931028363386833416499943995318423236e-13,
// [3.5714,3.7143]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002923383867735686108818084549307439398278,
-0.000346535562685193721161588506660558156195,
0.00002109283983257785463217412059406910648488,
-8.735109676630502597841345164658474350459e-7,
2.756805705396595363234548857375668669588e-8,
-7.05039542072396908298344033596560806409e-10,
1.518713626173388001915268318161315791771e-11,
-2.828235250155531164188701078212369088087e-13,
// [3.5714,3.7143]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002425748938796355931615463560243738233111,
-0.0002952997576558360606842150942744545238394,
0.00001834373032093524101482196682758812013977,
-7.719056076673000562119473074047011317623e-7,
2.467638382339264934133014052046723020376e-8,
-6.377622224057004847473234801065776856507e-10,
1.385882997870017779389258191892866916863e-11,
-2.600026226527584288220007723758824266963e-13,
// [3.5714,3.7143]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00206709830359085231640645258642557262187,
-0.0002568122244928584777241519780325947339163,
0.00001621001776102370600646214689116409263202,
-6.90938756450934554156417218765434445182e-7,
2.232167764546808014977009811127880462301e-8,
-5.8198065755918173828625065239416609267e-10,
1.274057245148608951584251248159321329497e-11,
-2.405384445079937082286006099607691248264e-13,
// [3.5714,3.7143]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001797685571450009242770103679111072655542,
-0.0002269402486541133829911699985236929955109,
0.00001450971388547935038730991454621042025418,
-6.250069828132262752402857643484933034921e-7,
2.036932288490762837518903441456029555105e-8,
-5.35020137323336835501499970870721535587e-10,
1.178679870568634768473273901891117407133e-11,
-2.237492980822177572946053606801457204688e-13,
// [3.5714,3.7143]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001588581740578793585863449651743230207725,
-0.000203135994396506685904434089698439776562,
0.00001312514663908687900223355349499223058764,
-5.703410489462659612670660616227943833166e-7,
1.872570468462011648453962655696681788807e-8,
-4.949671241975391854103302303684670273558e-10,
1.096410503618560353094673356981365407158e-11,
-2.091247168963620478893327331704219421848e-13,
// [3.5714,3.7143]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001421951960775546711766935928163427232167,
-0.0001837520529470246345851962703122893399983,
0.00001197716202788018338513068163474468223538,
-5.243197388362888253743187055746822930445e-7,
1.732384923227086721753192317390547870848e-8,
-4.604188086009665481226295363528679550775e-10,
1.024747798629767020128328213056478631799e-11,
-1.962752757679779837432284026205940260211e-13,
// [3.5714,3.7143]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001286264370629172357444446504068133715137,
-0.0001676802683901420054312841137438823486096,
0.00001101071451557019196034187948979792140182,
-4.850677857261269004121722989944074675873e-7,
1.611465819267845648357715273121876828007e-8,
-4.30324738604506069740030730216696972283e-10,
9.617835250548085367219976185231785195701e-12,
-1.848989692499914695619693541402022538125e-13,
// [3.5714,3.7143]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001173761878730993957772556109813340195094,
-0.0001541500032178120284657854746151282769904,
0.00001018642350025636860664144635346291169031,
-4.512104362214169598238178825572229708851e-7,
1.506136574844141571036676487229005194985e-8,
-4.038835464987888092103221700470185989982e-10,
9.060378186089439724565852112251854379859e-12,
-1.747581417977355736176358115506808770514e-13,
// [3.5714,3.7143]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001079050022524684122986919316649406385797,
-0.0001426099290034273830240301856533519774463,
9.475419160657078458508202352645483565769e-6,
-4.217182474275165357075037187834660192499e-7,
1.413592402982660322489204058464864919255e-8,
-3.804737603374722625020896981510625660676e-10,
8.563461367964537371066187701755142385424e-12,
-1.65663317492679279307214889811397898267e-13,
// [3.5714,3.7143]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0009982695030239916084952096687617245137149,
-0.0001326558682490453302529846884794495290663,
8.856083195984823894969743162267003028634e-6,
-3.958058789859270509470401587077582155887e-7,
1.331658151878929871137625763507779026036e-8,
-3.596063295904672363496680918783975342176e-10,
8.117800228909525455628524792448933944482e-12,
-1.57461647548778462334866566785372584554e-13,
// [3.5714,3.7143]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000928591077743317242375884578711600747277,
-0.0001239851647436410251546761114573949129016,
8.311923458711130026104020532504168783561e-6,
-3.728642883865284780550112258833337672284e-7,
1.258622144683998698689558134580257488762e-8,
-3.40891349164346200758658526125041118965e-10,
7.715904974866853311756598714539246906316e-12,
-1.500285138187995780592723252663206047832e-13,
// [3.5714,3.7143]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0008678961532054871096844019057634096898248,
-0.000116366928421815919450229423661562174307,
7.830150056123472312934852675806410074977e-6,
-3.524142061076071629250727546388064682634e-7,
1.193119713576152783733727877339066965319e-8,
-3.240142861782671672288821443976182419711e-10,
7.351669147587126258823557916113965383514e-12,
-1.43261330432105807849497054010966057613e-13,
// [3.5714,3.7143]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0008145684989527113725031850740745076802006,
-0.0001096221007855947527904517718557787520594,
7.400698328265860702545739869329808227059e-6,
-3.340735251557552973042466271305887940169e-7,
1.134049993476867421899467679582716275073e-8,
-3.08718701333574419968567096225481707053e-10,
7.020065897896732995821648981724640164399e-12,
-1.370749029209042604474225330269074362727e-13,
// [3.5714,3.7143]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0007673547054991632084170220319955646637339,
-0.0001036097765955937340847966131131338527215,
7.015544028276728424371957637687435482191e-6,
-3.175340033061999327967548877563754606183e-7,
1.080515446844577286908409870126173004192e-8,
-2.947934937110022168179236782145506470004e-10,
6.71692057752389324419715503636693860761e-12,
-1.313979083080764048763956043365175944513e-13,
// [3.5714,3.7143]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000725268436169156079816829508963800431809,
-0.00009821761639575098728759875430486498909397,
6.668214069435841186809181576839094604036e-6,
-3.025443300449530714925151124177937875698e-7,
1.031777220465019346609377577977224755781e-8,
-2.820633506404269377854654926271107390674e-10,
6.438738259243533305095802112842536761131e-12,
-1.261701936110243503504442529872237868458e-13,
// [3.5714,3.7143]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006875233147702568544040940245213967570371,
-0.00009335499697198328219750318790114221726403,
6.35343093094944900402128134263451949725e-6,
-2.888976264700258056298209081681275649713e-7,
9.872217199954672372241555371450322001998e-9,
-2.703815043329726212948101360886881049216e-10,
6.182571360306409207527895911122016002335e-12,
-1.213406797801646064450005137937486636195e-13,
// [3.5714,3.7143]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006534849788038829207875031913466570423666,
-0.00008894803303317816163697568744518059552961,
6.066850155875783068428953450564858538018e-6,
-2.764220861637497739231911326988124863241e-7,
9.463352581771812106014135372811109770119e-9,
-2.596241726810539330984422849102174407504e-10,
5.94591693284704463575227459500861622593e-12,
-1.168657189745723011732397111202127743404e-13,
// [3.5714,3.7143]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006226362312322470787400221137196197839434,
-0.00008493590218215089893612342911872065384773,
5.804863809443806285380278600690198202418e-6,
-2.649738766922123872808175422125341405094e-7,
9.086845976356228112623416700804629587652e-9,
-2.496862459428029584855183932143708641156e-10,
5.726636168333147386767800323216216431526e-12,
-1.127077951089770376017054870012789337328e-13,
// [3.5714,3.7143]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005945513152750562415860416866301741906712,
-0.0000812680933321070064305817460614241274553,
5.564451410541352974437951235058148482719e-6,
-2.544316915892764381574034165208832141312e-7,
8.73901854276000354808302082154929243106e-9,
-2.404779063177484808789400673037236890486e-10,
5.522890722713822036395393820164496230095e-12,
-1.088344870404708829084565473304011123509e-13,
// [3.5714,3.7143]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005688766533247489956869132139711592328089,
-0.0000779023197474761920938972403925848983582,
5.343065523379540634364324643760308302311e-6,
-2.446925233073002598186847799808913884314e-7,
8.416726657981883025296416556557997464217e-9,
-2.319219539157593018137554913217150746372e-10,
5.333091911227474709182585618896448780126e-12,
-1.052176346502019936243073111142918562523e-13,
// [3.5714,3.7143]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005453162382323332968678973103876149680545,
-0.00007480291732721412490723373126618284462961,
5.138542989457893262032956742060353338018e-6,
-2.356683504012887544437102284427636969534e-7,
8.117268325880630726106535581911237671035e-9,
-2.239516731923865312400676737458997043177e-10,
5.155859845310678014959003027014188609446e-12,
-1.018326630804616220135594648924205076046e-13,
// [3.5714,3.7143]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005236204212904988280063249454680648182056,
-0.00007193960185231416186264482189348279741013,
4.949035358431512921513253177199919513192e-6,
-2.272835169784458796801215509492823355683e-7,
7.838308502412291405678824100333960348273e-9,
-2.165091169058264315080548139157141911093e-10,
4.989990319316732829676822426462108289016e-12,
-9.865803129153798417116770722723337264989e-14,
// [3.5714,3.7143]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000503577212966199088054686742918948855567,
-0.00006928649501794775037258640998997317111534,
4.772953856551681944323608709989848395802e-6,
-2.194726418048002635460190400449585381157e-7,
7.577819034124122035893533898633500747179e-9,
-2.095437155289557041495692444411584438175e-10,
4.834427789058774102480663348143234192519e-12,
-9.567477911173433230221090415755635434673e-14,
// [3.5714,3.7143]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004850054651256342089073712824497157523436,
-0.00006682135399163285943902312507665586322413,
4.608925477905000827900559203244907679025e-6,
-2.121789365830206678458477791883770266738e-7,
7.334029987576019776766611748172169786956e-9,
-2.03011142488519530106517928772644975652e-10,
4.688243176643266384366710110709040706418e-12,
-9.286615289588351113141250726611070112502e-14,
// [3.5714,3.7143]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000467749477941429973584536136245545066227,
-0.00006452495669058191053586659153674983486793,
4.45575766824751295523870902006950583153e-6,
-2.053528431762038164801112656299125257512e-7,
7.105389932712264633969300517770825301086e-9,
-1.968723820848258684617421478102582763393e-10,
4.550615527221217759529774503934109129569e-12,
-9.021729435768513392800399691333877723259e-14,
// [3.5714,3.7143]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004516746968340733324092815393704763260769,
-0.0000623806073553795239379015575411515230984,
4.312409706704248979235090341370561807272e-6,
-1.989509215422730310986373258643545927716e-7,
6.890533320050955376436441358096134716525e-9,
-1.910929591728209688750639768113389448312e-10,
4.420816761339222262876552290474182345046e-12,
-8.771498050342814992467414611399128331379e-14,
// [3.5714,3.7143]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004366642514876566273102226291298687832793,
-0.00006037373589377614031495889073793683187136,
4.177969352391598162831370587760677326115e-6,
-1.929349362952739026543917497520045318396e-7,
6.688253519521766823606130572559683602989e-9,
-1.856422988427612775194796766890334116054e-10,
4.298198931325938740887370256553163540805e-12,
-8.534740515625358344425472994225786252409e-14,
// [3.5714,3.7143]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004226161512564329429807761768241115035253,
-0.00005849157093340121967401022740964107850634,
4.051633662204517474370763007459694401538e-6,
-1.872711017928242912447551263168441371842e-7,
6.497480409289563529108614336084208583117e-9,
-1.804931912576062133917703912977883156252e-10,
4.182183515670641593764590298105980212359e-12,
-8.310399452708746825568947394536643035018e-14,
// [3.5714,3.7143]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004094409965338084994738876617868649900534,
-0.00005672287127078416884690126765846017152504,
3.932693137652981578069939873595949107579e-6,
-1.819294546231693997487815794466108477582e-7,
6.317261645063229550520060440716268203785e-9,
-1.756213420755688399488688783358119557304e-10,
4.072252381734617685059061968825717102547e-12,
-8.097525080999469795630694284992568752107e-14,
// [3.5714,3.7143]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003970600988954891446162195282990590020413,
-0.00005505770392706464148705134452948864904188,
3.820518547090139375103143913739366599923e-6,
-1.768833291458238036310193648234090450239e-7,
6.146746924885012601110849919155050833703e-9,
-1.710049929339507710771887311386423637873e-10,
3.967940121697442300986073292854321769674e-12,
-7.89526189650881675374943414549902007189e-14,
// [3.5714,3.7143]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003854039274894524539850955580555692775294,
-0.00005348725965918672975589564036923404048169,
3.714549912065796624933095199465647173648e-6,
-1.721089169056686365715360905341735904905e-7,
5.985174706064806416604668402993659462214e-9,
-1.666245996016119121917621830332487001466e-10,
3.868827524725421687373098884384895634902e-12,
-7.702837278172831459924157812456027130399e-14,
// [3.5714,3.7143]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003744108176143070727136372264418800523991,
-0.0000520036987688477211170536300957397883685,
3.614287255022456839620694078272824332999e-6,
-1.675848947071076752770409898144376555698e-7,
5.831860940516643971474179213226078964394e-9,
-1.624625578467312257205999497209773018324e-10,
3.774535993901081235241918032714758084625e-12,
-7.519551704836813124002311397855781574872e-14,
// [3.5714,3.7143]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003640258913819340130499781302651070203442,
-0.0000506000215702426709390022033389227023611,
3.519282788852599061467389389320506151794e-6,
-1.632921092034858560213946486700479493377e-7,
5.686189480130369487005960634804449337223e-9,
-1.585029689794631432459291585505447303885e-10,
3.684722752400879250366085828953823339234e-12,
-7.344770323783192739725773185162939934054e-14,
// [3.5714,3.7143]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003542001509916986625760064997171621962482,
-0.00004926995904386629145328393988254267935864,
3.429134293276466707422799611339960425138e-6,
-1.592133082474932422452567594267837959713e-7,
5.547603870764652097456063484587308476825e-9,
-1.547314385387484917262361679311366616899e-10,
3.599076711948294256373496316367667139176e-12,
-7.177915658186061610953005056410161005905e-14
}
,{
// [3.7143,3.8571]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.4527807534019084552167044423155395533947,
-0.008114869914146189319722943106940102013854,
0.0001990830461522053130942468538041451549213,
-4.804135840226548691770526655758819569008e-6,
1.066150793202620512007605609686668611863e-7,
-2.134831022339996038162388637916904593194e-9,
3.846863269654497669018869276686067177772e-11,
-6.257692365350409907764607606742255210243e-13,
// [3.7143,3.8571]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.05680408939902332502495696464232851259061,
-0.002787162646130368948439565761157920336531,
0.0001008868526447779805671610023619127633506,
-2.985222241184880591823505284661968861944e-6,
7.471908550912526838917908428195108616065e-8,
-1.615488483893604520479754683769724314723e-9,
3.066356547046441055815954791036412370832e-11,
-5.17573433842195476290516229193563030073e-13,
// [3.7143,3.8571]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0195101385229125824541221064273372219009,
-0.001412415937026462109524482040511751121263,
0.00006268966706490024817544892666751379166682,
-2.092134411440374682436255718601434253459e-6,
5.654209669953221336098677013477236620592e-8,
-1.287704774199257588527933921494034475993e-9,
2.536185584027661883402176259579169150223e-11,
-4.399364186453131105997623586124381352087e-13,
// [3.7143,3.8571]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00988691155918523460349909448749641589926,
-0.0008776553389082306041448587679440784830316,
0.00004393482264026353294337195705180920573622,
-1.583178722501831116263074585426711251742e-6,
4.506966688811193265155241380827671980822e-8,
-1.065054761225064366821030643006028877035e-9,
2.155755287353762964229249152809240225064e-11,
-3.818255999362918933437680114987010144084e-13,
// [3.7143,3.8571]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006143587372357614083156300554809538327054,
-0.0006150875169633605046930780197549237902003,
0.00003324675317255245584863275743560567703392,
-1.261950686025500109133702987359760215133e-6,
3.727691645617795643690519208515955543362e-8,
-9.052908997002754224145848092092341518602e-10,
1.871005183576803950361875638410029005068e-11,
-3.368572540637100749570789496876970629705e-13,
// [3.7143,3.8571]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004305612618743523401063183589649914171569,
-0.0004654545444154403316436365421130578430891,
0.00002650096440654815403603454832442821834025,
-1.043753672535088383724952332556007853447e-6,
3.16851813208192253799328860238720100279e-8,
-7.857092602733270360739004573283142609366e-10,
1.650654525949617138868656440331805989809e-11,
-3.011127155088364092700404082787179111959e-13,
// [3.7143,3.8571]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003258181810908082201361067738434473505306,
-0.0003710135016914084701623510099567252733291,
0.00002191882712324838997625993868691190839327,
-8.871850876104802809296470896902548585055e-7,
2.749982395578041818623807294897618770117e-8,
-6.931728759308971450961998174095746736056e-10,
1.475501517619192833909699228705545714497e-11,
-2.720676574250285355401912266676076977751e-13,
// [3.7143,3.8571]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002597094511839859180779178651453168476708,
-0.0003068635797252352476538632739075542525345,
0.00001863088683983068025103621234344582616127,
-7.699950804504138765396301626726779329317e-7,
2.426105051632295760958281309816298703009e-8,
-6.196176266809072823179607319070217653762e-10,
1.333176724174318116451984433459594525152e-11,
-2.480295793608719104686816848633057647291e-13,
// [3.7143,3.8571]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00214804505807664663155570176555147695181,
-0.0002608324157574070423790471999910979052397,
0.00001616989668946848550539032424216577794545,
-6.793094233563641025919227233567891480672e-7,
2.168661680324338630994239071094379623163e-8,
-5.598487901832559962868446694830143780074e-10,
1.215386727230367651031781066159882279275e-11,
-2.278248448011538832356648535029389700402e-13,
// [3.7143,3.8571]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00182582691030184920181349677642169372715,
-0.000226378553652353121246340842639925976147,
0.00001426549789049275082471291254434636432462,
-6.072252787179188537319758102093910050579e-7,
1.959470753501798606248835397543903579938e-8,
-5.103834447844144570385534496194654110996e-10,
1.116380586316214718050251367399545698196e-11,
-2.106159368796954463715506399368238854486e-13,
// [3.7143,3.8571]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00158464987556647176013338957218260228064,
-0.0001997169704667073136858974627095735461659,
0.00001275173085308480070757299753553670718247,
-5.486518186284844971967710679598019629523e-7,
1.7863420454057101539993652665040293902e-8,
-4.68806425210789743510321195093290031828e-10,
1.032054377954267096855293767936074324789e-11,
-1.957902558688129904936820655118882601983e-13,
// [3.7143,3.8571]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001398018793266951112695322560602930795351,
-0.0001785242319430086097460382348612304044349,
0.00001152168819120615265441016272517823453401,
-5.001757798576678450316181361017819551105e-7,
1.640822477600113478978575761139581320766e-8,
-4.333942552777149512362422662637763264279e-10,
9.594062943094064653916851874051078696259e-12,
-1.828899980352054924914718184935989843723e-13,
// [3.7143,3.8571]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001249669623601060189968290750637430361045,
-0.000161303634676718594874928143812547064039,
0.00001050369137701853716643889672971969729279,
-4.594303004297823919414792857504410444753e-7,
1.5168798834554101013289458888434753483e-8,
-4.028863064254875418027553180341958439154e-10,
8.961930435341856371271856778999420194927e-12,
-1.715665462512084759493153837222507812051e-13,
// [3.7143,3.8571]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001129125442737030090192184242468703062267,
-0.0001470516792781017596758348671748147337609,
9.648036309032527769846839476135514547604e-6,
-4.247263736779970475355215609725527901361e-7,
1.410102063025791321371276295355557271216e-8,
-3.76340497296352392129053127803213251248e-10,
8.407063596223948172220038342530427469688e-12,
-1.615499845919175816455341101559576637798e-13,
// [3.7143,3.8571]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001029361754946712247671673931872306589838,
-0.0001350725083263063406301621144889931399708,
8.919253847244663713869987368495626831154e-6,
-3.948285836092007321110120478066191886201e-7,
1.317191731569584307569069323913570862198e-8,
-3.5303943570061922271140635553874728182e-10,
7.916236210376340754674961871758200539974e-12,
-1.526282246337237100377152748076115861206e-13,
// [3.7143,3.8571]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0009455075582841443178424894412401082443637,
-0.0001248695538612840521307611779599596690711,
8.291400255799605997975446193787263536793e-6,
-3.688136904891288680078571106653831766471e-7,
1.235638016431308801982029672151823991653e-8,
-3.324276839166492224526074118467370235689e-10,
7.479055675098718667462191271021673794208e-12,
-1.446324027887849559287475911282927188861e-13,
// [3.7143,3.8571]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0008740868770289883015082604358596242127222,
-0.0001160796035810602810323424678709912296574,
7.745087500277793338901362882563165703276e-6,
-3.459786499689324608573171236313503158076e-7,
1.163496885592098848590303200909910152261e-8,
-3.140688036506677129047921056493400257796e-10,
7.087247454750230468376051833664567737905e-12,
-1.374264662571518597157239612071635849842e-13,
// [3.7143,3.8571]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0008125572250674219066960954838573830725222,
-0.0001084312250037612775695425538499203538466,
7.265551649353392617860328337358161523907e-6,
-3.257791330790010015790754906887798490226e-7,
1.099240805029392478300126256039137720676e-8,
-2.976153059551438073007851082312298976461e-10,
6.734144781352086535343772914170222954849e-12,
-1.30899616322047261446468299415212237191e-13,
// [3.7143,3.8571]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000759018575026328885085068674451602664444,
-0.0001017177230908254670571795726465232184767,
6.841361794664579628813553494954104037061e-6,
-3.077874302894580006496356788982331525188e-7,
1.04165356343151873510625783187751666392e-8,
-2.827872207428516609575399502769158944971e-10,
6.414318367793235101138911655212485202701e-12,
-1.24960738329250970722747088093641189571e-13,
// [3.7143,3.8571]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0007120240616357782139095355544127890230788,
-0.00009577906512518738443303839117749302244507,
6.463536036083945151410827927263978646784e-6,
-2.916630024300826766227808577758005621656e-7,
9.897552654971076480803759688860691668203e-9,
-2.693565463032313903687259567908959286376e-10,
6.123303470561144011663839497586139993458e-12,
-1.195342370822743704181549776082332569032e-13,
// [3.7143,3.8571]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006704534558763116377598006452722870612831,
-0.00009048950450506336235965163000249448694403,
6.124923051036850297467690884497106811686e-6,
-2.771314788140215226959366927308462847811e-7,
9.42747905242503385153446935704025266707e-9,
-2.571357871200522228993063043107543029971e-10,
5.857395819307366890673007666324243834215e-12,
-1.145568822459979074933087400284373836427e-13,
// [3.7143,3.8571]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006334265315354434852954924459208973287355,
-0.00008574892271440850843648032095641039562964,
5.81976105509936932170626378414317728016e-6,
-2.639694177637695291901216444337042319121e-7,
8.999752483637014810117391954939084078164e-9,
-2.459693838199914741937321218403686287383e-10,
5.613497037907247974676778909475097334547e-12,
-1.099753901392103319165321737532592132949e-13,
// [3.7143,3.8571]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006002424590008595097315387809219164646939,
-0.00008147665477128790638049261278972453909193,
5.543357773043895225602448114785344832716e-6,
-2.519930736724393986294641505106179145244e-7,
8.60892837056466233902708375724473359605e-9,
-2.357272215599488481188612339417982763954e-10,
5.388996149383112496412267612382566861281e-12,
-1.057445495779393790819577080435315368752e-13,
// [3.7143,3.8571]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005703365833990152971027794126480517702933,
-0.00007760700882251509591289030494700701721717,
5.291854547125793220135011184266740384975e-6,
-2.410499983533371018581380858929284876222e-7,
8.250452693720026499621791731303357811567e-9,
-2.262996537840466169173015351839256017798e-10,
5.181677739926841842590666711516946580807e-12,
-1.01825754601261137502661329406742298252e-13,
// [3.7143,3.8571]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005432490617576056254714995151069654543965,
-0.00007408596365966522236993138992285430436238,
5.062049965424487361258280528265414320468e-6,
-2.31012679259504728750454341287762020141e-7,
7.92048782366514384481111870550679015049e-9,
-2.175936455473991546291802206981129715691e-10,
4.989650060630586441428339395463868832367e-12,
-9.818584494411960967689236165201827018308e-14,
// [3.7143,3.8571]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005186017456176565122035472781694190132984,
-0.00007086869951585025050218897373603590529917,
4.851266264453860380272392131432553343558e-6,
-2.217736627655606058845417376889540678172e-7,
7.615777537344431645541375778706538600645e-9,
-2.095297541354208277136542729686926538891e-10,
4.811288209181140473600542624808344910222e-12,
-9.479618171995394826449442071830134006242e-14,
// [3.7143,3.8571]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000496080896610951710563437682474423737148,
-0.00006791772770226456282510930748765479668045,
4.657246918080896127002648841574015208608e-6,
-2.132417746249773197116374870009758231471e-7,
7.333541339760836969293843290558645433126e-9,
-2.020397429735366032508371932342538209681e-10,
4.645188837115373007488225243319316611255e-12,
-9.163190462672750447252825820534388971436e-14,
// [3.7143,3.8571]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004754240939158518981684962732619734195593,
-0.00006520145685304595441084905533162103708544,
4.478077267128518033549449556981998294642e-6,
-2.053391609769903975480470861183606221827e-7,
7.071390950816012036809039195904167011954e-9,
-1.950646795574459324454865338149318077303e-10,
4.490133751942626938555848874433903031731e-12,
-8.867133051629075902005133103512082410481e-14,
// [3.7143,3.8571]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004564101979713216405319150183133752415118,
-0.00006269308173971537185968246504590355318647,
4.312122380520671396401086435253711522904e-6,
-1.979989499781040611081205622457235582069e-7,
6.827263732869799723452304617776160803247e-9,
-1.885534069261663014782836597127538073453e-10,
4.345060446244849379072781702100286974995e-12,
-8.589546298535647495298801612326614642282e-14,
// [3.7143,3.8571]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004388515721780075638627635348399407957367,
-0.00006036971332720806564266575341749882775169,
4.157977949543944185167026423949010650668e-6,
-1.911633877737410970264952067337510759381e-7,
6.599369192296963962434382939863521342251e-9,
-1.824613060351851463138878361432206796584e-10,
4.209038066974778122171315200178351612723e-12,
-8.328758985035928192882807236511151279659e-14,
// [3.7143,3.8571]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004225879932904564214647735572154226918562,
-0.00005821169129353628174854636950924339251179,
4.01443114325221431070303284411708019407e-6,
-1.84782340541818322326012732872083787725e-7,
6.386145662547677989855372558476874445062e-9,
-1.767492865916481499486515217845051659565e-10,
4.081247691176134742111150688535648798551e-12,
-8.083295070779066249945913708817673419176e-14,
// [3.7143,3.8571]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004074818390547539352648155126584507957235,
-0.00005620203600545432370719594554520507079144,
3.880429151381734389211031437112152590749e-6,
-1.788120816184294679539545015764371422439e-7,
6.186224983379257767317396529080107160349e-9,
-1.713829587381661528800207937852604602033e-10,
3.960966035982377553897876269325193191138e-12,
-7.851846077648339974920458636062012098539e-14,
// [3.7143,3.8571]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003934142520381802299770229866844248944662,
-0.00005432600811926973951226249560257216162116,
3.755053713990472287494324367730742472109e-6,
-1.732143025163247043889183528984405787284e-7,
5.998403509789523924995470072439743629071e-9,
-1.663319489592613549058909323519666870645e-10,
3.847551926504393656385220647596199443936e-12,
-7.633248023396639233608873362333748627351e-14,
// [3.7143,3.8571]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003802820568348881415614372601542041298744,
-0.0000525707519957940894438634778794486859498,
3.637500352846181153103936175236125286086e-6,
-1.679553011750372111904870130605622621216e-7,
5.821618168742520132436045051417773436758e-9,
-1.615693318052733450374447836500995036698e-10,
3.740434992987699299293223234935092488864e-12,
-7.426462055457281704056338245848595777818e-14,
// [3.7143,3.8571]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003679952639705585919829258293480394780026,
-0.00005092500493977592665019390740168964574385,
3.527061324679057368441540009390956507641e-6,
-1.630053115491969089651241116440132065874e-7,
5.654926569505310013861224182318499655937e-9,
-1.57071155234233007362569077490727372446e-10,
3.639106181104960648756826025701109536745e-12,
-7.230558111824197511662108351402165663249e-14,
// [3.7143,3.8571]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003564750345784314532824963047867092388533,
-0.00004937885854543800864020951307021803641415,
3.423111542536328916560702835086273911641e-6,
-1.583379466979553632680460446637360868644e-7,
5.497490390613636954523649142095938363261e-9,
-1.528160420960820189641560740129826759356e-10,
3.543109745553360227507045074316860784083e-12,
-7.044701072037446467896438656070181607026e-14,
// [3.7143,3.8571]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003456520098180660280261273314868877817539,
-0.00004792356159544153451997805598121090979816,
3.325096880660178358419365748577740023485e-6,
-1.539297336200196537922208390184792682534e-7,
5.34856143181966901905855691483192142075e-9,
-1.487848539079741887402790910300210752475e-10,
3.452036463837528142174035452388536434916e-12,
-6.868138967289466832898886843880924503431e-14,
// [3.7143,3.8571]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003354649311680907099592697172812594955833,
-0.00004655135632917706677263736507723268784884,
3.232524406023454081826395196139514104971e-6,
-1.497597227081852951209005264843211374764e-7,
5.207469846227599442661339313323773759279e-9,
-1.449604058709535824818377991219471920125e-10,
3.365517859049979771996699835663931323423e-12,
-6.700192901719598588207076312019625263104e-14,
// [3.7143,3.8571]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000325859494304239436466588777988754783038,
-0.00004525534168426448882811789678597564612602,
3.144954176874861633779568979187623803862e-6,
-1.458091582491296716219245874639798260692e-7,
5.073614165877427211510320390536965993646e-9,
-1.413272242591701087908980100107753207638e-10,
3.283221261157583258687518133986628139755e-12,
-6.54024840246285150397212435450434611486e-14,
// [3.7143,3.8571]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003167873917898513915600534615451883149604,
-0.00004402935847618568378864353960119412169457,
3.061992323234625850250096151378734634446e-6,
-1.420611991397549829178798814865977645097e-7,
4.946452810367542514712516671626118570241e-9,
-1.378713390218547963707767915006607403543e-10,
3.204845568397632263177589756290741735649e-12,
-6.387747967985445895109375066360233132943e-14,
// [3.7143,3.8571]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003082055093332997569574690521328917777037,
-0.00004286789252522380430862773229921078132958,
2.983285181937692708457046052996619399568e-6,
-1.385006811286181200081463271892243126757e-7,
4.825496827923896101941624827174377291155e-9,
-1.345801057861136818274514068809250604698e-10,
3.130117595851372788701886855572378735177e-12,
-6.242184625706629375474824207411058916214e-14
}
,{
// [3.8571,4.]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.4448602669161706423865397846323453158936,
-0.007730700216720573751018156286328779541343,
0.0001852895448761267358800703010886635495654,
-4.398275942656419497931958549795390945712e-6,
9.6496616184706461631089863936848099422e-8,
-1.916684607298471167954177338309948071998e-9,
3.433675946824298810924206344097629005682e-11,
-5.561394889594158458643338936805946858338e-13,
// [3.8571,4.]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.05411490151704401606884920039639173892518,
-0.002594053628265326591570223927094010557494,
0.00009236379479580288421429498512427045004874,
-2.701905271080341411642629857856462140569e-6,
6.708396101444908172233825694961746250547e-8,
-1.441971974650620050020207768496888346006e-9,
2.725160615208495741229551645276182154571e-11,
-4.584629652237474219783775952408105262049e-13,
// [3.8571,4.]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01815837539785728597792070095595757136316,
-0.001293093127140860809460962283163182008292,
0.0000567400106927028245190381645471468943811,
-1.878350923587470752887318512623204995054e-6,
5.04690189040394703283260577019604332433e-8,
-1.144421701846156415233311828041451259421e-9,
2.246535324032199292255852314158426466423e-11,
-3.886855111047811667242249701851331532321e-13,
// [3.8571,4.]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.009051651889986025522603006364342681072731,
-0.0007943601496975107912341184333178614265108,
0.00003944536939535067375307444690356400938228,
-1.413132542463289028789745524253399538278e-6,
4.00547593807757268062171046924024564142e-8,
-9.43418593672064895933741733930545534973e-10,
1.904617833242606572664219073565194529387e-11,
-3.366477075261773591236664182680058909068e-13,
// [3.8571,4.]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005560521047882575410433963092267484164306,
-0.000552235171534619886052300920185378622728,
0.00002967578339174137733036439216941393830475,
-1.121533274243672541775586856586204272708e-6,
3.301965061441879176235116888683657448353e-8,
-7.998283026284619010220246315671668672024e-10,
1.649626280091835704878852798527129106359e-11,
-2.965006842007215846873837825933026151024e-13,
// [3.8571,4.]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003865646200742339086663103399008341124093,
-0.0004154609674841208206574179265960214818627,
0.00002355219875912823091940516041092927054966,
-9.245502275422891763073319067597489326143e-7,
2.799399044389517435504815684996454711688e-8,
-6.927437868941834473679915907015437160528e-10,
1.452900744992897329359374018010183361008e-11,
-2.646696793622488888302192043734273640112e-13,
// [3.8571,4.]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002908226772388845639226415222369529610205,
-0.0003297307826275619747383787633766974426588,
0.00001941555477839818880116244108330569775528,
-7.838317417594675800767345649474335333763e-7,
2.424603240641470675663035205787042520435e-8,
-6.101287405359669107518496109033457101848e-10,
1.296924591108329793836590164764991068587e-11,
-2.388605726087280505423287201638272990837e-13,
// [3.8571,4.]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002308115478392933726459739693036010071495,
-0.0002718177668973622053754321462988569717698,
0.00001646046657695810328332294280752236797068,
-6.788889158771968923610853292774176089535e-7,
2.135450579497044950867607063634501309523e-8,
-5.446267509911005277550205066330687878668e-10,
1.170456418147162186878603781993711894465e-11,
-2.175402696501414327289706682134417300896e-13,
// [3.8571,4.]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001902724368281535348290301515012565029708,
-0.0002304465320772184800461418106271046930481,
0.00001425666723342971120416446487321474936564,
-5.979261700578757969540700255555477942769e-7,
1.90619361703353123812764310181344415814e-8,
-4.915168276460512965015579496848255164403e-10,
1.065983914385242414158601777560030192947e-11,
-1.996487447525451124160609689194319739519e-13,
// [3.8571,4.]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001613125724540529277327770910146555764422,
-0.0001995933412678358513033316086909857458903,
0.00001255644957122335931775905478904871085875,
-5.337342199736729308960619037128070954599e-7,
1.720308886137704892008387143034378375846e-8,
-4.476440825169244228110692886019291179767e-10,
9.783128444753460074906670950323263032765e-12,
-1.844314886450921998271635414366491639057e-13,
// [3.8571,4.]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001397153388874850881640835744266601747115,
-0.0001757902939969597114211199388639991585462,
0.00001120841861945456990548314725864931166581,
-4.816864948113765881158816326563408734841e-7,
1.566754278891396146789714498139477974928e-8,
-4.108271432431911327349486790321896309974e-10,
9.037460315117676722967552889902823509936e-12,
-1.713378723718708815657682612333956157881e-13,
// [3.8571,4.]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001230532057978717907299525965516610176146,
-0.0001569178606722077733643728024932135032905,
0.00001011541639104588262447962494392565301591,
-4.386912043378581168768147631514580990514e-7,
1.437894992052108150949504674545408347031e-8,
-3.79513349519780012032422205610260731139e-10,
8.395853316781090661765306849620889942437e-12,
-1.599572405935906068521444668637765173588e-13,
// [3.8571,4.]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001098425024705454345174777463086973636695,
-0.0001416158294744958971554617694666721331921,
9.212515291101584568344447734157882369201e-6,
-4.026106036330254305470369894672236309003e-7,
1.328296714567050897824514421643457566718e-8,
-3.525695979967680186266533733826441193298e-10,
7.838184859399590595016243938027963686138e-12,
-1.499774511779554792072208112378312675647e-13,
// [3.8571,4.]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000991310806321471215514888281241702507755,
-0.0001289752140752843377231346802224849881186,
8.454822676299733114859366491120149347138e-6,
-3.719230855926725327578394222545675475574e-7,
1.233993584723231032854242173350245633912e-8,
-3.291508303581030008637246510707623832065e-10,
7.349159602896753623244978532345109408191e-12,
-1.411572276791999740167817565139538852585e-13,
// [3.8571,4.]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0009028264985269903028940024816494948690886,
-0.0001183675174680660832181323766988054275454,
7.810384797451995340973786245594722822299e-6,
-3.455182089297667654438215524054453299417e-7,
1.152027898423798456777838068660542803081e-8,
-3.08614713308255953264661680463804436419e-10,
6.916954702791806811115036515625759254348e-12,
-1.333072690431398395376585925335559249376e-13,
// [3.8571,4.]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000828572622276462524425975472250426645074,
-0.0001093453871642046197030844474080727473503,
7.255882387530679795019095938330413710575e-6,
-3.225678164913106491157259836356910984208e-7,
1.080151489141911394809273479664853027234e-8,
-2.904647438220150461757720495886119755117e-10,
6.532294167117528923264207669865320499896e-12,
-1.262770601893223224698637672908957602024e-13,
// [3.8571,4.]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0007654177101494322825969391017093641488199,
-0.0001015823534253123851327277178919301908192,
6.773924146322834826128812450793590679493e-6,
-3.024424216450573315745659170096169223662e-7,
1.016626596295437247066842429598669641778e-8,
-2.743113756560482016770010064113907097633e-10,
6.187802561449701372732926438176425821215e-12,
-1.19945482625502348810830690098800084643e-13,
// [3.8571,4.]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0007110764739771866431292560536302026017999,
-0.00009483493804840815260828065697058046009524,
6.351290854551272774438236588954862743095e-6,
-2.846554514241611248878574925841457375748e-7,
9.600898080377320008905282362426512253796e-9,
-2.598448775108802861082448946510650177876e-10,
5.877544919083735652405331669887891985469e-12,
-1.142140124559781153323990969010068467723e-13,
// [3.8571,4.]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006638445663388570177639512460093370483634,
-0.00008891807196361137392897244805266423349926,
5.977764479912231073771400623368059179596e-6,
-2.688251505084001903872518002699746377976e-7,
9.094570648247921570598835509091682207001e-9,
-2.468160111356028066995153128657925688611e-10,
5.596693436027031330973276012479955803414e-12,
-1.090017142579815160282496062302429475464e-13,
// [3.8571,4.]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006224265037452795691227850008383726136124,
-0.00008368870271866943867975009528004414047148,
5.645328160681048505102527052295355810454e-6,
-2.546479822228330735769860447183804331695e-7,
8.638560327819139867790347324476334369079e-9,
-2.350220339192008691183787760203134834289e-10,
5.341282165331898295585360816419027177496e-12,
-1.042415033436598491937594040623227421552e-13,
// [3.8571,4.]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005858209190306860243232514590026700609062,
-0.00007903459424943714454100182040987339078044,
5.347607626683952328960648724672030147173e-6,
-2.418796930803528449212627225901581738809e-7,
8.225771127734721138478744477117530931236e-9,
-2.242963971128712850369193965037261882536e-10,
5.108023863637270202868828822778769607659e-12,
-9.987731819116293997824313693688037950348e-14,
// [3.8571,4.]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005532421597460599671474541816346793400048,
-0.00007486650677348171925635003661797577920716,
5.079473554691695164029267411396899528465e-6,
-2.303215953211405372125799689302904722652e-7,
7.850373841811368051084859924035604205689e-9,
-2.145010541966079279731137935375440355555e-10,
4.894171436966721936906378755211127844616e-12,
-9.586195554940891110432609796843086646806e-14,
// [3.8571,4.]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005240655474143719918172918715295667702951,
-0.0000711126297655937385696866537846452564426,
4.836753501748077110930208866191046323683e-6,
-2.198104711705005208923263720981146511458e-7,
7.507536841870039841773211990169515414444e-9,
-2.05520642231194675167442687910256055391e-10,
4.697411858261403386805638457871821994302e-12,
-9.215539449663948143968781783996559717275e-14,
// [3.8571,4.]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004977884083591561285542067542923750354213,
-0.00006771454902438643723811873897879367041202,
4.616019894584488585836497474270860251747e-6,
-2.102110350380857087088502589899021205136e-7,
7.193222425056346876446775371081718577303e-9,
-1.972580268861028647526121715239707657851e-10,
4.515784044196137829313851721103315278896e-12,
-8.872348573189081409634115706822848574948e-14,
// [3.8571,4.]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00047400184317070502067020339864565224333,
-0.00006462427852409930971305987480727396499294,
4.414431735803639579968762731282927666955e-6,
-2.014102312428281827523676864589169817587e-7,
6.904030889817472626821980820659201858783e-9,
-1.896308536538077979638006922296196941139e-10,
4.34761462882637059128586378409121274671e-12,
-8.553691679754729969886119673793400643655e-14,
// [3.8571,4.]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004523699496686951293358087458087095978397,
-0.00006180204430117032057796319533814609473919,
4.229614856103102796558179806764730838299e-6,
-1.933128681402608182438079664351775897032e-7,
6.637079828403661517091526054800696946385e-9,
-1.825688506520007066640429371688226634404e-10,
4.191467258180440649558881808118413975087e-12,
-8.25703879698161374363009395328175492168e-14,
// [3.8571,4.]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004326143101081922066444626700479657639522,
-0.00005921460798536550911197467761635878858533,
4.059570230949067725518456724140592648183e-6,
-1.858382383125330961874058100115702878951e-7,
6.389909724945969705002376629067625810641e-9,
-1.760116992445969603604318877696225451801e-10,
4.046102207830696849807528393147994334457e-12,
-7.980195056920067686443435957215685069682e-14,
// [3.8571,4.]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004145022558975585275583022977060374442854,
-0.00005683398323321154685810965789394239047504,
3.902603004566672688867461530978852656053e-6,
-1.789174753145672143544261938408556933638e-7,
6.160409427191822765497723897361119753819e-9,
-1.699073381804683446449336392213909324521e-10,
3.910443959241288456681862144872222888949e-12,
-7.721247163775109987705944123584382609876e-14,
// [3.8571,4.]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003978378826324807928855709105594771114903,
-0.00005463644206386038668509420551586219233218,
3.757266981609283154749562758730429059724e-6,
-1.724914668826366647669849496579469286976e-7,
5.946756791360867248430619293000333275464e-9,
-1.642106019636125148348429397483722374618e-10,
3.78355496824364813023191171514247380241e-12,
-7.478519777166822138729283423711988548826e-14,
// [3.8571,4.]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003824550944470226727136505817932246015357,
-0.00005260173774245915953303109740911109029111,
3.622320804538641850866004387831055363834e-6,
-1.66509192990316103090726650439631205306e-7,
5.747371025101084621186740200489410466912e-9,
-1.588821192632764325860270065546116581178e-10,
3.664614292248268556038058250307348150346e-12,
-7.250539737840799834137008978513279380944e-14,
// [3.8571,4.]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003682121641972140836288427255471156615901,
-0.0000507124912634722762897518158935046842874,
3.496693052799816010323195377409357682068e-6,
-1.609263914512410058790447176615124394426e-7,
5.560874131843263523132601921278037402755e-9,
-1.538874153675697242736369009006361295193e-10,
3.552900060356509045976442289742732372952e-12,
-7.036006542574188854742053159458330251297e-14,
// [3.8571,4.]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003549874388443059018509311890690805735809,
-0.00004895370273913068947204925377727831564147,
3.379454220479150164246975036362103036293e-6,
-1.557044783610233193650364836215781586982e-7,
5.386059496677594167344437263576055495991e-9,
-1.491961760200745769143182402117824693116e-10,
3.447775005658605561994293451837466460663e-12,
-6.833767833689599994375548130034138442502e-14,
// [3.8571,4.]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003426759191739147950018943486470379344563,
-0.00004731235908664323252207874193017198343995,
3.269794045584494758430251048625305068684e-6,
-1.50809668501788110601319153806606529766e-7,
5.221866120635121143727190305031088124214e-9,
-1.447816398530867606804435735160462314591e-10,
3.348674454754861772796170647815387909939e-12,
-6.642798939092985425378011138251089818774e-14,
// [3.8571,4.]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003311865136065025971807824251097955521691,
-0.00004577711663811982060807661130060890167118,
3.167003038540475820537347576719063997148e-6,
-1.462122539020475398346522221898341827571e-7,
5.067357355851268964399128181173138777629e-9,
-1.406200940118846394391955329467565524462e-10,
3.255096302086984356066290432157452302786e-12,
-6.462185704486063383221689309662066054552e-14,
// [3.8571,4.]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003204398164668387145688068220176637528057,
-0.00004433804253950522610658846431102815294165,
3.07045733194584837426826119022303326319e-6,
-1.418860084212739210403523031359105318782e-7,
4.921703252415333470282626127013794896557e-9,
-1.366904531309309812418841673814674619233e-10,
3.166592597478849970483551694454135193205e-12,
-6.291110017122944857174952414872969511405e-14,
// [3.8571,4.]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003103662977765365538049885714263982595566,
-0.00004298640264718202652052775633027346204237,
2.979606176849530705728936040065084008372e-6,
-1.378076934616806909123453923594861077773e-7,
4.784165822537609897545850746326258417626e-9,
-1.329739060565731793759532061422836083561e-10,
3.082762452570214172371090108831878070376e-12,
-6.128837542299153578302905379012676236089e-14,
// [3.8571,4.]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003009048185302741574126115930098768986518,
-0.00004171448647583508431026071409547837190469,
2.893961562698004708076217250669425244792e-6,
-1.339566453648979186512555743454586739769e-7,
4.654086675843955896486203499111492326476e-9,
-1.294536179563895104286694991065861003883e-10,
3.003246031519428304455443402708297629729e-12,
-5.97470728851987701135586202921158036697e-14,
// [3.8571,4.]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000292001405330845562616854172255714010823,
-0.00004051546187771515180570462618884585613348,
2.813089552665501583510971158845823371757e-6,
-1.303144292002165877099000526039601847191e-7,
4.530876593202957987117102751679005323952e-9,
-1.261144779621381260734543917590510083946e-10,
2.927719437784127475519515488452272451191e-12,
-5.82812269150019807679433367842271568475e-14,
// [3.8571,4.]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002836082331440060357295088964656806982618,
-0.00003938325373726147110892304973020148889089,
2.736603013207131756870811998346786170376e-6,
-1.26864546831746268866729339977626369675e-7,
4.414006694229187107382044099287570422228e-9,
-1.229428844432278460742421067636585048545e-10,
2.855890345150669661218345572240540673776e-12,
-5.688543965632112161411528242103839530007e-14,
// [3.8571,4.]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002756827761608302714671766715619356726615,
-0.00003831244218484559294873996983251127054382,
2.664155483469196007525618317358921444086e-6,
-1.235921896085037029172602357755497664883e-7,
4.303000921854711658100809421693892170497e-9,
-1.199265615346522451086142979804366540544e-10,
2.787494249840046501979054695386514241645e-12,
-5.555481517922654576610262168489993816433e-14,
// [3.8571,4.]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000268187095293919124933589745780122878587,
-0.0000372981767685157325828582545822658980055,
2.595435981781045703602494965796458546372e-6,
-1.204840279324129426669247709176130722658e-7,
4.197429620806821143600435443064608845507e-9,
-1.170544017467813629640045394406360890636e-10,
2.722291243239639757098693214836977426318e-12,
-5.428490256385166883675515997055261284583e-14
}
,{
// [4.,4.1429]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.4373105525821003748163472294875372257109,
-0.007372939349584218077544945353641757315369,
0.0001726550334404652221441608050834058310888,
-4.030788681824830111103908010383538717537e-6,
8.740927963754514521597340125301338806483e-8,
-1.721921511851684938783708512923229103256e-9,
3.066393569132825943873790294293200368451e-11,
-4.944523952694460724500384606523001715682e-13,
// [4.,4.1429]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.05161057544708952637642735276689468498171,
-0.002417170468166116405195816069101261633594,
0.00008464656231833740558096744662482239614917,
-2.447459845719572008031196282385241092364e-6,
6.026725270183177272811719773666990929753e-8,
-1.287732962543254582296958109580518339618e-9,
2.422884889644432305947393308876795139912e-11,
-4.062320492131386154864697374744116221715e-13,
// [4.,4.1429]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01692019327716281469256374802356219364945,
-0.001185051872456388240258453095223528483302,
0.00005139665676012481769997251204934375173834,
-1.687483089068905790995019031621170971227e-6,
4.5070653504939657644391312247011681917e-8,
-1.017482843888207590881140385570199508883e-9,
1.99059594501231379511119742625010644862e-11,
-3.434939329666020927029903972965654649572e-13,
// [4.,4.1429]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.008295363107194717555369347940717004767259,
-0.0007195531946414575319346797744517148841045,
0.00003543714487045915989051790652784440059794,
-1.26197830973503517515749856028173621285e-6,
3.561189937424308501871604181600532389794e-8,
-8.359389677704141558154079703378026955645e-10,
1.683172061770666399826441781327395215299e-11,
-2.968787918354128718905989327345114744569e-13,
// [4.,4.1429]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005036872362490202610837114200086409889771,
-0.0004961200281861733348590478996104752814828,
0.00002650154450444655847159836280330829459009,
-9.971331926750319832285981374394884658505e-7,
2.925786372770017333977576713385566987421e-8,
-7.068343816577335346967154736236981556396e-10,
1.45475224466440109907594040643288392589e-11,
-2.610257533087260554524350081857578812411e-13,
// [4.,4.1429]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003472840197303213242419158209026370824176,
-0.0003710216230620246031833425793509285320285,
0.00002093979704618542473597458894410525744007,
-8.192201934642955602362596372022362661816e-7,
2.473920322797912066928864907254185107355e-8,
-6.109086908550109682048531608011194560135e-10,
1.279067804590242578945452615812879409175e-11,
-2.326726455373460619130367976473647923904e-13,
// [4.,4.1429]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002597151361434172129850855878592674422054,
-0.000293157158646391131672127777135036627266,
0.00001720362406275908033255561348807614003815,
-6.926976985760684442354238213044500265951e-7,
2.138180406161080093415626325710614441948e-8,
-5.371298280245196067244294224243089472346e-10,
1.14013382387385865589661952575957364483e-11,
-2.097339022671395147695940976838249188637e-13,
// [4.,4.1429]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002052100110524737836948046535578514275589,
-0.0002408507368784407799395072582444842732674,
0.00001454665167010557402676980218651425966706,
-5.986905211789536666154229979888184332967e-7,
1.879954387236855847076494511864207874425e-8,
-4.787846486545363564326179800032174513549e-10,
1.027730837858864667823073615275673371869e-11,
-1.908204229624758185460556987487893599744e-13,
// [4.,4.1429]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001685955158149085381338795110017678312881,
-0.0002036531233813071659090885134244990747841,
0.00001257250094476553786119078696104895918559,
-5.263872352611963725459110085009304001925e-7,
1.675746260276364048793482081144951716221e-8,
-4.315813367121713958684348544828770132819e-10,
9.350521190227435298076663947416589346624e-12,
-1.749745445068239298266634367964126307916e-13,
// [4.,4.1429]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001425571863669150088728148475538180892878,
-0.0001760150132265598021250881224224911162967,
0.00001105413194049209686403984684235048903073,
-4.692089591865534136694568854375592039998e-7,
1.510534669195184172346968125870442631449e-8,
-3.926613215963552757629713575992691271914e-10,
8.574050198740480463559341538102307066895e-12,
-1.615163761257393163705661699810030817991e-13,
// [4.,4.1429]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001232105092585918547104092695185730224688,
-0.000154757847166742922409422782492158260714,
9.85338814292412778663231221081815285675e-6,
-4.229497132320498615687426133063308559493e-7,
1.374314616912417429980847908168187384973e-8,
-3.600538769983117796368141387079590290578e-10,
7.914580025161897614909925474981338004856e-12,
-1.499508828144080133988735296732551640973e-13,
// [4.,4.1429]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001083304930167200393355450792698770510476,
-0.0001379474340008011610769867287935973208732,
8.881943977879144133169220195502255130273e-6,
-3.848080982006421128957814617660407899777e-7,
1.260188561364679147511558759524956304873e-8,
-3.32359895164659398597352820400986428234e-10,
7.347853399628353091189770308912554473562e-12,
-1.399096283920845019060454230479487111227e-13,
// [4.,4.1429]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0009656320380056080677909729869997459170294,
-0.0001243472156901799801661228463027877620789,
8.080970062219220202211684987610701052744e-6,
-3.528528023036632122589495454944411971144e-7,
1.163259625428508914308166515956819909668e-8,
-3.085606755863247828691823085260058978845e-10,
6.85581652128684327792954767895346979069e-12,
-1.31113078229358312620404352676598956784e-13,
// [4.,4.1429]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0008704305098312598047609170742406917112343,
-0.0001131335808709486305102723715957804163182,
7.409908848382342072176804637564069038055e-6,
-3.25712699938118049259398570355868886224e-7,
1.079962357332627841149601152572507643686e-8,
-2.87898039519290613829915512733128211296e-10,
6.424771858005039215386892520619616252691e-12,
-1.233455202021388701048278314663773404761e-13,
// [4.,4.1429]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0007919350660966403601643680176148831815308,
-0.0001037387238772390822326738104617983682641,
6.839966698705606185019654809566555103986e-6,
-3.023894646014474771610803191231843622446e-7,
1.007643131481294890569338320262819006907e-8,
-2.697967539844209647095974648088550019907e-10,
6.044149249475612763905752779977749719647e-12,
-1.164379660092913058641158831761451798918e-13,
// [4.,4.1429]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0007261710671406735249162317738690612644562,
-0.00009575953378177081655363929826269181105919,
6.350178756635265444567880586261401024809e-6,
-2.821400811216026538763334162834808162964e-7,
9.442886324542206215037949127859706649701e-9,
-2.5381292256600069174263931217901932884e-10,
5.705668054980940326023968350296124074682e-12,
-1.102562360565289157735807126191083882801e-13,
// [4.,4.1429]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006703167364723956676011360990759873323891,
-0.00008890250259279147943568759387098841418383,
5.924941703558290092454921363135217585468e-6,
-2.64400821176690150775265488305525460542e-7,
8.883452228018347445106344647791186830538e-9,
-2.395987987929651958170469687845033519252e-10,
5.402753300553859853449876404213273552399e-12,
-1.046924924692741559017088961528513302832e-13,
// [4.,4.1429]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006223175181495403099915412968646600329256,
-0.000082949183849718739826651047498559398003,
5.552417244714914783856449091422807703824e-6,
-2.487366662774076651739502588605469427455e-7,
8.385957898798693022039473756837955480881e-9,
-2.268782666160090779126851664067247069626e-10,
5.130120787679378766901444115568250699306e-12,
-9.965911545851263164435583993657223085248e-14,
// [4.,4.1429]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005806442869480311347511141987599514282787,
-0.00007773384142591595311445487656416135633596,
5.223469991829788393722252943084641227912e-6,
-2.348068248805515311083904926426747784578e-7,
7.940739275194469907120314826845154429127e-9,
-2.154294166605497380009605720285085106047e-10,
4.883477028564632941279982189572240014313e-12,
-9.508420333050865073785587364112517635163e-14,
// [4.,4.1429]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005441368899814116296194109393282462563692,
-0.00007312857988552826169078645330288641662471,
4.93094332249563162515602603483615548812e-6,
-2.223407032565103916754583936448115357972e-7,
7.540029529126109207296953961789648731665e-9,
-2.050719447662240652640167238815811359708e-10,
4.65929874158566499736693900893880263561e-12,
-9.090821738854742092051208640120148089212e-14,
// [4.,4.1429]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005119000591986977913585009084219789365448,
-0.00006903320651485380394375257014074020272795,
4.669154768390604038436656164324816020669e-6,
-2.111208302171145179366560099528484255718e-7,
7.177518015006832181481744278940428792182e-9,
-1.956578917461930761973037080673619428907e-10,
4.454668447626870991723413348001080755069e-12,
-8.708144715265474300661870030648463610957e-14,
// [4.,4.1429]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004832324456039765887022936574950827051312,
-0.00006536816675738685455708364692759378135156,
4.433537434563139678296270751001897538402e-6,
-2.009705076843005003952413181070258273604e-7,
6.848026161319241921780664877201691918724e-9,
-1.870647391602818925507675640705856927621e-10,
4.267150262873348681653459665716175559838e-12,
-8.356207199882001864626536760233158153829e-14,
// [4.,4.1429]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004575771673017079444515355708102928227853,
-0.00006206952408380552475574459291783431186398,
4.220380661373905540486511133915265969156e-6,
-1.917447356541972669834490840644646476741e-7,
6.547265822675948627999541144870390500835e-9,
-1.791901931743318960874802852153367993587e-10,
4.094694916809472517526562543420121009624e-12,
-8.031466229189303691595860296634221976899e-14,
// [4.,4.1429]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004344866685866386371935332635225225216509,
-0.00005908532925915918198138256536233500393531,
4.026639448741607906563046713792590398121e-6,
-1.833234460547778496671665558161530483722e-7,
6.271656714897467653786727648907212894101e-9,
-1.719481957552570831287017951391691337995e-10,
3.935566305898418783241461280105932764193e-12,
-7.730900840556625704632494962366882842413e-14,
// [4.,4.1429]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004135973048141142390306438463231918648672,
-0.00005637295228230973948119902853236564856682,
3.849792367153679408412003972964585198905e-6,
-1.756063909280044539550481330363631974721e-7,
6.018186806839647540211736850971189272892e-9,
-1.652658402719130981223266927195815194069e-10,
3.788284114103296852582590073261795284577e-12,
-7.451919717564955666731102116929988746059e-14,
// [4.,4.1429]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000394610665976168142702664889509705945254,
-0.00005389709314008127592892613149317668934708,
3.687734209491325460528383340529241815285e-6,
-1.685092334009677393991967558975480580197e-7,
5.784304366424451698046943511633434736808e-9,
-1.59080961832548547636108768567902695975e-10,
3.651578557927986432704209585136849004155e-12,
-7.192287706101584664783752746110557669915e-14,
// [4.,4.1429]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003772796519805688989338998946375731367759,
-0.00005162827893281068605239687126016607568972,
3.538693901423449128370867876744596727012e-6,
-1.619605249747256796556301119744391072051e-7,
5.56783362245104985822266354192731821239e-9,
-1.533402367975807436095160464014565250567e-10,
3.524354378357997013098592805240348271655e-12,
-6.950066864689094273869864119359378753452e-14,
// [4.,4.1429]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003613979525296747708263373627185337474761,
-0.00004954171461986262925586152442892848280166,
3.401171024472267171590636839945650937177e-6,
-1.558993440549955090094032280006559577383e-7,
5.366908247543209632564739679996980351932e-9,
-1.479976706198078299555471887231834129874e-10,
3.405661954751920216932381110668121090731e-12,
-6.72356881206054850272385568087543126549e-14,
// [4.,4.1429]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003467920023390383742160198388872895899449,
-0.0000476163943425481546040512195158295965331,
3.273886225157840906272648845348496713694e-6,
-1.502734334746655315663417411329599315837e-7,
5.179918432556918095877600774966738099749e-9,
-1.430133847733337173306513401867756708773e-10,
3.294673954522389590916509332575215932135e-12,
-6.5113159313193392525447415271233345631e-14,
// [4.,4.1429]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003333147603978370525616159484791246461594,
-0.00004583440715214813320406774409235965355934,
3.155742102970824185767141044477033494089e-6,
-1.45037718577196104993343431133934361995e-7,
5.005468429092917101943138785085165086602e-9,
-1.383526361594062909126740260873850156778e-10,
3.190666322654087079634039712660690404033e-12,
-6.312009573063982865881118864005165227005e-14,
// [4.,4.1429]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000320840850065036903617754432389464881573,
-0.0000441803894415317301931171948142755899475,
3.045792090123884049118920035140118486637e-6,
-1.401531184069603855895546649513036389621e-7,
4.842342228701175449960631709236623851156e-9,
-1.339850187645143896502086556991295125896e-10,
3.093002700803231156365835340298869871918e-12,
-6.124503831113761413072510917330755431449e-14,
// [4.,4.1429]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003092627260907220833492476122448140341962,
-0.00004264108926167629402897772525746573543565,
2.943215486548856359122195790263823626962e-6,
-1.355855847269610505807047769888329830589e-7,
4.689475620914395690169460631624310755435e-9,
-1.298838093441509986698995513870885338072e-10,
3.001121577043038397932090400639144101476e-12,
-5.947783786519246467218319389859776069824e-14,
// [4.,4.1429]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002984876248317340309644393503527886802246,
-0.00004120501681162753560000435910067032686714,
2.847297279268796963423964861754041888008e-6,
-1.313053196437613991413100903370633026072e-7,
4.545933292179820226856044579801449280928e-9,
-1.260254277801824917604219035718607896703e-10,
2.914525625126899630465829658196362432661e-12,
-5.780947358206681821044013087041747013689e-14,
// [4.,4.1429]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002884351176813927226853008735918083223301,
-0.00003986216190970824462920663276278697040465,
2.757411712521534810025635303487397967553e-6,
-1.272861343775699780256719383152241678069e-7,
4.41088993836723433428598466254288783563e-9,
-1.22388989386976729522284081273150140555e-10,
2.832772811049634258593313955529358659809e-12,
-5.623190083006672385206187879014253853446e-14,
// [4.,4.1429]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002790351333679576865760268783258365542736,
-0.00003860376397524803341652907810331586266991,
2.673008821931449080483744311786484889224e-6,
-1.235049204124596556688344994217944460267e-7,
4.283614595483516928661433001919207235714e-9,
-1.189559314350752836224624088286483058259e-10,
2.755468935046069976937929259125605381143e-12,
-5.473792289065905082672840885557052279838e-14,
// [4.,4.1429]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002702263478267362087390436484918084097018,
-0.00003742212350698821680971622719956877952327,
2.593603328664069741735092831103265112845e-6,
-1.199412107563708015725120641502180105305e-7,
4.163457568001225683578230147349440458668e-9,
-1.157096999558593775440712445828620293267e-10,
2.682261346379532753417500720821239778003e-12,
-5.332108235651788618904320268109650056463e-14,
// [4.,4.1429]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002619548645489174931111022976528132047338,
-0.00003631044660124622002005899022774652239281,
2.518765425886144309533175641550311977041e-6,
-1.165768139343080617319436129717220023052e-7,
4.049839467021953863623455891843899581082e-9,
-1.126354857973015706992382021694272066304e-10,
2.612833621689616499663858990695448134541e-12,
-5.197556877092552441952332189184764060113e-14,
// [4.,4.1429]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002541731262087235161735361759461534870327,
-0.00003526271596235651338753349627650915544993,
2.448113092622770129266983009673398161432e-6,
-1.133955070569112688012822986232617270956e-7,
3.942241972227681203694345658912250233261e-9,
-1.09720001144151512243959119450682913948e-10,
2.546901039188638151584231463026460133773e-12,
-5.069613974902371470483368503028987858388e-14,
// [4.,4.1429]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002468390117364955703082609540305686224368,
-0.00003427358329667046437831228741611729473582,
2.381305648197383486727135850742632469343e-6,
-1.103827771550906247762003026724008878512e-7,
3.840200010087311386531770399012725566575e-9,
-1.069512894595497870938304507354579648463e-10,
2.484206713487447742152725070263609974456e-12,
-4.947805334362242387673288448015552052643e-14,
// [4.,4.1429]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000239915083076693227780377339859582492167,
-0.00003333827907471618519233807941222155773332,
2.318038320259098442102181718371235942382e-6,
-1.075256021698074764331976377332584101785e-7,
3.743295101813187832099272622179881301083e-9,
-1.04318563169541790718937111808047871837e-10,
2.42451828142028536011137197664218120276e-12,
-4.831700983205595870898961968469555147064e-14,
// [4.,4.1429]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002333679535230132739912085175628597352132,
-0.00003245253648358127648837838385218390948785,
2.258037645568103112177137791615102681685e-6,
-1.048122646948547931583948776026541251883e-7,
3.651149682319106341366926578564044711212e-9,
-1.018120644865788440967075994608015015553e-10,
2.367625049514247909441440214829446693758e-12,
-4.720910143034339371201367981114020728664e-14
}
,{
// [4.1429,4.2857]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.4301063231948065162989592685422184617592,
-0.007039380440677408352460096017200787490906,
0.0001610703535828170672648491517771205761488,
-3.697774476002217457360325469803864295549e-6,
7.924276832550226268772031989042351882578e-8,
-1.547947917726895770366968868587192104469e-9,
2.739786116602570670927533190653877889866e-11,
-4.397844464169461577292541147367518866546e-13,
// [4.1429,4.2857]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.04927566308474185832014010363710857303332,
-0.002254984950159087318272933733250582838135,
0.00007765326399606068640110605547126073295099,
-2.218797527179102229666403776599915016155e-6,
5.417817693217690238436746114218098285789e-8,
-1.150575143950433940442936098056088802774e-9,
2.155004032172655739990107374027200661546e-11,
-3.600679641096039954777724767321262712019e-13,
// [4.1429,4.2857]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01578489465111361110106414858359233747051,
-0.001087145695944553094177324007332349889642,
0.00004659474807077332413135335251081519326816,
-1.516988965961659858174315557940255455219e-6,
4.027012987590063522735057531980100779068e-8,
-9.049878301592188225225819342307933789195e-10,
1.76438498088805059177929551425772258983e-11,
-3.036366945450439588118776633902250348349e-13,
// [4.1429,4.2857]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007610019871611871547909056228695857878798,
-0.0006523264729905708146207909081556896752888,
0.00003185676828520554496365193045437734451762,
-1.12756364675422578830488710211712033814e-6,
3.167457391306636831725253653914616400982e-8,
-7.409434929887953008072028169018139864146e-10,
1.487865405368413897154508755321610781621e-11,
-2.618649242464358361150431045522756850008e-13,
// [4.1429,4.2857]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004566285310933995603250018386175702127241,
-0.0004459947559926531829410339345617275811969,
0.00002367883658184825476925975937239459194409,
-8.868880785437916900346172938436364481253e-7,
2.593302212776460970836415074261686974627e-8,
-6.24817281637520618323911270072195826161e-10,
1.283178718716823995771272613181204255006e-11,
-2.298371826526565036329527835623689360775e-13,
// [4.1429,4.2857]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003121963291948572191369632303281448202782,
-0.0003315037121456757894682476839171773384399,
0.00001862464964942819042217747979195806132419,
-7.261246275685659976799878421069777773327e-7,
2.186860474311380485102773317275291876369e-8,
-5.38858346339405694598159352218493279528e-10,
1.126238738881957170827376882284454088108e-11,
-2.045755152347695080458510653154660728553e-13,
// [4.1429,4.2857]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002320525985019730445189117196714219998785,
-0.0002607450950918148025429748630883529492276,
0.00001524861717894767049661062010824944752995,
-6.123209400017806671383684346946212524894e-7,
1.886004201808495645819076985257829699863e-8,
-4.729512018464013343265989024884322955516e-10,
1.002453238873173374570292954970716019687e-11,
-1.841833485531460583087803554610064672374e-13,
// [4.1429,4.2857]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001825215665642703543511477690386906897544,
-0.0002134806405051039116785742167622993410472,
0.00001285873974004452582253904658905320311414,
-5.280811830454445576483568529417527150369e-7,
1.65532919695329285196622689505623971598e-8,
-4.209675849444390811504163296340519005328e-10,
9.025288371286225661307506115874382951372e-12,
-1.674016877119261094575538037660089488943e-13,
// [4.1429,4.2857]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001494364483535727313227064322793151376421,
-0.0001800223563604735936153666486913468221997,
0.00001108970484396091394730890873640632573722,
-4.634921811376888643098088240668331616212e-7,
1.473386538534528130137863180515856032114e-8,
-3.790045999066134482776356549615169341503e-10,
8.202963370724420764015122143800054869953e-12,
-1.533651197507793364466956529919529311393e-13,
// [4.1429,4.2857]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00126015649452331509173375445502735786769,
-0.0001552558678153146524457347622285884156633,
9.733335803897569267370329539489210603942e-6,
-4.125482363154279649862939445962379224264e-7,
1.326516091535633193028437202976766502906e-8,
-3.444714139700540937362188545897080362483e-10,
7.515151268759654128777152432163246696604e-12,
-1.414608457958804597286904646417468773253e-13,
// [4.1429,4.2857]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001086791074707202507838380138215099025923,
-0.0001362667012544378044332490596609184115601,
8.663512962629678343038811753416654300696e-6,
-3.714245107566341114548839275657529197705e-7,
1.205649941307061644617609260850472229626e-8,
-3.155871370981687234593020050555515649181e-10,
6.931824264579971298019668490693185732312e-12,
-1.312437141374057893090986207158115205475e-13,
// [4.1429,4.2857]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0009538669087810645755072183918841643317163,
-0.0001212891814766959842932925325607637984704,
7.799914725894646818375845746325413691831e-6,
-3.375819883465193994441197022553046876656e-7,
1.104554972736264830901202562567473410589e-8,
-2.910907256405021405317113799212284266333e-10,
6.431169427623309867619869003540194835307e-12,
-1.223831008057655693721844931491331653305e-13,
// [4.1429,4.2857]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0008490242703368718378413853519083019658998,
-0.0001091988061624131155509335476823949903817,
7.08922175528191973386982979577064585766e-6,
-3.092753968437898190648536481846070578276e-7,
1.018817533058609336427800947116977101336e-8,
-2.700661304055348928642555344151567395138e-10,
5.996985800666461274828045284448972630719e-12,
-1.146286261890860516586346408373314990351e-13,
// [4.1429,4.2857]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0007643916431368917595894449918756989693715,
-0.00009924910457384161713888935076834322377153,
6.49478333372431586562376909401955798048e-6,
-2.852689134668133324437349187896918824261e-7,
9.452314501131320661230748565622180605871e-9,
-2.518329835231317779412837470293850770049e-10,
5.617004483367578398006861135134743848958e-12,
-1.077874010833917570433293942423165015672e-13,
// [4.1429,4.2857]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006947437320168912733383686489335375312297,
-0.00009092696667204109926320399085162158139431,
5.990647182807556855017057819867047367963e-6,
-2.646648100046266116692109979530903474035e-7,
8.814154363617775349751240218845421668148e-9,
-2.358760477583217636480446448657890136756e-10,
5.281773667364342798700639546750959882638e-12,
-1.017085466709258560656302448948120699133e-13,
// [4.1429,4.2857]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006364887667042876505769796162843456078265,
-0.00008386906055921178173197593631541696924889,
5.557961010101408350454672538817130635131e-6,
-2.467963259419009392600442584806452843308e-7,
8.255661614881010809218702313397031491706e-9,
-2.217983920226248637935592026336637572236e-10,
4.983900100059965019952120671890306426094e-12,
-9.627242861716570594082764238850231538988e-14,
// [4.1429,4.2857]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005870834239144824299997173432447113650094,
-0.00007781145414133047739729531752706159328838,
5.182722844783963660799994338406778939083e-6,
-2.311585287862808082646275653450856705135e-7,
7.76294366687254804531561769382112403465e-9,
-2.092895357167494721807306403387154627706e-10,
4.7175215444369208260755778201099402478e-12,
-9.138302055752029950156616749184841523081e-14,
// [4.1429,4.2857]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005446801789893133016021578372333919150575,
-0.00007255811982689056868782618458270747163928,
4.854329104515754169328282956821422825059e-6,
-2.173624260693646032943892728237983986624e-7,
7.325133698656790816097145994669258886259e-9,
-1.981032941104446954368236496342155693079e-10,
4.477932581878128173977769953259726685533e-12,
-8.696239028200581969659278712933238733549e-14,
// [4.1429,4.2857]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005079068387882339424105485050693292036566,
-0.0000679606074631395573549275101705714899309,
4.564610947460343495360153566506555041615e-6,
-2.051037468024601922784595349676303006419e-7,
6.933615244707724858510411714445749269677e-9,
-1.880420635781004648113922544086651175666e-10,
4.261314429239793548828161840477918622184e-12,
-8.294665409603238563734248086671941872839e-14,
// [4.1429,4.2857]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004757242522419768647057560235805252181552,
-0.00006390455326436738567694827200848805551525,
4.307178682855194815427553492926626585738e-6,
-1.941412299487748470772806801816266274589e-7,
6.581472178156330897757607913377060651247e-9,
-1.789454750453255848558531379565307740489e-10,
4.064536698020847757536200684490470098378e-12,
-7.928296476286408854958904672805835950544e-14,
// [4.1429,4.2857]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004473318728505716644541727110285950369466,
-0.00006030050155989858117487177199213827417871,
4.076965828927659115443039280371776064453e-6,
-1.842812239542540343994430142131201933482e-7,
6.263091581421892573881710125781803812819e-9,
-1.70682068726291476919336480246329455214e-10,
3.885009800099569914126440362028110624151e-12,
-7.592723889077853951817525973407333824917e-14,
// [4.1429,4.2857]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004221035109192900343178761751516120072633,
-0.00005707752160491609383773155557980741781019,
3.869905703042589737609361799744033395671e-6,
-1.753665671251902354990220723203395299089e-7,
5.973872362019858718789346805060194712101e-9,
-1.631430958155745531908099807059978302926e-10,
3.720573587044386548207780335917671618512e-12,
-7.284242118367986046463576371200957420251e-14,
// [4.1429,4.2857]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003995426512344126242330475576114180593867,
-0.00005417867984252790108856786799841691231568,
3.68269790963212754540973148581702058212e-6,
-1.672684288707907207221786496322493160878e-7,
5.710008311776975166699313587922782425186e-9,
-1.562378418419562401468128269353871302554e-10,
3.569412296318004430267404151882462162853e-12,
-6.999714377997533372325123770464372413687e-14,
// [4.1429,4.2857]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003792507588976952761720660458218524154613,
-0.00005155777073478400111602070132723323995506,
3.512637006289624150751712313537726394758e-6,
-1.598802353611603707724259804996687684665e-7,
5.468324424214796705368257384130728881653e-9,
-1.498900548013446844374928927753446222101e-10,
3.429988857246746208169373603244422773777e-12,
-6.736467998780712488889045638891173244206e-14,
// [4.1429,4.2857]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003609043951434879774649776307254279080675,
-0.00004917691808799133885937248924190374081923,
3.357484942587281130064611883394281893734e-6,
-1.531130864140078100142509311767128575474e-7,
5.246151879202353092626043615181194002389e-9,
-1.440351863166209269494368538014864004839e-10,
3.300993622746709197715511306952859160187e-12,
-6.492211992503153073148529846359070944448e-14,
// [4.1429,4.2857]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003442384266159393426950483905094195957005,
-0.00004700478919616075567384866878226358032852,
3.215374814696978799251064672610845800262e-6,
-1.468922550648944572876886835098418935868e-7,
5.041231483551090808809757694446990387503e-9,
-1.386182386158760489170488417566227975865e-10,
3.1813039746406341906800874443931221307e-12,
-6.264971521460985172103867975071239261473e-14,
// [4.1429,4.2857]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003290335243731252613560739153582860958363,
-0.00004501524740569859269285310937536270027017,
3.084737356365506261065755925410706990144e-6,
-1.411544839038723783122000924081599995444e-7,
4.851638315253436202531132804631375166739e-9,
-1.335920681526804948013258904612545421781e-10,
3.069952212890354926665739246891500741469e-12,
-6.053035375409024994955825437765252897983e-14,
// [4.1429,4.2857]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000315106731839890121388194479900986904311,
-0.00004318632298905991190554412551225872128545,
2.96424416198395628954853188905241418947e-6,
-1.358458751141474827035039262674348846145e-7,
4.675722350192435769459724769926507278186e-9,
-1.289160371129298717803539037595826884932e-10,
2.966099818616678376129339536302697729209e-12,
-5.854913549314809004790578457980775039573e-14,
// [4.1429,4.2857]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003023042609234193567210640419978288910612,
-0.00004149941826772002487591232666740668515264,
2.852763377399652454234460062983677474729e-6,
-1.309202280199359711169722936809534831934e-7,
4.512061264881534380464142124193836859258e-9,
-1.24554932591094859681255298587524367882e-10,
2.869016666638073440295796390655659572409e-12,
-5.669302732598405725960965572934706119638e-14,
// [4.1429,4.2857]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002904959278740401483075080291304220440092,
-0.00003993868728354147275773297060538712560862,
2.749324788421134499332742145360923255491e-6,
-1.263377175631670851003536111469280726691e-7,
4.359422607633466561457676468061144254047e-9,
-1.204780936228357535689108897183294950791e-10,
2.778064114735994868047131700893340393873e-12,
-5.495058045143265105017230307271748760193e-14,
// [4.1429,4.2857]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002795708109847902842282833606846974320467,
-0.00003849054703784382183055075336535604554981,
2.653092068828916081653142925862475730827e-6,
-1.220638350962029636667078109093680585824e-7,
4.216733244701885227673668869510473282293e-9,
-1.166587010214993899006281459919992703203e-10,
2.692681153917019077555866530461951979626e-12,
-5.331169743001052318838210293983728647902e-14,
// [4.1429,4.2857]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002694338292649067284440499698439302672074,
-0.00003714328896355427201924546599756984629104,
2.563340537022601751150366828221522209326e-6,
-1.180685328737967039757356356438928036556e-7,
4.083054504558853929495000473486828099546e-9,
-1.130731957615504181591143170575300600421e-10,
2.612372993888754867777071913841900881993e-12,
-5.176743905910001637754155299133045176167e-14,
// [4.1429,4.2857]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002600030227448798804323957656075595890191,
-0.0000358867675182672947788856424289008532097,
2.479439190352006218956920306006526020621e-6,
-1.143255280928558548779099918511228548423e-7,
3.957561821315025027186167069420787693002e-9,
-1.097007996289374879215470064754111996812e-10,
2.536701599676100119175405527681597205988e-12,
-5.030986336457975068014043674634063054719e-14,
// [4.1429,4.2857]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002512073726278710403818672227640245342745,
-0.00003471214866488030297896462064135027501605,
2.400836089952187716524576456576889969263e-6,
-1.108117329082021347301971659942567605786e-7,
3.839527957482526709071179384223317687241e-9,
-1.065231178095786058860300361965366126454e-10,
2.465277801985633718979398872120592587884e-12,
-4.893189066016867502888722554780246844329e-14,
// [4.1429,4.2857]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002429850406541620983816683997240008732497,
-0.0000336117052592841180423721143627628758646,
2.327046391074402065281901715110490128451e-6,
-1.07506784669927828230673014741930016112e-7,
3.728309094572010122527969736877457322744e-9,
-1.035238075672866733971788452672460151326e-10,
2.397754684922641545067305879951142852006e-12,
-4.7627189890835415389226616700268084192e-14,
// [4.1429,4.2857]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002352819368149888043945015421810319985039,
-0.00003257864947499632701452449472437732180672,
2.257642478070587006359404464214261895086e-6,
-1.043926564601093490630052311360540941464e-7,
3.623333236820093878064689783202987111851e-9,
-1.006883005639521751151913093444530245656e-10,
2.333822016655964072136575376563254836585e-12,
-4.639008245220901028052977055678250680199e-14,
// [4.1429,4.2857]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002280505463249742677405128593315952318477,
-0.00003160699469294406325935441464136353133258,
2.192245785664347012878405498081798629437e-6,
-1.01453332397172492485658380661507661559e-7,
3.524090492395801422943177521755710161189e-9,
-9.800356897812476615196544421256783810366e-11,
2.273201536430657516314085827996035272988e-12,
-4.52154604356625588361545333255449150844e-14,
// [4.1429,4.2857]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002212489628506084219692488666030570228617,
-0.00003069144099925779389950533020516283904504,
2.130519980342623591543013638806742474819e-6,
-9.867453550966995576001514209983677946805e-8,
3.430124887550953675440787979370425021657e-9,
-9.545792758571508778585497241016854628673e-11,
2.215642948458421992987667276448941850591e-12,
-4.409871684127643017850764280924591309703e-14,
// [4.1429,4.2857]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002148400869948045369410429606934347806068,
-0.00002982727972475470409702703637727269503132,
2.07216524570502320923783325094480233304e-6,
-9.604349853248999557740762190763362490092e-8,
3.341027439444764479463780716073788395566e-9,
-9.304086552582165036154148081903377132714e-11,
2.160920502250703888833907517306976798642e-12,
-4.30356857672276517062030059824107040108e-14,
// [4.1429,4.2857]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002087909580732829087919056912680688883921,
-0.0000290103134398292880759931331703025312822,
2.016913469184199096929516163748762556587e-6,
-9.354876994594309079093515464336472740885e-8,
3.256430267947809054190755444197717667679e-9,
-9.074290269401720301833719441850113554786e-11,
2.108830061810621660590711248942507006949e-12,
-4.202259095342110881536428077794164473101e-14,
// [4.1429,4.2857]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002030721940788049970918656718358890349469,
-0.00002823678856853869442051092972899672569384,
1.964524168866671165244757159838104323867e-6,
-9.118004910627132346542522755429445571486e-8,
3.176001569407070803670466277903516638509e-9,
-8.855546666507088476798180491652695172237e-11,
2.059186584194513344019673602256889188828e-12,
-4.105600135128470664775329352897765215789e-14
}
,{
// [4.2857,4.4286]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.4232243930550180505217427016557375910012,
-0.006728023664263783517307145951252755534568,
0.000150437409150622032352925505352857291668,
-3.395750025029104434176484024639281173872e-6,
7.189899692123292184433010042570767933098e-8,
-1.392465160171756310353456016862966695745e-9,
2.44923108076068379130715072814913896474e-11,
-3.913211462840062166067685245402130701505e-13,
// [4.2857,4.4286]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.04709616564984648449110096514734807771336,
-0.002106123728108396685114612917035309780397,
0.00007131075052562367788170238654928668890562,
-2.013171926265324393425024617788258226574e-6,
4.873628043954751434056941114485006243677e-8,
-1.028557333641719676808869456315769904743e-9,
1.917526885350853173190357863174739572284e-11,
-3.192551654452584275455900877880652395769e-13,
// [4.2857,4.4286]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01474286609675877668389130624877672585573,
-0.0009983505073584693105545724547702198284473,
0.00004227661045158255576490735755586866163398,
-1.364615862794600409639779877038133636372e-6,
3.599950653421309424908810157674102404182e-8,
-8.052606135532767072604997403874473529318e-10,
1.56439614983476361842423942507277686992e-11,
-2.684764070852681384781744059803377722452e-13,
// [4.2857,4.4286]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00698845355150928507583405989177420261558,
-0.0005918725463219301673724195358392549452837,
0.00002865693311869602124898826711093181271413,
-1.007986191982569803797625743151372270392e-6,
2.818412134886238180979862035353972717476e-8,
-6.569597462940496033679140007895276089788e-10,
1.315574555529189576686187421293667591661e-11,
-2.310318806512103849292837877865716481844e-13,
// [4.2857,4.4286]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004143107824253511084464295625318355003077,
-0.0004011970636615466321117472867614533567303,
0.00002116771003164233161289584499863315937615,
-7.891554056748243411174047840281797671182e-7,
2.299359100874831072484944693219500643014e-8,
-5.524654088156168761713885627966875105874e-10,
1.132091909154637422876865712862091262726e-11,
-2.024127819042248265458799717799923392297e-13,
// [4.2857,4.4286]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002808379445630826346422880692991500213545,
-0.0002963479404428169623692668268466067924031,
0.00001657226351917883369734052031251305523081,
-6.438205552722180669076046300465055997171e-7,
1.933628920824584903806965166515121590415e-8,
-4.754111397331662083059804133445052036212e-10,
9.918547276298811736066013574514109445918e-12,
-1.798996585855646663073594064092366055863e-13,
// [4.2857,4.4286]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002074435583099718665439690355941899509455,
-0.0002320116892683456987205305305698520404826,
0.00001352023166072340937549869206480427045891,
-5.414161041498575590027750992987684899788e-7,
1.663938979959427752448576784292941909437e-8,
-4.165183231227857462337713105066690379204e-10,
8.815374684190048225866465874610362289871e-12,
-1.617672531249874238955092534840596702842e-13,
// [4.2857,4.4286]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001624081824878419825922515849202083172373,
-0.0001892832432499843020336515732122812207238,
0.000011369738187153260405025044697982681719,
-4.659029201258567192719502269867000298154e-7,
1.457814122594170711866547020421546345611e-8,
-3.701906591459753209234779031485062591565e-10,
7.926862142193773008306830295300609185748e-12,
-1.468741586871648965792042799581368040132e-13,
// [4.2857,4.4286]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001324982702749890054215761167803642918957,
-0.0001591763346200143608295798935833650237607,
9.783961322648753034359552565310711425247e-6,
-4.081879595778059473534462193157431438325e-7,
1.295667299328317667443901127650050084101e-8,
-3.328777958810055652060856526412333006397e-10,
7.197079619234274938251698947325221920093e-12,
-1.344381190455167396319066509844345071972e-13,
// [4.2857,4.4286]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001114234342340100470160021225928159274764,
-0.0001369754585169615420931063241633640788942,
8.571947151139267037093500040903988616393e-6,
-3.627868486519859509103842720396199531397e-7,
1.165072278460640966943551501312843345376e-8,
-3.022308791950829349200531986571262644135e-10,
6.587695765803833674886077993639787396181e-12,
-1.239066731013084666589427192495570135455e-13,
// [4.2857,4.4286]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0009588282096187307427925783543707005066402,
-0.0001200072601158375536493071116091356711753,
7.618523821696683474441592853894358377229e-6,
-3.262202424564131084631251525097307592576e-7,
1.057808070544762857460653572643751879386e-8,
-2.766401425522764741853798246195721836113e-10,
6.071639399274812473488110518882394211413e-12,
-1.148794374127755428334764117664191459317e-13,
// [4.2857,4.4286]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0008400508208108628269976202978705885356481,
-0.0001066593335036490201488987211143044781895,
6.850625091589335864509341281502336576457e-6,
-2.961862639345098339243347605896803369909e-7,
9.682404927188327852292382300583743827387e-9,
-2.549687075640799154354692007830437883807e-10,
5.629291285950791138897877069899291781199e-12,
-1.070596595530367897466105752337359785542e-13,
// [4.2857,4.4286]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0007466153345255430954137261061954317457269,
-0.00009590875128215282989245216881118585107314,
6.219911542629086877309799802898953791896e-6,
-2.711073418761960362741612536847843874099e-7,
8.923904706337750406817194502058092042284e-9,
-2.363926505301630267290647993066771550729e-10,
5.246110214634839759903391985368168980434e-12,
-1.002230324338403240858613934944840522221e-13,
// [4.2857,4.4286]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006713612589750697662101564628003979044758,
-0.00008707876159671522872522751562699626528265,
5.693254179404248336100252291814620837952e-6,
-2.498693354569918724010666293852772994313e-7,
8.273742713467875982390409305081646305984e-9,
-2.203013052715058864605767363336535605787e-10,
4.91110487068106365951698071859330583746e-12,
-9.419704554819411695551088063142651624994e-14,
// [4.2857,4.4286]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006095513311770065603554080214694326558328,
-0.00007970555851157271374456151610048544074716,
5.247256044600738574601890329917686033741e-6,
-2.316647994476498726113395742243965156174e-7,
7.710545632379153292477123685974744980234e-9,
-2.062330870970573206185454584677227107447e-10,
4.615822027579431525143308003828923034725e-12,
-8.884696762169193860827263716054351341215e-14,
// [4.2857,4.4286]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005579389095810089575722638217461726727722,
-0.00007346158462432824580213035077998921725753,
4.864960788404356752358725689021067924718e-6,
-2.158952809904153908377987763873776116795e-7,
7.218157998937816543121593103713879778501e-9,
-1.938330004987843705497694541622383980944e-10,
4.353659683202828274893382639587593515297e-12,
-8.406611729122585553266506627149535951986e-14,
// [4.2857,4.4286]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005142310923702976838560377132067913349073,
-0.0000681094510375830966461278283829458237736,
4.533800900802252078211072052852387569909e-6,
-2.021084270862023797460789999311659453597e-7,
6.784154970405695601029481169687734614078e-9,
-1.828237934572979843129009232803651485284e-10,
4.119390313211806689824619852616958703711e-12,
-7.976897442254559230268881157285421476528e-14,
// [4.2857,4.4286]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004767661572630816414715838759005862883745,
-0.00006347321261115742289898577203196902137607,
4.24427696881361491836489206190830776345e-6,
-1.899563421356341107272192380647763065189e-7,
6.398832726139370697479213337888114786311e-9,
-1.72985935946920544310927252148551661119e-10,
3.908823318397365854246657298439964645806e-12,
-7.588621458292492837116210639352376360397e-14,
// [4.2857,4.4286]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004443124882781019267986495693004056464711,
-0.00005941987756331994512275818478887990107493,
3.989083184851531790471380826870938919022e-6,
-1.791673191584774082852100884204504709483e-7,
6.054507715269212762381098796883456416806e-9,
-1.641434440882219067650392917652974139892e-10,
3.718561708475623294533900486338800995958e-12,
-7.236107132661493672506517145715771725148e-14,
// [4.2857,4.4286]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004159391429432395837905514153128449036941,
-0.00005584716458785392037729358486285305936535,
3.762513702331104191001207362547202567241e-6,
-1.695262187285501364616093121315816869304e-7,
5.745020502039408331828722447256078356771e-9,
-1.561536618815716056972236703154960554479e-10,
3.545823850031735063695543886432016937936e-12,
-6.914663171823599736736763278792752985996e-14,
// [4.2857,4.4286]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000390930152114977411882134748538123972311,
-0.00005267519183257080779352219918563255060417,
3.560050593302505737872824190198468458299e-6,
-1.608605766431622938696939509238384137912e-7,
5.465378126483250537734712778244384876154e-9,
-1.488997753849692928285165570514797595928e-10,
3.388310944081779315932174761188503467885e-12,
-6.620379849832201661282248677503146336051e-14,
// [4.2857,4.4286]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003687263428279956250033649088028078093405,
-0.00004984070830617307008831208146837065700932,
3.378072109509245110399462796683435090593e-6,
-1.530305900219634386506136052876049300317e-7,
5.211492100647948108510192734484719419571e-9,
-1.422852473545846014841419634469705293092e-10,
3.244107169804886096189120035073406900137e-12,
-6.349973557022701738499593761273017151134e-14,
// [4.2857,4.4286]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003488849581432114621837729497508090216929,
-0.00004729300953306985589490030175687931538824,
3.213642390463961929891504368082389751828e-6,
-1.459217812011904932771154810126109710798e-7,
4.979983621014100298313273486381725133873e-9,
-1.362296237312416897554311540626664268099e-10,
3.111603511546641355846590967713253936408e-12,
-6.100666870307003352136750125662272418328e-14,
// [4.2857,4.4286]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003310510667314889638658304363433344467138,
-0.00004499099346643814300433921166872304387233,
3.064357405227630626313400721687047004942e-6,
-1.394395436813765218230248914160976343062e-7,
4.768036795523112182792944253280051611752e-9,
-1.306653347251628536110610567839524608888e-10,
2.989438991803272374716450499388072658852e-12,
-5.870095059392233457196921826898360840802e-14,
// [4.2857,4.4286]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003149369542650669745953564018754703934944,
-0.00004290100367313159321734015456474220748323,
2.92823041731144473376502875847584878064e-6,
-1.335050324840896121900260197849709023063e-7,
4.573286681543583488611641105720611977857e-9,
-1.255352268772675942129572511620107546512e-10,
2.87645485810923402476713144901077421585e-12,
-5.65623249635175938350589201362762668097e-14,
// [4.2857,4.4286]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003003070257119211269846500248266612106803,
-0.00004099522584230693305174441067292380366559,
2.803605682168333395480261319009885547132e-6,
-1.280520292149689339086194727590196127543e-7,
4.393732908017066134499445452826598844355e-9,
-1.207906391276146641929713193728517865518e-10,
2.771658522798413598897599288201968561628e-12,
-5.457334211867244235158797709741628387598e-14,
// [4.2857,4.4286]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002869665808961485066649701995773835827999,
-0.00003925047955030518526954313106683635261813,
2.689092613516718560083342504835850521936e-6,
-1.230245234837876675709865114537230916366e-7,
4.227672337853770034590377253993697256784e-9,
-1.163898884608779117672055401904353239037e-10,
2.67419492481334911228318547521032211342e-12,
-5.271889093685929908704487203268777455763e-14,
// [4.2857,4.4286]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000274753356852136272975823792228699603319,
-0.00003764729658918426999330467534331374530252,
2.583514993161836485402055573129862467592e-6,
-1.183748274515180137037122509774819299185e-7,
4.0736460655244080026162019334901480084e-9,
-1.122970672439688311900603558221782562403e-10,
2.583323596340144413309755799782389249294e-12,
-5.098582117066796537933799251619081208064e-14,
// [4.2857,4.4286]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002635310761242898667800787126240585802033,
-0.00003616920990421750605937530193626822192407,
2.485871376484102913110869214257493697972e-6,
-1.140620917628908573514310732603432165122e-7,
3.930397323877140805380041083110793261839e-9,
-1.084810801398337549288744522177260265865e-10,
2.498400155228566260404113344352447343239e-12,
-4.936263643147469331278380700383695106519e-14,
// [4.2857,4.4286]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002531844693295225199364689843461179898907,
-0.00003480219927073072370824785945822473709794,
2.395303927022866015364623896899161084474e-6,
-1.100511269372604149219112530936175631144e-7,
3.79683777612060719407452858670346543518e-9,
-1.049148668833190169917702628819784905563e-10,
2.418861260781251356674654953177239613357e-12,
-4.783924293240266303895913640460851441275e-14,
// [4.2857,4.4286]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000243615394895115044132258632554859300437,
-0.00003353425497827480603951342962368831898053,
2.311073665684563970572625809538462691857e-6,
-1.063114595441209918062449108323997498795e-7,
3.672020312979310923457201381638733040473e-9,
-1.015747705019172671515335042796009037097e-10,
2.344212301818600545548887914201839901041e-12,
-4.640674255533045944941569808692939232785e-14,
// [4.2857,4.4286]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002347397848479236210679723784328182159142,
-0.00003235503131953989524012483921096723068862,
2.232540650428576866779986474020805726641e-6,
-1.02816570523451121821522791598552630238e-7,
3.555116940419830318562092189545456579052e-9,
-9.844002027878542856650841203281264733959e-11,
2.274017256678773301715612945516971543966e-12,
-4.505726140392311292718339990079492277667e-14,
// [4.2857,4.4286]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002264852192367792460552981795268556942542,
-0.0000312555691059573193712309251930936484866,
2.159147980994453624399935677670001662964e-6,
-9.954327604204185521930758793995277136963e-8,
3.445400683356520976380542949961813006089e-9,
-9.549230592631136692326750875270616286098e-11,
2.207890292078816497650116178170068668422e-12,
-4.378380695808824420351754315387532719361e-14,
// [4.2857,4.4286]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002187889837417012155249758730889981624455,
-0.00003022807173388076940326981466760752166862,
2.090408796884806039708148995543657102279e-6,
-9.647122079725176066373099865247919623621e-8,
3.342230681726411634174121905241267637285e-9,
-9.271542478314873196988496263090019380161e-11,
2.145488763483180546272410519034989043661e-12,
-4.258014842708474271101248512916309242403e-14,
// [4.2857,4.4286]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002115965021371653662724913990543019934082,
-0.00002926572315634681592324527644106531286707,
2.025895636744163822421476318277794993529e-6,
-9.358246070710007528555953145252030843461e-8,
3.245039842385476440735135054053051947546e-9,
-9.009498786746369847662852474621913497753e-11,
2.086507352237028049114571574539485063548e-12,
-4.144071603167963163804028023692060903963e-14,
// [4.2857,4.4286]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002048600620944276924090463176133343857496,
-0.00002836253891437887974401634658933880855386,
1.965231674850930743467887279238121674269e-6,
-9.086111716335900075651424039473701828077e-8,
3.153324550972315762610649788007118765239e-9,
-8.761817366847828395345736813047790442963e-11,
2.030673130248327621224831659049641534162e-12,
-4.036051581891105612868159167750219527733e-14,
// [4.2857,4.4286]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001985377724006521396266150980829802500262,
-0.00002751324344787461804375066289763607847233,
1.908083460432322849675818400364369905109e-6,
-8.829308896373393887910745394997448669557e-8,
3.066636054612210589210891787025950448647e-9,
-8.527352089025747106587078626918839089186e-11,
1.97774138578872811655686107969580965708e-12,
-3.933505729068702015237592857222798433516e-14,
// [4.2857,4.4286]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001925927041351223081741440821186182929026,
-0.00002671316844601505943187398181340532984686,
1.854154868240153408701009122765290193714e-6,
-8.586581102757460296577487914758932592686e-8,
2.98457320794970461391874436906272399496e-9,
-8.305075315850089138074562211806210133181e-11,
1.927492077189947578599704486468969918517e-12,
-3.836029165699376320147486710069076245494e-14,
// [4.2857,4.4286]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001869921791221053983192201357733404848699,
-0.00002595816815532559322970027392250707499722,
1.803182031580766245873061493093196492845e-6,
-8.356805128478524991908647463452778508132e-8,
2.906776337886341368127402685001596072665e-9,
-8.094063009562669086679146572962078344448e-11,
1.879726807160734155405868130004407452179e-12,
-3.743255894096477199692622799454363722386e-14,
// [4.2857,4.4286]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001817071770872791353125003344830130559647,
-0.00002524454844209503623070302129320970444136,
1.754929076982150616054261953763275136139e-6,
-8.138973888847954319315531392741692170134e-8,
2.832922031208623855240619927036026857278e-9,
-7.893482025914117244934200507965128584771e-11,
1.834266230857354787411981816374686732378e-12,
-3.654854249262916035071572035416541976831e-14,
// [4.2857,4.4286]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001767118390946652367096275146921873788758,
-0.00002456900707771524094487642279045563523463,
1.709184516659693324248019604659309067451e-6,
-7.932181826856188902618472500353412505601e-8,
2.762718687430973122117512852084311008358e-9,
-7.702579229532118099097011378257140223084e-11,
1.790947826989803215437009023574322004908e-12,
-3.570522973047851155813557086553799703514e-14
}
,{
// [4.4286,4.5714]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.4166434815805158410529965651626247378141,
-0.006437055318131325945742309905119232415919,
0.0001406679876899466657877994538113108013259,
-3.121602279982007934206433543501426835405e-6,
6.529070777281709875338311329265942896869e-8,
-1.25343547736347512151241248762434740411e-9,
2.190641810151193382005490913432517424571e-11,
-3.483439856147316813834606422360346765501e-13,
// [4.4286,4.5714]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.04505938722691928150517492506452049701586,
-0.001969351827658976798356802492104973443194,
0.00006555364787963320870585389544083113200359,
-1.828139828699864665190256987684360490505e-6,
4.387024156049341111631080011522217096539e-8,
-9.199633742932287880005047582596217378598e-10,
1.706932642624027173903038356193392937509e-11,
-2.831635505038968585566747827719632824769e-13,
// [4.4286,4.5714]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01378546279361283748974114469814815766204,
-0.0009177510703146330382683758614006105334803,
0.00003839093640270663863303714166376761992207,
-1.228366772969229145887349914394432844311e-6,
3.219871797385381242297926604956001730898e-8,
-7.168226654890110310284514944303728069623e-10,
1.387541848484348777882727724170664941895e-11,
-2.374526104798515306138134066148906838085e-13,
// [4.4286,4.5714]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006424257492202431181513516329388246285148,
-0.0005374731096376938471167099261196121780986,
0.00002579570223236210315295581207627725070205,
-9.015641112317178947270196141952715873298e-7,
2.508879318156722198237665327346665875067e-8,
-5.826911233840015176440066821990120868352e-10,
1.163553166829169179311482345374759537674e-11,
-2.038753592593654603022005687957134920125e-13,
// [4.4286,4.5714]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003762311767463856853173178428394839857524,
-0.0003611398312528953314444076313394981336505,
0.0000189328463358734336279547656335676853444,
-7.024862160484451595772516968356529764249e-7,
2.039418922033525931501224990534513052755e-8,
-4.886254699115691834191133232537023656448e-10,
9.990206539642083021922001375422944223823e-12,
-1.782944224278557212694817510626894423218e-13,
// [4.4286,4.5714]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002527978818770267251278821460603232045316,
-0.0002650598487020735563410771624338500395035,
0.00001475221053702395625796542324942785945854,
-5.710373043500152146846201259911796107723e-7,
1.710189135879924264176723285427570910294e-8,
-4.195293403169779269803478920646994121762e-10,
8.736708637679320894436165391043419654394e-12,
-1.582255367980220487329396951452358648425e-13,
// [4.4286,4.5714]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001855418940914514831959106996200853485889,
-0.0002065309475181966217160903656753312062167,
0.00001199178339135631266722135709103538513355,
-4.788529635970602828728861654180009419584e-7,
1.468352683118520874883758717142617346928e-8,
-3.668884759479362450156392723551682127258e-10,
7.75330701246233047463797728307928194708e-12,
-1.420987823545110403656798537691319174134e-13,
// [4.4286,4.5714]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001445716632627376294922243351554331557072,
-0.0001678849674788625211419245637666906167932,
0.00001005591223554374664480676618468923710994,
-4.111387563074758728076117136564923648398e-7,
1.284109658510149199436396519234434423138e-8,
-3.255905650699594896186415285007998658161e-10,
6.963074179918820663468027758617041158427e-12,
-1.288790553025394101831498715133122557379e-13,
// [4.4286,4.5714]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001175194772352037595416892499669900639553,
-0.0001407827712974973583598159005171432066745,
8.633913882462040705838073586827289711804e-6,
-3.595507089866674446398212959607735996604e-7,
1.139566971015002174791381542026881315136e-8,
-2.924049185807627851842587850740159990319e-10,
6.315289065648636507280785020376391657027e-12,
-1.178590772702499670653623213454627853421e-13,
// [4.4286,4.5714]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0009854793990824814598061867924869180987277,
-0.0001208747943543625750851270362579616471401,
7.550564888724692763415460637673929372881e-6,
-3.190787561240287681714765084543234309867e-7,
1.023417208797416208125877545719200636003e-8,
-2.652014381746247803217088645195763861976e-10,
5.775294314758753397014412210792802561291e-12,
-1.085406627720490431132136564572898712429e-13,
// [4.4286,4.5714]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0008461235604805379802267112094350699713941,
-0.0001057079084420474938525056472843927776274,
6.700653878608959594888235010713886176394e-6,
-2.865568223915039097208922251396071066799e-7,
9.282050278038845920360285374134550272128e-9,
-2.425246500197627489576374897764151687959e-10,
5.318678309877652727309345585862497670224e-12,
-1.005636142603121433333591615183830718376e-13,
// [4.4286,4.5714]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000739955359094332414518357365391513354003,
-0.00009380915430043396970297749871946779360862,
6.017693270225657246552009373537542451814e-6,
-2.5989741144370460599854826000856807353e-7,
8.488362696356296632183339585184120604463e-9,
-2.233493660882536971740111475244867737703e-10,
4.927790972391086060212576443722619487298e-12,
-9.366152652396657193261062421843072769424e-14,
// [4.4286,4.5714]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006566640801030377480430537911023291505487,
-0.00008424770578307362355869876847264569920936,
5.457845640321625013476419343108861130658e-6,
-2.37674158921122103945277350564121288039e-7,
7.817227762044887640683239008104690230451e-9,
-2.069343584482680331302473792729070914844e-10,
4.589578140785778035161208939831231436706e-12,
-8.763341241098430147298806466576818003492e-14,
// [4.4286,4.5714]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005897339404815153273141818451359436125429,
-0.00007640983896442235624338390017019985023537,
4.991157337347173485739548170067053836192e-6,
-2.188823805530431129799364639174252718416e-7,
7.242702497565192454458340672878973897193e-9,
-1.927314101828500994846553150009219766629e-10,
4.294191205862491149415856833833755950662e-12,
-8.232496053347733171317689967127823361033e-14,
// [4.4286,4.5714]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005348688727509564581438305776820174377659,
-0.00006987620272278463352729771151221448489082,
4.596529991617319136678062881467490262386e-6,
-2.027956729636632949586901901789018404706e-7,
6.745599310882755990630821653256610829034e-9,
-1.803269248296261324553144439401408090946e-10,
4.034068720837072544848512878170065644608e-12,
-7.761583988036821626687269030675083732671e-14,
// [4.4286,4.5714]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000489133419059492400960752109706329365986,
-0.00006435141988257077895072186022428420861767,
4.258709132240167323011700579264409310713e-6,
-1.888767835723013409281091159172754172028e-7,
6.311442325861726755232939480742733892275e-9,
-1.694033573028543485669527493313563135929e-10,
3.803314315029732343751420637143090556375e-12,
-7.341090564660395922227804923286010158868e-14,
// [4.4286,4.5714]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004504599391779954205767933637088489223231,
-0.00005962192785129434189512330053690598559976,
3.96641245502140768867664627027050880543e-6,
-1.767203878441778938823227814687657490583e-7,
5.92911746453938297375953455256370315074e-9,
-1.597130885990543674185893307392559059801e-10,
3.597265770622611290540759878235308940582e-12,
-6.963398407174791386732343071610865453844e-14,
// [4.4286,4.5714]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004173534949590603626866599317418833063725,
-0.00005552977437023503760443396548010722641896,
3.711128144730671390596676658678538043943e-6,
-1.660152915939275749816393710251495740822e-7,
5.589958061825190779554784863371237383999e-9,
-1.510603286980032697401113849849521392856e-10,
3.412190473259922518490823701866485111429e-12,
-6.622339995326111081998221578202626557545e-14,
// [4.4286,4.5714]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003887084205916452340180502881327686836004,
-0.0000519557940261677515405549151845954574278,
3.486321123475283536335878856592845602616e-6,
-1.565188281970448237937582246395205698759e-7,
5.287111467037172092725978509401878428511e-9,
-1.432883267147208068667072772485464556762e-10,
3.245066255380020033909228462626456345033e-12,
-6.312870522652141365142253153909417565183e-14,
// [4.4286,4.5714]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003636905581831742328210700938148310479417,
-0.0000488084957285950758821968374313903420284,
3.28689539214062574417826645835315972743e-6,
-1.480391234328073302662425021632477158655e-7,
5.015091399222520098685279737338078995903e-9,
-1.362701672301586466138262921998585037707e-10,
3.093421093010262207469308034855858286883e-12,
-6.030825031192710146004081640397925996198e-14,
// [4.4286,4.5714]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000341659470100165504360790165742671061329,
-0.00004601653548991238715069610298400970594468,
3.108821592091528149800785699384539437852e-6,
-1.404225614331818796526847464647327675207e-7,
4.769455818732586174505935062880791484054e-9,
-1.299020382419166418319009527438785749702e-10,
2.955214099012457131343275265147314906951e-12,
-5.772735655782301273973354132848496973871e-14,
// [4.4286,4.5714]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003221157484293866842984786080070740404834,
-0.00004352350228922733566316311603432761373325,
2.948873790099292099827211347891393926711e-6,
-1.335447650868695970763223725254531933122e-7,
4.546571305498614564167109665402212594301e-9,
-1.24098233402691601506557220552999554455e-10,
2.828745970657357991065365838647422398738e-12,
-5.535692386304376951999647196323200670635e-14,
// [4.4286,4.5714]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003046645160245913248640440044994031380169,
-0.00004128423306133816429004322973125593631502,
2.804440066826640248808935268978436016475e-6,
-1.273039986309846059273693524771546391544e-7,
4.343438137377967436675010499332647917852e-9,
-1.187873912207678172642389922485644040041e-10,
2.712590761471455885744440951953018266034e-12,
-5.317235770922100378306888349130882318563e-14,
// [4.4286,4.5714]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002889896314293671261592800822626883227528,
-0.000039262160935523010628805001073154008934,
2.673383971252968355016372934865059910196e-6,
-1.216162698447156909998008958408934952514e-7,
4.157558662171700682110983538435179292974e-9,
-1.139096297909131644965168533604689138548e-10,
2.605543304516453279457336478440382578777e-12,
-5.115273360951364431598239605962904398479e-14,
// [4.4286,4.5714]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002748351265486610513739360062602729908247,
-0.0000374273755974934327866230937035860460165,
2.553941666741240182070340950366981026236e-6,
-1.164116444657927564906117646084187466188e-7,
3.986837013206250666384468414603794893415e-9,
-1.094143388184952065547030747832263201693e-10,
2.506578269342757222697451907328068762964e-12,
-4.928014009805302010601002878393140169741e-14,
// [4.4286,4.5714]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002619916291824540072646987282791956268647,
-0.0000357551833343309385122802561051763653253,
2.444644533783783097531592050542454030057e-6,
-1.116314382267536924948632365034126539234e-7,
3.829501830177756477161928315656713932202e-9,
-1.052584602070100692604813239444674826678e-10,
2.414817967644770986196692568458415417536e-12,
-4.75391574425962652393705508717283639254e-14,
// [4.4286,4.5714]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002502862833403165480786697383347642611868,
-0.00003422502347292812398371314990405074478746,
2.344260202763892253657952537177997254372e-6,
-1.072260530385691060923086514923296985114e-7,
3.684046079715779050204574701066710714068e-9,
-1.014051360520066410670522885214239439072e-10,
2.32950680951365078294552283698143430542e-12,
-4.591644057136516028050558778227745781307e-14,
// [4.4286,4.5714]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002395751643104968470663181775286457594192,
-0.00003281964283865113266702781757456863195385,
2.251747113811949927521195398953948862208e-6,
-1.031532919664133201765438895159472209358e-7,
3.549179735170817538916404099286575149012e-9,
-9.782263592981530911956379946925125735238e-11,
2.249990866310443644459502009505029533497e-12,
-4.440038277078093006844598065632668612151e-14,
// [4.4286,4.5714]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002297374998705579084946973153675164455839,
-0.00003152445959332532634488320531240733605565,
2.166219131296616486031634295850559114492e-6,
-9.937703426370416183914334275052789422326e-8,
3.423792231719953577668194243486588783131e-9,
-9.448349864065185281256767316152868845289e-11,
2.175701391411887943806688343381975454939e-12,
-4.298084253232175881413649102481445042338e-14,
// [4.4286,4.5714]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002206712171532772648462182623368889539889,
-0.00003032706783811195884497417431844461860772,
2.086917719539665933455908431979991535549e-6,
-9.586618411505227613592658983115987163477e-8,
3.30692242737560154418911614109020218082e-9,
-9.136384016370726859390549206621820285634e-11,
2.10614143534166983421318130693259801159e-12,
-4.164892017526282978718003536317058812892e-14,
// [4.4286,4.5714]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002122894748667836929181047808679273911271,
-0.00002921684807351587388481081099986013087854,
2.013189866417921493430284065789391016183e-6,
-9.259382954449895217193038523476615975145e-8,
3.197734381413746805246877696198335055571e-9,
-8.844279156160096985079421813152672407009e-11,
2.040874899982676564293162127805174974755e-12,
-4.039677400493235987328886596370457391699e-14,
// [4.4286,4.5714]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002045179365146110987359098495856610635031,
-0.00002818465812981260336854878625157625381347,
1.94447042043624995085132953444427103393e-6,
-8.953656421150084817381425558851033854829e-8,
3.09549768102988629600993802761442290095e-9,
-8.570203931437543300210030672908178897512e-11,
1.979517530080075043165344161952186547865e-12,
-3.921746809885880307223101246856843600474e-14,
// [4.4286,4.5714]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001972926069086882056313795697841906086631,
-0.00002722258588607028829722716530557484328844,
1.880267848443240873462827942695409945495e-6,
-8.667393655729137694303344663279901964443e-8,
2.99957135302890691483762051284680830774e-9,
-8.312544701015310070515537174222351236686e-11,
1.921729454552023471016692831934183229997e-12,
-3.810484556607326886052034694635923885872e-14,
// [4.4286,4.5714]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001905581012024920006141610706345906197542,
-0.00002632374987816918797459079195307049069608,
1.820152667704795702221710558110104678303e-6,
-8.39879993321931566564603789220035292323e-8,
2.909390622998132478303394081358935364914e-9,
-8.069874211997129108374276876144388225959e-11,
1.86720897601839263292983578168449450008e-12,
-3.705342245336443553526903363612742842682e-14,
// [4.4286,4.5714]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001842662491471842988125957472674667699037,
-0.00002548213734783192735937794378194761356147,
1.763747985977689214602434025645716650045e-6,
-8.146293885245983899870079847069123106114e-8,
2.824455952426592326603647012702221159326e-9,
-7.840925519146009098898356668353998133335e-11,
1.81568737205755262775696025134349513016e-12,
-3.605829848737876766477169798641808734148e-14,
// [4.4286,4.5714]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001783749614348234749397638821322436358715,
-0.0000246924718036533576252636848997284222228,
1.710721715903247912396275984846572898073e-6,
-7.908476803961268280949844712665824516651e-8,
2.744323910483787312370214683879952211621e-9,
-7.624570152992165290861593972144620845802e-11,
1.766924521442164327083506307009674891126e-12,
-3.511508162301762345360734026306144886035e-14,
// [4.4286,4.5714]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001728473026255734872131145795710938600318,
-0.00002395010402261205365261616019118919909363,
1.660780128833418065766133043928106014865e-6,
-7.684107083024349910613268012298125276666e-8,
2.668599532857499068583270394307325790873e-9,
-7.419799752425913005669801518594941128545e-11,
1.720705206902233969137406512998941934374e-12,
-3.42198239746994227943362036802064962279e-14,
// [4.4286,4.5714]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001676507281582843597967824864140568499743,
-0.0000232509218036352666986184185438650859508,
1.61366248743662755654142538244197402809e-6,
-7.472078822347117638333356897888614709138e-8,
2.596929893161331742723959302008273554422e-9,
-7.225710538351340484514153159647765081448e-11,
1.676835975664139991992008961673358202935e-12,
-3.336896718034161703516409046104478235461e-14,
// [4.4286,4.5714]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001627564526254468514924600326021334131599,
-0.00002259127482408099024908138632493877664438,
1.569136552694372907646413537372725678036e-6,
-7.271403828035102178375103693020853374382e-8,
2.528998668713522425347850153600602966012e-9,
-7.041490129704520563838597653798997357509e-11,
1.635142462205966790716365892455662344048e-12,
-3.255929561989078359078411066783207514729e-14,
// [4.4286,4.5714]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001581389237685669167021050494546851036006,
-0.00002196791173769017846970817247284248883537,
1.526994803888815446066866012753058108357e-6,
-7.081196396567988237156992807668214208643e-8,
2.464521526143336745677922466528671363655e-9,
-6.866406300533817287728886370754498745856e-11,
1.595467095896649859214897648012115175368e-12,
-3.178789620438522087599921652538578039902e-14,
// [4.4286,4.5714]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001537753821638312345868054530939361486431,
-0.00002137792725441309252150425113345197779918,
1.487051243280688848199382407217602301034e-6,
-6.900660394497386672556031077433155909016e-8,
2.403242186369194807234084791745720382664e-9,
-6.699797353385182184813052921119388183356e-11,
1.557667130600454698626853113987787450919e-12,
-3.105212368552021242867924859726185353189e-14
}
,{
// [4.5714,4.7143]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.4103440367066277146454685860183603995503,
-0.006164829124911141319276857126750386165678,
0.000131682712127193831013130472431616464021,
-2.872547637619953215742118361616212806675e-6,
5.9340243501170184151159824815138096858e-8,
-1.129051796305643460337614846707091131292e-9,
1.960404412204315992945330527315548569019e-11,
-3.102189881441119876718290072046668673319e-13,
// [4.5714,4.7143]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.04315380387437798913317962414075752122953,
-0.001843557969780468288161898430122810064596,
0.0000603235003900287863770054580650150099101,
-1.661526827846675767394848968971757704909e-6,
3.951681274044589670193265237888933481047e-8,
-8.23275639137645351615150144824293998073e-10,
1.520114722498236839102769193617117882329e-11,
-2.512381518779957057290990931057477533375e-13,
// [4.5714,4.7143]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01290490578846327792996500069416876040854,
-0.000844529005460197863700861818617406780824,
0.00003489206338478855930849502134381949997767,
-1.106470764938369066903712451659486970932e-6,
2.881464725824137773008216175822669555173e-8,
-6.383694065736762780436711890248364215037e-10,
1.231102648653286440037315478664790536497e-11,
-2.100724102707667507337022255017706430623e-13,
// [4.5714,4.7143]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005911703038221384969817032644942182788176,
-0.0004884888873868640984367750573955155613099,
0.00002323588606371305498264312426966490954658,
-8.068101302600877988211634256805016065831e-7,
2.234292913268403160493589717804038114585e-8,
-5.169956305286520487716374877046282981396e-10,
1.029385976668399458701466910634021198203e-11,
-1.799524061221696250206487728360149956965e-13,
// [4.5714,4.7143]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003419422211708048621636608875626981333171,
-0.0003253024048918293737830976208130635961612,
0.00001694301273546831620412362231297132405743,
-6.256020218510402179529798585555072029164e-7,
1.809484698220352604115959206234740808958e-8,
-4.322832053727177338597780269395754634269e-10,
8.817944058252856719527431662956154370638e-12,
-1.570801311013573114107678049842403350682e-13,
// [4.5714,4.7143]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002277116834242805556010168053422310834579,
-0.0002372021782964205059935636308812697970349,
0.00001313764245887764987034011243767281298207,
-5.066557209385771244334983736332818630771e-7,
1.512991211064098677374117755356043845736e-8,
-3.703014561335206309139157158948451299287e-10,
7.697174117670498519871707699456740692342e-12,
-1.391853692280416590039237114156800114309e-13,
// [4.5714,4.7143]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001660415248074943487168689544779421424998,
-0.0001839269944241651871548912070251003969482,
0.00001063977013971537911930332762889245944156,
-4.23637543974428868253007205374897902957e-7,
1.296055089454626765922380640166472088179e-8,
-3.232344984659888327319272870166664477361e-10,
6.820307498865723894950796587895600875182e-12,
-1.248390912828747032282611243676483212243e-13,
// [4.5714,4.7143]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001287488960969156260029050540297832316773,
-0.0001489567819559048581596823186014570667044,
8.896388423467811555087892658241286415394e-6,
-3.628954294653127651947519896526003592617e-7,
1.131320738223846226365212560810558656065e-8,
-2.864105017507316697640890009781326492314e-10,
6.117320500935973232177307891033310286091e-12,
-1.131023202109191482270712650557107085019e-13,
// [4.5714,4.7143]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001042697473691333961057130680140791123235,
-0.0001245494379284484501247239169638381973904,
7.620804018775989913155425432998098894964e-6,
-3.167698107391769198187576630975381056085e-7,
1.002436750231751082260202079277889687595e-8,
-2.568887104211922646029159602632696039375e-10,
5.542202356624474940108574754988096282973e-12,
-1.033353975460255480106597918449906955027e-13,
// [4.5714,4.7143]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0008718460654991391082274408912652401275109,
-0.0001066912562627710001618950413914174511452,
6.652166025526809317000208394766303649098e-6,
-2.806822937792669493675694926147751264269e-7,
8.991104810154886565199958912808799673204e-9,
-2.327368407591892745677395050122689930727e-10,
5.063609158004474508473762115466241417277e-12,
-9.50889712166490677976814398397685207447e-14,
// [4.5714,4.7143]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0007468387938393969614390077326854485661513,
-0.00009313032435728935651748008860991271868556,
5.894328169368416604843201674727338996922e-6,
-2.517509381233233398774243721390847870886e-7,
8.14578937576256109526285358155629170431e-9,
-2.126385701755837836429417239783871049544e-10,
4.659522178950127345223930086159403190586e-12,
-8.803892361247355206419608923240919829614e-14,
// [4.5714,4.7143]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006519122705010254585030745595294882024063,
-0.00008252059437107780852772110667586237894679,
5.286769700593353607168908341733740282478e-6,
-2.280821057223371968119790416882105479864e-7,
7.442349908632356270367521733204593555762e-9,
-1.956692018767873686923635018474291365319e-10,
4.314059299166909253919207093415235413568e-12,
-8.194604422251739750628451435528001885692e-14,
// [4.5714,4.7143]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000577644160597544624840209872234642747962,
-0.00007401477580823211772107181089996980132783,
4.789724220172427118771490998491642732486e-6,
-2.083858004350434237035545543170142314295e-7,
6.848422021074276698849589596366761380675e-9,
-1.811617543573379997834746726003140935035e-10,
4.015498929698193243817236323561203079716e-12,
-7.663020745104442558122359365263180683732e-14,
// [4.5714,4.7143]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005181034306576247912016562109100655699891,
-0.00006705613908234371404333890951584238388153,
4.376101809139065115424235179037032763049e-6,
-1.917558194007293691604608566980353442831e-7,
6.34066136046379688551826727455756821419e-9,
-1.686239726519599071061654305080139250192e-10,
3.755014703086826497803635044375135120677e-12,
-7.195335715696350229161004780091407210639e-14,
// [4.5714,4.7143]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004693929735764059672491705964450083568263,
-0.00006126542532788069412177761354767901303407,
4.026872207418297971320782282888776075491e-6,
-1.775385207417096358527400739004149780406e-7,
5.901839003068886083614283818635983140361e-9,
-1.576851896349822282930766881867292376017e-10,
3.525841700031261842073456737014965278123e-12,
-6.780800627422808913683794826925028359056e-14,
// [4.5714,4.7143]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004288579772951648294064098381762596182479,
-0.00005637621090379356607322807716532719172867,
3.728308935578729186968040261413364995976e-6,
-1.652514945901722164384885207143340290204e-7,
5.518981599533138393345182069838069414204e-9,
-1.480613105212818362914012703143534830201e-10,
3.322712919657306767620571254140082218606e-12,
-6.410928643198639702542307901594692700196e-14,
// [4.5714,4.7143]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003946334763265549345178372894096041820628,
-0.00005219632509804284517157410942938642383381,
3.470281386396304056277746308977039806394e-6,
-1.545314871614870568641067682274662348056e-7,
5.182145832411269891430353210811417111224e-9,
-1.395311467208668997977066585952052672702e-10,
3.141469702911378032836847615942575109134e-12,
-6.078933948707699072036550011452633271881e-14,
// [4.5714,4.7143]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00036537427568629988952235915434064859275,
-0.00004858393940949181912160749809360722570866,
3.245161230393789359040104633211493011603e-6,
-1.451000855650425918484355545967450402375e-7,
4.883590101081367671370394122255496035291e-9,
-1.319200551322867603810334939823724791498e-10,
2.978786911814618577068788330231857186865e-12,
-5.779328846523418749500402024642885355087e-14,
// [4.5714,4.7143]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003400875758664427083714313613512816335479,
-0.00004543225722545926662712076222848064839491,
3.047101796868340504656063050586576168803e-6,
-1.36740524981673778902127312295329669968e-7,
4.617201897015588006479571866455231984266e-9,
-1.250883967747243466933615459606542106922e-10,
2.831975501254072398631114997808249567628e-12,
-5.507629510133416784879183207830994621922e-14,
// [4.5714,4.7143]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003180258005782148420065028576154448791648,
-0.0000426594251561053995332835834095736349625,
2.871551024617490170317337118862524161854e-6,
-1.292816551711564414268474837544026799626e-7,
4.378093855904407206362043920506368250367e-9,
-1.18923245621201083616321214979584298535e-10,
2.698838335202390410626304413192662164525e-12,
-5.260137856533123801112226867332582536705e-14,
// [4.5714,4.7143]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002986159760927377733562548235089510969142,
-0.0000402017143445957053619242724160986633622,
2.714914758596529448086611638954930389663e-6,
-1.225866299316222618794157640720389809424e-7,
4.162313566819566812339221294537371496223e-9,
-1.133323334945027643183120770361855204192e-10,
2.577563301813902931434251888020889291611e-12,
-5.033777626245283102366365004551241780704e-14,
// [4.5714,4.7143]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002814120004121699150840359915804525893402,
-0.00003800880662030428458871519931994046374744,
2.574319228566222656699348816916488082501e-6,
-1.165447817560725244500070806819831545787e-7,
3.966631643572074389486676217912803801763e-9,
-1.082395613689552671859241426778716045939e-10,
2.466642990729307917619078025970293668074e-12,
-4.825969652121197716404427298617276147383e-14,
// [4.5714,4.7143]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000266061646342129970528522413000815052945,
-0.00003604046919988185894594568631567105036647,
2.447440416879595903077324154415376234175e-6,
-1.110656878303646351087110448604372822873e-7,
3.788384620274816669365941987807021514018e-9,
-1.035816261772433690838646407769876668545e-10,
2.36481357330651945246779770832169549427e-12,
-4.634535853221606473763347253199954524917e-14,
// [4.5714,4.7143]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000252283284399172991823411873702737092888,
-0.00003426416583627081201301330347779636355642,
2.332379444439653976026307081606630773993e-6,
-1.060747711089361173500927940934114895481e-7,
3.625356889581581749161661972838571221534e-9,
-9.930545406016115366918210029660092056885e-11,
2.271007758458343187352510816303316253858e-12,
-4.457624553324533759527414783515488477287e-14,
// [4.5714,4.7143]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002398491608538956640310882136977382407748,
-0.0000326533122221132263010723547816213641779,
2.227570193289584235247852346335223872012e-6,
-1.01509994585474318386702650326516448577e-7,
3.475690866428611703142691756840157078538e-9,
-9.53662247317829721877299267709664516766e-11,
2.184318197798695859704628699613145282492e-12,
-4.293651817125089176750317178601270621984e-14,
// [4.5714,4.7143]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002285731855547925647353407856313708838123,
-0.00003118598270601373815330229308575043477842,
2.131709886296820424117522031940387100612e-6,
-9.731934587766171802051525304148703715788e-8,
3.337817840815816684139944917118943093105e-9,
-9.172583466998220088349854531584894532361e-11,
2.103968739642990484577605695906826684259e-12,
-4.141254949417287726218543759253215155312e-14,
// [4.5714,4.7143]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002183018789420961483433884236989633147575,
-0.00002984393840811643148679761951116523731327,
2.043706263432694142058906135259221831148e-6,
-9.345890110503540359847993735615472738156e-8,
3.210404189475117115984273044499131536613e-9,
-8.835168992383892578667196147087905982317e-11,
2.029291643013041372862600730028229866801e-12,
-3.999255323910769392244087399521563210921e-14,
// [4.5714,4.7143]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002089075688568150022792264703116731200516,
-0.0000286118876880199586982541472073931978966,
1.962636923207483807307959268664044820618e-6,
-8.989131881568895332479097874280361310669e-8,
3.09230912412986335772391979498588011496e-9,
-8.521574921414797555475827730028759629332e-11,
1.95970936327502674323250905396089640533e-12,
-3.866628435931089397421286906531339341859e-14,
// [4.5714,4.7143]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002002832138161396933235268274715275482814,
-0.00002747691692486822638001615567698992102625,
1.887717695131154197234958223837913622191e-6,
-8.658465693752669293805971669963014834768e-8,
2.982551200012739996256588180103632431304e-9,
-8.22937590213553970746348284564429503121e-11,
1.894719877568572109117122009532477268426e-12,
-3.742479597796982422983773443552598849297e-14,
// [4.5714,4.7143]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001923384184740775676260471585836244433685,
-0.00002642804773180074907853369571540209163329,
1.818277795689695830969418330144331748139e-6,
-8.351143501675725785667849146556667752527e-8,
2.880281543943643768620095316603715945588e-9,
-7.956463732805755690424058332583032923776e-11,
1.833884775217525328178311134875445273765e-12,
-3.626024078063401077992511444037443392702e-14,
// [4.5714,4.7143]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000184996334122605227014897488622440868052,
-0.00002545588913962140081062093995262852554775,
1.753740135353489768168014382878322302891e-6,
-8.064788460406779457851821635163728885939e-8,
2.784762285317235908944431196100585347017e-9,
-7.700997347750604689379036976338306159972e-11,
1.776819525690896242272575674684227039441e-12,
-3.516570767507444789673042680525651599651e-14,
// [4.5714,4.7143]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001781912239773497896104031330731969488734,
-0.00002455236189491552237818611407897973671673,
1.693605576686965833178593959909931768788e-6,
-7.797334532227014719759853267368811127198e-8,
2.695349051150683573840909492563432435042e-9,
-7.461361947864768307825319389141899242948e-11,
1.72318547471282980043230510678369045513e-12,
-3.413508664274844660618087059246909984307e-14,
// [4.5714,4.7143]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000171866533264408641028259789720736322944,
-0.00002371047807358513661634993349979146335261,
1.637440251769172527515204634349212208197e-6,
-7.54697747276332243943200076863941975304e-8,
2.611476661760118137730272594735188666759e-9,
-7.236135388491561141117468345287623584104e-11,
1.672683221797714239006281793918107078574e-12,
-3.316295627913374781494740997251792979958e-14,
// [4.5714,4.7143]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001659733465150959411164009144815662907571,
-0.00002292416352473692725997186708589472287543,
1.584865269281756732974306930311738516944e-6,
-7.312134778882015664344085661159351172861e-8,
2.532647366518372942491495784173791524526e-9,
-7.024060368598460470497881951164727217657e-11,
1.625047109569595675832180719899632406818e-12,
-3.224448971140379667844134009836155571504e-14,
// [4.5714,4.7143]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000160469144673158476020713942031355565321,
-0.00002218811376991395486452897440864991906419,
1.535548303566644009894332527083712436562e-6,
-7.091412748810186310802869425885420347667e-8,
2.45842111006646012984940379010402480957e-9,
-6.824021288885153992446191784648342468775e-11,
1.580040613595660005231289384713653910736e-12,
-3.137537549132155121976648314857624155163e-14,
// [4.5714,4.7143]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001553167963893976696312196224123737720677,
-0.00002149767624990318104701667189873180049374,
1.489196677251523499290967516511636273093e-6,
-6.883579227527579224952665664416298734701e-8,
2.388407432651422516116558930757428892005e-9,
-6.635024891590444683852884477257160373758e-11,
1.53745246602548702723412142596181034863e-12,
-3.055175076085359184998012833514651716503e-14,
// [4.5714,4.7143]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001504837337493222532684008032388722763109,
-0.00002084875348149225717123442045172858427633,
1.44555163778214147342029835969166449286e-6,
-6.687540927712355863561205220543192383576e-8,
2.322258694058780271648667114231567486854e-9,
-6.456183981904185924311262783510806716975e-11,
1.497093380610216387657551959507314617034e-12,
-2.977014453016720155982279739509271560181e-14,
// [4.5714,4.7143]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001459412743704457864802222857640072016067,
-0.00002023772292892163410288281863963430074417,
1.404383594820911706819723835650978129925e-6,
-6.502324456751756142917914877074477987796e-8,
2.259664376106733868642574686293222192112e-9,
-6.286703674860781598879773731042260065721e-11,
1.458793273242308379851250568746335598397e-12,
-2.902742933071164512931084208436349412844e-14,
// [4.5714,4.7143]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001416640605024514253278276973364445402982,
-0.00001966137032746510744454284568762777771133,
1.365488135919154462923468704231763150787e-6,
-6.327060363725705065577582822961964753294e-8,
2.2003462690589111588251473790187476497e-9,
-6.12586972315780913926962703824640458336e-11,
1.422398892885992993296897115140849834773e-12,
-2.832077983833538240177293005647865132317e-14,
// [4.5714,4.7143]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001376295922922557390304090416703788420327,
-0.00001911683390284116338300864151056982231424,
1.328682676383653884253201717684482577823e-6,
-6.160969661362365674506174112237994658522e-8,
2.14405438636090444527510530295127323484e-9,
-5.97303856839700058456869395523303001345e-11,
1.387771794049189558308339023684476612825e-12,
-2.764763732395923704316217656783883904417e-14,
// [4.5714,4.7143]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001338178373198881308965490094722382189243,
-0.00001860155746934478218189271636160764242423,
1.293803628887324111555015819913538326256e-6,
-6.003352387300323107059404311267582255817e-8,
2.090563482574630302266959378834620090696e-9,
-5.827628826615883178392226244551843831571e-11,
1.354786594813856433083939503872700324698e-12,
-2.70056789980518635026125658285977983054e-14
}
,{
// [4.7143,4.8571]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.4043080759766942195596630448448718900453,
-0.005909849512378405504121902935047455864834,
0.0001234101077959702075496444678043528695015,
-2.646095754527402969386502161718608669803e-6,
5.397846159291404586411535844595193492392e-8,
-1.01771107353678188762957205627156149801e-9,
1.755322179324698030433433823691639017856e-11,
-2.763866375678587799495876174991353170183e-13,
// [4.7143,4.8571]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.04136894658664883843880287657413429169218,
-0.001727741509143365145238496685607238133091,
0.0000555680108450841072370554977555066746049,
-1.511396933312073111058940498032729467955e-6,
3.561988745852200167044869762347554073942e-8,
-7.371516943160118783413096098487535280408e-10,
1.354331409062804521454337603821499143409e-11,
-2.229900848199480870285748640048785725924e-13,
// [4.7143,4.8571]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01209419056400355593971153528518707989184,
-0.0007779521518309959590142045553762737471262,
0.00003173933559956092332763779920064442069826,
-9.973568561003618681887440535093346966894e-7,
2.580030920255354628842841750662380095924e-8,
-5.68749478703203900247577328941580603301e-10,
1.092682937872316460120206843177753713092e-11,
-1.859022623488158055549979029412008740696e-13,
// [4.7143,4.8571]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005445665062816971646050695549610564198474,
-0.0004443506983936977788375933549175356720185,
0.00002094449397811403594022516916452310075278,
-7.224086638774567231072967601048086618551e-7,
1.990623166878912310155215731508159156214e-8,
-4.588672564113686866616845576437758008006e-10,
9.109485489239597049567014695941305276626e-12,
-1.588738984984054811304418878242921704786e-13,
// [4.7143,4.8571]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003110454888755884392545625032692386099681,
-0.0002932229156934613324276646074392268747336,
0.00001517058194143228569474778754838238812816,
-5.573744921329673731773972995082039573405e-7,
1.606035389847089666526961182644323462105e-8,
-3.825464843055250488287330029320292223157e-10,
7.785063993302209478734636084125701848931e-12,
-1.384171649512195145528593307618605123536e-13,
// [4.7143,4.8571]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002052560409854229273859638168295301437939,
-0.0002123881471799324151519753857812847233279,
0.00001170486433479741572650636107840460174078,
-4.496899139406065543928545485220196604271e-7,
1.338912688268131752879166430961582880854e-8,
-3.269267666265974989306992839248354184598e-10,
6.782658721616058903855052799679506052275e-12,
-1.224567145352890621898640884590919202866e-13,
// [4.7143,4.8571]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001486717030259526857978357402052743082267,
-0.0001638681006870567015996869216619192588009,
9.443488192757353869859743802587522589017e-6,
-3.748955569998548598373401577092510967988e-7,
1.144243677038103130798984978266179174999e-8,
-2.848305322146476019135356514410505244061e-10,
6.000575972232200179389080240419011575282e-12,
-1.096913433192101207399753101730005267016e-13,
// [4.7143,4.8571]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001147076704809396867306213830262720185747,
-0.00013220883469850601350304689214692945375,
7.872806697001165670448861519977688028385e-6,
-3.203882334483281607048982106962162734338e-7,
9.96906857133098162088495945648679182139e-9,
-2.519869650980031986636568270615471011576e-10,
5.375055604386191139153279167062442844634e-12,
-9.926902223173019713171585148165474470882e-14,
// [4.7143,4.8571]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0009254618428895420541653192705512372523189,
-0.0001102192937579278335921009919196156362279,
6.728152902418765570995214943754946623113e-6,
-2.791339235367291223453320109815203941996e-7,
8.819543726774008074207143084158853197749e-9,
-2.257183563613383776516656302805168124011e-10,
4.864347389217108835679414392596506549649e-12,
-9.06110918130046016771588608980551577627e-14,
// [4.7143,4.8571]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0007715350563054947978073012481386166564465,
-0.00009419414063378135996410186234023394780438,
5.861812394274895978265548908996425913315e-6,
-2.469472276040212044292255201484036297161e-7,
7.90014242485457850100108169902265093404e-9,
-2.042713484191006453300554868722578650394e-10,
4.440096434392663511127073081079291746319e-12,
-8.331214676475066513457762833477733645935e-14,
// [4.7143,4.8571]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006593589844364694850169946698799881331325,
-0.00008206537351977327119074240706112345974813,
5.185891779687779554932756929799604828085e-6,
-2.212039909068543653744031457385993561644e-7,
7.149497150211561351813348015085320510631e-9,
-1.864551451870869824938420895999315147064e-10,
4.082437454038116881216687290786344990905e-12,
-7.708046964915659951511715879852253742209e-14,
// [4.7143,4.8571]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005744576146384128658744608603007330103786,
-0.00007260248491555889434788447463687753854642,
4.645283809047057919489361417474408320108e-6,
-2.001859230067249362708729549801818873282e-7,
6.525930039997961585459962000196251532314e-9,
-1.714354852219258043593739610536852259746e-10,
3.777075973346230303792327824335486466536e-12,
-7.170122423617762206697803703666746582829e-14,
// [4.7143,4.8571]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005082173944089122299705728163981709979825,
-0.00006503397332659336976822015360325083851963,
4.203904383144148280159636066527656057668e-6,
-1.827260437376100668815271732837889616385e-7,
6.000241943772369443091599921955882962113e-9,
-1.586120611290464579493395789935755480591e-10,
3.513484771938193593737534233900230060791e-12,
-6.701295051651349675282362567323666263152e-14,
// [4.7143,4.8571]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004552378132861535596806916878828322854476,
-0.00005885466136395665900477782833460808642849,
3.837246918492566316361141468929897732775e-6,
-1.680067768823247390764483729503302717446e-7,
5.551422102784353576584691237347650764161e-9,
-1.475427759779575610278679554863506992909e-10,
3.283752118825197533311082656439808435489e-12,
-6.289211368277129762010373574774577772941e-14,
// [4.7143,4.8571]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004119826295476965859126914644972072968668,
-0.00005372145685883807534699398098984363294918,
3.528142314531423126523729959438536004372e-6,
-1.554398211921057480541597780841605645234e-7,
5.163997124513660271434662936569919409733e-9,
-1.378953730780987891298464728220088865563e-10,
3.081824658221840046511267096056211686085e-12,
-5.92426841969085863492953841468553677523e-14,
// [4.7143,4.8571]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003760501980118665017219135619976713998997,
-0.00004939399240338524811050461950772807115464,
3.264236245036688598345741767642202310289e-6,
-1.445919216734284661614689733062427413944e-7,
4.826338024828164532627928766668100778585e-9,
-1.294156398788313813907169512719176110735e-10,
2.902996822807502212550236466276068538504e-12,
-5.598894889989354322668785677263747771794e-14,
// [4.7143,4.8571]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003457579468236967123414652794783165371244,
-0.00004569930743046181476391995871966873124901,
3.036430355144343280197904425999238120799e-6,
-1.351374667682317447547325191731703817727e-7,
4.529547364485788630490519570808490400451e-9,
-1.21905965224616512198954653531904131607e-10,
2.743558570896991889355068339354577516304e-12,
-5.307044913098959106092615056379734289221e-14,
// [4.7143,4.8571]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003198951520132326800709247646728583945679,
-0.00004251002497197155067354573486079662964977,
2.837886802135101197118532041839813427605e-6,
-1.268273281758298300554146080273320820793e-7,
4.266708753067386329798439335961074906235e-9,
-1.152105456775000936137126545792011735451e-10,
2.600547349033575443960140664115311684291e-12,
-5.043835053777860246529660593836931123969e-14,
// [4.7143,4.8571]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002975701748038008324903393353248184188318,
-0.00003973041522984449111223265602431330717516,
2.663373891694559992664455697641000126177e-6,
-1.194678469629297243289252699571287286028e-7,
4.032369070264925854157149873213275876919e-9,
-1.092049689386372229587648408559002042719e-10,
2.471570208792494529175926936015106311046e-12,
-4.805279609024129260492795248259146495811e-14,
// [4.7143,4.8571]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002781129066089114165228839193521510555278,
-0.00003728723448367903515871985061321834549701,
2.50882478622356544528684366409398487033e-6,
-1.129063357596237712372811189885608194537e-7,
3.822173885635757284070779013683026940831e-9,
-1.037887434888040986421641198217713612225e-10,
2.354674101553358075184470098903923841129e-12,
-4.588094670424822204814821462988016989232e-14,
// [4.7143,4.8571]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002610106413857532257308065112176953087634,
-0.00003512354700708705036287069376012208085206,
2.371033050954055708770445084708817850097e-6,
-1.070208705124517036783751996782687144126e-7,
3.632605996021222500496501998796510719236e-9,
-9.887985152025795858010676011867279800063e-11,
2.248249866833950785219697284343480290371e-12,
-4.389551078556844972782847241861464533913e-14,
// [4.7143,4.8571]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000245864829049609332972495122805199991492,
-0.00003319446271331569320312065069924802487167,
2.247438280763364276906185534563206346815e-6,
-1.017129695320780700095306880886387788754e-7,
3.460794778162285725092612675723601600706e-9,
-9.441071686581977398864492126150403942419e-11,
2.150960178242029399212849146103828628466e-12,
-4.207362672608393749989731889183574247222e-14,
// [4.7143,4.8571]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002323612389932098336049360610881468675804,
-0.00003146413593064765143088645111543477164791,
2.135972360175445903141772674301534995099e-6,
-9.690225536649632925798186848406875949708e-8,
3.304375066217841724117447638802647655722e-9,
-9.032517905102618975194568352735053122317e-11,
2.06168478446527880356628220022927090684e-12,
-4.039600376398869756391504292991096379393e-14,
// [4.7143,4.8571]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002202489515145335418948319052901179917773,
-0.0000299036130424183075975985097615876049481,
2.034947362698162575641062570209469377736e-6,
-9.252250337151540371162561847835615599481e-8,
3.161381243590358531822260317431383329179e-9,
-8.657619366604904671747395768645758817138e-11,
1.979478410381675118089752222341968826329e-12,
-3.884625440485387926855811129863918692723e-14,
// [4.7143,4.8571]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002093252912969281357080294025488839414129,
-0.00002848926307773774321710705610066075208402,
1.942972570803501102398440153231305251223e-6,
-8.851867628185721204558524643270224297524e-8,
3.0301667559433179500342466819781687458e-9,
-8.312406440877748604001148748978146886283e-11,
1.903538044868572117098616446764742773619e-12,
-3.741037056178343963781346268932406451748e-14,
// [4.7143,4.8571]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001994248415441641856465228643166550623951,
-0.00002720161599121378536240254049618102341024,
1.858892201920621291489666483037623843806e-6,
-8.484467057562879844399365996479673854806e-8,
2.909342232709294627241304343727909520244e-9,
-7.993506932836695911596855396216921304872e-11,
1.833177271012961548942957349907528248165e-12,
-3.607630870780690799756093868143499854033e-14,
// [4.7143,4.8571]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001904113119384964812256261993634687697102,
-0.00002602449082685468151267408355145885635451,
1.781738082089770650227069381417670937637e-6,
-8.146158387653560486705041933458271629148e-8,
2.797727405614336770085911228240703503592e-9,
-7.698038281831339162889463343269090267769e-11,
1.767805938049553652623533720710816645305e-12,
-3.483365855957602093368616477756510516246e-14,
// [4.7143,4.8571]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001821714357879827548034868230666222117251,
-0.00002494433314922390560192929680311433746045,
1.710693261408763092710366588225450603727e-6,
-7.833636867255371301834241007060228964472e-8,
2.69431337843569598360390482103295143347e-9,
-7.42352219378653665088218338001298831969e-11,
1.706913926432795936024044976456443775594e-12,
-3.367337637572709821830309813123098272893e-14,
// [4.7143,4.8571]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001746103320445673239215099465184530478208,
-0.00002394970565969086013591049857194188801184,
1.64506374212509601292811923501507935227e-6,
-7.544077586913783162762043391584036824414e-8,
2.59823274825136317213569004434118881795e-9,
-7.167816462615241410046463165293816538109e-11,
1.650058079106237574714672680915894394292e-12,
-3.258756868019146127845803927997987187459e-14,
// [4.7143,4.8571]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001676479396178360061228423911701767559581,
-0.00002303089238972051538797280529297533294805,
1.584256293253318010946859322406550606544e-6,
-7.275051818420140332010111197586078503131e-8,
2.508735742934648684507572862914144124001e-9,
-6.929060088172283417408773096067299475947e-11,
1.596851603656773715835389860180278799995e-12,
-3.156931566291582722322668481306657842282e-14,
// [4.7143,4.8571]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001612162467280435933235773585269177142037,
-0.00002217958810551655770746459022592281216164,
1.527760881869611131585890272082587122531e-6,
-7.024460199795917933591108231858123757604e-8,
2.425171012438080491784283536971817967418e-9,
-6.705628770744792226910662954800085349738e-11,
1.546955418711840810633550839895212481956e-12,
-3.0612526043482659800602582044624718342e-14,
// [4.7143,4.8571]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001552571167386158899714659815593160948551,
-0.00002138865234614554097836591390555005166579,
1.475136641958484928884914070987988107711e-6,
-6.790478950887168082591586592775267335807e-8,
2.346970051865113985459629867718743410281e-9,
-6.496098570420612051330051897684068175774e-11,
1.500071042058809761903403530486572960305e-12,
-2.971181706552440598275761318553484968691e-14,
// [4.7143,4.8571]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000149720566423018773256426754282962891551,
-0.00002065191298739060361928993705966685144167,
1.426000579687610149436444526213033613233e-6,
-6.571516257964917498857147241364568107425e-8,
2.27363448224912898949571354265729299501e-9,
-6.299216040932054358090000258381792625392e-11,
1.455934710205736470596623442441410098331e-12,
-2.886241470168889435477619088218840574577e-14,
// [4.7143,4.8571]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001445633909117342121105893858496256997727,
-0.00001996400811559914023161648322286342215499,
1.380018414173902227999165574440778696153e-6,
-6.366176659906033219299781824238219008494e-8,
2.20472559739878664263645845516122310432e-9,
-6.113873534939910312157854186528473235821e-11,
1.414312488285550028323869453293369639974e-12,
-2.806007021708905866047098876694624135299e-14,
// [4.7143,4.8571]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001397480568091939687452584176031353572783,
-0.00001932025779840797060655002550467755498691,
1.336897098581503084329300595661752401822e-6,
-6.173231779359947982547881600223905086691e-8,
2.139855720747470462480845043819375648756e-9,
-5.939088668260357587381149912644425652746e-11,
1.374996181547619986929271322668641681549e-12,
-2.730099005396161832537443084235828110491e-14,
// [4.7143,4.8571]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001352418045888557817003434572366841071528,
-0.00001871655938011508493857542500472847390838,
1.296378673666793451621733528322106637175e-6,
-5.991596121926863940421250613246178056223e-8,
2.078681017832734867939540508605994017125e-9,
-5.773987150349725638049819872088116941555e-11,
1.337799899606388370711890476682920007668e-12,
-2.658177662659191544418330656160931440868e-14,
// [4.7143,4.8571]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001310159156608055823385700722477034302573,
-0.00001814930143130981647267657632730804286351,
1.258235185605815653930196463706241570856e-6,
-5.820306951100012347591807761783959008567e-8,
2.020895486965999551885085840499952218747e-9,
-5.61778835603202211856883681237659709082e-11,
1.302557155307798213265306492603460247966e-12,
-2.589937810055519917119950501089080646898e-14,
// [4.7143,4.8571]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001270451100191687033760280368407642176091,
-0.00001761529259845676042999986297044953458729,
1.222264459732148139294092903476073417556e-6,
-5.658507462140630680469889843011107199238e-8,
1.966225909337206420418113653585243617285e-9,
-5.469793142342336061621384533476675794757e-11,
1.26911840383950074941868184231900872365e-12,
-2.525104560847952563434270499412169894517e-14,
// [4.7143,4.8571]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000123307048189197311361830744148800815409,
-0.00001711170243622601750737005656519086220929,
1.18828656705065067324306329762720912373e-6,
-5.505432642370859104605299994346261937403e-8,
1.914427584910030192716353277119112403148e-9,
-5.329373514162155229116894886574995735222e-11,
1.237348946240877429129706193225975684839e-12,
-2.463429665132725012641765668311608346675e-14,
// [4.7143,4.8571]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001197819170535821111747396238496874837014,
-0.00001663601193868562661704692505106612204605,
1.156140854898972595691397327093534250303e-6,
-5.360397331680207382593827386362216116134e-8,
1.865280715394256271348848333140919829137e-9,
-5.195963820140072421195678095426180723804e-11,
1.207127136012080997572903128614743595235e-12,
-2.404688366858163814127721938555209634961e-14,
// [4.7143,4.8571]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001164520835707993752014717509662256773089,
-0.00001618597196856268050970294473516712425316,
1.125683439653910870508647395918936783124e-6,
-5.222786094848107086408452184433124094135e-8,
1.818587322818042480541021153981130156311e-9,
-5.069053221465678723667890141318677804606e-11,
1.178342839006682991399358530709497744783e-12,
-2.348676694693181816575626673182446339221e-14
}
,{
// [4.8571,5.]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.3985190434543868679871748218571308921514,
-0.005670756655353602039196264703080313234737,
0.0001157857717675472632550319987721670084099,
-2.440017447062346480808482639228484901718e-6,
4.91437742963182375084857974222307635154e-8,
-9.17990767045166779360732777483631975524e-10,
1.572566632624148360030567421080708825777e-11,
-2.46353017453513260612543817951291734062e-13,
// [4.8571,5.]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.03969529658747521419466115396846570672052,
-0.001621000804745468340245923419882582460169,
0.00005124036638831692854896813936733074763366,
-1.376025688030777998620106700213360681094e-6,
3.212967674454788095163083068600623420207e-8,
-6.604037402276446347103162097849390181028e-10,
1.207162436124373031159097075665852290714e-11,
-1.979885947639205630320136556417318158347e-13,
// [4.8571,5.]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0113470056332182783137616375304769463855,
-0.0007173651294362762983472495940854015666953,
0.00002889653944865286214340905019321467300976,
-8.996309552754414738990151622662451846175e-7,
2.311413082097836464310037206334471781749e-8,
-5.069465131019248848020639448251602178466e-10,
9.701719509360156219178331105087125335084e-12,
-1.64560763808856661793741202593536610762e-13,
// [4.8571,5.]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005021555906053934029336382672764579005623,
-0.0004045515522810030625231747282887361813481,
0.00001889225006078994403685845185965059180525,
-6.471956684677353492952773841171124998294e-7,
1.774312788292603000345485676894497683791e-8,
-4.074196078509417595827098444570066857497e-10,
8.063719479368605140987239656653487590948e-12,
-1.40297954870951577485429208759190781137e-13,
// [4.8571,5.]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002831860865967021385465245105159341728021,
-0.0002644915008509400818527334349256715943411,
0.00001359110903782745326858090011776056706065,
-4.968075854873527259974216980104441163632e-7,
1.425968620797032857421456807629905728967e-8,
-3.386304698251065768725644375616571332949e-10,
6.874813589503210760339230956914093220052e-12,
-1.219959443729204959304882194685788703725e-13,
// [4.8571,5.]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001851440505956580526275577675773964751165,
-0.0001902755265294791162687814038444901972937,
0.00001043295929523888984878045462899687162335,
-3.992712180323826213998577005523582607782e-7,
1.185206638411052100893861125877317167326e-8,
-2.887017620941783593778813006477339530489e-10,
5.977992532908112324364728230088298500399e-12,
-1.077568515269374100592667867681227996253e-13,
// [4.8571,5.]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001331928685706353771671986901887864021689,
-0.0001460614301332503233233724204812472014845,
8.384695578684087179439001260792420274782e-6,
-3.318578625205077630124456604286583195014e-7,
1.010456161926768456763752527561557623918e-8,
-2.510395382185891774434338893024574910059e-10,
5.280258616541463172553920105663007458134e-12,
-9.639547818413872929580546076485330184491e-14,
// [4.8571,5.]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00102243001093275222477249042094257906666,
-0.0001173857381014921258731076263635251758819,
6.969015112934358188643421412447015053458e-6,
-2.829277287433090181537841018748251390965e-7,
8.786383788381603881260867461827146207529e-9,
-2.2173818510073004849419197841943223352e-10,
4.723536092188798564424530382925008836332e-12,
-8.713846222003894052908045935876920986136e-14,
// [4.8571,5.]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0008217001667104448457505124075129633606564,
-0.00009756621158100341625289682168986454864273,
5.941482303612884077278601311228925074004e-6,
-2.460187491786465869840034554315287995545e-7,
7.760836433262802385029551832697211040653e-9,
-1.983587176724320333068737544956189364809e-10,
4.26992948986839925938762736351977650071e-12,
-7.94621841578167054502593762320830054601e-14,
// [4.8571,5.]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006829634810670238810776322783929037050798,
-0.00008318075225050908854380494856933728066789,
5.166393732754716836429974051089542800054e-6,
-2.173034229829243038427255710723479733798e-7,
6.942555076688197882426922459474588361907e-9,
-1.793096633868312777477804255752279736423e-10,
3.893780934157189457247310844102712005438e-12,
-7.300079677018887340311132550908273913784e-14,
// [4.8571,5.]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005822652657535635894147866780840934140335,
-0.00007232951225850012707866937143905259165705,
4.563371882644328012923015924210615413402e-6,
-1.94391544783637505480003447920355198586e-7,
6.275838179637213138337137352077077499413e-9,
-1.63513489956790386280494187919164862457e-10,
3.577163528013275450114571923836707384647e-12,
-6.749168435521936905191513247369313644217e-14,
// [4.8571,5.]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005063065858095008611619569113630771717258,
-0.00006388720635695932197334589185031114470257,
4.082222440459112946989349002631247563399e-6,
-1.757234714806715745033465210871841555805e-7,
5.722972112149793087271545480687079454691e-9,
-1.502173400759901941159531050181781696644e-10,
3.307208814829304206100047797967805558621e-12,
-6.274186121629450818218571115756390600179e-14,
// [4.8571,5.]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004472104444987152271836888361662666374453,
-0.0000571511141663703493528785706324233512963,
3.69019290109665953206473863092414669983e-6,
-1.60243221429490463773332931338346954554e-7,
5.2576068685733180664247046956567782642e-9,
-1.388807928503856206049693762319295921834e-10,
3.074460276107358956909939322798454886982e-12,
-5.86065710447333207665358281488131301469e-14,
// [4.8571,5.]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004000577991645924203968557499865491131292,
-0.00005166270061529954769304371701524301986306,
3.365107650021706787979119168565201209325e-6,
-1.472129944675020895287421221275669113293e-7,
4.860827717669413598693632390665055686798e-9,
-1.291067159630704527461149751113143481445e-10,
2.871824682471118303393862594694181224687e-12,
-5.497525350462366873435781327150822557917e-14,
// [4.8571,5.]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003616389043070968101646156648582477276317,
-0.00004711150710025334706647897042031898372205,
3.091472883819817814245788337239868546431e-6,
-1.361031781166801447915296066963895760901e-7,
4.518735028388249003712190987449920285178e-9,
-1.205972259578603480096397091550891650137e-10,
2.693884443423008506901933182935129338919e-12,
-5.1762101351715301541111406876648125394e-14,
// [4.8571,5.]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003297805497017734070215282780504217524673,
-0.0000432806203734296968378382994902924763471,
2.858166740452437659079012345538997269429e-6,
-1.265245827049904949569599993609576217576e-7,
4.220902879796775660024689501849469688927e-9,
-1.131248093671703246700183713470281143756e-10,
2.53643489710382957720458570811704966375e-12,
-4.889955887904827201440552192024748003429e-14,
// [4.8571,5.]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003029643426140078565412246035366293345357,
-0.00004001433436628888029225964217204796672221,
2.657016236806847474661071613742721821121e-6,
-1.181852824442033593648430173083223910108e-7,
3.959368300556467513827537064245341056703e-9,
-1.06512890595426752424262150312444680423e-10,
2.396165727637828821207276524286927413981e-12,
-4.633375255184924740985138124015891790269e-14,
// [4.8571,5.]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002801003405640221417368340738162948461667,
-0.00003719822731525287600374203772145295177686,
2.481890931330220219390222156165687056278e-6,
-1.108623141351422593224766147248528094518e-7,
3.727951144844217393384347490425654451895e-9,
-1.006224526743125367101199907795497115622e-10,
2.270437061517357136885244662182079492188e-12,
-4.402121966029501198819351291513443999681e-14,
// [4.8571,5.]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002603875912067701126404771553716686085778,
-0.00003474647303858213999232331399860716043371,
2.328108596839848484503854749346493196476e-6,
-1.043826336933760861479157277064191612283e-7,
3.52178581878700991094470280092390439414e-9,
-9.534263420400205437817146382029548025084e-11,
2.157119168095633530623176058441664503911e-12,
-4.192652676829000579844864900571991517279e-14,
// [4.8571,5.]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002432253112700749614041482094558383535737,
-0.00003259352035571879701583463562254369459303,
2.192035307562677861638158028073586804473e-6,
-9.861000448932118082287238605174976497379e-8,
3.336992173405962199333150975069359847948e-9,
-9.058399743397111476576432957040795792962e-11,
2.054475760959279169155267798127885973516e-12,
-4.002050940513239480939028534841503357488e-14,
// [4.8571,5.]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002281546424900315613424823739492166063425,
-0.00003068849430584010900406341139503544208728,
2.070810094277450568974561663860807235377e-6,
-9.343578235062294121427521485956953224224e-8,
3.17043988744529341027986153485462554703e-9,
-8.627362741536596504113641598831193659604e-11,
1.96107774083319769277231201884702584236e-12,
-3.827895280168122565575892663934439406588e-14,
// [4.8571,5.]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002148194601408807459722683794010793775528,
-0.00002899134131984848680265639047220165654485,
1.962151429364719167135808573601718749948e-6,
-8.87723182813278536805841872081288502831e-8,
3.019576937705716658171642874662274284536e-9,
-8.23515095785074741938328231700431070785e-11,
1.875738550127142183557069079367500886835e-12,
-3.668159055630813814574121890623551029652e-14,
// [4.8571,5.]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002029393892389393912200593200679366032716,
-0.00002747012001107168294655430848801674628961,
1.864218683909459195165072571278328245009e-6,
-8.454815563118843173011145793716576581663e-8,
2.882302814257455507076934851821275623014e-9,
-7.876781491219146641626923389258589120201e-11,
1.797465106383195821893182291137234532341e-12,
-3.52113357139653134404879054109386232437e-14,
// [4.8571,5.]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000192290840077501764836441802804872160205,
-0.00002609906157469936914652240424260294303009,
1.775511268256472860594517681575069813186e-6,
-8.070448012160436459950506838867178659021e-8,
2.756873501716045722113473098472633675436e-9,
-7.548083939228918553009829759235830573602e-11,
1.725420124221834346879911806238396252274e-12,
-3.385368394769866419206822923515748066952e-14,
// [4.8571,5.]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001826934310228955688019288570454791691699,
-0.00002485715775555878840698962062770884438664,
1.694794082555153142381248356751879080434e-6,
-7.719245932132669100277346838895323717629e-8,
2.64182935924358102098455051593712973119e-9,
-7.245542167892542779125316812352234877986e-11,
1.658892870501682117104196258027553606195e-12,
-3.259624570789406021939179615543950279051e-14,
// [4.8571,5.]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000174000104288911504151977403164110397777,
-0.00002372711715574145282860796804401359547262,
1.621041645749271425902831692263874482247e-6,
-7.39712232864782218398357702521696627837e-8,
2.535939739950130842473956035689040166095e-9,
-6.966171497198889642859096229723399970957e-11,
1.597276239046591264209220729545821024052e-12,
-3.142837608622189254215973247406954848016e-14,
// [4.8571,5.]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001660898200901901555950188595178245655469,
-0.00002269458304046017078660344092975487510519,
1.553395689017406368790750822866121725763e-6,
-7.100631390378170109557357581475087654233e-8,
2.438160005836750036996386957761721085453e-9,
-6.707422426042154398876485675940405918121e-11,
1.540048613507623174704496725838212991706e-12,
-3.03408794760134404159306079017777120393e-14,
// [4.8571,5.]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001588620812832211817610087914119135224257,
-0.00002174753964621505128441230842271694748286,
1.491132591980735270860065066367815046409e-6,
-6.826848130895479839666133138361163342227e-8,
2.347597831520725187989131253875449045907e-9,
-6.467104465149845843529060907612614585974e-11,
1.486759395339604053538703597991221975364e-12,
-2.932577203374869424821531489630989068391e-14,
// [4.8571,5.]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001522327775235053456769570169659749466701,
-0.0000208758562877025833206551868127907006041,
1.433638107489328910499980359459737643588e-6,
-6.573274039100948760733736821841958798356e-8,
2.263486545760467997169487003609625861416e-9,
-6.243325361802018736450559655493794050886e-11,
1.437017364102438028348153127027562559346e-12,
-2.837608920647230063765792013678029524469e-14,
// [4.8571,5.]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001461309940139180703358085878355173891768,
-0.00002007093350482376375196509894752330415695,
1.380387548212438489187556077105110235789e-6,
-6.337762435494292717238300259160338395848e-8,
2.185163860107326598751422448030840242562e-9,
-6.034442218985881727776239974267853281798e-11,
1.390481246048594742972159034438092253479e-12,
-2.748572868881308466824669860514390796335e-14,
// [4.8571,5.]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001404965345337663337362918260535915067977,
-0.00001932542567494811464212644392520132315731,
1.330930111455004113734629915706697598688e-6,
-6.11845891239831389979006522783563795412e-8,
2.112054760609764409281533932151902410092e-9,
-5.839021888299667812324589302598462159839e-11,
1.346852018805624619987266139643404183277e-12,
-2.664932145129737115506685222484999558403e-14,
// [4.8571,5.]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001352779797246367903269340115905340955041,
-0.00001863302156034480211760021293690749053069,
1.284876371604814048665769599981368896855e-6,
-5.913753430730185348734525838089723585391e-8,
2.043657645329769698586550546419375757114e-9,
-5.655808653615465785542210254633456453653e-11,
1.305866591587792967279291952023248928827e-12,
-2.586212517286998289146846093276599002779e-14,
// [4.8571,5.]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001304311509224136029947737388937793864206,
-0.00001798826920244286598730139849809196759877,
1.241888220454474458533736640796019144148e-6,
-5.722241505047051806055706279751763356498e-8,
1.979533013624892482081843798785203372916e-9,
-5.483697691289187702362119538067898148872e-11,
1.267292583242783921858016766545773113802e-12,
-2.511993567782167682018577977890949521366e-14,
// [4.8571,5.]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001259178844171000504038328866709732480726,
-0.00001738643508633879620728053917939674523228,
1.201670716060985583929067110356462485756e-6,
-5.542692533535443951598989708760066173468e-8,
1.919294177221771666235097029746424929668e-9,
-5.321713140731735962960674809369110801421e-11,
1.23092398253783305518747984503134418583e-12,
-2.441901293526078935215483318507124686609e-14,
// [4.8571,5.]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001217050456043715622479128376296558981649,
-0.0000168233900248305994055500530985647677929,
1.163965432043518728647819604598865199973e-6,
-5.374023789016911639835660940207881405251e-8,
1.862599584916076049277613191801838071425e-9,
-5.168989879941368407341766522185080895358e-11,
1.196577522029771591159003595820420801489e-12,
-2.37560189093524462350408399382994742472e-14,
// [4.8571,5.]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001177637301738141849244003527764379415214,
-0.0000162955160485866767536526299357136578906,
1.128544995694599237344294322563817238802e-6,
-5.215278928107653820705124770714056639252e-8,
1.809146444008859675510423583698123175687e-9,
-5.024758297724752456096532072917517524975e-11,
1.164089632638519695433224680456898555998e-12,
-2.312796510931501919029520364370797845328e-14,
// [4.8571,5.]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001140686123401067266352646566261378641314,
-0.00001579962993970238569722378992328329517873,
1.095208574903628777147145608770016375244e-6,
-5.06561013124013983244995135282592994737e-8,
1.758665390583954261340862404511486774326e-9,
-4.888331504573157486531183872733240092835e-11,
1.133313873521787056631235313629346442634e-12,
-2.253216812205223264606349041282558719465e-14,
// [4.8571,5.]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001105974095779166895010110320624821940432,
-0.00001533292004862935193559324388880162162113,
1.063778127561425807649269644342557692326e-6,
-4.924263179439660290303617111719731640328e-8,
1.710916013314656599936780645603080773696e-9,
-4.759094539556084626482519625559302084144e-11,
1.104118753109812375762682471149560898981e-12,
-2.196621174835914544062322526893339412045e-14,
// [4.8571,5.]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001073304403404054534179038016194316416408,
-0.0000148928937858390360332581993951347579973,
1.034095267683301266239970189437225953523e-6,
-4.790564920982925359080630899802957152006e-8,
1.665683075876516219237960587191214079468e-9,
-4.63649521988119878802054618743719437242e-11,
1.076385873724319811282563118174659629691e-12,
-2.142791462877762870480454351946026829712e-14,
// [4.8571,5.]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001042502565008732423382910682455943694187,
-0.00001447733374754579304167111194019430386702,
1.006018633407364204235493097728671360863e-6,
-4.663912694153761645083031667907650995878e-8,
1.622773314293326439017146522607490521193e-9,
-4.520036349336035915492847327919360472479e-11,
1.050008345197932464326053439712190832264e-12,
-2.091530245442713336639057734372280603708e-14,
// [4.8571,5.]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001013413362328205416231290017839355619511,
-0.00001408426086768315142844837709999862072181,
9.794216657732181332166907636926063970843e-7,
-4.543765359811793630339495299094956836939e-8,
1.582012709891734743730141462917930482596e-9,
-4.409269056387389183131530884047571585917e-11,
1.024889423163804603872286643922176178451e-12,
-2.042658402426491827677908056849958601874e-14
}
,{
// [5.,5.1429]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.39296168079461859711252679729341732912,
-0.005446313086368333853265017271238941646192,
0.0001087516329489741550865650054235121027161,
-2.252316205927364162149311017487943721653e-6,
4.478129911567130643380128201148672849283e-8,
-8.286280667939424048098572482376443579182e-10,
1.409634386902929210666703636621539219304e-11,
-2.196820167567312333737900136147379770606e-13,
// [5.,5.1429]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.03812419160457833690227253798732814486714,
-0.00152252286128546644129398374908504252831,
0.00004729864032448142336208943783356856826074,
-1.253876382108041319530923242457989144225e-6,
2.900198224744165949259283470988651726024e-8,
-5.919804974423876329548799510480407689913e-10,
1.076470792981363621150812470692716229815e-11,
-1.758540703337299564289056161084005388784e-13,
// [5.,5.1429]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01065766002899826502902946465962599145243,
-0.0006621809645425976321085244698837607641095,
0.000026331404024274630464190026445322595632,
-8.120555086202065699436751888341894360513e-7,
2.071931733364664206145872925487897329963e-8,
-4.520630911204511343086161105456500326838e-10,
8.617095324949437326218785949340902674501e-12,
-1.457123259672906984121705503027769646491e-13,
// [5.,5.1429]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004635266751798183372665412221132009389048,
-0.0003686396563397238087831243690127434334413,
0.00001705316568102933904052944544583826106398,
-5.801408901828513500923805623357413951643e-7,
1.58222081225346508020167968055322673217e-8,
-3.618715322570943299405206175653047233775e-10,
7.140117352431556951022304126399253740297e-12,
-1.239241567674468196660577006751359023655e-13,
// [5.,5.1429]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002580477594378066615542797002926967347133,
-0.0002387443195343057241680470056223591716239,
0.0000121829586938442885236282628743209875058,
-4.430218316318990285810517003951801414304e-7,
1.266550357019584737311328878449106655178e-8,
-2.998445996754637047078013267134706214301e-10,
6.072471849810250305581935904280361619893e-12,
-1.075447379999412764031016672701595814131e-13,
// [5.,5.1429]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001671210236740140028136562036462298620646,
-0.0001705614217137274258215878132231318775493,
9.303458464273819436623688838389696346405e-6,
-3.546341036700537394552835524878905390117e-7,
1.049456093610992706097100188910197998294e-8,
-2.550082536301785019644407567341825167386e-10,
5.269860262486129567529536794288117383239e-12,
-9.48378596582957098348450402976793833822e-14,
// [5.,5.1429]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001193929951996091943694905699871958922199,
-0.0001302484184997507356355594694314677558887,
7.447316177074685941871202841884815254036e-6,
-2.938477095205640513981793894408688546321e-7,
8.925288829623932122103036705979061557104e-9,
-2.213023597850734509353495499642891052894e-10,
4.647206906960270458962145378584032652439e-12,
-8.472363414619361162009687457222993873159e-14,
// [5.,5.1429]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0009117389294982551156900951147496689826765,
-0.0001042624264789708975871655106967893239948,
6.170801899935085942127904646384051592771e-6,
-2.499080902177188505487553684063613468165e-7,
7.745582549265939763669381591792290440865e-9,
-1.95154002745664361737991247849425901909e-10,
4.151596350654686270163913682579046338712e-12,
-7.649989331525953588135439698267556234696e-14,
// [5.,5.1429]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0007298369853527962521225189823284703746446,
-0.00008639122659902314514605726619054870452323,
5.248069894575070689727561140368237228168e-6,
-2.16876314101790964993604695610829613225e-7,
6.830390056433758188066435821077273148762e-9,
-1.743409120722659222473768877932199376119e-10,
3.748621699082667800955822688475298882296e-12,
-6.969269314138060167990670837564055824871e-14,
// [5.,5.1429]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006047385861931619873942830338014071208084,
-0.00007347297852398851833298633881672180749699,
4.554402596140358577996641967781072405298e-6,
-1.912509240790194340521653540593712035392e-7,
6.101931885885021644061209103633806062093e-9,
-1.57418122032933720514113721431480764401e-10,
3.415059225877532555949425874533557961054e-12,
-6.397180214577172622010238422968014335689e-14,
// [5.,5.1429]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00051431084966791960173464027213408828139,
-0.00006376163634590730558069173124790584806033,
4.016269405661961369405529733430666358258e-6,
-1.708540951133809196509936476298899923118e-7,
5.50963423710918576441447363042736467739e-9,
-1.434103247967205982837729377213524027125e-10,
3.134727244546192938333560835887042073492e-12,
-5.910074917254170864451067618113367659277e-14,
// [5.,5.1429]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004463314544213511142356413933435069460572,
-0.00005622777167921384089162840270194805365335,
3.587935997383382928134210306120289348759e-6,
-1.542697607838070263723378061802661159754e-7,
5.019361336103583878241263220854844160289e-9,
-1.316379545534324200723796430101808604477e-10,
2.896038410901249587891522239785115200025e-12,
-5.490614037938825628655821285543420681439e-14,
// [5.,5.1429]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003935944017544968629624101136591931813913,
-0.00005023110396331730512605130104283325708981,
3.239664976462182348249479379111332082845e-6,
-1.405421194131525883086879292705633169667e-7,
4.607328379572783512526269000650146780693e-9,
-1.216143915241801799668657308861093192386e-10,
2.690496230310352212402146085802883236458e-12,
-5.125817011933309206703779825168481119252e-14,
// [5.,5.1429]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003516177277432211139738774534858898323259,
-0.00004535530967042362224618847748136250574327,
2.951384507678307579627690405211756732658e-6,
-1.290051965052796611372854346160081774844e-7,
4.256503675303217109716274185784877160791e-9,
-1.129828200465714342898008118818765459268e-10,
2.511740073918866222248355976047066305697e-12,
-4.805787233086562543729360395321017457352e-14,
// [5.,5.1429]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003174871676929653350348423154090161849342,
-0.00004131938310745213843763318572541808743312,
2.709109126612858988036248420365531813235e-6,
-1.191821046752128416688864987059309272948e-7,
3.954398675148528620073090084661275164702e-9,
-1.054761224657417527022583096052333391836e-10,
2.354920485098261960275525112710002417358e-12,
-4.522856137324736588759240291712219230629e-14,
// [5.,5.1429]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002892356817521649494677145676758922013334,
-0.00003792752777253831769362838238479992118302,
2.502824198181350873811679292596986293393e-6,
-1.107231645724992455475889471473582745711e-7,
3.691664261218232241069557030892996720845e-9,
-9.889064421057248453013179265665789955534e-11,
2.216279772190578556569635445486543147676e-12,
-4.270995098480011603014151996743435434034e-14,
// [5.,5.1429]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002654926944077682052439347809085868773648,
-0.00003503953877449940710532222325391401273028,
2.325186456024270866445115600675724111338e-6,
-1.033666008943298006223736641872208517114e-7,
3.461172523547162387273028893070159879642e-9,
-9.306858023601633857079525576914695120093e-11,
2.092863831614463031920856138706647973709e-12,
-4.045402878452094496059072316740138963251e-14,
// [5.,5.1429]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000245276771421495832016800801812568996142,
-0.00003255261038430227126529133080265370683637,
2.170698618782626985726773015964961743098e-6,
-9.69128321601686724323008434386257309638e-8,
3.25740028557819775804130560987923875111e-9,
-8.788587269922219502645241787895454650854e-11,
1.982319994192220264561511226844766278999e-12,
-3.842210780815795302590139070279156884144e-14,
// [5.,5.1429]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002278682726901158819468967975789678332481,
-0.00003038978066292105321637773173967932526496,
2.035169475365165503382320890101187979595e-6,
-9.120720942518582507253304010746713679194e-8,
3.076005522827610460467698673393569633728e-9,
-8.324372130891922286419783231134299626084e-11,
1.882752547279264445684516682908796857402e-12,
-3.658268344123469341597737074000670289566e-14,
// [5.,5.1429]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002127284646404473563445252404625767044293,
-0.00002849237265507822605920468046776390006884,
1.915351397930454666210703828431170288915e-6,
-8.612815600282501202364735872761712216688e-8,
2.913530225114244100634626981536204210855e-9,
-7.906251584798835273293378730541681726393e-11,
1.792617722133564885872845200174745730533e-12,
-3.490985172277959459582181206408697655881e-14,
// [5.,5.1429]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001994466085855475669227523504895141551908,
-0.00002681491957099376623212720817447480678281,
1.808691276060812461770415822376562294671e-6,
-8.157884760717452159814166200502937475785e-8,
2.767188034850071727280784617042363039553e-9,
-7.527742608689535030414506387940657414971e-11,
1.710646189017932218681478604055597203142e-12,
-3.338212556514267058932156529782939199259e-14,
// [5.,5.1429]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001877044369969563487573953312929608723404,
-0.00002532167786482014310715420732914548653171,
1.713155799752092240750366680934101778716e-6,
-7.748126622506774658646320820107212365976e-8,
2.634709894010780143148944531447665702673e-9,
-7.183514691449481612013139103182208110963e-11,
1.635785050605453135570477490748153462858e-12,
-3.198153739686321943796849938749704580597e-14,
// [5.,5.1429]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001772517450537409874587354462409042378737,
-0.00002398418119649931837526107901779912921404,
1.627106590727794654721868072646454117086e-6,
-7.377187823123265999212196110957893130518e-8,
2.514230123714240761924363138801553968017e-9,
-6.869146231915531404155044148106906933105e-11,
1.567153870421246889814848066816046963362e-12,
-3.0692950902376882820813150476164120267e-14,
// [5.,5.1429]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001678892683754952148689089679844840586127,
-0.00002277949227016031369602404420119481791622,
1.549209442857206627143794476846883000433e-6,
-7.039844461646815463771132574895233843529e-8,
2.404201163560148200714118556016377095756e-9,
-6.580939878342009078110468374005437469388e-11,
1.504010947259000197313723356441443099398e-12,
-2.950352740193175116905786404807791419067e-14,
// [5.,5.1429]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001594564458911221826095073431145985269165,
-0.00002168893219997315669976865549480887888271,
1.478367336947104469730319961913599473628e-6,
-6.731763368913760060196004958023950647002e-8,
2.303328940443350246831456492970247988198e-9,
-6.315780896620445748715718737172865948105e-11,
1.445727167141549476664329881415516010474e-12,
-2.840230797590254841927065167869878330022e-14,
// [5.,5.1429]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000151822525399812084096744314372226268901,
-0.00002069714271723272493925051796018360911234,
1.413670307473118576385145883105428482082e-6,
-6.449321140192918291146600277742949789139e-8,
2.210523297430919465088189957027295804047e-9,
-6.071027360899399956638171196523762510657e-11,
1.391765526889869143141455106168173524258e-12,
-2.737988318710913019594736581223912035389e-14,
// [5.,5.1429]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001448799990206290622033234570264012908084,
-0.00001979138430459785186185536813447168010039,
1.35435743944170050488875211488160246877e-6,
-6.189465336040362164285020628549327255877e-8,
2.124859560479221730664065159968542159306e-9,
-5.844424162447756940096067406339811343843e-11,
1.341664950096872331846037953546740292013e-12,
-2.642812978538225342908175685354639964788e-14,
// [5.,5.1429]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001385396901321849510639217111564781550004,
-0.00001896100415215886615984273678748011098101,
1.299787720569625091077609495343938301948e-6,
-5.94960686910638259031569429235263012691e-8,
2.045548441536177188084498195418667798038e-9,
-5.634035044679068654742553289200542611557e-11,
1.295027385311248678996319603514973366235e-12,
-2.553999912351759478875338358076913840094e-14,
// [5.,5.1429]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001327270290651120515269626738972286761345,
-0.0000181970280879506215361412439814568779206,
1.249417442513453175937873847914920134102e-6,
-5.727535732821150918398124149791482991149e-8,
1.971912250799865854494935704541451853819e-9,
-5.438188421947538240409166111942735682541e-11,
1.251507438142102581918721182575690869494e-12,
-2.470934585364669024404245462375278659135e-14,
// [5.,5.1429]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001273791966156543395152391848265729061356,
-0.00001749184419516497502004907084455744161967,
1.202782503893520522712124910956397337039e-6,
-5.521354395718269066625721978232814754466e-8,
1.903365933297193249033762680701223012624e-9,
-5.255433839624887603157611992997621085995e-11,
1.210803977153717635168880352434624823125e-12,
-2.393078826313944404972139907444449747655e-14,
// [5.,5.1429]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001224429093661548142358624724268606249585,
-0.00001683895505448663191865690866711384353864,
1.159484423101883339898875331018728612836e-6,
-5.329424703854585566807718108596434969753e-8,
1.839401829910852727308116429290478675605e-9,
-5.084506723156475178308417967092228181468e-11,
1.172653290137000844254201383371510581761e-12,
-2.319959365804264691356140590179642881654e-14,
// [5.,5.1429]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001178726853814064128402608718755510752764,
-0.00001623278192340438323126297262419509287261,
1.119179187810479647001641625097078006668e-6,
-5.150325211685320410906671829387915494491e-8,
1.779577339549015433640283960099530762224e-9,
-4.924299637950402816337069023460880889843e-11,
1.13682346774201716595251157344481698428e-12,
-2.251158372167155512415862959853037884304e-14,
// [5.,5.1429]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001136294734638306723251167151066366285022,
-0.00001566850862932536484661765731256968262299,
1.081568294454905489228584501249686722751e-6,
-4.982816636138889884545196049386714708615e-8,
1.723504860106558407723098749701122653038e-9,
-4.773838703581689234414758112449407891203e-11,
1.103109765918747578163753541722882364602e-12,
-2.18630559135927720549434080413606989242e-14,
// [5.,5.1429]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000109679560405277543913105606361743532311,
-0.00001514195612234792461281541692467367479556,
1.046391493590128149966978355103334529599e-6,
-4.825813691308088876177600225184095282116e-8,
1.670843533436559451979898976842316400158e-9,
-4.632264118504804283786977087751814475722e-11,
1.071330754357940266202008062294887676096e-12,
-2.125071783339475587637726785842029841323e-14,
// [5.,5.1429]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001059936928564354625421620926716460071088,
-0.00001464948091024160736573808499611808092594,
1.013420875175634433548357931082799404883e-6,
-4.678361974370061301184730060906327253239e-8,
1.621292428999712937197360241691797188544e-9,
-4.498813985571217022954154073118244846052e-11,
1.041325100223666785455634041296050816735e-12,
-2.067163212788605530636233722155168188603e-14,
// [5.,5.1429]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001025463663716912420646019754624623510954,
-0.00001418789225243923093314575092618445129327,
9.824560146186244524773830683467235408681e-7,
-4.539618879804482841054239092726048425053e-8,
1.574584882795496042221701589317025394565e-9,
-4.372810805448699191906723547562708405684e-11,
1.012948868527910415836462069827813969947e-12,
-2.012317002246790696430214439723997345747e-14,
// [5.,5.1429]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00009931524576707460727577467891677481424526,
-0.0000137543842046415991943329138007938052627,
9.533199647598300011013482144744260562357e-7,
-4.408837748400672605230650908115210495625e-8,
1.530483770058945747960877914493628667566e-9,
-4.253650139671172676957715813166574202607e-11,
9.860732451013336553307809930057280670531e-13,
-1.960297194562891439069361952082634883242e-14,
// [5.,5.1429]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00009628068943249118533167019928735083942489,
-0.0000133464795066189593442268825702742876126,
9.258559271650080032669247042529769875813e-7,
-4.285354630808437812301370385829811654529e-8,
1.48877753732812320706545777152524325008e-9,
-4.140791048372598470882605232825394715432e-11,
9.605826071366950282818739477491270686202e-13,
-1.910891401774931211306629162224600303155e-14,
// [5.,5.1429]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00009342535654633270659761980852540730083944,
-0.00001296198298029191018854932786132329367182,
8.999244724706178942265893356715273990852e-7,
-4.168577177326862155643479749174722044159e-8,
1.449276855650990115134143099048335837052e-9,
-4.033747987643279328653197235780765496611e-11,
9.36372881091505439433043267620342563495e-13,
-1.86390794122793744795128010064617541294e-14,
// [5.,5.1429]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00009073388086204336271443716737471652408241,
-0.00001259894261457088551450156850298347309406,
8.75401207239467176440407869183218075698e-7,
-4.057975266883463660944604241384048783549e-8,
1.411811784660127760837132575018694694713e-9,
-3.932083913642446794446203687140936603707e-11,
9.133501393437841609784785055219946344485e-13,
-1.819173378417728191878698574740595167995e-14,
// [5.,5.1429]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00008819259830199619019322419039391882331431,
-0.00001225561690133519189265443767108220450769,
8.521748060463345688049164127185632225779e-7,
-3.953073066443325366345058739796342091676e-8,
1.376229359012153825404312922233259107337e-9,
-3.835404389345978232447800846944010148795e-11,
8.914293961492261031999050830261388713583e-13,
-1.7765304108717252919838486181784744315e-14
}
,{
// [5.1429,5.2857]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.3876219109915428005139661698037668373403,
-0.005235391703975331705656878961233007792479,
0.0001022552927418292746127598049363198420704,
-2.081202909428314453799554570974534836847e-6,
4.084210694797290808540575599864219069827e-8,
-7.485015560737312567398521693897430795917e-10,
1.264309137173709328746062073962947028117e-11,
-1.959884716360693352273716571205520918624e-13,
// [5.1429,5.2857]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.03664774192782732187708111251906565088238,
-0.00143157409838545725710184062349506547687,
0.00004370526109800060519121590272070330567402,
-1.143579000646782926568521963024780468171e-6,
2.61975543825582371265715126359837834953e-8,
-5.309512433430135957579206266011371088005e-10,
9.603691182145663626750745746427437555875e-12,
-1.562519037438671048875959954133225935001e-13,
// [5.1429,5.2857]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.01002101868869820074667704862144518215146,
-0.0006118736553718824379063936723926597646113,
0.00002401515901358753292067738024698900143863,
-7.33531527753058210681741610686013858622e-7,
1.858329344911912594791151134767991269604e-8,
-4.033066317102594088595283717557663708436e-10,
7.656560520153304908261407420578425096251e-12,
-1.290616209107713024090924016069922496812e-13,
// [5.1429,5.2857]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004283115587603177019411883421878609884567,
-0.0003362122261901185402770073030942155483557,
0.00001540416208281863200013490338275181912496,
-5.203322208521923760904682055081471544143e-7,
1.411573205106457443263526979819437519469e-8,
-3.21534483813638734456989697151065941967e-10,
6.32420756738608677880956491188139425076e-12,
-1.094884818384610301183394308559085864532e-13,
// [5.1429,5.2857]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002353485583330829741500556651345145227429,
-0.0002156582691593682470034035466121133674483,
0.00001092697663789992201147786580829885353896,
-3.95240501133876883998110815740788172805e-7,
1.125370688171569646004968301918429532023e-8,
-2.655811585850286349109325545610693450894e-10,
5.365101247703239290234579987935094788005e-12,
-9.482500565821308373930420082580164223204e-14,
// [5.1429,5.2857]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00150960788411557769294789745080900300604,
-0.0001529776729305173838600576176240900227501,
8.300050523814877869469477541377984890463e-6,
-3.151037959490375549492240046165238854547e-7,
9.295340504298463583358473856933790411981e-9,
-2.253029466556319261111221087041395884833e-10,
4.646573045657365146629262194081536540079e-12,
-8.348231171699507706213840565290939878766e-14,
// [5.1429,5.2857]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00107084371051362165448414380842060082956,
-0.0001162007073333355608337905742777199576755,
6.617179714932912149962400028816169857374e-6,
-2.602695370295542207053214284340451762371e-7,
7.885603091300382774914597903494768404143e-9,
-1.951281394724907301423536090602126783039e-10,
4.090766543941573627694074841535776374097e-12,
-7.447614243356512665581125329889569898035e-14,
// [5.1429,5.2857]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0008134049513333488962248986137857171965722,
-0.00009264051600899517675340722240029279508289,
5.465660277623481365764672715923212546542e-6,
-2.207968891801662862491371408482897959071e-7,
6.829484843633984550348814301113816608672e-9,
-1.717870066530414381323016754917017443519e-10,
3.649452269694055942635499329090154952892e-12,
-6.71687781972743077760443463130633984071e-14,
// [5.1429,5.2857]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006484836120629662101162229985458393737484,
-0.00007651924388666904183581085065068281822789,
4.63673467278609915643257802208049460521e-6,
-1.912255780096630198561365985727676267289e-7,
6.01254519809441068365607494678792543602e-9,
-1.53254071248843449468630700060161293475e-10,
3.291381370414427300327297077252655718989e-12,
-6.113111998768689005261185054590106632118e-14,
// [5.1429,5.2857]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005356347072066832677796065296161035846545,
-0.00006491428541895063820129715116494364435429,
4.015737138205330251731057475800980781305e-6,
-1.683512677366616705043258362238620289576e-7,
5.363892461618294002283926230446032353975e-9,
-1.382169932637857250868646719484725893995e-10,
2.995527571528484806522916262941283894694e-12,
-5.606500998835884364394599059340553770315e-14,
// [5.1429,5.2857]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004543999979326544441325999284572956250257,
-0.00005622031993482408005084532479766464998093,
3.535376622472129634010204998308550516317e-6,
-1.501889909470685414461529051639913678705e-7,
4.837594734438364170518158212390521054113e-9,
-1.257927490324332769533634668234950259322e-10,
2.74728083085876237126573066975381281268e-12,
-5.175746983188680827229715973841605083017e-14,
// [5.1429,5.2857]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003935422395437685386384520990114645524091,
-0.00004949527271456288930719389988571309796508,
3.15396880989052425768114541500062220656e-6,
-1.354526544413132354578107363245012939813e-7,
4.402746188336581958813052444390458805947e-9,
-1.153677752170993845174897232331770797071e-10,
2.536204977409054723151478393008448448758e-12,
-4.805267572142405916482265534298446493524e-14,
// [5.1429,5.2857]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003464669090019402047992521169719400007147,
-0.0000441555633384235570237083543002892092225,
2.844505743269531660769199373868047811008e-6,
-1.23276895024742963160538947377553738679e-7,
4.037872106548849260362730799179734341825e-9,
-1.065037962939465598362674290834002590573e-10,
2.354664469334864872278158241903020787765e-12,
-4.483420599147245653142710365068972603871e-14,
// [5.1429,5.2857]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000309088943368964880021761390912163402028,
-0.0000398230804057324152584448292157857796908,
2.588814795521440078814767513455163172812e-6,
-1.130604206245019286465378997337376655478e-7,
3.727632845783354164549846002743863116591e-9,
-9.888015273170898274592749834616454743697e-11,
2.196954509026051418801901323426398345879e-12,
-4.201345717970231215486247066023710026375e-14,
// [5.1429,5.2857]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002787615628401268887384370319744695189478,
-0.00003624340713726156624653334978232705340949,
2.374268833116275295588971625261318980273e-6,
-1.043737212257418792581783857915024071184e-7,
3.460805322479155241069513451930036508146e-9,
-9.225726873505078987843204623934812159683e-11,
2.058733420069019802565050239368512735147e-12,
-3.952188431041455697397975350678968480638e-14,
// [5.1429,5.2857]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002537038499608309466159189849001840524057,
-0.00003323976366359142350619309210945456775854,
2.191848145742222015216251135146727081964e-6,
-9.690255048665462466186480066505564081514e-8,
3.229004383826031537093753435674227215198e-9,
-8.645281407225716245911544578251182889371e-11,
1.936642413744738564316481092418440837919e-12,
-3.730568013783723631578447560532872961992e-14,
// [5.1429,5.2857]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002326783456451399482984243776648264304342,
-0.000030685874040356614686503253307653665259,
2.034953560221306639126954544366853093379e-6,
-9.041212412688231384043817391275889334513e-8,
3.025848471735318633723595267390878820036e-9,
-8.132573566517590357645832265443924526978e-11,
1.828044866676713760763405224640031209712e-12,
-3.532204973729580628429392093301652489957e-14,
// [5.1429,5.2857]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002148011182824962873430779040475719502731,
-0.0000284893498430655430403067817882269051344,
1.898654606666012993205457915388330532016e-6,
-8.47237585185970214250075324274600602371e-8,
2.846400728489059750658262361824828514159e-9,
-7.676530824713907763677551426044332986747e-11,
1.73084377197144787268256834780693412131e-12,
-3.353655261392757607860886306321226116634e-14,
// [5.1429,5.2857]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001994254489014587865309264682041789540237,
-0.00002658116449329300948782194825892292007412,
1.779198928891953574659289167380281004571e-6,
-7.969922164460163669751835126121774983267e-8,
2.686785769768144453301642596051246094642e-9,
-7.268346803414742969341782835751162193838e-11,
1.643351499725199962100964768176579568907e-12,
-3.192117393727591945428103505909551636013e-14,
// [5.1429,5.2857]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001860681514530510523127347861361792317427,
-0.00002490878500445761146089509479055915427558,
1.673683654537988171670867449823465825061e-6,
-7.523000274306222367770416080661572366214e-8,
2.543921363144422026587852161208953570238e-9,
-6.900934319918394224352242672877613174253e-11,
1.564195285411267827659417358794565361257e-12,
-3.045290312514362809149366260779012370562e-14,
// [5.1429,5.2857]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001743614950312032667191442221762281635754,
-0.00002343157116350340461557037082948669786055,
1.579830057605603387822428422663735096956e-6,
-7.122979930524569753604806312485180444148e-8,
2.415326994682174644676293652245273216493e-9,
-6.568528478286253372384143952922836703268e-11,
1.492247578893018335907677640788173488264e-12,
-2.911267148642100270540465405447753861484e-14,
// [5.1429,5.2857]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001640209981445238193489593968147271760113,
-0.00002211762080645121695946031465848712517037,
1.495825785411403818116364481522057601308e-6,
-6.762915694032965629523204030763290476537e-8,
2.298984950811203843799004243097314597444e-9,
-6.266394165358019597735228651055357435554e-11,
1.426573987699980589426972315808131867666e-12,
-2.788454793919554810967801257157708066458e-14,
// [5.1429,5.2857]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001548233456451585062609733969630401631814,
-0.00002094156099573352591771903088367299317918,
1.420212295748118492516137223142430796576e-6,
-6.43715796678247343886374581022326698367e-8,
2.193237941932451194534535054114525841778e-9,
-5.990607435616357285285374838483449259259e-11,
1.366393866268464612686759029308128579871e-12,
-2.67551228738804830560001974675995617877e-14,
// [5.1429,5.2857]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001465909269701346694359608344761575150014,
-0.0000198829721404485490085945773381744024778,
1.351803173025470283316377752771689481329e-6,
-6.141066337851334218530096808667519084618e-8,
2.096712587120859998770387666561371681299e-9,
-5.737890003891294364651947066483354690346e-11,
1.311050124494230128982982723129500131583e-12,
-2.571303098382600653845811261680372324678e-14,
// [5.1429,5.2857]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001391808049831398315056873270161122527465,
-0.00001892524442233241591113668968039762219347,
1.289623930949889421460746203255105285207e-6,
-5.870795340611521236108118862475297711452e-8,
2.00826148657209746008960499359226065242e-9,
-5.505482455279387424507302904249313192831e-11,
1.25998584584763795898666291131191461304e-12,
-2.474857798537696062777341706885018828757e-14,
// [5.1429,5.2857]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001324767109563269002269772832351874623538,
-0.00001805473503327515798147708048252305255484,
1.232867021529489959419943368558946444815e-6,
-5.623132255578410738581027805464086315701e-8,
1.926918845074408027292988323425420958264e-9,
-5.291046052280156658427727250155104880313e-11,
1.212725996190703562544152996343633852583e-12,
-2.385344587567384489200003588735194326747e-14,
// [5.1429,5.2857]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001263831452329260950957672972159108275419,
-0.00001726013830139037896236767226123253310409,
1.180857773672500619656981481864049912444e-6,
-5.395372856131054324002257660021242386975e-8,
1.851866104506482491210931056786950855166e-9,
-5.092585920630854373993237918924474695534e-11,
1.168862981035375726661457653473887427884e-12,
-2.302045818143561830778551631789353661614e-14,
// [5.1429,5.2857]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001208209681097326423139476389546172261794,
-0.00001653200883139328704575738952334147580016,
1.133028299788521985587968458504409360386e-6,
-5.18522517950547615575031328529896842055e-8,
1.78240505887972150333789287755799487566e-9,
-4.908390396859331745475224271935659269095e-11,
1.128045142431363951898651361087991935668e-12,
-2.224339147580586784502936778888076063898e-14,
// [5.1429,5.2857]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001157240618197529992274599700371215984452,
-0.00001586239619701829569514387198545390245736,
1.088897287697118910799154090278838147548e-6,
-4.99073424891241520782250711169770367776e-8,
1.717936625981815959092527930333414499735e-9,
-4.736982720930765175713540536721037781059e-11,
1.089967523045136903994658213157853651606e-12,
-2.151682290183587650721969311655629753344e-14,
// [5.1429,5.2857]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001110367733791280600828086805105498857817,
-0.00001524456202773931749591794563828260755461,
1.048054192272546385821539863240057729428e-6,
-4.810222634138865111408628718814482333793e-8,
1.657943939803164385623758744436109504016e-9,
-4.577082250037645564864054129556712533077e-11,
1.054364394608476749743974426367101251788e-12,
-2.083600595353362030917836957142431219371e-14,
// [5.1429,5.2857]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000106711934194175212979512903063671951154,
-0.00001467275869179592638968664524815731947667,
1.010146753170072902002527783499566933255e-6,
-4.642243110341644924104257586689814898118e-8,
1.60197877536342130362180050416288253243e-9,
-4.427573081903723602929260065181134549527e-11,
1.021003171023102942988800009431505372695e-12,
-2.01967686084120154845161811552748116728e-14,
// [5.1429,5.2857]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001027093108425714755104014643536258829819,
-0.0000141420545443618848505614821650390504375,
9.748710531726303097964965353543068367154e-7,
-4.485540647561406199988477847198277067051e-8,
1.549650566867921336069479268674542260833e-9,
-4.287478492975536844730293561451738962783e-11,
9.896794167171157431251819806579189036012e-13,
-1.959542927102257746356421763139715016724e-14,
// [5.1429,5.2857]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00009899438181053318499565896196001818689506,
-0.0000136481947443982419693550590010915221392,
9.419635359887553007736898709574807950767e-7,
-4.339021661560347978041072480328743049866e-8,
1.500617461074749783660679748402080761324e-9,
-4.155939976135639030942697790402151703027e-11,
9.602127277626279451985350863873492942044e-13,
-1.902872700828354258260691398211851209433e-14,
// [5.1429,5.2857]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00009553736321078768507222572529178534527545,
-0.00001318748950382451425856152115695392826472,
9.111945489285095529166349320419131436291e-7,
-4.201728963249786967662418509808528951646e-8,
1.454578980494403169804197391429763370622e-9,
-4.032199943579949822471075360789804636584e-11,
9.324433133099382309476330720618029465676e-13,
-1.849376332804622457099762436248408858597e-14,
// [5.1429,5.2857]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00009231242652677159132868091986793263152865,
-0.00001275672368498156773420141110581754957052,
8.823630822832694676647269487286364700824e-7,
-4.072821215649016293341935028821770813609e-8,
1.411269969396887175629403602891746450137e-9,
-3.915587370672792409115569769965639781565e-11,
9.062291426555929816768322782162991171798e-13,
-1.798795333867604580620499938234856646439e-14,
// [5.1429,5.2857]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00008929706579487096587817341766682265447025,
-0.00001235308315194867427465227599754719521834,
8.552924552870865046731218620508710820847e-7,
-3.951555982705018885837268186152759129388e-8,
1.370455569161001422826998273701067334457e-9,
-3.805505815176698111901936740443034532272e-11,
8.814435519926733092266747252260998956846e-13,
-1.75089845770460657472846400630187255396e-14,
// [5.1429,5.2857]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00008647158206364071187025041790154496199487,
-0.00001197409437400255634132424321865026639642,
8.298267563688269925810748473014368047456e-7,
-3.837275660270328668831217044332473223428e-8,
1.331927025004789475049047921967617697118e-9,
-3.701423366911189049667036613868982759336e-11,
8.579732269236279787965274894294357513621e-13,
-1.705478213969365329719089948839709132366e-14,
// [5.1429,5.2857]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00008381866061801788653559874857336305387831,
-0.00001161757458914734435838182918156761281866,
8.058278886575229770317168699346257648015e-7,
-3.729395734948174701732011916423025577906e-8,
1.295498168366128480292573275411412371682e-9,
-3.602864175418948553958793057370639778435e-11,
8.357164938366812169077172253591863375623e-13,
-1.662347902212747021070409125702871112652e-14,
// [5.1429,5.2857]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00008132302212403140284410805281757352324075,
-0.00001128159044118948860976252293610366131114,
7.83173104339852489649725980831025157303e-7,
-3.627394934758033210775247103374730916661e-8,
1.261002451585902287978126025956419503963e-9,
-3.509401274696827293487850530542381149863e-11,
8.145818664887648605494624351971225335395e-13,
-1.621339078291695395095873086038531478392e-14,
// [5.1429,5.2857]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00007897113308832641278400993422442135937937,
-0.00001096442346074248302608229423816629677941,
7.617529362999054737029654877554529459823e-7,
-3.530806926248427443777984555179896579888e-8,
1.228290436563865189440793493862371564077e-9,
-3.420650479658943740804543330870435507677e-11,
7.944868045087927673361630870298859864849e-13,
-1.582299381600424343754105041481186592087e-14,
// [5.1429,5.2857]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00007675096422519737387022434052447868481072,
-0.00001066454110818358816216363209770332860372,
7.414694545128717528494308364301794673664e-7,
-3.43921328273327308255415359669288492064e-8,
1.197227658520737100158585112829217039046e-9,
-3.336265172551825379414397437752305246894e-11,
7.753566487088846684748245594536630012533e-13,
-1.545090664693975362481184343523421133517e-14
}
,{
// [5.2857,5.4286]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.3824867334834539459830204982066929160692,
-0.005036965026798728964970656418584602995522,
0.00009624943722784210176634579323510565273946,
-1.925073368167410469361696510473348709859e-6,
3.728255644419268493242307470415975805723e-8,
-6.766150152751655751006881024939289882849e-10,
1.134628151885515267508417908281234814642e-11,
-1.749321300688015886198158060576945500902e-13,
// [5.2857,5.4286]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.03525875518759110269940443905145116921948,
-0.0013474921211896537948087832124004770547,
0.0000404265407315209373341061832598659698084,
-1.043911585862629475833185691248548471804e-6,
2.36815254637309146168312385840564512765e-8,
-4.764917413026863693622581454391434701165e-10,
8.571901253373707964496648947658301780092e-12,
-1.388870950375139158563739258990826599279e-13,
// [5.2857,5.4286]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.009432444848327576516791827285259372151612,
-0.0005659715702411814557549782657062664286247,
0.00002192214330311971849921194256073183613825,
-6.630827174511749163288729251770525997848e-7,
1.6677210885600441533724120142516099658e-8,
-3.599769720249438168914363781549275596825e-10,
6.805659636636834089368142478381124621224e-12,
-1.143487063952796940215489774047162557962e-13,
// [5.2857,5.4286]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00396180099168827014977609059674489521521,
-0.0003069100062435815694404225118700044683628,
0.00001392473706647856210125666301031679217902,
-4.669619085764227867702475086990157353332e-7,
1.259919396902144769454589217494689821257e-8,
-2.858014202963922456412549947690882106901e-10,
5.60325253874485663072324024951361774079e-12,
-9.675886015878692256128974150537384690969e-14,
// [5.2857,5.4286]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002148370043705070950480093796213217521981,
-0.0001949463189306182034746154476146383158488,
9.806200080108296413000853616591216481131e-6,
-3.52777434399263628153180265467066117241e-7,
1.000304966480172370414253213506583456512e-8,
-2.353052464992020918334483471490979498907e-10,
4.741329978467616960095512302036853059358e-12,
-8.362731753733746354120343790063678268094e-14,
// [5.2857,5.4286]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001364624232514327392605030621716749384777,
-0.0001372868011214443741488813645359590981726,
7.408326122387581064315975878988416480346e-6,
-2.800853934854964458136932431066367086181e-7,
8.235683586873649559453822998047376790984e-9,
-1.991082968868069379628927851847103895042e-10,
4.097868474531984848937348694488823226334e-12,
-7.349950219334328721155399841904663793797e-14,
// [5.2857,5.4286]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0009610076078501105904708138622773126436754,
-0.0001037165657133621926328101425448626072209,
5.881793263198168225482102763564711541432e-6,
-2.305991429901736734981769332924827259459e-7,
6.968790354466624913774122406561727149507e-9,
-1.720859217670783132465740839294867877143e-10,
3.601592636877256737898095935562237708273e-12,
-6.547802204301950974642514205469067495399e-14,
// [5.2857,5.4286]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0007260159599935353224528126619691492014699,
-0.00008234510568471675508728821590082687139947,
4.842582002796140934083175826751265579384e-6,
-1.951261322290873185366018311161895202577e-7,
6.023007228597101573639003315201975642785e-9,
-1.512447720174909212735775372047013834041e-10,
3.208529482362581881035828297445612746625e-12,
-5.89835157496594105185650723543841466379e-14,
// [5.2857,5.4286]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005764157397930172618076650125246455135561,
-0.00006779614803909360353114480788903109625303,
4.097648776813118822447429436936609345325e-6,
-1.686442044955182509997698118889183082292e-7,
5.293566990143649148711762124276700683599e-9,
-1.347381280724977929860164356988320188315e-10,
2.890289771232737544664852395073526105828e-12,
-5.36273787568563977823343105609082892322e-14,
// [5.2857,5.4286]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000474573036273655202760114203568382119944,
-0.00005736708287533567577042304590630245851893,
3.541528294407991239915789535401806125044e-6,
-1.482198776435482277673182858157419430513e-7,
4.715834454431085892737069018652879814238e-9,
-1.213737428373719812481621454377616620955e-10,
2.627831499544373714952984120993142743826e-12,
-4.914034375652233874533049471260141320085e-14,
// [5.2857,5.4286]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004015695801273497100203089079384269268733,
-0.00004958139612166761006082867757941734042248,
3.112617430516468567648269386945997922135e-6,
-1.320433664947775221612607405064190542611e-7,
4.248080973230812992054525872718094522917e-9,
-1.103519240952745191055557881469845985637e-10,
2.407960291298931328121806201750540351314e-12,
-4.533054683733133799024805473317070322827e-14,
// [5.2857,5.4286]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003470697728516732514287790109778903501869,
-0.00004357664402718948851806188697561799249867,
2.772910696392151688784043425879399189969e-6,
-1.189462688933341582535746727573087220774e-7,
3.862317319018221710211377721575240823269e-9,
-1.011185605780704956232203109749676323084e-10,
2.221274607624817296318144935494016017455e-12,
-4.205792414562477222798386428633145990993e-14,
// [5.2857,5.4286]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003050365081903264018338580542885086937739,
-0.00003882074974945182549507930411590311833006,
2.49787164676172541932538885192305245142e-6,
-1.081448864644495073496131302449141860904e-7,
3.539149597457783724751026804453994258092e-9,
-9.327882681731208916856787597657509474105e-11,
2.060911162273145212723081874874414145516e-12,
-3.921803763435193858730135739411903227082e-14,
// [5.2857,5.4286]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002717452482461627617359257173277449352654,
-0.00003497020305462828589554663402639673697616,
2.271042615755045707080580463968701932815e-6,
-9.909619016362957245450857961857521641245e-8,
3.2647589171918200826725938263455849924e-9,
-8.654449454302326950195510627942888469626e-11,
1.921752369354881850617623292704466112086e-12,
-3.673154713741159458545360045453995940462e-14,
// [5.2857,5.4286]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002447914213823979854839084913051413897322,
-0.00003179459662053691283041864947520647683841,
2.081019993437736381559754286714575164528e-6,
-9.141325103046566254355086495914161373989e-8,
3.029057288800951093670340728618549739907e-9,
-8.070064812729052754683018088552684633523e-11,
1.799910465430547371579653303413296442238e-12,
-3.453717400555203679608097329927172394306e-14,
// [5.2857,5.4286]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002225621763437583748729694966127906146761,
-0.0000291342799080964868183492364497828762996,
1.91967827164121315718502246028743235871e-6,
-8.481360535933891392607407010501822808134e-8,
2.824522665331858113463330438939248097505e-9,
-7.558401951756243682313672826674401051731e-11,
1.692382720993338793122307969011384638788e-12,
-3.258688634264343594326785896647551095729e-14,
// [5.2857,5.4286]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002039399593566753935485395939971595509832,
-0.00002687549580294686511645508526184563938534,
1.781085712547478470495641199486012144469e-6,
-7.908663583406618662258720483142312776098e-8,
2.645440664964254016135424808474910098574e-9,
-7.106850838070728354258115564264833689521e-11,
1.596815512291837583058656794441665386752e-12,
-3.084253428695756540936633185759000980022e-14,
// [5.2857,5.4286]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001881284706206280423225042196169441552598,
-0.00002493519997563611178135201375681934271274,
1.660819352516685223331755084794446318301e-6,
-7.407233976248163053634452314991186598496e-8,
2.487397776053976729527836493293681636486e-9,
-6.705527401819767200258649283205978511335e-11,
1.511339446667969382709437844364600326875e-12,
-2.927345365610204019838045670057636001841e-14,
// [5.2857,5.4286]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001745463998294527696010764993943174185036,
-0.00002325147093520639176918219021667457995399,
1.555519135013349613027398393925904328792e-6,
-6.964713881757548588271852946901294164511e-8,
2.346934574165242558587214988904029109074e-9,
-6.346581128136538934146916992037413811539e-11,
1.434451938623947807725644954266319879979e-12,
-2.785472978103216138600716939846109397665e-14,
// [5.2857,5.4286]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001627602965464447300854496939112125106856,
-0.00002177726789016095180592472734643147899169,
1.462589915170265897086260172074544470777e-6,
-6.571416911434727162055072302958927502826e-8,
2.221303379105157667227430176951445163451e-9,
-6.023701924529920822263256514193529182032e-11,
1.364932135796962909968151583226032832696e-12,
-2.656591993954363011744261260485630785499e-14,
// [5.2857,5.4286]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001524408752311266508643205859336705464024,
-0.00002047625881235892802401812845979139403276,
1.379997551402423316713891406448324942104e-6,
-6.21964956067348602611041993409370235521e-8,
2.10829565851058818864287915506582432639e-9,
-5.731762845735132886488687488327117476618e-11,
1.3017783167698698269321357318116749048e-12,
-2.53900998349288975442952081261314125039e-14,
// [5.2857,5.4286]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001433338116865124848705138170622847722174,
-0.00001931996571961018359585997865501225769009,
1.306126407742516642153138050173815055059e-6,
-5.903227938801867883983821415723080892073e-8,
2.00611698154622773137703403240540722943e-9,
-5.466557191554368386371557571566661872654e-11,
1.244161167430836370683473803489368416052e-12,
-2.431314264094377301967610991852248406004e-14,
// [5.2857,5.4286]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001352397600372712743156864382837880551506,
-0.00001828576970837245690720659131689980493253,
1.239677867149434373246953179912739920025e-6,
-5.617127639434605081373191428996944166232e-8,
1.913295003149082641349957053894331118947e-9,
-5.22460228825373809130715922742622568309e-11,
1.191388453330007112805528179208523837451e-12,
-2.332316736224808725906135131470732614338e-14,
// [5.2857,5.4286]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001280003879586071879042869727478631900986,
-0.00001735549014007019678178557476480512838489,
1.179596804282269903774561672864600820944e-6,
-5.357226096356011396825814796887107420004e-8,
1.828610787517609266021578225853549863539e-9,
-5.002991128463977283283700053953420555396e-11,
1.142877988679434434808671550433552028077e-12,
-2.241011209402043994823606226280861166797e-14,
// [5.2857,5.4286]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001214884309804913674059708068994890794385,
-0.00001651435525993071910766218144614780123236,
1.125017480235728785265103701322814685627e-6,
-5.120110289288263733109097113333290110066e-8,
1.751046882077124509801937903934213637603e-9,
-4.799278853485631632545754676987380920256e-11,
1.098136725552266150978476733879718203751e-12,
-2.156540054162364913096871809245604973393e-14,
// [5.2857,5.4286]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001156004868195150240402660301193831765363,
-0.00001575024472327990878751904257744909160924,
1.0752231607514678724515083361818072994e-6,
-4.902931350993524110949206028923405967748e-8,
1.679747586286750677279860997980289576053e-9,
-4.611394937790276395991636747190200661762e-11,
1.056744412931034315142645181902718826687e-12,
-2.078167896297995896138128765638129061189e-14,
// [5.2857,5.4286]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001102517130629593521286129174144324973135,
-0.00001505312425050096797942389242692547192065,
1.029615583709540934146918920898399289256e-6,
-4.703293319932567570836250820865483577142e-8,
1.613988216214946308757630849059911467543e-9,
-4.437574564872891725343295877272258780413e-11,
1.01834070655086838623419601080192719708e-12,
-2.00526068462679834877006349285694563518e-14,
// [5.2857,5.4286]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00010537186975350676677980374705803263175,
-0.00001441461817191465481308185975395513387975,
9.876915971867105063119808004446271246884e-7,
-4.519167081075612578894849195861270826744e-8,
1.55315108608792169563881919686675543525e-9,
-4.276304494890348180137777494371927644758e-11,
9.826149118373409663050879717894787030865e-13,
-1.937268898966160595257529875718613061959e-14,
// [5.2857,5.4286]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001009023272034025749038023623660452955599,
-0.00001382768236059564946532518169394715772993,
9.490250870267222721495570504972537291021e-7,
-4.348823114237354275718624340874873254203e-8,
1.496706561963177318180322222732448063539e-9,
-4.126279990090246096924206428216229796178e-11,
9.492977555944472666234872497704575243551e-13,
-1.873713977057505953179333901024412561379e-14,
// [5.2857,5.4286]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00009679377652416953774021602808138739186839,
-0.00001328635121835639558933929932118547217198,
9.132528539906620401494738740281478031979e-7,
-4.190778444362440498881258158323726253531e-8,
1.444197985629653764831324757037616964814e-9,
-3.986370260043838532256400246697794138816e-11,
9.181547350179653948660936065063033583392e-13,
-1.814177265439861577085020510414329227147e-14,
// [5.2857,5.4286]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00009300445852849476086291426497674543327893,
-0.00001278553995585209809570969395525747424677,
8.800634733169057053439242267470299953762e-7,
-4.043754428445537873082839197289072289757e-8,
1.395229580439301054626164324217441683494e-9,
-3.855590530897526253697771586216170597381e-11,
8.889807034753693454248035680699337281894e-13,
-1.758290965069178192142370331815708331032e-14,
// [5.2857,5.4286]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00008949877969096467864738670092115839984974,
-0.0000123208886264200226829077415703797066399,
8.491884299743331253584847367262261705807e-7,
-3.906642891859434270033836795296727757208e-8,
1.349456675545140280344718683881493926202e-9,
-3.733079308448276142374652084361854787944e-11,
8.615954337364272819061288996961562608976e-13,
-1.705730665203682778874953361274828238777e-14,
// [5.2857,5.4286]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00008624622038494015098416906984889013279632,
-0.00001188863801962449016247893425701934118277,
8.203950072912296345860024937246307008698e-7,
-3.778478756221369511687880007071447480681e-8,
1.306577747977692072788493497796639351563e-9,
-3.618079746034361835001692946659217474259e-11,
8.358399594745493610330773269526303194703e-13,
-1.656209150775808555734313175783983007635e-14,
// [5.2857,5.4286]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00008322046613737142355518173663954918974458,
-0.00001148553010206149770791395276352157750097,
7.934805388072154898519040170110085213773e-7,
-3.658417757206833322171129089057143839115e-8,
1.266327901406762673217545269699590969716e-9,
-3.509924280777814851326480625418634980063e-11,
8.115735407931205523134409270902548458075e-13,
-1.609471237600915564152056944944488613511e-14,
// [5.2857,5.4286]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00008039871071443047657584487642011499975928,
-0.00001110872754328573113624801994630759389861,
7.682677290141434386228911679826966363629e-7,
-3.545718185082397189297416685652203570209e-8,
1.228473488826324447640711469067250775391e-9,
-3.408021890418786654406245458284339193349e-11,
7.886711334053906131454612186715696952111e-13,
-1.565289442324789149413473885124700095423e-14,
// [5.2857,5.4286]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00007776109280300011076628890684070903949315,
-0.00001075574820618313089860852459718940835912,
7.446008188679934084891348150393370871922e-7,
-3.439725828223237300039302629832762957305e-8,
1.192807652446561754440594493806771709559e-9,
-3.311847465239136656620562097989675347636e-11,
7.67021266847479351272140380565445720293e-13,
-1.523460334277609245476527374658569313676e-14,
// [5.2857,5.4286]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00007529023744328190928520057230525294259784,
-0.00001042441146413741776312141021939815379466,
7.22342423927552322392316973950897367457e-7,
-3.339861484810742346299537025021841909602e-8,
1.15914660386714302491101863963079096386e-9,
-3.220932897715046234704397238180830924271e-11,
7.465242568347305002955805391414747382337e-13,
-1.483801447484927454007384046729030732291e-14,
// [5.2857,5.4286]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00007297088024896191751017852325247460938853,
-0.00001011279393497161025374220754211500670363,
7.013709118109117367952801122176117834064e-7,
-3.245610547317572795901172230523454032483e-8,
1.127326505455649566660159456995351483447e-9,
-3.134859575398233113949141097946805036875e-11,
7.270906921026163593327789489041856657359e-13,
-1.446148655250304864719101708405936609681e-14,
// [5.2857,5.4286]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00007078955754480126510955985095917426233802,
-9.819192765338991606520620780932623186689e-6,
6.815782149373302876964554681587586977541e-7,
-3.15651427036717398349166652954854115404e-8,
1.097200842856012511381002440843954779667e-9,
-3.053252026488317751729416891031987720887e-11,
7.086401479141076989771428456894268344572e-13,
-1.410353928635776770167474086961006604693e-14,
// [5.2857,5.4286]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000068734349357372934736280199690163106203,
-9.542095009109184032565905297923064066034e-6,
6.628679967777314389509221217769033042908e-7,
-3.072162413757324510222148784554930971122e-8,
1.068638200938851299815068636686905239605e-9,
-2.975772517284299632338348451202105443369e-11,
6.911000876822990807284594321214976928429e-13,
-1.376283415061268790074186796898073425016e-14,
// [5.2857,5.4286]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006679466506376428186863832598097161116289,
-9.28015195487511721068653571672823127685e-6,
6.451541068896486455391460982518574030939e-7,
-2.992187015121019421801946300917650501305e-8,
1.041520372909499237054278392787508655795e-9,
-2.902116439619553178322254324117353601543e-11,
6.744049214557959579569178684144082006555e-13,
-1.343815785046566619893499504514528700576e-14
}
,{
// [5.4286,5.5714]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.3775441294376290757418829759728898659535,
-0.004850095558430714399327529154322846064337,
0.00009069131288366439870253466245947569004761,
-1.782488376823474132325621071536513750215e-6,
3.406370453101305016404677417160898703719e-8,
-6.120831142758442551852471533206777764811e-10,
1.018852732497295588167425997846920271678e-11,
-1.562123396341914633477242085386602599336e-13,
// [5.4286,5.5714]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0339506689090150007462016217514433514276,
-0.00126967838037118097862150068151650501547,
0.00003743225591329766954262965315772136151623,
-9.537837316925271085447927934172444940767e-7,
2.142290893681753927119264139188347968721e-8,
-4.278718354843798711967104820400930001534e-10,
7.65460572084572086748238076852883800869e-12,
-1.234995093428428583550465830660082385284e-13,
// [5.4286,5.5714]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.008887748662598266808919448843708314154002,
-0.0005240515827860684056500002193881492794616,
0.0000200294583655470466782589253885683598175,
-5.998414541896375582125829963094592307863e-7,
1.497551418892117165283107949033965486616e-8,
-3.214554361265660046007503034233340251634e-10,
6.051645660893389557629291174488001868726e-12,
-1.013447456916181461773287856183962763158e-13,
// [5.4286,5.5714]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003668361079502478803817165799214611753478,
-0.0002804124171175751282018694048134897295295,
0.00001259667053798581909021758131828338605562,
-4.193144006308296371985189452510045707623e-7,
1.125094021869144926785244596011961559377e-8,
-2.541370436434281876648811272744334970168e-10,
4.96603890190906989204110615927889276433e-12,
-8.55312766661414677302399443435843081751e-14,
// [5.4286,5.5714]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001962886919823025866061937196405766745588,
-0.0001763533875317294296126633782116466379712,
8.805602413250432105404194143818617020982e-6,
-3.150263290048889062330676348294281138864e-7,
8.894796487390218367876913115027844052735e-9,
-2.085459710654503855168531912542647618354e-10,
4.191160972137530936560496563595461670539e-12,
-7.376777793745599879016241947118040222631e-14,
// [5.4286,5.5714]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001234473712722105979399113475552198603135,
-0.0001232784337854428453318912295073038478457,
6.615552909105344438475756791633318467671e-6,
-2.490543041751119391694177383694820113003e-7,
7.299108951591894262509112682148361129894e-9,
-1.760044901173821372213593593946864864604e-10,
3.614735355533372075261833438881629460392e-12,
-6.472214443369597648368383381431117910144e-14,
// [5.4286,5.5714]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000862949036498099892230296529118332221799,
-0.00009261774072741859663864704706393924173215,
5.230140387679759656610405565164668468629e-6,
-2.043750528936112724113814059962202411341e-7,
6.160157121989161194656027182432696870982e-9,
-1.517972940486754377769015493289832519669e-10,
3.17148785893284965393424820213223111221e-12,
-5.757591220166763132534944814572414993507e-14,
// [5.4286,5.5714]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006483241850919301536796679277329327864054,
-0.00007322196542746604763163405236244931761166,
4.291876110768024654038185129221314181318e-6,
-1.724844014392156421288632400133529489062e-7,
5.312905262531109362289965398861623564203e-9,
-1.331830641847135755291255330744958475251e-10,
2.821313050163479798863170647652369628391e-12,
-5.180257806424457354899866202363601857977e-14,
// [5.4286,5.5714]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005125537579922623125559745973757111334455,
-0.00006008626555070639860510557904458261690619,
3.622172430225531579153083136520809026947e-6,
-1.487613491887485055616519188490023954413e-7,
4.661407219756969782564044101763848587728e-9,
-1.184775043849367110279345900834443943847e-10,
2.538411790934529814263894516901258128977e-12,
-4.705011271276438917029096682517707452726e-14,
// [5.4286,5.5714]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004206038588549447710027342530454193233279,
-0.00005071041402311537717107321176649910317082,
3.123988332965564994504965418247914636084e-6,
-1.305194038358069025617779280059301440211e-7,
4.146712628854341419455154117683314626368e-9,
-1.06597142055161210450639204095739862163e-10,
2.305534302102407105891439834183194372954e-12,
-4.307527814664798231152286519242593694054e-14,
// [5.4286,5.5714]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003549728981618076223650258125247224795244,
-0.00004373583666147913603431680108594917618739,
2.740907480553656881176096280514989136494e-6,
-1.161079551588904045386889384772492751653e-7,
3.73089994910487471411007330106330171917e-9,
-9.681755130230457365740071437578522477988e-11,
2.110761671790009204172460651232164806284e-12,
-3.970519139418595026905478275728981500557e-14,
// [5.4286,5.5714]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003061508566303539356216994583500435219938,
-0.00003837270472771524590191653986959580297655,
2.438267058338293881192155965858415481746e-6,
-1.044652000129663191685971060363893658797e-7,
3.388614274308783249423705824781923475873e-9,
-8.86381850491896179214287132841523661873e-11,
1.945622448459842337342070422332223621679e-12,
-3.681392769004074648279155596416086466163e-14,
// [5.4286,5.5714]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002686089330940067057567370129599566842492,
-0.00003413573881670261127119015374739089004395,
2.193769200273786151376690540373882896596e-6,
-9.488120102078024721179099259137610986708e-8,
3.102336456808924081240320677470385488372e-9,
-8.170327747102127793890075913980206668745e-11,
1.803946177623125303798361693391893769992e-12,
-3.430778069624582995150545189958471792565e-14,
// [5.4286,5.5714]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002389501717169182642779035869373566875657,
-0.00003071276880380164372952842822457636387641,
1.992505221437788759747596866282828479063e-6,
-8.686542204515649516842868458423354473772e-8,
2.859614692771442639461532325368902508794e-9,
-7.575369612601130742461547740621589467436e-11,
1.681141140006651373228280272673675320371e-12,
-3.211569223538235198584117567570146503248e-14,
// [5.4286,5.5714]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002149893816266114923177071476970519425547,
-0.0000278950731000995677360505470579649031612,
1.824173862949610148643820746485781601544e-6,
-8.006921257660684714975395008209363530462e-8,
2.651379346760339221319520447815666678573e-9,
-7.059660935153347142020857643482590457968e-11,
1.573725399832236233277780317939484357556e-12,
-3.01828705509072091794714121797517160522e-14,
// [5.4286,5.5714]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001952655117006969611064582175903007299734,
-0.00002553843408126674336091925170872386606925,
1.681453464109996202676845641489910944739e-6,
-7.423862282124820419693681073473164436696e-8,
2.470881310604785312876957719219731474841e-9,
-6.60857919260016503574149443478516724856e-11,
1.479014093542095113494029805803130946821e-12,
-2.846643258683729617351314523831424392218e-14,
// [5.4286,5.5714]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001787690385688671911485227367416848047243,
-0.00002354034849751364620488477906364557704074,
1.559011079247400573938354983444457560776e-6,
-6.918467774896871175353752173058686635223e-8,
2.313002701566197073580847549903626281462e-9,
-6.210849233514303932708353614761717784849e-11,
1.394905897215870399497394521251173365572e-12,
-2.693236449407210628034931764247888561072e-14,
// [5.4286,5.5714]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001647824394825955116598867649600032429983,
-0.00002182615510943865406243847256164150218271,
1.452878232729473286236510232341257411574e-6,
-6.476407664202140693442310729061802221365e-8,
2.173797216658766452064525420329396674749e-9,
-5.857646521385502262532522014069894330563e-11,
1.31973408827918268829421247284951923617e-12,
-2.555336071372909145829705480994802049591e-14,
// [5.4286,5.5714]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000152783085766070567210788065232923568454,
-0.00002034029525818888890751437404795163710824,
1.360045609483527274337685380175352032426e-6,
-6.086632301593803368856393926599514632645e-8,
2.050176268115163870160127427742196663128e-9,
-5.541971652410480567518997467572846283338e-11,
1.252160658308420781654404177802974675117e-12,
-2.430726093179983227962782705581234937013e-14,
// [5.4286,5.5714]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001423820668073222116260418703021348998657,
-0.00001904063853274674956438846120322084021332,
1.278192783335728462581196742329963508647e-6,
-5.740493641252385710273824716210619059461e-8,
1.939690064613556061200805615085195575423e-9,
-5.258205672339634474987673936700675619123e-11,
1.191099722110584402460607452453627427957e-12,
-2.317590163409666857512438976125960873436e-14,
// [5.4286,5.5714]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001332844697292272366816486339288658514607,
-0.00001789469896667857364416820870079586969895,
1.20550366466398683520042561304885336759e-6,
-5.431132267418072734881537634132731853831e-8,
1.840371972174349366783714662870878857382e-9,
-5.001788426710600784088926536376700463641e-11,
1.135661242633246697984002271890344983631e-12,
-2.214426015120781859632102399816752833133e-14,
// [5.4286,5.5714]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001252628927667500056604602301572300165915,
-0.0000168770513052751131604779383255472145772,
1.140537776158740756245654779220416952358e-6,
-5.153041604899064425989238760170494915779e-8,
1.750625936742244135956897292869404079854e-9,
-4.768982229708018539887932474305708691419e-11,
1.085109088187293543263341105764202670219e-12,
-2.119980831219933946247496101632237014674e-14,
// [5.4286,5.5714]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001181393591369257826621622920763271786388,
-0.00001596752886620251549075050537359255624253,
1.082138737029711807933632493726210523636e-6,
-4.901752702299398551305538873370282472343e-8,
1.669143768287387759050138442960757589586e-9,
-4.556695723022302696919023238515786418672e-11,
1.038829360721013238308883742520428441631e-12,
-2.033201848937851649042190454815878996365e-14,
// [5.4286,5.5714]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001117727020634175993324603466042269528415,
-0.00001514994231839689148585629583784896113949,
1.02936806748374756862983146360299703592e-6,
-4.67360262750068820815909305876868911644e-8,
1.594843491406130193374594758650742207173e-9,
-4.362350868913994331829619452859310175583e-11,
9.963061914223226684162086072566947036805e-13,
-1.953198191255839150980614346023045163221e-14,
// [5.4286,5.5714]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001060495962287782316305965333045698751411,
-0.00001441115294475411465282492585565855650438,
9.814565517759864864259501279017250966455e-7,
-4.465561849343073407929490514392135254957e-8,
1.526822792893692123811396578296495385469e-9,
-4.183781302916477081567448592955296689142e-11,
9.571030376518527808701208174497805999519e-13,
-1.879211071173421407021063289051579632041e-14,
// [5.4286,5.5714]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001008780706132787941084919214080981282825,
-0.00001374039172484612961422777019391353276919,
9.377679883628577031899616769577355850614e-7,
-4.275103890827774729855152071701301109824e-8,
1.464323445190231455609000006378990384376e-9,
-4.019153789760053317479276554117020278628e-11,
9.208480826637279562533702434839283853998e-13,
-1.810590311325207082504861646323521348659e-14,
// [5.4286,5.5714]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00009618274207392289912649668232982435128757,
-0.00001312875183706294982716769113650702320569,
8.977718170746173149083263882749400122871e-7,
-4.100105714765350340012905129159781489071e-8,
1.406703815954362741478316690969706491964e-9,
-3.866906909199191662106941480795626728137e-11,
8.872227299209766821641510448243255418443e-13,
-1.746775677326052940013524761399946530094e-14,
// [5.4286,5.5714]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0000919012628594406408863767240642270882763,
-0.00001256880543902816537417625653762104768214,
8.610222001014823400525910915433474845964e-7,
-3.938770750580965958394092843115572796864e-8,
1.353417408102768734128534579123770140941e-9,
-3.725702737747792581998073132564639799097e-11,
8.559524561949124293700998956558590878712e-13,
-1.68728191628259659887030201994786536644e-14,
// [5.4286,5.5714]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00008798163807319714996764182854489320645922,
-0.00001205431080140481863834736776231364488576,
8.271418576227374080946995020683699802293e-7,
-3.789568806424835197944217827352210374445e-8,
1.303995948417604286195203191949851170073e-9,
-3.59438843622788541110088597967405554377e-11,
8.267994802407685769337646129971684573512e-13,
-1.631686673295652174620711561101240396449e-14,
// [5.4286,5.5714]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0000843801756098337230535200341858013665537,
-0.00001157998600670289803853564544362347958682,
7.958094493499272269625546881438848918801e-7,
-3.651188717272566545627181538120254464593e-8,
1.258035943188590212049772708286668470911e-9,
-3.47196546189312333487892822798477371883e-11,
7.995568417241804342559697483345338213414e-13,
-1.579620662553417091153681200931627732837e-14,
// [5.4286,5.5714]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00008105990204692027907737846196375751491962,
-0.00001114133229088403265233225176322432447865,
7.667496306279294459794218491717997376505e-7,
-3.52250070072271178157077237364903529092e-8,
1.215187902456277653505152391230525164141e-9,
-3.357564702938203160422042368978731154655e-11,
7.740435849252683189167032733954839920525e-13,
-1.530759618784762778230994553107697468172e-14,
// [5.4286,5.5714]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00007798932603618822158358069998345007019333,
-0.00001073449482877651236198230904529802049911,
7.397251471524398213281248205295996533077e-7,
-3.402526184877647138147842796714398388032e-8,
1.175147637090379084183965399163911397036e-9,
-3.250426252543048593291877888236963670756e-11,
7.501008148416783676739702385082237326738e-13,
-1.484817665145989497836419633819028810664e-14,
// [5.4286,5.5714]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00007514146380143557974893092862843403306846,
-0.00001035615206012008022456707843970851937538,
7.145304988249572573273651418660241513551e-7,
-3.290413440162685377009231550872794745285e-8,
1.13764917970526086619256938987078549091e-9,
-3.149882846567049067879239645379270093362e-11,
7.27588447362180962735497121061987703551e-13,
-1.441541815956225618012156787688609080388e-14,
// [5.4286,5.5714]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00007249306442084055497396776812891199973069,
-0.00001000342698353572309461767688246053507691,
6.909868224347973408233369350955797900944e-7,
-3.18541775788927596551827104362582133919e-8,
1.102458987852950644788948188971598002951e-9,
-3.05534621599641078167939094090173004251e-11,
7.063825155308433130749376382353326469379e-13,
-1.400707394715105588370057869750874207026e-14,
// [5.4286,5.5714]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00007002398888475005524127595657216012210937,
-9.673815514073861143088925514838598453875e-6,
6.689377291573643767376756165128501917756e-7,
-3.086885219195275815800563252675726265317e-8,
1.069371167379730254400371574195499401941e-9,
-2.966295774693362114729585612072168708168e-11,
6.863729243115506873252972374952369254581e-13,
-1.362114194941642466293404913475857677449e-14,
// [5.4286,5.5714]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006771670859851702174831653659136409662565,
-9.365128208190156386599857019143274443298e-6,
6.482458960316082419961210234263267761103e-7,
-2.994239320443283143791368615851841783946e-8,
1.038203513138374572666822483283684158352e-9,
-2.882269190619415471353555497339481011936e-11,
6.674615693447906914325288601926228468679e-13,
-1.325583247436146246755791365880486760498e-14,
// [5.4286,5.5714]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006555589745733108861212166423120029777605,
-9.075442544429908669142560527284011422497e-6,
6.287902572936744948477430188658377999306e-7,
-2.906969887214799429544129007071215618199e-8,
1.008794208916307574769045912085720014639e-9,
-2.80285448563944389042851546236898423459e-11,
6.495607528598820761509251609396955348351e-13,
-1.290954085383804043659596505921749089759e-14,
// [5.4286,5.5714]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006352809781100935474126613001527793326765,
-8.803063602099157215246566014246059598603e-6,
6.104636763156783856074834764374403664793e-7,
-2.824623834108975587786827864854934158301e-8,
9.809990623670414436913892033535328109062e-10,
-2.727683383224549247970372311023418522857e-11,
6.325918435363796166115610912193276261034e-13,
-1.258082420327423391251189905133934585251e-14,
// [5.4286,5.5714]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006162144521469409470802628059956961240154,
-8.546491468407516799741618341591966230267e-6,
5.931710051634415516828221867381416206117e-7,
-2.7467974225505640073465999232392134185e-8,
9.54689176706190976750107734137351351214e-10,
-2.65642568061367176468748618340663266164e-11,
6.164841376970042768270543217531494362135e-13,
-1.226838158936492821639755252875054512679e-14,
// [5.4286,5.5714]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005982544027885261193672651614719377846709,
-8.304394072276491494724548208269068061158e-6,
5.76827458736161945173351208833710757272e-7,
-2.673129741538692514747100364748426753506e-8,
9.297489809680459745099756525793884937857e-10,
-2.588784466461096771623631290644135371227e-11,
6.011738874954867694569027108431510096835e-13,
-1.197103703801023327911483290827509574645e-14,
// [5.4286,5.5714]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005813075850593543493251349756944112796044,
-8.075584422294853670224766856310880251976e-6,
5.613572457236563646396036363286060885238e-7,
-2.603297192365209976936446190448725823216e-8,
9.060745561822064303632175603502739386568e-10,
-2.524492039773428261389364179270225561342e-11,
5.866034682805206965743448712886901642415e-13,
-1.168772492011452612258374337802395852001e-14
}
,{
// [5.5714,5.7143]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.3727829761653601840261030367018397329136,
-0.004673927143342093987257530013476243791116,
0.00008554225874218597653396893390790476298353,
-1.652155986495088480328227189748542511819e-6,
3.11507842203362595775644972815201082776e-8,
-5.541187700482267082482459682979161407819e-10,
9.154421644476904008413243331557581913448e-12,
-1.395633709899312854928233917675415443932e-13,
// [5.5714,5.7143]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.03271749000339465786728049124828650912372,
-0.001197591622390496388174952970702871671318,
0.00003469527571640103623744910201318592219182,
-8.722219624607765989299626674943674167831e-7,
1.939415689597912032639875713452675546148e-8,
-3.84444511811693560706410932539179831952e-10,
6.838783447012386177415652837855589747283e-12,
-1.098597074908295432971629057653478547374e-13,
// [5.5714,5.7143]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.008383141356733474680591523370075075505771,
-0.0004857338600295267662444892559434252323812,
0.00001831666121167982537505031967337904687541,
-5.430363965970837823531786296402935881361e-7,
1.345555786651851806256912791392192976846e-8,
-2.871952117963362605727102915775053015656e-10,
5.383275717733935341058245221408275162524e-12,
-8.984824928603207162723685571224513282873e-14,
// [5.5714,5.7143]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003400137020206687332184545022645863708903,
-0.0002564332569634437025744785582016676936714,
0.00001140376432854178602352602090374965973401,
-3.767556232166476444010702559791516330666e-7,
1.005183237251707178404255830905546157592e-8,
-2.260692203637199540936861322377201458533e-10,
4.402693350280034047278052189261177727968e-12,
-7.562635223281560973043284133961323194781e-14,
// [5.5714,5.7143]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001795032798744105890409090946293823882092,
-0.0001596527005995214459167749789758164810503,
7.911868087552251321550208848172563080851e-6,
-2.814513089728341354053685647692017739424e-7,
7.912422677386245216363515079112221564449e-9,
-1.848887139673763527669478260927723176137e-10,
3.705804360267498129940734916590893029688e-12,
-6.508489089574941397300606577470056511084e-14,
// [5.5714,5.7143]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001117568904196650096889854473086301837663,
-0.00011076615322567585198726740147176176241,
5.910477488431871501399562406964270936134e-6,
-2.215478371934930760963114021750840201282e-7,
6.471104957462644812257714285794530652077e-9,
-1.556224069075691586595489777074683560707e-10,
3.189260119769792403852609861819775027964e-12,
-5.700347806067139816708103803574931489681e-14,
// [5.5714,5.7143]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0007753630725797309418698982606014606528741,
-0.0000827466848379967532559111206221356235469,
4.65250458106547054166420871217075854468e-6,
-1.811909407868806002660209084717262101103e-7,
5.446784213552257216345351992973986746617e-9,
-1.33929936833105105503129150542776199069e-10,
2.793260705669332087421230037383231133385e-12,
-5.063538763077980254306962203667936370581e-14,
// [5.5714,5.7143]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005792267938659772527934259378628081422202,
-0.00006513506413487215281433361958177335792531,
3.80500975652641241834651207758248047302e-6,
-1.525099597568685941667364962472839243742e-7,
4.687547763561100207983385761298130942327e-9,
-1.172998864526927683800348334866478331101e-10,
2.481215906319397992655954985308645530907e-12,
-4.550200617545570359400057864979270098709e-14,
// [5.5714,5.7143]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004559454489441050514082917349884021808087,
-0.00005327013659132945783264817241371068598311,
3.202709154895996522375559385472665714029e-6,
-1.312513389923652520320216108146016132511e-7,
4.105496002430245731834933687110661021396e-9,
-1.04195586496470833276515854604754776394e-10,
2.229673227549328127789770649069016345354e-12,
-4.128434815623731266005572397145286725682e-14,
// [5.5714,5.7143]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003728909561393061879807658759777951217151,
-0.00004483792816850707441144628308616834334571,
2.756278118841287686812177139006466697793e-6,
-1.149538895431354410595719283847330295184e-7,
3.646845505811159232209022998582483154439e-9,
-9.363211462696596594423620656439325969027e-11,
2.023002068817516576755867691668769992126e-12,
-3.776263251960146743358438826986001780697e-14,
// [5.5714,5.7143]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003138654971795495052698409378254274459968,
-0.00003858789366374406237619784330039723804892,
2.41403168040734285699408868874243508988e-6,
-1.021116755213336563693987370898448995594e-7,
3.277123991962484999514198771218821246866e-9,
-8.495304405239755892257013467035732719376e-11,
1.850432933825870381558103418178716265777e-12,
-3.478104295670858859465481789401677365591e-14,
// [5.5714,5.7143]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002701152556462084220945411935528999364107,
-0.00003379644352567132954364249932764976094814,
2.144345185949402519865333636446633411408e-6,
-9.175947303377863296586756965245381034858e-8,
2.973356523224042894280068248538192860588e-9,
-7.770609839213279285378701160854678018264e-11,
1.704330656589435407611536219872918596729e-12,
-3.222634213463640387129279503221282455459e-14,
// [5.5714,5.7143]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000236575104679699293202877245886751247149,
-0.00003002083260326232485303530352082596526379,
1.926948933710657150531696948139467820542e-6,
-8.325398382270042641568770870591624135051e-8,
2.719713426313150800126191723508241833835e-9,
-7.157063220997018905042177138542782396005e-11,
1.579146481501848471555282030382806455869e-12,
-3.001443604432152574156334428266313777169e-14,
// [5.5714,5.7143]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002101458282228362611933698767875506596772,
-0.00002697728507192177711551278435873142391656,
1.748333660277935637286009205287529942954e-6,
-7.615197703369753639985599761460488711056e-8,
2.504972110993138469268033415592424441157e-9,
-6.63136216399876418687481575581139353303e-11,
1.470759704898043938956742096100442776026e-12,
-2.808167238447591708711227315247398583587e-14,
// [5.5714,5.7143]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001888409955034524277625234065225049539609,
-0.00002447667124386533861790576141095425667142,
1.599191517708804692755039956163237943109e-6,
-7.013922013822915497916607391050865202025e-8,
2.320976741980474578814465321229333921528e-9,
-6.176201550312122982396911448605676501585e-11,
1.37605128803919725063504695197874546251e-12,
-2.637905136158968858138118566017195527768e-14,
// [5.5714,5.7143]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001713366987070573589328456552776860674445,
-0.00002238868124789898073111586039999017073517,
1.47292362290390593977015379453806371632e-6,
-6.49873497468606306149655312702881993954e-8,
2.161670528026727856672668162708462924988e-9,
-5.778482853183581079597840746256622739194e-11,
1.292620180864097524381024862438614395426e-12,
-2.486828087281195440477507026924868267288e-14,
// [5.5714,5.7143]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001567207687352928543124276131440608475412,
-0.00002062093072063171579410872186969779738614,
1.364734344685110565172735474420937842598e-6,
-6.052677570345110698815503306250378371499e-8,
2.022468984783246322313964118367186539452e-9,
-5.428122799750414269147430733629781314841e-11,
1.218590022083397328651233106950720214508e-12,
-2.351903030871178899156274176844195934136e-14,
// [5.5714,5.7143]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001443465150444220002835776492113204269223,
-0.00001910628082556976417042863474611353542955,
1.2710622897734596695651163020585625193e-6,
-5.662913244528841179323166606327700045884e-8,
1.899842966760299152338532532993345931631e-9,
-5.117241584521542976808680612819961413057e-11,
1.152474572722440376791709683994973187326e-12,
-2.230698161491831111176218802377910381572e-14,
// [5.5714,5.7143]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001337439657789883393989241018232771360494,
-0.0000177948720568077205191196938109431354586,
1.189211781351996883970416667298763380169e-6,
-5.319560389789005234878288477903513711694e-8,
1.791034542046016003975127122388519222701e-9,
-4.839597743035159476459176417268641421721e-11,
1.093082216092976083581307709115223715881e-12,
-2.121242188179975894606521039790934285505e-14,
// [5.5714,5.7143]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001245641043976540342782356316037355466757,
-0.00001664896493890821143627929183369468893697,
1.117107681856589241916731314781203617524e-6,
-5.014896796709318380303147044020116085851e-8,
1.693859198087033023093469323427982385511e-9,
-4.59018709045961110821194074814940363171e-11,
1.039446993162685152801954110574227296759e-12,
-2.021921081687025873202022596943651407483e-14,
// [5.5714,5.7143]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001165427545723574710995763924589565224989,
-0.00001563950754597338841469609557135063764155,
1.053128327309816484803777988667557663044e-6,
-4.742805830088267881147346563232172404997e-8,
1.606565470199161301889545963395735656557e-9,
-4.364953098963322296796098471766615244876e-11,
9.907780075532493722846255486347433499559e-13,
-1.931401226497875880513383448486285140857e-14,
// [5.5714,5.7143]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001094765528218137103169861411984769811951,
-0.00001474379658231937868492840260752815461273,
9.959892243193605045625704588658443419043e-7,
-4.49838338876672064087250877424228790587e-8,
1.527733573647133862207781287900795068275e-9,
-4.160574420439197976571636472440580639383e-11,
9.464217691519221122298335453190742776188e-13,
-1.848571467308848821174528766040882441055e-14,
// [5.5714,5.7143]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001032065760762356425481941226602129837983,
-0.00001394384914045373784452442436463971841669,
9.44660511641802984086971098171108427657e-7,
-4.277654075449486804394240486398591660086e-8,
1.456201036598358551183030821494950964974e-9,
-3.974306746261603414073310187964977810457e-11,
9.058337962082907949243684389821462798688e-13,
-1.772498873211973555416289368513745476266e-14,
// [5.5714,5.7143]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00009760694398317615697929002086124055484409,
-0.00001322524716296861715748564317979910961816,
8.983073558451537415963328894518127403012e-7,
-4.077362968974494171273138846205788821974e-8,
1.391007351038026061772511793291207145793e-9,
-3.803863548896286788793896584972629380123e-11,
8.68556939256307839127934590148289515838e-13,
-1.702394594875411322799975732270110760568e-14,
// [5.5714,5.7143]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00009257673014078031246106899166476698789436,
-0.00001257630298181616063534818587472883910196,
8.562462234853773476583935680727562507186e-7,
-3.894820646874050629210975085577141373695e-8,
1.331352232332712799210335680031643638476e-9,
-3.647325052357180783321289554531038928261e-11,
8.34204650716816817006916931713303153251e-13,
-1.637587239740480320532347037941494845198e-14,
// [5.5714,5.7143]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00008803412087271311707668192014813085183645,
-0.00001198744712877987788019473025251127550047,
8.179123358442582284817768537319032457502e-7,
-3.727786312152113409403107124603525775645e-8,
1.276563758890364660418937642864894817477e-9,
-3.503067972397244421393830482566501233745e-11,
8.024479384137808394567179996036996660062e-13,
-1.577501911340065500676185413576249202502e-14,
// [5.5714,5.7143]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00008391212990145913804278334488777541111031,
-0.00001145077270180475569332503144990143575668,
7.82835125552627203340409981578738844685e-7,
-3.574378584331593362685023766373985655582e-8,
1.226073781227174338705247736216559858194e-9,
-3.36971072751958854057936445398963536282e-11,
7.730050945757865486108326575012226573516e-13,
-1.521643561337784179532211871959702767811e-14,
// [5.5714,5.7143]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00008015540891263328297027725570617780059161,
-0.00001095969175772242972938902878412563944054,
7.506195027102953775614037418094466032136e-7,
-3.433006644841090644247470929506030314079e-8,
1.17939874582154194460923492354768562796e-9,
-3.246070305783191215239417208674938313609e-11,
7.456335381217776224157069149786188961408e-13,
-1.469583657756933060874329129501576303541e-14,
// [5.5714,5.7143]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00007671784230405700144329119827192807532987,
-0.0000105086730379302591032778553550932225843,
7.209313954172686322489725452192211879644e-7,
-3.302316543805564206483499425924105119772e-8,
1.136124598496129620821021950306872732778e-9,
-3.131128006437353298208895920846254162937e-11,
7.20123281919744530030534529452997558626e-13,
-1.420949426613827736230425752994397601041e-14,
// [5.5714,5.7143]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00007356071126551180726744800012564507018597,
-0.00001009303953582832933165003014624839494173,
6.934864741997882144550554972513827608201e-7,
-3.18114892951574315322851909035864418863e-8,
1.095894793989965345326832810219764271446e-9,
-3.024002005687258273137199959585272745294e-11,
6.962916610450972766387516630684420620893e-13,
-1.375415107722290870926205339193722495275e-14,
// [5.5714,5.7143]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00007065127675079829906057765479136248280698,
-9.708810638784020665192529805106953486577e-6,
6.680412751989071188299338845415040721096e-7,
-3.068505475229737897170769607224800762972e-8,
1.058400693976425410401137693420244396444e-9,
-2.923925218062267309025492206815124109772e-11,
6.73979048007966274736674276943375656847e-13,
-1.332694799658527105814400939411817010271e-14,
// [5.5714,5.7143]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006796167447148813857856846450591373781699,
-9.352577852772077489265053802691567889788e-6,
6.443861497988284284089553705433991767834e-7,
-2.963521993623160956972599166303900722715e-8,
1.023373818542167951430493947404369690146e-9,
-2.830227302579975946419072457424812841504e-11,
6.530453466891597347986140001555183806001e-13,
-1.292536568027180732429772340478488118979e-14,
// [5.5714,5.7143]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006546804496940453651989658818734075894528,
-9.02140609717134514257308858204810387567e-6,
6.223396186614306801175769628206677363613e-7,
-2.865446740929950200956421046183445087824e-8,
9.905795483445779928166718945632593017495e-10,
-2.74231993910735898052238094914712158266e-11,
6.333671053098734397398056956545860700223e-13,
-1.254717565115008198341401962275715143894e-14,
// [5.5714,5.7143]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006314984268019941025635398009280281272882,
-8.712754661248125073937409615666204275673e-6,
6.017438155958407443624060085908171588187e-7,
-2.77362278298305617892124576551859927433e-8,
9.598119713381892639176450438195783989451e-10,
-2.659684704358007056308039278120054195902e-11,
6.14835124994448238715952645737943871825e-13,
-1.219039964662564996303904249631521351019e-14,
// [5.5714,5.7143]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006098928262873686993044878131535021300565,
-8.424413418330195189811832493649208610034e-6,
5.824607844269781633804652961224196950088e-7,
-2.687473566048290233130164063253739172295e-8,
9.308896393737347701354667230159975167835e-10,
-2.581863029132261164084639599837129149518e-11,
5.973524677516555690629130860725901077569e-13,
-1.185327557711371567070806633285475405986e-14,
// [5.5714,5.7143]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005897089392831136088804131108752799242263,
-8.154450981966430619149238116807101169398e-6,
5.643694488706632534201258537902793201813e-7,
-2.606491035301554191304051279163927597508e-8,
9.036520532322088855845353026406426701484e-10,
-2.508447832912915036669756323988483478781e-11,
5.80832788391747165406811829028576515515e-13,
-1.153422887785647553114455486334822102378e-14,
// [5.5714,5.7143]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005708115687376500903241620891415497405167,
-7.901172284178317167589820968374385799811e-6,
5.473631174138353393130658300805773234419e-7,
-2.530225792924119604656358094999061729092e-8,
8.779567347333759483884265853634802782283e-10,
-2.439076518825194068730088258741780078184e-11,
5.651989307246846587486021035141836374384e-13,
-1.123184828563645852747909001887270053966e-14,
// [5.5714,5.7143]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005530820598924821500360867269474650500008,
-7.66308364378300662155959436517647345056e-6,
5.313474165145613936455519080201498418259e-7,
-2.458278900006371449818164631555555795944e-8,
8.536767749717737628053575489356176668731e-10,
-2.373425078440826458345181928478499175995e-11,
5.503817405842760630145223540401742753218e-13,
-1.094486526519018988310961829244061702967e-14,
// [5.5714,5.7143]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005364158550648104130710127494401023878697,
-7.438863831193437713804831471212737822896e-6,
5.162385690018222134638692765415848327404e-7,
-2.390295011608549388283999045814204498318e-8,
8.306987709981478165513851693408919919641e-10,
-2.311203107137466107197725426139477814776e-11,
5.363190576926283613112214689560874717409e-13,
-1.06721364611563161829087680863545456075e-14,
// [5.5714,5.7143]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005207204681835405907257130785843411249259,
-7.227339966015342611943530042033812157179e-6,
5.019619524382680841571756692254478731892e-7,
-2.325956599468704872704256421307394875698e-8,
8.089210811952573061260476964792786753521e-10,
-2.252149570492937487745649447223576693815e-11,
5.229548557799727635336099872299811358302e-13,
-1.041262867018281666898539380232370345619e-14,
// [5.5714,5.7143]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0000505913797621073934737564360311609468144,
-7.027467334125826225441482024394395765055e-6,
4.88450885888889771260554664447666094791e-7,
-2.264979067054907552970764197681784571741e-8,
7.882523435158677517362411634833651663802e-10,
-2.19602919327296645060398910578404084219e-11,
5.102385061958014976274128241905034165119e-13,
-1.016540592182206557466517162588006409155e-14
}
,{
// [5.7143,5.8571]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.3681929697304552015140725706889920762844,
-0.004507677207314331258127398028234568159852,
0.00008076728872629128753000681607564803946339,
-1.532915744431305982891301773592775042905e-6,
2.851274188253548060344902968223960508049e-8,
-5.020219728282743645917521019177353030754e-10,
8.230307419871385258348481174041163964612e-12,
-1.247503002329111950267836648172785472456e-13,
// [5.7143,5.8571]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.03155374045120031876829417335544031523881,
-0.001130742042167982552902272297782135144469,
0.00003219123063306113102703535288130320201047,
-7.98356776529920089222641833615528029236e-7,
1.757076899958432797164927083401240894107e-8,
-3.456362497730428247716086436098212226275e-10,
6.112922808556256484503809985588439321725e-12,
-9.776528012007631399859597640806333552248e-14,
// [5.7143,5.8571]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007915194295175877837916386247832444192128,
-0.0004506772288627780213380791847027168583144,
0.00001676549230713143224111784963306884951551,
-4.919815351009050573905598532983423295286e-7,
1.209726870058482466351378982960657114696e-8,
-2.567128773957214933122327768637770617068e-10,
4.790631435472242513847808243980921350904e-12,
-7.968177437591824347354691921835153643126e-14,
// [5.7143,5.8571]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.003154740602039446121544373348652586873689,
-0.0002347168922997747337225363132559333690587,
0.00001033161223712167714658014773553516944611,
-3.387235262291006740068402553285711292909e-7,
8.984950673237604491410800874722443516949e-9,
-2.011814379996957229606079344521436817628e-10,
3.904520905097806774494543225256030691712e-12,
-6.688634425106224901548638379394650446506e-14,
// [5.7143,5.8571]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001643018246098423111733933876557555517229,
-0.0001446425713196473903392107446232862373297,
7.113194050813449251670894717985961080241e-6,
-2.515786210942587015111250900268376363281e-7,
7.041350298854630391187307891085192181333e-9,
-1.639683392882005372654802261885429481259e-10,
3.277530499589697671015324082289910248252e-12,
-5.743681282055229189482283252640124590673e-14,
// [5.7143,5.8571]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001012497999237531710800158065526584963823,
-0.00009958471671133925252561131177963632079985,
5.283151042981503875947275492184841085591e-6,
-1.971578103294250523434518765575604235394e-7,
5.73889184747168448129496982936972305682e-9,
-1.376374505277764829799344155501753811694e-10,
2.814492197443597362607266514537499855548e-12,
-5.021473547611455820098197459745160378239e-14,
// [5.7143,5.8571]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006970930169793747483164250289620819454731,
-0.0000739641146016975602803561021825493789865,
4.140314016919784942137396444431138500946e-6,
-1.606889734689805210795649102332885266822e-7,
4.817310743687533162279230655781859750188e-9,
-1.18191970378564787963863025430540467649e-10,
2.460601349341929573215416999894865171438e-12,
-4.45386086583700028098470926907259942873e-14,
// [5.7143,5.8571]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005177488022118829044131133235296593029411,
-0.0000579643962368379535301550019444648296176,
3.374468442850275691307348379442088441609e-6,
-1.348847023846901493034478263606836278782e-7,
4.136718940786386706064289963996576362765e-9,
-1.033302667737291534308593702854183865524e-10,
2.182463707219124906119357508346723645892e-12,
-3.997321942386679219291319649018289475408e-14,
// [5.7143,5.8571]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004057507736578656586734119726324021740269,
-0.00004724255819986847999527188100325864892931,
2.832578750080032759590549028432416341253e-6,
-1.158281317572179566514480267881303044019e-7,
3.616559316552136611317620933547382979578e-9,
-9.16498897159966252337650555309907478556e-11,
1.958753442727315059705205612058138358384e-12,
-3.622944315980164690564771999679978233456e-14,
// [5.7143,5.8571]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003306979073990793452072104487611957465551,
-0.00003965610250108812656118145615729024470456,
2.432390766902994027261182357410499406775e-6,
-1.012636621567536813428253117001289640027e-7,
3.20774612116732416718064651664522302431e-9,
-8.225522890355727220047936286944319842701e-11,
1.77530317113580228723418491805249084975e-12,
-3.310864258359618093742737012196861111685e-14,
// [5.7143,5.8571]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002775927175076168722621679333147779811016,
-0.00003405347073661216072549867589357404460842,
2.126536905293139260636603109639799084013e-6,
-8.981689258292149916781658049998384478052e-8,
2.878932994131752400313554120169108411128e-9,
-7.455130685291992980679453398667550916295e-11,
1.622379463516006679394782678395428077727e-12,
-3.047035014362463608837745737244423626108e-14,
// [5.7143,5.8571]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002383742951562851123581528056850827487077,
-0.00002977151667407639867862387875994924889167,
1.886154744242572641180931680756798653716e-6,
-8.061012493773739447845766716189090786426e-8,
2.609295723570032391625279019238389756503e-9,
-6.812935774282946636924504056129177023595e-11,
1.493099260072178800714663596305207002626e-12,
-2.821271558321566921125903568537059800967e-14,
// [5.7143,5.8571]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002084006167185347788555920733031140562922,
-0.00002640616641937037267374755356528884308043,
1.692812623693627188312491392728023755401e-6,
-7.306028128574195284420787090891882375044e-8,
2.384527505773594265010543093411703761361e-9,
-6.270032136781018427813619096466588345506e-11,
1.38247178507612047033573970419056210322e-12,
-2.626025617325004670833768664050731097715e-14,
// [5.7143,5.8571]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001848431649355925975480491276439099820681,
-0.0000236993767316868006741462140869593408467,
1.534265907001653160905532631818365731042e-6,
-6.676677112086754768236559281233909463271e-8,
2.19451123357796312107185932444746917379e-9,
-5.805460653299507162811167677355582693045e-11,
1.286798297842410562272949254265562598425e-12,
-2.455594317267739997773039672561983112352e-14,
// [5.7143,5.8571]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001658956371218075941950250097901729675935,
-0.00002147972269800062910379762584995062297019,
1.402102193539228810434796538892944126735e-6,
-6.144631544079688179704766005456689118382e-8,
2.031911215183997436843653023602808222222e-9,
-5.40368825648636508945667494858545896722e-11,
1.203284322207021439299176355326964174729e-12,
-2.305594907448594535901933525865898932045e-14,
// [5.7143,5.8571]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000150358058886004393777467108289785406345,
-0.00001962943070952798687937994414752946149396,
1.290372624257689637891140353449647761906e-6,
-5.689351487381814288033488942071994420946e-8,
1.891290877035252478073363683200927730847e-9,
-5.052979428873726418275643534539371844817e-11,
1.129782256655961856978486565177776584279e-12,
-2.172607588167729285838353891381024325631e-14,
// [5.7143,5.8571]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001374060149666958987226147247342128191432,
-0.00001806521673958759743055062396682293688701,
1.194763812351086577551798283616729906568e-6,
-5.295614535929424015877083847693368831388e-8,
1.768542788031405335555613322209967673295e-9,
-4.744315258479863345467736870761637381474e-11,
1.064616356359869949890223270934045270684e-12,
-2.05392734571234882684054685224717905079e-14,
// [5.7143,5.8571]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000126456517177113173046565406598306688435,
-0.00001672669337289619498955861927608704818694,
1.112079052546039907616186612448485925204e-6,
-4.951919882557003190904453405148882589817e-8,
1.660510328989725410143639868698794407413e-9,
-4.470658429281268468366969950374722463186e-11,
1.006461129802198494580206391924247135465e-12,
-1.9473881471397689862385734280182463968e-14,
// [5.7143,5.8571]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001170868536102733563820100821476209214351,
-0.00001556910673562648057789882837983836733385,
1.039903175337790984860070824115920894523e-6,
-4.649428993484398864395336834504376766589e-8,
1.564730439310878928216907721597469923009e-9,
-4.22644253458118261110171058340686902466e-11,
9.542551923807714953432481501910247682127e-13,
-1.851236192885514946496505757014882607085e-14,
// [5.7143,5.8571]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001089837471493853558851484332423891956265,
-0.00001455864445471184719814835344065746464311,
9.763800886325071394521690709227595771195e-7,
-4.381245298977445222731396158412093859927e-8,
1.479254876658341529748007068697730635515e-9,
-4.007210296952886831245602838723217375368e-11,
9.071391588166333587660342446788022311569e-13,
-1.764037071839556208208872671828953231873e-14,
// [5.7143,5.8571]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001019105111829829225787944536587645192115,
-0.00001366932124083864903525941491910316038721,
9.200615127860130921999977500917826520984e-7,
-4.141913720447623376766515256482126652636e-8,
1.402523593938872701723147977705658407838e-9,
-3.809352742233356498526278340829943555041e-11,
8.64410148110263123565392515089103501576e-13,
-1.684606757192351599052068998057470406565e-14,
// [5.7143,5.8571]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00009568524868587053576161876909448752798142,
-0.00001288086117898844180551373866422124928967,
8.698018812947194918933532334724652449384e-7,
-3.927066125995359713371327898951327197258e-8,
1.333273450200540363284367845351524836793e-9,
-3.6299181398280323313563559010265946984e-11,
8.254879707200311188325002157817307486488e-13,
-1.611959636500495588439020889066300329991e-14,
// [5.7143,5.8571]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00009016602825291908545097730772125488900964,
-0.00001217722633811098266624090327411660394547,
8.246838864597155549243640968892837755753e-7,
-3.733165720922947086198266067560498100906e-8,
1.27047133973958011956777097125259293152e-9,
-3.466470003717814651330453031533073993055e-11,
7.898896626883134295079793023757465075962e-13,
-1.545268891637541580134049314810319397058e-14,
// [5.7143,5.8571]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00008524058436677687175105498942650156340203,
-0.00001154557441042152746878910767938361080247,
7.839648013944825042833703270074697537828e-7,
-3.557319809232557559031418656706727866982e-8,
1.213264492452990781510295708816587636119e-9,
-3.316980147243197562680261997418410345079e-11,
7.572100713452962815103905642056904915719e-13,
-1.48383595349646745695580856076443060127e-14,
// [5.7143,5.8571]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00008081902087295068562375174605656737204259,
-0.00001097550721950881913682769296865039263541,
7.470371599394762369602343388380358748204e-7,
-3.397140634612580414639514078201416640555e-8,
1.160943043013097357706037097746653540207e-9,
-3.17974715198297161443164187400793968503e-11,
7.271068877418906614261557723622042543228e-13,
-1.427066707988209081392703918265045543179e-14,
// [5.7143,5.8571]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00007682855053656172753687210280448120119588,
-0.00001045852023913924519271069266279489113223,
7.133995332692582988982950345634835706583e-7,
-3.250640574125667546585547780814527487973e-8,
1.112911494975189055331685501555308363303e-9,
-3.053333510989954394102958496178276389395e-11,
6.99288987294391945572491032342743128931e-13,
-1.374452782681026098781635763567770585332e-14,
// [5.7143,5.8571]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00007320964167397471014872758003650227515667,
-9.987593465756671551840687068161972714694e-6,
6.826345205669854117495775329958830325772e-7,
-3.116152237709539824115717269878481782786e-8,
1.068666720910098652269407697260866758648e-9,
-2.936516665066946669809717255339178094551e-11,
6.735072600020429345014832588700995641463e-13,
-1.325556697726714989314289701027705951952e-14,
// [5.7143,5.8571]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006991315426029669486873007628663191713658,
-9.556883287925296013262694650514750824646e-6,
6.543919699195788050468102622546394724509e-7,
-2.992266868547745247268027087481959760339e-8,
1.02778082510084654243749003165228733358e-9,
-2.828250494135750050529043001925283243326e-11,
6.495473342108120549142540340850142053979e-13,
-1.279999985149415827007026324861875550236e-14,
// [5.7143,5.8571]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006689818301547706629158679335325613426212,
-9.161487578862019003620175789938647424504e-6,
6.283760423955834251584119109751589607012e-7,
-2.877786358619888080275183276367659315489e-8,
9.898876655218451545970667139509567775007e-10,
-2.727634760637309030605484859581244532045e-11,
6.272237549105477453921310712245733931836e-13,
-1.237453609419304351175903651015641182626e-14,
// [5.7143,5.8571]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006413041305203412740502307627416363658821,
-8.797264593526472578499006229187490134881e-6,
6.043351353107160503449732776045211932911e-7,
-2.771685510243097325570491571627091484222e-8,
9.546721590289881097024491092219434120576e-10,
-2.633890661297199459975650072987168563117e-11,
6.063752896898736974936944936823033240806e-13,
-1.197630187587730686259769159348539098756e-14,
// [5.7143,5.8571]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006158085215468530259921558284952768334589,
-8.460691894338694095338814199800344416908e-6,
5.82053957151573667866369248315179425054e-7,
-2.673082090604027880324739546360725588621e-8,
9.218617244776036405794432766452714472127e-10,
-2.546341114521646125824463093424818628705e-11,
5.868611164985503434727181086261447283462e-13,
-1.160277628046566075172162110514885972481e-14,
// [5.7143,5.8571]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005922484326037085337719263238577890210209,
-8.148755400111043544049162221193135501184e-6,
5.613472390273537148395652479280218146884e-7,
-2.581212872488925643691282350516398690898e-8,
8.912193833110876114208103667216576369825e-10,
-2.464394750962341347125397399152894873687e-11,
5.685577065535836862610372406221269256839e-13,
-1.125173896110240263207870564565578588904e-14,
// [5.7143,5.8571]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005704128780077729966916495055217559965061,
-7.858861346372286961332514228158803218458e-6,
5.420547032231677490827411588052077894425e-7,
-2.495414315931630715516305961098364015533e-8,
8.625381562586123905179256249180106717879e-10,
-2.387532823346965966565861149440545389736e-11,
5.513562594028566445096914412097710280653e-13,
-1.092122681031065369700321649764253501668e-14,
// [5.7143,5.8571]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005501202942460600373280106148249074675906,
-7.588765845113987857726164303797235349236e-6,
5.240370063461221194407087368718912533845e-7,
-2.415106878967021342228123384991703486767e-8,
8.35636481775827913985828042694092739013e-10,
-2.315298435099065786303479094184223494413e-11,
5.351605797027446872479895992101623999854e-13,
-1.06094978898781099546124835679380238977e-14,
// [5.7143,5.8571]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005312136091579791014253308744403507860556,
-7.33651808883563663163654870616435359722e-6,
5.071724445835411932409992063439269776894e-7,
-2.339782189264858980563046214075165870558e-8,
8.103544460618341805119753291562316511581e-10,
-2.247287623936023715860199211090154963288e-11,
5.19885309732142403741001064263263899155e-13,
-1.031500124444948735662253563018225567264e-14,
// [5.7143,5.8571]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005135562662184945168780732633336281376639,
-7.100414224159775778518406044128981732864e-6,
4.913542597460748186785344203854221808917e-7,
-2.268992488177212106420681975130662423969e-8,
7.865506623184848066442178296202352886538e-10,
-2.183141939380069718460113086114164295236e-11,
5.054544502159219632366107090615142921466e-13,
-1.003635151215106690633425421797975033743e-14,
// [5.7143,5.8571]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004970289956911842583734719570205534120991,
-6.878959636435504385213153174895724802777e-6,
4.764884225176573238282474217299641298893e-7,
-2.202341892664422357149943956742298214494e-8,
7.640996728792518012628081783198497519463e-10,
-2.122542231021410388064039685893395160946e-11,
4.918001162093973520267636605475669283994e-13,
-9.772307468409445138182964587486067625366e-15,
// [5.7143,5.8571]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004815271745504852619952938564893280338702,
-6.670837915237904133912385870369383782286e-6,
4.624917974599604058030669101851938510387e-7,
-2.139479121255854752304342943735151388862e-8,
7.428897751013305452684213653715000210144e-10,
-2.065203423916000598276023798277534992116e-11,
4.788614857146343714626708097555464749107e-13,
-9.521753811972658126271666516935496414024e-15,
// [5.7143,5.8571]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004669586540666532455013480184509988580781,
-6.474885164430379765533276511906979786264e-6,
4.492906154641506764944432436988914366393e-7,
-2.080091406547730981865255910011442853878e-8,
7.228211927548681098575168053702659620918e-10,
-2.010870103356335397189694200795240323386e-11,
4.665839071691139631688956448258528484305e-13,
-9.283685637125475831130675002446210091061e-15,
// [5.7143,5.8571]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004532419615101265407598188180209715939688,
-6.290068616489264733065716238918463320515e-6,
4.368191953754346525681216962877822131701e-7,
-2.023899375092916777918695176097756118001e-8,
7.038045306927475012458518987875074737335e-10,
-1.959312766822265560992159618281375703291e-11,
4.54918138561620705485928661723209511202e-13,
-9.057195142187197328919254492842320551095e-15,
// [5.7143,5.8571]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004403048031542484894836136132586245961535,
-6.115468735247451072666506353135484454905e-6,
4.250188687699141030582062646489124991409e-7,
-1.970652720476273186087613824963404169956e-8,
6.857594630333792118820322368566930686864e-10,
-1.910324628696023811247161621920923216943e-11,
4.438196961290908986408918797320682387191e-13,
-8.841460208283852097102468619274585745016e-15
}
,{
// [5.8571,6.]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.36376455491494918671129485168473225319,
-0.004350629787719654440480810635458371669342,
0.00007633471859544055918919177753024189418101,
-1.423724677384925039751385741654908543867e-6,
2.612182708271268380672970394413043514599e-8,
-4.551699076877819240376180834657135950524e-10,
7.404074997602442520485493606650982859087e-12,
-1.115653826916767898368462452400836317776e-13,
// [5.8571,6.]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.03045440851403758104912411878802862922388,
-0.001068686060336082832197435565681603570945,
0.00002989821822508671303779109105476182248617,
-7.314111617158361267472178639586972400025e-7,
1.593094672524288449843686718216880808089e-8,
-3.109385108951391138012181410592999888788e-10,
5.466844006470540039370455034634881937356e-12,
-8.703762396811999663993489450545898767727e-14,
// [5.8571,6.]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007480802422352579796719079794995366585857,
-0.0004185750551511449513296089927302607761065,
0.00001535963439603531031865942852192304237337,
-4.46066511068068391146838946920995095513e-7,
1.088284784464101398669691144608061774395e-8,
-2.295809399768223294407644411292870241465e-10,
4.264960978977442139382927498783300709453e-12,
-7.068902601031983520144879807450586554443e-14,
// [5.8571,6.]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002930025386058014634748879194610899024143,
-0.0002150348815444365597205700368926284514132,
9.36739673243179382948770583893954907945e-6,
-3.047197419613551628348430750310577426825e-7,
8.035332867753824966050749271110729635268e-9,
-1.791061715023221137697867318201300345236e-10,
3.463862866546966315251232070937333253857e-12,
-5.917251495212372260324017509018215038346e-14,
// [5.8571,6.]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001505244170811055896612891343565170769729,
-0.0001311435542539956037386380050864623122167,
6.399114581190515814496260710767848275579e-6,
-2.249893222775172502900847289821854467879e-7,
6.268715975149265370652863181713893573382e-9,
-1.454632283603447987516785660474511637135e-10,
2.899541015243220295850754212787015909304e-12,
-5.069894410495957967200047545777956466814e-14,
// [5.8571,6.]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0009180048797779692071907389455305332143348,
-0.00008958760413662401615287845950380034160969,
4.724775767829684318465060439355905751397e-6,
-1.755240490324030430287067374946188047135e-7,
5.09121296831783439763224061864053803546e-9,
-1.217641316064300614094491043083709293334e-10,
2.484326002836715128895406392088348002583e-12,
-4.424292267297621554991090597613921999179e-14,
// [5.8571,6.]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006271132289563680960576404086797171365881,
-0.00006614686074957731718867665212601562448679,
3.68600502968209711227529847266607728379e-6,
-1.425539646434424783607013992756827550761e-7,
4.261744584448873900857446703966746062605e-9,
-1.043269988263755001277893879397872461889e-10,
2.167972894879810381413112601728940002024e-12,
-3.918226425481845149018225809184754968042e-14,
// [5.8571,6.]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004630280252470412049182697109191688702898,
-0.00005160407041551506222625728545316862438747,
2.993633257513770689590116958342486171648e-6,
-1.193288497364735430865212247980237252532e-7,
3.651444939207831386264377799356817544717e-9,
-9.104169122424261764483897181395858123398e-11,
1.919994037604519578503455508344402289717e-12,
-3.512109903576699224252119230180335277603e-14,
// [5.8571,6.]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003612284929086054215211853826135644365062,
-0.0000419108656051617381635256104230972372166,
2.505905844467294422575480097849949716007e-6,
-1.022404595398892534387095085175452617396e-7,
3.186459174848201561164234040632341842874e-9,
-8.062782564142387898073017928981472849535e-11,
1.720991453794418918230793079864691790225e-12,
-3.179729402307848355954305794483713616858e-14,
// [5.8571,6.]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002933760592361321542130785765502894445358,
-0.00003508268182251377157420415326212513702566,
2.147049650338915762074489364279270541691e-6,
-8.922085802977288975340494758566618519508e-8,
2.821973880897254718194676355593908675111e-9,
-7.227075437494283151156974465060310591526e-11,
1.558120375495918040371843935563357649968e-12,
-2.903127514830503298608523162126603836212e-14,
// [5.8571,6.]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002455787727575963890543915980286415240148,
-0.00003005869510471875046085753623165256057498,
1.873638018626379334208594858804190349382e-6,
-7.901526970794036351562949583091384904219e-8,
2.529476387807627625243726604492878654942e-9,
-6.543104466842115448060571092578624272786e-11,
1.42258149155448395223517718203510699812e-12,
-2.669638163138167957994521238194390094531e-14,
// [5.8571,6.]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002104108657330312420959029713987590861756,
-0.00002623093226074518906880847826696001002597,
1.659320663867816128840508997697197402226e-6,
-7.082533982348629952597294325304333439092e-8,
2.290086549148096777039021806625154344538e-9,
-5.973915981386032823001358889958041108278e-11,
1.308168289290045275520994221005745113892e-12,
-2.470098525258240437120117012240952225065e-14,
// [5.8571,6.]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001836165258252163130796263601962568932024,
-0.00002323048929412698743386334516482303199841,
1.487332136294210890329806976882607486305e-6,
-6.412242427368931530745130391528985482338e-8,
2.090870580170400088912872273213552400357e-9,
-5.493445169118815666532502552510649078121e-11,
1.210390884540355787504856410536211627971e-12,
-2.297731918454964809986582733473618094404e-14,
// [5.8571,6.]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001626134250588889022751985528975284704587,
-0.00002082264990809798188252677634875088910749,
1.346570909748412763399441758942683378174e-6,
-5.854437708360184182867544711475852399076e-8,
1.922705796696320888043885103135599626986e-9,
-5.082836432948039967329752328300148060822e-11,
1.125928624972174512255566639510095924356e-12,
-2.147427957604433199401705446508156860215e-14,
// [5.8571,6.]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001457585493566858639829486619321310940737,
-0.00001885199273645809872935780700102911427403,
1.229431918756521377907938747211883573044e-6,
-5.383576309470226539292454706719270266425e-8,
1.77899273976245057612454913998831802237e-9,
-4.728144503365061049236596312997835698185e-11,
1.05227736126731970559661328232480246304e-12,
-2.015265855395373821178569056087672375838e-14,
// [5.8571,6.]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001319639491552066824164774840264990144962,
-0.00001721204686257276262247987610324270460123,
1.130551024989581724215110173345340118952e-6,
-4.981179745482193360688076878065861469326e-8,
1.65485056505572367083013008290176370366e-9,
-4.418853098721476207198307759654724848714e-11,
9.875158597707025338485121632941817984149e-13,
-1.898190968150790407476243335480825541977e-14,
// [5.8571,6.]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001204843280380093301219958779102255750795,
-0.00001582771434983662698910078069314884703502,
1.046047746552051204766401811213259662089e-6,
-4.63358165222525201012425404252205404025e-8,
1.546598574011163184619065864699877623742e-9,
-4.1468939424629321918343425628611999024e-11,
9.301473067958666340911815556824198813408e-13,
-1.793790496445892248098281454651780564855e-14,
// [5.8571,6.]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001107940004488563810975174506218693750186,
-0.00001464466845171211430663319480323323699289,
9.730521469680542401369798835498826553977e-7,
-4.330476073642093722696610858782622029597e-8,
1.451412869844421096227414468870388326436e-9,
-3.905981140692192211995427731655947201464e-11,
8.78989399662644921830539237739783138466e-13,
-1.700134871911377169016712161117983206996e-14,
// [5.8571,6.]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001025126791619847926911066572646345594774,
-0.00001362273005753698170217573456179094503004,
9.093999754655553968238282225022998431376e-7,
-4.063956098675587446303076453597457589855e-8,
1.367093389699369522242485703629816715545e-9,
-3.691149607338450535226282059206014246357e-11,
8.330966245740599341040971774387897416361e-13,
-1.615663593857995235921568201951066289724e-14,
// [5.8571,6.]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00009535911040275886479754643735343542141542,
-0.00001273159965650274555701021751226460073366,
8.534307807225566649913809451562721578578e-7,
-3.827861551278773472391482304329910378595e-8,
1.291902353457745060524617634014011392524e-9,
-3.498428662552266250179940434573358352904e-11,
7.917043153328119797599386822915962430079e-13,
-1.53910173051683582418373670558939862224e-14,
// [5.8571,6.]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00008912119759551921209004481006806865391922,
-0.00001194803093010144400020625214809168148686,
8.038509257691960992535626161226435640865e-7,
-3.61732664707944692085912328979495012353e-8,
1.224450023177664607046897836357307960387e-9,
-3.324607102553397659577202190872496354218e-11,
7.541877380242185631319602931037627992869e-13,
-1.469397949313904867230462952372687370298e-14,
// [5.8571,6.]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00008363621651071010147563891592246454455864,
-0.0000112539129607550183347435195861985292907,
7.596385958873103339988059016227927447743e-7,
-3.428460119806181321735537412555371668522e-8,
1.163612477540587445758547621396812416163e-9,
-3.167061372777988138394831674604791709312e-11,
7.200317251465888089562916870985522889728e-13,
-1.405677906500273132987680127119836157542e-14,
// [5.8571,6.]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00007877739072528512207818337500612127442369,
-0.00001063494034240918859864344665982212193529,
7.199766251598995228342836737109058917513e-7,
-3.258114989738436132855102851418489935599e-8,
1.10847147245299975466634241293058355455e-9,
-3.023628044534997332849605934633587836059e-11,
6.888078359877370592459498600208001266493e-13,
-1.347208756761973930482625735753076828383e-14,
// [5.8571,6.]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00007444458239686431416637501354580633581422,
-0.00001007967275222596298408042954509418477743,
6.842041478456499074076871325390825196302e-7,
-3.103720173390205782513432113867800574928e-8,
1.058269807876297385328173459747275047394e-9,
-2.892507898836313674092349846885107865269e-11,
6.601569659118806227778223908300352376558e-13,
-1.293371823126652096495997960584348391297e-14,
// [5.8571,6.]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00007055770926558173508761236627079895102554,
-9.578858069826954008565373255238806615758e-6,
6.51781236412500108595090032199288267101e-7,
-2.963155510632860732333219429503019959649e-8,
1.01237775716742884258828975848965967109e-9,
-2.772192893376667052174042497054954179175e-11,
6.337759542823774758837753888925585596683e-13,
-1.243641330388653569584513886879963863569e-14,
// [5.8571,6.]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006705200648878867246634461577178929645129,
-9.124937309763306754745207052984644787823e-6,
6.222626572334377435431152812811149755752e-7,
-2.834657766848295138754667003881270546431e-8,
9.702675055219468494674522205184397901061e-10,
-2.661409922074898395489054175011638752506e-11,
6.094071635772069547556104338666037988299e-13,
-1.197567696335708152652850551487024493686e-14,
// [5.8571,6.]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006387456116834314188271949136595487933707,
-8.711677201256851638345997927164628954324e-6,
5.952781310386604296579718864950230843486e-7,
-2.716749060568953820297958294349455614757e-8,
9.314934658135183528183034678414859283517e-10,
-2.559077052326178262525994096143144521626e-11,
5.868302918799548698232606854179610670254e-13,
-1.154764285842807178473446141743790670565e-14,
// [5.8571,6.]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006098174040879795624822638477605495965396,
-8.333893834530358567422384672740042369203e-6,
5.705173027199814437637101737457746406604e-7,
-2.608181747828046659048248225291440666101e-8,
8.95676961632253922082380276784393790576e-10,
-2.464269141434859251205112541667640368895e-11,
5.658558822165600401207306939066588841004e-13,
-1.114896822266357621558685580344009441032e-14,
// [5.8571,6.]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005833725684171250492047592103961861868292,
-7.987242238069216253877227330016744636033e-6,
5.477181670443747431949516886832606835152e-7,
-2.507895534666537698678143918316185629292e-8,
8.624941930362490142324438682061044182032e-10,
-2.37619057902772304333568166404447229693e-11,
5.463201340011002365397654311153136652844e-13,
-1.077674856986529433248473258503385974219e-14,
// [5.8571,6.]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005591069566648450888386490200625639061437,
-7.668054338611062576361246216824459593829e-6,
5.266580622804426737514471889254996357865e-7,
-2.414983781237190278802275926983669967758e-8,
8.316666963962555082550252799296302933446e-10,
-2.294153497722204746727066456290352772578e-11,
5.28080722999321647118365936764191794377e-13,
-1.042844846922937164354099557782699163793e-14,
// [5.8571,6.]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005367638037027743328989630793046809577317,
-7.373212871916332546870207598996075410595e-6,
5.071465940602654457498046891247529575725e-7,
-2.328666789369426396542420920669794003349e-8,
8.029537181295890613838569110293866816904e-10,
-2.217560219094202056764509693809901281841e-11,
5.110134092190459422824912396897838415831e-13,
-1.010184498555378899835151940993809529748e-14,
// [5.8571,6.]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005161249010341432322337823339481854971893,
-7.100052316834151020886446251424066928238e-6,
4.890200257680215981630799445715067698965e-7,
-2.2482704490240856334210814793577109336e-8,
7.761460707888861701795355597518862136972e-10,
-2.145889008561830244707918399272697169464e-11,
4.950092654039977396199209185323191694036e-13,
-9.794981171212114398213290157963273625201e-15,
// [5.8571,6.]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004970036621783905267343004920778207530931,
-6.846280360743019232903889428925468930766e-6,
4.721367942954873717800409781079388231823e-7,
-2.173209035341794642318375527285842173895e-8,
7.510611472714382616456992546665946091207e-10,
-2.078682436502231184117305312590447572745e-11,
4.799723980770136619142568548575599614112e-13,
-9.506127593129058878992013582887593633672e-15,
// [5.8571,6.]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004792396252520113028217053736941366818454,
-6.60991512012780605191386376153195126686e-6,
4.563738974221943002212614538286546955869e-7,
-2.102971248428977577311243330447900509865e-8,
7.275388472100914505643571403350960202612e-10,
-2.015537807833299721959636400338573790734e-11,
4.658180623087306113904502124210440558365e-13,
-9.23375032605210995467504895251433608497e-15,
// [5.8571,6.]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004626940584089463813312943960118722295893,
-6.389234563901954281772300128531275245264e-6,
4.416239621704913991589245693894023148682e-7,
-2.037108807252270845885278031377498990114e-8,
7.054382273268647218124295696377012201059e-10,
-1.95609924504603508431276941709349550428e-11,
4.524710933432479609895713587561559233931e-13,
-8.976484182844308484789194388485327116823e-15,
// [5.8571,6.]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004472464194731367585382709809054638403679,
-6.182735470378351332251988982758252612264e-6,
4.277928495233722633974676426871455442501e-7,
-1.975227070628566118792805523363225724751e-8,
6.84634730494284706918771941061979752391e-10,
-1.900051101876172110159355332541964506328e-11,
4.39864594844762963581355006910202762965e-13,
-8.733110211740590829297028451344159077282e-15,
// [5.8571,6.]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004327914829264845531314805413018596105325,
-5.989099893318908596732223614439273803417e-6,
4.147976848323840981938842683837781780363e-7,
-1.916977278596673461403870355839342589011e-8,
6.650178805204666145122910814636708921883e-10,
-1.847112454652786261301955331307461603163e-11,
4.279388362311358786567539656845085821856e-13,
-8.502536689981777264938476261323022961918e-15,
// [5.8571,6.]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004192369925323235626517520701329151433225,
-5.807167587645287906430802509786604789064e-6,
4.025652285056769762024479962207616239556e-7,
-1.862050097815484947416715564149915684998e-8,
6.464893541211345035921726305399628984311e-10,
-1.797032471703988894941423811307641372075e-11,
4.166403213346919984703406030525650298751e-13,
-8.283782997845955335421747393300499678909e-15,
// [5.8571,6.]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004065017311351701152881952530267400931533,
-5.635913199071591141048798741572913110493e-6,
3.910305205416181958072138868491673599767e-7,
-1.81017022308557774907154806379428699637e-8,
6.2896136021162191827186904597749526725e-10,
-1.74958650224757772933766007306553673216e-11,
4.059209982064334991734988551171086157479e-13,
-8.075965877748622445291295779755737867249e-15,
// [5.8571,6.]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003945139239350113426233832378762412735644,
-5.474427287574961256906254175786562368079e-6,
3.801357468483289295973498223872654379454e-7,
-1.761091839366769991406679657939281839903e-8,
6.123552710186058115689016720201364130829e-10,
-1.70457275800958983690582638215398019555e-11,
3.957375857919118059383889622013526545908e-13,
-7.878287677849074980777696478238490357687e-15,
// [5.8571,6.]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003832099101302472516028933953131812328625,
-5.321900455869095375457556211805319173519e-6,
3.698292862673709533561306770434274925146e-7,
-1.714594788890936295624578638499939398177e-8,
5.966004606466054902751820227351171176902e-10,
-1.661809485641622601209191202537388513938e-11,
3.860509978506693125379592183752212042255e-13,
-7.690026254494207264599362014687641642299e-15
}
,{
// [6.,6.1429]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.3594888617950981536388895568320806228048,
-0.004202129269450515024025431786128728691232,
0.00007221583257702762477617317653583063448065,
-1.323644820765836163161292444968884967433e-6,
2.395322889223877451601622281202675720324e-8,
-4.130082190197826011482012280604454025229e-10,
6.664983283090413836520522514522806976395e-12,
-9.982485890653911540514847865653982548155e-14,
// [6.,6.1429]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02941490488615360513779067341194912906239,
-0.001011021656078311045700959779076012783472,
0.00002779654123608547662396781738310903864396,
-6.706904120107525963722257234419126194581e-7,
1.445528762679632675360713910047082292305e-8,
-2.79900228317596894653291433614309843469e-10,
4.891542554032095460914137081421519527767e-12,
-7.751910656438054857383534718724513461396e-14,
// [6.,6.1429]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.007077151592548177294542339796801770985653,
-0.0003891515773051354116426711393983710359524,
0.00001408449865222823951467931418564966627856,
-4.047480560007582100558808608840211548204e-7,
9.796507958649262385717504995534830115357e-9,
-2.054212627321489869344966489037851761726e-10,
3.79854011504661243038930320302576778123e-12,
-6.273231065303056799930399640542002350158e-14,
// [6.,6.1429]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002724061041135947859816235151314698361052,
-0.0001971829811311442184464490832498130660256,
8.499709176018003936899419728950657446201e-6,
-2.743022248875848179685785200047571282954e-7,
7.189744167871463838361241077201726378038e-9,
-1.595190488427381947024464697163236112144e-10,
3.073972034159384267795898381535948911259e-12,
-5.236282210279082723151914864398373239241e-14,
// [6.,6.1429]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001380280867918009510239083647823552943721,
-0.0001189959284642083431197221678770549853462,
5.760346722641094247414467928851517347871e-6,
-2.013128384488942220807504464646874656025e-7,
5.583166685321502591970437147223708557552e-9,
-1.290900398141386006428098396517511423025e-10,
2.565855641048268828986246078826218031192e-12,
-4.476181767877258931092495810217244185757e-14,
// [6.,6.1429]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0008329714992494583851381329938066960165725,
-0.00008064485411693724503107028774740686754933,
4.227569607428381863749983920450954598314e-6,
-1.563286687119913414136676538792884720422e-7,
4.518151372118790100513216189822937642206e-9,
-1.077513161505048984917827922065620033009e-10,
2.193397469783232232110858672127048485617e-12,
-3.898887518695905282260800826079975784169e-14,
// [6.,6.1429]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005645139788185607002679208712783428391951,
-0.00005918597450396367892647606816198520131086,
3.282902042953253335380488171436903031349e-6,
-1.265082397660235795433869447428390471052e-7,
3.771296046132073957271518973912479949997e-9,
-9.210976536608975798456938832449184923841e-11,
1.910516118190103984521033965118998272165e-12,
-3.447577133339337571170088898584231184025e-14,
// [6.,6.1429]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004143018215277457389654814112638002817613,
-0.00004596062860131540824779408141094095899465,
2.656673035087793082419998027771287411485e-6,
-1.055962904972458377042672075141556318063e-7,
3.223841770507597625047516809472473115814e-9,
-8.023010364228197125845803548617973580502e-11,
1.689368173184402554061131112131664765613e-12,
-3.086231147376451996469098964061174725848e-14,
// [6.,6.1429]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003217244002092078454024546930402978759179,
-0.00003719342249120184703112352739871910680011,
2.217522100443346479592789419951932128289e-6,
-9.026757066446670846715459106414713164988e-8,
2.808053611694651636622986386395975657154e-9,
-7.094299677757075983131943010226348946989e-11,
1.512303775009522173266220730479781490521e-12,
-2.791076696854942166984336906706841237017e-14,
// [6.,6.1429]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002603539574384129178868721067184919700589,
-0.00003104530940618198909764301021819159154195,
1.895618983954888658519110588033750611685e-6,
-7.862550212192329324946026028545071157298e-8,
2.483004872711190443317641199713477934882e-9,
-6.350721155550477219812211342879863703278e-11,
1.36767399366711849010563682982154720239e-12,
-2.545875522534117715334500677766181276604e-14,
// [6.,6.1429]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002173171658432739132069226708979711449649,
-0.0000265386657753455978496773834762017834911,
1.651135544561394914828412079879146079033e-6,
-6.952413734965590745187698105041808695611e-8,
2.222752391032538657031561299998789041787e-9,
-5.74335357554331334904876323044631745887e-11,
1.247521918538915826827390035715915278111e-12,
-2.33920378777716484018983172980925913222e-14,
// [6.,6.1429]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001857706604274191752083569601994352029025,
-0.00002311589762383840794284119042426188483213,
1.460006884343709042172244104804803265679e-6,
-6.223706779375295036876363571274548126943e-8,
2.010173738973645906341593824260739989196e-9,
-5.238781005012431489485829008387223145924e-11,
1.146249748935917297138993264897515795181e-12,
-2.162816474666645657022035494225483674126e-14,
// [6.,6.1429]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001618112833668688465026636984892539208186,
-0.00002044009638079229191323495235231057566288,
1.306978423669685295780359745926956953339e-6,
-5.62848654766559988530137582725674939418e-8,
1.833573340109808098744501833545545993028e-9,
-4.81349496300498261863443932993658150835e-11,
1.059817335200321664006534218101023325427e-12,
-2.01062839079567615874145479927305173468e-14,
// [6.,6.1429]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001430806746655460348595398046969621825265,
-0.00001829769793135725406302833136129414985638,
1.181982175010595158203809653490354929058e-6,
-5.134005425668416961481801901461555166973e-8,
1.684723226129279585642272552343401801087e-9,
-4.450528538566921275497528225796934438696e-11,
9.852428634478270483603092223715263161621e-13,
-1.878059233353493263093947459938719736819e-14,
// [6.,6.1429]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001280838855195007704073917069958597133777,
-0.00001654775045013112940646294042280048683937,
1.078141139391138811347790179509531078539e-6,
-4.717225101973823507612577134522002372534e-8,
1.557684978215064610201859293872362878288e-9,
-4.137359428924958542790743522767652796254e-11,
9.202819311565859855331026853088966402708e-13,
-1.761600883427830749912302432210579048718e-14,
// [6.,6.1429]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001158342531509178982563372099254264234874,
-0.00001509397595145974713937486141577740361296,
9.906172714152314751241749058692757041872e-7,
-4.361518003787634085031042360447666845428e-8,
1.448075790409858068830103428543590899611e-9,
-3.864562166825866246827648550773047066733e-11,
8.632155173522682698309196780340405989732e-13,
-1.65852443014754689540647563639126502956e-14,
// [6.,6.1429]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001056578316602182227855564263315701259152,
-0.00001386864179979794136044317264174781726562,
9.159187807960934079669393760716617090096e-7,
-4.054612274345314214147935630443601254818e-8,
1.352596749185689416727812033937749291442e-9,
-3.624917671350721081587889979887745586822e-11,
8.127064215981511379038460429823219743e-13,
-1.56667740491630698582521789930671490709e-14,
// [6.,6.1429]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00009708049259858558269252965173252278649869,
-0.00001282286293113081247611345974553551756576,
8.514685776131717233920703921033505826511e-7,
-3.787270955701392450635737296580847832858e-8,
1.268721176229545787167000801300949996956e-9,
-3.412810345345719533035426489404900388912e-11,
7.676999067290584085041146100618742029023e-13,
-1.484340654561373945148439488234445234662e-14,
// [6.,6.1429]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00008976004051791568082786574759127358090931,
-0.00001192056008657063363677342385294678349606,
7.953269006979168910715503361980320000334e-7,
-3.552419348525187750323719600684049158313e-8,
1.194483612544622890145982605938170197989e-9,
-3.223810813471603490008053142082258094019e-11,
7.273535652041055064188304391537920054954e-13,
-1.410125496054069503015327790539992287289e-14,
// [6.,6.1429]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00008344392060599442924880101628507738836272,
-0.00001113457660975772248522506711935410981106,
7.460080631908854575779033120160162506028e-7,
-3.344554167581378223610288236901200488508e-8,
1.128333776767969089076778328894972237382e-9,
-3.054381389049406526532352183099489375739e-11,
6.909869238153627683115162698879492938334e-13,
-1.342898611733114645346379429119408494149e-14,
// [6.,6.1429]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00007794203626830405145870702288874448848766,
-0.00001044411288465987979065963840026927749563,
7.023563751926594653685183897968134936104e-7,
-3.159334625017230332667797973995535193936e-8,
1.069033478566755830369275911090153666907e-9,
-2.901664434709406833470539225757042777278e-11,
6.580446415177526616552912683873222067786e-13,
-1.281726388975635534373987500879586130431e-14,
// [6.,6.1429]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00007310879019261915284506923036609210332449,
-9.832989252685261722725489132602287995625e-6,
6.634602712541645694131218814109284152823e-7,
-2.993293787870522935479210070295698154302e-8,
1.015582544865608310437714972341072855144e-9,
-2.763327808955080662604661861713448837859e-11,
6.280692352369908345670954576398184826965e-13,
-1.22583311105463241399908945915419737413e-14,
// [6.,6.1429]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006883092476879682659808045642997364701533,
-9.288443797546833794808765134528983675864e-6,
6.285916954533340750687288141719851073956e-7,
-2.843631171504831161769864178230967356751e-8,
9.671647261441408180730813923593258451157e-10,
-2.637450326479622071605035350973771439168e-11,
6.006805929045069755336503016510477211782e-13,
-1.174569161936281666254412791568527231493e-14,
// [6.,6.1429]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006501910658282783131369779912921977787653,
-8.800283736335667633110217768197612298648e-6,
5.971625460165185431853163361697511681333e-7,
-2.708061277241670055906128759211000122386e-8,
9.231076075478565120861974370426625128924e-10,
-2.522435722083678682360401517063856576788e-11,
5.755603934309481841008711470569148560796e-13,
-1.127386570027724136488039511802010705464e-14,
// [6.,6.1429]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006160198615434966837725258970575985495499,
-8.36027564422067563376445552717955044588e-6,
5.686928682212359481761419133866495968837e-7,
-2.584701343470271390421120066046811988721e-8,
8.82852496259447448905373299834089589259e-10,
-2.416947221692708344770061941952753349714e-11,
5.524401228465558332389220639545105205035e-13,
-1.083819998407889381923747199076064553253e-14,
// [6.,6.1429]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0000585219295095447245633423951408896725006,
-7.961700155087113321530646244306973878502e-6,
5.427872821292248033629461591631041342691e-7,
-2.471987030286640914640986772269498804527e-8,
8.459315213549426843762688264021196322631e-10,
-2.31985721574174037834345314535736033547e-11,
5.310917592798067892053666155028926936996e-13,
-1.043471824307124773553973083898476956867e-14,
// [6.,6.1429]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005573190108560978854671165077541714197507,
-7.599021949799323220024422416825312827176e-6,
5.191172763606461787986081312757959403288e-7,
-2.368608299090311440517698413077650831547e-8,
8.119500194884332266476107476185752197673e-10,
-2.230208140517039922658157981193529256787e-11,
5.113204617151727452558532034552252968553e-13,
-1.006000322081368036718919192829449338834e-14,
// [6.,6.1429]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005319315364859525799391179752773600898525,
-7.26764186903956319340113183952224363044e-6,
4.974077428094018457592229126049304485087e-7,
-2.273460092501204044016013587400289501241e-8,
7.805728433617016630465308654810958501812e-10,
-2.147181774477912669921074001225597969583e-11,
4.929587794997310470041679707092260516775e-13,
-9.711102252749640109967260521090914808366e-15,
// [6.,6.1429]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005087349308327693795511387584575310974164,
-6.963708399322460543433828289050560575733e-6,
4.774266194256751261707598695915541462185e-7,
-2.185603998074294262797266661454733489662e-8,
7.515136154368920112982950155491824222886e-10,
-2.070074921033355449495070583139400068915e-11,
4.758620276317212799918421317015532175136e-13,
-9.385451295699245899209065429413261700352e-15,
// [6.,6.1429]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004874595879525721954367882243355471058043,
-6.683972671950583961643141492470714383685e-6,
4.589768395960107917839027544368835005974e-7,
-2.104238158694847051415769528988769669346e-8,
7.245262169083686075857252241193681635418e-10,
-1.998279987064035045805726040960606627248e-11,
4.599045641058953626599150038184744232745e-13,
-9.08081332656672633195221307588069262672e-15,
// [6.,6.1429]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004678780870365408360108657816781913581306,
-6.425675754335562166858852792797305221445e-6,
4.418900133263144028435453021134579849791e-7,
-2.028673441699425211304380435897453292182e-8,
6.993979901854343161947875233898047496379e-10,
-1.931269349658750237465069006821750471666e-11,
4.449767713681871223075939739482424735617e-13,
-8.795228049068401397919328773085054238122e-15,
// [6.,6.1429]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004497973028034893115988366631104163816264,
-6.186460186560074686949521997744238089614e-6,
4.260214227572640767970930604140326987417e-7,
-1.958314405827338949927523757261754763644e-8,
6.759442672501134254940829178275897294882e-10,
-1.86858267977375846499070065251250235776e-11,
4.309825918775999748886816331568763856943e-13,
-8.526970567879873480007148822330465168806e-15,
// [6.,6.1429]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004330522130592051891580433885631328246582,
-5.964299918593616654100703216230155225721e-6,
4.112460252241148943957854236767113171279e-7,
-1.892643980622304540642888504047131454026e-8,
6.540039329379335925078192111272761394724e-10,
-1.809816592873819230473780423160306715023e-11,
4.178375030830644041669752706731242832654e-13,
-8.274517225453351907397578464364733108894e-15,
// [6.,6.1429]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004175009943015531279472958763436967587047,
-5.757444353129760517970482744884560131513e-6,
3.974552359310472171616236347501654431661e-7,
-1.831211043618522818615170087620093642525e-8,
6.334358026623057532183807826704260841461e-10,
-1.754616144900109806318291235966847944411e-11,
4.054668433803814472660184247281330948289e-13,
-8.036517198875528370701482920226333676307e-15,
// [6.,6.1429]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004030211047190831994478557111968645690281,
-5.564373303027032513411052285317167614354e-6,
3.845543191602431706017390530327839569288e-7,
-1.773620277968850190686035798313067872114e-8,
6.141156460033069875559007303771981446589e-10,
-1.702667802179686231791379314459395505543e-11,
3.938044203186843456592482100291957032903e-13,
-7.811768758498510544578622436724241282947e-15,
// [6.,6.1429]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003895061312118922401040278290229446121789,
-5.383760468235983444947714285582453081215e-6,
3.724602583738025554981113372396120095715e-7,
-1.719523838493312745393270813992520512297e-8,
5.959337261760023078780109087094586146351e-10,
-1.653693596640580524219375254246318296643e-11,
3.827913472399472379055968654004713749156e-13,
-7.599199322318317270051286649597292915016e-15,
// [6.,6.1429]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003768632327765188062367600538772982558774,
-5.214443617226011461277387122019605462046e-6,
3.611000060839308103476205725136797292997e-7,
-1.668614462190341490440354943685278822015e-8,
5.787927543557375358246745330811327050264e-10,
-1.607446240329134832114975458374165147618e-11,
3.723750659152019798125896215875868847343e-13,
-7.397848618646935787674820064729964194274e-15,
// [6.,6.1429]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003650110532058207682585973650456560067661,
-5.055400085167993543372661023798555578746e-6,
3.504090370602984106651742530397422238361e-7,
-1.62061974034753663418243162338065861431e-8,
5.626061797592137957204028070221648716566e-10,
-1.563705021017542251560488548456337269114e-11,
3.625085214913415231324773734277977207847e-13,
-7.206854407925283916217397616583832389706e-15,
// [6.,6.1429]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003538780059617595148410161849672210317739,
-4.905726518837317106600731401757806081545e-6,
3.403301454733013676121759302218486141734e-7,
-1.575297330768628666184261758534653585259e-8,
5.472967531071332481228631159742945233756e-10,
-1.522272337436432526136913142583450283179e-11,
3.531494628393220206093438674427996003081e-13,
-7.025440322374117604053088012577997805808e-15,
// [6.,6.1429]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003434008563186121650627929580756454881679,
-4.764622036619526991677231903815461262599e-6,
3.308124394617230544985338316923831508427e-7,
-1.532430935468845789770286062880999670856e-8,
5.327953139556096203015770988288889486257e-10,
-1.48297076112728810206964834827546818199e-11,
3.44259846679343654120143649810637597662e-13,
-6.852905466831983481606973363268642599249e-15,
// [6.,6.1429]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003335235425633668577768097295995273623331,
-4.631374152457591044412742157548037440221e-6,
3.218104964487613672604291476686016596534e-7,
-1.491826905202828533558876276036978933105e-8,
5.190397623445185954820998796114781633556e-10,
-1.445640534102993794926400217691609887321e-11,
3.358053280066464650066845874121474284234e-13,
-6.688615490985431699455487555163209676629e-15
}
,{
// [6.1429,6.2857]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.3553576482601905168195809423477237820682,
-0.004061574751585917941646817208658541168403,
0.00006838458531288545901036920787230479129881,
-1.23183211866046740833617911667609049767e-6,
2.198475330389504983365608793237675237678e-8,
-3.750432836213797298685899924960432448891e-10,
6.003501895641215721182514267696218796846e-12,
-8.936614075591092177554701061579398432768e-14,
// [6.1429,6.2857]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0284310232611014255645514987613570365724,
-0.0009573841943803289753767872868755537660235,
0.00002586847449187240530300011635680148367513,
-6.15573095207110091849918796775268508107e-7,
1.312651489221849941559427372567020422558e-8,
-2.521211782344920468330291005571222760564e-10,
4.37905139255313578444988969977897507401e-12,
-6.907057212859578734355171624760652735946e-14,
// [6.1429,6.2857]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006701689360662302805185773907456626140652,
-0.0003621586428861592900049868599327493987153,
0.00001292703499935146730487502597378114241523,
-3.675424191575027823584804076891457801387e-7,
8.824241209468800495007685584424838554748e-9,
-1.838992746951274150311935244851284110139e-10,
3.384549997406980360312476301194218482855e-12,
-5.569029901499606588965436187656820554272e-14,
// [6.1429,6.2857]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002535110500203115010887081875316210736041,
-0.0001809784899908752794149250554382524913512,
7.7183908023093966290176081604163641318e-6,
-2.470787556756538639904767744720182751703e-7,
6.436474589820067561282504501620307020998e-9,
-1.421337187423419055501337552859429342416e-10,
2.728903081928695943770790736453424907039e-12,
-4.634989234680965139356178135931030862106e-14,
// [6.1429,6.2857]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001266849429936126939257932877121413381328,
-0.0001080574712322929506551375798340978753942,
5.188653869190329219193727866413157541291e-6,
-1.802212900590596931093134077134249418079e-7,
4.97468013467423627629112464230222357088e-9,
-1.145991060268664029740650303824434226566e-10,
2.271212909855814613457533046982903749455e-12,
-3.952924769535043244551706000281922371611e-14,
// [6.1429,6.2857]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0007564022986260506398895369549671424740039,
-0.00007264115416863104951956916568197433814111,
3.784647091241664418807445811056543270588e-6,
-1.392910451132710922998975298525473246717e-7,
4.010968692128757765026654346588241592539e-9,
-9.537805517889942588073862019456719153997e-11,
1.936993334197198051828465437073488163851e-12,
-3.436555491944435431361949993100973866409e-14,
// [6.1429,6.2857]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005084880791804173334983112589537118184532,
-0.00005298505927735367376038188793412372407132,
2.925111947379954256916086857195860989489e-6,
-1.123071245647388292817670379557543143685e-7,
3.338231914443850170858352030374557660715e-9,
-8.134234269284031343377508236077834392572e-11,
1.683966007570663682210991785174089777803e-12,
-3.033969826124974855323075752539298016896e-14,
// [6.1429,6.2857]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003708954149414757044538549573554711637472,
-0.0000409515672632928719339337665697185923956,
2.358449615860654826996375001007672728247e-6,
-9.347049466394297293359769247243387856772e-8,
2.846981979057205701088526969947212474206e-9,
-7.071640091697736849259127338370323100211e-11,
1.486693829952418501153207264980753571095e-12,
-2.712384145366378848518165867178768022441e-14,
// [6.1429,6.2857]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002866609708430500927218205713711532881634,
-0.00003301829462202523995007428284381695515285,
1.962880387943840745427885227561936327993e-6,
-7.971549637071481810888479257289872732138e-8,
2.475074018249982750732645089980286496218e-9,
-6.243195252180161295853135902191194366884e-11,
1.329112532837138752734587122077351701176e-12,
-2.450232670228920429854757752144447894108e-14,
// [6.1429,6.2857]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000231128062354176669721076208419812993194,
-0.00002748032543119196586954905804716275827598,
1.674025423785964411632375013547289354578e-6,
-6.930207338318951384840792106186536358348e-8,
2.185118325553267226135584072218447760593e-9,
-5.581435330821928193725579560576257091159e-11,
1.200654679818659384272219890997949245399e-12,
-2.232827864367723629462060168644886582465e-14,
// [6.1429,6.2857]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001923622780183437519127874290202634664426,
-0.00002343635593298348392672845463405161039108,
1.455343541047860504618843229279904297943e-6,
-6.118331391621174572029182470000760142359e-8,
1.953502354044788742853725018873501035857e-9,
-5.041980959421716227088879543644081441939e-11,
1.094123230851364295534341634450798291619e-12,
-2.04986382310573160438346452054316733319e-14,
// [6.1429,6.2857]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001640544915308843789640209640441650942756,
-0.0000203748095746515520954816176338302670859,
1.284849592241264879786851092663521091935e-6,
-5.469806665305921732236838167058527545403e-8,
1.76469332488797198885388319005172485201e-9,
-4.594607352574733293946291561973765833566e-11,
1.004468184204476002887056695765525426714e-12,
-1.893919762258118323882985145290536408249e-14,
// [6.1429,6.2857]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001426236670225608567117501003750050201481,
-0.00001798789429136052572351513350167815358114,
1.148659399715007403314213156960537978924e-6,
-4.941141378417292324274924772411568594607e-8,
1.608112563216598136426371450469430574017e-9,
-4.218106552520308057828881224567299990351e-11,
9.28053274160411992189160819030745995092e-13,
-1.759530310743195360218756967416684424625e-14,
// [6.1429,6.2857]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000125915260039523672605128274671806361015,
-0.00001608123159599406303419748145961615999037,
1.037639689468347501642275010335248955122e-6,
-4.502715241169485169373697886785634853069e-8,
1.476337283833897712552115167589797926562e-9,
-3.897207782979868575298421641183100790538e-11,
8.622004065994733811698535474139821454475e-13,
-1.642589496310027724463129456874048133274e-14,
// [6.1429,6.2857]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001125686211719584342195459323728179785501,
-0.00001452695565254182665577231994784375305417,
9.455702006462657933605353922132326113315e-7,
-4.133744454888912739571044344326522375087e-8,
1.364022715057488821467591705100419581947e-9,
-3.620664225932276303970177772061869551199e-11,
8.048976067401213306747612477040007841547e-13,
-1.539957892294690273730147136291048715121e-14,
// [6.1429,6.2857]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000101688689567792779962011597181569851259,
-0.00001323798280903356905971931170833032060859,
8.680863355273080045138146553041727059703e-7,
-3.819263658769660226375055308855109430724e-8,
1.267232470591880775140666744890928295156e-9,
-3.380026501655670783385803122671699575607e-11,
7.54606517412009883272974741776490879536e-13,
-1.449197196520030459515562080341460336468e-14,
// [6.1429,6.2857]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00009266587966323497714030336825687640061568,
-0.00001215320869736894916539624542610055969802,
8.020453683422313129519388896186557282366e-7,
-3.548250971109334303134708722489262405181e-8,
1.183009267543920659116405523889503742354e-9,
-3.168834230223905807338917440958071741731e-11,
7.101323401538984721170909356636286058342e-13,
-1.368386898329251171199244167059179291681e-14,
// [6.1429,6.2857]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00008507246088158263819591263646890536033777,
-0.00001122863515677858242239234415191214173968,
7.451327039335325453364307934721790598722e-7,
-3.312425999747267652302859224146337525375e-8,
1.109091972947120542608708372428476028059e-9,
-2.982069832558177740661115382904626012698e-11,
6.705340002215459249934795027750522801315e-13,
-1.295995103877985670135176863367894903454e-14,
// [6.1429,6.2857]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00007860044609745007128084452706151624852679,
-0.00001043185785505743647353550972633130510009,
6.956094599474710964464008886842323722837e-7,
-3.105457572329015480027880075447989716205e-8,
1.043724434130146314673301194661051287591e-9,
-2.815781258209171526633551065626854204395e-11,
6.350608496402645029473896501048153238018e-13,
-1.230785871921585488834889028889922991982e-14,
// [6.1429,6.2857]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00007302300498540204989891081236352861192354,
-9.738532439253152684980397916048063967869e-6,
6.521460901896131739913482157285773631237e-7,
-2.922428461335485020510752432161158002967e-8,
9.855234334408788066710346137744389566444e-10,
-2.666816163514555174169909008252134660004e-11,
6.031072607486260784614678030847447398214e-13,
-1.171751648924533391454814977778907897512e-14,
// [6.1429,6.2857]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006816972707477206361652812378827172475163,
-9.130045262643666061867596017321623369709e-6,
6.137099768809489770842297185454968658162e-7,
-2.75946565730835354799110616782036143804e-8,
9.333856506017694221788295340910968311339e-10,
-2.532631223232673108099270126859135788056e-11,
5.741795186578834088968409276961709463884e-13,
-1.118063268586124659799675384370138270049e-14,
// [6.1429,6.2857]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006391031683850565747247019235950385739106,
-8.591939676322846113226625477467010831284e-6,
5.794877880352304655107420267544980407652e-7,
-2.613479863443599011891273039386428482836e-8,
8.864209217818094471124348148739184136984e-10,
-2.411153092937404423763941659184463667414e-11,
5.478713204543131086561089628871047951039e-13,
-1.069032444299987894824916228264099157709e-14,
// [6.1429,6.2857]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006014357773425991803229488849896955188657,
-8.112829032483225899949195638676157292216e-6,
5.48830771323612782931478728795047325359e-7,
-2.481978620991901285339239189949429603928e-8,
8.43903576434866147041626018787292437384e-10,
-2.300675516351115797491460668249134902545e-11,
5.238453960701139697249586117235500838275e-13,
-1.024083280773996529233684954247469510709e-14,
// [6.1429,6.2857]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005678980322738257672423931876728542283902,
-7.683630798520982173359209638699732335162e-6,
5.212155104087385111689506947669866864222e-7,
-2.36293005240512827327121106550279999607e-8,
8.052364248663216567455146468387051525181e-10,
-2.199782141210125528498833044044956436303e-11,
5.018195486399762693670170743643338603759e-13,
-9.827303867910656130810719474499500246486e-15,
// [6.1429,6.2857]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005378541558964687080927441366541916003071,
-7.29701714571311510118971003807317906624e-6,
4.962153110054997467126385529364438090659e-7,
-2.254662026522260964456333678723975835131e-8,
7.699237437860680148787042240005802504785e-10,
-2.107287895135172589337268405248268997613e-11,
4.815559294894878090047159131170010602634e-13,
-9.445618808181229455247693942731006248041e-15,
// [6.1429,6.2857]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005107912001999180146299461885914490387245,
-6.947014354068117468779721329042603998832e-6,
4.734790255700823567412692307524487074496e-7,
-2.155786518117259020870358354759648161794e-8,
7.375507578632367039634805424326047000263e-10,
-2.022193945579616693745570189107699900457e-11,
4.628527106743379805289865857778649064759e-13,
-9.092260659138197031819804983743963195274e-15,
// [6.1429,6.2857]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004862910047847681818403654575880355046853,
-6.628706357972594366347236561323368378772e-6,
4.527151688050177489789817423619011673534e-7,
-2.065142156251891642591484598078742119164e-8,
7.077678757081208383159293808827046332918e-10,
-1.943652728447844146561377800617039957181e-11,
4.45537555518170354161440895244412950527e-13,
-8.764208863665086508539574820214719583052e-15,
// [6.1429,6.2857]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004640094450580815660502265419773995749983,
-6.338012363261988049133862614893860965026e-6,
4.336798528132773501769173024358175307666e-7,
-1.981750085024791058572058911329342178155e-8,
6.802784498886591654337121389537072249885e-10,
-1.870940527509162354959758928567309819138e-11,
4.294624522309392556265558790744358525799e-13,
-8.458855145653000959559596047376925430239e-15,
// [6.1429,6.2857]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004436608654283391251360032504404034194909,
-6.071517939377900702224295766339465239842e-6,
4.161675178555738366259187677856490198982e-7,
-1.904779691617343419935302452071292870179e-8,
6.548291797253331364395859735541430936413e-10,
-1.803435778132290199631147323849834301721e-11,
4.144995913669541828031964848370690140733e-13,
-8.1739358459827495700830837653737259856e-15,
// [6.1429,6.2857]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004250062557564530120619985373989503517479,
-5.826345249970311721260571185040839327352e-6,
4.000037352399982196690360702162424575876e-7,
-1.833521734119186449964801969321497515577e-8,
6.312025175982662668241183046174041493468e-10,
-1.740601754664967315380883656382268981713e-11,
4.005380501992654122930133651200778877257e-13,
-7.907477100411877938041011405718761904192e-15,
// [6.1429,6.2857]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004078441674979217845305198579983538268655,
-5.600052293352496953293055699664985379338e-6,
3.850395641653743504915661250342658269821e-7,
-1.767367079187912949341624336182845706507e-8,
6.092106095301101547524029251324846322375e-10,
-1.681972646483547650842195058368582770668e-11,
3.874811063632640582115455800962514866586e-13,
-7.657750114791025084993330004915074502381e-15,
// [6.1429,6.2857]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003920036605346747518415956203737379001833,
-5.390553898307991799703791754954168476405e-6,
3.711470866297966548100385303760397159856e-7,
-1.705789735681008195285358350994931000138e-8,
5.886904218034210377870534178801225661277e-10,
-1.627142276673266297557625384656345304069e-11,
3.752440462820888337907940953345720351288e-13,
-7.423234441035986855564723476334950859002e-15,
// [6.1429,6.2857]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003773387728815593920977313651489226189941,
-5.196059212810119531479246112803323475016e-6,
3.582158444933369855196453516314453228126e-7,
-1.648333209184385791766861862371604284595e-8,
5.694997924987688122914216027269410840214e-10,
-1.575754898552635782777960346488841161517e-11,
3.637523656392435575024479905949640745605e-13,
-7.202587638684399952521033033642543906893e-15,
// [6.1429,6.2857]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0000363724144896708334273104174371133920764,
-5.015021822899887250890273460333968801361e-6,
3.461499739290371502597153468804546786887e-7,
-1.594599446318994459365633283522377268562e-8,
5.515142102782887919770054856041368497312e-10,
-1.527497638603702907814465399298192914823e-11,
3.529402827530951852309557837581256068182e-13,
-6.994620067790549728958303879274350956683e-15,
// [6.1429,6.2857]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003510515276029920755312457667001493985601,
-4.846099634999881297971992886045719172222e-6,
3.348658837272963364615893861934794834195e-7,
-1.544239815334680705675227196804215779031e-8,
5.346241694112825636553174177020893598525e-10,
-1.482094253434578045017622882493827979029e-11,
3.427495033936398824454745537106330164079e-13,
-6.798273831853253624514763982462028430561e-15,
// [6.1429,6.2857]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003392269744499916596787061071261394596593,
-4.688122372175691218459652416489960604001e-6,
3.242903612205822714400935095884486355418e-7,
-1.496947700181802158483888699832205580652e-8,
5.187329847111124900599078881915730894595e-10,
-1.439299942672859852597458216649726631217e-11,
3.331281889563728487636019548877341121408e-13,
-6.612605097563823858945123597446986186224e-15,
// [6.1429,6.2857]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003281685660522983549206251663569744911992,
-4.540065057081866019636186886393352170821e-6,
3.143590170384700217735481861524595867748e-7,
-1.452452382334473905518933332854873885921e-8,
5.037549760479081990393682935454927639071e-10,
-1.398897015855265516888457785628797398268e-11,
3.240300901048536287651472239989023980469e-13,
-6.436769178035783827430019719565710029955e-15,
// [6.1429,6.2857]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0000317804553995730591770114103623463789788,
-4.401026238532457373998739017482896984832e-6,
3.050150002905237241496197632571667746969e-7,
-1.410513957426097278264870858063791353808e-8,
4.896139517599438412782186202528313253264e-10,
-1.360691254200512193173920545661774950957e-11,
3.154138158276088089075274838158201332772e-13,
-6.270007889904467025197538896585953262305e-15,
// [6.1429,6.2857]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003080718366972719873049563024966956228147,
-4.270210004061363861610404822151452437159e-6,
2.962079310597576295393457511007173433522e-7,
-1.370919088801174133975531374884021454667e-8,
4.762419352741522983428835518135250863981e-10,
-1.324508841051890511555981974722178755004e-11,
3.072422139177603096973208682981107166186e-13,
-6.11163879109829063429586778134605445869e-15,
// [6.1429,6.2857]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002989147002842954421322677870585150798418,
-4.146911034830785597519973924145427586554e-6,
2.878930086485171020525957490419615336417e-7,
-1.333477442052710704393970572208687447472e-8,
4.635780907610308042229820600570743655475e-10,
-1.290193760236669655100645187958182841489e-11,
2.994818436081554066996362389416690958085e-13,
-5.961045981706608538449839674737027434592e-15,
// [6.1429,6.2857]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002902837724381549643079119795529624303005,
-4.030502121073558223447171848417589216751e-6,
2.800302628313334268530722226005036147825e-7,
-1.298018676855922500683046550146905960286e-8,
4.515678125604369384174289764672035214938e-10,
-1.257605581429664602007218611494350583852e-11,
2.921025248004011517210511537266898383192e-13,
-5.817672210054509227186833186853053907204e-15
}
,{
// [6.2857,6.4286]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.3513632478774236791741262953650557164369,
-0.00392841497811743147065970305276882344466,
0.00006481733524518389544882822512433897683267,
-1.14752653998192917813624283431813094049e-6,
2.019653700969342236713854279593769011354e-8,
-3.408353740447583832773570312663983977006e-10,
5.41117182560385408663937997437948037063e-12,
-8.004533200809541530651861851366048781675e-14,
// [6.2857,6.4286]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02749890484682202027066154473934491755339,
-0.0009074426934325144110151937011579873919275,
0.00002409805733962281256243125324550985216785,
-5.655030386764424656299869667801813106828e-7,
1.192923806090218050277308047838204196624e-8,
-2.272461283178425226578631357855540505015e-10,
3.922319394518655185757721346028853993663e-12,
-6.156914725521463818125263705052010792724e-14,
// [6.2857,6.4286]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006352098854027600857227078736765300542135,
-0.0003373728027546710796649938623914338761449,
0.00001187556381220720019700622763838260790929,
-3.340186676371229465336496100150397648826e-7,
7.953614465678839261452472702870567991549e-9,
-1.647188686090189474919351704103731328302e-10,
3.016969641721821230228529568020996157648e-12,
-4.945606031936263350262669876343677286798e-14,
// [6.2857,6.4286]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002361609619282697540741428334825499461344,
-0.0001662578933708607260018974203184321713473,
7.014392020381205583161609803142592137622e-6,
-2.227012066420894872388043783033830536828e-7,
5.765160379666192832216056432967660587593e-9,
-1.266973352900939736484494689840878494663e-10,
2.423416234076699299506344845689559592795e-12,
-4.103924238147432911320264970721757797136e-14,
// [6.2857,6.4286]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001163805253596025067337745067662393121312,
-0.000098201488285302780372805531302440913492,
4.676725339485288089982643219684520704918e-6,
-1.614244919945754058867340976227452446285e-7,
4.434406716368464751618615836172876783984e-9,
-1.017703881137286408997012872118145614615e-10,
2.010982988239787799779369079504568364423e-12,
-3.491670613249373309636044573601973317892e-14,
// [6.2857,6.4286]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006874104179971194496741430818693517902231,
-0.00006547415475276444727242992349192988531,
3.389914331887325337002510469094275939987e-6,
-1.241633892417657451024932898945058204854e-7,
3.561963567422944283862860442435176161092e-9,
-8.444992433911357047718045942134952216304e-11,
1.710971584777251731597220953260457242045e-12,
-3.029655784855879963747822094082152671734e-14,
// [6.2857,6.4286]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004583190832693511193581873131280249179624,
-0.0000474588006463964766640978347204935917634,
2.607431174078189339359020376129161628336e-6,
-9.973498093097293394227705377893019287289e-8,
2.95574733708636996981246951319987649761e-9,
-7.185079245454831117337071136100116831494e-11,
1.484578639005486570539222812598866850559e-12,
-2.670438455022612620937843048858370181836e-14,
// [6.2857,6.4286]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003322116045247753262279616897850340250011,
-0.00003650403643707136823932645948454228722044,
2.094434599551432007053900509734155495984e-6,
-8.276092636972641654053535534315139634875e-8,
2.514777722570561929377929847020915404056e-9,
-6.2343362232307134871851161091891852843e-11,
1.308557526656418731974239611791298830385e-12,
-2.384170831464651088171155139198508685081e-14,
// [6.2857,6.4286]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002555282550594995681884555495018677852971,
-0.00002932208439369903984176874044797701562795,
1.737979453765165487470875371240406880958e-6,
-7.041377707231297506686063781686770910307e-8,
2.182017665987512352236013173958994317341e-9,
-5.495134883750099656268041766125922428597e-11,
1.168282565853573928669180583886330321908e-12,
-2.151283704904669684706163752555115692847e-14,
// [6.2857,6.4286]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002052545907558932701902867951128856957719,
-0.00002433171235269319130294699166881252711706,
1.47868931851940788161694749454340256311e-6,
-6.109649541267763893756117529756829484408e-8,
1.923297198173765494784175794805127198097e-9,
-4.906052346270184728752224944478134679192e-11,
1.054164659536175081660731167443188110644e-12,
-1.958488674751720907406976265471822583435e-14,
// [6.2857,6.4286]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001703219864688523310864715716573109593741,
-0.00002070165045925416685254069202534794030859,
1.283026403667001700669788433117685347025e-6,
-5.385232225061100504817805497754460269606e-8,
1.717118310910760586913828658215550735643e-9,
-4.426817890484473810258736406598267834682e-11,
9.596923588674695329862046629888715529346e-13,
-1.796486121095339766622583634717107129573e-14,
// [6.2857,6.4286]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000144911553214779160508548696021872296833,
-0.00001796236965132182688485177162650753349636,
1.130898767263547196311570266943717048716e-6,
-4.807931335338384498829155424467422949878e-8,
1.549386252121666132076099974780316731352e-9,
-4.030085936433509655387304917466881975215e-11,
8.803087526777073942590139376087072209409e-13,
-1.65859561346706602800408707201631474813e-14,
// [6.2857,6.4286]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001257365875592527812344646015532898840931,
-0.00001583258274167462286900329202206259913761,
1.009665580421728859957036053155810469882e-6,
-4.338281566092704305934433766172682719893e-8,
1.410530068843498360615509047436078752138e-9,
-3.696719298331755162322177439639635126547e-11,
8.127403569931909378214737908505523996823e-13,
-1.539907465492677191235887015157673873411e-14,
// [6.2857,6.4286]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001108280791917223535616921754316937842927,
-0.00001413531812589017363606953943498450651477,
9.110391288800939552320533938530203469566e-7,
-3.949484248883899455170786056963765376812e-8,
1.293851746068742049572201132694124312972e-9,
-3.412970724031876922775698532632638062535e-11,
7.545813697376317190588632779221918421351e-13,
-1.436740221019744991293191582262102433955e-14,
// [6.2857,6.4286]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00009894722688123120931830641695006580545027,
-0.00001275454780430816831758409373147732589146,
8.293916922662077699511518576451613463275e-7,
-3.62278494158116357239070008442848874884e-8,
1.194539745559340979104552145157419565674e-9,
-3.168736898549712550918352164355014435725e-11,
7.040278341627632812210668010578926553818e-13,
-1.346283908621313612872671200487772038675e-14,
// [6.2857,6.4286]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00008928183463015717243334402116787098198333,
-0.00001161148369171454222188188342488671665259,
7.607848377326001685833322260480134337394e-7,
-3.344711337032822986935302624354353662092e-8,
1.109057907081464471522702442049031037802e-9,
-2.956442020661977808395301895532785571216e-11,
6.597028302656231325206541995585499420393e-13,
-1.266359536688356650160946755000264411316e-14,
// [6.2857,6.4286]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00008128038584200179007176258703260497870564,
-0.00001065098772824473018767553294668433857496,
7.0238938077741904543027786895658078239e-7,
-3.105362186517206788497523758372612070964e-8,
1.034754700215427862101379642231967747843e-9,
-2.770303669031081344362610457365750520763e-11,
6.205386250703272767142275842466881454175e-13,
-1.195253300420801127806877868187537378508e-14,
// [6.2857,6.4286]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00007455691409771310610987954373566753231154,
-9.833451330872816067236356776906695927929e-6,
6.521260591691129977359340748891659656299e-7,
-2.897313204805869377700878363829866906544e-8,
9.696062774998954148638463777114068065141e-10,
-2.605837877114498385693799799127118083424e-11,
5.856954323968482482813840446843423252557e-13,
-1.13159997908524901265962656261219572848e-14,
// [6.2857,6.4286]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006883415931610970751793358176503874076922,
-9.129764828357090965393293582713445460996e-6,
6.084357730097080330386360311081004781131e-7,
-2.714897618964096753592199736841450122357e-8,
9.120432506505380404284827004137419655741e-10,
-2.459517955507766240302172829184675296395e-11,
5.545042763111330269764794330565814119891e-13,
-1.074299431462079387010624292534832092399e-14,
// [6.2857,6.4286]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006390835379849963203329479648476108845094,
-8.518100822125927736120013324512170937613e-6,
5.701284999829138686544715383697377163355e-7,
-2.553721141760773481309409109380165147431e-8,
8.608312783803602575401655498588978139863e-10,
-2.328534541231629107331592850102856656056e-11,
5.264260730028687486462555899095681377553e-13,
-1.022455804620952392270157048682370335175e-14,
// [6.2857,6.4286]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005962670575488148963673745296932152288148,
-7.981798999751269613994012342918476111e-6,
5.362814397701959792380110076891174820282e-7,
-2.410327617563543406408250108062473649641e-8,
8.149870836504095370809374851582761567428e-10,
-2.210623758464585803639643101553695239072e-11,
5.010218424177653157469140210313321545431e-13,
-9.753326113155676200694027649533635141539e-15,
// [6.2857,6.4286]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005587259299825888297277731250279860898174,
-7.507940156773639208686120476544702516495e-6,
5.061687996887593349194352507465968470969e-7,
-2.281963870639481253351889914406751461013e-8,
7.737183099263262931527801678328127547901e-10,
-2.1039421200183468211387583667529984539e-11,
4.779306956744685061058582817230625731716e-13,
-9.323190769042029227879934547249849836477e-15,
// [6.2857,6.4286]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0000525555810974154703111930782769846561617,
-7.086363195633911088181314325653971145349e-6,
4.792124128346894278773157706124425948912e-7,
-2.166411302672435390557813366043508785471e-8,
7.363797366948753205497487459673441111819e-10,
-2.006974084062122922235189983310844209954e-11,
4.568533446669277536909136702458656476117e-13,
-8.929046106018323870158419270302183140275e-15,
// [6.2857,6.4286]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004960454236943737362964568973780026487515,
-6.708973779677286342108246596021124749247e-6,
4.549463735615942459326930784106808088143e-7,
-2.06186329620855060828666513713250564438e-8,
7.024409243175410136237864715995728424644e-10,
-1.918462801798970854326494719004205308471e-11,
4.375395926764600437526585506056744487669e-13,
-8.566592150211422764894728620219042586541e-15,
// [6.2857,6.4286]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004696281645774100055704267888389698693439,
-6.369249229854280360437382503910367174367e-6,
4.329912922041640503712277386301565076294e-7,
-1.966834620245741203660918528781766648211e-8,
6.714619757173221599314513120923659640669e-10,
-1.837357583734351976021605796622477720516e-11,
4.197787348107079474128023132964614009466e-13,
-8.232182917540432628611033915594899060078e-15,
// [6.2857,6.4286]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004458474460897995882447069699145375232426,
-6.06187809085055983921564211026339446746e-6,
4.130352702519607194222031301764977947488e-7,
-1.88009356295625150185841632164491115112e-8,
6.430751495727857917767363567688372144454e-10,
-1.762773585982824370078358994649675043277e-11,
4.033921125519167668687948766405366468406e-13,
-7.922707398205915110354899574978448031119e-15,
// [6.2857,6.4286]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004243314663595391530537434239358126941167,
-5.782493783519993681145634529668236245553e-6,
3.948196482211554542213376221130471885822e-7,
-1.800610448629639145314444072683081930677e-8,
6.169707505254559302631545679589388351091e-10,
-1.693960542898774932323831074272646464012e-11,
3.882272818480911660544656448868810183331e-13,
-7.635495477256388157034181054876698733229e-15,
// [6.2857,6.4286]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004047745648463995231963999524183559708869,
-5.527475075088980952309426471683882060957e-6,
3.781281942125552667404605638918876521886e-7,
-1.727518130253170597584429606323061796432e-8,
5.928861856006071866270097244171769242547e-10,
-1.630278275871892359817518700754280908147e-11,
3.741534031011688788314335032251250799716e-13,
-7.368242931287441386437751164634422398347e-15,
// [6.2857,6.4286]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003869232552562286333068583264514584497319,
-5.293794718968821772041903504245673545643e-6,
3.627788073534860333296768124325464337148e-7,
-1.660081347489807793946113844267861253181e-8,
5.705973922857105615297835757717122055123e-10,
-1.571177333537664647661992200823419871513e-11,
3.610575659084662192149856616591544868501e-13,
-7.118951157109234737965361770174379781409e-15,
// [6.2857,6.4286]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003705656303278174917459462499687971483992,
-5.078903302942080110170082058169234118417e-6,
3.486170829731696894941369522997466606064e-7,
-1.59767272529766578211899084878401736297e-8,
5.499120626041322314753665019993685245041e-10,
-1.516183557521192399083416623111932713034e-11,
3.488418356884652840768485180792570629489e-13,
-6.885878378467862337890877059562358001817e-15,
// [6.2857,6.4286]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003555232312059455764080472940580883307593,
-4.880639161617864552731257346345955271976e-6,
3.355112723128103329274113270828777631733e-7,
-1.539753801336213949729088282958722434888e-8,
5.306642411254884434981964408609333584038e-10,
-1.464885679757096992441643196022007415612e-11,
3.37420862745404967740456195882041445723e-13,
-6.66749986986698427528377857535162864807e-15,
// [6.2857,6.4286]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003416447413132504883215017671236986091967,
-4.69715781237303377631208758768248634967e-6,
3.233482982808964800261370387949294553714e-7,
-1.485859900395142241177657031475415100397e-8,
5.127099840276300797490390000758219270452e-10,
-1.41692528178469566886073410837196999664e-11,
3.267199331830812098662062777452790369349e-13,
-6.462475319320103648865142505715300113231e-15,
// [6.2857,6.4286]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003288010468661123348524509815093747121386,
-4.52687617592642816556572334223741668292e-6,
3.120305790832629703920774517993909183338e-7,
-1.435587979767812689472197892803233888825e-8,
4.959238448499678097370283682767593980208e-10,
-1.371988609596357693496688835684656471324e-11,
3.166733696353194702871907646917575172609e-13,
-6.269621884392462575217090452212590875709e-15,
// [6.2857,6.4286]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003168813323148499429311267805917191555509,
-4.368428107159736498088292492423000943667e-6,
3.014734757515157876480873495314640981711e-7,
-1.388586789360482451289111037648492272058e-8,
4.801960096904083274745252103224223727053e-10,
-1.329799857544448301596790485725246593472e-11,
3.072232109749693289084579709490185905964e-13,
-6.087891819872288191135997455030609360924e-15,
// [6.2857,6.4286]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003057899675011815269932814080871587416837,
-4.220628660515443454081955230933683742294e-6,
2.916032257659688960724283230583151643228e-7,
-1.34454885024365231229597879199727507442e-8,
4.654299465727944505020045863961799608927e-10,
-1.290115623819512655679905011381128340928e-11,
2.983181160386981951347513175216144317402e-13,
-5.916353800198023273962888958376886036355e-15,
// [6.2857,6.4286]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002954440062360810146566653757161963986813,
-4.082445160717945344540966367117833588897e-6,
2.823552585514274261018678985790903611723e-7,
-1.303203872880837741399940533364646138685e-8,
4.515404648642776993251313840689466467769e-10,
-1.252720306682153206344227957746270739879e-11,
2.899124483997034077312643694232553886071e-13,
-5.754177246237939059067593051522527373655e-15,
// [6.2857,6.4286]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000028577116125025614769410605573030919064,
-3.952973619714514721201236523515772965813e-6,
2.73672813305229595203217695174036766875e-7,
-1.264313323497160488108734329617843342308e-8,
4.384521039564823153226303540088469379198e-10,
-1.217422261004023476672171423934940709316e-11,
2.819655083577000249271519631336682226504e-13,
-5.600619109166070580688869086707761275157e-15,
// [6.2857,6.4286]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002767081533800160047300231054704676047882,
-3.831419386267887279674888085392496276805e-6,
2.655057979346509428761985988454661439227e-7,
-1.227665912386564100270408240705199436717e-8,
4.260977880548589913506532041235674915521e-10,
-1.184050573045609861301417184090022231421e-11,
2.744408853299422234975977454407761368962e-13,
-5.455012674874955429186733371148050436844e-15,
// [6.2857,6.4286]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002681993570387520844598868042331644560859,
-3.717081171079921158792383827718787625701e-6,
2.578098416014195890003426877925061500813e-7,
-1.193073827321967161430294776940252892769e-8,
4.144176973509135472632500361488476649707e-10,
-1.152452340855723252158538424781741374331e-11,
2.673059092513007642884270617322236723684e-13,
-5.316758038562670206407874372585753118669e-15,
// [6.2857,6.4286]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002601956819755944566042008689087045341615,
-3.609337782414810565406332718106135946438e-6,
2.505455037378484133553636724859903459857e-7,
-1.160369572837471793340056072887033163633e-8,
4.033583161620333268529517232515709507e-10,
-1.122490370456999787530993781305804143033e-11,
2.605311838151783664299577179408408605806e-13,
-5.185313966693938130159229197172593883209e-15,
// [6.2857,6.4286]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002526536447690367156447979702393312777244,
-3.507637052324936294493147664604922427784e-6,
2.436776102960988408685986627507356682081e-7,
-1.129403305019850301303754068265483666228e-8,
3.928716265964161861386031156467763138697e-10,
-1.094041215719501152118832304719199232547e-11,
2.54090187697680398809572720677666069677e-13,
-5.0601909168228438546922568902779186545e-15
}
,{
// [6.4286,6.5714]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.3474985225690448351727050368751981668931,
-0.003802143773363376513413848330278449278866,
0.00006149260600287143925773987133031324733814,
-1.070043273844242265441430510808609105821e-6,
1.857079335618973486734948477645461133304e-8,
-3.099926079507808168559656598189077625866e-10,
4.880482396359488358980982181139417197382e-12,
-7.173504312511053595405794313723600689591e-14,
// [6.4286,6.5714]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0266150064135436355726147834877848711809,
-0.0008608964840401465309195402077748021569279,
0.00002247090875073113066945040593625030902014,
-5.19982216118074214266336752527367607981e-7,
1.084974125103599297999852800502341546665e-8,
-2.049596708452277706210668751330764857806e-10,
3.515104285546317264140197048821574069726e-12,
-5.490631002940606052397653173034972818499e-14,
// [6.4286,6.5714]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.006026275388281025698830062402057351551895,
-0.0003145927225101929244123310196414687622622,
0.00001091962653848124875182450034731881035679,
-3.037927567452181571921746321927682530376e-7,
7.173588457046217933068329733243772910937e-9,
-1.476179042970384010737069654133288676691e-10,
2.690481309176743004326312816847003407625e-12,
-4.39353348206434781030681042625206174093e-14,
// [6.4286,6.5714]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002202149057571350455942734285499652953365,
-0.0001528747715387019872622934931696573796977,
6.37964789165101589132748551023939299028e-6,
-2.008604782171149844574384854841235756276e-7,
5.166626631268422775416073351513187622575e-9,
-1.129865846386974179883534120969633027273e-10,
2.152892615667739531676916384217941427187e-12,
-3.634771746995103323828880335694647847047e-14,
// [6.4286,6.5714]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001070123400770913897895503509649058091552,
-0.00008931507048308409611203342642416963466891,
4.218070042560656972147526048392336028345e-6,
-1.446655468805796097137951309597395363481e-7,
3.954530445790383871677798605179081632547e-9,
-9.040992118752985355289213638945208827862e-11,
1.781091161006467935422438841582158856432e-12,
-3.08498990961327442554296014891507718135e-14,
// [6.4286,6.5714]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006252054933815886613966987885077166392455,
-0.00005905298059582310936706766001896955196938,
3.037976484493265012461491199633544713593e-6,
-1.107268535256685905289878536435192391467e-7,
3.164347226987391173331519879957603187327e-9,
-7.479581074700481943981883912946662495417e-11,
1.511691699489268111254006365893632186088e-12,
-2.67148063927740916299246408072723964125e-14,
// [6.4286,6.5714]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004133708641707617554166144376272650602699,
-0.0000425316707828827528203196957879482921842,
2.325263924040015080222167358004547860324e-6,
-8.860172327394843881782657668560052395805e-8,
2.617853363149410112594730319012115644861e-9,
-6.348223563743370077564594790465864116278e-11,
1.309067099752399483046242947190824410977e-12,
-2.350873221380161812439283433406207058054e-14,
// [6.4286,6.5714]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002977216954801792605917570117552575697532,
-0.00003255369493653974288334351604617493071264,
1.860636188753795665581909722779925301757e-6,
-7.329989498691973148166875775581394317509e-8,
2.221878235597473465310426592755395894757e-9,
-5.497295827900757307398796102379673703241e-11,
1.151965359208127900009786512613690619645e-12,
-2.095984274263457447321302898335224988166e-14,
// [6.4286,6.5714]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002278758645557781918612569843901087377758,
-0.00002604890664253469187939820630348015568087,
1.539297794726113291203667188274420740706e-6,
-6.221259133463290867283926343669094571464e-8,
1.924053529112832544966082706648756259947e-9,
-4.837546117263445836515572889150488838372e-11,
1.027066382239974230671759414459122774151e-12,
-1.889051200180353855773573142888194354932e-14,
// [6.4286,6.5714]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001823423464977428355285866528835484974526,
-0.00002155016912614880856326608215241037898363,
1.306464418028023297039285365184969741344e-6,
-5.387349948626548381749738583698607217546e-8,
1.693141131279311510791659958741677335807e-9,
-4.313034539879559357763869960333199257031e-11,
9.256663294128916674143075995905812326092e-13,
-1.718048271151590825189265853929822884285e-14,
// [6.4286,6.5714]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001508511838830416529063602118580759957634,
-0.00001829050185237694966240398868104221277945,
1.131343489212250667782602896165575067536e-6,
-4.740795229088579103888106747780075032575e-8,
1.509562079951050466578608916685388386331e-9,
-3.887208117725483876386192185653523867181e-11,
8.418724746670946539311139530506180293984e-13,
-1.574581856783899267128589208446072057152e-14,
// [6.4286,6.5714]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001280335129666386411081443857960515317361,
-0.00001583880884895732370484976623838920119376,
9.955669981092283686091834430964011733262e-7,
-4.226773880606004748549123996988003505246e-8,
1.360522832847136734193282496591742140526e-9,
-3.535319657075835401321469796071361868602e-11,
7.715718515826614888663149466035189769847e-13,
-1.452636672906811129818374414814053796807e-14,
// [6.4286,6.5714]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001108716619427012598461962683355310011952,
-0.00001393793797351603528269075297845969627656,
8.876225149278454243800210881237804936643e-7,
-3.809463984619950324362583344206392501111e-8,
1.237361872184156059110423757618181948606e-9,
-3.240096353240310508655696588178056527783e-11,
7.118169054115177054490532303377451314646e-13,
-1.347801311989609403987641710312243904286e-14,
// [6.4286,6.5714]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00009756556581461224127731241393197265341217,
-0.00001242671520897756298436439801189561672311,
7.99987436770736917125260315179918641964e-7,
-3.464613291207893629814132433572082094624e-8,
1.134033716336099370778186565360630517002e-9,
-2.989159714319799538716606778421576804665e-11,
6.604459965527325709092078078071058502609e-13,
-1.256774317356950358813025600569960011137e-14,
// [6.4286,6.5714]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00008698700646284293552998897883526144414477,
-0.0000111998241147788225236978329879879496213,
7.275687911541722788584087984733013136652e-7,
-3.175294451718747054188293327722426737808e-8,
1.046205893150353894268062584597734358332e-9,
-2.773431797311433368394265921695630562748e-11,
6.158413725933926044308662544186085604148e-13,
-1.177040159009851224990566573376336943946e-14,
// [6.4286,6.5714]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00007839876880345175260842512212878423663911,
-0.0000101859630761476049679029006682113320108,
6.668118348614224000397720081322974089821e-7,
-2.929376544049118265195163190517243561484e-8,
9.707011225853993168710613994257146201918e-10,
-2.586118772418993208912914976070696942179e-11,
5.767703934855339700712144537327817658798e-13,
-1.106651263152317752303674438766541324267e-14,
// [6.4286,6.5714]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0000713017415330332299889981456233102973921,
-9.335365688049717720489883035566907777017e-6,
6.151690742507743250236844679935732531574e-7,
-2.717963184023001743699661638724324813154e-8,
9.051415642201039435095759844212901559751e-10,
-2.422044125030438020756535687084360012197e-11,
5.422787239254636493121900929910643860507e-13,
-1.044078053405494559171888358201701673486e-14,
// [6.4286,6.5714]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006534755981634801950102416197741549788631,
-8.612367039501191285569944232515336910379e-6,
5.707722686452664393526927871830764099059e-7,
-2.534396418413695832854310979833971080695e-8,
8.477154379463259456174716407644279160307e-10,
-2.277200103188174909572319480246705928414e-11,
5.116168520553995553091934411049326107218e-13,
-9.88103678225312934097267777487715565175e-15,
// [6.4286,6.5714]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006028656927650833467714018597221430216957,
-7.990811761024572624951322295591155098493e-6,
5.32223247867291024636250008877121125649e-7,
-2.373603262880146308803715329864881257404e-8,
7.970200305838471895509957421260635297189e-10,
-2.148439124359338852837850790315748921619e-11,
4.84188504812709063030782769881640584725e-13,
-9.377487483351407058493805301826476716931e-15,
// [6.4286,6.5714]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000055935682327172004253173010377329794774,
-7.451125470133361460663649618690964028994e-6,
4.984566852052263896273570682045301722005e-7,
-2.231656120486624198134237222624915765179e-8,
7.519536882502214254045289025599220365111e-10,
-2.033257140416929591414682813591657928171e-11,
4.595137684709960497324976153685830815966e-13,
-8.922166316896151157559950458991744065829e-15,
// [6.4286,6.5714]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005215787829093352628595573198385810257987,
-6.978393592864860504227927800642399468749e-6,
4.686477853025691978437997649807825953439e-7,
-2.10547036033672390971416063687599535699e-8,
7.116399941043594420341170344120833069479e-10,
-1.929638759049683645229217756329854607059e-11,
4.372022825732006601579399057354114665414e-13,
-8.508530872160371582997922798171364177652e-15,
// [6.4286,6.5714]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004884875515005401975832840367023618946552,
-6.561068994228028338260722045197252670802e-6,
4.421487756710740646150456120764894303897e-7,
-1.992592015254222034456649762435885802832e-8,
6.753735608401261333373218910456257272901e-10,
-1.835944669605626128583262363833746276331e-11,
4.169334600109273753316500135400841382437e-13,
-8.131160659773896952821632505641482116328e-15,
// [6.4286,6.5714]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004592748295959619475045234332259384267578,
-6.190082859387433998560831233380985778504e-6,
4.184443232037338968774169554678404214376e-7,
-1.891046000764255405181504096370623796301e-8,
6.425806297316923678174939006275429536948e-10,
-1.750828576005049265053711565632348498647e-11,
3.984416892463739414723681507063218853368e-13,
-7.7855283163981768512869491589973073055e-15,
// [6.4286,6.5714]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004333058001571203451448263786106349637813,
-5.858220524844981902497590485991719555669e-6,
3.971196601608272794269390024114724722303e-7,
-1.79922579241962130375981861562635214071e-8,
6.127899971531617105869707558512793068413e-10,
-1.673175052648905554529014232720034416529e-11,
3.815051230718225734528747880804910662492e-13,
-7.467824236446308602024211528606142976297e-15,
// [6.4286,6.5714]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004100754367391486997333775874238351057886,
-5.55967524224457538923116931169397496213e-6,
3.778374164084415134759341780060504901568e-7,
-1.71581202005520155776647322335425282251e-8,
5.856112641465739490978032633720345319085e-10,
-1.60205246230365123200476193486127168002e-11,
3.659370853529857015031171645571167792638e-13,
-7.174820705542309459288622086633842580674e-15,
// [6.4286,6.5714]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003891772669571202450228231743404966080749,
-5.289723829711439363324278338101557145127e-6,
3.603205242119016730458604339168983427872e-7,
-1.63971156657795713239180555578951034288e-8,
5.607183576816523596290157896130547059396e-10,
-1.536676868410004278640258268275874954911e-11,
3.515794134019311533574777317242681195215e-13,
-6.903765588048545922791401395845816717872e-15,
// [6.4286,6.5714]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003702806680798007243424140529783060214404,
-5.044487338960127166229712058088198589882e-6,
3.443394289816694661590452987463246017888e-7,
-1.570011427493892523299621388451397442742e-8,
5.378368999639104741466013859445037174802e-10,
-1.476384076197595243480433843917204350125e-11,
3.382972485332444222144173492861516097481e-13,
-6.652298370410582050422304227992141372401e-15,
// [6.4286,6.5714]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003531141137272088716023940078042694456944,
-4.820752005737104698278632913183346511713e-6,
3.297023997740057548420937427120586758259e-7,
-1.50594334497049348612263941583337627413e-8,
5.167344228248131583067723209496666156013e-10,
-1.420607755528549580718927704574655279985e-11,
3.259749220813967894178107578729730758717e-13,
-6.418383288930359860058912881198414044137e-15,
// [6.4286,6.5714]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003374526404015972998333899601148685096788,
-4.615833596830025751160497621831550280508e-6,
3.162481024440824763002743932720213950134e-7,
-1.446856408128968324635746779485837352378e-8,
4.972127107170573583231117558125282219357e-10,
-1.368862164183852803233368332492587040038e-11,
3.145126785754587904389188291655397294184e-13,
-6.200255638424718183775646213741691680676e-15,
// [6.4286,6.5714]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003231083517781017744601781816590140935513,
-4.427473434211298946777228653084619200153e-6,
3.038398457073533117887385540384855630647e-7,
-1.39219561343086432980211625280772824984e-8,
4.791017538648218402840976844599371392879e-10,
-1.32072838682581518021714837790777088236e-11,
3.038240447931736720566730688234243122046e-13,
-5.996378340937946452339490821844051451771e-15,
// [6.4286,6.5714]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003099231403947908990216168098243589282042,
-4.253757839897277135988707864581631669063e-6,
2.92361078820743137463283992360904682547e-7,
-1.341484933498628855225414440271755584801e-8,
4.622549319006478199780040262154957235844e-10,
-1.275843286351854706352858089583362775755e-11,
2.93833701570318344330585567861821436783e-13,
-5.805406567643426386818082780255617585269e-15,
// [6.4286,6.5714]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002977630487928093730829463239454475630911,
-4.093055103484909738850781790698408811969e-6,
2.817118360349658491124663586516436214823e-7,
-1.29431383129876177064095759241016937945e-8,
4.465451468392778401700374143065484910572e-10,
-1.233890566581594203291262504906407658473e-11,
2.844757502264019346624940848737466554681e-13,
-5.626158731253738068344771481040297768771e-15,
// [6.4286,6.5714]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002865138572439436560525461061280860679368,
-3.943965704484192314234699021007961167546e-6,
2.718059045729863764673078838912407831776e-7,
-1.250326432468470838139623936418570412052e-8,
4.318616950181520686035970745411824665775e-10,
-1.194593492142208815659265718736700705102e-11,
2.75692291153360973703165818381251719967e-13,
-5.457592554913886409554291733618679484334e-15,
// [6.4286,6.5714]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0000276077599313893437055375313252529014509,
-3.80528266401663477955512898018941868919e-6,
2.625685508186183114378629067837022579355e-7,
-1.209212766748983777644872259452165314221e-8,
4.181077190572901943758162112502405338624e-10,
-1.157708919285610932450067528662905725892e-11,
2.674322511584545657585465458348996605797e-13,
-5.298785214398578852608357344223335864216e-15,
// [6.4286,6.5714]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002663697864811644103140014983144620706722,
-3.675959711455628222261641883075151054344e-6,
2.539346810175194412172443286090834148773e-7,
-1.170701633473152858060174270196035932623e-8,
4.05198118645314765744802897477094287456e-10,
-1.123022371349912366163398365619770729635e-11,
2.596504104043619241778261157985539305174e-13,
-5.148916769991214922549851801053598444659e-15,
// [6.4286,6.5714]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002573171798018939519530609307326966052016,
-3.555085534240382376864163029230178537028e-6,
2.458473430295887118780369738561272459148e-7,
-1.134554751766265966409378796251295155643e-8,
3.930578269509702134793777999618636639623e-10,
-1.090343952428892971490373312305488886467e-11,
2.523065905480187385756756024273295656314e-13,
-5.007256271531181455670267171665629199401e-15,
// [6.4286,6.5714]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0000248855987396826743391088422820124984389,
-3.441862802409483126718245385489370289689e-6,
2.382564978711365524307239667865468543562e-7,
-1.10056193449825413094712042062813316208e-8,
3.816203804074430147490336543592909418694e-10,
-1.059504937994241259707134812316084158352e-11,
2.45364973868263087060332164532739117664e-13,
-4.873150048297435003426942870783747522444e-15,
// [6.4286,6.5714]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002409303961686637964655272511755652403947,
-3.335590970191277050525082002288903655944e-6,
2.311180062448484542848009842480118468899e-7,
-1.068537083679749251599127856852038128841e-8,
3.708267254301511357958347558989051847399e-10,
-1.030354915603462150794943308861291597012e-11,
2.387935294534025164254149064932413113243e-13,
-4.746011794443269750824860525439589931095e-15,
// [6.4286,6.5714]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002334913679133893716877649011647977222413,
-3.235652087423361543020600565085087147252e-6,
2.243927875729570943064363391358783241872e-7,
-1.038314849271861605224299954477502604425e-8,
3.606242176645162092502430302695495188013e-10,
-1.002759375203828745826602681749071270922e-11,
2.325635273732326095673092315810956698547e-13,
-4.62531413776620951693115304259341901809e-15,
// [6.4286,6.5714]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002264956461196352866913995850678264528324,
-3.141499026016994544813596061346551972704e-6,
2.180461183472956106364700444404889728013e-7,
-1.009747827079913818327182280598118163392e-8,
3.509657785923504816348899500709694582735e-10,
-9.765976689248498371466355638463824191613e-12,
2.266491255365147331276010575549773246576e-13,
-4.510581439962193665979761460723434914043e-15,
// [6.4286,6.5714]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0000219904931821189597320935259269145640048,
-3.05264565685784040986548674468770698347e-6,
2.12047043686981736713020015668658954567e-7,
-9.827041972513002088144803020598702175055e-9,
3.4180918145922373545460901512296970658e-10,
-9.517612761111456251191018754364935447725e-12,
2.210270168928918425358468750703519898725e-13,
-4.40138362409252758331391786570097323432e-15
}
,{
// [6.5714,6.7143]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.3437568196240205373778143586414716197968,
-0.003682295929182903562958633854997854498315,
0.00005839087273643234843651684541754908314979,
-9.987648829946914052240612103390824023563e-7,
1.709158678183774376005622601368549250915e-8,
-2.82165591573762694123965921402945616635e-10,
4.404762597537635401220094945397142446374e-12,
-6.43224650143960098560823687355015565521e-14,
// [6.5714,6.7143]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02577607150428032492179713743612007809834,
-0.0008174722183100050407029258554940746333157,
0.00002097406254289033519381232149535574273952,
-4.785644318049656364650699815569105702524e-7,
9.875795680872514064025134268535328783132e-9,
-1.849816593321287025748446344685817479408e-10,
3.15187825520758867338212689744675545676e-12,
-4.898619066217656379298172571371827196233e-14,
// [6.5714,6.7143]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005722305528170035269321874599129328122533,
-0.0002936368756004265633814256437350325457867,
0.00001004985306790577583833207934533257389109,
-2.765222805896142366611727432802998978003e-7,
6.474358056658152897088984106611550953795e-9,
-1.323642448862108699009916833951628418294e-10,
2.400387234880481941724329182778419237326e-12,
-3.904501548100153483969617979439462009648e-14,
// [6.5714,6.7143]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.002055458129202985930463626053760275513371,
-0.0001406979429506494148424144337846289606747,
5.806967892383166785396611470124857561637e-6,
-1.812820268443131808708893094276110998303e-7,
4.632748554113128975318912170972048317059e-9,
-1.0080418811150796505069950838130776101e-10,
1.913259852269025608304968694366948427338e-12,
-3.220212056386640809666423437496221549638e-14,
// [6.5714,6.7143]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0009848856006545458924838458088784139642766,
-0.00008129755049333771089584292018201559371058,
3.806922563731672456559066486062089157075e-6,
-1.297169605801396208908617844708036875674e-7,
3.528146569293962089612271789314540999939e-9,
-8.034669001267114859410043556989234371131e-11,
1.577950655927650277679661707961805728735e-12,
-2.72635181242067265493154947917262096764e-14,
// [6.5714,6.7143]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005690828534533639662443163010217320711318,
-0.00005329691589222040559133472341112096515592,
2.724056172183894595258270465281964386755e-6,
-9.878810486059009971719645348314174008812e-8,
2.812134137609367083213081418268518844812e-9,
-6.626509205530772989674205694760302171204e-11,
1.33595345735644095328352518797333891852e-12,
-2.356140346886426555015480282966888741682e-14,
// [6.5714,6.7143]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000373078411245542830212362737651222015414,
-0.00003813678641055431066693432130841464044412,
2.07455020207324908823615509121933433452e-6,
-7.873975666161537442747176260589466014798e-8,
2.31927821050914903604792560060284229049e-9,
-5.610228305810317878158613553610691321065e-11,
1.154545335232536906422641596581710585309e-12,
-2.069914735090422878707175404711508585833e-14,
// [6.5714,6.7143]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002669575048738801666324177952455280179888,
-0.00002904370282900749037659410250839345731035,
1.653534889894694334453601741738620553731e-6,
-6.493979061413636246092011507872339697705e-8,
1.963579896747295132242314671973598332241e-9,
-4.848399319256539662528482469268549938102e-11,
1.014291136469415410862529390466058530737e-12,
-1.842910374126437267031245714573513287843e-14,
// [6.5714,6.7143]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000203305919803052425334879620686795475686,
-0.00002314948845850951979870465277934093744547,
1.363735602897564537902946427095615096807e-6,
-5.498023775696495807807464257166459800151e-8,
1.696939752394078457421654152535758244795e-9,
-4.259400650681855820473909587623999570301e-11,
9.030559896538574342356110817524232700931e-13,
-1.658999826028973135316666966818078468315e-14,
// [6.5714,6.7143]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001620464192095666319051889289674512468791,
-0.00001909229844055118409595294884301373401332,
1.154584992896905954209693335858972394956e-6,
-4.751431365581651491149146192960916871e-8,
1.490790219180829823507659806712879404678e-9,
-3.792269922365237017120207571381875240644e-11,
8.129372998315640934556488543131803938044e-13,
-1.50729715989685724185871130588564693335e-14,
// [6.5714,6.7143]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001336460890838582825083972806857118204116,
-0.00001616418990054320484780569313135541374263,
9.978005867727384903480768183215786334771e-7,
-4.174212667620825194605327844073975745506e-8,
1.327294464938779819752139308707982786863e-9,
-3.41381907715045938152330721079643582749e-11,
7.386008533700835784976083106452096582228e-13,
-1.380224618784363769524486853174308839691e-14,
// [6.5714,6.7143]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001131493293038024282199756804216945547586,
-0.00001396920821480591365736323607394019508768,
8.7658466020092190140215582729270360616e-7,
-3.716424551529839576271591835571798167905e-8,
1.194836669687831452942510293553503304643e-9,
-3.101646449369290913190110973832895646754e-11,
6.763334707055494248255447562004411297157e-13,
-1.272364620527510331013500324624616921386e-14,
// [6.5714,6.7143]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00009778445750364139027597081652757934896909,
-0.00001227218524280138581134210048046594968306,
7.804491558217775685207157002936288781174e-7,
-3.345542721209560023545302233941369402686e-8,
1.085576250462464244284277855091561090823e-9,
-2.840158171544181947685113329828935321186e-11,
6.234804778232217432393892735775264762364e-13,
-1.17975264598086681333439159691877814948e-14,
// [6.5714,6.7143]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00008590529669960969569430864075695138302957,
-0.0000109262881814941495638804039711747007964,
7.025639714544861756654815862077822214883e-7,
-3.039613544240856300638364412387722188833e-8,
9.940553536595010636245432762541810242197e-10,
-2.618205723277336438968339675694434985268e-11,
5.780992156529692756073068456796231536071e-13,
-1.099427449289168270473407635807403819264e-14,
// [6.5714,6.7143]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00007648401727045904226238943359287790348824,
-9.83589560035275648565126797270513409641e-6,
6.38318844291029563706143476165722615737e-7,
-2.783355030446851078147561842421355813231e-8,
9.163719971505081823811409142715165920859e-10,
-2.427630781098693104252650359454130088293e-11,
5.387386391820221835135459373838666823784e-13,
-1.029136692029078941088497227717746307825e-14,
// [6.5714,6.7143]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006885126920246929098153881330629623148322,
-8.93646382006496935072771751393051503234e-6,
5.845045563942628585439196223015414105214e-7,
-2.565841629799921906221318931640920279397e-8,
8.496707677294298251367890000647504465137e-10,
-2.262339608834203230590300267213129729081e-11,
5.042950754927504340524432500624388217706e-13,
-9.671393568505399306505440728233119693957e-15,
// [6.5714,6.7143]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006255524674045478127552250424378219982798,
-8.183063789510773255122068954132143818749e-6,
5.388267422583848412713511682145555610503e-7,
-2.3790781852697786947326895051586331896e-8,
7.918188577420748060138727054628201314606e-10,
-2.117697292237605349409911838134362850334e-11,
4.739154045608288762247230038289835116086e-13,
-9.120700855417208805931147967939048171156e-15,
// [6.5714,6.7143]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005728144652657540882065069250813726761463,
-7.543574391608961727332172227361322266991e-6,
4.996064189070341874589399203631373780007e-7,
-2.217092835382313039579386472419718453672e-8,
7.411940472076583280756320243339072257723e-10,
-1.990121133990093941753420874672062700645e-11,
4.469305835609392871958057243390637674739e-13,
-8.628441191287363715614803705438980060178e-15,
// [6.5714,6.7143]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005280502074126272831984021926035510196162,
-6.994489864690484740897369225247713686086e-6,
4.655894954306478027823127295965451218895e-7,
-2.075343364157265074830708750891420637863e-8,
6.965423920689913086349768201730808298673e-10,
-1.876801481226145297965973980623162314085e-11,
4.228090665387390399715907881304132612857e-13,
-8.185894526982292551175038703517786241147e-15,
// [6.5714,6.7143]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004896142905283338959068201632087588640784,
-6.518252936021465893991132964566804634214e-6,
4.35822106473370845103236381460894585023e-7,
-1.950318728206830323738113171438913290375e-8,
6.568805138267443617851260102611044761142e-10,
-1.775506106619147739617898547336586044614e-11,
4.011235595541447469419021431477652883475e-13,
-7.785975996378218082028058342370050497459e-15,
// [6.5714,6.7143]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004562777055215025782272086625418444632253,
-6.101509490619943072821653070525911518874e-6,
4.095669329237641505783478510821249957289e-7,
-1.839265467710181776365786539677531195826e-8,
6.214271329195864229422021500206771232027e-10,
-1.684440593587148954391170271996838520415e-11,
3.815268946214026194475307925582746214579e-13,
-7.422873143231503142787006751799467869179e-15,
// [6.5714,6.7143]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004271056643433959822135884067233887000221,
-5.733937060925772681827148562466855477324e-6,
3.862457482194538604249093072994747405742e-7,
-1.739995999876799339868884736753854571081e-8,
5.895542035463234718526121336384563058574e-10,
-1.602147017004695478871402114617919183969e-11,
3.637342534188363026862156902198687550509e-13,
-7.091774892597688433793251973547981415783e-15,
// [6.5714,6.7143]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004013755942648040561929854219758215204648,
-5.407440475065724618666252746467645335887e-6,
3.653991599744305981781762993500928938542e-7,
-1.650751796447657178946379008641837402794e-8,
5.607514519151396992687518963190228079093e-10,
-1.527429290474209830626822770241729070266e-11,
3.475098865768666556991526336127721923642e-13,
-6.788666471958156418930583081691013698826e-15,
// [6.5714,6.7143]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003785208332546006930153465512063929373646,
-5.115588239635670909143350850563334212335e-6,
3.466578772542988055362599076780620748298e-7,
-1.570104090792485690120484423962938393976e-8,
5.34600247788654839812529745500854256699e-10,
-1.459297393223304299615131436505355656474e-11,
3.326570645721407027655439912441374626805e-13,
-6.510172407285717144996685683384104554871e-15,
// [6.5714,6.7143]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003580911767744969344987731489327799153697,
-4.853210281554076527691502831674152176365e-6,
3.297218590667017525422102265117555754788e-7,
-1.496880718235457411440054870504515274274e-8,
5.107540838980428870496230539540677652433e-10,
-1.396925168416515346640375863115678834482e-11,
3.190103843463032022389299641624923120509e-13,
-6.25343501888256303411010358593690866415e-15,
// [6.5714,6.7143]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0000339724719708785328863688641317323280766,
-4.616106026927949632670063475822602861467e-6,
3.143449508297155751372663438271005519433e-7,
-1.43011145841434842857242765074441852966e-8,
4.88923805352185592558814293012442417916e-10,
-1.339618014515721695308213779783321412784e-11,
3.064298154533854611037908185240542990277e-13,
-6.016019445698506020406637444712319924413e-15,
// [6.5714,6.7143]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003231274218849564472039607724514796391364,
-4.400829311610358165283844291212563041301e-6,
3.003234062672731676672762980993263585992e-7,
-1.368986677625875475695221646305212479594e-8,
4.688663016138557697651648212222540550103e-10,
-1.28678788190907563905717618812665723596e-11,
2.947960461330895005150302683395018634496e-13,
-5.795838714484050186082798062097628204582e-15,
// [6.5714,6.7143]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003080580518127250454114746445991071219322,
-4.204527687736364402903159835306295980419e-6,
2.874872023016849718455720222611937790942e-7,
-1.312825666358776264063282982819727521957e-8,
4.503757553198729101711053263605334913478e-10,
-1.237933728654858918050382074578269896657e-11,
2.840068116043804056914203319469936505814e-13,
-5.591094111277574703333643658774030625532e-15,
// [6.5714,6.7143]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002943169381415454829087232388127619948125,
-4.024820832218316051256493420207307149393e-6,
2.756933899355858439541548612304133782492e-7,
-1.261052135990058356570307557015482208569e-8,
4.332768017914767119651087773473591143665e-10,
-1.192626101108346893512590925277293555532e-11,
2.739739721916487423992578286362556665063e-13,
-5.400227347221879167445576142304200074775e-15,
// [6.5714,6.7143]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002817374582552820991024464493025924835705,
-3.859707459093102423002396432763841535396e-6,
2.648209485581473170382546475684521826345e-7,
-1.213175065413894025645670182082042059564e-8,
4.174191322537490528409068959493029495372e-10,
-1.150494863501510283160830755313798775731e-11,
2.646211693872834164712722472535197397791e-13,
-5.221881896586748792944080544303148306504e-15,
// [6.5714,6.7143]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002701795221365171458837889313166986748017,
-3.707493279809126139184303405576446263968e-6,
2.547667637371455198656625277329276850369e-7,
-1.168773590055878852372451321690655713989e-8,
4.026731991885255841201528309349256484509e-10,
-1.111219354585904290577481397852328942424e-11,
2.558819313636197770928150728647819813191e-13,
-5.054871527785352931849521783446684906837e-15,
// [6.5714,6.7143]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002595245295866388067302401234291126274844,
-3.566734692315254019840196586997508534204e-6,
2.45442453911955481753026366064476827765e-7,
-1.127484976861083473327802900489972357204e-8,
3.889267711594201059341703181029305485648e-10,
-1.074520431749735360405061530884012211897e-11,
2.476981309505673928228250765364117940187e-13,
-4.898154519698403274579205053100382003959e-15,
// [6.5714,6.7143]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002496714284620677590483961530753959428631,
-3.436194354762737372329017518331007157274e-6,
2.367718451410419985826221416146398147194e-7,
-1.088994977804038797418593436664170389944e-8,
3.760821482528088609743449602281234593584e-10,
-1.040153995321024746104873430474802633299e-11,
2.400187222005195636628372567515448304177e-13,
-4.750812404887161165651333948969200082722e-15,
// [6.5714,6.7143]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002405336048333915943568798433843365722234,
-3.314805831970084132779264981089019187897e-6,
2.286889453390565276543509224085030310786e-7,
-1.053030033123423172711547100372690402514e-8,
3.640538955839468964100778437352853780616e-10,
-1.007905682801245957139624532062095068191e-11,
2.327986987766283840122477611952235903932e-13,
-4.612032342402001240713146069948502167057e-15,
// [6.5714,6.7143]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002320364082379058681878022436882199524458,
-3.201645234742415408370198156905346269707e-6,
2.211363069561214921471933581840167955904e-7,
-1.019350925139130786419332729917819695602e-8,
3.527669862787488087511020709670499500124e-10,
-9.775864946472510849780260387012381135743e-12,
2.259982301959816056035744631171940736759e-13,
-4.481092419838417312818113423376485449606e-15,
// [6.5714,6.7143]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0000224115166431969058046497113636514566592,
-3.095908297381445751822557378885896505986e-6,
2.14063694279414644633762332260019387582e-7,
-9.877475786012096287433748083126114439319e-9,
3.421552704974693916457428971405376005791e-10,
-9.490291669545848132020135267686892102215e-12,
2.195819416097590623078145602365785177592e-13,
-4.357349334063921545131699132234161191287e-15,
// [6.5714,6.7143]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002167135808167011826259139090454064718514,
-2.996891719907664260735338740206183562528e-6,
2.074269915064460390789723011643087109775e-7,
-9.58034773956126774269843248763351651864e-9,
3.32160205873868983711723394286610437414e-10,
-9.220851469232546683655645540864859413035e-12,
2.135183101414289617957964841905494359221e-13,
-4.240228014829010199059328520809187817476e-15,
// [6.5714,6.7143]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002097824203935364787602281558937923425027,
-2.903977881086212194141897109117435294864e-6,
2.011873025309737395861422939019264520529e-7,
-9.300485925763970567200219469808608420086e-9,
3.227297989282376837465998210537657930875e-10,
-8.966220578074291133158661636304305613112e-12,
2.077791564288774705622460488545641559467e-13,
-4.129212844105915804669667604047320104681e-15,
// [6.5714,6.7143]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002032784516760348345837995174316676756216,
-2.816622235429702902243533220395594391372e-6,
1.953102044412258417983538856137537832629e-7,
-9.036434527170217187475911984781158333421e-9,
3.138177177997936329925952481823935682103e-10,
-8.725215636722499606838805097971755474068e-12,
2.023392143593688209115308828576578408072e-13,
-4.0238401929119834007000792814401744052e-15,
// [6.5714,6.7143]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001971635564800791846125493320291517708602,
-2.734342862173330132249107267987730587005e-6,
1.897651250707525891020415154471433840253e-7,
-8.786896251661787196903617337623138147328e-9,
3.053825449115707443869038830382481550676e-10,
-8.496775625195388488198330457742991252061e-12,
1.971757653630602365260842208754375259815e-13,
-3.923692051313446699139583761085016553519e-15,
// [6.5714,6.7143]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001914040003521330911527563091510580310258,
-2.656711750986797660555917040335804669771e-6,
1.845248212850713370365085897504579822837e-7,
-8.550711407068870548593057945313248365344e-9,
2.97387144564418880453941610598200988746e-10,
-8.27994650525258007982882777435593350178e-12,
1.922683262738707409186592386127166928786e-13,
-3.82839056979084433819331626278736583833e-15
}
,{
// [6.7143,6.8571]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.3401319326164531978806632083667294527825,
-0.003568443496856959150304151841873814027014,
0.00005549437069094595482607901425455147557841,
-9.331343080387045552817681104151423656629e-7,
1.574463247258520199517748452152253239098e-8,
-2.570426762955396768606158463838088128919e-10,
3.978085087706292003694246475640879994805e-12,
-5.770767069884042357801014997413664252717e-14,
// [6.7143,6.8571]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02497910447799871403531475301207683593542,
-0.0007769211896732006688649435156613148748784,
0.00001959582046881440984116319900274585990652,
-4.40849710940344721261961790873387832609e-7,
8.99649364882143342893773454780421860049e-9,
-1.670631772058592035744607020124558393632e-10,
2.827744736210908163007242264583254236473e-12,
-4.372407562770998088190417009254627022629e-14,
// [6.7143,6.8571]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005438448327712404668230881380957930330127,
-0.0002743414865633678400214383332774528009074,
9.257843929748566229534756827435757440628e-6,
-2.519018235229196690121927012398609167498e-7,
5.847211184510586848478104077830356930656e-9,
-1.187522620335704917441459954611630772234e-10,
2.142536328206310178037326946320914295637e-12,
-3.471181328329767207699981099359716807249e-14,
// [6.7143,6.8571]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001920390405943574868476041040316154029877,
-0.0001296098150164520584959314612199757804738,
5.28993829398243376079063049880765228438e-6,
-1.637219142810531882904800094272059408441e-7,
4.156329156232107433709219653439737786066e-9,
-8.997582406825433496986879809377251708669e-11,
1.70092666811674831153906376164642397473e-12,
-2.853800861544909514203643671447557403008e-14,
// [6.7143,6.8571]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0009072687051151643994035238872740521470965,
-0.0000740591361157305377317763625776995851675,
3.43816019990308348611813967297462158464e-6,
-1.163772173159028436742883701311206525876e-7,
3.149153829501773195176249757459333462437e-9,
-7.142988253887356365842165570949185379377e-11,
1.398403661042867904380085312455242654323e-12,
-2.410014485854200382568733267080230670506e-14,
// [6.7143,6.8571]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0005184139528101137552925667727808333146806,
-0.00004813424279862287165366431008959245840438,
2.443921563634807388394057068333073387024e-6,
-8.817630803794200925111449273158816433947e-8,
2.500045877558260179502066895172899544945e-9,
-5.872515955695299592798615548349464975982e-11,
1.180943265542161297851966311923302740713e-12,
-2.078462815253913399413237012540612110216e-14,
// [6.7143,6.8571]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003369396995903600937253289656273037454229,
-0.00003421490189086950235993552503902915622616,
1.851702468797535828674715611694298292396e-6,
-7.000128528368003495024044756100013646436e-8,
2.055380574444867648060700599568807308178e-9,
-4.959278144863258659210865097110429995945e-11,
1.018478934694341845267695782150672964156e-12,
-1.82286132047678515874097330922531363583e-14,
// [6.7143,6.8571]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002395043132360865094611424175059846737054,
-0.00002592383456314967529565771246440174334204,
1.470026990957958344217929879421339846091e-6,
-5.755065671751365073152985096281915616445e-8,
1.735747341667310928369751069574017795832e-9,
-4.277003787373859979258290577270244985536e-11,
8.932309585440266760134689142008774938151e-13,
-1.62064182741123867284375170484549589168e-14,
// [6.7143,6.8571]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001814668419420477206632273831161283728659,
-0.00002058037787339718701875346995314718789629,
1.208563791068401680521012678422987747266e-6,
-4.860092613588140672042373750016962754718e-8,
1.496951317380624834963465055855764275203e-9,
-3.75102359405134988612854935656157014053e-11,
7.941407362064958758428655358005002085936e-13,
-1.457157272119517106202116780521265057171e-14,
// [6.7143,6.8571]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001440626451137803032701888749979760369313,
-0.0000169198930749447082229800604734211630213,
1.020619448854072208112992821574657912137e-6,
-4.191463740327398701625269221309083397444e-8,
1.312858250415723731092550039542711024983e-9,
-3.33489513747108796879164261503357512156e-11,
7.140310705820153887519882481851221587249e-13,
-1.322550855057900028005090573681032038257e-14,
// [6.7143,6.8571]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001184392515246129521619682278731281452773,
-0.00001428867228394519491990894283437698974621,
8.802073854692720235852315593225037312609e-7,
-3.676003148428401337810175302694964322427e-8,
1.16721329120424345794284892844042352297e-9,
-2.998476755884224903213230742909166111473e-11,
6.480720330620878167204754839218130559172e-13,
-1.209979705968349014113749484890346761659e-14,
// [6.7143,6.8571]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001000207059876163594368279083238577521579,
-0.00001232290339655892412117195246163869526853,
7.719606611704445266966044810973574824848e-7,
-3.268197258909090495328170538663621309599e-8,
1.049466858156182554409820294313602694889e-9,
-2.721484579270689958123238370096721496766e-11,
5.92910546513614137673881618254656801853e-13,
-1.11456345065205286191085848326795162548e-14,
// [6.7143,6.8571]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00008626032377591246418905931764980835508384,
-0.00001080744925637613822418808475976831265161,
6.863214243713562841553225921980848057204e-7,
-2.938507243178258122501679059177608047451e-8,
9.52519596780988102329588762176048803007e-10,
-2.489837020034491348327976811680026383254e-11,
5.461551748692243392328433302861771759153e-13,
-1.032738444849476504065799366779144389174e-14,
// [6.7143,6.8571]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00007565214479463296321036594201095137281372,
-9.608499941189595105944334605641058033277e-6,
6.170865210678526671360490976426910054153e-7,
-2.667054908558583157092244108863498756898e-8,
8.714429514325693104536447301107707791946e-10,
-2.293491042916292867531826896368203765652e-11,
5.060596924214227288245262342476155990971e-13,
-9.61848084892819192088991515963331136957e-15,
// [6.7143,6.8571]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006725949958832716164719700221790495265447,
-8.63921129494114966030461384691066239053e-6,
5.600815307976955287014600552771050660844e-7,
-2.440040299162221236731428243031947903041e-8,
8.027218597798100907453399381753559030438e-10,
-2.125113256333045731368230927762971338373e-11,
4.713223324878731047503212995952392401848e-13,
-8.998753292020246055465022570185316106098e-15,
// [6.7143,6.8571]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006047447906458804376250448150358864194825,
-7.84114143115948303134642783667984002817e-6,
5.124084628244369859104613914806728078729e-7,
-2.247621240401242139971913127252976404318e-8,
7.437896347762013017204029992916304475856e-10,
-1.97923682395101598083400152929505111616e-11,
4.409547205089264001871774533474036163672e-13,
-8.452635543333806761622053349987969506664e-15,
// [6.7143,6.8571]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005488799001811637756951573267680466420011,
-7.17371847953433676160285659439660703174e-6,
4.720004604846112425151449238436791557549e-7,
-2.082611008497805136165196514105731404276e-8,
6.927328837109326576040085905103535118663e-10,
-1.851711029726894460307144389978077677985e-11,
4.141940918080151014911451979285281915777e-13,
-7.967938040402264873417016343272383862761e-15,
// [6.7143,6.8571]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005021602935674035386972364777319598052004,
-6.608006446777199148218655151647907777623e-6,
4.373483117848713839900704452633881161268e-7,
-1.939652103823862719062463360721203685595e-8,
6.480988559736604884640569565802045423128e-10,
-1.739332624696238867275372763050499335848e-11,
3.904431424177410820655261601163565517673e-13,
-7.534989363022676238007617635105941859931e-15,
// [6.7143,6.8571]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0000462560451274403907461940468944824329623,
-6.122876364981220970705512384181529736256e-6,
4.073269418033271409784868765169016859939e-7,
-1.814676824640120941313267727763031712503e-8,
6.087664144307372576312949541408365811422e-10,
-1.639593223381304719748673524888002508838e-11,
3.692279602448789558374162178802054536822e-13,
-7.146024544726973502179361433345235409562e-15,
// [6.7143,6.8571]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004286013455486854365801043954108446122707,
-5.702577185239944611998038496207665027859e-6,
3.810821331747265443738530862790540717457e-7,
-1.704545986947750818249120991619461056902e-8,
5.738576241681545767219581535334503843663e-10,
-1.550502631304071706596807797278982226218e-11,
3.501680516855000384900497510806657627484e-13,
-6.794741871305632702788365191622928394045e-15,
// [6.7143,6.8571]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003991804029667960928778377972406306110054,
-5.335149864439847548022075152288018192723e-6,
3.579546572593153088014522189368016786733e-7,
-1.606801372967355054748000104809326071444e-8,
5.426759171212532677483969391558113924243e-10,
-1.470462968997475829663227195548894853378e-11,
3.329546242699449875399010066645649098294e-13,
-6.475976807632385217488066215248631285199e-15,
// [6.7143,6.8571]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003734604905107892996871969100805031086001,
-5.011365201624373954008842329234788384565e-6,
3.37428288323419836710713638671709638336e-7,
-1.519492592101205927511350455495066133579e-8,
5.146620354787686027572132316704312220584e-10,
-1.398177468237129031917763353109891075366e-11,
3.173346087124665134347405503977636052881e-13,
-6.185458721412383968514701940824369371465e-15,
// [6.7143,6.8571]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003507955641137061492888559118136491376956,
-4.723996036522096941321029410704675629425e-6,
3.190934443415171671160767970343953098621e-7,
-1.441053722463853817504119810192702533664e-8,
4.893621103640193379554987520087022657144e-10,
-1.332583371543975584902580913438297385857e-11,
3.030987380959634601353126779406065394174e-13,
-5.91962703848214092119885494606083116323e-15,
// [6.7143,6.8571]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003306797225565467594904597076498058065137,
-4.467308220775697977108223962699758295514e-6,
3.026212817176627623558590352878392891761e-7,
-1.370213931188907368881358765533493983765e-8,
4.664041766609048299855157469027290067139e-10,
-1.272801870031907890619103202130842765434e-11,
2.900725392660592952854703761882245873711e-13,
-5.675490662870967089146533054653070442321e-15,
// [6.7143,6.8571]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003127115754542988330029149908614162306349,
-4.236697944041956005055302657421545040559e-6,
2.87744925549914337520781412592277681908e-7,
-1.305931715941400914998766861435661363775e-8,
4.454806512606231808367002047647735893113e-10,
-1.218100271338108489983406963608658658088e-11,
2.7810944424574682073958468819810275213e-13,
-5.450519304155691647670729801931540848889e-15,
// [6.7143,6.8571]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002965688560829368958931015841992421264721,
-4.028428957693681140248991567262962396931e-6,
2.742456603479290168059951392496693079678e-7,
-1.24734584400827381331090101873451033565e-8,
4.263350918373324100736486921782086349872e-10,
-1.16786307074669124680459993728665506895e-11,
2.670854651430675534975453396563199236747e-13,
-5.24255862031880247227885874927988483557e-15,
// [6.7143,6.8571]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002819900270385576562247892376821234062566,
-3.839439244866074923420135869514161162712e-6,
2.619426272419639913669548498691327317713e-7,
-1.193738276869960470359391253889047459995e-8,
4.087520717414578393094380633129485267075e-10,
-1.12156958830903491935293113013416999774e-11,
2.568950360456561948761516666174881380538e-13,
-5.049763335409197527117857752992903823863e-15,
// [6.7143,6.8571]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002687607471406252218558029582106888843692,
-3.667196781382739582853642839857206544654e-6,
2.506850381429104225850136267658854786689e-7,
-1.144505819901443306095751747404489220108e-8,
3.92549352991835268051865899832940958517e-10,
-1.078776506794915611133782066686166796772e-11,
2.474477357015547680474128045528688962134e-13,
-4.870544064828112751325172792342482230957e-15,
// [6.7143,6.8571]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002567037746967917487718857462635050245876,
-3.509590533996152721727312770023879769592e-6,
2.403462221795145629746264904134122846284e-7,
-1.099138206750087163163676382008669254583e-8,
3.775717745586286726815678720221583041818e-10,
-1.039104108550170728318087759657120657073e-11,
2.386656818899311159030082960225923004263e-13,
-4.703524695341416145121565230562852217178e-15,
// [6.7143,6.8571]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002456713373797306692005523633185944546465,
-3.364847110508763044424982510081999710513e-6,
2.308190234177229808308367910396678917401e-7,
-1.057200986527674413970881790387425088895e-8,
3.636864352635299488690113648613553598161e-10,
-1.002225333144442997264399538571397883829e-11,
2.304814429861378120794509234363405794757e-13,
-4.547507965602626934555231091301998889118e-15,
// [6.7143,6.8571]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002355392977356133924531204033680019879438,
-3.231466327843823528939020293600098226662e-6,
2.220122071710099316485329063740943661443e-7,
-1.018322035930854887983582064865665341694e-8,
3.5077886395648352881219886150195664096e-10,
-9.678570069942067157898468296757814635988e-12,
2.228363513618644920876682767889398742936e-13,
-4.401447471961713693336711306231957796354e-15,
// [6.7143,6.8571]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0000226202642949067626992976398398361900771,
-3.108170900389974649122337006331130278325e-6,
2.138476275456718419761943123120289308562e-7,
-9.821808357358852697156625222890279957132e-9,
3.387499498837572771245410046258849349484e-10,
-9.357527605029020782586469924795395449984e-12,
2.156791316323034835664237285824361660021e-13,
-4.264424748536922177978565944551042230612e-15,
// [6.7143,6.8571]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002175719630272982059932924071339372710634,
-2.993866785635367167080780755509041689013e-6,
2.062579755047225822659357509242834330941e-7,
-9.484998758291538469110932418076926868067e-9,
3.275134636869992070703625281126061961177e-10,
-9.0569726740713297151088943573493435154e-12,
2.089647775486214654562334232579158590101e-13,
-4.135630384429205306540579030202863606116e-15,
// [6.7143,6.8571]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002095706749944756828045567992610828908962,
-2.887611657062195968355070254351301054552e-6,
1.991849739243036633854336067870645575563e-7,
-9.170377140044781903199334930244827002801e-9,
3.169940411744147942326973235427694682187e-10,
-8.775015283118572131725560532291537728727e-12,
2.026536267155941979422715037719867916161e-13,
-4.014348375423428453482600485186391840796e-15,
// [6.7143,6.8571]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002021328159943536994173625030977905874916,
-2.788589634936442825821742320682751216686e-6,
1.925779199411167488629414982695078259182e-7,
-8.875833305223506956989120351557549908767e-9,
3.071255325580903325430842034907044841694e-10,
-8.50998984993442882673097860541416854102e-12,
1.967105938032931579947434022868453158222e-13,
-3.899943084199191088349126441395148482511e-15,
// [6.7143,6.8571]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001952012744455509799355178715841676907261,
-2.696090879171931581790212481696267676913e-6,
1.863924994098652182798159258087286926943e-7,
-8.599515059744013130241814063093491752697e-9,
2.978496424600683514348289378968444336448e-10,
-8.26042300298363757741797066417829600934e-12,
1.91104531579171618512576932485210767886e-13,
-3.791848317318962372822342273375883723009e-15,
// [6.7143,6.8571]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001887263615420351933228785957805827738718,
-2.609494991734510044477646243302154144233e-6,
1.805898162547913400435499700812525139428e-7,
-8.339790133003727962156598779937875306935e-9,
2.89114802876895840558255075162270935497e-10,
-8.025006748227782113339244645361693009692e-12,
1.858076956650530537681660919466812850971e-13,
-3.689558130067150521572388142816920856742e-15,
// [6.7143,6.8571]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001826646494214156861566206961994420845049,
-2.528257427563570414425962527598234085327e-6,
1.751355927932410735226600186589190583288e-7,
-8.095214620888253539946709141476001301939e-9,
2.808752340174809600497218952729597791394e-10,
-7.802575991835140249366152370673163521224e-12,
1.807952939611372123148408476047448569514e-13,
-3.592619049529865619721185431530178658517e-15,
// [6.7143,6.8571]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001769780199294499124764580649701009340059,
-2.451898299101956520841968531975117211576e-6,
1.699995070388120454646941301010851294091e-7,
-7.864506689231096315673746463696828079147e-9,
2.730901575979412198099308369996444591827e-10,
-7.592089618466429623274508836746374897817e-12,
1.760451055657256879441750159854930354068e-13,
-3.500623467930312831182241719271163867329e-15,
// [6.7143,6.8571]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001716328809371369403284752905557218337857,
-2.379993098540035496953485972634479971567e-6,
1.651546404740078759252597149896429464366e-7,
-7.646524546069195964065938194589515291748e-9,
2.65723134581607573566204718563123180163e-10,
-7.39261448801684692369848811600787661615e-12,
1.715371570390777807320508985351262258718e-13,
-3.413204006435190212543901721724636600997e-15,
// [6.7143,6.8571]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0000166599516897802469040076600858071595142,
-2.312164966632858347730085616768682071653e-6,
1.605770154676042840891879865350180526829e-7,
-7.440247898362851440994054202654376704057e-9,
2.587415050649983654442030885357881522569e-10,
-7.203311840508345509133229222233030752942e-12,
1.672534462216884769950391087261814459623e-13,
-3.330028687587221413128383677605448769859e-15
}
,{
// [6.8571,7.]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.3366180658474897624399453787444784029041,
-0.003460192441614033008560705412728012847397,
0.0000527869236104343380821093232148125129084,
-8.726486274698052201758995952871775212438e-7,
1.451711835245215847323570515986176756789e-8,
-2.343457569115331418702723866320460675152e-10,
3.595181368576256390504887996679151132078e-12,
-5.180211762858082076894245394855213502745e-14,
// [6.8571,7.]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02422134709129823104497106558928819688896,
-0.0007390169305460426037033348027789587617653,
0.0000183256211768673452011797379019652648641,
-4.064793153938480886666073551469362145496e-7,
8.20210147276258086000750671329562588508e-9,
-1.50982975616005903248219599994485210694e-10,
2.538365015351647777550878048724173767055e-12,
-3.904509070299247265325585000554863259635e-14,
// [6.8571,7.]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.005173118513822298213668761619977213279885,
-0.0002565586964761126857161233082494750179215,
8.536065623271986306814976701711049628025e-6,
-2.296588424432445374505035911321628886184e-7,
5.284404130874236176264375190799431087973e-9,
-1.065997540260956208526983819374488947946e-10,
1.91325963963641406065575092085274076149e-12,
-3.087108379507812983413751409558770814483e-14,
// [6.8571,7.]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001795910875332788789690715522012183242911,
-0.0001195049187257831029773417900478609102538,
4.822835691309126216962321828211909410985e-6,
-1.479633166526984422862104105604284572991e-7,
3.730991377700905409913942465505762287138e-9,
-8.03474179088849118726052511883683204808e-11,
1.512725385844929694070818941036169118395e-12,
-2.52986355526237368991537634110094113856e-14,
// [6.8571,7.]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0008365344310804817119580219600830453200931,
-0.00006751969967830695751713606633399784973369,
3.10722964970752009542809102686166257995e-6,
-1.044677594080123868036147695824367010511e-7,
2.812159615440175602922656713637033446633e-9,
-6.352647525012774009869434847374851331316e-11,
1.239669528692350189916197557516174245994e-12,
-2.13092901396924952038391102847609745611e-14,
// [6.8571,7.]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004726378977481486948425484234621423235872,
-0.00004350121509588737239815542090569758105095,
2.193822947569006757114868435614930177699e-6,
-7.874046994868795401987224570345786552591e-8,
2.223426623799320108862566291628106761137e-9,
-5.205924308457079119498203715001918613974e-11,
1.044187073386373273941803572183309358796e-12,
-1.833905535732249786192716770883056698439e-14,
// [6.8571,7.]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003045085056712115998824959893914825229158,
-0.00003071352126595041529669449680833190688969,
1.653549868923109878374149272364401437999e-6,
-6.225594609355803993142888600213504078237e-8,
1.8220734991220332354843531451332299967e-9,
-4.384983615054326077270829089966421155374e-11,
8.986419939844642993175734629085747703149e-13,
-1.605587829453267506067013417368152660796e-14,
// [6.8571,7.]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002149946488616529008763836982547566308201,
-0.00002314969816490961858874611760777559763039,
1.307374867965314087286886370756774618596e-6,
-5.101805853220976768457672006206060188439e-8,
1.534744257332341554681261935168421751772e-9,
-3.77376185073278220788593753469667829844e-11,
7.867634338331448480040147353152693402663e-13,
-1.425402821191414096299459645242847922064e-14,
// [6.8571,7.]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000162047887154367324499405433310268041357,
-0.0000183032481515018970120985568796236084067,
1.071379229176944818478000044973423045909e-6,
-4.297283970531807114336800180180669818636e-8,
1.3208166405604913544419290496386950151e-9,
-3.303926407460910236221106780575663338263e-11,
6.984704095724592727074792733736258494292e-13,
-1.280044046889496797330778711033244226361e-14,
// [6.8571,7.]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001281227370605132739460187839980578734347,
-0.00001499930920846589383183683132533857040012,
9.024296338121728088752060634160888338557e-7,
-3.698286638904261588537468886143520810604e-8,
1.156374236033767555999670546345692689737e-9,
-2.933140502879069273270968276315377264493e-11,
6.272426311807396124763730406893922506779e-13,
-1.160584146646277279028697858719948467289e-14,
// [6.8571,7.]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001049951644592612520931200169246766473241,
-0.00001263401487336005972347366393820514782897,
7.766401941703489906907179226322787382976e-7,
-3.237847902333302630109700971286062462986e-8,
1.026599169953561456918542224324605922333e-9,
-2.634021236683467988925621982059953403087e-11,
5.687056050563215101756999330049585232131e-13,
-1.060842339421752936434053807791189062335e-14,
// [6.8571,7.]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00008843810411352041368481821868051272790918,
-0.0000108729627183753506812028024552958594662,
6.799480594904139861977084024731935427431e-7,
-2.874477714011052057450327537163854990179e-8,
9.219074272334118283871576881002058952505e-10,
-2.388197384781650002541609027090757203702e-11,
5.198306849191630332423610901314567741065e-13,
-9.76421188531145804045613869243264280814e-15,
// [6.8571,7.]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00007611073902862745069201654018123830164462,
-9.519272832856966705299908015996145967972e-6,
6.03640319942712268751750563034193145957e-7,
-2.581340831570263924813060762076529134535e-8,
8.358690794557390706897938712199629551092e-10,
-2.182949834285168605610267363407769281166e-11,
4.784630794972316985375455646240993696817e-13,
-9.041167181613574107938078702885145505344e-15,
// [6.8571,7.]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006663490982999876312539598662633992519923,
-8.450964479189753701388481215861047178751e-6,
5.420815746301213496109760422345176381903e-7,
-2.340433455348600231664895769500001748675e-8,
7.640324371207886612963015185103396139711e-10,
-2.009229355760104416887579370550274911892e-11,
4.430328047962177405415047263346458273944e-13,
-8.415451654244530895398323722929865987306e-15,
// [6.8571,7.]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005915675135432827233106383286008565247764,
-7.589142044814014469906777499014694879222e-6,
4.914910256235496013551439031439095968345e-7,
-2.139290854676176595491836380365838075323e-8,
7.03230269935319612837498018288145335674e-10,
-1.860442693921562345941136199346187061212e-11,
4.123717893824460787903419279582956166116e-13,
-7.868998819629041687550837268044750385243e-15,
// [6.8571,7.]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0000531239943136980979173601480497640768931,
-6.880874358722479820435081755526055626945e-6,
4.492510794823207975154689121306988022566e-7,
-1.969044784677544062834542427542360135894e-8,
6.511549385563672031913658073317735463678e-10,
-1.73168447074245862949857645957298765108e-11,
3.855947539653053505388263966683529693359e-13,
-7.38788878765169804672099932164737267444e-15,
// [6.8571,7.]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004816612051105735555552437641933594033397,
-6.289515112745693211359466965714867353198e-6,
4.13499404782590256828778903797450734206e-7,
-1.823233855149885347503768405849343708969e-8,
6.060895606797992834465746636018842849631e-10,
-1.619236921358157417397362176213033986931e-11,
3.620196068181763994989990178243471175174e-13,
-6.9612373674606167218342171535688555457e-15,
// [6.8571,7.]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004402660578921984945760883577703920050695,
-5.788991666949837526759395283718855063382e-6,
3.828791095817660276126281915836688679246e-7,
-1.697050795607943159823816370094187771098e-8,
5.667329186072810853332204189818484376362e-10,
-1.520235583916066708066730444809584102716e-11,
3.411130088684021109869860335897312315864e-13,
-6.580420553681600325764334073235803884268e-15,
// [6.8571,7.]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004052294166864885981488520766424989110598,
-5.360307534138632196311639811609643858617e-6,
3.563806670779438183512376620257199650665e-7,
-1.586852196469367046743263823808495261569e-8,
5.320824506938812015794150400157443139818e-10,
-1.432440693637943030817217253880887499144e-11,
3.224523727300908789028641098258972581441e-13,
-6.238522940196725613396435495304859283283e-15,
// [6.8571,7.]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003752215273897042263731398112846379587638,
-4.989329339085422613152056446910737450308e-6,
3.332389612588298204875331917214437156969e-7,
-1.489830885106451493928103263318485147687e-8,
5.013542392700599745939987214111913915587e-10,
-1.354077593720692062860795048271518144796e-11,
3.056988343976670525065135494941463000143e-13,
-5.9299387050844802656099800796108869788e-15,
// [6.8571,7.]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0000349253053735979556786826463439236129832,
-4.665345457618099939219417572999945685007e-6,
3.128644858726056997067649629057224804119e-7,
-1.403791892026558386502466302044278194233e-8,
4.739271544570851252282963325090041252975e-10,
-1.283723227441586066017468261081179361279e-11,
2.905777010746160845093750519516421996521e-13,
-5.650078481692822973316895268938777598568e-15,
// [6.8571,7.]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003265741820332669707409237419125734210394,
-4.380102802211211196500663852783662653093e-6,
2.94796297325817305027033626997870262107e-7,
-1.326996053554427666377614452757586985629e-8,
4.493031264039600561784560422556763305346e-10,
-1.22022402722881548845095573223075115721e-11,
2.768640875290222518138613797434886933871e-13,
-5.395150962791682926300369867241355052656e-15,
// [6.8571,7.]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003066071961547847597873575952884556938678,
-4.127148162556401355263692171464843081443e-6,
2.786691712466599009842092238986851499297e-7,
-1.258048774094932716324926215370873774228e-8,
4.270784064621949683450814498785134974072e-10,
-1.162635593535879854533938886825854817081e-11,
2.643722144472623747139656016411423027798e-13,
-5.161998069914740795111409368782859370657e-15,
// [6.8571,7.]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002889003713789480718557342396962791715381,
-3.901368397448406707583677152960799860901e-6,
2.641902425601567937668733404097161370859e-7,
-1.195819557421947733017943028401778450914e-8,
4.069224547919039442165761635796891463464e-10,
-1.110177752647946734211794728979703258814e-11,
2.529473315388826326644909838218648084043e-13,
-4.947969063791929512067423652168404135155e-15,
// [6.8571,7.]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002730957878213884474005499598694280712395,
-3.698663395837555728160273991592834536387e-6,
2.511221070588214760291048463869377188506e-7,
-1.1393828919750398461416316850801721398e-8,
3.885622105940776011675860391138099203828e-10,
-1.062200637367607072023049281464262635892e-11,
2.424595487973460167029225484878061887549e-13,
-4.750823336795313296468986123598167221861e-15,
// [6.8571,7.]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002589064377086288796586845152856903472677,
-3.515709498819039175743481278969036051337e-6,
2.392704073149629691270187105682810499134e-7,
-1.08797420750953635834017997863257007602e-8,
3.717702203506344754486244017110744392319e-10,
-1.018158781153443412274937015767073107347e-11,
2.327990732114199630451056279432022934087e-13,
-4.568654587943720965572553316810111612992e-15,
// [6.8571,7.]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002460996649173327217494412596505374688844,
-3.349785702405184942749668767098655109787e-6,
2.284745835771999413024674497212361131389e-7,
-1.040956634168435400978884434508028692852e-8,
3.563555707729493055387666769503599215471e-10,
-9.775911145467737060509686254006825060559e-12,
2.238724932463087123185773555173770339602e-13,
-4.399831118303446971422890933875966972198e-15,
// [6.8571,7.]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002344849991683629261478530477440702235908,
-3.198644170076655756145407464215786309222e-6,
2.186008931755619436269819200242221485004e-7,
-9.977956147380998783049129591600733833824e-9,
3.421568875512369120604477817789510025658e-10,
-9.401053617708073974642992064370363888159e-12,
2.155998532429356668953000430023975940891e-13,
-4.242948406838749557422235030131791932044e-15,
// [6.8571,7.]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002239050919053658837466987871297342706685,
-3.060412504453866517749777660160214069377e-6,
2.095370790951851368504539697137535966549e-7,
-9.580393011463839816958384644390001584846e-9,
3.290368741642758290454700907127251171723e-10,
-9.053657546317928250503847346984861091115e-12,
2.079123295738142593548038599151259924788e-13,
-4.096791132654181909264251359597904756073e-15,
// [6.8571,7.]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002142288753117706376777856401913197944263,
-2.933519107328724510042528003388717052701e-6,
2.011882532409188582351774782615697696357e-7,
-9.213032631297396500071040288111243361062e-9,
3.168780117448253364722407826387588611036e-10,
-8.730832735228690173875234259525448372315e-12,
2.007503696835075423079484991422266507445e-13,
-3.960302529721053745296881094989919399037e-15,
// [6.8571,7.]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002053463375130106977186188599344408630308,
-2.816635545369121356161800404875716463063e-6,
1.934736852574179772812883521191980830106e-7,
-8.87258447856286975424800846948845472719e-9,
3.055791434309862224654337093586440421338e-10,
-8.43007832327938545063629107578133460398e-12,
1.940621904297967984891334924232954048878e-13,
-3.832559481662859182137557242550849527037e-15,
// [6.8571,7.]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001971644881758384774923237919957383564045,
-2.708631593600226019958353950181389510869e-6,
1.863242740499876801757609078257193786558e-7,
-8.556216161095447448304761963247931548656e-9,
2.950527390825666826691312002327948663881e-10,
-8.149219722186699533302471714161967043588e-12,
1.878025576948836616690951664362550720393e-13,
-3.71275214584244062878754073084255437802e-15,
// [6.8571,7.]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001896042115520158044715088438418174473112,
-2.608539836696311804663326419431674693822e-6,
1.796805393831668828329795336694226422274e-7,
-8.261476834941894303489686303538386925877e-9,
2.852226881100417855367711198974264545415e-10,
-7.886357366506154301343436414327541957298e-12,
1.819317879380967475672490446455741672417e-13,
-3.60016717814039545681719306229607935809e-15,
// [6.8571,7.]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001825977885687418098850648336005983179564,
-2.515527551360924149023023602557194132221e-6,
1.734910135339376183717745288581443930348e-7,
-7.986235403570874246407183547817477028262e-9,
2.760225057232018296205342388760880560017e-10,
-7.639824784055492456935186729079370073467e-12,
1.764149261870784127593409686629687094707e-13,
-3.494173840360260878886277898151117622855e-15,
// [6.8571,7.]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0000176086928595264674447647159181795126485,
-2.428874189471812063275611374455662421173e-6,
1.677109434751418060764017093914547135545e-7,
-7.72863029283465248234961836104481018311e-9,
2.673938653959768028451683323924084103406e-10,
-7.408154075883077009931137187345696697832e-12,
1.712210652814214300316618289102925270455e-13,
-3.394212430708389269509816159829526822214e-15,
// [6.8571,7.]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0000170021193263026828878083716715988802109,
-2.347953208648762904063338547338132900966e-6,
1.62301236149676994557569193008648711956e-7,
-7.487028359983802011960252914577125533526e-9,
2.592853906653354170823615862099461209656e-10,
-7.190047328132741566307664952137679809655e-12,
1.66322778950014530951690703379548314825e-13,
-3.29978459813494459750604935470889657035e-15,
// [6.8571,7.]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001643567246054133881432638671661735103916,
-2.272217306092342786644730240688690055706e-6,
1.572275955598051982820349933258302502496e-7,
-7.259991064036058284313981067098271392364e-9,
2.516516545465591392442766718967302260408e-10,
-6.984352804348141638287050427375036206521e-12,
1.616956472000840147029883349803423326757e-13,
-3.210445193403546816989148192793929312989e-15,
// [6.8571,7.]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001590552114264639803130203036242110620225,
-2.201186337834220303035752738178134148615e-6,
1.524598123448988450153198534607674076412e-7,
-7.046246449403722993662108355884452898745e-9,
2.444523462638981135815183975369093070965e-10,
-6.7900450143956237762690786588861166008e-12,
1.573178570080016838706294645494250016571e-13,
-3.125795380743549877362658052232502729856e-15,
// [6.8571,7.]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001540830436483954068300347491702743683997,
-2.134437372825609791919422054268984363383e-6,
1.479711754376162553214310318940061262412e-7,
-6.84466581435180143696504565916897380125e-9,
2.376515736628746735299069047300481881181e-10,
-6.606207945671450840942898678606173053695e-12,
1.531698647803513167118793536517355282405e-13,
-3.045476789051287277532148043046802806183e-15,
// [6.8571,7.]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001494106160977926714035312387829652743211,
-2.071596456123728056512180792722595961579e-6,
1.437379821015225268724984374860733020439e-7,
-6.654244178542306034425741892538803271747e-9,
2.312172763025388735803112786657364698571e-10,
-6.432020888336790230919855789172076393898e-12,
1.492341097543181778097731430172915848355e-13,
-2.969166524682358196812765532853760461325e-15,
// [6.8571,7.]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001450117519286609502599402740658447190074,
-2.012331749418486749068359124487429815465e-6,
1.397391277495199082093395195872657783287e-7,
-6.474083849617244438414997773819898972487e-9,
2.251207293386953524143277743342764121467e-10,
-6.266746399735202108223885221960848089396e-12,
1.454947696172304865887762533736577898863e-13,
-2.896572901759934050479134246035566983812e-15
}
,{
// [7.,7.1429]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.3332098019671750641489334417702775109827,
-0.003357179622318827357011431314613271050725,
0.0000502537898348395318312790018914178048752,
-8.168534893650761913774430860146119575291e-7,
1.339754686137654857400018331863190414928e-8,
-2.138265486090415032208924518996555765321e-10,
3.251366811917731943986253828270121714533e-12,
-4.65273266535310543547217176725575419113e-14,
// [7.,7.1429]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02350025735623179148577565832679836899976,
-0.0007035530576877193806336047352698319495342,
0.00001715392327666787724265145409696775297291,
-3.751313134811522256334498601920607042142e-7,
7.483929184286762294982344505382485345599e-9,
-1.365443249997966120359051915317166580643e-10,
2.279893501118756817724115503656612197278e-12,
-3.488304130721414580953948369886102268243e-14,
// [7.,7.1429]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004924871403814035653568259323964249353217,
-0.0002401549258733234597226035457011989625852,
7.87775758310523997215292200595082428475e-6,
-2.095500182329036282361092670597085194614e-7,
4.779051361083061840218305730441022586896e-9,
-9.574522740712022234580099263258065065238e-11,
1.709313535550623213061073165072669052171e-12,
-2.746579534499710007409698326250033978078e-14,
// [7.,7.1429]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001681084481113264208929018983555539331174,
-0.0001102886061634514517196433838634017171931,
4.400550382891852600532053273536601422408e-6,
-1.338134389866476232534793662337295060065e-7,
3.351082947563742957350141808144537987488e-9,
-7.178275576282568864491915298996755726519e-11,
1.345861365457381018742690799616379680539e-12,
-2.243402392516797938678688756147863950536e-14,
// [7.,7.1429]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0007720202431441601541976536154678833505781,
-0.00006160770536046753186619561591986038881718,
2.810082218720352717310444299699235578114e-6,
-9.38303232679719342101808475538124951437e-8,
2.512396441663818893287248548522758716328e-9,
-5.651910991766814150726478162758906222317e-11,
1.099299284647595314884742864652367280054e-12,
-1.884655093162011150439564722205355508582e-14,
// [7.,7.1429]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0004312539375232727162117101769024154513733,
-0.00003934115106206913284938663729878548532841,
1.970436788628068380722698820037429801474e-6,
-7.034710099879950635811431716102289763313e-8,
1.978168838348196578007663616391495106967e-9,
-4.616450068335894494966349760503525813573e-11,
9.235090603039311220737538157409049478494e-13,
-1.618478414042321907209655819835654468414e-14,
// [7.,7.1429]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000275388057434483923871948530167600258725,
-0.00002758611504077914433577980876934758118204,
1.477289120975372717027870557173039475622e-6,
-5.538872802627363572118252422950567815833e-8,
1.615757516143094337995687703501818967342e-9,
-3.878207627321087794100143949665396194525e-11,
7.930793012275578408982023405192385661791e-13,
-1.414474527141083807637338906543479929961e-14,
// [7.,7.1429]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001931028052854540049028864287064042028554,
-0.00002068204769364297329754816556815319133899,
1.163163288552269319586968369647994931176e-6,
-4.52412109418002222481623515874467337721e-8,
1.357372662589434770256410798198366027621e-9,
-3.330462860792029204229929674918820215863e-11,
6.931148317689958892232410003590551291239e-13,
-1.25388311498822976021170329377655661832e-14,
// [7.,7.1429]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001447743338555008081489121546065268445545,
-0.00001628428603972078812763550131002895382815,
9.500654297782783263182203404015458218527e-7,
-3.800643499180164142888902990364867432784e-8,
1.165661994961737012725690578822736232173e-9,
-2.910660565553691079733823337260225366106e-11,
6.144229358978634724771104844374356618038e-13,
-1.124612063773068295343065941925626091382e-14,
// [7.,7.1429]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001139900022780455123876818216476320824102,
-0.00001330091601688594973797496196982259918843,
7.981351348282670257427846906355614299595e-7,
-3.263853625680517064939283134122762801627e-8,
1.018731192176364990001331702346433066052e-9,
-2.580194367176436205989485821828205797186e-11,
5.510783670512673872425657449621911295663e-13,
-1.018573626863086328440914965638375116668e-14,
// [7.,7.1429]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00009310641211820164402192514258926490373007,
-0.00001117389188758665469989952689887538534147,
6.85409261393306399983199473518533361271e-7,
-2.852447374428770716264820079188445775137e-8,
9.030680232075188444106717546176800662111e-10,
-2.31418032414418921397074488873157905467e-11,
4.991180507450419589252155732775118423906e-13,
-9.301836624444021756464382889092693409835e-15,
// [7.,7.1429]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0000782172432131065790649295518507646676378,
-9.595729659497935465763863572322612421768e-6,
5.990139486304099515159843290179129924361e-7,
-2.528590498397874109805838926685920548488e-8,
8.099631085424367109524910624839913505642e-10,
-2.09597500981806166210672212018261952822e-11,
4.558057003238653852040611534024140736351e-13,
-8.554789747908801897186994410783516526669e-15,
// [7.,7.1429]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006717010761648554469354075456899998608939,
-8.386195280818009206773701819249173953205e-6,
5.310040046638959782041011606381088202443e-7,
-2.267896734839547258547605410319969487104e-8,
7.335912488707723886808567702646006544022e-10,
-1.914087100698660133734559536970311283518e-11,
4.191993074346623814546207424654324707244e-13,
-7.915783483682942298681549901344698584515e-15,
// [7.,7.1429]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005870336696572606111403880955652502818059,
-7.434056065287352984939478518236193408647e-6,
4.762583143166249301366343188385475430626e-7,
-2.054055525601252601513236837084102785088e-8,
6.699304809777733736813519276868978402003e-10,
-1.760360963960860061052384679443666513132e-11,
3.878870443530032772419347118870451012335e-13,
-7.363424231843342181477439511874005175571e-15,
// [7.,7.1429]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005203839245701146776652740801361912800651,
-6.667616400426028906890395876743499591225e-6,
4.313516603765633405655920503859354503067e-7,
-1.87580537361846122298760412710812649809e-8,
6.161263333823653217637941310205537310797e-10,
-1.628867530094753169656841508855582417599e-11,
3.608205999810062156621194774728687250166e-13,
-6.881527074569429523602029069252913998389e-15,
// [7.,7.1429]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004667331480298219940211936781122238742258,
-6.038923245265580595944667014394978193413e-6,
3.939191284601596851825043881697680147119e-7,
-1.725153758695533500804415674086114029521e-8,
5.701036317621071274105312670083365564926e-10,
-1.515204359353144109361746045021105901168e-11,
3.372068940597051501273177326106167125162e-13,
-6.457642271302766795545184055006959841714e-15,
// [7.,7.1429]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004227246271685906138776774648240616313111,
-5.514867798436296203950112837497477207241e-6,
3.622822893263292856767986815385044892692e-7,
-1.596290192691665435217884584666764132605e-8,
5.303215222102491708323150897582977855461e-10,
-1.416040879151384381841714025040186664717e-11,
3.164358740417048102697027111164126252519e-13,
-6.082049712707873074604406235711612058386e-15,
// [7.,7.1429]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003860407458905407078939099440929642926774,
-5.07195205056299774532112111216401760295e-6,
3.352209404655030156651723933651093997127e-7,
-1.484900284637914847929479333622892943862e-8,
4.956143043259836422429920276767832001727e-10,
-1.328815157206782510690930546236148502309e-11,
2.980312423482210547109429882585576198875e-13,
-5.747058272059901099683128001339511948348e-15,
// [7.,7.1429]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003550366435394098171030570815504036555912,
-4.69309316651172346585675782438376457702e-6,
3.118290597742027857793351413138614303141e-7,
-1.387720073387959052275172740383869335121e-8,
4.65085301813460873477622942707502026467e-10,
-1.25152697467332159312293283966763364145e-11,
2.816161238742054323270920003214511923975e-13,
-5.44650833338440457868359814279996838146e-15,
// [7.,7.1429]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003285165216558206187308107428336073687076,
-4.365606836833784984834296574806506550239e-6,
2.914212154117006421495184242746261809719e-7,
-1.302238865293937113594202635474438392511e-8,
4.380344380791038922403114744637200423733e-10,
-1.182593643136525444154239183074748118735e-11,
2.668886893442774109313870350735441233393e-13,
-5.175412501384697241969210548468627425264e-15,
// [7.,7.1429]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003055924785783649261428855955596271096142,
-4.079897015758994931150232523816458175982e-6,
2.734701617119456319633793088390429387489e-7,
-1.226496445877901107884301187373255310842e-8,
4.139077721799331532923692311804129257739e-10,
-1.120747632591545705221706282557856199129e-11,
2.536045497101061356398669632527101341449e-13,
-4.929692049497010480307618802055248616275e-15,
// [7.,7.1429]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002855927911031296233756060388900159167311,
-3.82858226396264325263404927209145787007e-6,
2.575642536345685609104224111243627743e-7,
-1.158941780486359408298298578773927635753e-8,
3.922616686159886690049802146393025135281e-10,
-1.064962635265417162666601260401052262626e-11,
2.415638418118566359046516737550636545816e-13,
-4.705980832639643422936523220869673935944e-15,
// [7.,7.1429]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002680007584773850067884160362057926993831,
-3.605899550879563964606716908731464727795e-6,
2.433777739023360781032184559702517815332e-7,
-1.098332689708481300960446461642672540772e-8,
3.727369196681892785509492672908105895634e-10,
-1.014399330937710803810147591885028193006e-11,
2.306016198791931561122088267869498734887e-13,
-4.501477484695842441832696951735208680011e-15,
// [7.,7.1429]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002524129685615694574633952221472463832391,
-3.407288834628492448882099055591801163508e-6,
2.30649864838973641354878592017206706036e-7,
-1.043663391921662518688575657762678771901e-8,
3.550397632606151370275826397918680646299e-10,
-9.683650354727967526270787692918831905778e-12,
2.205806130353976174238645237396774656859e-13,
-4.313832670221767911496509519616764880806e-15,
// [7.,7.1429]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002385102184239944521356856324349451813615,
-3.229098107741587052574552036367871397479e-6,
2.191693123037342761096236148502576045911e-7,
-9.941113533055765166375631380969176135521e-9,
3.389277599468417574855856815851641445858e-10,
-9.262832855982164749070083855737509167844e-12,
2.113857004966132562789951321389282415725e-13,
-4.141062122203611865186542943412991783379e-15,
// [7.,7.1429]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002260368675419110751102818433488861346175,
-3.068370372248391779299602555229139045888e-6,
2.087633841943493408492055331439856290507e-7,
-9.489977434036450975068474676127462032977e-9,
3.241991475824033322285333512696344944682e-10,
-8.876706372825642953856396491258828837888e-12,
2.029196503161848562512078919866075557294e-13,
-3.981478880304828297722483522525104553355e-15,
// [7.,7.1429]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002147859260573874066462833081425978430246,
-2.922687378717147056916643532010604618899e-6,
1.992895261149373564194489219076931797314e-7,
-9.077576282057274986917875415849599736515e-9,
3.106847207570774910261839927533616481069e-10,
-8.521187704605646420432114348116309220041e-12,
1.950997989748458116788150868520853049771e-13,
-3.833639988085704493511945923350979728547e-15,
// [7.,7.1429]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002045881165102002766990470069478805318982,
-2.790053365605513389408253409992303730052e-6,
1.906291019233687127581924461061534300602e-7,
-8.699172325583524832620636711805337687344e-9,
2.982415674486833212508758194485143003618e-10,
-8.192805448980918621472160097227967099018e-12,
1.878554394773261985688542604567082599267e-13,
-3.696304193297153368118851652957010674432e-15,
// [7.,7.1429]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001953037355923859205518719778008408506002,
-2.668807426923677284122069638696290459334e-6,
1.826826188374144067318251824276812507782e-7,
-8.350764027952205927162847089663005857204e-9,
2.867481885758552225703005039363746693579e-10,
-7.888590314670844378293539726089960538199e-12,
1.811257486126074669463363305435226386582e-13,
-3.568398103500892927591478814561701460576e-15,
// [7.,7.1429]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001868165198846573937230377980604829947714,
-2.557556663720433608129760416925698399118e-6,
1.753660445871515141833089185371808602741e-7,
-8.028949414848643495667880695213344709616e-9,
2.761006589442766217008779350080585187005e-10,
-7.605988076618885145298545538418189026127e-12,
1.748581285353454011736784724451313061666e-13,
-3.448988898681984845146291384510056088155e-15,
// [7.,7.1429]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001790289664604303369110203586920443569642,
-2.455124624216862218538667270226767267804e-6,
1.686079377119718316325610341146174022079e-7,
-7.730818580800161007499829882106729678451e-9,
2.662095806774114397248498279113698065951e-10,
-7.342789930721938621229058088553950529541e-12,
1.690068696479408963913678502679774412367e-13,
-3.337262172257686394356362601057799035411e-15,
// [7.,7.1429]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001718587236951803401163790395364245410008,
-2.360511127964448963988077455735779604407e-6,
1.623471901969491226995532854668256157384e-7,
-7.453868385235853619879396904741867402576e-9,
2.569976456320409194836517280329524371829e-10,
-7.097076341670698073492818661170011837615e-12,
1.635320647804024576998351197221009052831e-13,
-3.232503815321277020231948740020162184963e-15,
// [7.,7.1429]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001652357789575114127465555270721858827773,
-2.272860662754227149127133325430743751345e-6,
1.565312360900943598466487573674085723989e-7,
-7.195934200121037159865029421234350889024e-9,
2.483976700726838499048406552146668889168e-10,
-6.867171444092879083471637652350489087537e-12,
1.583987214938201409956965995043669280586e-13,
-3.134085112588816839664731803270095326366e-15,
// [7.,7.1429]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001591002463927958861293771968938318509005,
-2.191437305258350931053071093671539659858e-6,
1.511146182026791525228240751385585051278e-7,
-6.955134880840532581320837372878451852938e-9,
2.403509987116158671919060161689817055651e-10,
-6.65160576392465854956651466398813502428e-12,
1.535760317613883533864469459282262757436e-13,
-3.041450407602603919530027449600605884933e-15,
// [7.,7.1429]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001534006113680845512637754325897518696695,
-2.115604654834623323355337022965960845368e-6,
1.460578324977847203605845727884587873267e-7,
-6.72982807931880539522392158013264204405e-9,
2.328061999568751454251602900088332268309e-10,
-6.449085548863279087279734798035264661548e-12,
1.490367675463336499451156405056293162192e-13,
-2.954106836972302599625656667071281314103e-15,
// [7.,7.1429]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0000148092325838423619102908742924462296233,
-2.044809654966181829260024914153325220637e-6,
1.413263896658248208765686827354846626271e-7,
-6.518573710963789499119372238901570719292e-9,
2.257179924781080285692581589482400837957e-10,
-6.25846738586090237715460619116063051074e-12,
1.447567777654003828415398246876231918651e-13,
-2.871615741317094348640380552896861778446e-15,
// [7.,7.1429]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001431366758476327148743029425162760848338,
-1.978569455318819436491170774355409100075e-6,
1.368900479303660496063606121704968523287e-7,
-6.320103898510282757995713030821155765906e-9,
2.190463568188576808911667192687304878901e-10,
-6.078737076299326487594449526481654152141e-12,
1.407145674129097047243271978283114139402e-13,
-2.793585443054391110465145309216941306996e-15,
// [7.,7.1429]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001384998618723173477201668615341896912708,
-1.916460671022468825116712866931107451278e-6,
1.327221818688391470620774241289746237224e-7,
-6.133298097163790870974652222058382954567e-9,
2.127557960276824038969120115288454931939e-10,
-5.908991961488587090891570895628340197734e-12,
1.368909436621324634513103663190335451998e-13,
-2.719665144717456601650867078929678342415e-15,
// [7.,7.1429]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001341522469715728052466309266120757463182,
-1.858110546161160670011448245302769802715e-6,
1.287992600405597197276297292222252649892e-7,
-5.957162392271637974580588262639612624297e-9,
2.068147170506094152206747041897313478433e-10,
-5.748426060866429618213150668300310312538e-12,
1.332687168740963808766557576291468933772e-13,
-2.649539750776029260940834210732211860064e-15,
// [7.,7.1429]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001300677382312812346961724114234939413181,
-1.803189640565313739100384063905246072836e-6,
1.251004102378215625517657622464641246662e-7,
-5.790812178311500062313080414463631540487e-9,
2.011949105681187155958753230804939054259e-10,
-5.596317516023156307599158019719641320743e-12,
1.2983244685930361545483531201442222621e-13,
-2.582925454432893149190970896763845481e-15,
// [7.,7.1429]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001262232748395719498246657428279408357205,
-1.751405743327041411949277492211281764713e-6,
1.21607055744655860608234566686342467339e-7,
-5.633457594326805741839686520679729749967e-9,
1.958711115360147469896366554527316568883e-10,
-5.452017935109943546720055387755700864563e-12,
1.265682266243260706882707768722783109995e-13,
-2.519565961131429011033073887081588188043e-15
}
,{
// [7.1429,7.2857]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.3299020724681272167644343736334047673741,
-0.003259070062876772372648283095955385017373,
0.00004788152418871434792036362615880438086425,
-7.653381401931892423675450214133082466265e-7,
1.23755942685448061643070647264848322514e-8,
-1.95263287107011004557797696546354688381e-10,
2.942474377911244634971075127397237145057e-12,
-4.181371656973681298255984884964578781203e-14,
// [7.1429,7.2857]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02281349044013740659669658396291043548611,
-0.0006703413386419704226740257869870396145524,
0.00001607210094405811086839355804163735653553,
-3.465166407371907267974354814232440041653e-7,
6.834215033588300197217681667671653805189e-9,
-1.235722316358375153945889267706621570261e-10,
2.048920614665699855291893693726292380841e-12,
-3.117939116815114441910618839959274230777e-14,
// [7.1429,7.2857]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004692389370493792949078606999233646241246,
-0.0002250094132167896798263090139938372214263,
7.276849455481930665654835030614842506461e-6,
-1.913580218953721145270976986306691358777e-7,
4.32502809491557677530702908162807718719e-9,
-8.604549873740250497439564374365492180717e-11,
1.527829651261186701807477236120899880699e-12,
-2.44456215813589665473683477979406836295e-14,
// [7.1429,7.2857]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001575065892517527750707880103278840466969,
-0.0001018758923767275958760597087576675114834,
4.018518459803589731596726752777412289349e-6,
-1.211007874349794103447814390911684946305e-7,
3.011592445471373756754533153682801361478e-9,
-6.416138282223511699135097258477834076042e-11,
1.197868538228472672494471901619221485111e-12,
-1.990014945018036989123618490222899597994e-14,
// [7.1429,7.2857]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0007131312466370931642119628015562546987861,
-0.00005625925843723397440095352693485794491311,
2.543116536135231983909953375648835090762e-6,
-8.432458912447695096925989759747227764032e-8,
2.245648389919983040287485653718935080794e-9,
-5.030422630122176674915547877981283454895e-11,
9.751356694969726029442936825497436043092e-13,
-1.667287054174744074680375370561851196391e-14,
// [7.1429,7.2857]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000393814809060637814769429041189950327525,
-0.00003560363150587929608855826898473649663902,
1.770816371614595548600794328252145542029e-6,
-6.287815547583124734104964946262196236293e-8,
1.760647912815030571383647114186888403144e-9,
-4.095034060297964432156063152644626547986e-11,
8.169953853311414368130669131795788055937e-13,
-1.428676116293148342121573276011767882805e-14,
// [7.1429,7.2857]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002492254205411550672761477212488077631285,
-0.00002479142920259216654967066378717584894482,
1.320441264992969197547935989816768255308e-6,
-4.929814204566991498552225127559605183978e-8,
1.433261914264223122859116055198546339226e-9,
-3.430913240843356917813797159012418367322e-11,
7.000731852309510983529912537556725782143e-13,
-1.246344794724093668324368939603136755354e-14,
// [7.1429,7.2857]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000173540004418145161060940377335772161627,
-0.00001848617770989079570581550542066318369503,
1.035260982959527745304965106168633790084e-6,
-4.013133403032410621260898484154734575094e-8,
1.200819628168082541204645434423096767032e-9,
-2.939893686928249015973644233877803642496e-11,
6.107285561479547095074920758962148219925e-13,
-1.103180334810438457483345134445589889185e-14,
// [7.1429,7.2857]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.000129403243969235565609982669996994664194,
-0.0000144936537614237383017430524489708771518,
8.427580146372219821690147845004177923784e-7,
-3.362294997471477316676262740537703477091e-8,
1.028962784881514931891456171411264813434e-9,
-2.564689365674854414919577406993377110975e-11,
5.405761028827211660598578174356362301822e-13,
-9.881909108713969280646007178197064759679e-15,
// [7.1429,7.2857]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001014555763299661641609709520801030595297,
-0.00001179861220491237697400811914623936114939,
7.06081949469389556490487929271882885541e-7,
-2.881095832591637715848297931052877754532e-8,
8.976412729285842497866623777189602403051e-10,
-2.270084365647680081030847784583293735906e-11,
4.842297307262214943327554709383544208884e-13,
-8.940474337159539781489153416246744998989e-15,
// [7.1429,7.2857]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00008259028543438663518709485554227161736437,
-9.885147292563488079828954908087259837253e-6,
6.050301248445924944152271896476378034876e-7,
-2.513395596063148637712616849678590736396e-8,
7.945295233290196365852609650094394439774e-10,
-2.033458981448876643024835520974006918123e-11,
4.38098115089318851933781997403243045233e-13,
-8.157035442421432472016238595307572990602e-15,
// [7.1429,7.2857]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006919603104794441320143542090367941652884,
-8.470421747816974874090911895827930743556e-6,
5.278130751735835227995053647839048470814e-7,
-2.224682694601695560611494939777338316755e-8,
7.117106392096421130287883178392983038196e-10,
-1.83973098954756883150259331106878634443e-11,
3.997084885934207848484281529078649095399e-13,
-7.495865840460399286646159190937670746707e-15,
// [7.1429,7.2857]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005929295223471882099749246566617751964455,
-7.389383052423400840287887472703830346205e-6,
4.671833658666556990663699991694904030514e-7,
-1.992789816861146756334949744515117939935e-8,
6.43905842346552409918583463206837081268e-10,
-1.678515738770893333732551039508423373924e-11,
3.673102105179126032596162237964951039503e-13,
-6.931050336768887886507750260323929651878e-15,
// [7.1429,7.2857]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005172568136696380296674798761622008447694,
-6.540567122126887535933971991465796589158e-6,
4.184858615411206859166246958301055597292e-7,
-1.80293638373956947386184538132815954218e-8,
5.874805048382400545252278662183875865194e-10,
-1.542461258237084229302925192322379710891e-11,
3.396334075584867693877239965701265571973e-13,
-6.443385016982545582003197897581212873193e-15,
// [7.1429,7.2857]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004578396985488821001719523048944616911011,
-5.858802061569812400706594609389731381548e-6,
3.786166405855720877512998150984074475255e-7,
-1.644945437056086588398554248996841636662e-8,
5.398614368829921165823276442147821420952e-10,
-1.426234623889094113921503261968266701356e-11,
3.157370658148044270887666540577109056923e-13,
-6.018367945574437168754023858584482585518e-15,
// [7.1429,7.2857]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004101161443098868423078168268997850994739,
-5.300632968192496771786702567014925409051e-6,
3.454385417820253046357610879618658201762e-7,
-1.511612045322399363268950190066429491122e-8,
4.991821150662250560555805792522752210736e-10,
-1.325883994971507714737967497989326016331e-11,
2.949105732202558057570905639333154224146e-13,
-5.644862396592381600388778912059826796126e-15,
// [7.1429,7.2857]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003710443077734747497109701663756657069708,
-4.836139584943164728369934440592147832904e-6,
3.174385295179372828548189297963943899563e-7,
-1.397709942943760551667635804571004355351e-8,
4.640593951277970463934922477358698229875e-10,
-1.238425126373938081467235998789766863374e-11,
2.766082165919029871505919710691858270749e-13,
-5.314186494139217489639283812020948174539e-15,
// [7.1429,7.2857]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003385297709460215079516867152760893602643,
-4.444139413246220217715317693212267301178e-6,
2.935190880184108454133229640345483976141e-7,
-1.299366325964975780554393096245436118163e-8,
4.334487912824748763745389600213761317912e-10,
-1.161566279982830048962304425659605097596e-11,
2.604045731236599716789935711380586095207e-13,
-5.019480089675122458739409918927900436428e-15,
// [7.1429,7.2857]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003110897589272353933595947361954965486828,
-4.109267232253108120372541289688512618716e-6,
2.728669284528549676071558574673006177168e-7,
-1.213656634165957990845175436946231037812e-8,
4.065481951932657869314520058398110455174e-10,
-1.093520885780571111221865591301724763108e-11,
2.45963486732105760237953535474045273117e-13,
-4.755256020775492155792883168754335147308e-15,
// [7.1429,7.2857]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002876487062577175475907334437707232412479,
-3.820136998335558424159304663028756361012e-6,
2.548678931750511984425216384881248383893e-7,
-1.138334964185792639955544603214615532643e-8,
3.827323073562532216063687435379991239848e-10,
-1.032877254631779455238518575461933480212e-11,
2.330160792653972481531931517078409052989e-13,
-4.517076540350050985393680624572438717522e-15,
// [7.1429,7.2857]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002674095898834890698068203714690638160916,
-3.568150504446516355468407282010873522175e-6,
2.390503424792073449818245508805240504792e-7,
-1.071650477399352049726077484273982305279e-8,
3.615070365759068151878484096493445978852e-10,
-9.785062342480031402875118049295594330128e-12,
2.213448951856394578318868157110914504467e-13,
-4.301316322282683361824427137517033341548e-15,
// [7.1429,7.2857]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0000249770535311256125867389716444296520637,
-3.346704794704894132063026944637772088681e-6,
2.250466002540464792440622783795959334531e-7,
-1.012219718447475558795068640797321491399e-8,
3.424771795528092925255366840169299531199e-10,
-9.294946234567319071185808325255276332489e-12,
2.107722885822415086260331954050737496495e-13,
-4.104986379438825019442531467686116485571e-15,
// [7.1429,7.2857]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002342693356293425710259510208169316493892,
-3.150652403552817189124912897839511926098e-6,
2.12566140874144765506750065370258490515e-7,
-9.589361180820870655125832971487408845537e-9,
3.253231158778732197901558853779227467455e-10,
-8.850964026281397114067073953730406932327e-12,
2.011517949539007752634639528153611996601e-13,
-3.9256015126989709529420651125050015663e-15,
// [7.1429,7.2857]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002205456682486971857537395806529052220106,
-2.975925972234353860104459080519428486021e-6,
2.013765847974061406997092584482171406598e-7,
-9.109047391496074365793151420740016741791e-9,
3.097837386817491334517447575028230136936e-10,
-8.446964989465841448991605971285263778859e-12,
1.92361636056869292416238783817990490916e-13,
-3.761079319580421978982641356770476949859e-15,
// [7.1429,7.2857]: m=24 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002083148180564047533995149377031491618079,
-2.8192721871601609782150981506332354478e-6,
1.912899952215789174029158987329186518105e-7,
-8.673944824089933409235950549198776583437e-9,
2.956437724798878885831180810357128124833e-10,
-8.077835096910388538594147715775889493128e-12,
1.842997712072562762158970698803823196413e-13,
-3.609662388318609587971305397000131212926e-15,
// [7.1429,7.2857]: m=25 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001973490531012112522943989981342796341699,
-2.678059933098716377545676767093966944501e-6,
1.821528413060439367475108046238446135896e-7,
-8.278025764976752535089158794085576421338e-9,
2.827242263207215142036266986453888084975e-10,
-7.73928919975859332335228174723757568052e-12,
1.768800846965844603542298071564027966124e-13,
-3.469857735150100933174614499431804910672e-15,
// [7.1429,7.2857]: m=26 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0000187464195316910130829925079067441788466,
-2.550139778281353080150955543054153790546e-6,
1.738385410646615474607158902015181852432e-7,
-7.916278467462779436923899608627334417435e-9,
2.708751199949546364938503539492159067839e-10,
-7.427710916808304137999996874567661010972e-12,
1.700294181437914019709029712036754982712e-13,
-3.340389211879970514387742749141257884865e-15,
// [7.1429,7.2857]: m=27 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001785097844796947005545190178324223033927,
-2.433739574902117039528348497247628642933e-6,
1.662418478168629070184246983101987434135e-7,
-7.584503485644890676402658831692199791607e-9,
2.599698801610998033865962704199594077259e-10,
-7.140028007445466044403225484887212684329e-12,
1.63685238406182550552174446034805280466e-13,
-3.220159773104379671388318326334444880556e-15,
// [7.1429,7.2857]: m=28 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001703617702431481782169886404608743712893,
-2.327385869433045386210562254603434433191e-6,
1.592745731986823849224703393605705886596e-7,
-7.279156765924402509316997222796155646978e-9,
2.499009783981756677932975112637928285372e-10,
-6.873614435215518724706645810675742288472e-12,
1.577937886251245265795825895680408005997e-13,
-3.108221312466766450610756769639649276563e-15,
// [7.1429,7.2857]: m=29 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001629170108603131629581040158055224821567,
-2.229844024778620097377129019111978329105e-6,
1.528622920845475891305171524680798636177e-7,
-6.997227512481670818411570172392073798383e-9,
2.405765034307181392300747466871041508519e-10,
-6.626212720854850278531045193098187182166e-12,
1.523086101634771931078827471897541273725e-13,
-3.003750363018835493836397469549406067453e-15,
// [7.1429,7.2857]: m=30 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001560890817345033931834827885920013049188,
-2.140072089180828386127779512812695909598e-6,
1.469417777622459638963226349283700744839e-7,
-6.736142209575633158194813283509309415414e-9,
2.319174434848912539763927351881773714812e-10,
-6.395871871056903975243975522722707720708e-12,
1.471893518913228883191472754093568850105e-13,
-2.90602837993201010357173668021168487721e-15,
// [7.1429,7.2857]: m=31 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001498050462426579738127904321585924009352,
-2.057184888668695086975988382031431777727e-6,
1.414589864012151720949480816892420633667e-7,
-6.493688527514283920001325010066662082892e-9,
2.238555137953091222742844651433478738062e-10,
-6.180897374513147232586939318399795791577e-12,
1.424008040125527516066559428461074232799e-13,
-2.814425632869128735707546010904613721267e-15,
// [7.1429,7.2857]: m=32 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001440029422068086432643414804951379649703,
-1.980425809614348021334000601557663197393e-6,
1.36367459077923073179751985090644455429e-7,
-6.267954492845171566941987031687727848223e-9,
2.163314064664766334852072007581442041576e-10,
-5.979810627595671535195931025739463197975e-12,
1.37912108769348920972028628878081903304e-13,
-2.728387963343483160981225223008516380256e-15,
// [7.1429,7.2857]: m=33 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001386298066730043490390955030378509777212,
-1.909144427088337699606378836621207191317e-6,
1.316270443498681646927264138347520618458e-7,
-6.057279484475310549753557600665286315744e-9,
2.092933703716861014673867408436355098243e-10,
-5.79131578805261762339065832740410104279e-12,
1.33696111534703040281632279667933840845e-13,
-2.647425832221773303740007274886181499266e-15,
// [7.1429,7.2857]: m=34 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001336401098961836268672423171876516593318,
-1.842778620895643511230643821735757233261e-6,
1.27202869174097732143124088326285459815e-7,
-5.860214470839931537640782989548156682879e-9,
2.026960510323618690535642921585728555461e-10,
-5.614272524310030019711700322165906245104e-12,
1.297288241249478370979007279839885163708e-13,
-2.571105210154609292256272939092652006107e-15,
// [7.1429,7.2857]: m=35 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001289945034626950340110778225407244302428,
-1.780840168434927830416548698900501046323e-6,
1.230645038877516035574674545201231894868e-7,
-5.675489526523832801747381025463165149119e-9,
1.964995368436291854932395863119979732934e-10,
-5.447673477454599000205592286130852411553e-12,
1.259889784181508590328672590940123367992e-13,
-2.499039960442437535579535116124131576903e-15,
// [7.1429,7.2857]: m=36 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001246588117904449366667793871644673336703,
-1.722903054426148586097380880879974878436e-6,
1.191852800571105282810853075790887749222e-7,
-5.501987126577058734769023518744002497163e-9,
1.906685702437135268431465610294867717201e-10,
-5.290625515606677082198087930050441807692e-12,
1.224576531038203679509429489355766545448e-13,
-2.430885437735053227256767817263449500293e-15,
// [7.1429,7.2857]: m=37 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001206032138098303898610191863096894966503,
-1.66859392079723657042990451063405978392e-6,
1.15541729658225425676785009243540721277e-7,
-5.338720059257869683184093338069706355062e-9,
1.851718916169990157187041269789614701644e-10,
-5.142334059434867811427381969136496382714e-12,
1.191179600100848632437698249105938523935e-13,
-2.366333082836535747364026766352198971843e-15,
// [7.1429,7.2857]: m=38 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001168015744558065490459790926386573317181,
-1.617584215212904925052975988049978503633e-6,
1.121131212445197514200841608340416665084e-7,
-5.184813055318198610820492225093592006746e-9,
1.799816906870414503728729830885449098792e-10,
-5.002089909616832972857979135077602487901e-12,
1.159547792413860099112859501766012080624e-13,
-2.305105837976846257987174768780876961006e-15,
// [7.1429,7.2857]: m=39 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001132308950649033341374699441204012975661,
-1.569583697421082273034682520450036128085e-6,
1.08881074161784087278775069884894602153e-7,
-5.039487427007863356147008313613949695041e-9,
1.750731454776986282599388293140714221213e-10,
-4.869258124086949083206562426772851641739e-12,
1.129545345201591164134629943819382469717e-13,
-2.246954241322243076732988991044860125198e-15,
// [7.1429,7.2857]: m=40 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001098708588194757487512451362368016003594,
-1.524335038262836979048349992462032424438e-6,
1.058292359672645983826636544630381675433e-7,
-4.902048158986085206887874227072912274268e-9,
1.704240330168001016760431377087083752191e-10,
-4.743268583639551236424254357345456820936e-12,
1.101050018121525913344361102989639263099e-13,
-2.191653086526869197725087441628685134391e-15
}
,{
// [7.2857,7.4286]: m=0 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.3266901307745309234230823249669247093574,
-0.003165554485494221105674203178090735986941,
0.00004565785397139757157334753676448477308604,
-7.177309846561798199791044704494201460718e-7,
1.144198553047717320741698633300878973462e-8,
-1.784578029487886095154120120475376546109e-10,
2.664796002178579936698084753131285989053e-12,
-3.759957569465307926214195314910983888002e-14,
// [7.2857,7.4286]: m=1 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.02215888139845954772917595595031328401298,
-0.0006392099555995387735236708731474178452756,
0.00001507235067778078839630334665722685046613,
-3.203755959425078204133939491249074220622e-7,
6.246023089711879615806797297565393605165e-9,
-1.119109762367882781607453259192660139092e-10,
1.84242239541538289728631621982743765509e-12,
-2.788236265010417782007323217163969757871e-14,
// [7.2857,7.4286]: m=2 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.004474469689196771406112103564603380859376,
-0.0002110129094889097818552238956526684912463,
6.727887514793485376966655826080167205764e-6,
-1.748886473621660675022692886995069203153e-7,
3.916884157338918723692582711830730048516e-9,
-7.737357832984761593297191415715432067149e-11,
1.366270805660384336876119802149266853513e-12,
-2.176614329359562499061839108624477693636e-14,
// [7.2857,7.4286]: m=3 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.001477090366422368465839831790879747038108,
-0.00009419042520709155117941034676238067377349,
3.672661594606173507114959998483944483687e-6,
-1.096727570952585373488203616865675116121e-7,
2.708075232396781969278323053435393086013e-9,
-5.737675202584605484025197080186442752977e-11,
1.066570294667409358243919526659878289857e-12,
-1.76582246298840636623242674108598877743e-14,
// [7.2857,7.4286]: m=4 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0006593329764496408521456054540348994076902,
-0.00005141726232447202123251537559561912581663,
2.303127899001015872571834569141388842555e-6,
-7.582610708342883539179630109893600263828e-8,
2.008186313083414360468685644505763253648e-9,
-4.479041968696309281985444416042458345917e-11,
8.652780347411745880602240162971748308916e-13,
-1.47538894040200122488762518223954255436e-14,
// [7.2857,7.4286]: m=5 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0003599208362713041433069493887071243068639,
-0.0000322437905860019038750401284161220550389,
1.592348248752516328760147817816437094182e-6,
-5.622921725907300249827781317603558128365e-8,
1.567664682233215752152487038115919813533e-9,
-3.633694715600989809991353348115958276731e-11,
7.229623744129288729862284949936064688226e-13,
-1.261418751016030566710269569687737296271e-14,
// [7.2857,7.4286]: m=6 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0002257065341020133224230102497024071022575,
-0.00002229287548252450211738714732396631622662,
1.180813562440984472562111314045728830285e-6,
-4.389461153159281657874530480497106735002e-8,
1.271793144441394862047493738908038932705e-9,
-3.03603007011598546613943635542350910537e-11,
6.18114448678996293007037701253560157847e-13,
-1.098410551366727270330970653540701652515e-14,
// [7.2857,7.4286]: m=7 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001560501283776715106150197746353044875442,
-0.00001653138987416430280367995866673652311206,
9.217868421638529923469956945648269427923e-7,
-3.561020842355458508232006095774605370948e-8,
1.062610519155990238778677311633916824754e-9,
-2.595716654797514955417020967887452014984e-11,
5.382384009375713367416134302492338949883e-13,
-9.707494523167548112080844244667485068272e-15,
// [7.2857,7.4286]: m=8 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0001157197291191501158240233217786491809155,
-0.00001290501579028546117379813680084418851873,
7.478143768950112551929743510435544025842e-7,
-2.975309487559926130241463354287637827479e-8,
9.085008243128698768863314251227344224954e-10,
-2.260275619890333601258082288104992330838e-11,
4.756828036985779393317839307955661579172e-13,
-8.684408394147003849436634070499639963787e-15,
// [7.2857,7.4286]: m=9 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00009033511053199822475127387255416693257996,
-0.00001046940127652249324321642535272922745359,
6.248149923879171590431761608942280785554e-7,
-2.543802338733608139022180452093755038302e-8,
7.910964625259811036546551567814689304541e-10,
-1.997573461209099364215260317115269295771e-11,
4.25550205375231485869657750177640235628e-13,
-7.848413028362615494001861715776527137998e-15,
// [7.2857,7.4286]: m=10 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00007328580893565744952069883672703594541959,
-8.747409893423854128698524585333759912608e-6,
5.34198491134363165064380575830438608546e-7,
-2.215070123017373907910110005059605762791e-8,
6.991507073504276860513669810189871165947e-10,
-1.787042592653272503732349586429253433596e-11,
3.845852712384059672825538624490511279774e-13,
-7.153893276858570498323030437637136248306e-15,
// [7.2857,7.4286]: m=11 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00006123186925396697596093763426130728247357,
-7.478778875874669744731302616398013601281e-6,
4.651647258339307574758939083299697001362e-7,
-1.957622006239680708445438520741147950694e-8,
6.254649036654764928295187970423426591028e-10,
-1.615011816364249241243574151802462626826e-11,
3.505528127307173359044432142160479093303e-13,
-6.568635625505218457497948589757276848823e-15,
// [7.2857,7.4286]: m=12 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00005235145213112268548175022340913744447684,
-6.512306161669103638174815096899739879529e-6,
4.111006213105951615178519245855662840391e-7,
-1.751301753971404110438846621194372122656e-8,
5.652541322313066670270659492375285691471e-10,
-1.472094214692201713389879779706670492741e-11,
3.218743334502771599459278062529033649774e-13,
-6.069325581303761724358661528514300688179e-15,
// [7.2857,7.4286]: m=13 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004558614313168372291745239470982452308603,
-5.755408698342825799827798235916715086614e-6,
3.677733683342396424899122472964168728794e-7,
-1.582711592273695580938794401892046095191e-8,
5.152329718785366143338132668400410127082e-10,
-1.351660749312950853820588386784081580342e-11,
2.97407397454003498603355185623872215246e-13,
-5.638721397121924903197989715708657467869e-15,
// [7.2857,7.4286]: m=14 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00004028786088839977820846178293092968354781,
-5.148827156674214635447992421091617934924e-6,
3.323694343777055451283963701368943762422e-7,
-1.442652341821520105194433312733571252461e-8,
4.730812591998815927949786175125954504763e-10,
-1.248913676626362574413746773075683230912e-11,
2.763071393630066407937025619678729312156e-13,
-5.263826748052913149046038661989580506074e-15,
// [7.2857,7.4286]: m=15 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0000360417900967195001988394422209023653421,
-4.653172081283058701557156629570109711702e-6,
3.029569917827351556746503023497272511883e-7,
-1.324627545035559315546749885256724915061e-8,
4.371197839401034951228644370474395250187e-10,
-1.160304935685083658874684188018800629929e-11,
2.57936723868595768407174275951211433634e-13,
-4.934677765624523030394283517150993963769e-15,
// [7.2857,7.4286]: m=16 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00003257220456898140878720212355371074071811,
-4.241397884953757579408475948023819589389e-6,
2.781717844576713323331556408991462162857e-7,
-1.223935413170850824778554063175930112519e-8,
4.061067247714231991397748779508494790224e-10,
-1.083160109032418534342625567110043655369e-11,
2.418079092732034605826738355064174435914e-13,
-4.643518698585644512823533943833448986289e-15,
// [7.2857,7.4286]: m=17 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002968978519467630104469023737266845269243,
-3.894404982403117260189361220011236498691e-6,
2.570264367660717464484093028253009537512e-7,
-1.137098846485707639081971498886235907826e-8,
3.791060355870284598709479899772984530227e-10,
-1.015428811031899889717136825436549893191e-11,
2.275406540656528121836134221368567848904e-13,
-4.384229706524101678537669779412475280396e-15,
// [7.2857,7.4286]: m=18 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002726083487682181891152862855356001502823,
-3.59837011472094991685647129060379709774e-6,
2.38790757762181945675020838503146827544e-7,
-1.06149691586195881283497475877999431738e-8,
3.554000814166043339808173413825911645269e-10,
-9.555150506653170389358347527238623372929e-12,
2.148350782301652309852749062875294440696e-13,
-4.151922008020663488079570004964660195178e-15,
// [7.2857,7.4286]: m=19 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002518859080304664759997497915604216782356,
-3.343070608666697073131473878648406920323e-6,
2.229143523311858815690361857890190808296e-7,
-9.951202433672961733007855816349960789443e-9,
3.344302654057752638139568126854791048894e-10,
-9.0215947995983962214273351955981582855e-12,
2.034516250830043579654838538340775722183e-13,
-3.942646432143238110713525640138318478203e-15,
// [7.2857,7.4286]: m=20 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.0000234014942606668777773667373100554667129,
-3.120800932632937197926132334111108267043e-6,
2.089752511072987144209208174742058467556e-7,
-9.364047577968795651185809952211393268635e-9,
3.157558157656964248834940503912874732469e-10,
-8.543560816948234665552820382540978625965e-12,
1.931967799819245188298115566064634662366e-13,
-3.753180273848019494605169077193519857182e-15,
// [7.2857,7.4286]: m=21 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002184560652843055872715431416640457908311,
-2.925653515498685127435673739271993782902e-6,
1.966449991375039090699829345463601153433e-7,
-8.841162981315748933489761918866141907038e-9,
2.990246264705094923259544215318660954823e-10,
-8.112921939117237728163671680328560161707e-12,
1.839126260049381011796919372280226109363e-13,
-3.580869154093696731757128170784758147097e-15,
// [7.2857,7.4286]: m=22 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00002047957460849079430359848498739830778928,
-2.753029987921711522641752936096940515182e-6,
1.856644226077832197367695766209785321464e-7,
-8.372689674903658626923394593072701902439e-9,
2.83952265835868351645135200148079455544e-10,
-7.723046482232411765075975400931108347583e-12,
1.754690949164058981506706761301716709657e-13,
-3.423508130897159986603434307250487499755e-15,
// [7.2857,7.4286]: m=23 maxabs=7.297617753308889496258e-18 maxrel=1.16558690608846800714213e-16
0.00001927120991545197913432486208196632161476,
-2.599301916505762745316191125794048877119e-6,
1.758264831731231520213259222132206144757e-7,
-7.950663571498725647016245568059852872426e-9,
2.703066249271833881578868511969209128479e-10,
-7.36847227263582169563102761075718053615e-12,
