# -*- coding: utf-8 -*-
"""qt.gui.py"""

# Copyright (C) 2009, 2010, 2011, 2012, 2013 Federico Brega, Pierluigi Villani

# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 3
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

from __future__ import unicode_literals
from PyQt4 import QtGui, QtCore
import ui_qt
import glal
import os.path
from themes import ThemeManager
from version import VERSION
import StringIO

if QtCore.PYQT_VERSION >= 0x040700:
    iconFromTheme = QtGui.QIcon.fromTheme
else:
    #define fallback method if using an odl version of Qt
    def iconFromTheme(_, icon=None):
        if icon == None:
            icon = QtGui.QIcon()
        return icon
    
class Gui(QtGui.QMainWindow, ui_qt.Ui_MainWindow):
    """Main frame"""
    def __init__(self, parent, title, size):
        QtGui.QMainWindow.__init__(self, parent)
        self.setupUi(self)
        self.setWindowTitle(title)
        self.notebook.removeTab(0)
        self.cg_exit = None #check on exit callback
        self.setContextMenuPolicy(QtCore.Qt.PreventContextMenu)
        self.resize(QtCore.QSize(size[0], size[1]))

        # center Gui
        screen = QtGui.QDesktopWidget().screenGeometry()
        dimension =  self.geometry()
        self.move( (screen.width() - dimension.width()) / 2,
                   (screen.height() - dimension.height()) / 2)
        if size[2]:
            self.showMaximized()

        ##Set standard icons
        icon1 = QtGui.QIcon(QtGui.QPixmap(":/qt-icons/pixmaps/open.png"))
        icon1 = iconFromTheme("document-open",  icon1)
        self.menu_item_open.setIcon(icon1)
        icon2 = QtGui.QIcon(QtGui.QPixmap(":/qt-icons/pixmaps/save_as.png"))
        icon2 = iconFromTheme("document-save-as",  icon2)
        self.menu_item_save_as.setIcon(icon2)
        icon0 = QtGui.QIcon(QtGui.QPixmap(":/qt-icons/pixmaps/save.png"))
        icon0 = iconFromTheme("document-save",  icon0)
        self.menu_item_save.setIcon(icon0)
        icon3 = QtGui.QIcon(QtGui.QPixmap(":/qt-icons/pixmaps/new.png"))
        icon3 = iconFromTheme("document-new",  icon3)
        self.menu_item_new.setIcon(icon3)
        icon_about = iconFromTheme("help-about")
        self.menu_item_about.setIcon(icon_about)
        icon_bug = iconFromTheme("tools-report-bug")
        self.menu_item_bug.setIcon(icon_bug)
        icon_exit = iconFromTheme("application-exit")
        self.menu_item_exit.setIcon(icon_exit)
        icon_server = iconFromTheme("network-server")
        self.menu_Kml_server.setIcon(icon_server)
        icon_pref = iconFromTheme("preferences-desktop")
        self.preferences.setIcon(icon_pref)

        #Group the altitude services to have radio button.
        serv_Action_group = QtGui.QActionGroup(self)
        serv_Action_group.addAction(self.menu_item_s1)
        serv_Action_group.addAction(self.menu_item_s2)
        serv_Action_group.addAction(self.menu_item_s3)

        self.connect(self.menu_item_about, QtCore.SIGNAL("triggered()"),
                     self.onabout)

    def closeEvent(self, event):
        """If a close event is generated call the controller"""
        if self.cg_exit():
            event.accept()
        else:
            event.ignore()

    def quit(self):
        """ Exit from CycloGraph"""
        self.close()

    def getdimensions(self):
        """Return window width and height"""
        return (self.size().width(), self.size().height())

    def ismaximized(self):
        """Return true if window is maximized"""
        return self.isMaximized()

    def addstatusbartext(self, text):
        """Show text in the statusbar for 5000 ms"""
        self.statusbar.showMessage(text, 5000)

    def onabout(self):
        """About dialog"""
        description = """CycloGraph"""
        licence = """
CycloGraph is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.

CycloGraph is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
You should have received a copy of the GNU General Public License along with this program. If not, see http://www.gnu.org/licenses/.
"""
        text = 'Copyright (C) 2008-2013 Federico Brega, Pierluigi Villani\n'+\
        'http://cyclograph.sourceforge.net\n'+ licence
        QtGui.QMessageBox.about(self, description +" "+ VERSION, text)

class Page(QtGui.QFrame, ui_qt.Ui_Frame_Page):
    """Page showed in a tab"""
    def __init__(self, parent):
        QtGui.QFrame.__init__(self, parent)
        self.savedfilepath = None
        self.slopecounternew = 0
        self.modified = False
        self.setupUi(self)
    def insert_row(self, row_num, cp):
        """Insert item to this page at position row_num"""
        self.tableWidget.insertRow(row_num)
        for col in range(len(cp)):
            item = QtGui.QTableWidgetItem(unicode(cp[col]))
            self.tableWidget.setItem(row_num, col, item)
    def delete_row(self, row_num):
        """Removes row at position row_num in this page."""
        self.tableWidget.removeRow(row_num)
    def get_sel_row(self):
        """Gives the row number that is selected."""
        row = self.tableWidget.currentRow()
        return row
    def get_sel_multi_row(self):
        """Gives a list with the row number of selected items."""
        rows = [self.tableWidget.row(item) \
                for item in self.tableWidget.selectedItems()]
        rows = list(set(rows))
        rows.sort()
        return rows

class Managecp(QtGui.QDialog, ui_qt.Ui_manag_dlg):
    """Add and edit dialog"""
    def __init__(self, parent, title,
                 dist="", alt="", name=""):
        QtGui.QDialog.__init__(self, parent)
        self.setupUi(self)
        self.setWindowTitle(title)
        self.distance.setText(dist)
        self.altitude.setText(alt)
        self.name.setText(name)
    def show_modal(self):
        """Shows the dialog and returns True if the value has to be chenged."""
        val = self.exec_()
        return val == QtGui.QDialog.Accepted
    def destroy(self):
        """Destroy current managecp"""
        pass
    def getcp(self):
        """Gives the modified values."""
        dist = self.distance.text()
        alt = self.altitude.text()
        name = self.name.text()
        return (dist, alt, name)

class FormSlopeInfo(QtGui.QDialog, ui_qt.Ui_Slope_Dialog):
    """info dialog about the slope"""
    def __init__(self, parent, slopedata):
        QtGui.QDialog.__init__(self, parent)
        self.setupUi(self)
        self.s_name.setText(slopedata['name'])
        self.s_state.setText(slopedata['state'])
        self.s_author.setText(slopedata['author'])
        self.s_email.setText(slopedata['email'])
        self.average_grad.setText(slopedata['average_grad'])
        self.max_grad.setText(slopedata['max_grad'])
        self.height_difference.setText(slopedata['height_difference'])
        self.height_gain.setText(slopedata['height_gain'])
        self.s_comment.setText(slopedata['comment'])
    def getslopeinfo(self):
        """Returns the slope infos: name, state, comment"""
        return {'name' : self.s_name.text(),
                'state' : self.s_state.text(),
                'author' : self.s_author.text(),
                'email' : self.s_email.text(),
                'comment' : self.s_comment.toPlainText()}

FONT_TYP = {
            QtGui.QFont.Light : "light",
            QtGui.QFont.Normal : "normal",
            QtGui.QFont.Bold : "bold",
            QtGui.QFont.DemiBold : "bold",
            QtGui.QFont.Black : "bold",
           }
class Preferences(QtGui.QDialog, ui_qt.Ui_Pref_Dialog):
    """Set gradient's color"""
    def __init__(self, parent, pref, default_pref):
        QtGui.QDialog.__init__(self, parent)
        self.setupUi(self)

        self.default = default_pref
        #General
        self.olines.setChecked(pref['olines'])
        self.effect3d.setChecked(pref['3d'])
        self.showinfo.setChecked(pref['sinfo'])
        self.vcheck.setChecked(pref['vcheck'])
        res = 0
        if pref['width'] == 1600:
            res = 3
        if pref['width'] == 1280:
            res = 2
        elif pref['width'] == 1024:
            res = 1
        self.res.setCurrentIndex(res)

        current = 0
        for theme in pref['themelist']:
            self.theme.addItem("")
            self.theme.setItemText(current, theme)
            if theme == pref['theme']:
                self.theme.setCurrentIndex(current)
            current += 1

        self.serv.setCurrentIndex(0)
        if pref['serv'] == 'usgs.net':
            self.serv.setCurrentIndex(2)
        elif pref['serv'] == 'earthtools.org':
            self.serv.setCurrentIndex(1)
        elif pref['serv'] == 'google elevation':
            self.serv.setCurrentIndex(3)
        #Fonts
        self.fontbuttonlist = [self.fdesc,
                               self.fgrad,
                               self.ftitle]
        self.fontbuttongroup = QtGui.QButtonGroup(parent)
        for id_, button in enumerate(self.fontbuttonlist):
            self.fontbuttongroup.addButton(button, id_)
        desc = pref['fdesc']
        self.fdesc.setText(desc['des']+' | '+str(desc["dim"]))
        self.fdesc.desc = desc
        desc = pref['fgrad']
        self.fgrad.setText(desc['des']+' | '+str(desc["dim"]))
        self.fgrad.desc = desc
        desc = pref['ftitle']
        self.ftitle.setText(desc['des']+' | '+str(desc["dim"]))
        self.ftitle.desc = desc
        self.connect(self.fontbuttongroup, QtCore.SIGNAL("buttonClicked(int)"),
                     self.font_button_clicked)
        #Color group buttons
        self.buttonlist = [self.button1,
                           self.button2,
                           self.button3,
                           self.button4,
                           self.button5,
                           self.button6,
                           self.button7]
        self.buttongroup = QtGui.QButtonGroup(parent)
        for id_, button in enumerate(self.buttonlist):
            self.buttongroup.addButton(button, id_)
        self.colorlist = [ map(int, color[4:-1].split(',')) \
                           for color in pref['colors']]
        for index in range(len(self.buttonlist)):
            self.setcolor(index, self.colorlist[index])
        self.connect(self.buttonBox, QtCore.SIGNAL("clicked(QAbstractButton *)"), self.restore_default)
        self.bindsgc()
        #Author info
        self.aname.setText(pref['aname'])
        self.amail.setText(pref['amail'])

        #Level group SpinButtons
        self.levlist = [self.lev_1,
                           self.lev_2,
                           self.lev_3,
                           self.lev_4,
                           self.lev_5,
                           self.lev_6,
                           self.lev_7]
        for i in range(len(pref['levels'])):
            self.levlist[i].setValue(float(pref['levels'][i]))
            self.levlist[i].editingFinished.connect(self.updatelevelconstraints)
        self.updatelevelconstraints()

    def restore_default(self, button):
        button_id = self.buttonBox.standardButton(button)
        if button_id == QtGui.QDialogButtonBox.RestoreDefaults:
            self.olines.setChecked(bool(self.default['orizzontal lines']))
            self.showinfo.setChecked(bool(self.default['slopeinfo']))
            self.effect3d.setChecked(bool(self.default['effect-3d']))
            self.vcheck.setChecked(bool(self.default['vcheck']))
            res = 0
            if self.default['width'] == 1600:
                res = 3
            if self.default['width'] == 1280:
                res = 2
            elif self.default['width'] == 1024:
                res = 1
            self.res.setCurrentIndex(res)
            index = self.theme.findText(self.default['theme'])
            if index:
                self.theme.setCurrentIndex(index)
            self.serv.setCurrentIndex(0)
            if self.default['server'] == 'usgs.net':
                self.serv.setCurrentIndex(2)
            elif self.default['server'] == 'earthtools.org':
                self.serv.setCurrentIndex(1)
            elif self.default['server'] == 'google elevation':
                self.serv.setCurrentIndex(3)

            self.fdesc.setText(self.default['font-des 1']+' | '+self.default['font-dim 1'])
            self.fdesc.desc = {
                                   "des" : self.default['font-des 1'],
                                   "typ" : self.default['font-typ 1'],
                                   "dim" : self.default['font-dim 1']
                               }
            self.fgrad.setText(self.default['font-des g']+' | '+self.default['font-dim g'])
            self.fgrad.desc = {
                                   "des" : self.default['font-des g'],
                                   "typ" : self.default['font-typ g'],
                                   "dim" : self.default['font-dim g']
                               }
            self.ftitle.setText(self.default['font-des t']+' | '+self.default['font-dim t'])
            self.ftitle.desc = {
                                   "des" : self.default['font-des t'],
                                   "typ" : self.default['font-typ t'],
                                   "dim" : self.default['font-dim t']
                                }
            self.aname.setText(self.default['authorname'])
            self.amail.setText(self.default['authormail'])
            
            colorlist = [ self.default["color "+str(i)] for i in range(1, 8) ]
            self.colorlist = [ map(int, color[4:-1].split(',')) \
                               for color in colorlist]
            for index in range(len(self.buttonlist)):
                self.setcolor(index, self.colorlist[index])
            
            levellist = [self.default["level "+str(i)] for i in range(1, 8)]
            for i in range(len(self.levlist)):
                #disable range fot this widget
                self.levlist[i].setMinimum(0.0)
                self.levlist[i].setMaximum(90.0)
                #set the range for this widget
                self.levlist[i].setValue(float(levellist[i]))
            self.updatelevelconstraints()

    def bindsgc(self):
        """Bind Setgcolor buttons """
        self.connect(self.buttongroup, QtCore.SIGNAL("buttonClicked(int)"),
                     self.changecolor)
    def setcolor(self, index, color):
        """Set color in the colorlist"""
        qcolor = QtGui.QColor(*color)
        palette = self.buttonlist[index].palette()
        palette.setColor(QtGui.QPalette.Button, qcolor)
        self.colorlist[index] = (color[0], color[1], color[2])
        self.buttonlist[index].setStyleSheet("QWidget { background-color: %s }"
            % qcolor.name())
    def updatelevelconstraints(self):
        for i in range(1,len(self.levlist)):
            self.levlist[i-1].setMaximum(self.levlist[i].value()-0.1)
        for i in range(len(self.levlist)-1):
            self.levlist[i+1].setMinimum(self.levlist[i].value()+0.1)
    def changecolor(self, index):
        """Change the color used by plot to fill different classes of gradients
        """
        button = self.buttonlist[index]
        palette = button.palette()
        qcolor = palette.color(QtGui.QPalette.Button)
        newcolor = QtGui.QColorDialog.getColor(qcolor, self)
        if newcolor.isValid():
            self.setcolor(index,
                          (newcolor.red(),
                           newcolor.green(),
                           newcolor.blue()))
    def font_button_clicked(self, index):
        button = self.fontbuttonlist[index]
        qfont = QtGui.QFont()
        qfont.setFamily(button.desc["des"])
        qfont.setPointSize(int(button.desc["dim"]))
        try:
            res = eval('QtGui.QFont.'+button.desc["typ"].capitalize())
        except AttributeError:
            res = QtGui.QFont.Normal
        qfont.setWeight(res)
        (qfont, ok) = QtGui.QFontDialog.getFont(qfont, self)
        if ok:
            button.desc = {
                            "des" : str(qfont.family()),
                            "typ" : FONT_TYP[qfont.weight()],
                            "dim" : str(qfont.pointSize())
                          }
            button.setText(str(qfont.family())+' | '+str(qfont.pointSize()))
    def getconfig(self):
        """Returns selected config options"""
        res = str(self.res.currentText()).split(' ')
        settings = {
                    'olines': self.olines.isChecked(),
                    'theme' : str(self.theme.currentText()),
                    'sinfo' : self.showinfo.isChecked(),
                    'vcheck': self.vcheck.isChecked(),
                    '3d'    : self.effect3d.isChecked(),
                    'aname' : self.aname.text(),
                    'amail' : self.amail.text(),
                    'fdesc' : self.fdesc.desc,
                    'ftitle': self.ftitle.desc,
                    'fgrad' : self.fgrad.desc,
                    'width' : res[0],
                    'height': res[2],
                    'serv'  : str(self.serv.currentText()),
                    'colors': ['rgb(%d,%d,%d)' % (clr[0], clr[1], clr[2]) \
                                for clr in self.colorlist],
                    'levels': ['%f' %(lev.value()) for lev in self.levlist]
                   }
        return settings

class Plot(QtGui.QMainWindow):
    """ Plot window"""
    def __init__(self, parent, slope, plot_settings,
                 exportfunc, exportfunc_pdf, exportfunc_html):
        QtGui.QMainWindow.__init__(self, parent)
        self.panel = AnimatedPanel(slope)
        self.setCentralWidget(self.panel)
        title = "CycloGraph"
        if slope.name:
            title += " - " + slope.name
        self.setWindowTitle(title)
        self.myslope = slope

        self.menu_item_save = QtGui.QAction( _('Save'), self)
        icon_save = QtGui.QIcon(QtGui.QPixmap(":/qt-icons/pixmaps/save.png"))
        icon_save = iconFromTheme("document-save-as", icon_save)
        self.menu_item_save.setIcon(icon_save)
        self.menu_item_savepdf = QtGui.QAction( _('Save PDF'), self)
        self.menu_item_savehtml = QtGui.QAction( _('Save html'), self)
        self.menu_item_olines = QtGui.QAction(self)
        self.menu_item_olines.setCheckable(True)
        self.menu_item_olines.setChecked(plot_settings['olines'])
        self.menu_item_olines.setText(_("&Orizzontal lines"))
        self.menu_item_3d = QtGui.QAction(self)
        self.menu_item_3d.setCheckable(True)
        self.menu_item_3d.setChecked(plot_settings['3d'])
        self.menu_item_3d.setText(_("3D &effect"))

        self.themeitems = {}
        self.theme_action_group = QtGui.QActionGroup(self)
        self.thememanager = ThemeManager()
        for theme in self.thememanager.getthemeslist():
            themeitem = QtGui.QAction(self)
            themeitem.setCheckable(True)
            if plot_settings['theme'] == theme:
                themeitem.setChecked(True)
            themeitem.setText(theme)
            self.theme_action_group.addAction(themeitem)
            self.themeitems[themeitem] = theme

        menubar = self.menuBar()
        options_menu = menubar.addMenu( _('&Options') )
        options_menu.addAction(self.menu_item_save)
        options_menu.addAction(self.menu_item_savepdf)
        options_menu.addAction(self.menu_item_savehtml)
        options_menu.addAction(self.menu_item_olines)
        options_menu.addAction(self.menu_item_3d)
        themes_menu = QtGui.QMenu(options_menu)
        themes_menu.setTitle(_("&Theme"))
        for action in self.themeitems.keys():
            themes_menu.addAction(action)
        options_menu.addAction(themes_menu.menuAction())
        res_menu = QtGui.QMenu(options_menu)
        res_menu.setTitle(_("&Saving resolution"))
        self.resitems = {}
        res_action_group = QtGui.QActionGroup(self)
        rlist = [('800','600'), ('1024','768'), ('1280','800'), ('1600','1200')]
        for res in rlist:
            resitem = QtGui.QAction(self)
            resitem.setCheckable(True)
            restext = res[0]+'x'+res[1]
            resitem.setText(restext)
            if plot_settings['width'] == int(res[0]):
                resitem.setChecked(True)
            res_action_group.addAction(resitem)
            self.resitems[resitem] = restext
            res_menu.addAction(resitem)
        options_menu.addAction(res_menu.menuAction())
        self.exportfunc = exportfunc
        self.exportfunc_pdf = exportfunc_pdf
        self.exportfunc_html = exportfunc_html
        self.myslope.calc()
        self.settings = plot_settings
        self.panel.settings = self.settings
        self.panel.devc = glal.DeviceContext()
        self.resize(plot_settings['width'], plot_settings['height'])

        self.settings['colors'] = plot_settings['colors']

        self.animation3d = QtCore.QPropertyAnimation(self.panel, "depth")
        self.animation3d.setDuration(1000)
        self.animation3d.setStartValue(100)
        self.animation3d.setEndValue(0)
        self.animationTh = QtCore.QPropertyAnimation(self.panel, "opacity")
        self.animationTh.setDuration(1000)
        self.animationTh.setStartValue(1.0)
        self.animationTh.setEndValue(0.0)

        QtCore.QObject.connect(self.menu_item_save, QtCore.SIGNAL("triggered()"), self.saveimage)
        QtCore.QObject.connect(self.menu_item_savepdf, QtCore.SIGNAL("triggered()"), self.savepdf)
        QtCore.QObject.connect(self.menu_item_savehtml, QtCore.SIGNAL("triggered()"), self.savehtml)
        QtCore.QObject.connect(self.menu_item_olines, QtCore.SIGNAL("triggered()"), self.olines)
        QtCore.QObject.connect(self.menu_item_3d, QtCore.SIGNAL("triggered()"), self.change3d)
        QtCore.QObject.connect(self.theme_action_group, QtCore.SIGNAL("triggered(QAction *)"), self.changetheme)
        QtCore.QObject.connect(res_action_group, QtCore.SIGNAL("triggered(QAction *)"), self.changeres)

    def showplot(self):
        """ Show plot"""
        self.show()
    def saveimage(self):
        """ Save image"""
        self.exportfunc(self.myslope, self.settings)
    def savepdf(self):
        """ Save pdf"""
        self.exportfunc_pdf(self.myslope, self.settings)
    def savehtml(self):
        """ Save html report"""
        self.exportfunc_html(self.myslope, self.settings)
    def change3d(self):
        """ If active switch 3d effect off, else turn it on."""
        if self.settings['3d']:
            self.animation3d.setDirection(self.animation3d.Forward)
        else:
            self.animation3d.setDirection(self.animation3d.Backward)
        self.animation3d.start()
    def olines(self):
        """ If active draws orizzontal lines"""
        self.panel.shot()
        self.settings['olines'] = not self.settings['olines']
        self.animationTh.start()
    def changetheme(self, action):
        """Change the actual slope theme."""
        self.panel.shot()
        self.settings["theme"] = self.themeitems[action]
        self.animationTh.start()
    def changeres(self, action):
        """Change saving resolution."""
        text = self.resitems[action]
        width, height = text.split('x')
        self.settings['width'] = int(width) 
        self.settings['height'] = int(height)

class AnimatedPanel(QtGui.QWidget):
    """This class implements a personalized widget that draws a slope.
    It also implements the fade between different images and 2D to 3D.
    Using this class prevents from drawing on the menu."""
    def __init__(self, slope, parent=None):
        QtGui.QWidget.__init__(self, parent)
        self.__old_opacity = 0.0
        self.__depth = 100
        self.myslope = slope
    def shot(self):
        """Create an internal pixmap representing the current state
        of the widget"""
        self.__old_pixmap = QtGui.QPixmap(self.size())
        self.render(self.__old_pixmap)
        
    def oldOpacity(self):
        """Get the opacity of the pixamp of the old slope"""
        return self.__old_opacity
    def setOldOpacity(self, opacity):
        """Set the opacity of the pixamp of the old slope"""
        self.__old_opacity = opacity
        self.repaint()
    def changeres(self, action):
        """Change saving resolution."""
        text = self.resitems[action]
        width, height = text.split('x')
        self.settings['width'] = int(width) 
        self.settings['height'] = int(height)
    opacity = QtCore.pyqtProperty("double", oldOpacity, setOldOpacity)
    
    def depth(self):
        """Get the depth of the current slope (0 means 2D)"""
        return self.__depth
    def setDepth(self, depth):
        """Set the depth of the current slope"""
        self.__depth = depth
        self.settings['3d'] = (depth != 0)
        self.repaint()
    depth = QtCore.pyqtProperty("int", depth, setDepth)
    
    def paintEvent(self, event):
        """Actually draws the content."""
        self.myslope.depth = self.__depth
        self.devc.init_draw_surf(self)
        self.myslope.paint(self.settings, self.devc)
        if self.__old_opacity > 0.0:
            self.devc.paint.resetTransform()
            self.devc.paint.setOpacity(self.__old_opacity)
            self.devc.paint.drawPixmap(0, 0, self.__old_pixmap)
        self.devc.end_draw()

class Map(QtGui.QDialog, ui_qt.Ui_map):
    """Show map of slopes"""
    def __init__(self, parent, slopename, stringc, savefnct):
        QtGui.QDialog.__init__(self, parent)
        self.setupUi(self)
        self.savefnct = savefnct
        self.setWindowFlags(QtCore.Qt.Window)
        self.setWindowTitle(_(u"Map")+" - "+ slopename)
        QtCore.QObject.connect(self.buttonBox,
                               QtCore.SIGNAL("clicked(QAbstractButton *)"),
                               self.button_clicked)
        mapfile = QtCore.QFile(":/openlayers/cyclograph/map.html")
        if mapfile.open(QtCore.QFile.ReadOnly):
            mapdata = unicode(mapfile.readAll())
            htmlMap = mapdata % (stringc,)
            mapfile.close()
            self.webView.setHtml(htmlMap)
        self.show()
    def saveimage(self, filepath):
        """ Save map image"""
        page = self.webView.page()
        img = QtGui.QImage(page.viewportSize(), QtGui.QImage.Format_ARGB32)
        painter = QtGui.QPainter(img)
        self.webView.render(painter)
        painter.end()
        img.save(filepath)

    def button_clicked(self, button):
        button_id = self.buttonBox.standardButton(button)
        if button_id == QtGui.QDialogButtonBox.Save:
            self.savefnct(self)

class Create_kml(QtGui.QDialog, ui_qt.Ui_Create_KML_Dialog):
    """Create kml dialog"""
    MODE = {_("Open Route Service") : 'ORS',
            _(u'OSRM') : 'OSRM',
            _("Draw on the map") : 'DRAW'}

    def __init__(self, parent, downloadfunc, savefunc):
        QtGui.QDialog.__init__(self, parent)
        self.setupUi(self)
        self.setWindowFlags(QtCore.Qt.Window)
        self.downloadfunc = downloadfunc #used to downlaod from url
        self.savefunc = savefunc
        self.handler = None
        self.mode = None
        self.mode_changed(self.listWidget.currentItem().text())
        self.splitter.setStretchFactor(0, 1)
        self.splitter.setStretchFactor(1, 3)

        QtCore.QObject.connect(self.buttonBox, QtCore.SIGNAL("clicked(QAbstractButton *)"), self.button_clicked)
        QtCore.QObject.connect(self.listWidget, QtCore.SIGNAL("currentTextChanged(const QString&)"), self.mode_changed)

    def webview_init(self):
        """ Check mode and set webview accordingly"""
        if self.mode == 'ORS':
            self.webView.setUrl(QtCore.QUrl("http://openrouteservice.org/"))
        elif self.mode == 'OSRM':
            self.webView.setUrl(QtCore.QUrl("http://map.project-osrm.org/"))
        elif self.mode == 'DRAW':
            url = QtCore.QUrl("qrc:/openlayers/cyclograph/openlayers.html")#.fromLocalFile(os.path.join(os.path.dirname(__file__),"openlayers.html"))
            self.webView.setUrl(url)

    def button_clicked(self, button):
        button_id = self.buttonBox.standardButton(button)
        if button_id == QtGui.QDialogButtonBox.Ok:
            self.download()
            QtGui.QDialog.accept(self)
        elif button_id == QtGui.QDialogButtonBox.Save:
            self.save_kml()
            #Do not accept the dialog: it must stay open if the user wants to
            #import without opening the file saved or if he hasn't saved and
            #wants to go back.
        elif button_id == QtGui.QDialogButtonBox.Cancel:
            self.handler = None
            QtGui.QDialog.reject(self)

    def mode_changed(self, sel_string):
        """ Change current mode"""
        newmode = self.MODE[str(sel_string)]
        if newmode != self.mode:
            self.mode = newmode
            self.webview_init()

    def get_ors(self):
        page = self.webView.page()
        frame = page.currentFrame()

        gpxcontent = unicode(frame.evaluateJavaScript("document.getElementById('gpx').innerHTML").toString())
        from xml.sax.saxutils import unescape
        return StringIO.StringIO(unescape(gpxcontent))

    def get_osrm(self):
        """ Returns gpx content from ORSM service"""
        page = self.webView.page()
        frame = page.currentFrame()

        gpxcontent = unicode(frame.evaluateJavaScript("document.getElementsByClassName('text-link')[1].getAttribute('onclick')").toString())
        url = gpxcontent.split('\'')
        return self.downloadfunc(url[1])

    def get_draw(self):
        frame = self.webView.page().currentFrame()
        kmlstring = frame.evaluateJavaScript("createKml()").toString()
        return StringIO.StringIO(kmlstring)

    def download(self):
        if self.mode == 'ORS':
            self.handler = self.get_ors()
        elif self.mode == 'OSRM':
            self.handler = self.get_osrm()
        elif self.mode == 'DRAW':
            self.handler = self.get_draw()

    def save_kml(self):
        self.download()
        if not self.handler:
            return
        self.savefunc(self.handler, self.mode)
    def show_modal(self):
        """Shows the dialog and returns True if the value has to be chenged."""
        val = self.exec_()
        return val == QtGui.QDialog.Accepted

class NumcpsDialog(QtGui.QDialog, ui_qt.Ui_numcps_dlg):
    """ask cps number you want to import"""
    def __init__(self, max_num_cps, max_length, file_ext):
        QtGui.QDialog.__init__(self, None)
        self.setupUi(self)
        self.slider.setMaximum(max_num_cps)
        self.spinBox.setMaximum(max_num_cps)
        self.slider_2.setMaximum(int(max_length))
        self.spinBox_2.setMaximum(int(max_length))
        text = _("The %s file contains %d points.\nChoose how many will be imported.") % (file_ext, max_num_cps)
        self.label.setText(text)
        self.label_2.setText(_(u"The %s file contains a route of %d Km.\nChoose what range you want between points.") % (file_ext, max_length))
        self.label_3.setText(_("Choose the type of import you want"))
        
        self.proceedButton.clicked.connect(self.onProceedClicked)
        
    def onProceedClicked(self):
        btn = self.buttonGroup.checkedButton()
        if btn == self.all_ or btn == self.automatic:
            self.accept()
        elif btn == self.number:
            self.stackedWidget.setCurrentIndex(1)
        elif btn == self.distance_2:
            self.stackedWidget.setCurrentIndex(2)

def preferences(parent, cgpref, defaultpref):
    """ set Preference dialog"""
    dlg = Preferences(parent, cgpref, defaultpref)
    if dlg.exec_() == QtGui.QDialog.Rejected:
        return {}
    else:
        return dlg.getconfig()

def formslopeinfo(parent, slopedata):
    """ Show slope's info"""
    dlg = FormSlopeInfo(parent, slopedata)
    if dlg.exec_() == QtGui.QDialog.Rejected:
        return {}
    else:
        return dlg.getslopeinfo()

def FileDialog(parent, title, dirname, filename, other,
               file_types, rw, FD_OPEN, FD_SAVE ):
    """ Show File dialog"""
    filters = reduce(lambda x, y: x + ';;' + y,
                      [ ("%s (%s)" % elem) for elem in file_types])
    #print filters
    if rw == FD_OPEN:
        file_ = QtGui.QFileDialog.getOpenFileName(parent,  title,
                                                 dirname,  filters)
    elif rw == FD_SAVE:
        file_ = QtGui.QFileDialog.getSaveFileName(parent,  title,
                                                 dirname,  filters)
    if file_:
        return unicode(file_)
    else:
        return None

def numcpsdlg(max_num_cps, max_length, file_ext):
    """ask how many cps in a gpx or tcx files shall be imported"""
    dlg = NumcpsDialog(max_num_cps, max_length, file_ext)
    if dlg.exec_() == QtGui.QDialog.Rejected:
        return (-1, 0)
    elif dlg.buttonGroup.checkedButton() == dlg.automatic:
        return (0, -1)
    elif dlg.buttonGroup.checkedButton() == dlg.all_:
        return (0, 0)
    elif dlg.stackedWidget.currentIndex() == 1:
        return (0, dlg.slider.value())
    else:
        return (1, dlg.slider_2.value())

def geterrnoroute(parent):
    """Get an error message explaining that you should have selected one
       route"""
    return QtGui.QMessageBox.warning(parent, _('No route'), _("Specify one route first."),
                                        QtGui.QMessageBox.Ok)

def managecperr(parent):
    """Get an error message explaining that you should have inserted 
       altitude and distance"""
    return QtGui.QMessageBox.critical(parent, _("Form incomplete"),
                                      _("Distance and altitude are required."),
                                      QtGui.QMessageBox.Ok)

def mapcoorderr(parent):
    """Get an error message explaining that there are no coords in the slope"""
    return QtGui.QMessageBox.critical(parent, _("No coordinates"),
                                      _("there aren't coordinates associated with this slope."),
                                      QtGui.QMessageBox.Ok)
def geterrnocp(parent):
    """Get an error message explaining that you should have at least two cps
       to plot
    """
    return QtGui.QMessageBox.warning(parent, _('Not enough check points'), _(
"""There aren't enougth check points to plot.
 At least two points are required to plot."""),
                                        QtGui.QMessageBox.Ok)

def save_changes(parent, filename):
    """Show a message if you close a modified and not saved slope"""
    response = None
    SAVE = _('Save')
    DISCARD = _('Close without saving')
    CANCEL = _('Cancel')
    message = QtGui.QMessageBox(parent)
    message.setText(_("Save changes to %s before closing?") % filename)
    message.setWindowTitle(_('Save changes'))
    message.setIcon(QtGui.QMessageBox.Warning)
    message.addButton(SAVE, QtGui.QMessageBox.AcceptRole)
    message.addButton(DISCARD, QtGui.QMessageBox.DestructiveRole)
    message.addButton(CANCEL, QtGui.QMessageBox.RejectRole)
    message.exec_()
    response = message.clickedButton().text()
    if response == SAVE:
        return 'SAVE'
    if response == DISCARD:
        return 'DISCARD'
    if response == CANCEL:
        return 'CANCEL'
    return response

# vim:sw=4:softtabstop=4:expandtab
