/*
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 *
*/

function ContactAddress(pref, type, formatted, streetAddress, locality, region, postalCode, country) {
    if (pref)
        this.pref = pref;
    if (type)
        this.type = type;
    if (formatted)
        this.formatted = formatted;
    if (streetAddress)
        this.streetAddress = streetAddress;
    if (locality)
        this.locality = locality;
    if (region)
        this.region = region;
    if (postalCode)
        this.postalCode = postalCode;
    if (country)
        this.country = country;
}

ContactAddress.create = function(obj) {
    var result = new ContactAddress()
    result.pref = obj.pref
    result.type = obj.type
    result.formatted = obj.formatted
    result.streetAddress = obj.streetAddress
    result.locality = obj.streetLocality
    result.region = obj.region
    result.postalCode = obj.postalCode
    result.country = obj.country
    return result
}

ContactAddress.prototype = {
    pref: false,
    type: "",
    formatted: "",
    streetAddress: "",
    locality: "",
    region: "",
    postalCode: "",
    country: ""
};

function ContactField(type, value, pref) {
    this.type = String(type);
    this.value = String(value);
    this.pref = pref;
}

ContactField.create = function(obj) {
    var result = new ContactField()
    result.type = obj.type;
    result.value = obj.value;
    result.pref = obj.pref;
    return result;
}

ContactField.prototype = {
    type: "",
    value: "",
    pref: false
};

function ContactFindOptions(filter, multiple) {
    if (filter)
        this.filter = filter;
    if (multiple)
        this.multiple = multiple;
}

ContactFindOptions.create = function(obj) {
    var result = new ContactFindOptions();
    result.filter = obj.filter;
    result.multiple = obj.multiple;
    return result;
}

ContactFindOptions.prototype.filter = "";
ContactFindOptions.prototype.multiple = false;


function ContactName(formatted, familyName, givenName, middleName, honorificPrefix, honorificSuffix) {
    if (formatted)
       this.formatted = formatted;
    if (familyName)
       this.familyName = familyName;
    if (givenName)
       this.givenName = givenName;
    if (middleName)
       this.middleName = middleName;
    if (honorificPrefix)
       this.honorificPrefix = honorificPrefix;
    if (honorificSuffix)
       this.honorificSuffix = honorificSuffix;
}

ContactName.create = function(obj) {
    var result = new ContactName()
    result.familyName = obj.familyName
    result.givenName = obj.givenName
    result.formatted = obj.formatted
    result.middleName = obj.middleName
    result.honorificPrefix = obj.honorificPrefix
    result.honorificSuffix = obj.honorificSuffix
    var formattedArr = []

    if (typeof result.honorificPrefix === 'undefined')
        result.honorificPrefix = "";
    else if (result.honorificPrefix !== "")
        formattedArr.push(result.honorificPrefix);

    if (typeof result.givenName === 'undefined')
        result.givenName = "";
    else if (result.givenName !== "")
        formattedArr.push(result.givenName);

    if (typeof result.middleName === 'undefined')
        result.middleName = "";
    else if (result.middleName !== "")
        formattedArr.push(result.middleName);

    if (typeof result.familyName == 'undefined')
        result.familyName = "";
    else if (result.familyName !== "")
        formattedArr.push(result.familyName);

    if (typeof result.honorificSuffix == 'undefined')
        result.honorificSuffix = "";
    else if (result.honorificSuffix !== "")
        formattedArr.push(result.honorificSuffix);

    return result;
}

ContactName.prototype = {
    formatted: "",
    familyName: "",
    givenName: "",
    middleName: "",
    honorificPrefix: "",
    honorificSuffix: ""
};

function ContactOrganization(pref, type, name, department, title) {
    if (pref)
        this.pref = pref;
    if (type)
        this.type = type;
    if (name)
        this.name = name;
    if (department)
        this.department = department;
    if (title)
        this.title = title;
}

ContactOrganization.create = function(obj) {
    var result = new ContactOrganization()
    result.pref = obj.pref
    result.type = obj.type
    result.name = obj.name
    result.department = obj.department
    result.title = obj.title
    return result
}

ContactOrganization.prototype = {
    pref: false,
    type: "",
    name: "",
    department: "",
    title: ""
};

function ContactError(code) {
    if (code)
        this.code = code;
}

ContactError.UNKNOWN_ERROR = 0;
ContactError.INVALID_ARGUMENT_ERROR = 1;
ContactError.TIMEOUT_ERROR = 2;
ContactError.PENDING_OPERATION_ERROR = 3;
ContactError.IO_ERROR = 4;
ContactError.NOT_SUPPORTED_ERROR = 5;
ContactError.PERMISSION_DENIED_ERROR = 20;

ContactError.prototype.code = ContactError.UNKNOWN_ERROR

function Contact(p_id, p_displayName, p_name, p_nickname,
                 p_phoneNumbers,
                 p_emails,
                 p_addresses,
                 p_ims,
                 p_organizations,
                 p_birthday,
                 p_note,
                 p_photos,
                 p_categories,
                 p_urls) {  
    this.id = p_id || "";
    this.displayName = p_displayName || "";

    if (typeof(p_name) == 'object') {
        this.name = ContactName.create(p_name);
    } else {
        this.name = new ContactName();
    }

    this.nickname = p_nickname || "";
    this.phoneNumbers = p_phoneNumbers || null;
    this.emails = p_emails || null;
    this.addresses = p_addresses || null;
    this.ims = p_ims || null;
    this.organizations = p_organizations || null;
    this.birthday = p_birthday || "";
    this.note = p_note || "";
    this.photos = p_photos || null;
    this.categories = p_categories || null;
    this.urls = p_urls || null;
}

Contact.create = function(obj) {
    var result = new Contact();
    result.id = obj.id;
    result.displayName = obj.displayName;
    result.name = ContactName.create(obj.name);
    result.nickname = obj.nickname || null;
    var subObj;
    if (obj.phoneNumbers)
        result.phoneNumbers = [];
    for (subObj in obj.phoneNumbers)
        result.phoneNumbers.push(ContactField.create(obj.phoneNumbers[subObj]));
    if (obj.emails)
        result.emails = [];
    for (subObj in obj.emails)
        result.emails.push(ContactField.create(obj.emails[subObj]));
    if (obj.addresses)
        result.addresses = [];
    for (subObj in obj.addresses)
        result.addresses.push(ContactAddress.create(obj.addresses[subObj]));
    if (obj.ims)
        result.ims = [];
    for (subObj in obj.ims)
        result.ims.push(ContactField.create(obj.ims[subObj]));
    if (obj.organizations)
        result.organizations = [];
    for (subObj in obj.organizations)
        result.organizations.push(ContactOrganization.create(obj.organizations[subObj]));
    result.birthday = obj.birthday;
    result.note = obj.note;
    result.gender = obj.gender;
    if (obj.photos)
        result.photos = [];
    for (subObj in obj.photos)
        result.photos.push(ContactField.create(obj.photos[subObj]));
    if (obj.categories)
        result.categories = [];
    for (subObj in obj.categories)
        result.categories.push(ContactField.create(obj.categories[subObj]));
    if (obj.urls)
        result.urls = [];
    for (subObj in obj.urls)
        result.urls.push(ContactField.create(obj.urls[subObj]));

    return result;
}

Contact.prototype = {
    id: null,
    displayName: "",
    nickname: "",
    note: ""
};

Contact.prototype.clone = function() {
    var newContact = Contact.create(this);
    newContact.id = null;
    return newContact;
}

Contact.prototype.remove = function(contactSuccess, contactError) {
    if (typeof(contactSuccess) !== "function") contactSuccess = function() {}
    if (typeof(contactError) !== "function") contactError = function() {}

    if (!this.id) {
        contactError(new ContactError(ContactError.UNKNOWN_ERROR));
        return;
    }

    Cordova.exec(contactSuccess, contactError, "com.cordova.Contacts", "removeContact", [ this.id ]);
}

Contact.prototype.save = function(contactSuccess,contactError) {
    if (typeof(contactSuccess) !== "function") contactSuccess = function() {}
    if (typeof(contactError) !== "function") contactError = function() {}
    var clone = this.clone();
    clone.id = this.id;
    if (clone.birthday)
        clone.birthday = clone.birthday.getTime();
    Cordova.exec(contactSuccess, contactError, "com.cordova.Contacts", "saveContact", [ clone ]);
}

function ContactsManager() {
}

ContactsManager.prototype.create = function(properties) {
    return Contact.create(properties);
}

ContactsManager.prototype.find = function(contactFields, contactSuccess, contactError, contactFindOptions) {
    if (typeof contactSuccess !== "function") {throw "no callback";}
    if (typeof contactError !== "function") {
        contactError = function() {}
    }
    Cordova.exec(contactSuccess, contactError, "com.cordova.Contacts", "findContacts", [ contactFields, contactFindOptions.filter, contactFindOptions.multiple ])
}

Cordova.addConstructor("com.cordova.Contacts", function () {
    navigator.contacts = new ContactsManager()
    window.ContactFindOptions = ContactFindOptions;
    window.Contact = Contact;
    window.ContactName = ContactName;
    window.ContactField = ContactField;
    window.ContactAddress = ContactAddress;
    window.ContactOrganization = ContactOrganization;
    window.ContactError = ContactError;
});
