# -*- coding: UTF-8 -*-

#    Copyright (C) 2008-2010 Ivan Garcia <contact@ivangarcia.org>
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License along
#    with this program; if not, write to the Free Software Foundation, Inc.,
#    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

"""
   This file contains a list of language and country names.

   >>> languages == getLanguages()
   True

   >>> len(languages) == len(getNativeLanguageNames())
   True

   >>> combined == getCombined()
   True

   >>> len(combined) == len(getCombinedLanguageNames())
   True

   >>> len(countries) == len(getCountries())
   True
"""
def locale2name(locale):
    languages = getLanguages()
    if not languages.has_key(locale):
        languages = getCombined()
        locale = locale.replace('_','-').lower()
        if not languages.has_key(locale):
            print locale,  "not found locale into table"
            return ''

    name = languages[locale]['english']
    if languages[locale].has_key('native'):
        native = languages[locale]['native']
        name += '/ %s' % native
    return name

def getLanguages():
    """ Get all languages"""
    return languages.copy()

def getNativeLanguageNames():
    """ Get all native language names"""
    native_languages = {}
    for lc in languages:
        native_languages[lc] = languages.get(lc).get('native')
    return native_languages

def getCombined():
    """ Get all combined languages"""
    return combined.copy()

def getCombinedLanguageNames():
    """ Get all combined language names"""
    combined_languages = {}
    for lc in combined:
        combined_languages[lc] = combined.get(lc).get('english')
    return combined_languages

def getCountries():
    """ Get all countries"""
    return countries

# This is a dictionary of dictonaries:
#
# 'langcode-variation' : {native : 'Native name', english : 'English name', flag : 'flag-*.gif'}
#
# The current structure is to have language codes following ISO639-1 and to
# have flags which are named according to country codes following ISO3166-1.
#

languages = {
'aa' : {'native' : u'магIарул мацI', 'english' : u'Afar', 'flag' : 'flag-dj.gif'},
'ab' : {'native' : u'бызшәа', 'english' : u'Abkhazian', 'flag' : 'flag-ge.gif'},
'af' : {'native' : u'Afrikaans', 'english' : u'Afrikaans', 'flag' : 'flag-za.gif'},
'am' : {'native' : u'አማርኛ', 'english' : u'Amharic', 'flag' : 'flag-et.gif'},
'ar' : {'native' : u'العربية', 'english' : u'Arabic', 'flag' : 'flag-ly.gif'},
'as' : {'native' : u'অসমিয়া', 'english' : u'Assamese'},
'ay' : {'native' : u'Aymara', 'english' : u'Aymara', 'flag' : 'flag-bo.gif'},
'az' : {'native' : u'Azəri Türkçəsi', 'english' : u'Azerbaijani', 'flag' : 'flag-az.gif'},
'ba' : {'native' : u'Bashkir', 'english' : u'Bashkir'},
'be' : {'native' : u'Беларускі', 'english' : u'Belarussian', 'flag' : 'flag-by.gif'},
'bg' : {'native' : u'Български', 'english' : u'Bulgarian', 'flag' : 'flag-bg.gif'},
'bh' : {'native' : u'Bihari', 'english' : u'Bihari'},
'bi' : {'native' : u'Bislama', 'english' : u'Bislama', 'flag' : 'flag-vu.gif'},
'bn' : {'native' : u'বাংলা', 'english' : u'Bengali'},
'bo' : {'native' : u'བོད་སྐད་', 'english' : u'Tibetan'},
'bs' : {'native' : u'Bosanski', 'english' : u'Bosnian', 'flag' : 'flag-ba.gif'},
'br' : {'native' : u'Brezhoneg', 'english' : u'Breton'},
'ca' : {'native' : u'Català', 'english' : u'Catalan', 'flag' : 'flag-cat.gif'},
'ch' : {'native' : u'Chamoru', 'english' : u'Chamorro'},
'co' : {'native' : u'Corsu', 'english' : u'Corsican'},
'cs' : {'native' : u'Čeština', 'english' : u'Czech', 'flag' : 'flag-cz.gif'},
'cy' : {'native' : u'Cymraeg', 'english' : u'Welsh', 'flag' : 'flag-cym.gif'},
'da' : {'native' : u'Dansk', 'english' : u'Danish', 'flag' : 'flag-dk.gif'},
'de' : {'native' : u'Deutsch', 'english' : u'German', 'flag' : 'flag-de.gif'},
'dz' : {'native' : u'རྫོང་ཁ', 'english' : u'Bhutani', 'flag' : 'flag-bt.gif'},
'el' : {'native' : u'Ελληνικά', 'english' : u'Greek', 'flag' : 'flag-gr.gif'},
'en' : {'native' : u'English', 'english' : u'English', 'flag' : 'flag-gb.gif'},
'eo' : {'native' : u'Esperanto', 'english' : u'Esperanto', 'flag' : 'flag-eo.gif'},
'es' : {'native' : u'Español', 'english' : u'Spanish', 'flag' : 'flag-es.gif'},
'et' : {'native' : u'Eesti', 'english' : u'Estonian', 'flag' : 'flag-ee.gif'},
'eu' : {'native' : u'Euskara', 'english' : u'Basque', 'flag' : 'flag-eu.gif'},
'fa' : {'native' : u'فارسی', 'english' : u'Persian', 'flag' : 'flag-ir.gif'},
'fi' : {'native' : u'Suomi', 'english' : u'Finnish', 'flag' : 'flag-fi.gif'},
'fj' : {'native' : u'Fiji', 'english' : u'Fiji', 'flag' : 'flag-fj.gif'},
'fo' : {'native' : u'Føroyska', 'english' : u'Faroese', 'flag' : 'flag-fj.gif'},
'fr' : {'native' : u'Français', 'english' : u'French', 'flag' : 'flag-fr.gif'},
'fy' : {'native' : u'Frysk', 'english' : u'Frisian'},
'ga' : {'native' : u'Gaeilge', 'english' : u'Irish Gaelic', 'flag' : 'flag-ie.gif'},
'gd' : {'native' : u'Gàidhlig', 'english' : u'Scottish Gaelic'},
'gl' : {'native' : u'Galego', 'english' : u'Galician'},
'gn' : {'native' : u'Guarani', 'english' : u'Guarani', 'flag' : 'flag-py.gif'},
'gu' : {'native' : u'ગુજરાતી', 'english' : u'Gujarati'},
'gv' : {'native' : u'Gaelg', 'english' : u'Manx Gaelic'},
'ha' : {'native' : u'هَوُس', 'english' : u'Hausa'},
'he' : {'native' : u'עברית', 'english' : u'Hebrew', 'flag' : 'flag-il.gif'},
'hi' : {'native' : u'हिंदी', 'english' : u'Hindi', 'flag' : 'flag-in.gif'},
'hr' : {'native' : u'Hrvatski', 'english' : u'Croatian', 'flag' : 'flag-hr.gif'},
'hu' : {'native' : u'Magyar', 'english' : u'Hungarian', 'flag' : 'flag-hu.gif'},
'hy' : {'native' : u'Հայերէն', 'english' : u'Armenian', 'flag' : 'flag-am.gif'},
'ia' : {'native' : u'Interlingua', 'english' : u'Interlingua'},
'id' : {'native' : u'Bahasa Indonesia', 'english' : u'Indonesian', 'flag' : 'flag-id.gif'},
'ie' : {'native' : u'Interlingue', 'english' : u'Interlingue'},
'ik' : {'native' : u'Inupiak', 'english' : u'Inupiak'},
'is' : {'native' : u'Íslenska', 'english' : u'Icelandic', 'flag' : 'flag-is.gif'},
'it' : {'native' : u'Italiano', 'english' : u'Italian', 'flag' : 'flag-it.gif'},
'iu' : {'native' : u'ᐃᓄᒃᑎᑐᑦ', 'english' : u'Inuktitut', 'flag' : 'flag-it.gif'},
'ja' : {'native' : u'日本語', 'english' : u'Japanese', 'flag' : 'flag-jp.gif'},
'jbo': {'native' : u'lojban', 'english' : u'Lojban'},
'jw' : {'native' : u'Basa Jawi', 'english' : u'Javanese', 'flag' : 'flag-id.gif'},
'ka' : {'native' : u'ქართული', 'english' : u'Georgian', 'flag' : 'flag-ge.gif'},
'kk' : {'native' : u'ﻗﺎﺯﺍﻗﺸﺎ', 'english' : u'Kazakh', 'flag' : 'flag-kz.gif'},
'kl' : {'native' : u'Greenlandic', 'english' : u'Greenlandic', 'flag' : 'flag-gl.gif'},
'km' : {'native' : u'ខ្មែរ', 'english' : u'Cambodian/Khmer', 'flag' : 'flag-kh.gif'},
'kn' : {'native' : u'ಕನ್ನಡ', 'english' : u'Kannada', 'flag' : 'flag-in.gif'},
'ko' : {'native' : u'한국어', 'english' : u'Korean', 'flag' : 'flag-kr.gif'},
'ks' : {'native' : u'काऽशुर', 'english' : u'Kashmiri', 'flag' : 'flag-in.gif'},
'ku' : {'native' : u'Kurdí', 'english' : u'Kurdish'},
'kw' : {'native' : u'Kernewek', 'english' : u'Cornish', 'flag' : 'flag-con.gif'},
'ky' : {'native' : u'Кыргыз', 'english' : u'Kyrgyz'},
'la' : {'native' : u'Latin', 'english' : u'Latin', 'flag' : 'flag-va.gif'},
'lb' : {'native' : u'Lëtzebuergesch', 'english' : u'Luxemburgish', 'flag' : 'flag-lu.gif'},
'li' : {'native' : u'Limburgs', 'english' : u'Limburgish'},
'ln' : {'native' : u'Lingala', 'english' : u'Lingala'},
'lo' : {'native' : u'ພາສາລາວ', 'english' : u'Laotian', 'flag' : 'flag-la.gif'},
'lt' : {'native' : u'Lietuviskai', 'english' : u'Lithuanian', 'flag' : 'flag-lt.gif'},
'lv' : {'native' : u'Latviešu', 'english' : u'Latvian'},
'mg' : {'native' : u'Malagasy', 'english' : u'Madagascarian', 'flag' : 'flag-mg.gif'},
'mi' : {'native' : u'Maori', 'english' : u'Maori', 'flag' : 'flag-nz.gif'},
'mk' : {'native' : u'Македонски', 'english' : u'Macedonian', 'flag' : 'flag-mk.gif'},
'ml' : {'native' : u'മലയാളം', 'english' : u'Malayalam'},
'mn' : {'native' : u'Монгол', 'english' : u'Mongolian', 'flag' : 'flag-mn.gif'},
'mo' : {'native' : u'Moldavian', 'english' : u'Moldavian', 'flag' : 'flag-md.gif'},
'mr' : {'native' : u'मराठी', 'english' : u'Marathi'},
'ms' : {'native' : u'Bahasa Melayu', 'english' : u'Malay', 'flag' : 'flag-my.gif'},
'mt' : {'native' : u'Malti', 'english' : u'Maltese', 'flag' : 'flag-mt.gif'},
'my' : {'native' : u'Burmese', 'english' : u'Burmese', 'flag' : 'flag-my.gif'},
'na' : {'native' : u'Nauru', 'english' : u'Nauruan', 'flag' : 'flag-nr.gif'},
'ne' : {'native' : u'नेपाली', 'english' : u'Nepali'},
'nb' : {'native' : u'Bokmål', 'english' : u'Norwegian Bokmal', 'flag' : 'flag-no.gif'},
'nl' : {'native' : u'Nederlands', 'english' : u'Dutch', 'flag' : 'flag-nl.gif'},
'no' : {'native' : u'Norsk', 'english' : u'Norwegian', 'flag' : 'flag-no.gif'},
'nn' : {'native' : u'Nynorsk', 'english' : u'Norwegian Nynorsk', 'flag' : 'flag-no.gif'},
'oc' : {'native' : u'Languedoc', 'english' : u'Occitan', 'flag' : 'flag-fr.gif'},
'om' : {'native' : u'Oromo', 'english' : u'Oromo'},
'or' : {'native' : u'ଓଡ଼ିଆ', 'english' : u'Oriya'},
'pa' : {'native' : u'ਪੰਜਾਬੀ', 'english' : u'Punjabi'},
'pl' : {'native' : u'Polski', 'english' : u'Polish', 'flag' : 'flag-pl.gif'},
'ps' : {'native' : u'پښتو', 'english' : u'Pashto'},
'pt' : {'native' : u'Português', 'english' : u'Portuguese', 'flag' : 'flag-pt.gif'},
'qu' : {'native' : u'Runa Simi', 'english' : u'Quechua'},
'rm' : {'native' : u'Rumantsch', 'english' : u'Rhaeto-Romance'},
'rn' : {'native' : u'Kirundi', 'english' : u'Kirundi'},
'ro' : {'native' : u'Română', 'english' : u'Romanian', 'flag' : 'flag-ro.gif'},
'ru' : {'native' : u'Русский', 'english' : u'Russian', 'flag' : 'flag-ru.gif'},
'rw' : {'native' : u'Kiyarwanda', 'english' : u'Kiyarwanda'},
'sa' : {'native' : u'संस्कृत', 'english' : u'Sanskrit', 'flag' : 'flag-in.gif'},
'sd' : {'native' : u'Sindhi', 'english' : u'Sindhi', 'flag' : 'flag-pk.gif'},
'se' : {'native' : u'Northern Sámi', 'english' : u'Northern Sámi'},
'sg' : {'native' : u'Sangho', 'english' : u'Sangho', 'flag' : 'flag-cf.gif'},
'sh' : {'native' : u'Serbo-Croatian', 'english' : u'Serbo-Croatian'},
'si' : {'native' : u'Singhalese', 'english' : u'Singhalese'},
'sk' : {'native' : u'Slovenčina', 'english' : u'Slovak', 'flag' : 'flag-sk.gif'},
'sl' : {'native' : u'Slovenščina', 'english' : u'Slovenian', 'flag' : 'flag-si.gif'},
'sm' : {'native' : u'Samoan', 'english' : u'Samoan'},
'sn' : {'native' : u'Shona', 'english' : u'Shona'},
'so' : {'native' : u'Somali', 'english' : u'Somali', 'flag' : 'flag-so.gif'},
'sq' : {'native' : u'Shqip', 'english' : u'Albanian', 'flag' : 'flag-al.gif'},
'sr' : {'native' : u'српски', 'english' : u'Serbian', 'flag' : 'flag-cs.gif'},
'ss' : {'native' : u'Siswati', 'english' : u'Siswati'},
'st' : {'native' : u'Sesotho', 'english' : u'Sesotho'},
'su' : {'native' : u'Sudanese', 'english' : u'Sudanese', 'flag' : 'flag-sd.gif'},
'sv' : {'native' : u'Svenska', 'english' : u'Swedish', 'flag' : 'flag-se.gif'},
'sw' : {'native' : u'Kiswahili', 'english' : u'Swahili'},
'ta' : {'native' : u'தமிழ', 'english' : u'Tamil'},
'te' : {'native' : u'తెలుగు', 'english' : u'Telugu'},
'tg' : {'native' : u'Тоҷики', 'english' : u'Tadjik', 'flag' : 'flag-tj.gif'},
'th' : {'native' : u'ไทย', 'english' : u'Thai', 'flag' : 'flag-th.gif'},
'ti' : {'native' : u'ትግርኛ', 'english' : u'Tigrinya'},
'tk' : {'native' : u'түркmенче', 'english' : u'Turkmen', 'flag' : 'flag-tm.gif'},
'tl' : {'native' : u'Tagalog', 'english' : u'Tagalog', 'flag' : 'flag-ph.gif'},
'tn' : {'native' : u'Setswana', 'english' : u'Setswana', 'flag' : 'flag-bw.gif'},
'to' : {'native' : u'Lea faka-Tonga', 'english' : u'Tongan'},
'tr' : {'native' : u'Türkçe', 'english' : u'Turkish', 'flag' : 'flag-tr.gif'},
'ts' : {'native' : u'Tsonga', 'english' : u'Tsonga'},
'tt' : {'native' : u'татарча', 'english' : u'Tatar'},
'tw' : {'native' : u'Twi', 'english' : u'Twi'},
'ug' : {'native' : u'Uigur', 'english' : u'Uigur'},
'uk' : {'native' : u'Українська', 'english' : u'Ukrainian', 'flag' : 'flag-ua.gif'},
'ur' : {'native' : u'اردو', 'english' : u'Urdu', 'flag' : 'flag-pk.gif'},
'uz' : {'native' : u'Ўзбекча', 'english' : u'Uzbek', 'flag' : 'flag-uz.gif'},
'vi' : {'native' : u'Tiếng Việt', 'english' : u'Vietnamese', 'flag' : 'flag-vn.gif'},
'vo' : {'native' : u'Volapük', 'english' : u'Volapük'},
'wa' : {'native' : u'Walon', 'english' : u'Walloon'},
'wo' : {'native' : u'Wolof', 'english' : u'Wolof'},
'xh' : {'native' : u'isiXhosa', 'english' : u'Xhosa', 'flag' : 'flag-za.gif'},
'yi' : {'native' : u'ײִדיש', 'english' : u'Yiddish', 'flag' : 'flag-il.gif'},
'yo' : {'native' : u'Yorùbá', 'english' : u'Yorouba'},
'za' : {'native' : u'Zhuang', 'english' : u'Zhuang'},
'zh' : {'native' : u'中文', 'english' : u'Chinese', 'flag' : 'flag-cn.gif'},
'zu' : {'native' : u'isiZulu', 'english' : u'Zulu'}
}

combined = {
'ar-ae' : {'english' : u'Arabic (United Arab Emirates)', 'flag' : 'flag-ae.gif'},
'ar-bh' : {'english' : u'Arabic (Bahrain)', 'flag' : 'flag-bh.gif'},
'ar-dz' : {'english' : u'Arabic (Algeria)', 'flag' : 'flag-dz.gif'},
'ar-eg' : {'english' : u'Arabic (Egypt)', 'flag' : 'flag-eg.gif'},
'ar-il' : {'english' : u'Arabic (Israel)', 'flag' : 'flag-il.gif'},
'ar-iq' : {'english' : u'Arabic (Iraq)', 'flag' : 'flag-iq.gif'},
'ar-jo' : {'english' : u'Arabic (Jordan)', 'flag' : 'flag-jo.gif'},
'ar-kw' : {'english' : u'Arabic (Kuwait)', 'flag' : 'flag-kw.gif'},
'ar-lb' : {'english' : u'Arabic (Lebanon)', 'flag' : 'flag-lb.gif'},
'ar-ly' : {'english' : u'Arabic (Libya)', 'flag' : 'flag-ly.gif'},
'ar-ma' : {'english' : u'Arabic (Morocco)', 'flag' : 'flag-ma.gif'},
'ar-mr' : {'english' : u'Arabic (Mauritania)', 'flag' : 'flag-mr.gif'},
'ar-om' : {'english' : u'Arabic (Oman)', 'flag' : 'flag-om.gif'},
'ar-ps' : {'english' : u'Arabic (Palestinian West Bank and Gaza)'},
'ar-qa' : {'english' : u'Arabic (Qatar)', 'flag' : 'flag-qa.gif'},
'ar-sa' : {'english' : u'Arabic (Saudi Arabia)', 'flag' : 'flag-sa.gif'},
'ar-sd' : {'english' : u'Arabic (Sudan)', 'flag' : 'flag-sd.gif'},
'ar-so' : {'english' : u'Arabic (Somalia)', 'flag' : 'flag-so.gif'},
'ar-sy' : {'english' : u'Arabic (Syria)', 'flag' : 'flag-sy.gif'},
'ar-td' : {'english' : u'Arabic (Chad)', 'flag' : 'flag-td.gif'},
'ar-tn' : {'english' : u'Arabic (Tunisia)', 'flag' : 'flag-tn.gif'},
'ar-ye' : {'english' : u'Arabic (Yemen)', 'flag' : 'flag-ye.gif'},
'bn-bd' : {'english' : u'Bengali (Bangladesh)', 'flag' : 'flag-bd.gif'},
'bn-in' : {'english' : u'Bengali (India)', 'flag' : 'flag-in.gif'},
'bn-sg' : {'english' : u'Bengali (Singapore)', 'flag' : 'flag-sg.gif'},
'ch-gu' : {'english' : u'Chamorro (Guam)', 'flag' : 'flag-gu.gif'},
'ch-mp' : {'english' : u'Chamorro (Northern Mariana Islands)', 'flag' : 'flag-mp.gif'},
'da-dk' : {'english' : u'Danish (Denmark)', 'flag' : 'flag-dk.gif'},
'da-gl' : {'english' : u'Danish (Greenland)', 'flag' : 'flag-gl.gif'},
'de-at' : {'english' : u'German (Austria)', 'native' : u'Deutsch (Österreich)',  'flag' : 'flag-at.gif'},
'de-be' : {'english' : u'German (Belgium)', 'flag' : 'flag-be.gif'},
'de-ch' : {'english' : u'German (Switzerland)', 'flag' : 'flag-ch.gif'},
'de-de' : {'english' : u'German (Germany)', 'flag' : 'flag-de.gif'},
'de-dk' : {'english' : u'German (Denmark)', 'flag' : 'flag-dk.gif'},
'de-li' : {'english' : u'German (Liechtenstein)', 'flag' : 'flag-li.gif'},
'de-lu' : {'english' : u'German (Luxembourg)', 'flag' : 'flag-lu.gif'},
'el-cy' : {'english' : u'Greek (Cyprus)'},
'el-gr' : {'english' : u'Greek (Greece)', 'flag' : 'flag-gr.gif'},
'en-ag' : {'english' : u'English (Antigua and Barbuda)', 'flag' : 'flag-ag.gif'},
'en-ai' : {'english' : u'English (Anguilla)', 'flag' : 'flag-ai.gif'},
'en-as' : {'english' : u'English (American Samoa)', 'flag' : 'flag-as.gif'},
'en-au' : {'english' : u'English (Australia)', 'flag' : 'flag-au.gif'},
'en-bb' : {'english' : u'English (Barbados)', 'flag' : 'flag-bb.gif'},
'en-bm' : {'english' : u'English (Bermuda)', 'flag' : 'flag-bm.gif'},
'en-bn' : {'english' : u'English (Brunei)', 'flag' : 'flag-bn.gif'},
'en-bs' : {'english' : u'English (Bahamas)', 'flag' : 'flag-bs.gif'},
'en-bw' : {'english' : u'English (Botswana)', 'flag' : 'flag-bw.gif'},
'en-bz' : {'english' : u'English (Belize)', 'flag' : 'flag-bz.gif'},
'en-ca' : {'english' : u'English (Canada)', 'flag' : 'flag-ca.gif'},
'en-ck' : {'english' : u'English (Cook Islands)', 'flag' : 'flag-ck.gif'},
'en-cm' : {'english' : u'English (Cameroon)', 'flag' : 'flag-cm.gif'},
'en-dm' : {'english' : u'English (Dominica)', 'flag' : 'flag-dm.gif'},
'en-er' : {'english' : u'English (Eritrea)', 'flag' : 'flag-er.gif'},
'en-et' : {'english' : u'English (Ethiopia)', 'flag' : 'flag-et.gif'},
'en-fj' : {'english' : u'English (Fiji)', 'flag' : 'flag-fj.gif'},
'en-fk' : {'english' : u'English (Falkland Islands)', 'flag' : 'flag-fk.gif'},
'en-fm' : {'english' : u'English (Micronesia)', 'flag' : 'flag-fm.gif'},
'en-gb' : {'english' : u'English (United Kingdom)', 'flag' : 'flag-gb.gif'},
'en-gd' : {'english' : u'English (Grenada)', 'flag' : 'flag-gd.gif'},
'en-gh' : {'english' : u'English (Ghana)', 'flag' : 'flag-gh.gif'},
'en-gi' : {'english' : u'English (Gibraltar)', 'flag' : 'flag-gi.gif'},
'en-gm' : {'english' : u'English (Gambia)', 'flag' : 'flag-gm.gif'},
'en-gu' : {'english' : u'English (Guam)', 'flag' : 'flag-gu.gif'},
'en-gy' : {'english' : u'English (Guyana)', 'flag' : 'flag-gy.gif'},
'en-ie' : {'english' : u'English (Ireland)', 'flag' : 'flag-ie.gif'},
'en-il' : {'english' : u'English (Israel)', 'flag' : 'flag-il.gif'},
'en-io' : {'english' : u'English (British Indian Ocean Territory)', 'flag' : 'flag-io.gif'},
'en-jm' : {'english' : u'English (Jamaica)', 'flag' : 'flag-jm.gif'},
'en-ke' : {'english' : u'English (Kenya)', 'flag' : 'flag-ke.gif'},
'en-ki' : {'english' : u'English (Kiribati)', 'flag' : 'flag-ki.gif'},
'en-kn' : {'english' : u'English (St. Kitts-Nevis)', 'flag' : 'flag-kn.gif'},
'en-ky' : {'english' : u'English (Cayman Islands)', 'flag' : 'flag-ky.gif'},
'en-lc' : {'english' : u'English (St. Lucia)', 'flag' : 'flag-lc.gif'},
'en-lr' : {'english' : u'English (Liberia)', 'flag' : 'flag-lr.gif'},
'en-ls' : {'english' : u'English (Lesotho)', 'flag' : 'flag-ls.gif'},
'en-mp' : {'english' : u'English (Northern Mariana Islands)', 'flag' : 'flag-mp.gif'},
'en-ms' : {'english' : u'English (Montserrat)', 'flag' : 'flag-ms.gif'},
'en-mt' : {'english' : u'English (Malta)', 'flag' : 'flag-mt.gif'},
'en-mu' : {'english' : u'English (Mauritius)', 'flag' : 'flag-mu.gif'},
'en-mw' : {'english' : u'English (Malawi)', 'flag' : 'flag-mw.gif'},
'en-na' : {'english' : u'English (Namibia)', 'flag' : 'flag-na.gif'},
'en-nf' : {'english' : u'English (Norfolk Island)', 'flag' : 'flag-nf.gif'},
'en-ng' : {'english' : u'English (Nigeria)', 'flag' : 'flag-ng.gif'},
'en-nr' : {'english' : u'English (Nauru)', 'flag' : 'flag-nr.gif'},
'en-nu' : {'english' : u'English (Niue)', 'flag' : 'flag-nu.gif'},
'en-nz' : {'english' : u'English (New Zealand)', 'flag' : 'flag-nz.gif'},
'en-pg' : {'english' : u'English (Papua New Guinea)', 'flag' : 'flag-pg.gif'},
'en-ph' : {'english' : u'English (Philippines)', 'flag' : 'flag-ph.gif'},
'en-pk' : {'english' : u'English (Pakistan)', 'flag' : 'flag-pk.gif'},
'en-pn' : {'english' : u'English (Pitcairn)', 'flag' : 'flag-pn.gif'},
'en-pr' : {'english' : u'English (Puerto Rico)', 'flag' : 'flag-pr.gif'},
'en-pw' : {'english' : u'English (Palau)', 'flag' : 'flag-pw.gif'},
'en-rw' : {'english' : u'English (Rwanda)', 'flag' : 'flag-rw.gif'},
'en-sb' : {'english' : u'English (Solomon Islands)', 'flag' : 'flag-sb.gif'},
'en-sc' : {'english' : u'English (Seychelles)', 'flag' : 'flag-sc.gif'},
'en-sg' : {'english' : u'English (Singapore)', 'flag' : 'flag-sg.gif'},
'en-sh' : {'english' : u'English (St. Helena)', 'flag' : 'flag-sh.gif'},
'en-sl' : {'english' : u'English (Sierra Leone)', 'flag' : 'flag-sl.gif'},
'en-so' : {'english' : u'English (Somalia)', 'flag' : 'flag-so.gif'},
'en-sz' : {'english' : u'English (Swaziland)', 'flag' : 'flag-sz.gif'},
'en-tc' : {'english' : u'English (Turks and Caicos Islands)', 'flag' : 'flag-tc.gif'},
'en-tk' : {'english' : u'English (Tokelau)', 'flag' : 'flag-tk.gif'},
'en-to' : {'english' : u'English (Tonga)', 'flag' : 'flag-to.gif'},
'en-tt' : {'english' : u'English (Trinidad and Tobago)', 'flag' : 'flag-tt.gif'},
'en-ug' : {'english' : u'English (Uganda)', 'flag' : 'flag-ug.gif'},
'en-us' : {'english' : u'English (USA)', 'flag' : 'flag-us.gif'},
'en-vc' : {'english' : u'English (St. Vincent and the Grenadi)', 'flag' : 'flag-vc.gif'},
'en-vg' : {'english' : u'English (British Virgin Islands)', 'flag' : 'flag-vg.gif'},
'en-vi' : {'english' : u'English (U.S. Virgin Islands)', 'flag' : 'flag-vi.gif'},
'en-vu' : {'english' : u'English (Vanuatu)', 'flag' : 'flag-vu.gif'},
'en-ws' : {'english' : u'English (Western Samoa)', 'flag' : 'flag-ws.gif'},
'en-za' : {'english' : u'English (South Africa)', 'flag' : 'flag-za.gif'},
'en-zm' : {'english' : u'English (Zambia)', 'flag' : 'flag-zm.gif'},
'en-zw' : {'english' : u'English (Zimbabwe)', 'flag' : 'flag-zw.gif'},
'es-ar' : {'english' : u'Spanish (Argentina)', 'flag' : 'flag-ar.gif'},
'es-bo' : {'english' : u'Spanish (Bolivia)', 'flag' : 'flag-bo.gif'},
'es-cl' : {'english' : u'Spanish (Chile)', 'flag' : 'flag-cl.gif'},
'es-co' : {'english' : u'Spanish (Colombia)', 'flag' : 'flag-co.gif'},
'es-cr' : {'english' : u'Spanish (Costa Rica)', 'flag' : 'flag-cr.gif'},
'es-cu' : {'english' : u'Spanish (Cuba)', 'flag' : 'flag-cu.gif'},
'es-do' : {'english' : u'Spanish (Dominican Republic)', 'flag' : 'flag-do.gif'},
'es-ec' : {'english' : u'Spanish (Ecuador)', 'flag' : 'flag-ec.gif'},
'es-es' : {'english' : u'Spanish (Spain)', 'flag' : 'flag-es.gif'},
'es-gq' : {'english' : u'Spanish (Equatorial Guinea)', 'flag' : 'flag-gq.gif'},
'es-gt' : {'english' : u'Spanish (Guatemala)', 'flag' : 'flag-gt.gif'},
'es-hn' : {'english' : u'Spanish (Honduras)', 'flag' : 'flag-hn.gif'},
'es-mx' : {'english' : u'Spanish (Mexico)', 'flag' : 'flag-mx.gif'},
'es-ni' : {'english' : u'Spanish (Nicaragua)', 'flag' : 'flag-ni.gif'},
'es-pa' : {'english' : u'Spanish (Panama)', 'flag' : 'flag-pa.gif'},
'es-pe' : {'english' : u'Spanish (Peru)', 'flag' : 'flag-pe.gif'},
'es-pr' : {'english' : u'Spanish (Puerto Rico)', 'flag' : 'flag-pr.gif'},
'es-py' : {'english' : u'Spanish (Paraguay)', 'flag' : 'flag-py.gif'},
'es-sv' : {'english' : u'Spanish (El Salvador)', 'flag' : 'flag-sv.gif'},
'es-us' : {'english' : u'Spanish (USA)', 'flag' : 'flag-us.gif'},
'es-uy' : {'english' : u'Spanish (Uruguay)', 'flag' : 'flag-uy.gif'},
'es-ve' : {'english' : u'Spanish (Venezuela)', 'flag' : 'flag-ve.gif'},
'fr-ad' : {'english' : u'French (Andorra)', 'flag' : 'flag-as.gif'},
'fr-be' : {'english' : u'French (Belgium)', 'flag' : 'flag-be.gif'},
'fr-bf' : {'english' : u'French (Burkina Faso)', 'flag' : 'flag-bf.gif'},
'fr-bi' : {'english' : u'French (Burundi)', 'flag' : 'flag-bi.gif'},
'fr-bj' : {'english' : u'French (Benin)', 'flag' : 'flag-bj.gif'},
'fr-ca' : {'english' : u'French (Canada)', 'flag' : 'flag-ca.gif'},
'fr-cd' : {'english' : u'French (Democratic Republic of Congo)', 'flag' : 'flag-cd.gif'},
'fr-cf' : {'english' : u'French (Central African Republic)', 'flag' : 'flag-cf.gif'},
'fr-cg' : {'english' : u'French (Congo)', 'flag' : 'flag-cg.gif'},
'fr-ch' : {'english' : u'French (Switzerland)', 'flag' : 'flag-ch.gif'},
'fr-ci' : {'english' : u'French (Cote d\'Ivoire)', 'flag' : 'flag-ci.gif'},
'fr-cm' : {'english' : u'French (Cameroon)', 'flag' : 'flag-cm.gif'},
'fr-dj' : {'english' : u'French (Djibouti)', 'flag' : 'flag-dj.gif'},
'fr-fr' : {'english' : u'French (France)', 'flag' : 'flag-fr.gif'},
'fr-ga' : {'english' : u'French (Gabon)', 'flag' : 'flag-ga.gif'},
'fr-gb' : {'english' : u'French (United Kingdom)', 'flag' : 'flag-gb.gif'},
'fr-gf' : {'english' : u'French (French Guiana)', 'flag' : 'flag-gf.gif'},
'fr-gn' : {'english' : u'French (Guinea)', 'flag' : 'flag-gn.gif'},
'fr-gp' : {'english' : u'French (Guadeloupe)', 'flag' : 'flag-gp.gif'},
'fr-ht' : {'english' : u'French (Haiti)', 'flag' : 'flag-ht.gif'},
'fr-it' : {'english' : u'French (Italy)', 'flag' : 'flag-it.gif'},
'fr-km' : {'english' : u'French (Comoros Islands)', 'flag' : 'flag-km.gif'},
'fr-lb' : {'english' : u'French (Lebanon)', 'flag' : 'flag-lb.gif'},
'fr-lu' : {'english' : u'French (Luxembourg)', 'flag' : 'flag-lu.gif'},
'fr-mc' : {'english' : u'French (Monaco)', 'flag' : 'flag-mc.gif'},
'fr-mg' : {'english' : u'French (Madagascar)', 'flag' : 'flag-mg.gif'},
'fr-ml' : {'english' : u'French (Mali)', 'flag' : 'flag-ml.gif'},
'fr-mq' : {'english' : u'French (Martinique)', 'flag' : 'flag-mq.gif'},
'fr-nc' : {'english' : u'French (New Caledonia)', 'flag' : 'flag-nc.gif'},
'fr-pf' : {'english' : u'French (French Polynesia)', 'flag' : 'flag-pf.gif'},
'fr-pm' : {'english' : u'French (St. Pierre and Miquelon)', 'flag' : 'flag-pm.gif'},
'fr-re' : {'english' : u'French (Reunion)', 'flag' : 'flag-re.gif'},
'fr-rw' : {'english' : u'French (Rwanda)', 'flag' : 'flag-rw.gif'},
'fr-sc' : {'english' : u'French (Seychelles)', 'flag' : 'flag-sc.gif'},
'fr-td' : {'english' : u'French (Chad)', 'flag' : 'flag-td.gif'},
'fr-tg' : {'english' : u'French (Togo)', 'flag' : 'flag-tg.gif'},
'fr-vu' : {'english' : u'French (Vanuatu)', 'flag' : 'flag-vu.gif'},
'fr-wf' : {'english' : u'French (Wallis and Futuna)', 'flag' : 'flag-wf.gif'},
'fr-yt' : {'english' : u'French (Mayotte)', 'flag' : 'flag-yt.gif'},
'hr-ba' : {'english' : u'Croatian (Bosnia-Herzegovina)', 'flag' : 'flag-ba.gif'},
'hr-hr' : {'english' : u'Croatian (Croatia)', 'flag' : 'flag-hr.gif'},
'hu-hu' : {'english' : u'Hungarian (Hungary)', 'flag' : 'flag-hu.gif'},
'hu-si' : {'english' : u'Hungarian (Slovenia)', 'flag' : 'flag-si.gif'},
'it-ch' : {'english' : u'Italian (Switzerland)', 'flag' : 'flag-ch.gif'},
'it-hr' : {'english' : u'Italian (Croatia)', 'flag' : 'flag-hr.gif'},
'it-it' : {'english' : u'Italian (Italy)', 'flag' : 'flag-it.gif'},
'it-si' : {'english' : u'Italian (Slovenia)', 'flag' : 'flag-si.gif'},
'it-sm' : {'english' : u'Italian (San Marino)', 'flag' : 'flag-sm.gif'},
'ko-kp' : {'english' : u'Korean (Korea, North)', 'flag' : 'flag-kp.gif'},
'ko-kr' : {'english' : u'Korean (Korea, South)', 'flag' : 'flag-kr.gif'},
'ln-cd' : {'english' : u'Lingala (Democratic Republic of Congo)', 'flag' : 'flag-cd.gif'},
'ln-cg' : {'english' : u'Lingala (Congo)', 'flag' : 'flag-cg.gif'},
'ms-bn' : {'english' : u'Malay (Brunei)', 'flag' : 'flag-bn.gif'},
'ms-my' : {'english' : u'Malay (Malaysia)', 'flag' : 'flag-my.gif'},
'ms-sg' : {'english' : u'Malay (Singapore)', 'flag' : 'flag-sg.gif'},
'nl-an' : {'english' : u'Dutch (Netherlands Antilles)', 'flag' : 'flag-an.gif'},
'nl-aw' : {'english' : u'Dutch (Aruba)', 'flag' : 'flag-aw.gif'},
'nl-be' : {'english' : u'Dutch (Belgium)', 'flag' : 'flag-be.gif'},
'nl-nl' : {'english' : u'Dutch (Netherlands)', 'flag' : 'flag-nl.gif'},
'nl-sr' : {'english' : u'Dutch (Suriname)', 'flag' : 'flag-sr.gif'},
'pt-ao' : {'english' : u'Português (Angola)', 'flag' : 'flag-ao.gif'},
'pt-br' : {'native'  : 'Português (Brasil)', 'english' : u'Brazilian Portuguese', 'flag' : 'flag-br.gif'},
'pt-cv' : {'english' : u'Português (Ilhas Cabo Verde)', 'flag' : 'flag-cv.gif'},
'pt-gw' : {'english' : u'Português (Guiné-Bissau)', 'flag' : 'flag-gw.gif'},
'pt-mz' : {'english' : u'Português (Moçambique)', 'flag' : 'flag-mz.gif'},
'pt-pt' : {'english' : u'Português (Portugal)', 'flag' : 'flag-pt.gif'},
'pt-st' : {'english' : u'Português (São Tomé e Príncipe)', 'flag' : 'flag-st.gif'},
'sd-in' : {'english' : u'Sindhi (India)', 'flag' : 'flag-in.gif'},
'sd-pk' : {'english' : u'Sindhi (Pakistan)', 'flag' : 'flag-pk.gif'},
'sr-ba' : {'english' : u'Serbian (Bosnia-Herzegovina)', 'flag' : 'flag-ba.gif'},
'sr-yu' : {'english' : u'Serbian (Yugoslavia)', 'flag' : 'flag-yu.gif'},
'ss-sz' : {'english' : u'Swati (Swaziland)', 'flag' : 'flag-sz.gif'},
'ss-za' : {'english' : u'Swati (South Africa)', 'flag' : 'flag-za.gif'},
'sv-fi' : {'english' : u'Swedish (Finland)', 'flag' : 'flag-fi.gif'},
'sv-se' : {'english' : u'Swedish (Sweden)', 'flag' : 'flag-se.gif'},
'sw-ke' : {'english' : u'Swahili (Kenya)', 'flag' : 'flag-ke.gif'},
'sw-tz' : {'english' : u'Swahili (Tanzania)', 'flag' : 'flag-tz.gif'},
'ta-in' : {'english' : u'Tamil (India)', 'flag' : 'flag-in.gif'},
'ta-sg' : {'english' : u'Tamil (Singapore)', 'flag' : 'flag-sg.gif'},
'tn-bw' : {'english' : u'Tswana (Botswana)', 'flag' : 'flag-bw.gif'},
'tn-za' : {'english' : u'Tswana (South Africa)', 'flag' : 'flag-za.gif'},
'tr-bg' : {'english' : u'Turkish (Bulgaria)', 'flag' : 'flag-bg.gif'},
'tr-cy' : {'english' : u'Turkish (Cyprus)', 'flag' : 'flag-cy.gif'},
'tr-tr' : {'english' : u'Turkish (Turkey)', 'flag' : 'flag-tr.gif'},
'ur-in' : {'english' : u'Urdu (India)', 'flag' : 'flag-in.gif'},
'ur-pk' : {'english' : u'Urdu (Pakistan)', 'flag' : 'flag-ur.gif'},
'zh-cn' : {'english' : u'Chinese (China)', 'native' : u'简体中文(中国)',  'flag' : 'flag-cn.gif'},
'zh-hk' : {'english' : u'Chinese (Hongkong)', 'native' : u'繁體中文(香港)',  'flag' : 'flag-hk.gif'},
'zh-sg' : {'english' : u'Chinese (Singapore)', 'native' : u'简体中文(新加坡)',  'flag' : 'flag-sg.gif'},
'zh-tw' : {'english' : u'Chinese (Taiwan)', 'native' : u'繁體中文(臺灣)',  'flag' : 'flag-tw.gif'}
}

# countries list from http://alioth.debian.org/projects/pkg-isocodes/
countries = {
'AD':'Andorra',
'AE':'United Arab Emirates',
'AF':'Afghanistan',
'AG':'Antigua and Barbuda',
'AI':'Anguilla',
'AL':'Albania',
'AM':'Armenia',
'AN':'Netherlands Antilles',
'AO':'Angola',
'AQ':'Antarctica',
'AR':'Argentina',
'AS':'American Samoa',
'AT':'Austria',
'AU':'Australia',
'AW':'Aruba',
'AX':'Åland Islands',
'AZ':'Azerbaijan',
'BA':'Bosnia and Herzegovina',
'BB':'Barbados',
'BD':'Bangladesh',
'BE':'Belgium',
'BF':'Burkina Faso',
'BG':'Bulgaria',
'BH':'Bahrain',
'BI':'Burundi',
'BJ':'Benin',
'BM':'Bermuda',
'BN':'Brunei Darussalam',
'BO':'Bolivia',
'BR':'Brazil',
'BS':'Bahamas',
'BT':'Bhutan',
'BV':'Bouvet Island',
'BW':'Botswana',
'BY':'Belarus',
'BZ':'Belize',
'CA':'Canada',
'CC':'Cocos (Keeling) Islands',
'CD':'Congo, The Democratic Republic of',
'CF':'Central African Republic',
'CG':'Congo',
'CH':'Switzerland',
'CI':"Cote d'Ivoire",
'CK':'Cook Islands',
'CL':'Chile',
'CM':'Cameroon',
'CN':'China',
'CO':'Colombia',
'CR':'Costa Rica',
'CS':'Serbia and Montenegro',
'CU':'Cuba',
'CV':'Cape Verde',
'CX':'Christmas Island',
'CY':'Cyprus',
'CZ':'Czech Republic',
'DE':'Germany',
'DJ':'Djibouti',
'DK':'Denmark',
'DM':'Dominica',
'DO':'Dominican Republic',
'DZ':'Algeria',
'EC':'Ecuador',
'EE':'Estonia',
'EG':'Egypt',
'EH':'Western Sahara',
'ER':'Eritrea',
'ES':'Spain',
'ET':'Ethiopia',
'FI':'Finland',
'FJ':'Fiji',
'FK':'Falkland Islands (Malvinas)',
'FM':'Micronesia, Federated States of',
'FO':'Faroe Islands',
'FR':'France',
'GA':'Gabon',
'GB':'United Kingdom',
'GD':'Grenada',
'GE':'Georgia',
'GF':'French Guiana',
'GH':'Ghana',
'GI':'Gibraltar',
'GL':'Greenland',
'GM':'Gambia',
'GN':'Guinea',
'GP':'Guadeloupe',
'GQ':'Equatorial Guinea',
'GR':'Greece',
'GS':'South Georgia and the South Sandwich Islands',
'GT':'Guatemala',
'GU':'Guam',
'GW':'Guinea-Bissau',
'GY':'Guyana',
'HK':'Hong Kong',
'HM':'Heard Island and McDonald Islands',
'HN':'Honduras',
'HR':'Croatia',
'HT':'Haiti',
'HU':'Hungary',
'ID':'Indonesia',
'IE':'Ireland',
'IL':'Israel',
'IN':'India',
'IO':'British Indian Ocean Territory',
'IQ':'Iraq',
'IR':'Iran, Islamic Republic of',
'IS':'Iceland',
'IT':'Italy',
'JM':'Jamaica',
'JO':'Jordan',
'JP':'Japan',
'KE':'Kenya',
'KG':'Kyrgyzstan',
'KH':'Cambodia',
'KI':'Kiribati',
'KM':'Comoros',
'KN':'Saint Kitts and Nevis',
'KP':"Korea, Democratic People's Republic of",
'KR':'Korea, Republic of',
'KW':'Kuwait',
'KY':'Cayman Islands',
'KZ':'Kazakhstan',
'LA':"Lao People's Democratic Republic",
'LB':'Lebanon',
'LC':'Saint Lucia',
'LI':'Liechtenstein',
'LK':'Sri Lanka',
'LR':'Liberia',
'LS':'Lesotho',
'LT':'Lithuania',
'LU':'Luxembourg',
'LV':'Latvia',
'LY':'Libyan Arab Jamahiriya',
'MA':'Morocco',
'MC':'Monaco',
'MD':'Moldova, Republic of',
'MG':'Madagascar',
'MH':'Marshall Islands',
'MK':'Macedonia, the former Yugoslavian Republic of',
'ML':'Mali',
'MM':'Myanmar',
'MN':'Mongolia',
'MO':'Macao',
'MP':'Northern Mariana Islands',
'MQ':'Martinique',
'MR':'Mauritania',
'MS':'Montserrat',
'MT':'Malta',
'MU':'Mauritius',
'MV':'Maldives',
'MW':'Malawi',
'MX':'Mexico',
'MY':'Malaysia',
'MZ':'Mozambique',
'NA':'Namibia',
'NC':'New Caledonia',
'NE':'Niger',
'NF':'Norfolk Island',
'NG':'Nigeria',
'NI':'Nicaragua',
'NL':'Netherlands',
'NO':'Norway',
'NP':'Nepal',
'NR':'Nauru',
'NU':'Niue',
'NZ':'New Zealand',
'OM':'Oman',
'PA':'Panama',
'PE':'Peru',
'PF':'French Polynesia',
'PG':'Papua New Guinea',
'PH':'Philippines',
'PK':'Pakistan',
'PL':'Poland',
'PM':'Saint Pierre and Miquelon',
'PN':'Pitcairn',
'PR':'Puerto Rico',
'PS':'Palestinian Territory, occupied',
'PT':'Portugal',
'PW':'Palau',
'PY':'Paraguay',
'QA':'Qatar',
'RE':'Reunion',
'RO':'Romania',
'RU':'Russian Federation',
'RW':'Rwanda',
'SA':'Saudi Arabia',
'SB':'Solomon Islands',
'SC':'Seychelles',
'SD':'Sudan',
'SE':'Sweden',
'SG':'Singapore',
'SH':'Saint Helena',
'SI':'Slovenia',
'SJ':'Svalbard and Jan Mayen',
'SK':'Slovakia',
'SL':'Sierra Leone',
'SM':'San Marino',
'SN':'Senegal',
'SO':'Somalia',
'SR':'Suriname',
'ST':'Sao Tome and Principe',
'SV':'El Salvador',
'SY':'Syrian Arab Republic',
'SZ':'Swaziland',
'TC':'Turks and Caicos Islands',
'TD':'Chad',
'TF':'French Southern Territories',
'TG':'Togo',
'TH':'Thailand',
'TJ':'Tajikistan',
'TK':'Tokelau',
'TL':'Timor-Leste',
'TM':'Turkmenistan',
'TN':'Tunisia',
'TO':'Tonga',
'TR':'Turkey',
'TT':'Trinidad and Tobago',
'TV':'Tuvalu',
'TW':'Taiwan',
'TZ':'Tanzania, United Republic of',
'UA':'Ukraine',
'UG':'Uganda',
'UM':'United States Minor Outlying Islands',
'US':'United States',
'UY':'Uruguay',
'UZ':'Uzbekistan',
'VA':'Holy See (Vatican City State)',
'VC':'Saint Vincent and the Grenadines',
'VE':'Venezuela',
'VG':'Virgin Islands, British',
'VI':'Virgin Islands, U.S.',
'VN':'Viet Nam',
'VU':'Vanuatu',
'WF':'Wallis and Futuna',
'WS':'Samoa',
'YE':'Yemen',
'YT':'Mayotte',
'ZA':'South Africa',
'ZM':'Zambia',
'ZW':'Zimbabwe',
}

