// vim:expandtab:shiftwidth=2:tabstop=2:
// Copyright (C) 2013-2014 Canonical Ltd.

// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.

// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.

// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

#include "oxide_qquick_render_view_item.h"

#include <QGuiApplication>
#include <QInputMethod>
#include <QInputMethodEvent>
#include <QQuickWindow>
#include <QPointF>
#include <QRectF>
#include <QSizeF>
#include <QTouchEvent>

#include "qt/core/glue/oxide_qt_web_view_adapter.h"
#include "qt/quick/api/oxideqquickwebview_p.h"

#if defined(ENABLE_COMPOSITING)
#include "oxide_qquick_accelerated_render_view_node.h"
#endif
#include "oxide_qquick_painted_render_view_node.h"

namespace oxide {
namespace qquick {

void RenderViewItem::geometryChanged(const QRectF& new_geometry,
                                     const QRectF& old_geometry) {
  QQuickItem::geometryChanged(new_geometry, old_geometry);
  HandleGeometryChanged();
}

RenderViewItem::RenderViewItem() :
    QQuickItem(),
    is_compositing_enabled_(false),
    is_compositing_enabled_state_changed_(false),
    composite_requested_by_chromium_(false) {
  setFlag(QQuickItem::ItemHasContents);

  setAcceptedMouseButtons(Qt::AllButtons);
  setAcceptHoverEvents(true);
}

void RenderViewItem::Init(oxide::qt::WebViewAdapter* view) {
  setParentItem(adapterToQObject<OxideQQuickWebView>(view));
}

void RenderViewItem::Blur() {
  setFocus(false);
}

void RenderViewItem::Focus() {
  setFocus(true);
}

bool RenderViewItem::HasFocus() {
  return hasFocus();
}

void RenderViewItem::Show() {
  setVisible(true);
}

void RenderViewItem::Hide() {
  setVisible(false);
}

bool RenderViewItem::IsShowing() {
  return isVisible();
}

void RenderViewItem::UpdateCursor(const QCursor& cursor) {
  setCursor(cursor);
}

QRect RenderViewItem::GetViewBoundsPix() {
  if (!window()) {
    return QRect();
  }

  QPointF pos(mapToScene(QPointF(0, 0)) + window()->position());

  return QRect(qRound(pos.x()), qRound(pos.y()),
               qRound(width()), qRound(height()));
}

void RenderViewItem::SetSize(const QSize& size) {
  setSize(QSizeF(size));
  polish();
}

QScreen* RenderViewItem::GetScreen() {
  if (!window()) {
    return NULL;
  }

  return window()->screen();
}

void RenderViewItem::SetInputMethodEnabled(bool enabled) {
  setFlag(QQuickItem::ItemAcceptsInputMethod, enabled);
  QGuiApplication::inputMethod()->update(Qt::ImEnabled);
}

void RenderViewItem::SchedulePaintForRectPix(const QRect& rect) {
  if (is_compositing_enabled_) {
    is_compositing_enabled_state_changed_ = true;
    is_compositing_enabled_ = false;
  }

  if (rect.isNull() && !dirty_rect_.isNull()) {
    dirty_rect_ = QRectF(0, 0, width(), height()).toAlignedRect();
  } else {
    dirty_rect_ |= (QRectF(0, 0, width(), height()) & rect).toAlignedRect();
  }

  update();
  polish();
}

void RenderViewItem::ScheduleUpdate() {
#if defined(ENABLE_COMPOSITING)
  if (!is_compositing_enabled_) {
    is_compositing_enabled_state_changed_ = true;
    is_compositing_enabled_ = true;
  }

  composite_requested_by_chromium_ = true;

  update();
  polish();
#else
  Q_ASSERT(0);
#endif
}

void RenderViewItem::focusInEvent(QFocusEvent* event) {
  Q_ASSERT(event->gotFocus());
  HandleFocusEvent(event);
}

void RenderViewItem::focusOutEvent(QFocusEvent* event) {
  Q_ASSERT(event->lostFocus());
  HandleFocusEvent(event);
}

void RenderViewItem::keyPressEvent(QKeyEvent* event) {
  HandleKeyEvent(event);
}

void RenderViewItem::keyReleaseEvent(QKeyEvent* event) {
  HandleKeyEvent(event);
}

void RenderViewItem::mouseDoubleClickEvent(QMouseEvent* event) {
  HandleMouseEvent(event);
}

void RenderViewItem::mouseMoveEvent(QMouseEvent* event) {
  HandleMouseEvent(event);
}

void RenderViewItem::mousePressEvent(QMouseEvent* event) {
  forceActiveFocus();
  HandleMouseEvent(event);
}

void RenderViewItem::mouseReleaseEvent(QMouseEvent* event) {
  HandleMouseEvent(event);
}

void RenderViewItem::wheelEvent(QWheelEvent* event) {
  HandleWheelEvent(event);
}

void RenderViewItem::hoverMoveEvent(QHoverEvent* event) {
  // QtQuick gives us a hover event unless we have a grab (which
  // happens implicitly on button press). As Chromium doesn't
  // distinguish between the 2, just give it a mouse event
  QPointF window_pos = mapToScene(event->posF());
  QMouseEvent me(QEvent::MouseMove,
                 event->posF(),
                 window_pos,
                 window_pos + window()->position(),
                 Qt::NoButton,
                 Qt::NoButton,
                 event->modifiers());

  HandleMouseEvent(&me);

  event->setAccepted(me.isAccepted());
}

void RenderViewItem::inputMethodEvent(QInputMethodEvent* event) {
  HandleInputMethodEvent(event);
}

void RenderViewItem::touchEvent(QTouchEvent* event) {
  if (event->type() == QEvent::TouchBegin) {
    forceActiveFocus();
  }
  HandleTouchEvent(event);
}

void RenderViewItem::updatePolish() {
  if (is_compositing_enabled_) {
    UpdateTextureHandle();
  } else {
    UpdateBackingStore();
  }
}

QSGNode* RenderViewItem::updatePaintNode(
    QSGNode* oldNode,
    UpdatePaintNodeData* data) {
  Q_UNUSED(data);

  if (is_compositing_enabled_state_changed_) {
    delete oldNode;
    oldNode = NULL;
    is_compositing_enabled_state_changed_ = false;
  }

  bool composite_requested_by_chromium = composite_requested_by_chromium_;
  composite_requested_by_chromium_ = false;

  if (width() <= 0 || height() <= 0) {
    delete oldNode;
    if (composite_requested_by_chromium) {
      DidComposite(true);
    }
    return NULL;
  }

#if defined(ENABLE_COMPOSITING)
  if (is_compositing_enabled_) {
    AcceleratedRenderViewNode* node =
        static_cast<AcceleratedRenderViewNode *>(oldNode);
    if (!node) {
      node = new AcceleratedRenderViewNode(this);
    }

    QSize size = texture_handle()->GetSize();
    size = size.boundedTo(QSizeF(width(), height()).toSize());

    node->setRect(QRect(QPoint(0, 0), size));
    node->updateFrontTexture(texture_handle());

    if (composite_requested_by_chromium) {
      DidComposite(false);
    }

    return node;
  }
#else
  Q_ASSERT(!is_compositing_enabled_);
#endif

  PaintedRenderViewNode* node = static_cast<PaintedRenderViewNode *>(oldNode);
  if (!node) {
    node = new PaintedRenderViewNode();
  }

  const QPixmap* backing_store = GetBackingStore();

  QSize size;
  if (backing_store) {
    size = QSize(backing_store->width(), backing_store->height());
    size = size.boundedTo(QSizeF(width(), height()).toSize());
  } else {
    size = QSizeF(width(), height()).toSize();
  }

  node->setSize(size);
  node->setBackingStore(backing_store);
  node->markDirtyRect(dirty_rect_);

  node->update();

  dirty_rect_ = QRect();

  return node;
}

QVariant RenderViewItem::inputMethodQuery(Qt::InputMethodQuery query) const {
  switch (query) {
    case Qt::ImEnabled:
      return (flags() & QQuickItem::ItemAcceptsInputMethod) != 0;
    default:
      return InputMethodQuery(query);
  }
}

} // namespace qquick
} // namespace oxide
