/*
 * unity-webapps-local-url-index.c
 * Copyright (C) Canonical LTD 2011
 *
 * Author: Robert Carr <racarr@canonical.com>
 * 
 unity-webapps is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * unity-webapps is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.";
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <glib/gstdio.h>
#include <gio/gio.h>

#include "unity-webapps-local-url-index.h"
#include "unity-webapps-application-collector.h"
#include "unity-webapps-local-available-application.h"

#include "unity-webapps-dpkg-available-application.h"

#include "../unity-webapps-debug.h"

struct _UnityWebappsLocalUrlIndexPrivate {
  UnityWebappsApplicationCollector *collector;
  
  GHashTable *available_applications_by_name;
};

enum {
  PROP_0,
  PROP_COLLECTOR
};

G_DEFINE_TYPE(UnityWebappsLocalUrlIndex, unity_webapps_local_url_index, UNITY_WEBAPPS_TYPE_URL_INDEX)

#define UNITY_WEBAPPS_LOCAL_URL_INDEX_GET_PRIVATE(object) (G_TYPE_INSTANCE_GET_PRIVATE ((object), UNITY_WEBAPPS_TYPE_LOCAL_URL_INDEX, UnityWebappsLocalUrlIndexPrivate))


static void
unity_webapps_local_url_index_get_property (GObject *object,
					    guint prop_id,
					    GValue *value,
					    GParamSpec *pspec)
{
  UnityWebappsLocalUrlIndex *self;
  
  self = UNITY_WEBAPPS_LOCAL_URL_INDEX (object);
  
  switch (prop_id)
    {
    case PROP_COLLECTOR:
      g_value_set_object (value, self->priv->collector);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
    }
}

static void
unity_webapps_local_url_index_set_property (GObject *object,
					    guint prop_id,
					    const GValue *value,
					    GParamSpec *pspec)
{
  UnityWebappsLocalUrlIndex *self;
  
  self = UNITY_WEBAPPS_LOCAL_URL_INDEX (object);
  
  switch (prop_id)
    {
    case PROP_COLLECTOR:
      g_assert (self->priv->collector == NULL);
      self->priv->collector = g_value_get_object (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
    }
}


static GList *
local_url_index_lookup_url (UnityWebappsUrlIndex *url_index,
			     const gchar *url)
{
  UnityWebappsLocalUrlIndex *self;
  GList *available_applications, *results, *w;

  self = UNITY_WEBAPPS_LOCAL_URL_INDEX (url_index);
  
  results = NULL;
  
  available_applications = g_hash_table_get_values (self->priv->available_applications_by_name);
  
  for (w = available_applications; w != NULL; w = w->next)
    {
      UnityWebappsLocalAvailableApplication *app;
      gboolean matches;
      
      app = UNITY_WEBAPPS_LOCAL_AVAILABLE_APPLICATION (w->data);
      matches = unity_webapps_local_available_application_provides_url (app, url);
      
      if (matches)
	{
	  results = g_list_append (results, app);
	}
    }
  
  g_list_free (available_applications);

  return results;
}

static void
unity_webapps_local_url_index_finalize (GObject *object)
{
  UnityWebappsLocalUrlIndex *self;
  
  self = UNITY_WEBAPPS_LOCAL_URL_INDEX (object);

  g_hash_table_unref (self->priv->available_applications_by_name);
}

static void
unity_webapps_local_url_index_dispose (GObject *object)
{
  UnityWebappsLocalUrlIndex *self;
  
  self = (UnityWebappsLocalUrlIndex *)object;

  if (self->priv->collector)
    {
      g_object_unref (G_OBJECT (self->priv->collector));
      self->priv->collector = NULL;
    }

}

static void
unity_webapps_local_url_index_class_init (UnityWebappsLocalUrlIndexClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  UnityWebappsUrlIndexClass *index_class = UNITY_WEBAPPS_URL_INDEX_CLASS (klass);
  
  object_class->finalize = unity_webapps_local_url_index_finalize;
  object_class->dispose = unity_webapps_local_url_index_dispose;
  object_class->get_property = unity_webapps_local_url_index_get_property;
  object_class->set_property = unity_webapps_local_url_index_set_property;

  index_class->lookup_url = local_url_index_lookup_url;
  
  g_type_class_add_private (object_class, sizeof(UnityWebappsLocalUrlIndexPrivate));
  
  g_object_class_install_property (object_class, PROP_COLLECTOR,
				   g_param_spec_object ("collector", "Application Collector",
							"The Application Collector to use to build the index",
							UNITY_WEBAPPS_TYPE_APPLICATION_COLLECTOR,
							G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_STRINGS));
 
}

static void
unity_webapps_local_url_index_init (UnityWebappsLocalUrlIndex *self)
{
  self->priv = UNITY_WEBAPPS_LOCAL_URL_INDEX_GET_PRIVATE (self);
  
  self->priv->available_applications_by_name = NULL;
  self->priv->collector = NULL;
}



UnityWebappsUrlIndex *
unity_webapps_local_url_index_new (UnityWebappsApplicationCollector *collector)
{
  return UNITY_WEBAPPS_URL_INDEX (g_object_new (UNITY_WEBAPPS_TYPE_LOCAL_URL_INDEX, "collector", collector, NULL));
}

UnityWebappsUrlIndex *
unity_webapps_local_url_index_new_default ()
{
  UnityWebappsApplicationCollector *collector;

  collector = unity_webapps_application_collector_new_with_default_path ();
  return unity_webapps_local_url_index_new (collector);
}

gboolean
unity_webapps_local_url_index_load_applications (UnityWebappsLocalUrlIndex *index)
{
  GHashTable *manifests_by_name;
  
  unity_webapps_application_collector_search_path (index->priv->collector);
  
  if (index->priv->available_applications_by_name == NULL)
    {
      manifests_by_name = unity_webapps_application_collector_get_applications (index->priv->collector);
  
      index->priv->available_applications_by_name = g_hash_table_ref (manifests_by_name);
    }
  
  return TRUE;
}

UnityWebappsLocalAvailableApplication *
unity_webapps_local_url_index_get_application_by_name (UnityWebappsLocalUrlIndex *index,
						       const gchar *name)
{
  return UNITY_WEBAPPS_LOCAL_AVAILABLE_APPLICATION (g_hash_table_lookup (index->priv->available_applications_by_name, name));
}
