/*
 *========================================================================
 * $Id: host.c 88 2004-09-28 22:49:38Z rgb $
 *
 * See copyright in copyright.h and the accompanying file COPYING
 *========================================================================
 */

#include <wulfware/libwulf.h>

/*
 * This routine clears the host struct for a given host index, freeing
 * any allocated memory and nulling the pointers.  It does NOT clear
 * the hostname/hostip fields filled in when the hostfile was parsed,
 * as this is its only memory of this mapping and is needed to (re)connect.
 */

void clear_values(Host *hostptr)
{

 /* loop index */
 int i;

 if((verbose == D_ALL) || (verbose == D_INIT_HOST)){
   fprintf(OUTFP,"D_INIT_HOST: Starting clear_values() for host %s\n",hostptr->hostname);
   fprintf(OUTFP,"D_INIT_HOST: Use %d to focus\n",D_INIT_HOST);
 }

 /*
  * This is tricky and dangerous.  In order not to leak, we have to
  * free any memory previously allocated to hold cpu or interface
  * information (both requirements can vary from host to host according
  * to how many cpus and interfaces they can have).  In order for this
  * to work safely, the pointers MUST be set to be null already on the
  * very first call.
  */

 /* free cpu struct if already allocated */
 if(hostptr->val.cpu != NULL) {
   if((verbose == D_ALL) || (verbose == D_INIT_HOST)){
     fprintf(OUTFP,"D_INIT_HOST: Freeing cpu struct at %x on %s\n",&hostptr->val.cpu[-1],hostptr->hostname);
   }
   /* 
    * Don't forget to free the DISPLACED address!  cpu[-1] is always total cpu.
    */
   free(&hostptr->val.cpu[-1]);
   hostptr->val.cpu = NULL;
   if((verbose == D_ALL) || (verbose == D_INIT_HOST)){
     fprintf(OUTFP,"D_INIT_HOST: Freed cpu struct, now pointer value is %x on %s\n",&hostptr->val.cpu[-1],hostptr->hostname);
   }
 }

 /* free interface struct if already allocated */
 if(hostptr->val.interface != NULL) {
   if((verbose == D_ALL) || (verbose == D_INIT_HOST)){
     fprintf(OUTFP,"D_INIT_HOST: Freeing interface struct at %x on %s\n",&hostptr->val.interface[-1],hostptr->hostname);
   }
   /* Don't forget to free the DISPLACED address!  interface[-1] is totals. */
   free(&hostptr->val.interface[-1]);
   hostptr->val.interface = NULL;
   if((verbose == D_ALL) || (verbose == D_INIT_HOST)){
     fprintf(OUTFP,"D_INIT_HOST: Freed interface struct, now pointer value is %x on %s\n",&hostptr->val.interface[-1],hostptr->hostname);
   }
 }

 /* free pidlist linked list if already allocated */
 if(hostptr->pidlist != NULL) {
   if((verbose == D_ALL) || (verbose == D_INIT_HOST)){
     fprintf(OUTFP,"D_INIT_HOST: Deleting/freeing pidlist contents at %x on %s\n",hostptr->pidlist,hostptr->hostname);
   }
   /*
    * Delete the list contents.  The list itself remains.
    */
   delete_list(hostptr->pidlist);
   if((verbose == D_ALL) || (verbose == D_INIT_HOST)){
     fprintf(OUTFP,"D_INIT_HOST: Linked list at %x on %s is freed and empty.\n",&hostptr->val.interface[-1],hostptr->hostname);
   }
 }

 /*
  * We now zero all of hosts.val.  Note that this nulls the pointers
  * to cpu and interface, which are free if we got here.
  */
 if((verbose == D_ALL) || (verbose == D_INIT_HOST)){
   fprintf(OUTFP,"D_INIT_HOST: About to zero all of host %s val...",
       hostptr->hostname);
   fflush(stdout);
 }
 memset( &hostptr->val,0,sizeof(Value) );

 if((verbose == D_ALL) || (verbose == D_INIT_HOST)){
   fprintf(OUTFP,"Done!\n");
   
 }

}


void clear_host_values(List *hostlist)
{

 Host *hostptr;
 ListElement *element;

 if((verbose == D_ALL) || (verbose == D_INIT_HOST)){
   fprintf(OUTFP,"D_INIT_HOST: Starting clear_host values() for hostlist %0x\n",hostlist);
 }

 element = hostlist->head;
 while(element != NULL){
   hostptr = element->data;
   if(hostptr->client_fd && hostptr->connected){
     clear_values(hostptr);
   }
   element = element->next;
 }

}
