///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/scene/animation/AnimManager.h>
#include <core/utilities/ProgressIndicator.h>
#include <boost/iostreams/device/file.hpp>
#include <boost/iostreams/stream.hpp>

#include "LAMMPSDataParser.h"
#include "../CompressedTextParserStream.h"
#include <atomviz/atoms/AtomsObject.h>
#include <atomviz/atoms/datachannels/AtomTypeDataChannel.h>
#include <atomviz/utils/ChemicalElements.h>

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(LAMMPSDataParser, AtomsFileParser)

/******************************************************************************
* Checks if the given file has format that can be read by this importer.
******************************************************************************/
bool LAMMPSDataParser::checkFileFormat(const QString& filepath)
{
	// Open the input file for reading.
	CompressedTextParserStream stream(filepath);

	// Read first comment line.
	char buffer[1024];
	int count = stream.getline(buffer, sizeof(buffer)/sizeof(buffer[0]));
	if(count<0 || count >= sizeof(buffer)/sizeof(buffer[0])-1) return false;

	for(int i=0; i<20; i++) {
		if(stream.eof()) return false;
		string line = stream.readline();
		// Trim anything from '#' onward.
		size_t commentStart = line.find('#');
		if(commentStart != string::npos) line.resize(commentStart);
		// If line is blank, continue.
		if(line.find_first_not_of(" \t\n\r") == string::npos) continue;
		if(line.find("atoms") != string::npos) {
			int natoms;
			if(sscanf(line.c_str(), "%u", &natoms) != 1 || natoms < 0)
				return false;
			return true;
		}
	}

	return false;
}

/******************************************************************************
* Reads an atomic data set from the input file.
******************************************************************************/
EvaluationStatus LAMMPSDataParser::loadAtomsFile(AtomsObject* destination, int movieFrame, bool suppressDialogs)
{
	CHECK_OBJECT_POINTER(destination);
	scoped_ptr<ProgressIndicator> progress;

	CompressedTextParserStream stream(inputFile());
	setlocale(LC_NUMERIC, "C");

	// Read comment line
	stream.readline();

	// Read header
	int natoms = 0;
	int natomtypes = 0;
	FloatType xlo = 0, xhi = 0;
	FloatType ylo = 0, yhi = 0;
	FloatType zlo = 0, zhi = 0;
	FloatType xy = 0, xz = 0, yz = 0;

	while(true) {

		string line = stream.readline();

		// Trim anything from '#' onward.
		size_t commentStart = line.find('#');
		if(commentStart != string::npos) line.resize(commentStart);

    	// If line is blank, continue.
		if(line.find_first_not_of(" \t\n\r") == string::npos) continue;

		// These are the format strings used with the sscanf() parsing function.
		// We have to obey the floating point precision used to compile the program
		// because the destination variables are either single or double precision variables.
#ifdef USE_DOUBLE_PRECISION_FP
	#define FLOAT_SCANF_STRING_1   "%lg"
	#define FLOAT_SCANF_STRING_2   "%lg %lg"
	#define FLOAT_SCANF_STRING_3   "%lg %lg %lg"
	#define ATOM_SCANF_STRING      "%u %u %lg %lg %lg"
	#define VELOCITY_SCANF_STRING  "%u %lg %lg %lg"
#else
	#define FLOAT_SCANF_STRING_1   "%g"
	#define FLOAT_SCANF_STRING_2   "%g %g"
	#define FLOAT_SCANF_STRING_3   "%g %g %g"
	#define ATOM_SCANF_STRING      "%u %u %g %g %g"
	#define VELOCITY_SCANF_STRING  "%u %g %g %g"
#endif

    	if(line.find("atoms") != string::npos) {
    		if(sscanf(line.c_str(), "%u", &natoms) != 1)
    			throw Exception(tr("Invalid number of atoms (line %1): %2").arg(stream.lineNumber()).arg(line.c_str()));

			OVITO_ASSERT(!progress);
			progress.reset(new ProgressIndicator(tr("Loading LAMMPS data file (%1 atoms)").arg(natoms), natoms, suppressDialogs));
		}
    	else if(line.find("atom types") != string::npos) {
    		if(sscanf(line.c_str(), "%u", &natomtypes) != 1)
    			throw Exception(tr("Invalid number of atom types (line %1): %2").arg(stream.lineNumber()).arg(line.c_str()));
    	}
    	else if(line.find("xlo xhi") != string::npos) {
    		if(sscanf(line.c_str(), FLOAT_SCANF_STRING_2, &xlo, &xhi) != 2)
    			throw Exception(tr("Invalid xlo/xhi values (line %1): %2").arg(stream.lineNumber()).arg(line.c_str()));
    	}
    	else if(line.find("ylo yhi") != string::npos) {
    		if(sscanf(line.c_str(), FLOAT_SCANF_STRING_2, &ylo, &yhi) != 2)
    			throw Exception(tr("Invalid ylo/yhi values (line %1): %2").arg(stream.lineNumber()).arg(line.c_str()));
    	}
    	else if(line.find("zlo zhi") != string::npos) {
    		if(sscanf(line.c_str(), FLOAT_SCANF_STRING_2, &zlo, &zhi) != 2)
    			throw Exception(tr("Invalid zlo/zhi values (line %1): %2").arg(stream.lineNumber()).arg(line.c_str()));
    	}
    	else if(line.find("xy xz yz") != string::npos) {
    		if(sscanf(line.c_str(), FLOAT_SCANF_STRING_3, &xy, &xz, &yz) != 3)
    			throw Exception(tr("Invalid xy/xz/yz values (line %1): %2").arg(stream.lineNumber()).arg(line.c_str()));
    	}
    	else if(line.find("bonds") != string::npos) {}
    	else if(line.find("angles") != string::npos) {}
    	else if(line.find("dihedrals") != string::npos) {}
    	else if(line.find("impropers") != string::npos) {}
    	else if(line.find("bond types") != string::npos) {}
    	else if(line.find("angle types") != string::npos) {}
    	else if(line.find("dihedral types") != string::npos) {}
    	else if(line.find("improper types") != string::npos) {}
    	else if(line.find("extra bond per atom") != string::npos) {}
    	else break;
	}

	if(xhi < xlo || yhi < ylo || zhi < zlo)
		throw Exception(tr("Invalid simulation cell size in header of LAMMPS data file."));

	destination->setAtomsCount((size_t)natoms);

	// Get the required data channels.
	AtomTypeDataChannel* typeChannel = static_object_cast<AtomTypeDataChannel>(destination->createStandardDataChannel(DataChannel::AtomTypeChannel));
	DataChannel* posChannel = destination->createStandardDataChannel(DataChannel::PositionChannel);
	OVITO_ASSERT(posChannel->size() == natoms);

	// Define the simulation cell geometry.
	destination->simulationCell()->setCellShape(Point3(xlo, ylo, zlo), Vector3(xhi - xlo, 0, 0), Vector3(xy, yhi - ylo, 0), Vector3(xz, yz, zhi - zlo));

	// Allocate atom types.
	for(int i=1; i<=natomtypes; i++) {
		typeChannel->createAtomType(i);
	}

	// Read up to non-blank line plus 1 following line.
	while(!stream.eof() && stream.line().find_first_not_of(" \t\n\r") == string::npos) {
		stream.readline();
	}

	// This flag is set to true if the atomic coordinates have been parsed.
	bool foundAtomsSection = false;

	// Read identifier strings one by one in free-form part of data file.
	string keyword = extractKeyword(stream.line());
	for(;;) {

	    // Skip blank line after keyword.
		if(stream.eof()) break;
		stream.readline();
		if(stream.eof()) break;

		if(keyword.compare("Atoms") == 0) {

			for(int i = 0; i < natoms; i++) {
				stream.readline();
				if(stream.eof())
					throw Exception(tr("Unexpected end of LAMMPS data file in line %1.").arg(stream.lineNumber()));

				// Update progress indicator.
				if((i % 1000) == 0 && progress) {
					progress->setValue(i);
					if(progress->isCanceled()) return EvaluationStatus(EvaluationStatus::EVALUATION_ERROR);
				}

				Point3 pos;
				int index, atypeindex;

    			if(sscanf(stream.line().c_str(), ATOM_SCANF_STRING, &index, &atypeindex, &pos.X, &pos.Y, &pos.Z) != 5)
					throw Exception(tr("Invalid atom specification (line %1): %2").arg(stream.lineNumber()).arg(stream.line().c_str()));

				if(index < 1 || index > natoms)
					throw Exception(tr("Atom index out of range (line %1).").arg(stream.lineNumber()));
				if(atypeindex < 1 || atypeindex > natomtypes)
					throw Exception(tr("Atom type index out of range (line %i).").arg(stream.lineNumber()));

				posChannel->setPoint3((size_t)index-1, pos);
				typeChannel->setInt((size_t)index-1, atypeindex);
			}

			foundAtomsSection = true;
		}
		else if(keyword.compare("Masses") == 0) {
			for(int i = 0; i < natomtypes; i++) {
				stream.readline();
			}
		}
		else if(keyword.compare("Velocities") == 0) {

			// Create the velocity data channels.
			DataChannel* velocityChannel = destination->createStandardDataChannel(DataChannel::VelocityChannel);

			for(int i = 0; i < natoms; i++) {
				stream.readline();
				if(stream.eof())
					throw Exception(tr("Unexpected end of LAMMPS data file in line %1.").arg(stream.lineNumber()));

				// Update progress indicator.
				if((i % 1000) == 0 && progress) {
					progress->setValue(i);
					if(progress->isCanceled()) return EvaluationStatus(EvaluationStatus::EVALUATION_ERROR);
				}

				Vector3 v;
				int index;

    			if(sscanf(stream.line().c_str(), VELOCITY_SCANF_STRING, &index, &v.X, &v.Y, &v.Z) != 4)
					throw Exception(tr("Invalid velocity specification (line %1): %2").arg(stream.lineNumber()).arg(stream.line().c_str()));

				if(index < 1 || index > natoms)
					throw Exception(tr("Atom index out of range (line %1).").arg(stream.lineNumber()));

				velocityChannel->setVector3((size_t)index-1, v);
			}
		}
		else if(keyword.empty() == false) {
			throw Exception(tr("Unknown keyword in line %1 of LAMMPS data file: %2.\nNote that the parser routine supports only \"atomic style\" LAMMPS data files.").arg(stream.lineNumber()-1).arg(keyword.c_str()));
		}
		else break;

		// Read up to non-blank line plus 1 following line.
		while(!stream.eof() && stream.readline().find_first_not_of(" \t\n\r") == string::npos);

		// Read identifier strings one by one in free-form part of data file.
		keyword = extractKeyword(stream.line());
	}

	if(!foundAtomsSection)
		throw Exception("LAMMPS data file did not contain any atomic coordinates.");

	destination->invalidate();

	QString statusMessage = tr("Number of atoms: %1").arg(natoms);
	return EvaluationStatus(EvaluationStatus::EVALUATION_SUCCESS, statusMessage);
}

/******************************************************************************
* Removes whitespace characters in front and after the keyword.
******************************************************************************/
string LAMMPSDataParser::extractKeyword(const string& line)
{
	// Trim anything from '#' onward.
	size_t commentStart = line.find('#');
	if(commentStart == string::npos) commentStart = line.length();
	if(commentStart == 0) return string();

	size_t keywordStart = line.find_first_not_of(" \t\n\r");
	if(keywordStart == string::npos || keywordStart >= commentStart) return string();

	size_t keywordEnd = line.find_last_not_of(" \t\n\r", commentStart);
	OVITO_ASSERT(keywordEnd != string::npos && keywordEnd < commentStart);

	return line.substr(keywordStart, keywordEnd - keywordStart + 1);
}


};	// End of namespace AtomViz
