/* pdu-ipv6.c
 
   PDU builder for IPv6 packets

   Copyright (C) 2007, 2008, 2009, 2010, 2011 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "pbuild-priv.h"
#include "pdu-ipv6.h"

static const struct pdu_dict protocol_mappings[] = {
    {"udp", (uint8_t []) {IP_PROTO_UDP} },
    {"tcp", (uint8_t []) {IP_PROTO_TCP} },
    {"icmp6", (uint8_t []) {IP_PROTO_ICMPV6} },
    {"icmp6-unreachable", (uint8_t []) {IP_PROTO_ICMPV6} },
    {"icmp6-toobig", (uint8_t []) {IP_PROTO_ICMPV6} },
    {"icmp6-timeexceed", (uint8_t []) {IP_PROTO_ICMPV6} },
    {"icmp6-parmproblem", (uint8_t []) {IP_PROTO_ICMPV6} },
    {"icmp6-echo", (uint8_t []) {IP_PROTO_ICMPV6} },
    {"icmp6-echoreply", (uint8_t []) {IP_PROTO_ICMPV6} },
    {"icmp6-redirect", (uint8_t []) {IP_PROTO_ICMPV6} },
    {"icmp6-ns", (uint8_t []) {IP_PROTO_ICMPV6} },
    {"icmp6-na", (uint8_t []) {IP_PROTO_ICMPV6} },
    {"icmp6-rs", (uint8_t []) {IP_PROTO_ICMPV6} },
    {"icmp6-ra", (uint8_t []) {IP_PROTO_ICMPV6} },
    {"icmp6-rn", (uint8_t []) {IP_PROTO_ICMPV6} },
    {"ip6hbh", (uint8_t []) {IP_PROTO_HOPOPTS} },
    {"ip6dest", (uint8_t []) {IP_PROTO_DSTOPTS} },
    {"ip6frag", (uint8_t []) {IP_PROTO_FRAGMENT} },
    {"ip6rth", (uint8_t []) {IP_PROTO_ROUTING} },
    {NULL, NULL}
};

GHashTable *next_headers;

/*
 * Returns the value to use for the "next header" field of an IPv6 header or
 * extension header. This value depends on whether the extension header is
 * followed by another extension header, by an upper layer protocol (like ICMP,
 * TCP or UDP), or by nothing else.
 */
static uint8_t
ipv6_nxt_hdr(const GNode *pdu)
{
    uint8_t *nxt_hdr;
    void *field;
    const GNode *next_pdu;

    if (  !(field = _pb_pdata(pdu, "next-header") )
	&& (next_pdu = pb_nextpdu(pdu) ) ) {
	/*
	 * User didn't specify the IP protocol so we try to put in the right
	 * value. For this we multiplex based on the next PDU, i.e. if the next
	 * PDU is ICMP then the protocol field is 1, etc.
	 */

	nxt_hdr = g_hash_table_lookup(next_headers, pb_getname(next_pdu) );
	if (nxt_hdr)
	    return *nxt_hdr;
    } else
	/*
	 * Use for IP protocol whatever the user specified. Note that if
	 * there is no user-specified IP protocol *and* there is no
	 * next PDU then we end up here, but that's okay because
	 * num_next(NULL) is 0.
	 */
	return num_next(field);

    return IP_PROTO_NONE; /* Shouldn't happen */
}

/********************************************************************
 *			  IPv6 Basic Header                         *
 ********************************************************************/

static void
pdu_ipv6hdr_builder(const GNode *pdu, void *dest)
{
    struct ip6_hdr *ip6;
    uint32_t version;
    ip6_addr_t *addr;
    void *field;

    ip6 = dest;

    ip6->ip6_flow = 0;

    version = (field = _pb_pdata(pdu, "version") ) ? num_next(field) : 6;
    ip6->ip6_vfc |= (version << 4);

    ip6->ip6_nxt = ipv6_nxt_hdr(pdu);

    ip6->ip6_hlim = (field = _pb_pdata(pdu, "hlim") )
		    ? num_next(field) : IP6_HLIM_DEFAULT;

    if ( (addr = _pb_pdata(pdu, "src") ) )
	ip6->ip6_src = *addr;
    else
	memset(&ip6->ip6_src, 0, IP6_ADDR_LEN);

    if ( (addr = _pb_pdata(pdu, "dst") ) )
	ip6->ip6_dst = *addr;
    else
	memset(&ip6->ip6_dst, 0, IP6_ADDR_LEN);
}

static void
pdu_ipv6hdr_builder2(const GNode *pdu _U_, void *dest, void *prev_pdu_hdr _U_)
{
    struct ip6_hdr *ip6;
    uint16_t u16;
    size_t opts_len, payload_len;

    ip6 = dest;

    opts_len = ( (struct node_data *) pdu->data)->_data_pdu.opts_len;
    payload_len = ( (struct node_data *) pdu->data)->_data_pdu.payload_len;

    u16 = htons(opts_len + payload_len);
    SSVAL(ip6, offsetof(struct ip6_hdr, ip6_plen), u16);
}

#if 0
static void
pdu_ipv6hdr_dumper(pdu_t *p, const char *prefix)
{
    struct ipv6hdr_options *hdr_data;

    hdr_data = p->header_data;

    printf("%s  Parameters:\n", prefix);
    printf("%s    Next header: %s\n", prefix, num_info(hdr_data->nxt_hdr) );
    printf("%s    Hop Limit: %s\n", prefix, num_info(hdr_data->_basic.hlim) );
    printf("%s    Source address: %s\n", prefix,
	   pdu_ip6addr2str(hdr_data->_basic.source_ip) );
    printf("%s    Destination address: %s\n", prefix,
	   pdu_ip6addr2str(hdr_data->_basic.destination_ip) );
}
#endif

/********************************************************************
 *		 Hop-by-hop Options Extension Header                *
 ********************************************************************/

static void
pdu_ipv6hbh_builder(const GNode *pdu, void *dest)
{
    struct ip6_hbh *hbh;
    uint8_t *optsptr;
    const GNode *child;
    struct node_data *node_data;
    struct payload *data;
    size_t hdr_len;
    void *field;

    hbh = (struct ip6_hbh *) dest;

    hbh->ip6h_nxt = ipv6_nxt_hdr(pdu);

    optsptr = (uint8_t *) hbh + sizeof(struct ip6_hbh);

    for (child = g_node_first_child(pdu);
	 child;
	 child = g_node_next_sibling(child) ) {
	node_data = child->data;

	if (!strcmp(node_data->name, "pad1") ) {
	    *optsptr++ = IP6OPT_PAD1;
	} else if (!strcmp(node_data->name, "padn") ) {
	    *optsptr++ = IP6OPT_PADN;
	    /*
	     * Substract 2 per RFC 2460 section 4.2, which says "For N octets
	     * of padding, the Opt Data Len field contains the value N-2, and
	     * the Option Data consists of N-2 zero-valued octets.
	     */
	    *optsptr++ = num_next(node_data->_data_parm.data) - 2;
	    memset(optsptr, 0, optsptr[-1]);
	    optsptr += optsptr[-1];
	} else if (!strcmp(node_data->name, "raw-tlv") ) {
	    memcpy(optsptr,
		   ( (struct payload *) node_data->_data_parm.data)->data,
		   ( (struct payload *) node_data->_data_parm.data)->len);
	    optsptr += ( (struct payload *) node_data->_data_parm.data)->len;
	} else if (!strcmp(node_data->name, "tlv") ) {
	    /* TVL type */
	    *optsptr++ = num_next(_pb_pdata(child, "type") );

	    data = _pb_pdata(child, "value");
	    if (data) {
		/* TVL length */
		if ( (field = _pb_pdata(child, "length") ) )
		    /* User specified a fake length */
		    *optsptr++ = num_next(field);
		else
		    /* Use real length */
		    *optsptr++ = data->len;

		/* TLV data */
		memcpy(optsptr, data->data, data->len);

		optsptr += data->len;
	    } else {
		/* TVL length */
		if ( (field = _pb_pdata(child, "length") ) )
		    /* User specified a fake length */
		    *optsptr++ = num_next(field);
		else
		    *optsptr++ = 0;
	    }
	}
    }

    hdr_len = optsptr - (uint8_t *) dest;

    hbh->ip6h_len = (hdr_len - 8)/8; /* RFC 2460 4.3 */
}

/*
 * Calculates the length of an IPv6 hop-by-hop extension header.
 * This length is variable a depends on the PDU definition.
 */
static size_t
pdu_ipv6hbh_len(const GNode *pdu)
{
    const GNode *child;
    struct node_data *node_data;
    struct payload *data;
    size_t hdr_len;

    hdr_len = sizeof(struct ip6_hbh);

    for (child = g_node_first_child(pdu);
	 child;
	 child = g_node_next_sibling(child) ) {
	node_data = child->data;

	if (!strcmp(node_data->name, "pad1") ) {
	    hdr_len++;
	} else if (!strcmp(node_data->name, "padn") ) {
	    hdr_len += 1 /* type */ + 1 /* length */
		       + num_next(node_data->_data_parm.data) - 2 /* padding */;
	} else if (!strcmp(node_data->name, "raw-tlv") ) {
	    hdr_len += ( (struct payload *) node_data->_data_parm.data)->len;
	} else if (!strcmp(node_data->name, "tlv") ) {
	    data = _pb_pdata(child, "value");

	    hdr_len += 1 /* type */ + 1 /* length */;

	    if (data)
		hdr_len += data->len;
	}
    }

    return hdr_len;
}

/********************************************************************
 *		 Destination Options Extension Header               *
 ********************************************************************/

static void
pdu_ipv6destopts_builder(const GNode *pdu, void *dest)
{
    struct ip6_dest *destopts;
    uint8_t *optsptr;
    const GNode *child;
    struct node_data *node_data;
    struct payload *data;
    size_t hdr_len;
    void *field;

    destopts = (struct ip6_dest *) dest;

    destopts->ip6d_nxt = ipv6_nxt_hdr(pdu);

    optsptr = (uint8_t *) destopts + sizeof(struct ip6_dest);

    for (child = g_node_first_child(pdu);
	 child;
	 child = g_node_next_sibling(child) ) {
	node_data = child->data;

	if (!strcmp(node_data->name, "pad1") ) {
	    *optsptr++ = IP6OPT_PAD1;
	} else if (!strcmp(node_data->name, "padn") ) {
	    *optsptr++ = IP6OPT_PADN;
	    /*
	     * Substract 2 per RFC 2460 section 4.2, which says "For N octets
	     * of padding, the Opt Data Len field contains the value N-2, and
	     * the Option Data consists of N-2 zero-valued octets.
	     */
	    *optsptr++ = num_next(node_data->_data_parm.data) - 2;
	    memset(optsptr, 0, optsptr[-1]);
	    optsptr += optsptr[-1];
	} else if (!strcmp(node_data->name, "raw-tlv") ) {
	    memcpy(optsptr,
		   ( (struct payload *) node_data->_data_parm.data)->data,
		   ( (struct payload *) node_data->_data_parm.data)->len);
	    optsptr += ( (struct payload *) node_data->_data_parm.data)->len;
	} else if (!strcmp(node_data->name, "tlv") ) {
	    /* TVL type */
	    *optsptr++ = num_next(_pb_pdata(child, "type") );

	    data = _pb_pdata(child, "value");
	    if (data) {
		/* TVL length */
		if ( (field = _pb_pdata(child, "length") ) )
		    /* User specified a fake length */
		    *optsptr++ = num_next(field);
		else
		    /* Use real length */
		    *optsptr++ = data->len;

		/* TLV data */
		memcpy(optsptr, data->data, data->len);

		optsptr += data->len;
	    } else {
		/* TVL length */
		if ( (field = _pb_pdata(child, "length") ) )
		    /* User specified a fake length */
		    *optsptr++ = num_next(field);
		else
		    *optsptr++ = 0;
	    }
	}
    }

    hdr_len = optsptr - (uint8_t *) dest;

    destopts->ip6d_len = (hdr_len - 8)/8; /* RFC 2460 4.3 */
}

/********************************************************************
 *		      Fragment Extension Header                     *
 ********************************************************************/

static void
pdu_ipv6frag_builder(const GNode *pdu, void *dest)
{
    struct ip6_frag *frag;
    uint32_t u32;
    uint16_t u16;

    frag = (struct ip6_frag *) dest;

    frag->ip6f_nxt = ipv6_nxt_hdr(pdu);

    u32 = htonl(num_next(_pb_pdata(pdu, "id") ) );
    SIVAL(frag, offsetof(struct ip6_frag, ip6f_ident), u32);

    u16 = 0;
    if (_pb_pdata(pdu, "more") )
	u16 |= IP6F_MORE_FRAG;
    u16 |= htons(num_next(_pb_pdata(pdu, "offset") ) << 3);
    SSVAL(frag, offsetof(struct ip6_frag, ip6f_offlg), u16);
}

#if 0
static void
pdu_ipv6frag_dumper(pdu_t *p, const char *prefix)
{
    struct ipv6hdr_options *hdr_data;

    hdr_data = p->header_data;

    printf("%s      Parameters:\n", prefix);
    printf("%s        Fragment offset: 0x%04x\n", prefix,
	   hdr_data->_frag.frag_off);
    printf("%s        M flag: %s\n", prefix,
	   hdr_data->_frag.more ? "set" : "unset");
    printf("%s        Identification: %s\n", prefix,
	   num_info(hdr_data->_frag.id) );
}
#endif

/********************************************************************
 *		       Routing Extension Header                     *
 ********************************************************************/

static void
pdu_ipv6rthdr_builder(const GNode *pdu, void *dest)
{
    struct ip6_rthdr *rthdr;
    ip6_addr_t *s;
    const GNode *child;
    size_t hdr_len;
    struct node_data *node_data;

    rthdr = (struct ip6_rthdr *) dest;

    rthdr->ip6r_nxt = ipv6_nxt_hdr(pdu);
    rthdr->ip6r_type = num_next(_pb_pdata(pdu, "type") );
    rthdr->ip6r_segleft = num_next(_pb_pdata(pdu, "segs-left") );

    s = (ip6_addr_t *) ( (uint8_t *) rthdr + sizeof(struct ip6_rthdr)
			+ sizeof(uint32_t) /* reserved */ );

    for (child = g_node_first_child(pdu);
	 child;
	 child = g_node_next_sibling(child) ) {
	node_data = child->data;

	/* We only care about "router"; everything else ("next-header",
	 * "type", "segs-left") is handled above.
	 */
	if (!strcmp(node_data->name, "router") )
	    *s++ = *(ip6_addr_t *) node_data->_data_parm.data;
    }

    hdr_len = (void *) s - dest;

    rthdr->ip6r_len = (hdr_len - 8)/8; /* RFC 2460 4.4 */
}

/*
 * Calculates the length of an IPv6 routing extension header.
 * This length is variable a depends on the PDU definition.
 */
static size_t
pdu_ipv6rthdr_len(const GNode *pdu)
{
    const GNode *child;
    size_t hdr_len;
    struct node_data *node_data;

    hdr_len = sizeof(struct ip6_rthdr) + sizeof(uint32_t) /* reserved */;

    for (child = g_node_first_child(pdu);
	 child;
	 child = g_node_next_sibling(child) ) {
	node_data = child->data;

	/* We only care about "router"; everything else ("next-header",
	 * "type", "segs-left") is handled above.
	 */
	if (!strcmp(node_data->name, "router") )
	    hdr_len += sizeof(ip6_addr_t);
    }

    return hdr_len;
}

#if 0
static void
pdu_ipv6rthdr_dumper(pdu_t *p, const char *prefix)
{
    struct ipv6hdr_options *hdr_data;

    hdr_data = p->header_data;

    printf("%s      Parameters:\n", prefix);
    printf("%s        Routing type: %s\n", prefix,
	   num_info(hdr_data->_rthdr.type) );
    printf("%s        Segments left: %s\n", prefix,
	   num_info(hdr_data->_rthdr.segleft) );
    printf("%s        Number of routers: %d\n", prefix,
	   hdr_data->_rthdr.nrouters);
}
#endif

/* Basic IPv6 header */
static pdu_t pdu_ipv6 = {
    .name = "ip6",
    .description = "IPv6 packet",
    .documented_in = "RFC 2460",
    .len = sizeof(struct ip6_hdr),
    .fields = (field_t []) {
	{.name = "next-header", .type = PDU_FTYPE_NUMTYPE},
	{.name = "version", .type = PDU_FTYPE_NUMTYPE},
	{.name = "hlim", .type = PDU_FTYPE_NUMTYPE},
	{.name = "src", .type = PDU_FTYPE_IP6ADDR},
	{.name = "dst", .type = PDU_FTYPE_IP6ADDR},
	{.name = NULL}
    },
    .build = &pdu_ipv6hdr_builder,
    .postbuild = &pdu_ipv6hdr_builder2,
};

/* Example: ip6hbh(pad1, tlv(type = 5, length = 5, data = '\x12\x34\x56') ) */
static pdu_t pdu_ipv6hbh = {
    .name = "ip6hbh",
    .description = "Hop-by-hop options",
    .documented_in = "RFC 2460",
    .fields = (field_t []) {
	{.name = "next-header", .type = PDU_FTYPE_NUMTYPE},
	{.name = "pad1", .type = PDU_FTYPE_BIT},
	{.name = "padn", .type = PDU_FTYPE_NUMTYPE},
	{.name = "raw-tlv", .type = PDU_FTYPE_DATA},
	{.name = "tlv", .type = PDU_FTYPE_BRACED_ARGS, .data = (field_t []) {
	    {.name = "type", .type = PDU_FTYPE_NUMTYPE},
	    {.name = "length", .type = PDU_FTYPE_NUMTYPE},
	    {.name = "value", .type = PDU_FTYPE_DATA},
	    {.name = NULL} }, .defval = NULL},
	{.name = NULL}
    },
    .build = &pdu_ipv6hbh_builder,
    .get_len = &pdu_ipv6hbh_len
};

static pdu_t pdu_ipv6destopts = {
    .name = "ip6dest",
    .description = "Destination options",
    .documented_in = "RFC 2460",
    .fields = (field_t []) {
	{.name = "next-header", .type = PDU_FTYPE_NUMTYPE},
	{.name = "pad1", .type = PDU_FTYPE_BIT},
	{.name = "padn", .type = PDU_FTYPE_NUMTYPE},
	{.name = "raw-tlv", .type = PDU_FTYPE_DATA},
	{.name = "tlv", .type = PDU_FTYPE_BRACED_ARGS, .data = (field_t []) {
	    {.name = "type", .type = PDU_FTYPE_NUMTYPE},
	    {.name = "length", .type = PDU_FTYPE_NUMTYPE},
	    {.name = "value", .type = PDU_FTYPE_DATA},
	    {.name = NULL} }, .defval = NULL},
	{.name = NULL}
    },
    .build = &pdu_ipv6destopts_builder,
    .get_len = &pdu_ipv6hbh_len /* Yes, it's okay to use the one for HBH */
};

static pdu_t pdu_ipv6frag = {
    .name = "ip6frag",
    .description = "Fragment extension header",
    .documented_in = "RFC 2460",
    .len = sizeof(struct ip6_frag),
    .fields = (field_t []) {
	{.name = "next-header", .type = PDU_FTYPE_NUMTYPE},
	{.name = "id", .type = PDU_FTYPE_NUMTYPE},
	{.name = "offset", .type = PDU_FTYPE_NUMTYPE},
	{.name = "more", .type = PDU_FTYPE_BIT},
	{.name = NULL}
    },
    .build = &pdu_ipv6frag_builder,
};

static pdu_t pdu_ipv6rthdr = {
    .name = "ip6rth",
    .description = "Routing extension header",
    .documented_in = "RFC 2460",
    .fields = (field_t []) {
	{.name = "next-header", .type = PDU_FTYPE_NUMTYPE},
	{.name = "type", .type = PDU_FTYPE_NUMTYPE},
	{.name = "segs-left", .type = PDU_FTYPE_NUMTYPE},
	{.name = "router", .type = PDU_FTYPE_IP6ADDR},
	{.name = NULL}
    },
    .build = &pdu_ipv6rthdr_builder,
    .get_len = &pdu_ipv6rthdr_len
};

void
_pb_register_ipv6(void)
{
    next_headers = _pb_new_hash(protocol_mappings);

    _pb_register_protocol(&pdu_ipv6);
    _pb_register_protocol(&pdu_ipv6hbh);
    _pb_register_protocol(&pdu_ipv6destopts);
    _pb_register_protocol(&pdu_ipv6frag);
    _pb_register_protocol(&pdu_ipv6rthdr);
}
