/* pdu-dtp.c
 
   Builder for Cisco Dynamic Trunking Protocol (DTP) PDUs.

   Copyright (C) 2007, 2008, 2009, 2010 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*

Not a whole lot out there on DTP. Some references that I found:

- Yersinia (www.yersinia.net): src/dtp.[ch]

- Scapy DTP bug (http://trac.secdev.org/scapy/ticket/115)

A NetExpect DTP PDU example:

dot3(dst = 01:00:0c:cc:cc:cc)/llc(dsap=0xaa, ssap=0xaa, ctrl = 0x3)/snap(oui='\\x00\\x00\\x0c', code = 0x2004)/dtp(options(tlv(type='DTPDomain', value="12345"), tlv(type='DTPType', value="\\xa5"), tlv(type = 'DTPStatus', value="\\x10") ) )

*/

#include "pbuild-priv.h"

static struct pdu_dict dtp_tlv_types[] = { /* Enumeration for DTP TLV types */
    {"DTPDomain",   (uint32_t []) {0x01} },
    {"DTPStatus",   (uint32_t []) {0x02} },
    {"DTPType",     (uint32_t []) {0x03} },
    {"DTPNeighbor", (uint32_t []) {0x04} },
    {NULL,          NULL}
};

static const pdu_t pdu_dtp = {
    .name = "dtp",
    .description = "DTP packet",
    .documented_in = "www.cisco.com",
    .fields = (field_t []) {
	{
	    .name = "version",
	    .type = PDU_FTYPE_UINT8,
	    .offset = 0,
	    .defval = (defval_t []) { {.type = PDU_DEF_NUM, ._number = 1} }
	},
	{
	    .name = "options", /* DTP TLVs */
	    .type = PDU_FTYPE_BRACED_ARGS,
	    .data = (field_t []) {
		{
		    .name = "tlv",
		    .type = PDU_FTYPE_BRACED_ARGS,
		    .data = (field_t []) {
			{
			    .name = "type",
			    .type = PDU_FTYPE_UINT16,
			    .offset = 0,
			    .data = dtp_tlv_types /* Enumeration */
			},
			{
			    .name = "length",
			    .type = PDU_FTYPE_UINT16,
			    .offset = 2,
			    .defval = (defval_t []) {
				{.type = PDU_DEF_LEN_TLV2}
			    }
			},
			{
			    .name = "value",
			    .type = PDU_FTYPE_DATA,
			    .offset = 4
			},
			{
			    .name = NULL
			}
		    }, .defval = NULL
		},
		{
		    .name = NULL
		}
	    }, .defval = NULL
	},
	{
	    .name = NULL
	}
    },
    .len = sizeof(uint8_t),
    .options_class = "dtp"
};

static const pdu_option_t dtp_tlv = {
    .name = "tlv",
    .description = "DTP TLV",
    .get_len = &pdu_generic_tlv_len2 /* According to the Wireshark disector,
					type and length in these TVLs are
					two bytes each. Don't know if that's
					right but I'm going with what they
					are doing. */
};

void
_pb_register_dtp(void)
{
    _pb_register_protocol(&pdu_dtp);

    _pb_register_protocol_option("dtp", &dtp_tlv);
}
