/*
 * NASPRO - The NASPRO Architecture for Sound PROcessing
 * Portable runtime library
 *
 * Copyright (C) 2007-2014 Stefano D'Angelo
 *
 * See the COPYING file for license conditions.
 */

#define _WIN32_WINNT	0x501
#define UNICODE

#include "internal.h"

#include <windows.h>

static uint16_t *
msg_get(DWORD err)
{
	LPTSTR msgbuf;
	uint16_t *ret;

	if (FormatMessageW(FORMAT_MESSAGE_ALLOCATE_BUFFER
			   | FORMAT_MESSAGE_FROM_SYSTEM, NULL, err, 0,
			   (LPTSTR)&msgbuf, 0, NULL) == 0)
		return NULL;

	ret = nacore_string_utf16_to_utf8_a((char *)msgbuf);

	LocalFree(msgbuf);

	return ret;
}

_NACORE_DEF nacore_dl_handle
nacore_dl_open(const char *filename, nacore_msg_context msg_context,
	       void *msg_data)
{
	HMODULE ret;
	uint16_t *filenamew;
	char *msg;

	filenamew = nacore_string_utf8_to_utf16_a(filename);
	if (filenamew == NULL)
	  {
		nacore_msg_text(msg_context, nacore_msg_severity_err, msg_data,
				"Not enough memory");
		return NULL;
	  }

	ret = LoadLibraryW((LPCWSTR)filenamew);
	free(filenamew);
	if (ret == NULL)
	  {
		msg = msg_get(GetLastError());
		if (msg != NULL)
		  {
			nacore_msg_text(msg_context, nacore_msg_severity_err,
					msg_data, "Could not load the dynamic "
					"library (%s)", msg);
			free(msg);
		  }
		else
			nacore_msg_text(msg_context, nacore_msg_severity_err,
					msg_data, "Could not load the dynamic "
					"library");
	  }

	return (nacore_dl_handle)ret;
}

_NACORE_DEF void *
nacore_dl_sym(nacore_dl_handle handle, const char *symbol,
	      nacore_msg_context msg_context, void *msg_data)
{
	FARPROC ret;
	char *msg;

	ret = GetProcAddress((HMODULE)handle, symbol);
	if (ret == NULL)
	  {
		msg = msg_get(GetLastError());
		if (msg != NULL)
		  {
			nacore_msg_text(msg_context, nacore_msg_severity_err,
					msg_data, "Could not resolve symbol "
					"`%s' (%s)", symbol, msg);
			free(msg);
		  }
		else
			nacore_msg_text(msg_context, nacore_msg_severity_err,
					msg_data, "Could not resolve symbol "
					"`%s'", symbol);
		errno = NACORE_EUNKNOWN;
	  }

	return (void *)((NACORE_LIBC_SIZE_FORMAT_TYPE)ret);
}

_NACORE_DEF int
nacore_dl_close(nacore_dl_handle handle, nacore_msg_context msg_context,
		void *msg_data)
{
	char *msg;

	if (FreeLibrary((HMODULE)handle) != 0)
	  {
		msg = msg_get(GetLastError());
		if (msg != NULL)
		  {
			nacore_msg_text(msg_context, nacore_msg_severity_err,
					msg_data, "Could not unload the "
					"dynamic library (%s)", msg);
			free(msg);
		  }
		else
			nacore_msg_text(msg_context, nacore_msg_severity_err,
					msg_data, "Could not unload the "
					"dynamic library");
		return NACORE_EUNKNOWN;
	  }

	return 0;
}
