#
# debug.py: Helper routines for debugging
#
# Copyright 2008, Red Hat  Inc.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 2 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#

import logging
import logging.handlers
import optparse
import sys
import Queue
import threading
import time
import os

real_stdout = sys.stdout
real_stderr = sys.stderr
write_log_worker_thread = None

class Worker:
    def __init__(self, queue, out):
        self.queue = queue
        self.out = out
        self.stopevent = threading.Event()
        self.stopevent.clear()
        self.worker_thread = None

    def consumer_queue(self):
       while True:
           try:
               message = self.queue.get_nowait()
           except Queue.Empty:
               break 
           self.queue.task_done()
           for out in self.out:
               out.write(message)
       for out in self.out:
           out.flush()

    def write_log_worker(self):
        while True:
            if self.stopevent.isSet():
                self.consumer_queue()
                break
            self.consumer_queue()
            time.sleep(1)

    def start_write_log_worker(self, *args, **kwargs):
        self.worker_thread = threading.Thread(target=self.write_log_worker,
                                  args=args, kwargs=kwargs,
                                  name="mic2-worker")
        self.worker_thread.setDaemon(False)
        self.worker_thread.start()
        return self.worker_thread

    def terminate_write_log_worker(self):
        self.stopevent.set()
        self.worker_thread.join()
        self.consumer_queue()
        for out in self.out:
            if not out.isatty():
                out.close()

    def flush_log(self):
        self.consumer_queue()

class QueueWriter():
    def __init__(self, queue):
        self.queue = queue
        self.myfileno = 1
        self.mylogworker = None

    def write(self, string):
        self.queue.put(string)

    def close(self):
        pass

    def flush(self):
        if self.mylogworker:
            self.mylogworker.flush_log()

    def isatty(self):
        return False

    def fileno(self):
        return self.myfileno

_logworker = None
_logfile = None

def handle_logging(option, opt, val, parser, logger, level):
    if level < logger.level:
        logger.setLevel(level)

def handle_logfile(option, opt, val, parser, logger, stream):
    global _logworker, _logfile
    if _logworker and _logfile:
        """ Only need to initialize it once """
        return
    try:
        val = os.path.abspath(os.path.expanduser(val))
        logfile = logging.FileHandler(val,"a")
    except IOError, (err, msg):
        raise optparse.OptionValueError("Cannot open file '%s' : %s" %
                                        (val, msg))


    logger.removeHandler(stream)
    logger.addHandler(logfile)
    myout = QueueWriter(Queue.Queue())
    myout.myfileno = logfile.stream.fileno()
    sys.stdout = myout
    sys.stderr = myout
    _logworker = Worker(myout.queue, [logfile.stream, real_stdout])
    _logworker.start_write_log_worker()
    _logfile = val
    myout.mylogworker = _logworker

def get_logfile():
    return _logfile

def terminate_log_thread():
    if _logworker:
        _logworker.terminate_write_log_worker()

def setup_logging(parser = None):
    """Set up the root logger and add logging options.

    Set up the root logger so only warning/error messages are logged to stderr
    by default.

    Also, optionally, add --debug, --verbose and --logfile command line options
    to the supplied option parser, allowing the root logger configuration to be
    modified by the user.

    Note, to avoid possible namespace clashes, setup_logging() will only ever
    add these three options. No new options will be added in the future.

    parser -- an optparse.OptionParser instance, or None

    """
    logger = logging.getLogger()

    logger.setLevel(logging.WARN)

    stream = logging.StreamHandler(sys.stderr)

    logger.addHandler(stream)

    if parser is None:
        return

    group = optparse.OptionGroup(parser, "Debugging options",
                                 "These options control the output of logging information during image creation")

    group.add_option("-d", "--debug",
                     action = "callback", callback = handle_logging,
                     callback_args = (logger, logging.DEBUG),
                     help = "Output debugging information")

    group.add_option("-v", "--verbose",
                     action = "callback", callback = handle_logging,
                     callback_args = (logger, logging.INFO),
                     help = "Output verbose progress information")

    group.add_option("", "--logfile", type="string",
                     action = "callback", callback = handle_logfile,
                     callback_args = (logger, stream),
                     help = "Save debug information to FILE", metavar = "FILE")

    parser.add_option_group(group)
