/* libspe - A wrapper library to adapt the JSRE SPE usage model to SPUFS
 * Copyright (C) 2005 IBM Corp.
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License,
 * or (at your option) any later version.
 *  This library is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
 *  License for more details.
 *   You should have received a copy of the GNU Lesser General Public License
 *   along with this library; if not, write to the Free Software Foundation,
 *   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#define _GNU_SOURCE
#include <errno.h>
#include <fcntl.h>
#include <libspe.h>
#include <pthread.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include <sys/poll.h>

#include "spe.h"

#define __PRINTF(fmt, args...) { fprintf(stderr,fmt , ## args); }
#ifdef DEBUG
#define DEBUG_PRINTF(fmt, args...) __PRINTF(fmt , ## args)
#else
#define DEBUG_PRINTF(fmt, args...)
#endif


struct mfc_command_parameter_area {
	uint32_t pad;	/* reserved		 */
	uint32_t lsa;	/* local storage address */
	uint64_t ea;	/* effective address	 */
	uint16_t size;	/* transfer size	 */
	uint16_t tag;	/* command tag		 */
	uint16_t class;	/* class ID		 */
	uint16_t cmd;	/* command opcode	 */
};

enum mfc_cmd {
	MFC_CMD_PUT  = 0x20,
	MFC_CMD_PUTB = 0x21,
	MFC_CMD_PUTF = 0x22,
	MFC_CMD_GET  = 0x40,
	MFC_CMD_GETB = 0x41,
	MFC_CMD_GETF = 0x42,
};

static int spe_do_mfc_put(speid_t spe, unsigned src, void *dst,
			  unsigned size, unsigned tag, unsigned class,
			  enum mfc_cmd cmd)
{
	struct thread_store *thread_store = spe;
	struct mfc_command_parameter_area parm = {
		.lsa   = src,
		.ea    = (unsigned long) dst,
		.size  = size,
		.tag   = tag,
		.class = class,
		.cmd   = cmd,
	};
	int ret;

	DEBUG_PRINTF("queuing DMA %x %lx %x %x %x %x\n", parm.lsa,
		parm.ea, parm.size, parm.tag, parm.class, parm.cmd);

	if (!srch_thread(spe))
		return -1;

	if (thread_store->fd_mfc == -1) {
		/* the kernel does not support DMA, so just copy directly */
		memcpy(dst, thread_store->mem_mmap_base + src, size);
		return 0;
	}

	ret = write(thread_store->fd_mfc, &parm, sizeof (parm));
	if ((ret < 0) && (errno != EIO)) {
		perror("spe_do_mfc_put: internal error");
	}
	return ret < 0 ? -1 : 0;
}

static int spe_do_mfc_get(speid_t spe, unsigned dst, void *src,
			  unsigned size, unsigned tag, unsigned class,
			  enum mfc_cmd cmd)
{
	struct thread_store *thread_store = spe;
	struct mfc_command_parameter_area parm = {
		.lsa   = dst,
		.ea    = (unsigned long) src,
		.size  = size,
		.tag   = tag,
		.class = class,
		.cmd   = cmd,
	};
	int ret;

	DEBUG_PRINTF("queuing DMA %x %lx %x %x %x %x\n", parm.lsa,
		parm.ea, parm.size, parm.tag, parm.class, parm.cmd);

	if (!srch_thread(spe))
		return -1;

	if (thread_store->fd_mfc == -1) {
		/* the kernel does not support DMA, so just copy directly */
		memcpy(thread_store->mem_mmap_base + dst, src, size);
		return 0;
	}

	ret = write(thread_store->fd_mfc, &parm, sizeof (parm));
	if ((ret < 0) && (errno != EIO)) {
		perror("spe_do_mfc_get: internal error");
	}
	return ret < 0 ? -1 : 0;
}

int spe_mfc_put (speid_t spe, unsigned src, void *dst,
		unsigned size, unsigned short tag,
		unsigned char tid, unsigned char rid)
{
	return spe_do_mfc_put(spe, src, dst, size, tag & 0xf,
				 tid << 8 | rid, MFC_CMD_PUT);
}

int spe_mfc_putf(speid_t spe, unsigned src, void *dst,
		unsigned size, unsigned short tag,
		unsigned char tid, unsigned char rid)
{
	return spe_do_mfc_put(spe, src, dst, size, tag & 0xf,
				tid << 8 | rid, MFC_CMD_PUTF);
}

int spe_mfc_putb(speid_t spe, unsigned src, void *dst,
		unsigned size, unsigned short tag,
		unsigned char tid, unsigned char rid)
{
	return spe_do_mfc_put(spe, src, dst, size, tag & 0xf,
				tid << 8 | rid, MFC_CMD_PUTB);
}

int spe_mfc_get (speid_t spe, unsigned dst, void *src,
		unsigned size, unsigned short tag,
		unsigned char tid, unsigned char rid)
{
	return spe_do_mfc_get(spe, dst, src, size, tag & 0xf,
				tid << 8 | rid, MFC_CMD_GET);
}

int spe_mfc_getf(speid_t spe, unsigned dst, void *src,
		unsigned size, unsigned short tag,
		unsigned char tid, unsigned char rid)
{
	return spe_do_mfc_get(spe, dst, src, size, tag & 0xf,
				tid << 8 | rid, MFC_CMD_GETF);
}

int spe_mfc_getb(speid_t spe, unsigned dst, void *src,
		unsigned size, unsigned short tag,
		unsigned char tid, unsigned char rid)
{
	return spe_do_mfc_get(spe, dst, src, size, tag & 0xf,
				tid << 8 | rid, MFC_CMD_GETB);
}
/* MFC Read tag status functions
 *
 */
static int read_tag_status_noblock(speid_t speid)
{
	struct thread_store *spe = speid;

	int r_read = 0;
	unsigned int ret;

	r_read = read(spe->fd_mfc,&ret,4);

	if (r_read == 4)
	{
		return ret;
	}

	return -1;
}

static int read_tag_status_async(speid_t speid)
{
	struct thread_store *spe = speid;
	struct pollfd poll_fd;
	
	int r_read = 0;
	unsigned int ret;

	poll_fd.fd = spe->fd_mfc;
	poll_fd.events = POLLIN;

	ret = poll(&poll_fd, 1, -1);

	if (ret < 0 || !(poll_fd.revents | POLLIN))
		return -1;
	
	r_read = read(spe->fd_mfc,&ret,4);

	if (r_read == 4)
	{
		return ret;
	}

	return -1;
}

int spe_mfc_read_tag_status_all(speid_t speid, unsigned int mask)
{
	int status;
	
	if ((status = read_tag_status_noblock(speid)) == -1)
		return -1;

	while ((status & mask) != mask)
	{
		if ((status = read_tag_status_async(speid)) == -1)
			return -1;
	}

	return status & mask;
}

int spe_mfc_read_tag_status_any(speid_t speid, unsigned int mask)
{
	int status;
	
	if ((status = read_tag_status_noblock(speid)) == -1)
		return -1;

	while ((status & mask) == 0)
	{
		if ((status = read_tag_status_async(speid)) == -1)
			return -1;
	}

	return status & mask;
}

int spe_mfc_read_tag_status_immediate(speid_t speid, unsigned int mask)
{
	int status;
	
	if ((status = read_tag_status_noblock(speid)) == -1)
		return -1;
	
	return  status & mask ;
}

