/* IMSpector - Instant Messenger Transparent Proxy Service
 * http://www.imspector.org/
 * (c) Lawrence Manning <lawrence@aslak.net>, 2006
 * 
 * Contributions from:
 *     Ryan Wagoner <ryan@wgnrs.dynu.com>, 2006
 *          
 * Released under the GPL v2. */

#include "imspector.h"

#include <mysql/mysql.h>
#include <mysql/errmsg.h>

#define PLUGIN_NAME "MySQL IMSpector logging plugin"
#define PLUGIN_SHORT_NAME "MySQL"

#define CREATE_TABLE "CREATE TABLE IF NOT EXISTS `messages` ( " \
	"`id` int(11) NOT NULL auto_increment, " \
	"`timestamp` int(11) NOT NULL default '0', " \
	"`clientaddress` text NOT NULL, " \
	"`protocolname` text NOT NULL, " \
	"`outgoing` int(11) NOT NULL default '0', " \
	"`type` int(11) NOT NULL default '0', " \
	"`localid` text NOT NULL, " \
	"`remoteid` text NOT NULL, " \
	"`filtered` int(11) NOT NULL default '0', " \
	"`categories` text NOT NULL, " \
	"`eventdata` blob NOT NULL, " \
	"PRIMARY KEY  (`id`) " \
	") ENGINE=MyISAM DEFAULT CHARSET=latin1"

#define INSERT_STATEMENT "INSERT INTO messages " \
	"(id, timestamp, clientaddress, protocolname, outgoing, type, localid, remoteid, filtered, categories, eventdata) " \
	"VALUES (0, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)"
#define NO_FIELDS 10

extern "C"
{
	bool initloggingplugin(struct loggingplugininfo &ploggingplugininfo,
		class Options &options, bool debugmode);
	void closeloggingplugin(void);
	int logevents(std::vector<struct imevent> &imevents);
};

MYSQL *conn;
MYSQL_STMT *stmt;
MYSQL_BIND binds[NO_FIELDS];

time_t timestamp;
char clientaddress[STRING_SIZE]; unsigned long clientaddresslength;
char protocolname[STRING_SIZE]; unsigned long protocolnamelength;
int outgoing; int type;
char localid[STRING_SIZE]; unsigned long localidlength;
char remoteid[STRING_SIZE]; unsigned long remoteidlength;
int filtered;
char categories[STRING_SIZE]; unsigned long categorieslength;
char eventdata[BUFFER_SIZE]; unsigned long eventdatalength;

bool localdebugmode = false;
bool connected = false;
int retries = 0;

std::string server;
std::string database;
std::string username;
std::string password;

std::vector<struct imevent> mysqlevents;

bool connectmysql(void);

bool initloggingplugin(struct loggingplugininfo &loggingplugininfo,
	class Options &options, bool debugmode)
{
	server = options["mysql_server"];
	database = options["mysql_database"];
	username = options["mysql_username"];
	password = options["mysql_password"];

	if (server.empty()) return false;

	localdebugmode = debugmode;

	loggingplugininfo.pluginname = PLUGIN_NAME;

	if (!(conn = mysql_init(NULL))) return false;

	return connected = connectmysql();
}

void closeloggingplugin(void)
{
	mysql_close(conn);

	return;
}

/* The main plugin function. See loggingplugin.cpp. */
int logevents(std::vector<struct imevent> &imevents)
{	
	int rc = 0;

	/* store imevents locally in case of sql connection error */
	for (std::vector<struct imevent>::iterator i = imevents.begin(); i != imevents.end(); i++)
		mysqlevents.push_back(*i);

	/* if not connected try and connect */
	if (!connected)
	{
		retries++;
		
		if ((retries <= 2) || ((retries % 10) == 0))
		{
			debugprint(localdebugmode, PLUGIN_SHORT_NAME ": Trying to connect, retries: %d", retries);
			if ( (connected = connectmysql()) )
			{
				syslog(LOG_NOTICE, PLUGIN_SHORT_NAME ": Reconnected to database, "\
					"pending events will now be logged");
				retries = 0;
			}
			/* still not able to connect, user will get the connection attempt errors,
			 * lets not fill log with uneeded error messages */
			else
			{
				debugprint(localdebugmode, PLUGIN_SHORT_NAME ": Still not able to connect", retries);
				return 0;
			}
		}
		else
		{
			debugprint(localdebugmode, PLUGIN_SHORT_NAME ": Connection to server dead; " \
				"queued events: %d retries: %d", mysqlevents.size(), retries);
			return 0;
		}
	}
	
	/* try and process local imevents */
	while (mysqlevents.size())
	{
		struct imevent imevent = mysqlevents.front();
			
		timestamp = imevent.timestamp;

		memset(clientaddress, 0, STRING_SIZE);
		strncpy(clientaddress, imevent.clientaddress.c_str(), STRING_SIZE - 1);
		clientaddresslength = strlen(clientaddress);
		
		memset(protocolname, 0, STRING_SIZE);
		strncpy(protocolname, imevent.protocolname.c_str(), STRING_SIZE - 1);
		protocolnamelength = strlen(protocolname);
		
		outgoing = imevent.outgoing;
		type = imevent.type;
		
		memset(localid, 0, STRING_SIZE);
		strncpy(localid, imevent.localid.c_str(), STRING_SIZE - 1);
		localidlength = strlen(localid);
		
		memset(remoteid, 0, STRING_SIZE);
		strncpy(remoteid, imevent.remoteid.c_str(), STRING_SIZE - 1);
		remoteidlength = strlen(remoteid);

		filtered = imevent.filtered;

		memset(categories, 0, STRING_SIZE);
		strncpy(categories, imevent.categories.c_str(), STRING_SIZE - 1);
		categorieslength = strlen(categories);
		
		memset(eventdata, 0, BUFFER_SIZE);
		strncpy(eventdata, imevent.eventdata.c_str(), BUFFER_SIZE - 1);
		eventdatalength = strlen(eventdata);
		
		/* we're connected insert the data */
		if (connected)
		{
			debugprint(localdebugmode,PLUGIN_SHORT_NAME ": Connected, so logging one event");
			if ((rc = mysql_stmt_execute(stmt))) 
			{
				syslog(LOG_ERR, PLUGIN_SHORT_NAME ": mysql_stmt_execute(), Error: %s", mysql_stmt_error(stmt));
			
				/* connection lost */
				if (mysql_stmt_errno(stmt) == CR_SERVER_LOST ||
					mysql_stmt_errno(stmt) == CR_SERVER_GONE_ERROR)
				{
					debugprint(localdebugmode, PLUGIN_SHORT_NAME ": Connection lost");
					connected = false;
				}
			
				return mysql_stmt_errno(stmt);
			}
			else
				mysqlevents.erase(mysqlevents.begin());
		}
	}

	return 0;
}

bool connectmysql(void)
{
	/* Connect to database */
	if (!mysql_real_connect(conn, server.c_str(), username.c_str(), password.c_str(), 
		database.c_str(), 0, NULL, 0))
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Couldn't connect to database, Error: %s", mysql_error(conn));
		return false;
	}

	if (mysql_query(conn, CREATE_TABLE))
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Couldn't create table, Error: %s", mysql_error(conn));
		return false;
	}
		
	if (!(stmt = mysql_stmt_init(conn)))
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": mysql_stmt_init(), Error: out of memory");
		return false;
	}
	
	if (mysql_stmt_prepare(stmt, INSERT_STATEMENT, strlen(INSERT_STATEMENT)))
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": mysql_stmt_prepare(), Error: %s", mysql_stmt_error(stmt));
		return false;
	}
		
	if (mysql_stmt_param_count(stmt) != NO_FIELDS)
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": mysql_stmt_param_count(), Error: invalid parameter count");
		return false;
	}

	memset(binds, 0, sizeof(MYSQL_BIND) * NO_FIELDS);
	
	/* timestamp. */
	/* This is a number type, so there is no need to specify buffer_length. */
	binds[0].buffer_type = MYSQL_TYPE_LONG;
	binds[0].buffer=  (char *)&timestamp;
	binds[0].is_null = 0;
	binds[0].length = 0;

	/* clientaddress */
	binds[1].buffer_type = MYSQL_TYPE_STRING;
	binds[1].buffer = (char *)clientaddress;
	binds[1].buffer_length = STRING_SIZE;
	binds[1].is_null = 0;
	binds[1].length = &clientaddresslength;
	
	/* protocolname */
	binds[2].buffer_type = MYSQL_TYPE_STRING;
	binds[2].buffer = (char *)protocolname;
	binds[2].buffer_length = STRING_SIZE;
	binds[2].is_null = 0;
	binds[2].length = &protocolnamelength;
	
	/* outgoing */
	binds[3].buffer_type = MYSQL_TYPE_LONG;
	binds[3].buffer = (char *)&outgoing;
	binds[3].is_null = 0;
	binds[3].length = 0;
	
	/* type */
	binds[4].buffer_type = MYSQL_TYPE_LONG;
	binds[4].buffer = (char *)&type;
	binds[4].is_null = 0;
	binds[4].length = 0;

	/* localid */
	binds[5].buffer_type = MYSQL_TYPE_STRING;
	binds[5].buffer = (char *)localid;
	binds[5].buffer_length = STRING_SIZE;
	binds[5].is_null = 0;
	binds[5].length = &localidlength;
	
	/* remoteid */
	binds[6].buffer_type = MYSQL_TYPE_STRING;
	binds[6].buffer = (char *)remoteid;
	binds[6].buffer_length = STRING_SIZE;
	binds[6].is_null = 0;
	binds[6].length = &remoteidlength;

	/* filtered */
	binds[7].buffer_type = MYSQL_TYPE_LONG;
	binds[7].buffer = (char *)&filtered;
	binds[7].is_null = 0;
	binds[7].length = 0;

	/* categories */
	binds[8].buffer_type = MYSQL_TYPE_STRING;
	binds[8].buffer = (char *)categories;
	binds[8].buffer_length = STRING_SIZE;
	binds[8].is_null = 0;
	binds[8].length = &categorieslength;

	/* eventdata blob */
	binds[9].buffer_type = MYSQL_TYPE_BLOB;
	binds[9].buffer = (char *)eventdata;
	binds[9].buffer_length = BUFFER_SIZE;
	binds[9].is_null = 0;
	binds[9].length = &eventdatalength;

	/* Bind the buffers */
	if (mysql_stmt_bind_param(stmt, binds))
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": mysql_stmt_bind_param(), Error: %s", mysql_stmt_error(stmt));
		return false;
	}

	return true;
}
