/* Gnome Scan - Scan as easy as you print
 * Copyright © 2007  Étienne Bersac <bersace03@laposte.net>
 *
 * Gnome Scan is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * gnome-scan is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with gnome-scan.  If not, write to:
 *
 *	the Free Software Foundation, Inc.
 *	51 Franklin Street, Fifth Floor
 *	Boston, MA 02110-1301, USA
 */

/**
 * SECTION: gnome-scan-preview-area
 * @short_description: Preview processing and select ROI
 *
 * Not yet implemented ;).
 **/

#include "gnome-scan-preview-area.h"

#define	GET_PRIVATE(o)	(G_TYPE_INSTANCE_GET_PRIVATE ((o), GNOME_TYPE_SCAN_PREVIEW_AREA, GnomeScanPreviewAreaPrivate))

typedef struct _GnomeScanPreviewAreaPrivate GnomeScanPreviewAreaPrivate;
struct _GnomeScanPreviewAreaPrivate
{
    GSList			*plugins;
    GnomeScanPreviewPlugin	*plugin;
    
    gdouble			resolution;
    GdkPixbuf			*image;
    cairo_t			*cairo_context;
    cairo_surface_t		*buffer;
    gboolean			buffer_deprecated;
};

enum
{
    PROP_0,
    PROP_PIXBUF,
    PROP_RESOLUTION
};

enum
{
    PREVIEW_RECEIVED,
    LAST_SIGNAL
};


static GtkDrawingAreaClass* parent_class = NULL;
static guint scan_preview_area_signals[LAST_SIGNAL] = { 0 };

/* INTERNALS */
static gboolean	gspa_expose (GnomeScanPreviewArea *gspa,
                                GdkEventExpose *event);
static void     gspa_draw (GnomeScanPreviewArea *gspa,
                           GdkRegion *region);




G_DEFINE_TYPE (GnomeScanPreviewArea, gnome_scan_preview_area, GTK_TYPE_DRAWING_AREA);

static GObject*
gnome_scan_preview_area_constructor (GType type, guint n_construct_properties, GObjectConstructParam *construct_properties)
{
    GObject *object = 
        G_OBJECT_CLASS (gnome_scan_preview_area_parent_class)->constructor (type,
                                                                            n_construct_properties,
                                                                            construct_properties);
    GtkWidget *widget = GTK_WIDGET (object);
    GnomeScanPreviewArea *gspa = GNOME_SCAN_PREVIEW_AREA (widget);
    gtk_widget_set_size_request (widget, 100, 100);
    gtk_widget_set_sensitive (widget, FALSE);
    
    g_signal_connect (object, "expose-event",
                      G_CALLBACK (gspa_expose), gspa);
    
    return object;
}

static void
gnome_scan_preview_area_init (GnomeScanPreviewArea *object)
{
}

static void
gnome_scan_preview_area_finalize (GObject *object)
{
    /* TODO: Add deinitalization code here */
    
    G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gnome_scan_preview_area_set_property (GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec)
{
    g_return_if_fail (GNOME_IS_SCAN_PREVIEW_AREA (object));
    
    switch (prop_id)
    {
        case PROP_PIXBUF:
            /* TODO: Add setter for "pixbuf" property here */
            break;
        case PROP_RESOLUTION:
            /* TODO: Add setter for "resolution" property here */
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
gnome_scan_preview_area_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
    g_return_if_fail (GNOME_IS_SCAN_PREVIEW_AREA (object));
    
    switch (prop_id)
    {
        case PROP_PIXBUF:
            /* TODO: Add getter for "pixbuf" property here */
            break;
        case PROP_RESOLUTION:
            /* TODO: Add getter for "resolution" property here */
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
gnome_scan_preview_area_area_selected (GnomeScanPreviewArea *preview_area, GdkRectangle *area)
{
    /* TODO: Add default signal handler implementation here */
}

static void
gnome_scan_preview_area_class_init (GnomeScanPreviewAreaClass *klass)
{
    GObjectClass* object_class = G_OBJECT_CLASS (klass);
    parent_class = GTK_DRAWING_AREA_CLASS (g_type_class_peek_parent (klass));
    
    g_type_class_add_private (klass, sizeof (GnomeScanPreviewAreaPrivate));
    
    object_class->finalize		= gnome_scan_preview_area_finalize;
    object_class->constructor	= gnome_scan_preview_area_constructor;
    object_class->set_property	= gnome_scan_preview_area_set_property;
    object_class->get_property	= gnome_scan_preview_area_get_property;
    
    
    g_object_class_install_property (object_class,
                                     PROP_PIXBUF,
                                     g_param_spec_object ("pixbuf",
                                                          "Pixbuf",
                                                          "The image buffer where to select an area",
                                                          GDK_TYPE_PIXBUF,
                                                          G_PARAM_READABLE | G_PARAM_WRITABLE));
    
    g_object_class_install_property (object_class,
                                     PROP_RESOLUTION,
                                     g_param_spec_double ("resolution",
                                                          "Resolution",
                                                          "Image resolution for translation of pixel area into GtkUnit.",
                                                          1.0,
                                                          4800.0,
                                                          50.0,
                                                          G_PARAM_READABLE | G_PARAM_WRITABLE));
    
    /* SIGNALS */
    scan_preview_area_signals[PREVIEW_RECEIVED] =
        g_signal_new ("preview-received",
                      G_OBJECT_CLASS_TYPE (klass),
                      G_SIGNAL_RUN_FIRST,
                      G_STRUCT_OFFSET (GnomeScanPreviewAreaClass, preview_received),
                      NULL, NULL,
                      g_cclosure_marshal_VOID__OBJECT,
                      G_TYPE_NONE, 1, GDK_TYPE_PIXBUF);
}


GtkWidget*
gnome_scan_preview_area_new	(void)
{
    return GTK_WIDGET (g_object_new (GNOME_TYPE_SCAN_PREVIEW_AREA,
                                     NULL));
}

void
gnome_scan_preview_area_set_pixbuf (GnomeScanPreviewArea *gspa, GdkPixbuf *pixbuf, gdouble resolution)
{
    GnomeScanPreviewAreaPrivate *priv = GET_PRIVATE (gspa);
    gint width, height;
    
    if (priv->image)
        g_object_unref (priv->image);
    
    priv->image = g_object_ref (pixbuf);
    
    priv->resolution = resolution;
    width = gdk_pixbuf_get_width (pixbuf);
    height = gdk_pixbuf_get_height (pixbuf);
    gtk_widget_set_size_request (GTK_WIDGET (gspa),
                                 width, height);
    
    gtk_widget_set_sensitive (GTK_WIDGET (gspa), TRUE);
    
    gtk_widget_realize (GTK_WIDGET (gspa));
    while (gtk_events_pending ()) gtk_main_iteration();
    
    /* create buffer */
    cairo_t *cr;
    
    if (priv->buffer) {
        cairo_surface_destroy (priv->buffer);
        priv->buffer = NULL;
    }
    
    GdkPixmap *pixmap = gdk_pixmap_new (NULL, width, height, 24);
    cr = gdk_cairo_create (GDK_DRAWABLE (pixmap));
    priv->buffer = cairo_surface_create_similar (cairo_get_target (cr),
                                                 CAIRO_CONTENT_COLOR,
                                                 width, height);
    cairo_destroy (cr);
    g_object_unref (pixmap);
    
    gnome_scan_preview_plugin_changed (priv->plugin);
    gnome_scan_preview_area_update (gspa, NULL);
}


gdouble
gnome_scan_preview_area_get_resolution (GnomeScanPreviewArea *preview_area)
{
    g_return_val_if_fail (GNOME_IS_SCAN_PREVIEW_AREA (preview_area), 72.);
    return GET_PRIVATE (preview_area)->resolution;
}

void
gnome_scan_preview_area_get_size (GnomeScanPreviewArea *preview_area,
                                  gint *width,
                                  gint*height)
{
    g_return_if_fail (GNOME_IS_SCAN_PREVIEW_AREA (preview_area));
    GnomeScanPreviewAreaPrivate *priv = GET_PRIVATE (preview_area);
    *width = gdk_pixbuf_get_width (priv->image);
    *height = gdk_pixbuf_get_height (priv->image);
}


void
gnome_scan_preview_area_update (GnomeScanPreviewArea *preview_area,
                                GdkRectangle *r)
{
    g_return_if_fail (GNOME_IS_SCAN_PREVIEW_AREA (preview_area));
    GET_PRIVATE (preview_area)->buffer_deprecated = TRUE;
    
    if (r) {
        gtk_widget_queue_draw_area (GTK_WIDGET (preview_area),
                                    r->x, r->y,
                                    r->width, r->height);
    }
    else {
        gtk_widget_queue_draw (GTK_WIDGET (preview_area));
    }
    
    return;
}


void
gnome_scan_preview_area_select_plugin (GnomeScanPreviewArea *pa,
                                       GnomeScanPreviewPlugin *plugin)
{
    GnomeScanPreviewAreaPrivate *priv = GET_PRIVATE(pa);
    priv->plugin = plugin;
}

/* INTERNALS */
static void
gspa_draw (GnomeScanPreviewArea *gspa,
           GdkRegion *region)
{
    GnomeScanPreviewAreaPrivate *priv = GET_PRIVATE (gspa);
    GtkWidget *widget = GTK_WIDGET (gspa);
    cairo_t *cr;
    
    /* draw to buffer */
    cr = cairo_create (priv->buffer);
    
    /* clip to region */
        gdk_cairo_region (cr, region);
    cairo_clip (cr);
    
    /* draw image */
        gdk_cairo_set_source_pixbuf (cr, priv->image, 0, 0);
    cairo_paint (cr);
    
    /* delegate to current plugin */
        gnome_scan_preview_plugin_draw_buffer (priv->plugin, GTK_WIDGET (gspa), cr);
    
    cairo_destroy (cr);
    
    priv->buffer_deprecated = FALSE;
}

static gboolean
gspa_expose (GnomeScanPreviewArea *gspa,
             GdkEventExpose *event)
{
    GnomeScanPreviewAreaPrivate *priv = GET_PRIVATE (gspa);
    GtkWidget *widget = GTK_WIDGET (gspa);
    
    /* update buffer */
        /* TODO: draw optionnaly */
        gspa_draw (gspa, event->region);
    
    cairo_t *cr = gdk_cairo_create (GDK_DRAWABLE (widget->window));
    gdk_cairo_region (cr, event->region);
    cairo_clip (cr);
    cairo_set_source_surface (cr, priv->buffer, 0, 0);
    cairo_paint (cr);
    
    if (!GTK_WIDGET_IS_SENSITIVE (widget)) {
        gdk_cairo_set_source_color (cr, &widget->style->bg[GTK_WIDGET_STATE (widget)]);
        cairo_paint_with_alpha (cr, .5);
    }
    
    cairo_destroy (cr);
    
    return FALSE;
}
