# Copyright 2019-2020 by Christopher C. Little.
# This file is part of Abydos.
#
# Abydos is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Abydos is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Abydos. If not, see <http://www.gnu.org/licenses/>.

"""abydos.distance._goodman_kruskal_tau_a.

Goodman & Kruskal's Tau A similarity
"""

from typing import Any, Counter as TCounter, Optional, Sequence, Set, Union

from ._token_distance import _TokenDistance
from ..tokenizer import _Tokenizer

__all__ = ['GoodmanKruskalTauA']


class GoodmanKruskalTauA(_TokenDistance):
    r"""Goodman & Kruskal's Tau A similarity.

    For two sets X and Y and a population N, Goodman & Kruskal's :math:`\tau_a`
    similarity :cite:`Goodman:1954`, by analogy with :math:`\tau_b`, is

        .. math::

            sim_{GK_{\tau_a}}(X, Y) =
            \frac{\frac{\frac{|X \cap Y|}{|N|}^2 +
            \frac{|Y \setminus X|}{|N|}^2}{\frac{|Y|}{|N|}}+
            \frac{\frac{|X \setminus Y|}{|N|}^2 +
            \frac{|(N \setminus X) \setminus Y|}{|N|}^2}
            {\frac{|N \setminus X|}{|N|}} -
            (\frac{|X|}{|N|}^2 + \frac{|N \setminus X|}{|N|}^2)}
            {1 - (\frac{|X|}{|N|}^2 + \frac{|N \setminus X|}{|N|}^2)}

    In :ref:`2x2 confusion table terms <confusion_table>`, where a+b+c+d=n,
    after each term has been converted to a proportion by dividing by n, this
    is

        .. math::

            sim_{GK_{\tau_a}} =
            \frac{
            \frac{a^2 + c^2}{a+c} +
            \frac{b^2 + d^2}{b+d} -
            ((a+b)^2 + (c+d)^2)}
            {1 - ((a+b)^2 + (c+d)^2)}

    .. versionadded:: 0.4.0
    """

    def __init__(
        self,
        alphabet: Optional[
            Union[TCounter[str], Sequence[str], Set[str], int]
        ] = None,
        tokenizer: Optional[_Tokenizer] = None,
        intersection_type: str = 'crisp',
        normalizer: str = 'proportional',
        **kwargs: Any
    ) -> None:
        """Initialize GoodmanKruskalTauA instance.

        Parameters
        ----------
        alphabet : Counter, collection, int, or None
            This represents the alphabet of possible tokens.
            See :ref:`alphabet <alphabet>` description in
            :py:class:`_TokenDistance` for details.
        tokenizer : _Tokenizer
            A tokenizer instance from the :py:mod:`abydos.tokenizer` package
        intersection_type : str
            Specifies the intersection type, and set type as a result:
            See :ref:`intersection_type <intersection_type>` description in
            :py:class:`_TokenDistance` for details.
        normalizer : str
            Specifies the normalization type. See :ref:`normalizer <alphabet>`
            description in :py:class:`_TokenDistance` for details.
        **kwargs
            Arbitrary keyword arguments

        Other Parameters
        ----------------
        qval : int
            The length of each q-gram. Using this parameter and tokenizer=None
            will cause the instance to use the QGram tokenizer with this
            q value.
        metric : _Distance
            A string distance measure class for use in the ``soft`` and
            ``fuzzy`` variants.
        threshold : float
            A threshold value, similarities above which are counted as
            members of the intersection for the ``fuzzy`` variant.


        .. versionadded:: 0.4.0

        """
        super(GoodmanKruskalTauA, self).__init__(
            alphabet=alphabet,
            tokenizer=tokenizer,
            intersection_type=intersection_type,
            normalizer=normalizer,
            **kwargs
        )

    def sim(self, src: str, tar: str) -> float:
        """Return Goodman & Kruskal's Tau A similarity of two strings.

        Parameters
        ----------
        src : str
            Source string (or QGrams/Counter objects) for comparison
        tar : str
            Target string (or QGrams/Counter objects) for comparison

        Returns
        -------
        float
            Goodman & Kruskal's Tau A similarity

        Examples
        --------
        >>> cmp = GoodmanKruskalTauA()
        >>> cmp.sim('cat', 'hat')
        0.3304969657208484
        >>> cmp.sim('Niall', 'Neil')
        0.22137604585914503
        >>> cmp.sim('aluminum', 'Catalan')
        0.05991264724130685
        >>> cmp.sim('ATCG', 'TAGC')
        4.119695274745721e-05


        .. versionadded:: 0.4.0

        """
        self._tokenize(src, tar)

        a = self._intersection_card()
        b = self._src_only_card()
        c = self._tar_only_card()
        d = self._total_complement_card()

        if a + b == 0 or a + c == 0:
            return 0.0

        fp = (a * a + c * c) / (a + c)

        sp = b * b + d * d
        if sp:
            sp /= b + d

        num = fp + sp - (a + b) ** 2 - (c + d) ** 2
        if num > 1e-14:
            return num / (1 - (a + b) ** 2 - (c + d) ** 2)
        return 0.0  # pragma: no cover


if __name__ == '__main__':
    import doctest

    doctest.testmod()
